<?php
/* =============================================================================
 * File: broker_onboarding.php
 * Purpose: Broker Onboarding (Melody theme) with uploads & DB persistence
 * Stack  : Core PHP + MySQLi (NO PDO). DB via broker-config.php. Collation: utf8mb4_general_ci
 * Notes  : Uses base layout slots: $styles, $content, $scripts (layouts/base.php)
 * ============================================================================= */
@session_start();
require_once __DIR__ . '/../config.php';
require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);
ini_set('display_errors',1); ini_set('display_startup_errors',1); error_reporting(E_ALL);
if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ---- Required includes (as requested) ---- */
require_once 'broker-config.php';
require_once 'functions.php';

/* ---- Ensure DB ready & charset/collation ---- */
if (!isset($conn) || !($conn instanceof mysqli)) { die('DB connection missing.'); }
mysqli_set_charset($conn,'utf8mb4');
@mysqli_query($conn,"SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn,"SET collation_connection='utf8mb4_general_ci'");

/* ---- Create table (idempotent) ---- */
mysqli_query($conn, "CREATE TABLE IF NOT EXISTS brokers (
  id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  broker_code VARCHAR(30) COLLATE utf8mb4_general_ci NOT NULL,
  first_name VARCHAR(80) COLLATE utf8mb4_general_ci DEFAULT NULL,
  last_name VARCHAR(80) COLLATE utf8mb4_general_ci DEFAULT NULL,
  email VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  phone VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  alt_phone VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  whatsapp VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  dob DATE DEFAULT NULL,
  firm_name VARCHAR(180) COLLATE utf8mb4_general_ci DEFAULT NULL,
  business_type ENUM('Individual','Partnership','LLP','Pvt Ltd','Public Ltd','Other') COLLATE utf8mb4_general_ci DEFAULT NULL,
  rera_id VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  license_valid_till DATE DEFAULT NULL,
  gstin VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  pan VARCHAR(20) COLLATE utf8mb4_general_ci DEFAULT NULL,
  addr_line VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  city VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  state VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  pincode VARCHAR(20) COLLATE utf8mb4_general_ci DEFAULT NULL,
  bank_account_name VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  bank_account_number VARCHAR(40) COLLATE utf8mb4_general_ci DEFAULT NULL,
  bank_ifsc VARCHAR(20) COLLATE utf8mb4_general_ci DEFAULT NULL,
  bank_name VARCHAR(120) COLLATE utf8mb4_general_ci DEFAULT NULL,
  bank_branch VARCHAR(120) COLLATE utf8mb4_general_ci DEFAULT NULL,
  doc_rera_cert VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  doc_pan_card VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  doc_gst_cert VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  doc_cancelled_cheque VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  doc_address_proof VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  referral_source VARCHAR(120) COLLATE utf8mb4_general_ci DEFAULT NULL,
  notes TEXT COLLATE utf8mb4_general_ci,
  agreement_accept TINYINT(1) NOT NULL DEFAULT 0,
  status ENUM('Pending','Approved','Rejected') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'Pending',
  created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME NULL ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");

/* ---- Helpers (safe fallbacks) ---- */
if (!function_exists('esc')) { function esc($s){ return htmlspecialchars((string)$s,ENT_QUOTES,'UTF-8'); } }
if (!function_exists('v'))   { function v($k,$d=''){ return isset($_POST[$k]) ? trim((string)$_POST[$k]) : $d; } }
function gen_code(){ return 'BRK-'.date('Ymd').'-'.random_int(1000,9999); }
function ensure_dir($p){ if(!is_dir($p)) @mkdir($p,0775,true); }

/* ---- State ---- */
$success   = isset($_GET['ok']) ? 1 : 0;
$error_msg = '';

/* ---- Handle POST ---- */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  $first_name     = v('first_name');
  $phone          = v('phone');
  $email          = v('email');
  $firm_name      = v('firm_name');
  $business_type  = v('business_type');
  $rera_id        = v('rera_id');
  $agreement_accept = isset($_POST['agreement_accept']) ? 1 : 0;

  if ($first_name==='' || $phone==='' || $email==='' || $firm_name==='' || $business_type==='' || $rera_id==='') {
    $error_msg = 'Please fill all * required fields.';
  } elseif (!$agreement_accept) {
    $error_msg = 'Please accept the onboarding terms.';
  }

  if ($error_msg==='') {
    /* Optionals/normalize */
    $last_name   = v('last_name') ?: NULL;
    $alt_phone   = v('alt_phone') ?: NULL;
    $whatsapp    = v('whatsapp') ?: NULL;
    $dob         = v('dob')==='' ? NULL : v('dob');
    $license_valid_till = v('license_valid_till')==='' ? NULL : v('license_valid_till');
    $gstin       = ($tmp=strtoupper(v('gstin')))!=='' ? $tmp : NULL;
    $pan         = ($tmp=strtoupper(v('pan')))!=='' ? $tmp : NULL;
    $addr_line   = v('addr_line') ?: NULL;
    $city        = v('city') ?: NULL;
    $state       = v('state') ?: NULL;
    $pincode     = v('pincode') ?: NULL;
    $bank_account_name   = v('bank_account_name') ?: NULL;
    $bank_account_number = v('bank_account_number') ?: NULL;
    $bank_ifsc           = ($tmp=strtoupper(v('bank_ifsc')))!=='' ? $tmp : NULL;
    $bank_name           = v('bank_name') ?: NULL;
    $bank_branch         = v('bank_branch') ?: NULL;
    $referral_source     = v('referral_source') ?: NULL;
    $notes               = v('notes') ?: NULL;

    $broker_code = gen_code();

    /* Uploads */
    $saved = ['doc_rera_cert'=>NULL,'doc_pan_card'=>NULL,'doc_gst_cert'=>NULL,'doc_cancelled_cheque'=>NULL,'doc_address_proof'=>NULL];
    $klist = array_keys($saved);
    $base  = __DIR__."/uploads/brokers/$broker_code";
    ensure_dir($base);

    foreach ($klist as $k){
      if (!empty($_FILES[$k]['name']) && is_uploaded_file($_FILES[$k]['tmp_name'])) {
        $ext = strtolower(pathinfo($_FILES[$k]['name'], PATHINFO_EXTENSION));
        if (!in_array($ext,['jpg','jpeg','png','webp','gif','pdf'])) { $error_msg="Invalid file for $k"; break; }
        $safe = preg_replace('~[^A-Za-z0-9._-]+~','-', $_FILES[$k]['name']);
        $fn = $k.'_'.time().'_'.$safe;
        if (move_uploaded_file($_FILES[$k]['tmp_name'], $base.'/'.$fn)) {
          $saved[$k] = 'uploads/brokers/'.$broker_code.'/'.$fn;
        }
      }
    }

    if ($error_msg==='') {
      $status = 'Pending';

      $cols = "broker_code,first_name,last_name,email,phone,alt_phone,whatsapp,dob,
               firm_name,business_type,rera_id,license_valid_till,gstin,pan,
               addr_line,city,state,pincode,
               bank_account_name,bank_account_number,bank_ifsc,bank_name,bank_branch,
               doc_rera_cert,doc_pan_card,doc_gst_cert,doc_cancelled_cheque,doc_address_proof,
               referral_source,notes,agreement_accept,status";

      $sql = "INSERT INTO brokers ($cols) VALUES (".str_repeat('?,',31).'?)';
      $stmt = mysqli_prepare($conn,$sql);
      if(!$stmt){ $error_msg = 'Prepare failed.'; }
      else {
        $params = [
          $broker_code,$first_name,$last_name,$email,$phone,$alt_phone,$whatsapp,$dob,
          $firm_name,$business_type,$rera_id,$license_valid_till,$gstin,$pan,
          $addr_line,$city,$state,$pincode,
          $bank_account_name,$bank_account_number,$bank_ifsc,$bank_name,$bank_branch,
          $saved['doc_rera_cert'],$saved['doc_pan_card'],$saved['doc_gst_cert'],$saved['doc_cancelled_cheque'],$saved['doc_address_proof'],
          $referral_source,$notes,(int)$agreement_accept,$status
        ];
        $types = '';
        foreach ($params as $vv) { $types .= (is_int($vv) ? 'i' : 's'); }

        mysqli_stmt_bind_param($stmt, $types, ...$params);
        if (!mysqli_stmt_execute($stmt)) {
          $error_msg = 'Save failed.';
        } else {
          header('Location: broker_onboarding.php?ok=1'); exit;
        }
        mysqli_stmt_close($stmt);
      }
    }
  }
}


/* ================================ $styles ================================ */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.css">
<style>
  .form-section-title{ font-weight:600; margin:8px 0 14px; font-size:14px; text-transform:uppercase; letter-spacing:.4px; color:#64748b; }
  .page-hint-badge{ font-size:.8rem; }
</style>
<?php $styles = ob_get_clean();

/* =============================== $content =============================== */
ob_start(); ?>
<!-- Page Header (Melody) -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Broker Onboarding</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">Partners</a></li>
      <li class="breadcrumb-item active" aria-current="page">Onboarding</li>
    </ol>
  </div>
</div>

<?php if($error_msg): ?>
  <div class="alert alert-danger"><?php echo esc($error_msg); ?></div>
<?php endif; ?>

<?php if($success): ?>
  <div class="alert alert-success">Thank you! Your application has been submitted for Admin review.</div>
<?php endif; ?>

<form method="post" enctype="multipart/form-data" novalidate>
  <!-- Personal -->
  <div class="card custom-card mb-3">
    <div class="card-header justify-content-between">
      <div class="card-title"><i class="fa fa-id-card me-2"></i> Personal Information</div>
      <span class="badge bg-light text-dark page-hint-badge">Fields marked * are mandatory</span>
    </div>
    <div class="card-body">
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">First Name *</label><input type="text" name="first_name" class="form-control" value="<?= esc(v('first_name')) ?>" required></div>
        <div class="col-md-4"><label class="form-label">Last Name</label><input type="text" name="last_name" class="form-control" value="<?= esc(v('last_name')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Email *</label><input type="email" name="email" class="form-control" value="<?= esc(v('email')) ?>" required></div>
        <div class="col-md-4"><label class="form-label">Phone *</label><input type="text" name="phone" class="form-control" value="<?= esc(v('phone')) ?>" required><small class="text-muted">Primary contact number</small></div>
        <div class="col-md-4"><label class="form-label">Alternate Phone</label><input type="text" name="alt_phone" class="form-control" value="<?= esc(v('alt_phone')) ?>"></div>
        <div class="col-md-4"><label class="form-label">WhatsApp</label><input type="text" name="whatsapp" class="form-control" value="<?= esc(v('whatsapp')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Date of Birth</label><input type="date" name="dob" class="form-control" value="<?= esc(v('dob')) ?>"></div>
      </div>
    </div>
  </div>

  <!-- Firm & Compliance -->
  <div class="card custom-card mb-3">
    <div class="card-header"><div class="card-title"><i class="fa fa-building me-2"></i> Firm & Compliance</div></div>
    <div class="card-body">
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">Firm/Company Name *</label><input type="text" name="firm_name" class="form-control" value="<?= esc(v('firm_name')) ?>" required></div>
        <div class="col-md-4">
          <label class="form-label">Business Type *</label><?php $bt=v('business_type'); ?>
          <select name="business_type" class="form-control" required>
            <option value="">Select</option>
            <option <?= $bt==='Individual'?'selected':'' ?> value="Individual">Individual</option>
            <option <?= $bt==='Partnership'?'selected':'' ?> value="Partnership">Partnership</option>
            <option <?= $bt==='LLP'?'selected':'' ?> value="LLP">LLP</option>
            <option <?= $bt==='Pvt Ltd'?'selected':'' ?> value="Pvt Ltd">Pvt Ltd</option>
            <option <?= $bt==='Public Ltd'?'selected':'' ?> value="Public Ltd">Public Ltd</option>
            <option <?= $bt==='Other'?'selected':'' ?> value="Other">Other</option>
          </select>
        </div>
        <div class="col-md-4"><label class="form-label">RERA ID *</label><input type="text" name="rera_id" class="form-control" value="<?= esc(v('rera_id')) ?>" required></div>
        <div class="col-md-4"><label class="form-label">License Valid Till</label><input type="date" name="license_valid_till" class="form-control" value="<?= esc(v('license_valid_till')) ?>"></div>
        <div class="col-md-4"><label class="form-label">GSTIN</label><input type="text" name="gstin" class="form-control" value="<?= esc(v('gstin')) ?>"></div>
        <div class="col-md-4"><label class="form-label">PAN</label><input type="text" name="pan" class="form-control" value="<?= esc(v('pan')) ?>"></div>
      </div>
    </div>
  </div>

  <!-- Address -->
  <div class="card custom-card mb-3">
    <div class="card-header"><div class="card-title"><i class="fa fa-map-marker me-2"></i> Address</div></div>
    <div class="card-body">
      <div class="row g-3">
        <div class="col-md-6"><label class="form-label">Address Line</label><input type="text" name="addr_line" class="form-control" value="<?= esc(v('addr_line')) ?>"></div>
        <div class="col-md-3"><label class="form-label">City</label><input type="text" name="city" class="form-control" value="<?= esc(v('city')) ?>"></div>
        <div class="col-md-3"><label class="form-label">State</label><input type="text" name="state" class="form-control" value="<?= esc(v('state')) ?>"></div>
        <div class="col-md-3"><label class="form-label">Pincode</label><input type="text" name="pincode" class="form-control" value="<?= esc(v('pincode')) ?>"></div>
      </div>
    </div>
  </div>

  <!-- Bank -->
  <div class="card custom-card mb-3">
    <div class="card-header"><div class="card-title"><i class="fa fa-university me-2"></i> Bank Details</div></div>
    <div class="card-body">
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">Account Holder Name</label><input type="text" name="bank_account_name" class="form-control" value="<?= esc(v('bank_account_name')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Account Number</label><input type="text" name="bank_account_number" class="form-control" value="<?= esc(v('bank_account_number')) ?>"></div>
        <div class="col-md-4"><label class="form-label">IFSC</label><input type="text" name="bank_ifsc" class="form-control" value="<?= esc(v('bank_ifsc')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Bank Name</label><input type="text" name="bank_name" class="form-control" value="<?= esc(v('bank_name')) ?>"></div>
        <div class="col-md-4"><label class="form-label">Branch</label><input type="text" name="bank_branch" class="form-control" value="<?= esc(v('bank_branch')) ?>"></div>
      </div>
    </div>
  </div>

  <!-- Documents -->
  <div class="card custom-card mb-3">
    <div class="card-header"><div class="card-title"><i class="fa fa-file-upload me-2"></i> Documents</div></div>
    <div class="card-body">
      <p class="text-muted small mb-3">Allowed: JPG, PNG, WEBP, GIF, PDF</p>
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">RERA Certificate</label><input type="file" name="doc_rera_cert" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf"></div>
        <div class="col-md-4"><label class="form-label">PAN Card</label><input type="file" name="doc_pan_card" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf"></div>
        <div class="col-md-4"><label class="form-label">GST Certificate</label><input type="file" name="doc_gst_cert" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf"></div>
        <div class="col-md-4"><label class="form-label">Cancelled Cheque</label><input type="file" name="doc_cancelled_cheque" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf"></div>
        <div class="col-md-4"><label class="form-label">Address Proof</label><input type="file" name="doc_address_proof" class="form-control" accept=".jpg,.jpeg,.png,.webp,.gif,.pdf"></div>
      </div>
    </div>
  </div>

  <!-- Additional -->
  <div class="card custom-card mb-3">
    <div class="card-header"><div class="card-title"><i class="fa fa-clipboard-list me-2"></i> Additional</div></div>
    <div class="card-body">
      <div class="row g-3">
        <div class="col-md-4"><label class="form-label">Referral Source</label><?php $rs=v('referral_source'); ?>
          <select name="referral_source" class="form-control">
            <option value="">Select</option>
            <option <?= $rs==='Website'?'selected':'' ?> value="Website">Website</option>
            <option <?= $rs==='Walk-in'?'selected':'' ?> value="Walk-in">Walk-in</option>
            <option <?= $rs==='Referral'?'selected':'' ?> value="Referral">Referral</option>
            <option <?= $rs==='Campaign'?'selected':'' ?> value="Campaign">Campaign</option>
            <option <?= $rs==='Other'?'selected':'' ?> value="Other">Other</option>
          </select>
        </div>
        <div class="col-md-8"><label class="form-label">Notes</label><textarea name="notes" rows="3" class="form-control"><?= esc(v('notes')) ?></textarea></div>
        <div class="col-md-12">
          <div class="form-check">
            <input type="checkbox" class="form-check-input" id="agree" name="agreement_accept" required>
            <label class="form-check-label" for="agree">I confirm the information is true and I accept the onboarding terms. *</label>
          </div>
        </div>
      </div>

      <div class="mt-3">
        <button class="btn btn-primary" type="submit"><i class="fa fa-save me-2"></i> Submit Application</button>
      </div>
    </div>
  </div>
</form>
<?php $content = ob_get_clean();

/* =============================== $scripts =============================== */
ob_start(); ?>
<script src="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.js"></script>
<?php if($success): ?>
<script>
  setTimeout(function(){
    Swal.fire({icon:'success', title:'Submitted', text:'Your application has been submitted for Admin review.', timer:2400, showConfirmButton:false});
  }, 50);
</script>
<?php endif; ?>
<?php $scripts = ob_get_clean();

/* ------------------------------ Render base ------------------------------ */
include 'layouts/base.php';
