<?php
/* ============================================================================
 * File: pages/index8.php
 * Purpose: HRM Dashboard (dynamic data) - keep original look & layout
 * Tech   : MySQLi only (no PDO), utf8mb4_general_ci
 * Uses   : require_once __DIR__ . '/../config.php';
 *          $conn    -> broriserin_onb                (default)
 *          $hrd     -> broriserin_hrd                (HR dump)
 *          $conn_ic -> broriserin_ic_attendance      (Attendance)
 * KPIs   :
 *   - Total Employees           = hr_dump WHERE Status='Active'
 *   - New Employees (this month)= hr_dump WHERE Status='Active' AND created_at in current month
 *   - Total Job Applicants      = COUNT(*) FROM user_pif
 *   - Total Joined (this month) = COUNT(*) FROM employee_applications (created_at in current month)
 * Charts :
 *   - Candidate Statistics (12-month grouped columns: Hired vs Responses)
 *   - Attendance Overview (semi-donut radial with center “Total” like design image)
 * Lists :
 *   - Employees Status: top 4 HR_Designation (label + count) -> progress bars %
 *     (now widened to span up to Attendance Overview width)
 *   - Employees By Department: horizontal purple bars
 *   - Employees List: color-coded department badges (as in your design)
 *   - Today's Attendance -> replaced with Indian Corporate Holidays (Jan→Dec) + Day badge
 * Render : $styles, $content, $scripts -> include 'layouts/base.php'
 * ========================================================================== */

session_start();
require_once __DIR__ . '/../config.php';

require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);
/* ---------- Helpers ---------- */
if (!function_exists('dbq')) {
  function dbq($conn, $sql){ return $conn ? mysqli_query($conn, $sql) : false; }
}
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function fmt_num($n){ return number_format((int)($n ?? 0)); }
function safe_date($s){ if(!$s || $s==='0000-00-00' || $s==='0000-00-00 00:00:00') return ''; return date('M d, Y', strtotime($s)); }
function dept_badge_class($dept){
  $d = strtolower((string)$dept);
  if (strpos($d,'marketing') !== false) return 'bg-primary-transparent';
  if (strpos($d,'it') !== false || strpos($d,'develop') !== false || strpos($d,'tech') !== false) return 'bg-warning-transparent';
  if (strpos($d,'operation') !== false) return 'bg-secondary-transparent';
  if (strpos($d,'finance') !== false || strpos($d,'account') !== false) return 'bg-success-transparent';
  if (strpos($d,'data') !== false || strpos($d,'analyt') !== false) return 'bg-info-transparent';
  if (strpos($d,'sales') !== false) return 'bg-success-transparent';
  return 'bg-info-transparent';
}
function day_badge_class($dateStr){
  $day = date('l', strtotime($dateStr));
  $map = [
    'Monday'    => 'bg-primary-transparent',
    'Tuesday'   => 'bg-secondary-transparent',
    'Wednesday' => 'bg-info-transparent',
    'Thursday'  => 'bg-warning-transparent',
    'Friday'    => 'bg-success-transparent',
    'Saturday'  => 'bg-warning-transparent',
    'Sunday'    => 'bg-danger-transparent',
  ];
  return $map[$day] ?? 'bg-primary-transparent';
}

/* ---------- Date ranges ---------- */
$today           = date('Y-m-d');
$monthStart      = date('Y-m-01');
$nextMonthStart  = date('Y-m-01', strtotime('+1 month'));
$thisYearStart   = date('Y-01-01');
$nextYearStart   = date('Y-01-01', strtotime('+1 year'));

/* ============================== DATA QUERIES ============================== */
/* KPIs */
$totalEmployees = 0; // Active
$newEmployeesThisMonth = 0; // Active created this month
$totalApplicants = 0; // user_pif
$totalJoinedThisMonth = 0; // employee_applications created this month

$r = dbq($hrd, "SELECT COUNT(*) AS c FROM broriserin_hrd.hr_dump WHERE UPPER(TRIM(Status))='ACTIVE'");
if($r){ $row = mysqli_fetch_assoc($r) ?: []; $totalEmployees=(int)($row['c']??0); }

$r = dbq($hrd, "SELECT COUNT(*) AS c FROM broriserin_hrd.hr_dump 
                WHERE UPPER(TRIM(Status))='ACTIVE'
                  AND (created_at >= '{$monthStart}' AND created_at < '{$nextMonthStart}')");
if($r){ $row = mysqli_fetch_assoc($r) ?: []; $newEmployeesThisMonth=(int)($row['c']??0); }

$r = dbq($conn, "SELECT COUNT(*) AS c FROM broriserin_onb.user_pif");
if($r){ $row = mysqli_fetch_assoc($r) ?: []; $totalApplicants=(int)($row['c']??0); }

$r = dbq($conn, "SELECT COUNT(*) AS c FROM broriserin_onb.employee_applications 
                 WHERE (created_at >= '{$monthStart}' AND created_at < '{$nextMonthStart}')");
if($r){ $row = mysqli_fetch_assoc($r) ?: []; $totalJoinedThisMonth=(int)($row['c']??0); }

/* Candidate Statistics — monthly series (current year) */
$seriesHired = array_fill(1,12,0);
$seriesResponses = array_fill(1,12,0);

/* Hired per month (approved/hired) */
$qHired = "
  SELECT DATE_FORMAT(created_at,'%m') AS m, COUNT(*) AS c
  FROM broriserin_onb.employee_applications
  WHERE created_at >= '{$thisYearStart}' AND created_at < '{$nextYearStart}'
    AND LOWER(application_status) IN ('approved','hired')
  GROUP BY m
";
$r = dbq($conn, $qHired);
if($r){ while($row=mysqli_fetch_assoc($r)){ $seriesHired[(int)$row['m']] = (int)$row['c']; } }

/* Responses per month (user_pif) */
$qResp = "
  SELECT DATE_FORMAT(created_at,'%m') AS m, COUNT(*) AS c
  FROM broriserin_onb.user_pif
  WHERE created_at >= '{$thisYearStart}' AND created_at < '{$nextYearStart}'
  GROUP BY m
";
$r = dbq($conn, $qResp);
if($r){ while($row=mysqli_fetch_assoc($r)){ $seriesResponses[(int)$row['m']] = (int)$row['c']; } }

/* Candidate stat totals (header numbers in the middle card) */
$totalHiredAllTime = 0; $totalResponsesAllTime = $totalApplicants;
$r = dbq($conn, "SELECT COUNT(*) AS c FROM broriserin_onb.employee_applications WHERE LOWER(application_status) IN ('approved','hired')");
if($r){ $row = mysqli_fetch_assoc($r) ?: []; $totalHiredAllTime=(int)($row['c']??0); }

/* Attendance Overview (this month) */
$att_present=0; $att_absent=0; $att_late=0; $att_srt=0; $att_early=0;
$qAtt = "
  SELECT
    SUM(CASE WHEN UPPER(TRIM(FinalStatus))='P'     THEN 1 ELSE 0 END) AS p_cnt,
    SUM(CASE WHEN UPPER(TRIM(FinalStatus))='A'     THEN 1 ELSE 0 END) AS a_cnt,
    SUM(CASE WHEN UPPER(TRIM(FinalStatus))='LATE'  THEN 1 ELSE 0 END) AS late_cnt,
    SUM(CASE WHEN UPPER(TRIM(FinalStatus))='SRT'   THEN 1 ELSE 0 END) AS srt_cnt,
    SUM(CASE WHEN UPPER(TRIM(FinalStatus))='EARLY' THEN 1 ELSE 0 END) AS early_cnt
  FROM broriserin_ic_attendance.ams_final
  WHERE OfficeDate >= '{$monthStart}' AND OfficeDate < '{$nextMonthStart}'
";
$r = dbq($conn_ic, $qAtt);
if($r){
  $row = mysqli_fetch_assoc($r) ?: [];
  $att_present = (int)($row['p_cnt']??0);
  $att_absent  = (int)($row['a_cnt']??0);
  $att_late    = (int)($row['late_cnt']??0);
  $att_srt     = (int)($row['srt_cnt']??0);
  $att_early   = (int)($row['early_cnt']??0);
}
$att_total = $att_present + $att_absent + $att_late + $att_srt + $att_early;
$att_series_pct = [0,0,0,0,0];
if($att_total>0){
  $att_series_pct = [
    round(($att_present/$att_total)*100,1),
    round(($att_late/$att_total)*100,1),
    round(($att_srt/$att_total)*100,1),
    round(($att_early/$att_total)*100,1),
    round(($att_absent/$att_total)*100,1)
  ];
}

/* Employees Status = top 4 HR_Designation buckets */
$desigBuckets = []; $totalForPct = 0;
$r = dbq($hrd, "SELECT COALESCE(NULLIF(TRIM(HR_Designation),''),'Unassigned') AS d, COUNT(*) AS c
                FROM broriserin_hrd.hr_dump
                GROUP BY d
                ORDER BY c DESC
                LIMIT 4");
if($r){ while($row=mysqli_fetch_assoc($r)){ $desigBuckets[]=['label'=>$row['d'],'count'=>(int)$row['c']]; $totalForPct += (int)$row['c']; } }
while(count($desigBuckets)<4){ $desigBuckets[]=['label'=>'—','count'=>0]; }
$desigPct = [];
foreach($desigBuckets as $b){ $desigPct[] = $totalForPct>0 ? round(($b['count']/$totalForPct)*100) : 0; }

/* Employees By Department (top 6) */
$deptLabels=[]; $deptValues=[];
$r = dbq($hrd, "SELECT COALESCE(NULLIF(TRIM(Department),''),'Unassigned') AS dept, COUNT(*) AS c
                FROM broriserin_hrd.hr_dump
                GROUP BY dept
                ORDER BY c DESC
                LIMIT 6");
if($r){ while($row=mysqli_fetch_assoc($r)){ $deptLabels[]=$row['dept']; $deptValues[]=(int)$row['c']; } }

/* Employees List (latest 6) */
$recentJoins=[];
$r = dbq($hrd, "SELECT Employee_Full_Name, HR_Designation, Department, created_at, Status
                FROM broriserin_hrd.hr_dump
                ORDER BY COALESCE(updated_at, created_at) DESC
                LIMIT 6");
if($r){
  while($row=mysqli_fetch_assoc($r)){
    $recentJoins[] = [
      'name' => $row['Employee_Full_Name'] ?: '—',
      'dept' => $row['Department'] ?: '—',
      'role' => $row['HR_Designation'] ?: '—',
      'date' => safe_date($row['created_at'] ?? ''),
      'status' => $row['Status'] ?: ''
    ];
  }
}

/* Indian Corporate Holidays — Jan→Dec (removed FY Closing, Good Friday, Labour Day) */
$yr = (int)date('Y');
$holidays = [
  ['date'=>"$yr-01-01", 'name'=>"New Year's Day"],
  ['date'=>"$yr-01-26", 'name'=>"Republic Day"],
  ['date'=>"$yr-03-14", 'name'=>"Holi"],               // moveable; placeholder
  ['date'=>"$yr-08-15", 'name'=>"Independence Day"],
  ['date'=>"$yr-10-02", 'name'=>"Gandhi Jayanti"],
  ['date'=>"$yr-10-08", 'name'=>"Dussehra"],           // moveable; placeholder
  ['date'=>"$yr-10-20", 'name'=>"Diwali (Deepavali)"], // moveable; placeholder
  ['date'=>"$yr-12-25", 'name'=>"Christmas Day"],
];
usort($holidays, function($a,$b){ return strcmp($a['date'],$b['date']); });

/* ============================== VIEW: $styles ============================== */
ob_start();
?>
<style>
  /* restore theme spacing and badges */
  .dashboard-main-card .fw-semibold.mb-0.lh-1{font-size:clamp(1.25rem,3.5vw,1.75rem);line-height:1}
  .employee-status-marker{display:inline-block;padding:.25rem .5rem;border-radius:4px;background:#f5f5f7;font-weight:600}
  .employee-status-marker.remote{border-left:6px solid #7c3aed}
  .employee-status-marker.probation{border-left:6px solid #22c55e}
  .employee-status-marker.contract{border-left:6px solid #f59e0b}
  .employee-status-marker.work-home{border-left:6px solid #ef4444}
  .min-w-badge{min-width:86px;text-align:center}
  .avatar img{object-fit:cover}
  .avatar.avatar-md{width:40px;height:40px;display:inline-flex;align-items:center;justify-content:center;background:#f2f2f6;color:#6b7280}
  .badge-soft{padding:.25rem .5rem;border-radius:999px;font-size:.72rem}
  .badge-soft.purple{background:rgba(124,58,237,.12);color:#7c3aed}
  .table-holidays td,.table-holidays th{vertical-align:middle}
  /* Make Employees Status wide (span up to Attendance Overview) */
  @media (min-width: 1400px){
    .employees-status-wide{grid-column: 1 / span 9;}
  }
</style>
<?php
$styles = ob_get_clean();

/* ============================== VIEW: $content ============================== */
ob_start();
?>
	
                    <!-- Start::page-header -->
                    <div class="page-header-breadcrumb mb-3">
                        <div class="d-flex align-center justify-content-between flex-wrap">
                            <h1 class="page-title fw-medium fs-18 mb-0">HRM</h1>
                            <ol class="breadcrumb mb-0">
                                <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
                                <li class="breadcrumb-item active" aria-current="page">HRM</li>
                            </ol>
                        </div>
                    </div>
                    <!-- End::page-header -->

                    <!-- Start:: row-1 -->
                    <div class="row">
                        <!-- KPIs (left 4/12) -->
                        <div class="col-xxl-4 col-xl-6">
                            <div class="row">
                                <div class="col-xl-12">
                                    <div class="card custom-card overflow-hidden">
                                        <div class="card-body p-0">
                                            <div class="row g-0">
                                                <div class="col-xl-6">
                                                    <div class="card custom-card dashboard-main-card border-0 shadow-none border-lg-end border-bottom mb-0 rounded-0">
                                                        <div class="card-body">
                                                            <div class="mb-3">
                                                                <span class="avatar avatar-lg avatar-rounded bg-primary-transparent svg-primary">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" enable-background="new 0 0 24 24" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><g><rect fill="none" height="24" width="24"/></g><g><path d="M20,7h-5V4c0-1.1-0.9-2-2-2h-2C9.9,2,9,2.9,9,4v3H4C2.9,7,2,7.9,2,9v11c0,1.1,0.9,2,2,2h16c1.1,0,2-0.9,2-2V9 C22,7.9,21.1,7,20,7z M9,12c0.83,0,1.5,0.67,1.5,1.5S9.83,15,9,15s-1.5-0.67-1.5-1.5S8.17,12,9,12z M12,18H6v-0.75c0-1,2-1.5,3-1.5 s3,0.5,3,1.5V18z M13,9h-2V4h2V9z M18,16.5h-4V15h4V16.5z M18,13.5h-4V12h4V13.5z"/></g></svg>
                                                                </span>
                                                            </div>
                                                            <div class="text-muted mb-2">Total Employees</div>
                                                            <div class="d-flex align-items-end gap-2 flex-wrap">
                                                                <h5 class="fw-semibold mb-0 lh-1"><?php echo fmt_num($totalEmployees); ?></h5>
                                                                <div>
                                                                    <span class="badge bg-success-transparent rounded-pill"><i class="ti ti-arrow-up me-1"></i>0.00%</span>
                                                                    <span class="text-muted fs-12">This Year</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-xl-6">
                                                    <div class="card custom-card dashboard-main-card border-0 shadow-none border-bottom mb-0 rounded-0">
                                                        <div class="card-body">
                                                            <div class="mb-3">
                                                                <span class="avatar avatar-lg avatar-rounded bg-secondary-transparent svg-secondary">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><path d="M0 0h24v24H0z" fill="none"/><path d="M15 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm-9-2V7H4v3H1v2h3v3h2v-3h3v-2H6zm9 4c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>
                                                                </span>
                                                            </div>
                                                            <div class="text-muted mb-2">New Employees</div>
                                                            <div class="d-flex align-items-end gap-2 flex-wrap">
                                                                <h5 class="fw-semibold mb-0 lh-1"><?php echo fmt_num($newEmployeesThisMonth); ?></h5>
                                                                <div>
                                                                    <span class="badge bg-danger-transparent rounded-pill"><i class="ti ti-arrow-down me-1"></i>0.00%</span>
                                                                    <span class="text-muted fs-12">This Month</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-xl-6">
                                                    <div class="card custom-card dashboard-main-card dashboard-main-border border-0 shadow-none border-lg-end mb-0 rounded-0">
                                                        <div class="card-body">
                                                            <div class="mb-3">
                                                                <span class="avatar avatar-lg avatar-rounded bg-success-transparent svg-success">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><path d="M0 0h24v24H0zm10 5h4v2h-4zm0 0h4v2h-4z" fill="none"/><path d="M10 16v-1H3.01L3 19c0 1.11.89 2 2 2h14c1.11 0 2-.89 2-2v-4h-7v1h-4zm10-9h-4.01V5l-2-2h-4l-2 2v2H4c-1.1 0-2 .9-2 2v3c0 1.11.89 2 2 2h6v-2h4v2h6c1.1 0 2-.9 2-2V9c0-1.1-.9-2-2-2zm-6 0h-4V5h4v2z"/></svg>
                                                                </span>
                                                            </div>
                                                            <div class="text-muted mb-2">Total Job Applicants</div>
                                                            <div class="d-flex align-items-end gap-2 flex-wrap">
                                                                <h5 class="fw-semibold mb-0 lh-1"><?php echo fmt_num($totalApplicants); ?></h5>
                                                                <div>
                                                                    <span class="badge bg-success-transparent rounded-pill"><i class="ti ti-arrow-up me-1"></i>0.00%</span>
                                                                    <span class="text-muted fs-12">This Year</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-xl-6">
                                                    <div class="card custom-card dashboard-main-card border-0 shadow-none mb-0 rounded-0">
                                                        <div class="card-body">
                                                            <div class="mb-3">
                                                                <span class="avatar avatar-lg avatar-rounded bg-warning-transparent svg-warning">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" enable-background="new 0 0 24 24" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><g><rect fill="none" height="24" width="24"/></g><g><path d="M14,8c0-2.21-1.79-4-4-4S6,5.79,6,8s1.79,4,4,4S14,10.21,14,8z M17,10v2h6v-2H17z M2,18v2h16v-2c0-2.66-5.33-4-8-4 S2,15.34,2,18z"/></g></svg>
                                                                </span>
                                                            </div>
                                                            <div class="text-muted mb-2">Total Joined</div>
                                                            <div class="d-flex align-items-end gap-2 flex-wrap">
                                                                <h5 class="fw-semibold mb-0 lh-1"><?php echo fmt_num($totalJoinedThisMonth); ?></h5>
                                                                <div>
                                                                    <span class="badge bg-success-transparent rounded-pill"><i class="ti ti-arrow-up me-1"></i>0.00%</span>
                                                                    <span class="text-muted fs-12">This Month</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div><!-- row g-0 -->
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Candidate Statistics (middle 5/12) -->
                        <div class="col-xxl-5 col-xl-6">
                            <div class="card custom-card">
                                <div class="card-header">
                                    <div class="card-title">Candidate Statistics</div>
                                </div>
                                <div class="card-body p-0">
                                    <div class="row g-0 border-bottom border-block-end-dashed">
                                        <div class="col-xl-6 border-md-end">
                                            <div class="text-center p-3">
                                                <span class="d-block text-muted mb-1">Total Candidates Hired</span>
                                                <h5 class="fw-semibold mb-0"><?php echo fmt_num($totalHiredAllTime); ?></h5>
                                            </div>
                                        </div>
                                        <div class="col-xl-6">
                                            <div class="text-center p-3">
                                                <span class="d-block text-muted mb-1">Total Responses</span>
                                                <h5 class="fw-semibold mb-0"><?php echo fmt_num($totalResponsesAllTime); ?></h5>
                                            </div>
                                        </div>
                                    </div>
                                    <div id="candidate-statistics" class="p-3"
                                         data-hired="<?php echo h(json_encode(array_values($seriesHired))); ?>"
                                         data-resp="<?php echo h(json_encode(array_values($seriesResponses))); ?>"></div>
                                </div>
                            </div>
                        </div>

                        <!-- Attendance (right 3/12) -->
                        <div class="col-xxl-3">
                            <div class="card custom-card">
                                <div class="card-header">
                                    <div class="card-title">Attendance Overview</div>
                                </div>
                                <div class="card-body">
                                    <div id="attendance-overview"
                                         data-series="<?php echo h(json_encode($att_series_pct)); ?>"
                                         data-total="<?php echo (int)$att_total; ?>"></div>
                                    <ul class="list-unstyled my-4 hrm-attendance-overview-list">
                                        <li><div class="d-flex align-items-center justify-content-between gap-2"><div class="attendance-type present">Present</div><div class="fw-semibold"><?php echo fmt_num($att_present); ?></div></div></li>
                                        <li><div class="d-flex align-items-center justify-content-between gap-2"><div class="attendance-type late">Late</div><div class="fw-semibold"><?php echo fmt_num($att_late); ?></div></div></li>
                                        <li><div class="d-flex align-items-center justify-content-between gap-2"><div class="attendance-type permission">SRT</div><div class="fw-semibold"><?php echo fmt_num($att_srt); ?></div></div></li>
                                        <li><div class="d-flex align-items-center justify-content-between gap-2"><div class="attendance-type absent">Early</div><div class="fw-semibold"><?php echo fmt_num($att_early); ?></div></div></li>
                                        <li><div class="d-flex align-items-center justify-content-between gap-2"><div class="attendance-type absent">Absent</div><div class="fw-semibold"><?php echo fmt_num($att_absent); ?></div></div></li>
                                    </ul>
                                    <div class="d-grid"><button class="btn btn-light btn-lg">Total <?php echo fmt_num($att_total); ?> <i class="ti ti-arrow-narrow-right ms-1"></i></button></div>
                                </div>
                            </div>
                        </div>

                       <!-- Employees Status WIDE (full row) -->
                        <div class="col-xxl-12 mt-3">
                            <div class="card custom-card">
                                <div class="card-body">
                                    <h6 class="fw-semibold mb-3">Employees Status</h6>
                                    <div class="progress-stacked progress-xl mb-3">
                                        <div class="progress-bar" role="progressbar" style="width: <?php echo (int)$desigPct[0]; ?>%" aria-valuenow="<?php echo (int)$desigPct[0]; ?>" aria-valuemin="0" aria-valuemax="100"><?php echo (int)$desigPct[0]; ?>%</div>
                                        <div class="progress-bar bg-success" role="progressbar" style="width: <?php echo (int)$desigPct[1]; ?>%" aria-valuenow="<?php echo (int)$desigPct[1]; ?>" aria-valuemin="0" aria-valuemax="100"><?php echo (int)$desigPct[1]; ?>%</div>
                                        <div class="progress-bar bg-warning" role="progressbar" style="width: <?php echo (int)$desigPct[2]; ?>%" aria-valuenow="<?php echo (int)$desigPct[2]; ?>" aria-valuemin="0" aria-valuemax="100"><?php echo (int)$desigPct[2]; ?>%</div>
                                        <div class="progress-bar bg-danger" role="progressbar" style="width: <?php echo (int)$desigPct[3]; ?>%" aria-valuenow="<?php echo (int)$desigPct[3]; ?>" aria-valuemin="0" aria-valuemax="100"><?php echo (int)$desigPct[3]; ?>%</div>
                                    </div>
                                    <div class="row gy-2">
                                        <div class="col-xl-3"><div class="d-flex align-items-center gap-4 flex-wrap"><div class="employee-status-marker remote"><?php echo h($desigBuckets[0]['label']); ?> :</div><div class="fw-semibold"><?php echo fmt_num($desigBuckets[0]['count']); ?></div></div></div>
                                        <div class="col-xl-3"><div class="d-flex align-items-center gap-4 flex-wrap"><div class="employee-status-marker probation"><?php echo h($desigBuckets[1]['label']); ?> :</div><div class="fw-semibold"><?php echo fmt_num($desigBuckets[1]['count']); ?></div></div></div>
                                        <div class="col-xl-3"><div class="d-flex align-items-center gap-4 flex-wrap"><div class="employee-status-marker contract"><?php echo h($desigBuckets[2]['label']); ?> :</div><div class="fw-semibold"><?php echo fmt_num($desigBuckets[2]['count']); ?></div></div></div>
                                        <div class="col-xl-3"><div class="d-flex align-items-center gap-4 flex-wrap"><div class="employee-status-marker work-home"><?php echo h($desigBuckets[3]['label']); ?> :</div><div class="fw-semibold"><?php echo fmt_num($desigBuckets[3]['count']); ?></div></div></div>
                                    </div>
                                    <div class="mt-3 fw-semibold"><?php echo fmt_num($totalForPct); ?></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- End:: row-1 -->

                    <!-- Start:: row-2 -->
                    <div class="row">
                        <div class="col-xxl-4">
                            <div class="card custom-card">
                                <div class="card-header"><div class="card-title">Employees By Department</div></div>
                                <div class="card-body">
                                    <div id="employee-department"
                                         data-labels="<?php echo h(json_encode($deptLabels)); ?>"
                                         data-values="<?php echo h(json_encode($deptValues)); ?>"></div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xxl-4">
                            <div class="card custom-card">
                                <div class="card-header justify-content-between">
                                    <div class="card-title">Employees List</div>
                                    <a href="javascript:void(0);" class="text-muted fs-13">View All<i class="ti ti-arrow-narrow-right ms-1"></i></a>
                                </div>
                                <div class="card-body">
                                    <ul class="list-unstyled hrm-employee-list">
                                        <?php $facePool=[1,12,5,15,6,9,10,11,8,4,2,3]; $i=0; foreach($recentJoins as $ej): $img=$facePool[$i%count($facePool)]; $i++; ?>
                                        <li>
                                            <div class="d-flex align-items-center gap-2 flex-wrap">
                                                <div class="lh-1">
                                                    <span class="avatar avatar-md avatar-rounded">
                                                        <img src="<?php echo $baseUrl; ?>/assets/images/faces/<?php echo (int)$img; ?>.jpg" alt="">
                                                    </span>
                                                </div>
                                                <div class="flex-fill">
                                                    <span class="d-block fw-semibold">
                                                        <?php echo h($ej['name']); ?>
                                                        <span class="badge <?php echo dept_badge_class($ej['dept']); ?> ms-2"><?php echo h($ej['dept']); ?></span>
                                                    </span>
                                                    <span class="text-muted fs-13"><?php echo h($ej['role']); ?></span>
                                                </div>
                                                <div class="text-end">
                                                    <span class="fw-medium"><?php echo h($ej['date']); ?></span>
                                                    <span class="d-block fs-12 mt-1 text-muted"><?php echo ($ej['status']!=='') ? h($ej['status']) : 'Joined'; ?></span>
                                                </div>
                                            </div>
                                        </li>
                                        <?php endforeach; if(empty($recentJoins)): ?>
                                            <li class="text-muted">No recent employees found.</li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <div class="col-xxl-4">
                            <div class="card custom-card overflow-hidden">
                                <div class="card-header justify-content-between">
                                    <div class="card-title">Indian Corporate Holidays</div>
                                </div>
                                <div class="card-body p-0">
                                    <div class="table-responsive">
                                        <table class="table text-nowrap table-hover table-holidays">
                                            <thead><tr><th scope="col">Date</th><th scope="col">Day</th><th scope="col">Holiday</th></tr></thead>
                                            <tbody>
                                                <?php foreach($holidays as $h): $day = date('l', strtotime($h['date'])); ?>
                                                <tr>
                                                    <td><?php echo safe_date($h['date']); ?></td>
                                                    <td><span class="badge rounded-pill <?php echo day_badge_class($h['date']); ?>"><?php echo $day; ?></span></td>
                                                    <td><?php echo h($h['name']); ?></td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- End:: row-2 -->

                    <!-- Start:: row-3 -->
                    <div class="row">
                        <div class="col-xl-12">
                            <div class="card custom-card">
                                <div class="card-header justify-content-between">
                                    <div class="card-title">Applicant Details</div>
                                    <div class="d-flex flex-wrap gap-2"> 
                                        <div><input class="form-control form-control-sm" type="text" placeholder="Search Here" aria-label=".form-control-sm example"></div> 
                                        <div class="dropdown"> 
                                            <a href="javascript:void(0);" class="btn btn-primary btn-sm btn-wave waves-effect waves-light" data-bs-toggle="dropdown" aria-expanded="false"> Sort By<i class="ri-arrow-down-s-line align-middle ms-1 d-inline-block"></i></a> 
                                            <ul class="dropdown-menu" role="menu"> 
                                                <li><a class="dropdown-item" href="javascript:void(0);">New</a></li> 
                                                <li><a class="dropdown-item" href="javascript:void(0);">Popular</a></li> 
                                                <li><a class="dropdown-item" href="javascript:void(0);">Relevant</a></li> 
                                            </ul> 
                                        </div> 
                                    </div>
                                </div>
                                <div class="card-body p-0">
                                    <div class="table-responsive">
                                        <table class="table text-nowrap">
                                            <thead>
                                                <tr>
                                                    <th scope="row"><input class="form-check-input" type="checkbox" aria-label="..."></th>
                                                    <th scope="col">Application ID</th>
                                                    <th scope="col">Applicant Name</th>
                                                    <th scope="col">Position Applied</th>
                                                    <th scope="col">Date Of Application</th>
                                                    <th scope="col">Email</th>
                                                    <th scope="col">Work Experience</th>
                                                    <th scope="col">Status</th>
                                                    <th scope="col">Action</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php
                                                $appRows = [];
                                                $r8 = dbq($conn, "SELECT id, first_name, surname, title, created_at, email, experience_details, application_status
                                                                  FROM broriserin_onb.employee_applications
                                                                  ORDER BY COALESCE(updated_at, created_at) DESC
                                                                  LIMIT 5");
                                                if ($r8){ while($row = mysqli_fetch_assoc($r8)){
                                                    $appRows[] = [
                                                        'id'     => (int)$row['id'],
                                                        'name'   => trim(($row['first_name'] ?? '').' '.($row['surname'] ?? '')) ?: '—',
                                                        'title'  => $row['title'] ?: '—',
                                                        'date'   => safe_date($row['created_at'] ?? ''),
                                                        'email'  => $row['email'] ?: '—',
                                                        'exp'    => $row['experience_details'] ?: 'Fresher',
                                                        'status' => strtolower($row['application_status'] ?? '')
                                                    ];
                                                }}
                                                $badgeMap=['new'=>'bg-primary-transparent','under review'=>'bg-secondary-transparent','interviewed'=>'bg-success-transparent','hired'=>'bg-info-transparent','approved'=>'bg-info-transparent','rejected'=>'bg-danger-transparent','in-progress'=>'bg-warning-transparent'];
                                                if(empty($appRows)): ?>
                                                    <tr><td colspan="9" class="text-muted">No applications found.</td></tr>
                                                <?php else:
                                                    $idx=0; foreach($appRows as $ar):
                                                    $bClass = $badgeMap[$ar['status']] ?? 'bg-primary-transparent';
                                                    $face = ($idx%12)+1; $idx++;
                                                ?>
                                                <tr>
                                                    <th scope="row"><input class="form-check-input" type="checkbox" aria-label="..." <?php echo ($idx%2==0?'checked':''); ?>></th>
                                                    <td>#APP-<?php echo (int)$ar['id']; ?></td>
                                                    <td>
                                                        <div class="d-flex align-items-center">
                                                            <span class="avatar avatar-sm me-2 avatar-rounded">
                                                                <img src="<?php echo $baseUrl; ?>/assets/images/faces/<?php echo $face; ?>.jpg" alt="img">
                                                            </span><?php echo h($ar['name']); ?>
                                                        </div>
                                                    </td>
                                                    <td><?php echo h($ar['title']); ?></td>
                                                    <td><?php echo h($ar['date']); ?></td>
                                                    <td><?php echo h($ar['email']); ?></td>
                                                    <td><?php echo h($ar['exp']); ?></td>
                                                    <td><span class="badge <?php echo $bClass; ?>"><?php echo $ar['status']!=='' ? h(ucwords($ar['status'])) : 'New'; ?></span></td>
                                                    <td>
                                                        <div class="hstack gap-2 fs-15">
                                                            <a aria-label="call" href="javascript:void(0);" class="btn btn-icon waves-effect waves-light btn-sm btn-success-light rounded-circle"><i class="ri-phone-line"></i></a>
                                                            <a aria-label="edit" href="javascript:void(0);" class="btn btn-icon waves-effect waves-light btn-sm btn-primary-light rounded-circle"><i class="ri-edit-line"></i></a>
                                                            <a aria-label="delete" href="javascript:void(0);" class="btn btn-icon btn-wave waves-effect waves-light btn-sm btn-danger-light rounded-circle"><i class="ri-delete-bin-line"></i></a>
                                                        </div>
                                                    </td>
                                                </tr>
                                                <?php endforeach; endif; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                                <div class="card-footer border-top-0">
                                    <div class="d-flex align-items-center">
                                        <div> Showing <?php echo isset($appRows)?count($appRows):0; ?> Entries <i class="bi bi-arrow-right ms-2 fw-semibold"></i></div>
                                        <div class="ms-auto">
                                            <nav aria-label="Page navigation" class="pagination-style-2">
                                                <ul class="pagination mb-0 flex-wrap">
                                                    <li class="page-item disabled"><a class="page-link" href="javascript:void(0);">Prev</a></li>
                                                    <li class="page-item active"><a class="page-link" href="javascript:void(0);">1</a></li>
                                                    <li class="page-item"><a class="page-link" href="javascript:void(0);">2</a></li>
                                                    <li class="page-item"><a class="page-link" href="javascript:void(0);"><i class="bi bi-three-dots"></i></a></li>
                                                    <li class="page-item"><a class="page-link" href="javascript:void(0);">17</a></li>
                                                    <li class="page-item"><a class="page-link text-primary" href="javascript:void(0);">next</a></li>
                                                </ul>
                                            </nav>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- End:: row-3 -->

<?php $content = ob_get_clean(); ?>

<!-- ============================== VIEW: $scripts ============================== -->
<?php ob_start(); ?>

        <!-- Apex Charts JS -->
        <script src="<?php echo $baseUrl; ?>/assets/libs/apexcharts/apexcharts.min.js"></script>

        <!-- HRM Dashboard (charts in-page to ensure exact style) -->
        <script>
        (function(){
          /* ---------- Candidate Statistics (grouped columns, Jan→Dec) ---------- */
          var csEl = document.querySelector('#candidate-statistics');
          if(csEl && window.ApexCharts){
            var hired = JSON.parse(csEl.getAttribute('data-hired')||'[]');
            var resp  = JSON.parse(csEl.getAttribute('data-resp')||'[]');
            var options = {
              chart:{ type:'bar', height:320, toolbar:{show:false} },
              series:[
                { name:'Candidates Hired', data:hired },
                { name:'Received Responses', data:resp }
              ],
              plotOptions:{ bar:{ columnWidth:'38%', borderRadius:6 } },
              dataLabels:{ enabled:false },
              legend:{ position:'bottom' },
              xaxis:{ categories:['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'] },
              yaxis:{ labels:{ formatter:function(v){return v;} } },
              grid:{ strokeDashArray:3 },
              colors:['#6d28d9','#c4b5fd'] /* dark + light purple like your design */
            };
            new ApexCharts(csEl, options).render();
          }

          /* ---------- Attendance Overview (semi-donut radial with “Total”) ---------- */
          var atEl = document.querySelector('#attendance-overview');
          if(atEl && window.ApexCharts){
            var ser = JSON.parse(atEl.getAttribute('data-series')||'[0,0,0,0,0]');
            var total = parseInt(atEl.getAttribute('data-total') || '0', 10);
            var options2 = {
              chart:{ type:'radialBar', height:260, sparkline:{enabled:true} },
              series: ser,
              labels:['Present','Late','SRT','Early','Absent'],
              colors:['#6d28d9','#22c55e','#f59e0b','#ef4444','#8b5cf6'],
              plotOptions:{
                radialBar:{
                  startAngle:-90, endAngle:90,
                  hollow:{ margin:5, size:'58%', background:'transparent' },
                  track:{ background:'#f3f4f6' },
                  dataLabels:{
                    name:{ show:false },
                    value:{ show:false },
                    total:{
                      show:true,
                      label:'Total',
                      formatter:function(){ return total.toString(); },
                      color:'#0f172a',
                      fontSize:'16px',
                    }
                  }
                }
              },
              legend:{ show:false }
            };
            new ApexCharts(atEl, options2).render();
          }

          /* ---------- Employees By Department (horizontal purple bars) ---------- */
          var deptEl=document.getElementById('employee-department');
          if(deptEl && window.ApexCharts){
            var labels=[],values=[];
            try{labels=JSON.parse(deptEl.getAttribute('data-labels')||'[]');values=JSON.parse(deptEl.getAttribute('data-values')||'[]');}catch(e){}
            new ApexCharts(deptEl,{
              chart:{ type:'bar', height:320, toolbar:{show:false} },
              series:[{ name:'Employees', data:values }],
              plotOptions:{ bar:{ horizontal:true, barHeight:'60%', borderRadius:6 } },
              dataLabels:{ enabled:false },
              xaxis:{ categories:labels },
              grid:{ strokeDashArray:3 },
              colors:['#7c3aed']
            }).render();
          }
        })();
        </script>

<?php $scripts = ob_get_clean(); ?>

<!-- ============================== RENDER BASE ============================== -->
<?php include 'layouts/base.php'; ?>
