<?php
/* =============================================================================
 * File: unit_save.php
 * Purpose: Insert/Update Units (no UI). Redirects back to units.php with flash
 * Stack  : Core PHP + MySQLi (NO PDO). DB in /broker-config.php. Collation: utf8mb4_general_ci
 * Depends: functions.php for post(), compute_total_price(), redirect()
 * ============================================================================= */

@session_start();
require_once 'broker-config.php';
require_once __DIR__.'/functions.php';

if (!isset($conn) || !($conn instanceof mysqli)) { $_SESSION['flash_error']='DB connection missing.'; redirect('units.php'); }

/* ---- Charset/Collation ---- */
mysqli_set_charset($conn,'utf8mb4');
@mysqli_query($conn,"SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn,"SET collation_connection='utf8mb4_general_ci'");

/* --------- Read & normalize inputs (no PDO) --------- */
$id             = intval(post('id',0));
$project_id     = intval(post('project_id',0));
$tower_id_raw   = post('tower_id','');
$tower_id_is_null = ($tower_id_raw === '');
if(!$tower_id_is_null){ $tower_id = intval($tower_id_raw); }

$unit_no        = post('unit_no','');
$floor          = intval(post('floor',0));
$unit_type      = post('unit_type','');

$saleable_area  = floatval(post('saleable_area',0));
$carpet_area    = (post('carpet_area','')==='') ? null : floatval(post('carpet_area',0));
$plot_area      = (post('plot_area','')==='')   ? null : floatval(post('plot_area',0));

$facing         = post('facing','');
$aspect         = post('aspect','NA');
$furnishing     = post('furnishing','Bare');

$bedrooms       = (post('bedrooms','')==='')  ? null : intval(post('bedrooms',0));
$bathrooms      = (post('bathrooms','')==='') ? null : intval(post('bathrooms',0));
$balconies      = (post('balconies','')==='') ? null : intval(post('balconies',0));

$parking_slots  = intval(post('parking_slots',0));
$base_rate      = floatval(post('base_rate',0));
$plc_amount     = floatval(post('plc_amount',0));
$other_charges  = floatval(post('other_charges',0));
$gst_rate       = floatval(post('gst_rate',0));

$payment_plan   = post('payment_plan','');
$remarks        = post('remarks','');

/* --------- Compute price --------- */
$total_price = compute_total_price($saleable_area,$base_rate,$plc_amount,$other_charges,$gst_rate);

/*
  NOTE: Bind as strings to simplify nullable handling; MySQL will cast appropriately.
*/

/* --------- UPDATE --------- */
if ($id) {
  $sql = "UPDATE units SET
            project_id=?,
            unit_no=?,
            floor=?,
            unit_type=?,
            saleable_area=?,
            carpet_area=?,
            plot_area=?,
            facing=?,
            aspect=?,
            furnishing=?,
            bedrooms=?,
            bathrooms=?,
            balconies=?,
            parking_slots=?,
            base_rate=?,
            plc_amount=?,
            other_charges=?,
            gst_rate=?,
            payment_plan=?,
            total_price=?,
            remarks=?";
  if ($tower_id_is_null) { $sql .= ", tower_id=NULL"; } else { $sql .= ", tower_id=?"; }
  $sql .= " WHERE id=?";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { error_log("unit_save UPDATE prepare error: ".mysqli_error($conn)); $_SESSION['flash_error']='Failed to prepare update.'; redirect('units.php'); }

  if ($tower_id_is_null) {
    mysqli_stmt_bind_param($stmt, str_repeat('s', 22),
      $project_id, $unit_no, $floor, $unit_type,
      $saleable_area, $carpet_area, $plot_area,
      $facing, $aspect, $furnishing,
      $bedrooms, $bathrooms, $balconies,
      $parking_slots, $base_rate, $plc_amount, $other_charges, $gst_rate,
      $payment_plan, $total_price, $remarks,
      $id
    );
  } else {
    mysqli_stmt_bind_param($stmt, str_repeat('s', 23),
      $project_id, $unit_no, $floor, $unit_type,
      $saleable_area, $carpet_area, $plot_area,
      $facing, $aspect, $furnishing,
      $bedrooms, $bathrooms, $balconies,
      $parking_slots, $base_rate, $plc_amount, $other_charges, $gst_rate,
      $payment_plan, $total_price, $remarks,
      $tower_id, $id
    );
  }

  if (!mysqli_stmt_execute($stmt)) { $_SESSION['flash_error']='Update failed: '.mysqli_error($conn); }
  else { $_SESSION['flash_success']='Unit updated successfully.'; }
  mysqli_stmt_close($stmt);

/* --------- INSERT --------- */
} else {
  $columns = "project_id, unit_no, floor, unit_type, saleable_area, carpet_area, plot_area, facing, aspect, furnishing, bedrooms, bathrooms, balconies, parking_slots, base_rate, plc_amount, other_charges, gst_rate, payment_plan, total_price, remarks";
  $place   = "?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?";

  if ($tower_id_is_null) {
    $sql = "INSERT INTO units ($columns, tower_id) VALUES ($place, NULL)";
    $stmt = mysqli_prepare($conn, $sql);
    if (!$stmt) { error_log("unit_save INSERT(NULL tower) prepare error: ".mysqli_error($conn)); $_SESSION['flash_error']='Failed to prepare insert.'; redirect('units.php'); }
    mysqli_stmt_bind_param($stmt, str_repeat('s', 21),
      $project_id, $unit_no, $floor, $unit_type,
      $saleable_area, $carpet_area, $plot_area,
      $facing, $aspect, $furnishing,
      $bedrooms, $bathrooms, $balconies,
      $parking_slots, $base_rate, $plc_amount, $other_charges, $gst_rate,
      $payment_plan, $total_price, $remarks
    );
  } else {
    $columns .= ", tower_id"; $place .= ",?";
    $sql = "INSERT INTO units ($columns) VALUES ($place)";
    $stmt = mysqli_prepare($conn, $sql);
    if (!$stmt) { error_log("unit_save INSERT(+tower) prepare error: ".mysqli_error($conn)); $_SESSION['flash_error']='Failed to prepare insert.'; redirect('units.php'); }
    mysqli_stmt_bind_param($stmt, str_repeat('s', 22),
      $project_id, $unit_no, $floor, $unit_type,
      $saleable_area, $carpet_area, $plot_area,
      $facing, $aspect, $furnishing,
      $bedrooms, $bathrooms, $balconies,
      $parking_slots, $base_rate, $plc_amount, $other_charges, $gst_rate,
      $payment_plan, $total_price, $remarks, $tower_id
    );
  }

  if (!mysqli_stmt_execute($stmt)) { $_SESSION['flash_error']='Insert failed: '.mysqli_error($conn); }
  else { $_SESSION['flash_success']='Unit created successfully.'; }
  mysqli_stmt_close($stmt);
}

/* --------- Done --------- */
redirect('units.php');
