<?php
/* ============================================================================
 * File: users.php
 * Purpose: Manage Users (create / edit / delete / change password) + assign roles
 * Stack : Core PHP + MySQLi (NO PDO). DB connection lives in ../config.php
 * Rules : Collation utf8mb4_general_ci
 * Notes : - Uses the RBAC tables from roles.php (roles, permissions, role_permissions, user_roles)
 *         - Superuser fallback: if $_SESSION['user_id1'] is empty → allow access (so you can bootstrap)
 *         - Passwords are bcrypt-hashed with password_hash()
 *         - Safe prepared statements for writes
 * ============================================================================ */

date_default_timezone_set('Asia/Kolkata');
@session_start();
require_once __DIR__ . '/../config.php'; // DB connection must be here (NO PDO)
require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);
/* ---- Charset & collation enforcement ---- */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* ---------- Helpers ---------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function col_exists($conn,$table,$col){
  $t = mysqli_real_escape_string($conn,$table);
  $c = mysqli_real_escape_string($conn,$col);
  $q = mysqli_query($conn,"SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='{$t}' AND COLUMN_NAME='{$c}'");
  return ($q && mysqli_num_rows($q)>0);
}
function table_exists($conn,$table){
  $t = mysqli_real_escape_string($conn,$table);
  $q = mysqli_query($conn,"SELECT 1 FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA=DATABASE() AND TABLE_NAME='{$t}'");
  return ($q && mysqli_num_rows($q)>0);
}

/* ---------- RBAC (same model as roles.php) ---------- */
$CURRENT_USER_ID = (int)($_SESSION['user_id1'] ?? 0);

function rbac_ensure_schema($conn){
  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `roles` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `name` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
    `description` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `uniq_role_name` (`name`)
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `permissions` (
    `code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
    `label` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
    `group_name` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
    PRIMARY KEY (`code`)
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `role_permissions` (
    `role_id` INT UNSIGNED NOT NULL,
    `perm_code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
    PRIMARY KEY (`role_id`,`perm_code`),
    KEY `idx_perm_code` (`perm_code`),
    CONSTRAINT `fk_role_permissions_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `user_roles` (
    `user_id` INT NOT NULL,
    `role_id` INT UNSIGNED NOT NULL,
    PRIMARY KEY (`user_id`,`role_id`),
    KEY `idx_role_id` (`role_id`),
    CONSTRAINT `fk_user_roles_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
}
function rbac_seed_permissions($conn){
  $perms = [
    ['leads.create','Create leads','Leads'],
    ['leads.export','Export leads (CSV)','Leads'],
    ['activities.log','Log activities (call/mail/task/meeting)','Activities'],
    ['roles.manage','Manage roles & permissions','Admin'],
    ['users.manage','Manage users','Admin'],
  ];
  foreach($perms as $p){
    $code = mysqli_real_escape_string($conn,$p[0]);
    $label= mysqli_real_escape_string($conn,$p[1]);
    $grp  = mysqli_real_escape_string($conn,$p[2]);
    @mysqli_query($conn, "INSERT IGNORE INTO `permissions` (`code`,`label`,`group_name`) VALUES ('{$code}','{$label}','{$grp}')");
  }
}
function rbac_has($conn,$uid,$code){
  if(!$uid){ return true; } // superuser fallback if no auth wired yet
  $uid  = (int)$uid;
  $code = mysqli_real_escape_string($conn,$code);
  $q = mysqli_query($conn,"SELECT 1
                           FROM user_roles ur
                           JOIN role_permissions rp ON rp.role_id = ur.role_id
                           WHERE ur.user_id={$uid} AND rp.perm_code='{$code}'
                           LIMIT 1");
  return ($q && mysqli_num_rows($q)>0);
}
rbac_ensure_schema($conn);
rbac_seed_permissions($conn);
$CAN_MANAGE_USERS = rbac_has($conn,$CURRENT_USER_ID,'users.manage');

/* ---------- Gate: only admins (or superuser fallback) ---------- */
if(!$CAN_MANAGE_USERS){
  http_response_code(403);
  echo '<!doctype html><html><head><meta charset="utf-8"><title>Forbidden</title></head><body style="font-family:system-ui;padding:24px"><h2>403 — Not authorized</h2><p>You do not have permission to manage users.</p></body></html>';
  exit;
}

/* ---------- Ensure `users` table ---------- */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `users` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `username` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
  `full_name` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `email` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `phone` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `password_hash` VARCHAR(255) COLLATE utf8mb4_general_ci NOT NULL,
  `is_active` TINYINT(1) NOT NULL DEFAULT 1,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uniq_username` (`username`),
  UNIQUE KEY `uniq_email` (`email`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

/* ---------- Load roles for UI ---------- */
$all_roles = [];
$rr = mysqli_query($conn,"SELECT id,name FROM roles ORDER BY name");
if($rr){ while($r=mysqli_fetch_assoc($rr)){ $all_roles[]=$r; } }

/* ---------- Actions ---------- */
$flash = '';
if($_SERVER['REQUEST_METHOD']==='POST'){
  $action = $_POST['action'] ?? '';

  /* ---- Create user ---- */
  if($action==='create_user'){
    $username  = trim($_POST['username'] ?? '');
    $full_name = trim($_POST['full_name'] ?? '');
    $email     = trim($_POST['email'] ?? '');
    $phone     = trim($_POST['phone'] ?? '');
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    $pwd1      = (string)($_POST['password'] ?? '');
    $pwd2      = (string)($_POST['password2'] ?? '');
    $rolesSel  = isset($_POST['roles']) ? (array)$_POST['roles'] : [];

    if($username==='' || $pwd1===''){
      $flash = 'Username and password are required.';
    } elseif($pwd1 !== $pwd2){
      $flash = 'Passwords do not match.';
    } else {
      $hash = password_hash($pwd1, PASSWORD_BCRYPT);
      $stmt = mysqli_prepare($conn,"INSERT INTO `users` (`username`,`full_name`,`email`,`phone`,`password_hash`,`is_active`) VALUES (?,?,?,?,?,?)");
      mysqli_stmt_bind_param($stmt,'sssssi',$username,$full_name,$email,$phone,$hash,$is_active);
      $ok = @mysqli_stmt_execute($stmt);
      $uid = $ok ? mysqli_insert_id($conn) : 0;
      $err = mysqli_error($conn);
      mysqli_stmt_close($stmt);

      if($ok && $uid>0){
        // assign roles
        if($rolesSel){
          $vals = [];
          foreach($rolesSel as $rid){
            $rid = (int)$rid;
            if($rid>0) $vals[] = "({$uid},{$rid})";
          }
          if($vals){
            @mysqli_query($conn,"INSERT IGNORE INTO `user_roles` (`user_id`,`role_id`) VALUES ".implode(',',$vals));
          }
        }
        $flash = 'User created.';
      } else {
        $flash = 'Error creating user. '.($err ? 'DB: '.$err : '');
      }
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash));
    exit;
  }

  /* ---- Update user (profile + roles) ---- */
  if($action==='update_user'){
    $id        = (int)($_POST['id'] ?? 0);
    $username  = trim($_POST['username'] ?? '');
    $full_name = trim($_POST['full_name'] ?? '');
    $email     = trim($_POST['email'] ?? '');
    $phone     = trim($_POST['phone'] ?? '');
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    $rolesSel  = isset($_POST['roles']) ? (array)$_POST['roles'] : [];

    if($id<=0 || $username===''){
      $flash = 'Invalid data.';
    } else {
      $stmt = mysqli_prepare($conn,"UPDATE `users` SET `username`=?,`full_name`=?,`email`=?,`phone`=?,`is_active`=? WHERE `id`=?");
      mysqli_stmt_bind_param($stmt,'ssssii',$username,$full_name,$email,$phone,$is_active,$id);
      $ok = @mysqli_stmt_execute($stmt);
      $err = mysqli_error($conn);
      mysqli_stmt_close($stmt);

      // update roles mapping (reset then insert)
      @mysqli_query($conn,"DELETE FROM `user_roles` WHERE `user_id`={$id}");
      if($rolesSel){
        $vals = [];
        foreach($rolesSel as $rid){
          $rid = (int)$rid;
          if($rid>0) $vals[] = "({$id},{$rid})";
        }
        if($vals){
          @mysqli_query($conn,"INSERT IGNORE INTO `user_roles` (`user_id`,`role_id`) VALUES ".implode(',',$vals));
        }
      }

      $flash = $ok ? 'User updated.' : ('Error updating user. '.($err ? 'DB: '.$err : ''));
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash));
    exit;
  }

  /* ---- Change password ---- */
  if($action==='change_password'){
    $id   = (int)($_POST['id'] ?? 0);
    $pwd1 = (string)($_POST['password'] ?? '');
    $pwd2 = (string)($_POST['password2'] ?? '');
    if($id<=0 || $pwd1===''){
      $flash = 'Invalid password data.';
    } elseif($pwd1 !== $pwd2){
      $flash = 'Passwords do not match.';
    } else {
      $hash = password_hash($pwd1, PASSWORD_BCRYPT);
      $stmt = mysqli_prepare($conn,"UPDATE `users` SET `password_hash`=? WHERE `id`=?");
      mysqli_stmt_bind_param($stmt,'si',$hash,$id);
      $ok = @mysqli_stmt_execute($stmt);
      $err= mysqli_error($conn);
      mysqli_stmt_close($stmt);
      $flash = $ok ? 'Password changed.' : ('Error changing password. '.($err?'DB: '.$err:''));
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash));
    exit;
  }

  /* ---- Delete user ---- */
  if($action==='delete_user'){
    $id = (int)($_POST['id'] ?? 0);
    if($id>0){
      if($id === $CURRENT_USER_ID){
        $flash = "You can't delete your own account while logged in.";
      } else {
        @mysqli_query($conn,"DELETE FROM `user_roles` WHERE `user_id`={$id}");
        @mysqli_query($conn,"DELETE FROM `users` WHERE `id`={$id} LIMIT 1");
        $flash = 'User deleted.';
      }
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash));
    exit;
  }
}

/* ---------- Fetch Users ---------- */
$users = [];
$res = mysqli_query($conn,"SELECT u.*,
          (SELECT GROUP_CONCAT(r.name ORDER BY r.name SEPARATOR ', ') 
           FROM user_roles ur JOIN roles r ON r.id=ur.role_id WHERE ur.user_id=u.id) AS roles_names,
          (SELECT GROUP_CONCAT(r.id ORDER BY r.name SEPARATOR ',') 
           FROM user_roles ur JOIN roles r ON r.id=ur.role_id WHERE ur.user_id=u.id) AS roles_ids
        FROM users u
        ORDER BY u.created_at DESC");
if($res){ while($row=mysqli_fetch_assoc($res)){ $users[]=$row; } }

/* ---------- Base URL ---------- */
// $baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));
$flash = $_GET['flash'] ?? '';
?>
<?php ob_start(); ?>
<style>
.badge-soft { background: rgba(0,0,0,.06); color:#222; }
.role-chip{display:inline-block;border:1px solid #e5e7eb;border-radius:999px;padding:.15rem .5rem;margin:.1rem .25rem .1rem 0;font-size:.8rem;}
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>

<!-- Header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Users</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Settings</a></li>
      <li class="breadcrumb-item active" aria-current="page">Users</li>
    </ol>
  </div>
</div>

<?php if($flash): ?>
  <div class="alert alert-success py-2 px-3 mb-3"><?php echo h($flash); ?></div>
<?php endif; ?>

<div class="card custom-card">
  <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-3">
    <div class="card-title">Manage Users</div>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createUserModal"><i class="ri-add-line me-1"></i>Create User</button>
  </div>

  <div class="card-body p-0">
    <div class="table-responsive">
      <table class="table text-nowrap align-middle">
        <thead>
          <tr>
            <th style="width:60px;">ID</th>
            <th>Username</th>
            <th>Full Name</th>
            <th>Email</th>
            <th>Phone</th>
            <th>Status</th>
            <th>Roles</th>
            <th class="text-end pe-4">Actions</th>
          </tr>
        </thead>
        <tbody>
          <?php if(!$users): ?>
            <tr><td colspan="8" class="text-center py-4">No users yet.</td></tr>
          <?php else: foreach($users as $U): ?>
            <tr>
              <td><?php echo (int)$U['id']; ?></td>
              <td><?php echo h($U['username']); ?></td>
              <td><?php echo h($U['full_name'] ?: '—'); ?></td>
              <td><?php echo h($U['email'] ?: '—'); ?></td>
              <td><?php echo h($U['phone'] ?: '—'); ?></td>
              <td>
                <?php if((int)$U['is_active']===1): ?>
                  <span class="badge bg-success-transparent">Active</span>
                <?php else: ?>
                  <span class="badge bg-danger-transparent">Inactive</span>
                <?php endif; ?>
              </td>
              <td>
                <?php
                  $roles_names = array_filter(array_map('trim', explode(',', (string)$U['roles_names'])));
                  if(!$roles_names){ echo '<span class="text-muted">—</span>'; }
                  else { foreach($roles_names as $rn){ echo '<span class="role-chip">'.h($rn).'</span>'; } }
                ?>
              </td>
              <td class="text-end pe-4">
                <div class="btn-list">
                  <button type="button"
                          class="btn btn-sm btn-info-light btn-icon btn-edit"
                          title="Edit"
                          data-id="<?php echo (int)$U['id']; ?>"
                          data-username="<?php echo h($U['username']); ?>"
                          data-full_name="<?php echo h($U['full_name']); ?>"
                          data-email="<?php echo h($U['email']); ?>"
                          data-phone="<?php echo h($U['phone']); ?>"
                          data-active="<?php echo (int)$U['is_active']; ?>"
                          data-roles_ids="<?php echo h($U['roles_ids']); ?>">
                    <i class="ri-pencil-line"></i>
                  </button>

                  <button type="button"
                          class="btn btn-sm btn-warning-light btn-icon btn-pass"
                          title="Change Password"
                          data-id="<?php echo (int)$U['id']; ?>"
                          data-username="<?php echo h($U['username']); ?>">
                    <i class="ri-key-2-line"></i>
                  </button>

                  <form method="post" class="d-inline" onsubmit="return confirm('Delete this user?');">
                    <input type="hidden" name="action" value="delete_user">
                    <input type="hidden" name="id" value="<?php echo (int)$U['id']; ?>">
                    <button class="btn btn-sm btn-danger-light btn-icon" title="Delete"><i class="ri-delete-bin-line"></i></button>
                  </form>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<!-- Create User Modal -->
<div class="modal fade" id="createUserModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="create_user">
      <div class="modal-header">
        <h6 class="modal-title">Create User</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <div class="col-md-6">
            <label class="form-label">Username</label>
            <input type="text" class="form-control" name="username" required>
          </div>
          <div class="col-md-6">
            <label class="form-label">Full Name</label>
            <input type="text" class="form-control" name="full_name">
          </div>
          <div class="col-md-6">
            <label class="form-label">Email</label>
            <input type="email" class="form-control" name="email">
          </div>
          <div class="col-md-6">
            <label class="form-label">Phone</label>
            <input type="text" class="form-control" name="phone">
          </div>
          <div class="col-md-6">
            <label class="form-label">Password</label>
            <input type="password" class="form-control" name="password" required>
          </div>
          <div class="col-md-6">
            <label class="form-label">Confirm Password</label>
            <input type="password" class="form-control" name="password2" required>
          </div>
          <div class="col-12">
            <label class="form-label">Roles</label>
            <div class="row">
              <?php foreach($all_roles as $r): ?>
                <div class="col-6">
                  <label class="form-check">
                    <input class="form-check-input" type="checkbox" name="roles[]" value="<?php echo (int)$r['id']; ?>">
                    <span class="form-check-label"><?php echo h($r['name']); ?></span>
                  </label>
                </div>
              <?php endforeach; if(!$all_roles): ?>
                <div class="col-12 text-muted small">No roles yet — create some in Roles page.</div>
              <?php endif; ?>
            </div>
          </div>
          <div class="col-12">
            <label class="form-check">
              <input class="form-check-input" type="checkbox" name="is_active" checked>
              <span class="form-check-label">Active</span>
            </label>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
        <button class="btn btn-primary">Create</button>
      </div>
    </form>
  </div>
</div>

<!-- Edit User Modal -->
<div class="modal fade" id="editUserModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="update_user">
      <input type="hidden" name="id" id="eu-id">
      <div class="modal-header">
        <h6 class="modal-title">Edit User</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <div class="col-md-6">
            <label class="form-label">Username</label>
            <input type="text" class="form-control" name="username" id="eu-username" required>
          </div>
          <div class="col-md-6">
            <label class="form-label">Full Name</label>
            <input type="text" class="form-control" name="full_name" id="eu-full_name">
          </div>
          <div class="col-md-6">
            <label class="form-label">Email</label>
            <input type="email" class="form-control" name="email" id="eu-email">
          </div>
          <div class="col-md-6">
            <label class="form-label">Phone</label>
            <input type="text" class="form-control" name="phone" id="eu-phone">
          </div>
          <div class="col-12">
            <label class="form-label">Roles</label>
            <div class="row" id="eu-roles">
              <?php foreach($all_roles as $r): ?>
                <div class="col-6">
                  <label class="form-check">
                    <input class="form-check-input eu-role" type="checkbox" name="roles[]" value="<?php echo (int)$r['id']; ?>">
                    <span class="form-check-label"><?php echo h($r['name']); ?></span>
                  </label>
                </div>
              <?php endforeach; if(!$all_roles): ?>
                <div class="col-12 text-muted small">No roles yet — create some in Roles page.</div>
              <?php endif; ?>
            </div>
          </div>
          <div class="col-12">
            <label class="form-check">
              <input class="form-check-input" type="checkbox" name="is_active" id="eu-active">
              <span class="form-check-label">Active</span>
            </label>
          </div>
          <div class="col-12 small text-muted">Use "Change Password" for password updates.</div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
        <button class="btn btn-primary">Save</button>
      </div>
    </form>
  </div>
</div>

<!-- Change Password Modal -->
<div class="modal fade" id="passUserModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="change_password">
      <input type="hidden" name="id" id="pu-id">
      <div class="modal-header">
        <h6 class="modal-title" id="pu-title">Change Password</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <div class="col-md-6">
            <label class="form-label">New Password</label>
            <input type="password" class="form-control" name="password" required>
          </div>
          <div class="col-md-6">
            <label class="form-label">Confirm Password</label>
            <input type="password" class="form-control" name="password2" required>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
        <button class="btn btn-primary">Change</button>
      </div>
    </form>
  </div>
</div>

<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
(function(){
  // Edit modal populate
  document.querySelectorAll('.btn-edit').forEach(btn=>{
    btn.addEventListener('click', function(){
      const d = this.dataset;
      document.getElementById('eu-id').value = d.id || '';
      document.getElementById('eu-username').value = d.username || '';
      document.getElementById('eu-full_name').value = d.full_name || '';
      document.getElementById('eu-email').value = d.email || '';
      document.getElementById('eu-phone').value = d.phone || '';
      document.getElementById('eu-active').checked = (d.active === '1');

      const selected = (d.roles_ids||'').split(',').map(s=>s.trim()).filter(Boolean);
      document.querySelectorAll('#eu-roles .eu-role').forEach(ch => {
        ch.checked = selected.includes(ch.value);
      });

      new bootstrap.Modal(document.getElementById('editUserModal')).show();
    });
  });

  // Password modal populate
  document.querySelectorAll('.btn-pass').forEach(btn=>{
    btn.addEventListener('click', function(){
      const d = this.dataset;
      document.getElementById('pu-id').value = d.id || '';
      document.getElementById('pu-title').textContent = 'Change Password — ' + (d.username||('User #'+(d.id||'')));
      new bootstrap.Modal(document.getElementById('passUserModal')).show();
    });
  });
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
