<?php
/* ============================================================================
 * File: activities-calls.php
 * Purpose: Dashboard listing all CALL activities from `crm_activities`
 * Stack : Core PHP + MySQLi (NO PDO). DB connection via ../config.php
 * Rules : Collation utf8mb4_general_ci
 * Layout: Includes ../layouts/base.php (RBAC/bootstrap/UI gating)
 * Notes : - Filters: date_from/date_to (date_start), q (title/desc/lead name/company)
 *         - Pagination: 10/25/50/500
 *         - Source Of is derived from saved_from: contains 'opportunit' => Opportunity else Lead
 * ========================================================================== */

@session_start();
require_once __DIR__ . '/../config.php'; // MUST hold $conn (mysqli)

/* ---- Charset & collation enforcement ---- */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* ---------- Helper(s) ---------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function src_of($savedFrom){
  return (stripos((string)$savedFrom, 'opportunit') !== false) ? 'Opportunity' : 'Lead';
}
date_default_timezone_set('Asia/Kolkata');

/* ---------- Filters ---------- */
$perPageOptions = [10,25,50,500];
$perPage = (int)($_GET['per_page'] ?? 25); if(!in_array($perPage,$perPageOptions,true)) $perPage = 25;
$page    = max(1,(int)($_GET['page'] ?? 1));
$offset  = ($page-1)*$perPage;

$q       = trim($_GET['q'] ?? '');
$df      = trim($_GET['date_from'] ?? ''); // YYYY-MM-DD
$dt      = trim($_GET['date_to'] ?? '');   // YYYY-MM-DD

$where = ["a.`activity_type`='call'"];
if($q !== ''){
  $qEsc = mysqli_real_escape_string($conn,$q);
  $where[] = "(a.`title` LIKE '%{$qEsc}%' OR a.`description` LIKE '%{$qEsc}%' OR l.`contact_name` LIKE '%{$qEsc}%' OR l.`company` LIKE '%{$qEsc}%')";
}
if(preg_match('/^\d{4}-\d{2}-\d{2}$/', $df)){
  $dfEsc = mysqli_real_escape_string($conn,$df);
  $where[] = "a.`date_start` >= '{$dfEsc} 00:00:00'";
}
if(preg_match('/^\d{4}-\d{2}-\d{2}$/', $dt)){
  $dtEsc = mysqli_real_escape_string($conn,$dt);
  $where[] = "a.`date_start` <= '{$dtEsc} 23:59:59'";
}
$whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';

/* ---------- KPIs (Today / This Week / Overdue Reminders) ---------- */
$today = date('Y-m-d');
$weekStart = date('Y-m-d', strtotime('monday this week'));
$now = date('Y-m-d H:i:s');
$kpiToday = $kpiWeek = $kpiOverdue = 0;
$r = mysqli_query($conn, "SELECT 
  SUM(CASE WHEN DATE(`date_start`)='{$today}' THEN 1 ELSE 0 END) AS k1,
  SUM(CASE WHEN DATE(`date_start`) BETWEEN '{$weekStart}' AND '{$today}' THEN 1 ELSE 0 END) AS k2,
  SUM(CASE WHEN `remind`=1 AND `reminder_at` IS NOT NULL AND `reminder_at` < '{$now}' THEN 1 ELSE 0 END) AS k3
  FROM `crm_activities` WHERE `activity_type`='call'");
if($r){ $row = mysqli_fetch_assoc($r); $kpiToday=(int)$row['k1']; $kpiWeek=(int)$row['k2']; $kpiOverdue=(int)$row['k3']; }

/* ---------- Count & Fetch ---------- */
$totalRows = 0;
$cntRes = mysqli_query($conn, "SELECT COUNT(*) AS c
  FROM `crm_activities` a
  JOIN `leads` l ON l.`id`=a.`lead_id`
  {$whereSql}");
if($cntRes){ $cr = mysqli_fetch_assoc($cntRes); $totalRows = (int)$cr['c']; }
$totalPages = max(1,(int)ceil($totalRows/$perPage));

$sql = "SELECT a.`id`,a.`lead_id`,a.`status`,a.`direction`,a.`title`,a.`description`,
               a.`date_start`,a.`date_end`,a.`duration_minutes`,a.`assigned_user`,
               a.`created_by`,a.`remind`,a.`reminder_at`,a.`saved_from`,
               l.`contact_name`,l.`company`,l.`phone`,l.`email`
        FROM `crm_activities` a
        JOIN `leads` l ON l.`id`=a.`lead_id`
        {$whereSql}
        ORDER BY a.`date_start` DESC, a.`id` DESC
        LIMIT {$perPage} OFFSET {$offset}";
$rows = [];
$res = mysqli_query($conn,$sql);
if($res){ while($row = mysqli_fetch_assoc($res)){ $rows[] = $row; } }

/* ---------- Base URL ---------- */
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://')
         . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));

/* ---------- Styles ---------- */
ob_start(); ?>
<style>
.kpi { display:flex; gap:12px; flex-wrap:wrap; margin-bottom:12px; }
.kpi .card { padding:10px 14px; border:1px solid #e9ecef; border-radius:10px; background:#fff; min-width:180px; }
.table td, .table th { vertical-align: middle; }
.badge-soft { background: rgba(0,0,0,.06); color:#222; }
.small-muted { font-size:12px;color:#777; }
</style>
<?php $styles = ob_get_clean();

/* ---------- Content ---------- */
ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Activities — Calls</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Calls</li>
    </ol>
  </div>
</div>

<div class="kpi">
  <div class="card"><div class="text-muted">Today</div><div class="fs-4 fw-bold"><?php echo (int)$kpiToday; ?></div></div>
  <div class="card"><div class="text-muted">This Week</div><div class="fs-4 fw-bold"><?php echo (int)$kpiWeek; ?></div></div>
  <div class="card"><div class="text-muted">Overdue Reminders</div><div class="fs-4 fw-bold"><?php echo (int)$kpiOverdue; ?></div></div>
</div>

<div class="card custom-card">
  <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-3">
    <div class="card-title">All Calls</div>
    <form class="d-flex align-items-end flex-wrap gap-2" method="get">
      <div>
        <label class="form-label mb-1 small">Date From</label>
        <input type="date" class="form-control form-control-sm" name="date_from" value="<?php echo h($df); ?>">
      </div>
      <div>
        <label class="form-label mb-1 small">Date To</label>
        <input type="date" class="form-control form-control-sm" name="date_to" value="<?php echo h($dt); ?>">
      </div>
      <div>
        <label class="form-label mb-1 small">Search</label>
        <input type="text" class="form-control form-control-sm" name="q" placeholder="name, company, title..." value="<?php echo h($q); ?>">
      </div>
      <div>
        <label class="form-label mb-1 small">Per Page</label>
        <select class="form-select form-select-sm" name="per_page" onchange="this.form.submit()">
          <?php foreach($perPageOptions as $opt): ?>
            <option value="<?php echo $opt; ?>" <?php echo $opt===$perPage?'selected':''; ?>><?php echo $opt; ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div><button class="btn btn-sm btn-primary">Filter</button></div>
    </form>
  </div>

  <div class="card-body p-0">
    <div class="table-responsive">
      <table class="table table-hover text-nowrap align-middle">
        <thead>
          <tr>
            <th>When</th>
            <th>Contact</th>
            <th>Company</th>
            <th>Title</th>
            <th>Status</th>
            <th>Direction</th>
            <th>Duration</th>
            <th>Reminder</th>
            <th>Saved From</th>
            <th>Source Of</th>
            <th>Assigned</th>
          </tr>
        </thead>
        <tbody>
          <?php if(!$rows): ?>
            <tr><td colspan="11" class="text-center py-4">No call activities found.</td></tr>
          <?php else: foreach($rows as $r): ?>
            <tr>
              <td>
                <div class="fw-medium"><?php echo h($r['date_start'] ? date('Y-m-d H:i', strtotime($r['date_start'])) : '—'); ?></div>
                <?php if($r['date_end']): ?><div class="small-muted">to <?php echo h(date('Y-m-d H:i', strtotime($r['date_end']))); ?></div><?php endif; ?>
              </td>
              <td><?php echo h($r['contact_name'] ?: '—'); ?></td>
              <td><?php echo h($r['company'] ?: '—'); ?></td>
              <td>
                <div class="fw-medium"><?php echo h($r['title'] ?: 'Call'); ?></div>
                <?php if(!empty($r['description'])): ?>
                  <div class="small text-muted"><?php echo h(mb_strimwidth($r['description'],0,80,'…','UTF-8')); ?></div>
                <?php endif; ?>
              </td>
              <td><?php echo h($r['status']); ?></td>
              <td><?php echo h($r['direction']); ?></td>
              <td><?php echo $r['duration_minutes']!==null ? (int)$r['duration_minutes'].' mins' : '—'; ?></td>
              <td><?php echo ($r['remind'] && $r['reminder_at']) ? h(date('Y-m-d H:i', strtotime($r['reminder_at']))) : '—'; ?></td>
              <td><span class="badge badge-soft"><?php echo h($r['saved_from'] ?: '—'); ?></span></td>
              <td><?php echo src_of($r['saved_from']); ?></td>
              <td><?php echo h($r['assigned_user'] ?: '—'); ?></td>
            </tr>
          <?php endforeach; endif; ?>
        </tbody>
      </table>
    </div>
  </div>

  <div class="card-footer border-top-0">
    <div class="d-flex align-items-center flex-wrap w-100">
      <div>Showing <?php echo count($rows); ?> of <?php echo (int)$totalRows; ?> entries</div>
      <div class="ms-auto">
        <nav aria-label="Page navigation" class="pagination-style-4">
          <ul class="pagination mb-0">
            <li class="page-item <?php echo $page<=1?'disabled':''; ?>">
              <a class="page-link" href="<?php if($page>1){ $p=$page-1; echo h($_SERVER['PHP_SELF']).'?'.http_build_query(['page'=>$p,'per_page'=>$perPage,'q'=>$q,'date_from'=>$df,'date_to'=>$dt]); } else { echo 'javascript:void(0);'; } ?>">Prev</a>
            </li>
            <?php $start=max(1,$page-2); $end=min($totalPages,$page+2); for($p=$start;$p<=$end;$p++): ?>
              <li class="page-item <?php echo $p===$page?'active':''; ?>">
                <a class="page-link" href="<?php echo h($_SERVER['PHP_SELF']).'?'.http_build_query(['page'=>$p,'per_page'=>$perPage,'q'=>$q,'date_from'=>$df,'date_to'=>$dt]); ?>"><?php echo $p; ?></a>
              </li>
            <?php endfor; ?>
            <li class="page-item <?php echo $page>=$totalPages?'disabled':''; ?>">
              <a class="page-link" href="<?php if($page<$totalPages){ $p=$page+1; echo h($_SERVER['PHP_SELF']).'?'.http_build_query(['page'=>$p,'per_page'=>$perPage,'q'=>$q,'date_from'=>$df,'date_to'=>$dt]); } else { echo 'javascript:void(0);'; } ?>">Next</a>
            </li>
          </ul>
        </nav>
      </div>
    </div>
  </div>
</div>
<?php $content = ob_get_clean();

/* ---------- Scripts (if any) ---------- */
ob_start(); ?>
<script></script>
<?php $scripts = ob_get_clean();

/* ---------- Render via layout ---------- */
include 'layouts/base.php';
