<?php
/* =============================================================================
 * File: calendar.php
 * Purpose: Calendar page using ONE source of truth:
 *          - ALL items (Call / Meeting / Task) are stored in `crm_activities`
 * Views : Month (default) / Week / Timeline; drag/resize; filter by Assigned User
 * Stack : Core PHP + MySQLi (NO PDO). DB in ../config.php. Collation utf8mb4_general_ci
 * Notes : - Different colors per type (call/meeting/task)
 *         - "saved_from" is set to "Calendar" on create
 *         - Lead is required (per FK in your schema)
 *         - Top title shows "September 2025" style
 * ========================================================================== */

date_default_timezone_set('Asia/Kolkata');
@session_start();
require_once __DIR__ . '/../config.php';

mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* ---------- Ensure ONLY the activities table (matches your schema) ---------- */
/* This CREATE is safe (IF NOT EXISTS). It won't overwrite an existing table. */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `crm_activities` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_id` INT UNSIGNED NOT NULL,
  `activity_type` ENUM('call','meeting','task','email') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'task',
  `status` ENUM('Planned','Held','Not Held','Completed','Cancelled') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'Planned',
  `direction` ENUM('Inbound','Outbound','N/A') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'N/A',
  `title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `description` TEXT COLLATE utf8mb4_general_ci,
  `date_start` DATETIME DEFAULT NULL,
  `date_end` DATETIME DEFAULT NULL,
  `duration_minutes` INT DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_by` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `remind` TINYINT(1) NOT NULL DEFAULT 0,
  `reminder_at` DATETIME DEFAULT NULL,
  `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY `idx_lead_id` (`lead_id`),
  KEY `idx_reminder_at` (`reminder_at`),
  KEY `idx_saved_from` (`saved_from`),
  CONSTRAINT `fk_crm_activities_lead_id` FOREIGN KEY (`lead_id`) REFERENCES `leads`(`id`) ON DELETE CASCADE,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

/* ---------- Helpers ---------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function color_for_type($t){
  $t = strtolower((string)$t);
  if($t==='call')    return '#7b61ff'; // purple
  if($t==='meeting') return '#86b7fe'; // blue
  if($t==='task')    return '#6fcf97'; // green
  if($t==='email')   return '#f39c12'; // orange (not used by modal but supported)
  return '#9aa0a6';
}

/* ---------- Users for filter ---------- */
$assignedUsers = [];
$r2 = mysqli_query($conn, "SELECT DISTINCT TRIM(COALESCE(`assigned_user`,'')) u FROM `crm_activities` WHERE TRIM(COALESCE(`assigned_user`,''))<>''");
if($r2){ while($r = mysqli_fetch_assoc($r2)){ $assignedUsers[$r['u']] = 1; } }
$r3 = mysqli_query($conn, "SELECT DISTINCT TRIM(COALESCE(`assigned_user`,'')) u FROM `leads` WHERE TRIM(COALESCE(`assigned_user`,''))<>''");
if($r3){ while($r = mysqli_fetch_assoc($r3)){ $assignedUsers[$r['u']] = 1; } }
$assignedUsers = array_keys($assignedUsers);
sort($assignedUsers, SORT_NATURAL | SORT_FLAG_CASE);

/* ---------- FEED: ONLY crm_activities ---------- */
if(isset($_GET['action']) && $_GET['action']==='feed'){
    ini_set('display_errors', '0');
    while (ob_get_level()) { ob_end_clean(); }

    $startISO = $_GET['start'] ?? '';
    $endISO   = $_GET['end'] ?? '';
    $userFilt = trim($_GET['user'] ?? '');

    $start = $startISO ? date('Y-m-d H:i:s', strtotime($startISO)) : date('Y-m-d 00:00:00');
    $end   = $endISO   ? date('Y-m-d H:i:s', strtotime($endISO))   : date('Y-m-d 23:59:59');

    $es = mysqli_real_escape_string($conn, $start);
    $ee = mysqli_real_escape_string($conn, $end);
    $eu = mysqli_real_escape_string($conn, $userFilt);

    $events = [];

    $sqlA = "SELECT a.`id`,a.`activity_type`,a.`title`,a.`date_start`,a.`date_end`,a.`assigned_user`,
                    a.`description`,a.`duration_minutes`,a.`status`,a.`direction`,a.`lead_id`,a.`saved_from`,
                    l.`contact_name`
             FROM `crm_activities` a
             LEFT JOIN `leads` l ON l.`id`=a.`lead_id`
             WHERE (a.`date_start` < '$ee') AND (COALESCE(a.`date_end`,a.`date_start`) >= '$es')";
    if($userFilt !== '') { $sqlA .= " AND TRIM(COALESCE(a.`assigned_user`,'')) = '$eu'"; }
    $resA = mysqli_query($conn, $sqlA);
    if($resA){
        while($r = mysqli_fetch_assoc($resA)){
            $label = ucfirst($r['activity_type'] ?: 'activity');
            $title = trim((string)$r['title']);
            if(!empty($r['contact_name'])) $title = ($title!=='' ? $title.' — ' : '') . $r['contact_name'];
            $events[] = [
                'id'    => 'A'.(int)$r['id'],
                'title' => $title !== '' ? $title : $label,
                'start' => $r['date_start'] ? str_replace(' ', 'T', $r['date_start']) : null,
                'end'   => $r['date_end']   ? str_replace(' ', 'T', $r['date_end'])   : null,
                'allDay'=> false,
                'color' => color_for_type($r['activity_type']),
                'extendedProps' => [
                    'source'      => 'activity',
                    'kind'        => $label, // Call / Meeting / Task / Email
                    'assigned'    => $r['assigned_user'],
                    'lead_id'     => $r['lead_id'],
                    'description' => $r['description'],
                    'duration'    => $r['duration_minutes'],
                    'status'      => $r['status'],
                    'direction'   => $r['direction'],
                    'saved_from'  => $r['saved_from']
                ]
            ];
        }
    }

    http_response_code(200);
    header('Content-Type: application/json; charset=UTF-8');
    echo json_encode($events);
    exit;
}

/* ---------- AJAX: Move / Resize -> update crm_activities ---------- */
if(isset($_POST['ajax']) && $_POST['ajax']==='move_resize'){
    $id = $_POST['id'] ?? '';
    $start = $_POST['start'] ?? '';
    $end   = $_POST['end'] ?? '';

    $ok = false;
    if($id !== '' && $id[0]==='A'){
        $iid = (int)substr($id,1);
        if($iid>0 && $start){
            $start = mysqli_real_escape_string($conn, $start);
            $end   = $end ? "'".mysqli_real_escape_string($conn,$end)."'" : "NULL";
            $ok = mysqli_query($conn, "UPDATE `crm_activities` SET `date_start`='$start',`date_end`=$end WHERE `id`=$iid");
        }
    }
    header('Content-Type: application/json');
    echo json_encode(['ok'=>$ok?true:false]);
    exit;
}

/* ---------- POST: Create / Update / Delete (ALL in crm_activities) ---------- */
$flash = '';
if($_SERVER['REQUEST_METHOD']==='POST' && empty($_POST['ajax'])){
    $action = $_POST['action'] ?? '';
    $kind   = $_POST['kind'] ?? 'Meeting'; // Call / Meeting / Task

    if($action==='create' || $action==='update'){
        $title    = trim($_POST['title'] ?? '');
        $assigned = trim($_POST['assigned_user'] ?? '');
        $start    = trim($_POST['start_dt'] ?? '');
        $end      = trim($_POST['end_dt'] ?? '');
        $desc     = trim($_POST['description'] ?? '');
        $leadId   = (int)($_POST['lead_id'] ?? 0);
        $duration = (int)($_POST['duration_minutes'] ?? 0);

        // Enforce FK requirement
        if($leadId <= 0){
            $flash = 'Please select a Lead.';
            header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)."&user=".urlencode($_GET['user'] ?? ''));
            exit;
        }

        if($title!=='' && $start!==''){
            $atype = strtolower($kind)==='call' ? 'call' : (strtolower($kind)==='task' ? 'task' : 'meeting');
            if($action==='create'){
                $t = mysqli_real_escape_string($conn,$title);
                $s = mysqli_real_escape_string($conn,$start);
                $e = $end? "'".mysqli_real_escape_string($conn,$end)."'" : "NULL";
                $a = mysqli_real_escape_string($conn,$assigned);
                $d = mysqli_real_escape_string($conn,$desc);
                $cb = isset($_SESSION['user_name']) ? mysqli_real_escape_string($conn, (string)$_SESSION['user_name']) : '';
                $ok = mysqli_query($conn,"INSERT INTO `crm_activities`
                    (`lead_id`,`activity_type`,`status`,`direction`,`title`,`description`,
                     `date_start`,`date_end`,`duration_minutes`,`assigned_user`,
                     `created_by`,`remind`,`reminder_at`,`saved_from`)
                    VALUES ($leadId,'$atype','Planned','N/A','$t','$d',
                            '$s',$e,".($duration>0?$duration:"NULL").",'$a',
                            '$cb',0,NULL,'Calendar')");
                $flash = $ok ? (ucfirst($atype).' created.') : ('Error creating '.ucfirst($atype).'.');
            } else {
                $id = (int)($_POST['id'] ?? 0);
                $t = mysqli_real_escape_string($conn,$title);
                $s = mysqli_real_escape_string($conn,$start);
                $e = $end? "'".mysqli_real_escape_string($conn,$end)."'" : "NULL";
                $a = mysqli_real_escape_string($conn,$assigned);
                $d = mysqli_real_escape_string($conn,$desc);
                $ok = mysqli_query($conn,"UPDATE `crm_activities`
                    SET `lead_id`=$leadId,
                        `activity_type`='$atype',
                        `title`='$t',
                        `description`='$d',
                        `date_start`='$s',
                        `date_end`=$e,
                        `duration_minutes`=".($duration>0?$duration:"NULL").",
                        `assigned_user`='$a'
                    WHERE `id`=$id");
                $flash = $ok ? (ucfirst($atype).' updated.') : ('Error updating '.ucfirst($atype).'.');
            }
        } else {
            $flash = 'Title & Start are required.';
        }
        header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)."&user=".urlencode($_GET['user'] ?? ''));
        exit;
    }

    if($action==='delete'){
        $id = (int)($_POST['id'] ?? 0);
        if($id>0){
            mysqli_query($conn, "DELETE FROM `crm_activities` WHERE `id`=$id LIMIT 1");
            $flash = 'Deleted.';
        }
        header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)."&user=".urlencode($_GET['user'] ?? ''));
        exit;
    }
}

/* ---------- Base URL ---------- */
$rootFolder = basename($_SERVER['DOCUMENT_ROOT']);
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));
$flash = $_GET['flash'] ?? $flash;
$currentUserFilter = trim($_GET['user'] ?? '');
?>
<?php ob_start(); ?>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.css">
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/flatpickr/flatpickr.min.css">
<style>
.calendar-toolbar{display:flex;align-items:center;gap:10px;flex-wrap:wrap}
.fc .fc-toolbar-title{font-size:1.2rem}
.view-switch .btn{min-width:90px}
.fc-theme-standard .fc-scrollgrid { border-left:0;border-right:0; }
.fc-event { border-radius:6px }

/* Colors (server also sets) */
.tag-Call    { background:#7b61ff;border-color:#7b61ff;color:#fff }
.tag-Meeting { background:#86b7fe;border-color:#86b7fe;color:#fff }
.tag-Task    { background:#6fcf97;border-color:#6fcf97;color:#fff }
.tag-Email   { background:#f39c12;border-color:#f39c12;color:#fff }
.tag-Other   { background:#9aa0a6;border-color:#9aa0a6;color:#fff }

/* Larger time rows (week view) */
.fc .fc-timegrid-slot{ height:2.5rem }
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>

<!-- Start::page-header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Calendar</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Calendar</li>
    </ol>
  </div>
</div>
<!-- End::page-header -->

<?php if($flash): ?>
<div class="alert alert-success py-2 px-3 mb-3"><?php echo h($flash); ?></div>
<?php endif; ?>

<div class="card custom-card">
  <div class="card-header">
    <div class="calendar-toolbar w-100">
      <div class="btn-group" role="group">
        <button class="btn btn-light" id="btn-prev"><i class="ri-arrow-left-s-line"></i></button>
        <button class="btn btn-light" id="btn-today">Today</button>
        <button class="btn btn-light" id="btn-next"><i class="ri-arrow-right-s-line"></i></button>
      </div>

      <!-- Month + Year Title -->
      <h5 class="mb-0 ms-3 fw-semibold" id="cal-title"></h5>

      <div class="ms-auto d-flex align-items-center gap-2">
        <div class="d-flex align-items-center gap-2">
          <label class="form-label mb-0">Assigned User</label>
          <select id="filter-user" class="form-select form-select-sm">
            <option value="">All</option>
            <?php foreach($assignedUsers as $u): ?>
              <option value="<?php echo h($u); ?>" <?php echo $u===$currentUserFilter?'selected':''; ?>><?php echo h($u); ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="view-switch btn-group" role="group">
          <button class="btn btn-outline-primary active" id="view-month">Month</button>
          <button class="btn btn-outline-primary" id="view-week">Week</button>
          <button class="btn btn-outline-primary" id="view-timeline">Timeline</button>
        </div>

        <button class="btn btn-primary" id="btn-add">+ Add</button>
      </div>
    </div>
  </div>
  <div class="card-body p-0">
    <div id="calendar"></div>
  </div>
</div>

<!-- Create / Edit Modal -->
<div class="modal fade" id="eventModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="id" id="ev-id">
      <input type="hidden" name="action" id="ev-action" value="create">
      <div class="modal-header">
        <h6 class="modal-title" id="ev-title">Add</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <!-- Type first -->
          <div class="col-12">
            <label class="form-label d-block">Type</label>
            <div class="d-flex gap-3">
              <label class="form-check mb-0"><input class="form-check-input" type="radio" name="kind" value="Call" id="kind-call"> <span class="form-check-label">Call</span></label>
              <label class="form-check mb-0"><input class="form-check-input" type="radio" name="kind" value="Meeting" id="kind-meeting" checked> <span class="form-check-label">Meeting</span></label>
              <label class="form-check mb-0"><input class="form-check-input" type="radio" name="kind" value="Task" id="kind-task"> <span class="form-check-label">Task</span></label>
            </div>
          </div>

          <div class="col-12">
            <label class="form-label">Title</label>
            <input type="text" name="title" id="f-title" class="form-control" required>
          </div>

          <div class="col-md-6">
            <label class="form-label">Start</label>
            <input type="text" name="start_dt" id="f-start" class="form-control" required placeholder="YYYY-MM-DD HH:MM">
          </div>
          <div class="col-md-6">
            <label class="form-label">End</label>
            <input type="text" name="end_dt" id="f-end" class="form-control" placeholder="YYYY-MM-DD HH:MM">
          </div>

          <div class="col-md-6">
            <label class="form-label">Assigned User</label>
            <input type="text" name="assigned_user" id="f-assigned" class="form-control" placeholder="e.g. Amit">
          </div>

          <div class="col-md-6">
            <label class="form-label">Lead <span class="text-danger">*</span></label>
            <select name="lead_id" id="f-lead" class="form-select" required>
              <option value="" selected disabled>-- Select Lead --</option>
              <?php
                $lr = mysqli_query($conn, "SELECT `id`,`contact_name` FROM `leads` ORDER BY `contact_name` ASC LIMIT 1000");
                if($lr){ while($x=mysqli_fetch_assoc($lr)){ echo '<option value="'.(int)$x['id'].'">'.h($x['contact_name']).'</option>'; } }
              ?>
            </select>
          </div>

          <div class="col-md-6">
            <label class="form-label">Duration (mins, for Call/Task)</label>
            <input type="number" min="0" step="1" name="duration_minutes" id="f-duration" class="form-control" placeholder="e.g. 15">
          </div>

          <div class="col-12">
            <label class="form-label">Description / Notes</label>
            <textarea name="description" id="f-desc" class="form-control" rows="3"></textarea>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-danger me-auto d-none" id="btn-delete">Delete</button>
        <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
        <button type="submit" class="btn btn-primary">Save</button>
      </div>
    </form>
  </div>
</div>

<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.js"></script>
<script src="<?php echo $baseUrl; ?>/assets/libs/flatpickr/flatpickr.min.js"></script>
<script>
(function(){
  const fpStart = flatpickr('#f-start', { enableTime:true, dateFormat:'Y-m-d H:i' });
  const fpEnd   = flatpickr('#f-end',   { enableTime:true, dateFormat:'Y-m-d H:i' });

  const calendarEl = document.getElementById('calendar');
  const calendar = new FullCalendar.Calendar(calendarEl, {
    initialView: 'dayGridMonth',                  // default Month view
    height: 'auto',
    selectable: true,
    nowIndicator: true,
    editable: true,
    headerToolbar: false,
    titleFormat: { year: 'numeric', month: 'long' }, // "September 2025"
    datesSet(info){
      const el = document.getElementById('cal-title');
      if(el) el.textContent = info.view.title;
    },
    eventSources: [{
      url: '<?php echo h($_SERVER["PHP_SELF"]); ?>',
      method: 'GET',
      extraParams: function(){
        return { action: 'feed', user: document.getElementById('filter-user').value };
      },
      failure: function(){ alert('Failed to load events'); }
    }],
    select: function(info){
      openModal({ action:'create', kind:getCheckedKind(), start:info.startStr, end:info.endStr });
      calendar.unselect();
    },
    eventClick: function(info){
      const e = info.event, ext = e.extendedProps || {};
      openModal({
        action: 'update',
        id: e.id, // A45
        kind: ext.kind || 'Meeting',
        title: e.title,
        start: e.start ? e.start.toISOString().slice(0,16).replace('T',' ') : '',
        end:   e.end   ? e.end.toISOString().slice(0,16).replace('T',' ')   : '',
        assigned: ext.assigned || '',
        leadId: ext.lead_id || '',
        duration: ext.duration || '',
        description: ext.description || ''
      });
    },
    eventDrop: saveMoveResize,
    eventResize: saveMoveResize
  });

  function saveMoveResize(info){
    const e = info.event;
    const payload = new URLSearchParams({
      ajax: 'move_resize',
      id: e.id,
      start: e.start ? e.start.toISOString().slice(0,19).replace('T',' ') : '',
      end: e.end ? e.end.toISOString().slice(0,19).replace('T',' ') : ''
    });
    fetch('<?php echo h($_SERVER["PHP_SELF"]); ?>', {
      method: 'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body: payload
    }).then(r=>r.json()).then(js=>{
      if(!js.ok){ alert('Failed to update time'); info.revert(); }
    }).catch(()=>{ alert('Network error'); info.revert(); });
  }

  calendar.render();
  document.getElementById('cal-title').textContent = calendar.view.title;

  // Toolbar
  document.getElementById('btn-prev').addEventListener('click', ()=>calendar.prev());
  document.getElementById('btn-next').addEventListener('click', ()=>calendar.next());
  document.getElementById('btn-today').addEventListener('click', ()=>calendar.today());

  // View switchers
  function setActive(btn){ document.querySelectorAll('.view-switch .btn').forEach(b=>b.classList.remove('active')); btn.classList.add('active'); }
  document.getElementById('view-month').addEventListener('click', function(){ setActive(this); calendar.changeView('dayGridMonth'); });
  document.getElementById('view-week').addEventListener('click', function(){ setActive(this); calendar.changeView('timeGridWeek'); });
  document.getElementById('view-timeline').addEventListener('click', function(){ setActive(this); calendar.changeView('listWeek'); });

  // Filter by user
  document.getElementById('filter-user').addEventListener('change', function(){
    const params = new URLSearchParams(window.location.search);
    params.set('user', this.value);
    history.replaceState(null,'',window.location.pathname + '?' + params.toString());
    calendar.refetchEvents();
  });

  // Add
  document.getElementById('btn-add').addEventListener('click', function(){
    openModal({action:'create', kind:getCheckedKind()});
  });

  function getCheckedKind(){
    if(document.getElementById('kind-call').checked) return 'Call';
    if(document.getElementById('kind-task').checked) return 'Task';
    return 'Meeting';
  }

  // Modal
  const modalEl = document.getElementById('eventModal');
  const modal = new bootstrap.Modal(modalEl);

  function openModal(o){
    const isUpdate = (o.action==='update');
    const kind = o.kind || 'Meeting';
    document.getElementById('ev-action').value = o.action || 'create';
    document.getElementById('ev-title').textContent = (isUpdate ? 'Edit ' : 'Add ') + kind;
    document.getElementById('btn-delete').classList.toggle('d-none', !isUpdate);
    document.getElementById('ev-id').value = (isUpdate ? (o.id||'').replace(/^A/, '') : '');

    document.getElementById('f-title').value     = o.title || '';
    document.getElementById('f-assigned').value  = o.assigned || '';
    document.getElementById('f-desc').value      = o.description || '';
    document.getElementById('f-duration').value  = o.duration || '';
    document.getElementById('f-lead').value      = o.leadId ? String(o.leadId) : '';

    document.getElementById('kind-call').checked    = (kind==='Call');
    document.getElementById('kind-meeting').checked = (kind==='Meeting');
    document.getElementById('kind-task').checked    = (kind==='Task');

    fpStart.setDate(o.start ? o.start.replace(' ', 'T') : null, true);
    fpEnd.setDate(o.end ? o.end.replace(' ', 'T') : null, true);

    modal.show();
  }

  // Delete (from crm_activities)
  document.getElementById('btn-delete').addEventListener('click', function(){
    if(!confirm('Delete this item?')) return;
    const id = document.getElementById('ev-id').value || '';
    const f = document.createElement('form');
    f.method = 'POST';
    f.innerHTML = '<input type="hidden" name="action" value="delete">'+
                  '<input type="hidden" name="id" value="'+id+'">';
    document.body.appendChild(f); f.submit();
  });
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
