<?php
/* ============================================================================
 * File: opportunities.php
 * Purpose: Opportunities page with List / Kanban + Activity logger (Call/Mail/Task/Meeting)
 * Stack : Core PHP + MySQLi (NO PDO)
 * Rules : utf8mb4_general_ci
 * Changes:
 *   - RBAC: activities.log (show/hide activity buttons + block save)
 *   - saved_from = "<StageLabel> Opportunities" (unchanged)
 * ========================================================================== */

date_default_timezone_set('Asia/Kolkata');
@session_start();
require_once __DIR__ . '/../config.php';

mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function col_exists($conn,$table,$col){
  $t = mysqli_real_escape_string($conn,$table);
  $c = mysqli_real_escape_string($conn,$col);
  $q = mysqli_query($conn,"SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='{$t}' AND COLUMN_NAME='{$c}'");
  return ($q && mysqli_num_rows($q)>0);
}

/* ---------------------------- RBAC: schema & utils ---------------------------- */
$CURRENT_USER_ID = (int)($_SESSION['user_id1'] ?? 0);
function rbac_ensure_schema($conn){
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `roles` (`id` INT UNSIGNED NOT NULL AUTO_INCREMENT, `name` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL, `description` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL, `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP, `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP, PRIMARY KEY (`id`), UNIQUE KEY `uniq_role_name` (`name`)) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `permissions` (`code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL, `label` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL, `group_name` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL, PRIMARY KEY (`code`)) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `role_permissions` (`role_id` INT UNSIGNED NOT NULL, `perm_code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL, PRIMARY KEY (`role_id`,`perm_code`), KEY `idx_perm_code` (`perm_code`), CONSTRAINT `fk_role_permissions_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `user_roles` (`user_id` INT NOT NULL, `role_id` INT UNSIGNED NOT NULL, PRIMARY KEY (`user_id`,`role_id`), KEY `idx_role_id` (`role_id`), CONSTRAINT `fk_user_roles_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
}
function rbac_seed_permissions($conn){
  $perms=[['activities.log','Log activities','Activities'],['leads.create','Create leads','Leads'],['leads.export','Export leads (CSV)','Leads'],['roles.manage','Manage roles','Admin']];
  foreach($perms as $p){ @mysqli_query($conn,"INSERT IGNORE INTO `permissions` (`code`,`label`,`group_name`) VALUES ('".mysqli_real_escape_string($conn,$p[0])."','".mysqli_real_escape_string($conn,$p[1])."','".mysqli_real_escape_string($conn,$p[2])."')"); }
}
function rbac_has($conn,$uid,$code){
  if(!$uid) return true; // superuser default
  $uid=(int)$uid; $code=mysqli_real_escape_string($conn,$code);
  $q=mysqli_query($conn,"SELECT 1 FROM user_roles ur JOIN role_permissions rp ON rp.role_id=ur.role_id WHERE ur.user_id={$uid} AND rp.perm_code='{$code}' LIMIT 1");
  return ($q && mysqli_num_rows($q)>0);
}
rbac_ensure_schema($conn); rbac_seed_permissions($conn);
$CAN_LOG_ACTIVITIES = rbac_has($conn,$CURRENT_USER_ID,'activities.log');

$TAG_STATUS = ['Pending','Assigned','In Process','Prospect','Recycled','Dead'];
$STAGES = ['Prospecting','Qualification','Proposal','Negotiation','Closed Won'];
function stage_to_label($s){ return $s==='Prospecting' ? 'Prospect' : $s; }

/* ---- Leads table (same as before, with opp_* fields) ---- */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `leads` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `contact_name` VARCHAR(150) COLLATE utf8mb4_general_ci NOT NULL,
  `email` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `phone` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_status` ENUM('New','Follow-up','Closed','Contacted','Disqualified','Qualified') COLLATE utf8mb4_general_ci DEFAULT 'New',
  `company` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_source` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `tags` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `avatar` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `opp_title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `opp_amount` DECIMAL(12,2) DEFAULT NULL,
  `opp_close_date` DATE DEFAULT NULL,
  `opp_stage` ENUM('Prospecting','Qualification','Proposal','Negotiation','Closed Won') COLLATE utf8mb4_general_ci DEFAULT 'Prospecting',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
if(!col_exists($conn,'leads','assigned_user'))  @mysqli_query($conn,"ALTER TABLE `leads` ADD COLUMN `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL");
if(!col_exists($conn,'leads','opp_title'))      @mysqli_query($conn,"ALTER TABLE `leads` ADD COLUMN `opp_title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL");
if(!col_exists($conn,'leads','opp_amount'))     @mysqli_query($conn,"ALTER TABLE `leads` ADD COLUMN `opp_amount` DECIMAL(12,2) DEFAULT NULL");
if(!col_exists($conn,'leads','opp_close_date')) @mysqli_query($conn,"ALTER TABLE `leads` ADD COLUMN `opp_close_date` DATE DEFAULT NULL");
if(!col_exists($conn,'leads','opp_stage'))      @mysqli_query($conn,"ALTER TABLE `leads` ADD COLUMN `opp_stage` ENUM('Prospecting','Qualification','Proposal','Negotiation','Closed Won') COLLATE utf8mb4_general_ci DEFAULT 'Prospecting'");

/* ---- Activities table (unchanged) ---- */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `crm_activities` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_id` INT UNSIGNED NOT NULL,
  `activity_type` ENUM('call','meeting','task','email') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'task',
  `status` ENUM('Planned','Held','Not Held','Completed','Cancelled') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'Planned',
  `direction` ENUM('Inbound','Outbound','N/A') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'N/A',
  `title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `description` TEXT COLLATE utf8mb4_general_ci,
  `date_start` DATETIME DEFAULT NULL,
  `date_end` DATETIME DEFAULT NULL,
  `duration_minutes` INT DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_by` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `remind` TINYINT(1) NOT NULL DEFAULT 0,
  `reminder_at` DATETIME DEFAULT NULL,
  `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY `idx_lead_id` (`lead_id`),
  KEY `idx_reminder_at` (`reminder_at`),
  KEY `idx_saved_from` (`saved_from`),
  CONSTRAINT `fk_crm_activities_lead_id` FOREIGN KEY (`lead_id`) REFERENCES `leads`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
if(!col_exists($conn,'crm_activities','remind')) @mysqli_query($conn,"ALTER TABLE `crm_activities` ADD COLUMN `remind` TINYINT(1) NOT NULL DEFAULT 0 AFTER `created_by`");
if(!col_exists($conn,'crm_activities','reminder_at')) {@mysqli_query($conn,"ALTER TABLE `crm_activities` ADD COLUMN `reminder_at` DATETIME DEFAULT NULL AFTER `remind`"); @mysqli_query($conn,"CREATE INDEX `idx_reminder_at` ON `crm_activities` (`reminder_at`)");}
if(!col_exists($conn,'crm_activities','saved_from')) {@mysqli_query($conn,"ALTER TABLE `crm_activities` ADD COLUMN `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL AFTER `reminder_at`"); @mysqli_query($conn,"CREATE INDEX `idx_saved_from` ON `crm_activities` (`saved_from`)");}

/* ------------------------------- AJAX: save_activity ------------------------------- */
if(isset($_POST['ajax']) && $_POST['ajax']==='save_activity'){
  header('Content-Type: application/json');
  if(!$CAN_LOG_ACTIVITIES){ echo json_encode(['ok'=>false,'msg'=>'Not authorized to log activities']); exit; }

  $lead_id = (int)($_POST['id'] ?? 0);
  if($lead_id<=0){ echo json_encode(['ok'=>false,'msg'=>'Invalid lead']); exit; }
  $type = strtolower(trim($_POST['type'] ?? 'task')); if(!in_array($type,['call','meeting','task','email'],true)) $type = 'task';
  $status = trim($_POST['status'] ?? 'Planned'); if(!in_array($status,['Planned','Held','Not Held','Completed','Cancelled'],true)) $status = 'Planned';
  $direction = trim($_POST['direction'] ?? 'N/A'); if(!in_array($direction,['Inbound','Outbound','N/A'],true)) $direction = 'N/A';
  $duration_minutes = max(0,(int)($_POST['duration_minutes'] ?? 5));
  $date_end_raw = trim($_POST['date_end'] ?? '');
  $date_end = $date_end_raw ? date('Y-m-d H:i:00', strtotime($date_end_raw)) : date('Y-m-d H:i:00');
  $date_start = date('Y-m-d H:i:00', strtotime($date_end) - ($duration_minutes*60));
  $title = trim($_POST['title'] ?? '');
  $description = trim($_POST['description'] ?? '');
  $remind = isset($_POST['remind']) ? (int)$_POST['remind'] : 0;
  $reminder_at_raw = trim($_POST['reminder_at'] ?? '');
  $reminder_at = ($remind && $reminder_at_raw) ? date('Y-m-d H:i:00', strtotime($reminder_at_raw)) : NULL;

  // get lead data
  $assigned_user = NULL; $lead_name=''; $stage='Prospecting';
  $qr = mysqli_query($conn,"SELECT assigned_user, contact_name, opp_stage FROM leads WHERE id=".$lead_id." LIMIT 1");
  if($qr && mysqli_num_rows($qr)){
    $lr = mysqli_fetch_assoc($qr);
    $assigned_user = $lr['assigned_user'];
    $lead_name = $lr['contact_name'];
    if(!empty($lr['opp_stage'])) $stage = $lr['opp_stage'];
  }
  if($title===''){ $title = ucfirst($type).' - '.$lead_name; }
  $saved_from = ($stage==='Prospecting'?'Prospect':$stage).' Opportunities';
  $created_by = isset($_SESSION['namevfied1']) ? (string)$_SESSION['namevfied1'] : (isset($_SESSION['user_id1']) ? (string)$_SESSION['user_id1'] : NULL);

  $sql = "INSERT INTO `crm_activities`
    (`lead_id`,`activity_type`,`status`,`direction`,`title`,`description`,`date_start`,`date_end`,`duration_minutes`,`assigned_user`,`created_by`,`remind`,`reminder_at`,`saved_from`)
    VALUES (".
      (int)$lead_id.", '".
      mysqli_real_escape_string($conn,$type)."', '".
      mysqli_real_escape_string($conn,$status)."', '".
      mysqli_real_escape_string($conn,$direction)."', '".
      mysqli_real_escape_string($conn,$title)."', '".
      mysqli_real_escape_string($conn,$description)."', '".
      mysqli_real_escape_string($conn,$date_start)."', '".
      mysqli_real_escape_string($conn,$date_end)."', ".
      (int)$duration_minutes.", ".
      ($assigned_user===NULL ? "NULL" : "'".mysqli_real_escape_string($conn,$assigned_user)."'").", ".
      ($created_by===NULL ? "NULL" : "'".mysqli_real_escape_string($conn,$created_by)."'").", ".
      (int)$remind.", ".
      ($reminder_at===NULL ? "NULL" : "'".mysqli_real_escape_string($conn,$reminder_at)."'").", '".
      mysqli_real_escape_string($conn,$saved_from)."'".
    ")";
  $ok = mysqli_query($conn,$sql);
  if($ok){ echo json_encode(['ok'=>true,'id'=>mysqli_insert_id($conn)]); }
  else { echo json_encode(['ok'=>false,'msg'=>'DB error','err'=>mysqli_error($conn),'sql'=>$sql]); }
  exit;
}

/* --------------------------- Stage form (POST non-AJAX) --------------------------- */
if($_SERVER['REQUEST_METHOD']==='POST' && ($_POST['action'] ?? '')==='update_stage_form'){
  $id = (int)($_POST['id'] ?? 0);
  $stage = trim($_POST['opp_stage'] ?? '');
  if($id>0 && in_array($stage,$STAGES,true)){
    $stmt = mysqli_prepare($conn,"UPDATE `leads` SET `opp_stage`=? WHERE `id`=?");
    mysqli_stmt_bind_param($stmt,'si',$stage,$id);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
  }
  header("Location: ".$_SERVER['PHP_SELF']."?mode=".urlencode($_GET['mode'] ?? 'list')); exit;
}

/* ------------------------------ Tags form (POST) ------------------------------ */
$flash = '';
if($_SERVER['REQUEST_METHOD']==='POST' && empty($_POST['ajax']) && ($_POST['action'] ?? '')==='update'){
  $id = (int)($_POST['id'] ?? 0);
  $tagsArr = isset($_POST['tags_full']) ? (array)$_POST['tags_full'] : [];
  $tagsCsv = implode(',', array_map('trim', $tagsArr));
  if($id>0){
    $stmt = mysqli_prepare($conn,"UPDATE `leads` SET `tags`=? WHERE `id`=?");
    mysqli_stmt_bind_param($stmt,'si',$tagsCsv,$id);
    $flash = mysqli_stmt_execute($stmt) ? 'Tags updated.' : 'Error updating tags.';
    mysqli_stmt_close($stmt);
  }
  header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)."&mode=".urlencode($_GET['mode']??'list'));
  exit;
}

/* ------------------------------ Mode & fetching ------------------------------- */
$ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
$isMobileUA = (bool)preg_match('/Mobile|Android|iPhone|iPad|iPod|IEMobile|Opera Mini/i', $ua);
$mode = ($_GET['mode'] ?? 'board') === 'list' ? 'list' : 'board';
if($isMobileUA && $mode==='board'){ $mode = 'list'; }

$perPageOptions = [10,25,50,500];
$perPage = (int)($_GET['per_page'] ?? 25);
if(!in_array($perPage,$perPageOptions,true)) $perPage = 25;
$page   = max(1,(int)($_GET['page'] ?? 1));
$offset = ($page-1)*$perPage;

$whereProspect = "FIND_IN_SET('Prospect', REPLACE(COALESCE(`tags`,''), ' ', ''))";
$orderBy = "ORDER BY `updated_at` DESC";

$totalRows = 0;
$cntRes = mysqli_query($conn, "SELECT COUNT(*) AS c FROM `leads` WHERE $whereProspect");
if($cntRes){ $r = mysqli_fetch_assoc($cntRes); $totalRows = (int)$r['c']; }

$rows = [];
$res = mysqli_query($conn, "SELECT `id`,`contact_name`,`email`,`phone`,`lead_status`,`company`,`lead_source`,`tags`,`avatar`,`assigned_user`,
                                   `opp_title`,`opp_amount`,`opp_close_date`,`opp_stage`,`created_at`,`updated_at`
                            FROM `leads`
                            WHERE $whereProspect
                            $orderBy
                            ".($mode==='list' ? "LIMIT {$perPage} OFFSET {$offset}" : "")
);
if($res){ while($row = mysqli_fetch_assoc($res)){ $rows[] = $row; } }
$totalPages = max(1,(int)ceil($totalRows/$perPage));

$byStage = ['Prospecting'=>[], 'Qualification'=>[], 'Proposal'=>[], 'Negotiation'=>[], 'Closed Won'=>[]];
foreach($rows as $r){ $byStage[$r['opp_stage'] ?? 'Prospecting'][] = $r; }

$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));
$flash = $_GET['flash'] ?? $flash;
?>
<?php ob_start(); ?>
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/dragula/dragula.min.css">
<style>
@media (max-width: 991.98px){
  .app-sidebar, .main-sidebar, .sidemenu, [class*="sidebar"] { transform: translateX(-100%) !important; visibility: hidden !important; position: fixed !important; left: 0; top: 0; bottom: 0; z-index: 100; }
  .content, .main-content, .content-body, .page { margin-left: 0 !important; }
  .btn-mode-board { display:none !important; }
}
.quick-action-icon{ width:32px;height:32px;display:inline-flex;align-items:center;justify-content:center;border-radius:8px; }
.quick-action-icon i{ font-size:16px; }
a.opp-name { text-decoration:none; }
.stage-pill{ display:inline-block; padding:.35rem .8rem; border-radius:12px; color:#fff; font-weight:600; margin-right:.5rem; margin-bottom:.5rem; opacity:.95; font-size:.9rem; }
.stage-Prospecting{ background:#808080; } .stage-Qualification{ background:#808080; } .stage-Proposal{ background:#5b8fd9; } .stage-Negotiation{ background:#d99639; } .stage-Closed\ Won{ background:#6fcf97; }
.board { display:flex; gap:12px; align-items:flex-start; flex-wrap:nowrap; overflow-x:auto; }
.board-col { flex:1 1 0; min-width:230px; }
.board-head { border-radius:8px; padding:.35rem .6rem; font-weight:700; color:#fff; margin-bottom:.45rem; }
.board-Prospecting .board-head{ background:#808080; } .board-Qualification .board-head{ background:#808080; } .board-Proposal .board-head{ background:#5b8fd9; } .board-Negotiation .board-head{ background:#d99639; } .board-Closed\ Won .board-head{ background:#6fcf97; }
.board-body{ background:#f7f7f9; border:1px solid #eaeaea; border-radius:10px; padding:8px; min-height:120px; max-height:70vh; overflow:auto; }
.card-op { border:1px solid #e5e7eb; border-radius:10px; padding:10px 12px; background:#fff; margin-bottom:8px; box-shadow:0 1px 0 rgba(0,0,0,.03); cursor:grab; }
.card-op .title { font-weight:700; }
.card-op .company { color:#666; font-size:12px; }
.card-op .footer { color:#8a8a8a; font-size:12px; display:flex; justify-content:space-between; margin-top:6px; }
@media (max-width: 575.98px){ .table td, .table th { white-space:nowrap; } .table-responsive { overflow-x:auto; } }
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>

<!-- Header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Opportunities</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Opportunities</li>
    </ol>
  </div>
</div>

<?php if($flash): ?><div class="alert alert-success py-2 px-3 mb-3"><?php echo h($flash); ?></div><?php endif; ?>

<!-- Stage chips + Mode -->
<div class="d-flex align-items-center justify-content-between flex-wrap gap-2 mb-3">
  <div class="d-flex flex-wrap">
    <?php foreach($STAGES as $s): ?><span class="stage-pill stage-<?php echo str_replace(' ','\ ',$s); ?>"><?php echo h($s); ?></span><?php endforeach; ?>
  </div>
  <div class="btn-group">
    <a class="btn btn-outline-primary btn-mode-board <?php echo $mode==='board'?'active':''; ?>" href="<?php echo h($_SERVER['PHP_SELF']); ?>?mode=board">Board</a>
    <a class="btn btn-outline-primary <?php echo $mode==='list'?'active':''; ?>" href="<?php echo h($_SERVER['PHP_SELF']); ?>?mode=list">List</a>
  </div>
</div>

<?php if($mode==='board'): ?>
  <div class="board">
    <?php foreach($STAGES as $stage): ?>
    <div class="board-col board-<?php echo h($stage); ?>">
      <div class="board-head d-flex align-items-center justify-content-between">
        <span><?php echo h($stage); ?></span><span class="small opacity-75"><?php echo count($byStage[$stage]); ?></span>
      </div>
      <div class="board-body droppable" data-stage="<?php echo h($stage); ?>">
        <?php foreach($byStage[$stage] as $r):
          $avatar = $r['avatar'] ? ($baseUrl.'/'.ltrim($r['avatar'],'/')) : ($baseUrl.'/assets/images/faces/9.jpg');
          $date   = $r['opp_close_date'] ?: substr($r['created_at'],0,10);
        ?>
        <div class="card-op draggable" data-id="<?php echo (int)$r['id']; ?>">
          <div class="d-flex justify-content-between align-items-start">
            <a href="javascript:void(0);" class="opp-name lead-full"
               data-id="<?php echo (int)$r['id']; ?>"
               data-name="<?php echo h($r['contact_name']); ?>"
               data-email="<?php echo h($r['email']); ?>"
               data-phone="<?php echo h($r['phone']); ?>"
               data-status="<?php echo h($r['lead_status']); ?>"
               data-company="<?php echo h($r['company']); ?>"
               data-source="<?php echo h($r['lead_source']); ?>"
               data-tags="<?php echo h($r['tags']); ?>"
               data-avatar="<?php echo h($avatar); ?>"
               data-assigned="<?php echo h($r['assigned_user']); ?>"
               data-oppstage="<?php echo h($r['opp_stage']); ?>"
            ><div class="title"><?php echo h($r['contact_name']); ?></div></a>
          </div>
          <div class="company mt-1"><i class="ri-building-line me-1"></i><?php echo h($r['company'] ?: '—'); ?></div>
          <div class="footer"><span><i class="ri-user-line me-1"></i><?php echo h($r['assigned_user'] ?: 'Unassigned'); ?></span><span><?php echo h($date); ?></span></div>
          <div class="mt-2 d-flex gap-1">
            <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="call"    data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>" title="Call"><i class="ri-phone-fill"></i></button>
            <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="mail"    data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>" title="Mail"><i class="ri-mail-send-line"></i></button>
            <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="task"    data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>" title="Task"><i class="ri-task-line"></i></button>
            <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="meeting" data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>" title="Meeting"><i class="ri-calendar-event-line"></i></button>
          </div>
        </div>
        <?php endforeach; ?>
      </div>
    </div>
    <?php endforeach; ?>
  </div>
<?php else: ?>
  <div class="card custom-card">
    <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-3">
      <div class="card-title">Prospect Opportunities (<?php echo (int)$totalRows; ?>)</div>
      <form method="get" class="d-flex align-items-center gap-2">
        <input type="hidden" name="mode" value="list">
        <select name="per_page" class="form-select form-select-sm" onchange="this.form.submit()">
          <?php foreach($perPageOptions as $opt): ?><option value="<?php echo $opt; ?>" <?php echo $opt===$perPage?'selected':''; ?>><?php echo $opt; ?> / page</option><?php endforeach; ?>
        </select>
      </form>
    </div>
    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table align-middle text-nowrap">
          <thead><tr><th>Contact Name</th><th>Company</th><th>Stage</th><th>Assigned</th><th>Lead Source</th><th class="text-end pe-4">Actions</th></tr></thead>
          <tbody>
            <?php if(!$rows): ?>
              <tr><td colspan="6" class="text-center py-4">No opportunities found.</td></tr>
            <?php else: foreach($rows as $r):
              $avatar = $r['avatar'] ? ($baseUrl.'/'.ltrim($r['avatar'],'/')) : ($baseUrl.'/assets/images/faces/9.jpg');
            ?>
              <tr>
                <td>
                  <div class="d-flex align-items-center gap-2">
                    <span class="avatar avatar-sm avatar-rounded"><img src="<?php echo h($avatar); ?>" alt=""></span>
                    <a href="javascript:void(0);" class="fw-medium lead-full"
                       data-id="<?php echo (int)$r['id']; ?>"
                       data-name="<?php echo h($r['contact_name']); ?>"
                       data-email="<?php echo h($r['email']); ?>"
                       data-phone="<?php echo h($r['phone']); ?>"
                       data-status="<?php echo h($r['lead_status']); ?>"
                       data-company="<?php echo h($r['company']); ?>"
                       data-source="<?php echo h($r['lead_source']); ?>"
                       data-tags="<?php echo h($r['tags']); ?>"
                       data-avatar="<?php echo h($avatar); ?>"
                       data-assigned="<?php echo h($r['assigned_user']); ?>"
                       data-oppstage="<?php echo h($r['opp_stage']); ?>"
                    ><?php echo h($r['contact_name']); ?></a>
                  </div>
                </td>
                <td><?php echo h($r['company'] ?: '—'); ?></td>
                <td><a href="javascript:void(0);" class="badge bg-light text-dark stage-trigger" data-id="<?php echo (int)$r['id']; ?>" data-stage="<?php echo h($r['opp_stage']); ?>"><?php echo h($r['opp_stage']); ?></a></td>
                <td><?php echo h($r['assigned_user'] ?: '—'); ?></td>
                <td><?php echo h($r['lead_source'] ?: '—'); ?></td>
                <td class="text-end pe-4">
                  <div class="btn-list">
                    <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="call"    data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>"><i class="ri-phone-fill"></i></button>
                    <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="mail"    data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>"><i class="ri-mail-send-line"></i></button>
                    <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="task"    data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>"><i class="ri-task-line"></i></button>
                    <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="meeting" data-id="<?php echo (int)$r['id']; ?>" data-name="<?php echo h($r['contact_name']); ?>" data-email="<?php echo h($r['email']); ?>" data-phone="<?php echo h($r['phone']); ?>" data-company="<?php echo h($r['company']); ?>"><i class="ri-calendar-event-line"></i></button>
                  </div>
                </td>
              </tr>
            <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
    <div class="card-footer border-top-0">
      <div class="d-flex align-items-center flex-wrap w-100">
        <div>Showing <?php echo count($rows); ?> of <?php echo (int)$totalRows; ?> entries</div>
        <div class="ms-auto">
          <nav aria-label="Page navigation" class="pagination-style-4">
            <ul class="pagination mb-0">
              <li class="page-item <?php echo $page<=1?'disabled':''; ?>"><a class="page-link" href="<?php if($page>1){ echo h($_SERVER['PHP_SELF']).'?mode=list&page='.($page-1).'&per_page='.$perPage; } else { echo 'javascript:void(0);'; } ?>">Prev</a></li>
              <?php $start=max(1,$page-2); $end=min($totalPages,$page+2); for($p=$start;$p<=$end;$p++): ?>
                <li class="page-item <?php echo $p===$page?'active':''; ?>"><a class="page-link" href="<?php echo h($_SERVER['PHP_SELF']).'?mode=list&page='.$p.'&per_page='.$perPage; ?>"><?php echo $p; ?></a></li>
              <?php endfor; ?>
              <li class="page-item <?php echo $page>=$totalPages?'disabled':''; ?>"><a class="page-link" href="<?php if($page<$totalPages){ echo h($_SERVER['PHP_SELF']).'?mode=list&page='.($page+1).'&per_page='.$perPage; } else { echo 'javascript:void(0);'; } ?>">Next</a></li>
            </ul>
          </nav>
        </div>
      </div>
    </div>
  </div>
<?php endif; ?>

<!-- Stage Change Modal -->
<div class="modal fade" id="stageManageModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="update_stage_form">
      <input type="hidden" name="id" id="stage-id">
      <div class="modal-header"><h6 class="modal-title">Change Stage</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body px-4">
        <?php foreach($STAGES as $s): ?>
          <label class="form-check mb-2"><input class="form-check-input stage-radio" type="radio" name="opp_stage" value="<?php echo h($s); ?>"><span class="form-check-label"><?php echo h($s); ?></span></label>
        <?php endforeach; ?>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button><button type="submit" class="btn btn-primary">Save</button></div>
    </form>
  </div>
</div>

<!-- Activity Modal -->
<div class="modal fade" id="activityModal" tabindex="-1" aria-labelledby="activityModal" data-bs-keyboard="false" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header"><h6 class="modal-title">Log Activity</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body" id="activity-body"><div class="text-center text-muted">Select an action from the list/board.</div></div>
      <div class="modal-footer"><button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button><button type="button" class="btn btn-primary" id="activity-save">Save Activity</button></div>
    </div>
  </div>
</div>

<!-- Lead Full Modal (tags editable) -->
<div class="modal fade" id="leadFullModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-dialog-scrollable">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="update"><input type="hidden" name="id" id="full-id">
      <div class="modal-header"><h6 class="modal-title">Lead Details</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <div class="col-md-6"><label class="form-label">Contact Name</label><div id="v-name" class="form-control-plaintext border rounded px-2 py-1"></div></div>
          <div class="col-md-6"><label class="form-label">Assigned User</label><div id="v-assigned" class="form-control-plaintext border rounded px-2 py-1"></div></div>
          <div class="col-md-6"><label class="form-label">Email</label><div id="v-email" class="form-control-plaintext border rounded px-2 py-1"></div></div>
          <div class="col-md-6"><label class="form-label">Phone</label><div id="v-phone" class="form-control-plaintext border rounded px-2 py-1"></div></div>
          <div class="col-md-6"><label class="form-label">Company</label><div id="v-company" class="form-control-plaintext border rounded px-2 py-1"></div></div>
          <div class="col-md-6"><label class="form-label">Lead Source</label><div id="v-source" class="form-control-plaintext border rounded px-2 py-1"></div></div>
          <div class="col-md-6"><label class="form-label">Lead Status</label><div id="v-status" class="form-control-plaintext border rounded px-2 py-1"></div></div>
          <div class="col-12">
            <label class="form-label">Tag Status (tick to apply)</label>
            <div class="row">
              <?php foreach($TAG_STATUS as $tg): ?>
                <div class="col-6 col-md-4 mb-2">
                  <label class="form-check"><input class="form-check-input full-tags-check" type="checkbox" name="tags_full[]" value="<?php echo h($tg); ?>"><span class="form-check-label"><?php echo h($tg); ?></span></label>
                </div>
              <?php endforeach; ?>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Close</button><button type="submit" class="btn btn-primary">Save</button></div>
    </form>
  </div>
</div>

<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script src="<?php echo $baseUrl; ?>/assets/libs/dragula/dragula.min.js"></script>
<script>
(function(){
  // Mobile guard
  try{ if (window.innerWidth < 992) { const p = new URLSearchParams(location.search); if (p.get('mode') !== 'list') { p.set('mode','list'); location.replace(location.pathname+'?'+p.toString()); return; } document.querySelectorAll('[class*="sidebar"]').forEach(el=>el.classList.add('d-none')); } }catch(e){}

  function esc(s){ return (s||'').replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }

  // Lead view modal (tags editable)
  const fullEl = document.getElementById('leadFullModal');
  document.querySelectorAll('.lead-full').forEach(a=>{
    a.addEventListener('click', function(){
      const d=this.dataset;
      document.getElementById('full-id').value = d.id||'';
      document.getElementById('v-name').textContent = d.name||'';
      document.getElementById('v-assigned').textContent = d.assigned||'—';
      document.getElementById('v-email').textContent = d.email||'—';
      document.getElementById('v-phone').textContent = d.phone||'—';
      document.getElementById('v-company').textContent = d.company||'—';
      document.getElementById('v-source').textContent = d.source||'—';
      document.getElementById('v-status').textContent = d.status||'New';
      const sel = (d.tags||'').split(',').map(s=>s.trim());
      fullEl.querySelectorAll('.full-tags-check').forEach(ch => ch.checked = sel.includes(ch.value));
      new bootstrap.Modal(fullEl).show();
    });
  });

  // Stage change modal
  const stageModalEl=document.getElementById('stageManageModal');
  document.querySelectorAll('.stage-trigger').forEach(a=>{
    a.addEventListener('click', function(){
      const id=this.getAttribute('data-id'); const stage=this.getAttribute('data-stage')||'';
      document.getElementById('stage-id').value=id||'';
      stageModalEl.querySelectorAll('.stage-radio').forEach(r=>r.checked=(r.value===stage));
      new bootstrap.Modal(stageModalEl).show();
    });
  });

  // Activity modal (unified form + reminder)
  const activityModalEl=document.getElementById('activityModal');
  const activityBody=document.getElementById('activity-body');
  const activitySaveBtn=document.getElementById('activity-save');
  let currentActivity={type:null,id:null};

  function defaultEndISO(min){ const t=new Date(Date.now()+min*60000); const y=t.getFullYear(),m=String(t.getMonth()+1).padStart(2,'0'),d=String(t.getDate()).padStart(2,'0'); const hh=String(t.getHours()).padStart(2,'0'),mm=String(t.getMinutes()).padStart(2,'0'); return `${y}-${m}-${d}T${hh}:${mm}`; }
  function buildActivityForm(type,d){
    const isCall=(type==='call'); const pretty = type==='meeting'?'Meeting':type==='mail'?'Email':type==='call'?'Call':'Task';
    const endDefault=defaultEndISO(5);
    return `
      <div class="row g-2">
        <div class="col-md-6"><label class="form-label">Status</label>
          <select class="form-select" id="act-status">
            <option value="Planned">Planned</option><option value="Held">Held</option><option value="Not Held">Not Held</option><option value="Completed">Completed</option><option value="Cancelled">Cancelled</option>
          </select>
        </div>
        <div class="col-md-6"><label class="form-label">Direction</label>
          <select class="form-select" id="act-direction" ${isCall?'':'disabled'}>${isCall?'<option value="Outbound" selected>Outbound</option><option value="Inbound">Inbound</option><option value="N/A">N/A</option>':'<option value="N/A" selected>N/A</option>'}</select>
          ${isCall?'':'<div class="small text-muted mt-1">Direction applies to Calls only.</div>'}
        </div>
        <div class="col-md-6"><label class="form-label">Duration</label>
          <select class="form-select" id="act-duration"><option value="5">5m</option><option value="15">15m</option><option value="30">30m</option><option value="45">45m</option><option value="60">1h</option><option value="90">1h 30m</option><option value="120">2h</option></select>
        </div>
        <div class="col-md-6"><label class="form-label">Date End *</label><input type="datetime-local" class="form-control" id="act-dateend" value="${endDefault}"></div>
        <div class="col-12"><label class="form-label">Title</label><input type="text" class="form-control" id="act-title" value="${pretty + (d.name?(' - '+d.name):'')}"></div>
        <div class="col-12"><label class="form-label">Description</label><textarea class="form-control" id="act-desc" rows="4" placeholder="${pretty==='Meeting'?'Agenda / notes':'Notes'}"></textarea></div>
        <div class="col-12"><hr class="my-2"></div>
        <div class="col-md-6"><label class="form-label d-flex align-items-center gap-2"><input type="checkbox" id="act-remind"> Set Reminder</label></div>
        <div class="col-md-6"><label class="form-label">Reminder At</label><input type="datetime-local" class="form-control" id="act-remind-at" value="${endDefault}" disabled></div>
      </div>
    `;
  }

  document.addEventListener('click', function(ev){
    const btn=ev.target.closest('.btn-quick-action'); if(!btn) return;
    const d=btn.dataset;
    currentActivity={type:(d.action||'task'), id:(d.id||null)};
    activityModalEl.querySelector('.modal-title').textContent='Log '+(currentActivity.type==='call'?'Call':currentActivity.type==='meeting'?'Meeting':currentActivity.type==='mail'?'Email':'Task');
    activityBody.innerHTML=buildActivityForm(currentActivity.type,d);
    const cb=document.getElementById('act-remind'), at=document.getElementById('act-remind-at'); if(cb&&at){ at.disabled=!cb.checked; cb.addEventListener('change',()=>{ at.disabled=!cb.checked; }); }
    new bootstrap.Modal(activityModalEl).show();
  });

  if(activitySaveBtn){
    activitySaveBtn.addEventListener('click', function(){
      if(!currentActivity.id){ alert('Missing lead'); return; }
      const status=document.getElementById('act-status').value;
      const dirEl=document.getElementById('act-direction'); const direction=dirEl?(dirEl.disabled?'N/A':dirEl.value):'N/A';
      const duration=parseInt(document.getElementById('act-duration').value||'5',10);
      const dateend=document.getElementById('act-dateend').value;
      const desc=document.getElementById('act-desc').value||'';
      const title=document.getElementById('act-title').value||'';
      const remind=document.getElementById('act-remind').checked?1:0;
      const remindAt=document.getElementById('act-remind-at').value||'';
      const form=new URLSearchParams();
      form.append('ajax','save_activity'); form.append('id',currentActivity.id); form.append('type',currentActivity.type);
      form.append('status',status); form.append('direction',direction); form.append('duration_minutes', isNaN(duration)?5:duration);
      form.append('date_end', dateend.replace('T',' ')); form.append('description',desc); form.append('title',title);
      form.append('remind', String(remind)); form.append('reminder_at', remindAt ? remindAt.replace('T',' ') : '');
      fetch('<?php echo h($_SERVER["PHP_SELF"]); ?>', { method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body:form.toString() })
        .then(async r=>{ const t=await r.text(); let js=null; try{js=JSON.parse(t);}catch(e){} return {ok:r.ok, text:t, json:js}; })
        .then(res=>{ if(res.json && res.json.ok){ (bootstrap.Modal.getInstance(activityModalEl)||new bootstrap.Modal(activityModalEl)).hide(); alert('Activity saved.'); } else { alert((res.json && (res.json.err||res.json.msg)) ? (res.json.msg + (res.json.err?' — '+res.json.err:'')) : 'Save failed.'); console.log('RAW:',res.text); if(res.json&&res.json.sql) console.log('SQL:',res.json.sql); } })
        .catch(()=>alert('Network error'));
    });
  }

  // Dragula
  const cols=[...document.querySelectorAll('.board-body.droppable')];
  if(cols.length && window.innerWidth >= 992){
    const drake = dragula(cols, { revertOnSpill:true });
    drake.on('drop', function(el, target){
      const id=el.getAttribute('data-id'); const stage=target.getAttribute('data-stage'); if(!id||!stage) return;
      fetch('<?php echo h($_SERVER["PHP_SELF"]); ?>', { method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body:new URLSearchParams({ajax:'update_stage',id:id,stage:stage}) })
        .then(r=>r.json()).then(js=>{ if(!js.ok) alert('Failed to update stage'); }).catch(()=>alert('Network error'));
    });
  }
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
