<?php
/* ============================================================================
 * File: leads.php
 * Purpose: CRM Leads page + Activity logger (Call/Mail/Task/Meeting)
 * Stack : Core PHP + MySQLi (NO PDO). DB connection in ../config.php
 * Rules : utf8mb4_general_ci
 * Changes:
 *   - RBAC: leads.create, leads.export, activities.log
 *   - Tag Status set: Pending, Assigned, In Process, Prospect, Recycled, Dead
 *   - Do NOT show rows where Tag Status contains "Prospect" or "Dead"
 *   - Activities saved with saved_from = "In Process Lead"
 * ========================================================================== */

date_default_timezone_set('Asia/Kolkata');
@session_start();
require_once __DIR__ . '/../config.php';

mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function col_exists($conn,$table,$col){
  $t = mysqli_real_escape_string($conn,$table);
  $c = mysqli_real_escape_string($conn,$col);
  $q = mysqli_query($conn,"SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='{$t}' AND COLUMN_NAME='{$c}'");
  return ($q && mysqli_num_rows($q)>0);
}

/* ---------------------------- RBAC: schema & utils ---------------------------- */
$CURRENT_USER_ID = (int)($_SESSION['user_id1'] ?? 0);
function rbac_ensure_schema($conn){
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `roles` (`id` INT UNSIGNED NOT NULL AUTO_INCREMENT, `name` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL, `description` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL, `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP, `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP, PRIMARY KEY (`id`), UNIQUE KEY `uniq_role_name` (`name`)) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `permissions` (`code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL, `label` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL, `group_name` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL, PRIMARY KEY (`code`)) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `role_permissions` (`role_id` INT UNSIGNED NOT NULL, `perm_code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL, PRIMARY KEY (`role_id`,`perm_code`), KEY `idx_perm_code` (`perm_code`), CONSTRAINT `fk_role_permissions_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
  @mysqli_query($conn,"CREATE TABLE IF NOT EXISTS `user_roles` (`user_id` INT NOT NULL, `role_id` INT UNSIGNED NOT NULL, PRIMARY KEY (`user_id`,`role_id`), KEY `idx_role_id` (`role_id`), CONSTRAINT `fk_user_roles_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
}
function rbac_seed_permissions($conn){
  $perms=[['leads.create','Create leads','Leads'],['leads.export','Export leads (CSV)','Leads'],['activities.log','Log activities','Activities'],['roles.manage','Manage roles','Admin']];
  foreach($perms as $p){ @mysqli_query($conn,"INSERT IGNORE INTO `permissions` (`code`,`label`,`group_name`) VALUES ('".mysqli_real_escape_string($conn,$p[0])."','".mysqli_real_escape_string($conn,$p[1])."','".mysqli_real_escape_string($conn,$p[2])."')"); }
}
function rbac_has($conn,$uid,$code){
  if(!$uid) return true; // superuser default
  $uid=(int)$uid; $code=mysqli_real_escape_string($conn,$code);
  $q=mysqli_query($conn,"SELECT 1 FROM user_roles ur JOIN role_permissions rp ON rp.role_id=ur.role_id WHERE ur.user_id={$uid} AND rp.perm_code='{$code}' LIMIT 1");
  return ($q && mysqli_num_rows($q)>0);
}
rbac_ensure_schema($conn); rbac_seed_permissions($conn);

$CAN_CREATE_LEADS   = rbac_has($conn,$CURRENT_USER_ID,'leads.create');
$CAN_EXPORT_LEADS   = rbac_has($conn,$CURRENT_USER_ID,'leads.export');
$CAN_LOG_ACTIVITIES = rbac_has($conn,$CURRENT_USER_ID,'activities.log');

$TAG_STATUS = ['Pending','Assigned','In Process','Prospect','Recycled','Dead'];

/* ---------- Ensure `leads` table exists ---------- */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `leads` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `contact_name` VARCHAR(150) COLLATE utf8mb4_general_ci NOT NULL,
  `email` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `phone` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_status` ENUM('New','Follow-up','Closed','Contacted','Disqualified','Qualified') COLLATE utf8mb4_general_ci DEFAULT 'New',
  `company` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_source` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `tags` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `avatar` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
if(!col_exists($conn,'leads','assigned_user'))  @mysqli_query($conn,"ALTER TABLE `leads` ADD COLUMN `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL");

/* --------------------------- Activities table (ensure) --------------------------- */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `crm_activities` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_id` INT UNSIGNED NOT NULL,
  `activity_type` ENUM('call','meeting','task','email') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'task',
  `status` ENUM('Planned','Held','Not Held','Completed','Cancelled') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'Planned',
  `direction` ENUM('Inbound','Outbound','N/A') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'N/A',
  `title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `description` TEXT COLLATE utf8mb4_general_ci,
  `date_start` DATETIME DEFAULT NULL,
  `date_end` DATETIME DEFAULT NULL,
  `duration_minutes` INT DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_by` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `remind` TINYINT(1) NOT NULL DEFAULT 0,
  `reminder_at` DATETIME DEFAULT NULL,
  `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY `idx_lead_id` (`lead_id`),
  KEY `idx_reminder_at` (`reminder_at`),
  KEY `idx_saved_from` (`saved_from`),
  CONSTRAINT `fk_crm_activities_lead_id` FOREIGN KEY (`lead_id`) REFERENCES `leads`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
if(!col_exists($conn,'crm_activities','remind')) @mysqli_query($conn,"ALTER TABLE `crm_activities` ADD COLUMN `remind` TINYINT(1) NOT NULL DEFAULT 0 AFTER `created_by`");
if(!col_exists($conn,'crm_activities','reminder_at')) {@mysqli_query($conn,"ALTER TABLE `crm_activities` ADD COLUMN `reminder_at` DATETIME DEFAULT NULL AFTER `remind`"); @mysqli_query($conn,"CREATE INDEX `idx_reminder_at` ON `crm_activities` (`reminder_at`)");}
if(!col_exists($conn,'crm_activities','saved_from')) {@mysqli_query($conn,"ALTER TABLE `crm_activities` ADD COLUMN `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL AFTER `reminder_at`"); @mysqli_query($conn,"CREATE INDEX `idx_saved_from` ON `crm_activities` (`saved_from`)");}

/* ---------- File Upload Helper ---------- */
function save_lead_avatar($fileInputName, $oldPath = null){
  if(!isset($_FILES[$fileInputName]) || !is_uploaded_file($_FILES[$fileInputName]['tmp_name'])) return $oldPath;
  $f = $_FILES[$fileInputName]; if($f['error'] !== UPLOAD_ERR_OK) return $oldPath;
  $ext = strtolower(pathinfo($f['name'], PATHINFO_EXTENSION)); $allowed = ['jpg','jpeg','png','gif','webp']; if(!in_array($ext,$allowed,true)) return $oldPath;
  $dir = realpath(__DIR__ . '/../uploads'); if(!$dir){ @mkdir(__DIR__ . '/../uploads', 0775, true); $dir = realpath(__DIR__ . '/../uploads'); }
  if($dir){ $dir .= '/leads'; if(!file_exists($dir)) @mkdir($dir, 0775, true); }
  $newName = 'lead_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
  $destFs = $dir . '/' . $newName;
  if(@move_uploaded_file($f['tmp_name'], $destFs)){ if($oldPath && is_file(__DIR__ . '/../' . ltrim($oldPath, '/'))){ @unlink(__DIR__ . '/../' . ltrim($oldPath, '/')); } return 'uploads/leads/' . $newName; }
  return $oldPath;
}

/* ---------- CSV Export (permission check first) ---------- */
if(isset($_GET['export']) && $_GET['export'] === 'csv'){
  if(!$CAN_EXPORT_LEADS){
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode('Not authorized to export.')); exit;
  }
  $orderBy = "ORDER BY `created_at` DESC";
  $res = mysqli_query($conn, "SELECT `id`,`contact_name`,`email`,`phone`,`lead_status`,`company`,`lead_source`,`tags`,`assigned_user`,`created_at` FROM `leads` {$orderBy}");
  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename=leads_'.date('Ymd_His').'.csv');
  $out = fopen('php://output', 'w');
  fputcsv($out, ['ID','Contact Name','Email','Phone','Lead Status','Company','Lead Source','Tags','Assigned User','Created At']);
  if($res){ while($r = mysqli_fetch_assoc($res)){ fputcsv($out, [$r['id'],$r['contact_name'],$r['email'],$r['phone'],$r['lead_status'],$r['company'],$r['lead_source'],$r['tags'],$r['assigned_user'],$r['created_at']]); } }
  fclose($out); exit;
}

/* ---------- Handle Create / Update / Delete / Update_Tags ---------- */
$flash = '';
if($_SERVER['REQUEST_METHOD'] === 'POST' && empty($_POST['ajax'])){
  $action = $_POST['action'] ?? '';
  if($action === 'create'){
    if(!$CAN_CREATE_LEADS){ header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode('Not authorized to create leads.')); exit; }
    $name  = trim($_POST['contact_name'] ?? ''); $email = trim($_POST['email'] ?? ''); $phone = trim($_POST['phone'] ?? '');
    $company = trim($_POST['company'] ?? ''); $status  = trim($_POST['lead_status'] ?? 'New');
    $source  = trim($_POST['lead_source'] ?? ''); $assigned = trim($_POST['assigned_user'] ?? '');
    $tagsArr = isset($_POST['tags']) ? (array)$_POST['tags'] : []; $tagsCsv = implode(',', array_map('trim', $tagsArr));
    $avatar = save_lead_avatar('photo', null);
    $stmt = mysqli_prepare($conn, "INSERT INTO `leads` (`contact_name`,`email`,`phone`,`lead_status`,`company`,`lead_source`,`tags`,`avatar`,`assigned_user`) VALUES (?,?,?,?,?,?,?,?,?)");
    mysqli_stmt_bind_param($stmt, 'sssssssss', $name,$email,$phone,$status,$company,$source,$tagsCsv,$avatar,$assigned);
    $flash = mysqli_stmt_execute($stmt) ? 'Lead created successfully.' : 'Error creating lead.'; mysqli_stmt_close($stmt);
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)); exit;
  }
  if($action === 'update'){
    $id=(int)($_POST['id'] ?? 0); $name=trim($_POST['contact_name'] ?? ''); $email=trim($_POST['email'] ?? ''); $phone=trim($_POST['phone'] ?? '');
    $company=trim($_POST['company'] ?? ''); $status=trim($_POST['lead_status'] ?? 'New'); $source=trim($_POST['lead_source'] ?? '');
    $assigned=trim($_POST['assigned_user'] ?? '');
    $tagsArr = isset($_POST['tags']) ? (array)$_POST['tags'] : (isset($_POST['tags_full'])?(array)$_POST['tags_full']:[]);
    $tagsCsv = implode(',', array_map('trim', $tagsArr));
    $currentAvatar = null; $rr=mysqli_query($conn,"SELECT `avatar` FROM `leads` WHERE `id`={$id} LIMIT 1"); if($rr && mysqli_num_rows($rr)){ $row=mysqli_fetch_assoc($rr); $currentAvatar=$row['avatar']; }
    $avatar = save_lead_avatar('photo_edit', $currentAvatar); if(!$avatar && isset($_POST['photo_existing'])){ $avatar = $_POST['photo_existing']; }
    $stmt = mysqli_prepare($conn, "UPDATE `leads` SET `contact_name`=?,`email`=?,`phone`=?,`lead_status`=?,`company`=?,`lead_source`=?,`tags`=?,`avatar`=?,`assigned_user`=? WHERE `id`=?");
    mysqli_stmt_bind_param($stmt, 'sssssssssi', $name,$email,$phone,$status,$company,$source,$tagsCsv,$avatar,$assigned,$id);
    $flash = mysqli_stmt_execute($stmt) ? 'Lead updated successfully.' : 'Error updating lead.'; mysqli_stmt_close($stmt);
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)); exit;
  }
  if($action === 'delete'){
    $id = (int)($_POST['id'] ?? 0);
    $rr = mysqli_query($conn, "SELECT `avatar` FROM `leads` WHERE `id`={$id} LIMIT 1");
    if($rr && mysqli_num_rows($rr)){ $row = mysqli_fetch_assoc($rr); if($row['avatar'] && is_file(__DIR__ . '/../' . ltrim($row['avatar'],'/'))){ @unlink(__DIR__ . '/../' . ltrim($row['avatar'],'/')); } }
    mysqli_query($conn, "DELETE FROM `leads` WHERE `id`={$id} LIMIT 1"); $flash = 'Lead deleted.';
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)); exit;
  }
  if($action === 'update_tags'){
    $id = (int)($_POST['id'] ?? 0); $tagsArr = isset($_POST['tags']) ? (array)$_POST['tags'] : []; $tagsCsv = implode(',', array_map('trim', $tagsArr));
    $stmt = mysqli_prepare($conn, "UPDATE `leads` SET `tags`=? WHERE `id`=?"); mysqli_stmt_bind_param($stmt, 'si', $tagsCsv, $id);
    $flash = mysqli_stmt_execute($stmt) ? 'Tags updated.' : 'Error updating tags.'; mysqli_stmt_close($stmt);
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)."&page=".(int)($_GET['page'] ?? 1)."&per_page=".(int)($_GET['per_page'] ?? 10)."&sort=".urlencode($_GET['sort'] ?? 'newest')); exit;
  }
}

/* ------------------------------- AJAX: save_activity ------------------------------- */
if(isset($_POST['ajax']) && $_POST['ajax']==='save_activity'){
  header('Content-Type: application/json');
  if(!$CAN_LOG_ACTIVITIES){ echo json_encode(['ok'=>false,'msg'=>'Not authorized to log activities']); exit; }

  $lead_id = (int)($_POST['id'] ?? 0); if($lead_id<=0){ echo json_encode(['ok'=>false,'msg'=>'Invalid lead']); exit; }
  $type = strtolower(trim($_POST['type'] ?? 'task')); if(!in_array($type,['call','meeting','task','email'],true)) $type='task';
  $status = trim($_POST['status'] ?? 'Planned'); if(!in_array($status,['Planned','Held','Not Held','Completed','Cancelled'],true)) $status='Planned';
  $direction = trim($_POST['direction'] ?? 'N/A'); if(!in_array($direction,['Inbound','Outbound','N/A'],true)) $direction='N/A';
  $duration_minutes = max(0,(int)$_POST['duration_minutes']);
  $date_end_raw = trim($_POST['date_end'] ?? ''); $date_end = $date_end_raw ? date('Y-m-d H:i:00', strtotime($date_end_raw)) : date('Y-m-d H:i:00');
  $date_start = date('Y-m-d H:i:00', strtotime($date_end) - ($duration_minutes*60));
  $title = trim($_POST['title'] ?? ''); $description = trim($_POST['description'] ?? '');
  $remind = isset($_POST['remind']) ? (int)$_POST['remind'] : 0;
  $reminder_at_raw = trim($_POST['reminder_at'] ?? ''); $reminder_at = ($remind && $reminder_at_raw) ? date('Y-m-d H:i:00', strtotime($reminder_at_raw)) : NULL;

  $assigned_user = NULL; $lead_name='';
  $qr = mysqli_query($conn,"SELECT assigned_user, contact_name FROM leads WHERE id=".$lead_id." LIMIT 1");
  if($qr && mysqli_num_rows($qr)){ $lr=mysqli_fetch_assoc($qr); $assigned_user=$lr['assigned_user']; $lead_name=$lr['contact_name']; }
  if($title===''){ $title = ucfirst($type).' - '.$lead_name; }
  $created_by = isset($_SESSION['namevfied1']) ? (string)$_SESSION['namevfied1'] : (isset($_SESSION['user_id1']) ? (string)$_SESSION['user_id1'] : NULL);
  $saved_from = 'In Process Lead';

  $sql = "INSERT INTO `crm_activities`
    (`lead_id`,`activity_type`,`status`,`direction`,`title`,`description`,`date_start`,`date_end`,`duration_minutes`,`assigned_user`,`created_by`,`remind`,`reminder_at`,`saved_from`)
    VALUES (".
      (int)$lead_id.", '".
      mysqli_real_escape_string($conn,$type)."', '".
      mysqli_real_escape_string($conn,$status)."', '".
      mysqli_real_escape_string($conn,$direction)."', '".
      mysqli_real_escape_string($conn,$title)."', '".
      mysqli_real_escape_string($conn,$description)."', '".
      mysqli_real_escape_string($conn,$date_start)."', '".
      mysqli_real_escape_string($conn,$date_end)."', ".
      (int)$duration_minutes.", ".
      ($assigned_user===NULL ? "NULL" : "'".mysqli_real_escape_string($conn,$assigned_user)."'").", ".
      ($created_by===NULL ? "NULL" : "'".mysqli_real_escape_string($conn,$created_by)."'").", ".
      (int)$remind.", ".
      ($reminder_at===NULL ? "NULL" : "'".mysqli_real_escape_string($conn,$reminder_at)."'").", '".
      mysqli_real_escape_string($conn,$saved_from)."'".
    ")";
  $ok = mysqli_query($conn,$sql);
  if($ok){ echo json_encode(['ok'=>true,'id'=>mysqli_insert_id($conn)]); }
  else { echo json_encode(['ok'=>false,'msg'=>'DB error','err'=>mysqli_error($conn),'sql'=>$sql]); }
  exit;
}

/* ---------- Pagination + Sorting + FILTER (hide Prospect/Dead) ---------- */
$perPageOptions = [10,25,50,500];
$perPage = (int)($_GET['per_page'] ?? 10); if(!in_array($perPage,$perPageOptions,true)) $perPage=10;
$page   = max(1,(int)($_GET['page'] ?? 1)); $offset = ($page-1)*$perPage;
$sortParam = strtolower(trim($_GET['sort'] ?? 'newest'));
switch($sortParam){case 'az': $orderBy="ORDER BY `contact_name` ASC, `id` DESC"; break; case 'date': $orderBy="ORDER BY `created_at` DESC"; break; default: $orderBy="ORDER BY `created_at` DESC";}
$hideProspectDead = "NOT FIND_IN_SET('Prospect', REPLACE(COALESCE(`tags`,''), ' ', '')) AND NOT FIND_IN_SET('Dead', REPLACE(COALESCE(`tags`,''), ' ', ''))";

$totalRows = 0;
$cntRes = mysqli_query($conn, "SELECT COUNT(*) AS c FROM `leads` WHERE {$hideProspectDead}");
if($cntRes){ $r = mysqli_fetch_assoc($cntRes); $totalRows = (int)$r['c']; }
$leads = [];
$sql = "SELECT `id`,`contact_name`,`email`,`phone`,`lead_status`,`company`,`lead_source`,`tags`,`avatar`,`assigned_user`,`created_at`
        FROM `leads` WHERE {$hideProspectDead} {$orderBy} LIMIT {$perPage} OFFSET {$offset}";
$res = mysqli_query($conn, $sql); if($res){ while($row = mysqli_fetch_assoc($res)){ $leads[] = $row; } }
$totalPages = max(1, (int)ceil($totalRows / $perPage));

$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));
$flash = $_GET['flash'] ?? $flash;
?>
<?php ob_start(); ?>
<style>
.badge-soft { background: rgba(0,0,0,.04); color: #222; }
.quick-action-icon { width: 32px; height: 32px; display: inline-flex; align-items:center; justify-content:center; border-radius:8px; }
.quick-action-icon i { font-size:16px; }
.tags-cell { cursor: pointer; }
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>

<!-- Header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Leads</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Leads</li>
    </ol>
  </div>
</div>

<?php if($flash): ?><div class="alert alert-success py-2 px-3 mb-3"><?php echo h($flash); ?></div><?php endif; ?>

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card">
      <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-3">
        <div class="card-title">Leads</div>
        <div class="d-flex flex-wrap gap-2 align-items-center">
          <?php if($CAN_CREATE_LEADS): ?>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createLeadModal"><i class="ri-add-line me-1 fw-medium align-middle"></i>Create Lead</button>
          <?php endif; ?>
          <?php if($CAN_EXPORT_LEADS): ?>
            <a class="btn btn-success" href="<?php echo h($_SERVER['PHP_SELF']); ?>?export=csv">Export As CSV</a>
          <?php endif; ?>
          <div class="dropdown">
            <a href="javascript:void(0);" class="btn btn-light btn-wave waves-effect waves-light" data-bs-toggle="dropdown" aria-expanded="false">
              Sort By<i class="ri-arrow-down-s-line align-middle ms-1 d-inline-block"></i>
            </a>
            <ul class="dropdown-menu" role="menu">
              <li><a class="dropdown-item" href="<?php echo h($_SERVER['PHP_SELF']); ?>?sort=newest&per_page=<?php echo $perPage; ?>">Newest</a></li>
              <li><a class="dropdown-item" href="<?php echo h($_SERVER['PHP_SELF']); ?>?sort=date&per_page=<?php echo $perPage; ?>">Date Added</a></li>
              <li><a class="dropdown-item" href="<?php echo h($_SERVER['PHP_SELF']); ?>?sort=az&per_page=<?php echo $perPage; ?>">A - Z</a></li>
            </ul>
          </div>
          <form method="get" class="d-flex align-items-center gap-2">
            <input type="hidden" name="sort" value="<?php echo h($sortParam); ?>">
            <select name="per_page" class="form-select form-select-sm" onchange="this.form.submit()">
              <?php foreach($perPageOptions as $opt): ?><option value="<?php echo $opt; ?>" <?php echo $opt===$perPage?'selected':''; ?>><?php echo $opt; ?> / page</option><?php endforeach; ?>
            </select>
          </form>
        </div>
      </div>

      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table text-nowrap align-middle">
            <thead>
              <tr>
                <th scope="col"><input class="form-check-input" type="checkbox" id="checkAll"></th>
                <th scope="col">Contact Name</th>
                <th scope="col">Email</th>
                <th scope="col">Phone</th>
                <th scope="col">Lead Source</th>
                <th scope="col">Tags</th>
                <th scope="col">Assigned User</th>
                <th scope="col" class="text-end pe-4">Actions</th>
              </tr>
            </thead>
            <tbody>
              <?php if(!$leads): ?>
                <tr><td colspan="8" class="text-center py-4">No leads found.</td></tr>
              <?php else: foreach($leads as $L): ?>
              <?php
                $avatar = $L['avatar'] ? ($baseUrl . '/' . ltrim($L['avatar'],'/')) : ($baseUrl . '/assets/images/faces/9.jpg');
                $tagBadges = [];
                if(!empty($L['tags'])){
                  foreach(explode(',', $L['tags']) as $tg){
                    $tg = trim($tg); if($tg==='') continue;
                    $tagBadges[] = '<span class="badge bg-primary-transparent me-1 mb-1">'.h($tg).'</span>';
                  }
                }
                $assignedUser = trim((string)$L['assigned_user']);
              ?>
              <tr class="crm-contact">
                <td><input class="form-check-input row-check" type="checkbox" value="<?php echo (int)$L['id']; ?>"></td>
                <td>
                  <div class="d-flex align-items-center gap-2">
                    <span class="avatar avatar-rounded avatar-sm"><img src="<?php echo h($avatar); ?>" alt=""></span>
                    <a href="javascript:void(0);" class="d-block fw-medium lead-full"
                       data-id="<?php echo (int)$L['id']; ?>"
                       data-name="<?php echo h($L['contact_name']); ?>"
                       data-email="<?php echo h($L['email']); ?>"
                       data-phone="<?php echo h($L['phone']); ?>"
                       data-status="<?php echo h($L['lead_status']); ?>"
                       data-company="<?php echo h($L['company']); ?>"
                       data-source="<?php echo h($L['lead_source']); ?>"
                       data-tags="<?php echo h($L['tags']); ?>"
                       data-avatar="<?php echo h($avatar); ?>"
                       data-assigned="<?php echo h($assignedUser); ?>"
                    ><?php echo h($L['contact_name']); ?></a>
                  </div>
                </td>
                <td><span><i class="ri-mail-line me-2 text-muted"></i><?php echo h($L['email']); ?></span></td>
                <td><span><i class="ri-phone-line me-2 text-muted"></i><?php echo h($L['phone']); ?></span></td>
                <td><?php echo h($L['lead_source']); ?></td>
                <td class="tags-cell" data-id="<?php echo (int)$L['id']; ?>" data-tags="<?php echo h($L['tags']); ?>">
                  <div class="d-flex align-items-center flex-wrap">
                    <?php echo $tagBadges ? implode(' ', $tagBadges) : '<span class="text-muted">No Tags</span>'; ?>
                    <a href="javascript:void(0);" class="ms-2 small text-primary tags-trigger">Change</a>
                  </div>
                </td>
                <td><?php echo $assignedUser !== '' ? h($assignedUser) : '<span class="text-muted">—</span>'; ?></td>
                <td class="text-end pe-4">
                  <div class="btn-list">
                    <?php if($CAN_LOG_ACTIVITIES): ?>
                      <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="call"    data-id="<?php echo (int)$L['id']; ?>" data-name="<?php echo h($L['contact_name']); ?>" data-email="<?php echo h($L['email']); ?>" data-phone="<?php echo h($L['phone']); ?>" data-company="<?php echo h($L['company']); ?>"><i class="ri-phone-fill"></i></button>
                      <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="mail"    data-id="<?php echo (int)$L['id']; ?>" data-name="<?php echo h($L['contact_name']); ?>" data-email="<?php echo h($L['email']); ?>" data-phone="<?php echo h($L['phone']); ?>" data-company="<?php echo h($L['company']); ?>"><i class="ri-mail-send-line"></i></button>
                      <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="task"    data-id="<?php echo (int)$L['id']; ?>" data-name="<?php echo h($L['contact_name']); ?>" data-email="<?php echo h($L['email']); ?>" data-phone="<?php echo h($L['phone']); ?>" data-company="<?php echo h($L['company']); ?>"><i class="ri-task-line"></i></button>
                      <button type="button" class="btn btn-sm btn-light quick-action-icon btn-quick-action" data-action="meeting" data-id="<?php echo (int)$L['id']; ?>" data-name="<?php echo h($L['contact_name']); ?>" data-email="<?php echo h($L['email']); ?>" data-phone="<?php echo h($L['phone']); ?>" data-company="<?php echo h($L['company']); ?>"><i class="ri-calendar-event-line"></i></button>
                    <?php else: ?>
                      <span class="text-muted small">No activity permission</span>
                    <?php endif; ?>

                    <button type="button" class="btn btn-sm btn-warning-light btn-icon btn-view" data-id="<?php echo (int)$L['id']; ?>" data-name="<?php echo h($L['contact_name']); ?>" data-email="<?php echo h($L['email']); ?>" data-phone="<?php echo h($L['phone']); ?>" data-status="<?php echo h($L['lead_status']); ?>" data-company="<?php echo h($L['company']); ?>" data-source="<?php echo h($L['lead_source']); ?>" data-tags="<?php echo h($L['tags']); ?>" data-avatar="<?php echo h($avatar); ?>"><i class="ri-eye-line"></i></button>
                    <button type="button" class="btn btn-sm btn-info-light btn-icon btn-edit" data-id="<?php echo (int)$L['id']; ?>" data-name="<?php echo h($L['contact_name']); ?>" data-email="<?php echo h($L['email']); ?>" data-phone="<?php echo h($L['phone']); ?>" data-status="<?php echo h($L['lead_status']); ?>" data-company="<?php echo h($L['company']); ?>" data-source="<?php echo h($L['lead_source']); ?>" data-tags="<?php echo h($L['tags']); ?>" data-avatar="<?php echo h($avatar); ?>" data-assigned="<?php echo h($assignedUser); ?>"><i class="ri-pencil-line"></i></button>

                    <form method="post" class="d-inline" onsubmit="return confirm('Delete this lead?');"><input type="hidden" name="action" value="delete"><input type="hidden" name="id" value="<?php echo (int)$L['id']; ?>"><button class="btn btn-sm btn-danger-light btn-icon" title="Delete"><i class="ri-delete-bin-line"></i></button></form>
                  </div>
                </td>
              </tr>
              <?php endforeach; endif; ?>
            </tbody>
          </table>
        </div>
      </div>

      <div class="card-footer border-top-0">
        <div class="d-flex align-items-center flex-wrap w-100">
          <div>Showing <?php echo count($leads); ?> of <?php echo (int)$totalRows; ?> entries</div>
          <div class="ms-auto">
            <nav aria-label="Page navigation" class="pagination-style-4">
              <ul class="pagination mb-0">
                <li class="page-item <?php echo $page<=1?'disabled':''; ?>"><a class="page-link" href="<?php if($page>1){ echo h($_SERVER['PHP_SELF']).'?page='.($page-1).'&per_page='.$perPage.'&sort='.$sortParam; } else { echo 'javascript:void(0);'; } ?>">Prev</a></li>
                <?php $start=max(1,$page-2); $end=min($totalPages,$page+2); for($p=$start;$p<=$end;$p++): ?>
                  <li class="page-item <?php echo $p===$page?'active':''; ?>"><a class="page-link" href="<?php echo h($_SERVER['PHP_SELF']).'?page='.$p.'&per_page='.$perPage.'&sort='.$sortParam; ?>"><?php echo $p; ?></a></li>
                <?php endfor; ?>
                <li class="page-item <?php echo $page>=$totalPages?'disabled':''; ?>"><a class="page-link text-primary" href="<?php if($page<$totalPages){ echo h($_SERVER['PHP_SELF']).'?page='.($page+1).'&per_page='.$perPage.'&sort='.$sortParam; } else { echo 'javascript:void(0);'; } ?>">next</a></li>
              </ul>
            </nav>
          </div>
        </div>
      </div>

    </div>
  </div>
</div>

<?php if($CAN_CREATE_LEADS): ?>
<!-- Create Lead Modal -->
<div class="modal fade" id="createLeadModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post" enctype="multipart/form-data">
      <input type="hidden" name="action" value="create">
      <div class="modal-header"><h6 class="modal-title">Create Lead</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <div class="col-xl-12 text-center">
            <span class="avatar avatar-xxl avatar-rounded">
              <img src="<?php echo $baseUrl; ?>/assets/images/faces/9.jpg" alt="" id="create-profile-preview">
              <span class="badge rounded-pill bg-primary avatar-badge">
                <input type="file" name="photo" class="position-absolute w-100 h-100 op-0" id="create-profile-file" accept="image/*">
                <i class="fe fe-camera"></i>
              </span>
            </span>
          </div>
          <div class="col-xl-12"><label class="form-label">Contact Name</label><input type="text" class="form-control" name="contact_name" required></div>
          <div class="col-xl-12"><label class="form-label">Email</label><input type="email" class="form-control" name="email"></div>
          <div class="col-xl-12"><label class="form-label">Phone No</label><input type="tel" class="form-control" name="phone"></div>
          <div class="col-xl-12"><label class="form-label">Company Name</label><input type="text" class="form-control" name="company"></div>
          <div class="col-xl-12"><label class="form-label">Lead Status</label>
            <select class="form-control" name="lead_status">
              <option value="New">New</option><option value="Follow-up">Follow-up</option><option value="Closed">Closed</option><option value="Contacted">Contacted</option><option value="Disqualified">Disqualified</option><option value="Qualified">Qualified</option>
            </select>
          </div>
          <div class="col-xl-12"><label class="form-label">Lead Source</label>
            <select class="form-control" name="lead_source">
              <option value="">Select Source</option><option value="Social Media">Social Media</option><option value="Direct Mail">Direct Mail</option><option value="Blog Articles">Blog Articles</option><option value="Affiliates">Affiliates</option><option value="Organic Search">Organic Search</option><option value="Event">Event</option><option value="Referral">Referral</option><option value="Web">Web</option>
            </select>
          </div>
          <div class="col-xl-12"><label class="form-label">Assigned User</label><input type="text" class="form-control" name="assigned_user" placeholder="e.g. Amit Kumar"></div>
          <div class="col-xl-12"><label class="form-label">Tag Status</label>
            <select class="form-control" name="tags[]" multiple>
              <?php foreach(['Pending','Assigned','In Process','Prospect','Recycled','Dead'] as $tg): ?><option><?php echo h($tg); ?></option><?php endforeach; ?>
            </select>
          </div>
        </div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button><button type="submit" class="btn btn-primary">Create Lead</button></div>
    </form>
  </div>
</div>
<?php endif; ?>
<!-- View Lead Modal -->
<div class="modal fade" id="viewLeadModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header"><h6 class="modal-title">Lead Details</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body px-4">
        <div class="text-center mb-3"><span class="avatar avatar-xxl avatar-rounded"><img id="view-avatar" src="<?php echo $baseUrl; ?>/assets/images/faces/9.jpg" alt=""></span></div>
        <div class="row gy-2">
          <div class="col-12"><strong>Name:</strong> <span id="view-name"></span></div>
          <div class="col-12"><strong>Email:</strong> <span id="view-email"></span></div>
          <div class="col-12"><strong>Phone:</strong> <span id="view-phone"></span></div>
          <div class="col-12"><strong>Status:</strong> <span id="view-status"></span></div>
          <div class="col-12"><strong>Company:</strong> <span id="view-company"></span></div>
          <div class="col-12"><strong>Source:</strong> <span id="view-source"></span></div>
          <div class="col-12"><strong>Tags:</strong> <span id="view-tags"></span></div>
        </div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Close</button></div>
    </div>
  </div>
</div>

<!-- Edit Lead Modal -->
<div class="modal fade" id="editLeadModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post" enctype="multipart/form-data">
      <input type="hidden" name="action" value="update"><input type="hidden" name="id" id="edit-id"><input type="hidden" name="photo_existing" id="edit-photo-existing">
      <div class="modal-header"><h6 class="modal-title">Edit Lead</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <div class="col-xl-12 text-center">
            <span class="avatar avatar-xxl avatar-rounded">
              <img src="<?php echo $baseUrl; ?>/assets/images/faces/9.jpg" alt="" id="edit-profile-preview">
              <span class="badge rounded-pill bg-primary avatar-badge"><input type="file" name="photo_edit" class="position-absolute w-100 h-100 op-0" id="edit-profile-file" accept="image/*"><i class="fe fe-camera"></i></span>
            </span>
          </div>
          <div class="col-xl-12"><label class="form-label">Contact Name</label><input type="text" class="form-control" id="edit-name" name="contact_name" required></div>
          <div class="col-xl-12"><label class="form-label">Email</label><input type="email" class="form-control" id="edit-email" name="email"></div>
          <div class="col-xl-12"><label class="form-label">Phone No</label><input type="tel" class="form-control" id="edit-phone" name="phone"></div>
          <div class="col-xl-12"><label class="form-label">Company Name</label><input type="text" class="form-control" id="edit-company" name="company"></div>
          <div class="col-xl-12"><label class="form-label">Lead Status</label>
            <select class="form-control" id="edit-status" name="lead_status">
              <option value="New">New</option><option value="Follow-up">Follow-up</option><option value="Closed">Closed</option><option value="Contacted">Contacted</option><option value="Disqualified">Disqualified</option><option value="Qualified">Qualified</option>
            </select>
          </div>
          <div class="col-xl-12"><label class="form-label">Lead Source</label>
            <select class="form-control" id="edit-source" name="lead_source">
              <option value="">Select Source</option><option value="Social Media">Social Media</option><option value="Direct Mail">Direct Mail</option><option value="Blog Articles">Blog Articles</option><option value="Affiliates">Affiliates</option><option value="Organic Search">Organic Search</option><option value="Event">Event</option><option value="Referral">Referral</option><option value="Web">Web</option>
            </select>
          </div>
          <div class="col-xl-12"><label class="form-label">Assigned User</label><input type="text" class="form-control" id="edit-assigned" name="assigned_user"></div>
          <div class="col-xl-12"><label class="form-label">Tag Status</label>
            <select class="form-control" id="edit-tags" name="tags[]" multiple><?php foreach($TAG_STATUS as $tg): ?><option><?php echo h($tg); ?></option><?php endforeach; ?></select>
          </div>
        </div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button><button type="submit" class="btn btn-primary">Update Lead</button></div>
    </form>
  </div>
</div>

<!-- TAGS Manage Modal -->
<div class="modal fade" id="tagsManageModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="update_tags"><input type="hidden" name="id" id="tags-id">
      <div class="modal-header"><h6 class="modal-title">Change Tag Status</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body px-4">
        <div class="row"><?php foreach($TAG_STATUS as $tg): ?><div class="col-6 mb-2"><label class="form-check"><input class="form-check-input tags-check" type="checkbox" name="tags[]" value="<?php echo h($tg); ?>"><span class="form-check-label"><?php echo h($tg); ?></span></label></div><?php endforeach; ?></div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button><button type="submit" class="btn btn-primary">Save</button></div>
    </form>
  </div>
</div>

<!-- Combined Lead Modal (contact + tags) -->
<div class="modal fade" id="leadFullModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-dialog-scrollable">
    <form class="modal-content" method="post" enctype="multipart/form-data">
      <input type="hidden" name="action" value="update"><input type="hidden" name="id" id="full-id"><input type="hidden" name="photo_existing" id="full-photo-existing">
      <div class="modal-header"><h6 class="modal-title">Lead - Details & Actions</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body px-4">
        <div class="text-center mb-3">
          <span class="avatar avatar-xxl avatar-rounded">
            <img src="<?php echo $baseUrl; ?>/assets/images/faces/9.jpg" alt="" id="full-avatar">
            <span class="badge rounded-pill bg-primary avatar-badge"><input type="file" name="photo_edit" class="position-absolute w-100 h-100 op-0" id="full-file" accept="image/*"><i class="fe fe-camera"></i></span>
          </span>
        </div>
        <div class="row gy-3">
          <div class="col-xl-6"><label class="form-label">Contact Name</label><input type="text" class="form-control" id="full-name" name="contact_name" required></div>
          <div class="col-xl-6"><label class="form-label">Assigned User</label><input type="text" class="form-control" id="full-assigned" name="assigned_user" placeholder="e.g. Amit Kumar"></div>
          <div class="col-xl-6"><label class="form-label">Email</label><input type="email" class="form-control" id="full-email" name="email"></div>
          <div class="col-xl-6"><label class="form-label">Phone No</label><input type="tel" class="form-control" id="full-phone" name="phone"></div>
          <div class="col-xl-6"><label class="form-label">Company</label><input type="text" class="form-control" id="full-company" name="company"></div>
          <div class="col-xl-6"><label class="form-label">Lead Source</label>
            <select class="form-control" id="full-source" name="lead_source">
              <option value="">Select Source</option><option value="Social Media">Social Media</option><option value="Direct Mail">Direct Mail</option><option value="Blog Articles">Blog Articles</option><option value="Affiliates">Affiliates</option><option value="Organic Search">Organic Search</option><option value="Event">Event</option><option value="Referral">Referral</option><option value="Web">Web</option>
            </select>
          </div>
          <div class="col-xl-6"><label class="form-label">Lead Status</label>
            <select class="form-control" id="full-status" name="lead_status">
              <option value="New">New</option><option value="Follow-up">Follow-up</option><option value="Closed">Closed</option><option value="Contacted">Contacted</option><option value="Disqualified">Disqualified</option><option value="Qualified">Qualified</option>
            </select>
          </div>
          <div class="col-xl-12">
            <label class="form-label">Tag Status (tick to apply)</label>
            <div class="row">
              <?php foreach($TAG_STATUS as $tg): ?>
              <div class="col-6 col-md-4 mb-2"><label class="form-check"><input class="form-check-input full-tags-check" type="checkbox" name="tags_full[]" value="<?php echo h($tg); ?>"><span class="form-check-label"><?php echo h($tg); ?></span></label></div>
              <?php endforeach; ?>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Close</button><button type="submit" class="btn btn-primary">Save Changes</button></div>
    </form>
  </div>
</div>

<!-- Activity Modal (unified form) -->
<div class="modal fade" id="activityModal" tabindex="-1" aria-labelledby="activityModal" data-bs-keyboard="false" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header"><h6 class="modal-title">Log Activity</h6><button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button></div>
      <div class="modal-body" id="activity-body"><div class="text-center text-muted">Select an action from the leads table.</div></div>
      <div class="modal-footer"><button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button><button type="button" class="btn btn-primary" id="activity-save">Save Activity</button></div>
    </div>
  </div>
</div>

<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
(function(){
  const checkAll=document.getElementById('checkAll'); if(checkAll){ checkAll.addEventListener('change',()=>{ document.querySelectorAll('.row-check').forEach(cb=>cb.checked=checkAll.checked); }); }

  // Image previews
  const createFile=document.getElementById('create-profile-file'), createPrev=document.getElementById('create-profile-preview'); if(createFile&&createPrev){ createFile.addEventListener('change',e=>{ const f=e.target.files?.[0]; if(f) createPrev.src=URL.createObjectURL(f); }); }
  const editFile=document.getElementById('edit-profile-file'), editPrev=document.getElementById('edit-profile-preview'); if(editFile&&editPrev){ editFile.addEventListener('change',e=>{ const f=e.target.files?.[0]; if(f) editPrev.src=URL.createObjectURL(f); }); }

  // View
  document.querySelectorAll('.btn-view').forEach(btn=>{
    btn.addEventListener('click', function(){
      const d=this.dataset;
      document.getElementById('view-avatar').src=d.avatar||'<?php echo $baseUrl; ?>/assets/images/faces/9.jpg';
      document.getElementById('view-name').textContent=d.name||''; document.getElementById('view-email').textContent=d.email||'';
      document.getElementById('view-phone').textContent=d.phone||''; document.getElementById('view-status').textContent=d.status||'';
      document.getElementById('view-company').textContent=d.company||''; document.getElementById('view-source').textContent=d.source||'';
      document.getElementById('view-tags').textContent=d.tags||''; new bootstrap.Modal(document.getElementById('viewLeadModal')).show();
    });
  });

  // Edit
  document.querySelectorAll('.btn-edit').forEach(btn=>{
    btn.addEventListener('click', function(){
      const d=this.dataset;
      document.getElementById('edit-id').value=d.id||'';
      document.getElementById('edit-name').value=d.name||''; document.getElementById('edit-email').value=d.email||'';
      document.getElementById('edit-phone').value=d.phone||''; document.getElementById('edit-company').value=d.company||'';
      document.getElementById('edit-status').value=d.status||'New'; document.getElementById('edit-source').value=d.source||'';
      document.getElementById('edit-assigned').value=d.assigned||''; document.getElementById('edit-profile-preview').src=d.avatar||'<?php echo $baseUrl; ?>/assets/images/faces/9.jpg';
      document.getElementById('edit-photo-existing').value = d.avatar ? d.avatar.replace('<?php echo h($baseUrl); ?>/', '') : '';
      const tagsSelect=document.getElementById('edit-tags'); Array.from(tagsSelect.options).forEach(o=>o.selected=false);
      if(d.tags){ const sel=d.tags.split(',').map(s=>s.trim()); Array.from(tagsSelect.options).forEach(o=>{ if(sel.includes(o.value)) o.selected=true; }); }
      new bootstrap.Modal(document.getElementById('editLeadModal')).show();
    });
  });

  // Tags manage
  const tagsModalEl=document.getElementById('tagsManageModal');
  document.querySelectorAll('.tags-cell, .tags-trigger').forEach(node=>{
    node.addEventListener('click', function(){
      const td=this.closest('.tags-cell')||this.parentElement.closest('.tags-cell')||this;
      const id=td.getAttribute('data-id'); const tags=(td.getAttribute('data-tags')||'').split(',').map(s=>s.trim()).filter(Boolean);
      document.getElementById('tags-id').value=id||'';
      tagsModalEl.querySelectorAll('.tags-check').forEach(ch=>{ ch.checked=false; if(tags.includes(ch.value)) ch.checked=true; });
      new bootstrap.Modal(tagsModalEl).show();
    });
  });

  // Lead combined modal (basic fields + tags)
  const fullEl=document.getElementById('leadFullModal');
  document.querySelectorAll('.lead-full').forEach(a=>{
    a.addEventListener('click', function(){
      const d=this.dataset;
      document.getElementById('full-id').value=d.id||'';
      document.getElementById('full-avatar').src=d.avatar||'<?php echo $baseUrl; ?>/assets/images/faces/9.jpg';
      document.getElementById('full-photo-existing').value = d.avatar ? d.avatar.replace('<?php echo h($baseUrl); ?>/', '') : '';
      document.getElementById('full-name').value=d.name||''; document.getElementById('full-email').value=d.email||''; document.getElementById('full-phone').value=d.phone||'';
      document.getElementById('full-company').value=d.company||''; document.getElementById('full-source').value=d.source||''; document.getElementById('full-status').value=d.status||'New';
      document.getElementById('full-assigned').value=d.assigned||'';
      const sel=(d.tags||'').split(',').map(s=>s.trim()); fullEl.querySelectorAll('.full-tags-check').forEach(ch=>ch.checked=sel.includes(ch.value));
      const fullFile=document.getElementById('full-file'), fullPrev=document.getElementById('full-avatar'); if(fullFile&&fullPrev){ fullFile.onchange=(e)=>{ const f=e.target.files?.[0]; if(f) fullPrev.src=URL.createObjectURL(f); }; }
      new bootstrap.Modal(fullEl).show();
    });
  });

  // Activity modal (same pattern as Opportunities)
  const activityModalEl=document.getElementById('activityModal');
  const activityBody=document.getElementById('activity-body');
  const activitySaveBtn=document.getElementById('activity-save');
  let currentActivity={type:null,id:null};

  function defaultEndISO(min){ const t=new Date(Date.now()+min*60000); const y=t.getFullYear(),m=String(t.getMonth()+1).padStart(2,'0'),d=String(t.getDate()).padStart(2,'0'); const hh=String(t.getHours()).padStart(2,'0'),mm=String(t.getMinutes()).padStart(2,'0'); return `${y}-${m}-${d}T${hh}:${mm}`; }
  function buildActivityForm(type,d){
    const isCall=(type==='call'); const pretty = type==='meeting'?'Meeting':type==='mail'?'Email':type==='call'?'Call':'Task';
    const endDefault=defaultEndISO(5);
    return `
      <div class="row g-2">
        <div class="col-md-6"><label class="form-label">Status</label>
          <select class="form-select" id="act-status"><option value="Planned">Planned</option><option value="Held">Held</option><option value="Not Held">Not Held</option><option value="Completed">Completed</option><option value="Cancelled">Cancelled</option></select>
        </div>
        <div class="col-md-6"><label class="form-label">Direction</label>
          <select class="form-select" id="act-direction" ${isCall?'':'disabled'}>${isCall?'<option value="Outbound" selected>Outbound</option><option value="Inbound">Inbound</option><option value="N/A">N/A</option>':'<option value="N/A" selected>N/A</option>'}</select>
          ${isCall?'':'<div class="small text-muted mt-1">Direction applies to Calls only.</div>'}
        </div>
        <div class="col-md-6"><label class="form-label">Duration</label>
          <select class="form-select" id="act-duration"><option value="5">5m</option><option value="15">15m</option><option value="30">30m</option><option value="45">45m</option><option value="60">1h</option><option value="90">1h 30m</option><option value="120">2h</option></select>
        </div>
        <div class="col-md-6"><label class="form-label">Date End *</label><input type="datetime-local" class="form-control" id="act-dateend" value="${endDefault}"></div>
        <div class="col-12"><label class="form-label">Title</label><input type="text" class="form-control" id="act-title" value="${pretty + (d.name?(' - '+d.name):'')}"></div>
        <div class="col-12"><label class="form-label">Description</label><textarea class="form-control" id="act-desc" rows="4" placeholder="${pretty==='Meeting'?'Agenda / notes':'Notes'}"></textarea></div>
        <div class="col-12"><hr class="my-2"></div>
        <div class="col-md-6"><label class="form-label d-flex align-items-center gap-2"><input type="checkbox" id="act-remind"> Set Reminder</label></div>
        <div class="col-md-6"><label class="form-label">Reminder At</label><input type="datetime-local" class="form-control" id="act-remind-at" value="${endDefault}" disabled></div>
      </div>
    `;
  }

  document.addEventListener('click', function(ev){
    const btn=ev.target.closest('.btn-quick-action'); if(!btn) return;
    const d=btn.dataset; currentActivity={type:(d.action||'task'), id:(d.id||null)};
    activityModalEl.querySelector('.modal-title').textContent='Log '+(currentActivity.type==='call'?'Call':currentActivity.type==='meeting'?'Meeting':currentActivity.type==='mail'?'Email':'Task');
    activityBody.innerHTML=buildActivityForm(currentActivity.type,d);
    const cb=document.getElementById('act-remind'), at=document.getElementById('act-remind-at'); if(cb&&at){ at.disabled=!cb.checked; cb.addEventListener('change',()=>{ at.disabled=!cb.checked; }); }
    new bootstrap.Modal(activityModalEl).show();
  });

  if(activitySaveBtn){
    activitySaveBtn.addEventListener('click', function(){
      if(!currentActivity.id){ alert('Missing lead'); return; }
      const status=document.getElementById('act-status').value;
      const dirEl=document.getElementById('act-direction'); const direction=dirEl?(dirEl.disabled?'N/A':dirEl.value):'N/A';
      const duration=parseInt(document.getElementById('act-duration').value||'5',10);
      const dateend=document.getElementById('act-dateend').value;
      const desc=document.getElementById('act-desc').value||'';
      const title=document.getElementById('act-title').value||'';
      const remind=document.getElementById('act-remind').checked?1:0;
      const remindAt=document.getElementById('act-remind-at').value||'';
      const form=new URLSearchParams();
      form.append('ajax','save_activity'); form.append('id',currentActivity.id); form.append('type',currentActivity.type);
      form.append('status',status); form.append('direction',direction); form.append('duration_minutes', isNaN(duration)?5:duration);
      form.append('date_end', dateend.replace('T',' ')); form.append('description',desc); form.append('title',title);
      form.append('remind', String(remind)); form.append('reminder_at', remindAt ? remindAt.replace('T',' ') : '');
      fetch('<?php echo h($_SERVER["PHP_SELF"]); ?>', { method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body:form.toString() })
        .then(async r=>{ const t=await r.text(); let js=null; try{js=JSON.parse(t);}catch(e){} return {ok:r.ok, text:t, json:js}; })
        .then(res=>{ if(res.json && res.json.ok){ (bootstrap.Modal.getInstance(activityModalEl)||new bootstrap.Modal(activityModalEl)).hide(); alert('Activity saved.'); } else { alert((res.json && (res.json.err||res.json.msg)) ? (res.json.msg + (res.json.err?' — '+res.json.err:'')) : 'Save failed.'); console.log('RAW:',res.text); if(res.json&&res.json.sql) console.log('SQL:',res.json.sql); } })
        .catch(()=>alert('Network error'));
    });
  }
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
