<?php
/* ============================================================================
 * File: includes/rbac.php
 * Purpose: Central RBAC bootstrap + helpers (ONE place to manage roles/permissions)
 * Stack : Core PHP + MySQLi (NO PDO). Requires config.php to define $conn (mysqli)
 * Rules : Collation utf8mb4_general_ci
 * Notes : - All functions are guarded with function_exists to avoid redeclare fatals
 *         - Back-compat aliases: rbac_seed_permissions($conn), rbac_has($conn,$uid,$code)
 * Usage :
 *   1) In config.php after $conn is created: require_once __DIR__ . '/includes/rbac.php';
 *   2) In your global header: require_once __DIR__ . '/../includes/rbac.php'; rbac_auto_guard(); rbac_export_client_perms();
 *   3) Server checks: if (!rbac_can('leads.export')) { /* 403 */ /* }
 *   4) Client hide:  <button data-perm="leads.export">Export</button>
 * ========================================================================== */

date_default_timezone_set('Asia/Kolkata');
if (session_status() === PHP_SESSION_NONE) { @session_start(); }

/* ---- Require DB connection from config.php ---- */
if (!isset($conn) || !($conn instanceof mysqli)) {
  http_response_code(500);
  echo 'RBAC: $conn (mysqli) not found. Include config.php before rbac.php.'; exit;
}

/* ---- Charset & collation enforcement ---- */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* ---------- Tiny internals (guarded) ---------- */
if (!function_exists('_rbac_q')) {
  function _rbac_q($sql){ global $conn; return @mysqli_query($conn, $sql); }
}
if (!function_exists('_rbac_es')) {
  function _rbac_es($s){ global $conn; return mysqli_real_escape_string($conn,(string)$s); }
}
if (!function_exists('rbac_user_id')) {
  function rbac_user_id(){ return (int)($_SESSION['user_id1'] ?? 0); }
}

/* ---------- Ensure schema (guarded & idempotent) ---------- */
if (!function_exists('rbac_ensure_schema')) {
  function rbac_ensure_schema(){
    _rbac_q("CREATE TABLE IF NOT EXISTS `settings` (
      `k` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `v` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (`k`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    _rbac_q("CREATE TABLE IF NOT EXISTS `roles` (
      `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
      `name` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `description` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      UNIQUE KEY `uniq_role_name_core` (`name`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    _rbac_q("CREATE TABLE IF NOT EXISTS `permissions` (
      `code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `label` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `group_name` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
      PRIMARY KEY (`code`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    _rbac_q("CREATE TABLE IF NOT EXISTS `role_permissions` (
      `role_id` INT UNSIGNED NOT NULL,
      `perm_code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      PRIMARY KEY (`role_id`,`perm_code`),
      KEY `idx_perm_code_core` (`perm_code`),
      CONSTRAINT `fk_role_permissions_role_core` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    _rbac_q("CREATE TABLE IF NOT EXISTS `user_roles` (
      `user_id` INT NOT NULL,
      `role_id` INT UNSIGNED NOT NULL,
      PRIMARY KEY (`user_id`,`role_id`),
      KEY `idx_user_roles_role_core` (`role_id`),
      CONSTRAINT `fk_user_roles_role_core` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
  }
}

/* ---------- Seed default permissions ONCE (guarded) ---------- */
if (!function_exists('rbac_seed_default_permissions')) {
  function rbac_seed_default_permissions(){
    // Ensure `settings` exists (in case a page had its own ensure function)
    _rbac_q("CREATE TABLE IF NOT EXISTS `settings` (
      `k` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `v` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (`k`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    $seedFlag = 'rbac_seed_v1';
    $res = _rbac_q("SELECT `v` FROM `settings` WHERE `k` = '{$seedFlag}' LIMIT 1");
    $already = ($res && mysqli_num_rows($res) && mysqli_fetch_row($res)[0] === '1');

    $perms = [
      // Leads
      ['leads.create',        'Create/Update/Delete leads',               'Leads'],
      ['leads.export',        'Export leads (CSV)',                       'Leads'],
      // Activities
      ['activities.log',      'Log activities (call/mail/task/meeting)',  'Activities'],
      // Opportunities
      ['opportunities.view',        'View opportunities board/list',      'Opportunities'],
      ['opportunities.move',        'Move stage (drag & drop)',           'Opportunities'],
      ['opportunities.stage_change','Change stage via modal',             'Opportunities'],
      // Admin
      ['roles.manage',        'Manage roles & permissions',               'Admin'],
      ['users.manage',        'Manage users',                             'Admin'],
    ];

    if(!$already){
      foreach($perms as $p){
        $code = _rbac_es($p[0]); $label=_rbac_es($p[1]); $grp=_rbac_es($p[2]);
        _rbac_q("INSERT IGNORE INTO `permissions` (`code`,`label`,`group_name`)
                 VALUES ('{$code}','{$label}','{$grp}')");
      }
      _rbac_q("INSERT INTO `settings` (`k`,`v`) VALUES ('{$seedFlag}','1')
               ON DUPLICATE KEY UPDATE `v`='1', `updated_at`=CURRENT_TIMESTAMP");
    }
  }
}

/* ---------- Permissions cache (guarded) ---------- */
if (!function_exists('rbac_user_permissions')) {
  function rbac_user_permissions(){
    static $cache = null;
    if($cache !== null){ return $cache; }
    $uid = rbac_user_id();
    $codes = [];
    if($uid){
      $q = _rbac_q("SELECT rp.perm_code
                    FROM user_roles ur
                    JOIN role_permissions rp ON rp.role_id = ur.role_id
                    WHERE ur.user_id = {$uid}");
      if($q){ while($r = mysqli_fetch_row($q)){ $codes[$r[0]] = 1; } }
    }
    $cache = $codes;
    return $cache;
  }
}

/* ---------- Public API (guarded) ---------- */
if (!function_exists('rbac_bootstrap')) {
  function rbac_bootstrap(){
    rbac_ensure_schema();
    rbac_seed_default_permissions(); // safe to call every request
    rbac_user_permissions();         // warm cache
  }
}
if (!function_exists('rbac_can')) {
  function rbac_can($permCode){
    $perms = rbac_user_permissions();
    return isset($perms[$permCode]);
  }
}
if (!function_exists('rbac_any')) {
  function rbac_any(array $permCodes){
    if(!$permCodes){ return true; }
    foreach($permCodes as $c){ if(rbac_can($c)) return true; }
    return false;
  }
}
if (!function_exists('rbac_all')) {
  function rbac_all(array $permCodes){
    foreach($permCodes as $c){ if(!rbac_can($c)) return false; }
    return true;
  }
}
if (!function_exists('rbac_require_login')) {
  function rbac_require_login(){
    if(!rbac_user_id()){
      http_response_code(401);
      echo '<!doctype html><html><head><meta charset="utf-8"><title>Unauthorized</title></head><body style="font-family:system-ui;padding:24px"><h2>401 — Please sign in</h2><p>This page requires an authenticated user.</p></body></html>';
      exit;
    }
  }
}
if (!function_exists('rbac_require')) {
  function rbac_require($permCodes, $mode='any'){
    if(is_string($permCodes)){ $permCodes = array_filter(array_map('trim', explode(',', $permCodes))); }
    $ok = ($mode==='all') ? rbac_all($permCodes) : rbac_any($permCodes);
    if(!$ok){
      http_response_code(403);
      echo '<!doctype html><html><head><meta charset="utf-8"><title>Forbidden</title></head><body style="font-family:system-ui;padding:24px"><h2>403 — Not authorized</h2><p>You do not have permission to perform this action.</p></body></html>';
      exit;
    }
  }
}
if (!function_exists('rbac_auto_guard')) {
  /**
   * Optional page-wide guard. Set these BEFORE calling rbac_auto_guard():
   *   $RBAC_REQUIRE_LOGIN = true;                // default true
   *   $RBAC_REQUIRE_ALL   = ['roles.manage'];    // ALL must pass
   *   $RBAC_REQUIRE_ANY   = ['leads.create'];    // ANY may pass
   */
  function rbac_auto_guard(){
    $requireLogin = $GLOBALS['RBAC_REQUIRE_LOGIN'] ?? true;
    if($requireLogin){ rbac_require_login(); }
    $all = $GLOBALS['RBAC_REQUIRE_ALL'] ?? [];
    $any = $GLOBALS['RBAC_REQUIRE_ANY'] ?? [];
    if($all){ rbac_require($all, 'all'); }
    if($any){ rbac_require($any, 'any'); }
  }
}
if (!function_exists('rbac_export_client_perms')) {
  function rbac_export_client_perms(){
    $uid = rbac_user_id();
    $perms = array_keys(rbac_user_permissions());
    $json = json_encode(array_fill_keys($perms, 1), JSON_UNESCAPED_SLASHES);
    echo "<style>.rbac-hidden{display:none!important}</style>\n";
    echo "<script>\n";
    echo "window.CURRENT_USER_ID = ".(int)$uid.";\n";
    echo "window.CURRENT_PERMISSIONS = ".$json.";\n";
    echo "document.addEventListener('DOMContentLoaded', function(){\n";
    echo "  var has = function(code){ return !!(window.CURRENT_PERMISSIONS && window.CURRENT_PERMISSIONS[code]); };\n";
    echo "  document.querySelectorAll('[data-perm]').forEach(function(el){\n";
    echo "    var expr = (el.getAttribute('data-perm')||'').split(',').map(function(s){return s.trim();}).filter(Boolean);\n";
    echo "    var mode = (el.getAttribute('data-perm-mode')||'any').toLowerCase();\n";
    echo "    var ok = true;\n";
    echo "    if(expr.length){\n";
    echo "      if(mode==='all'){ ok = expr.every(has); }\n";
    echo "      else if(mode==='not'){ ok = expr.every(function(c){return !has(c);}); }\n";
    echo "      else { ok = expr.some(has); }\n";
    echo "    }\n";
    echo "    if(!ok){ el.classList.add('rbac-hidden'); el.setAttribute('disabled','disabled'); }\n";
    echo "  });\n";
    echo "});\n";
    echo "</script>\n";
  }
}

/* ---------- Backward-compatibility aliases (guarded) ---------- */
/* Some pages may still call these older helpers. Keep them working. */
if (!function_exists('rbac_seed_permissions')) {
  function rbac_seed_permissions($conn_ignored = null){ rbac_seed_default_permissions(); }
}
if (!function_exists('rbac_has')) {
  // Old signature: rbac_has($conn,$uid,$code)
  function rbac_has($conn_ignored, $uid, $code){
    $uid  = (int)$uid;
    $code = _rbac_es($code);
    if($uid === rbac_user_id()){
      return rbac_can($code);
    }
    // Check for an arbitrary user id (rare, but keep support)
    $q = _rbac_q("SELECT 1
                  FROM user_roles ur
                  JOIN role_permissions rp ON rp.role_id = ur.role_id
                  WHERE ur.user_id={$uid} AND rp.perm_code='{$code}'
                  LIMIT 1");
    return ($q && mysqli_num_rows($q)>0);
  }
}

/* ---------- Auto-run bootstrap ---------- */
rbac_bootstrap();
