<?php
/* ============================================================================
 * File: dashboard.php
 * Purpose: CRM Dashboard (dynamic) pulling data from `leads` and `crm_activities`
 * Stack : Core PHP + MySQLi (NO PDO). DB connection lives in ../config.php
 * Rules : DB collation utf8mb4_general_ci
 * Notes : - KPIs from leads/opportunities
 *         - Revenue analytics (monthly Closed Won)
 *         - Tasks (Today / Upcoming) from crm_activities
 *         - Leads Overview, Leads By Source, Deals Statistics, Top Deals
 * ========================================================================== */

@session_start();
require_once __DIR__ . '/../config.php'; // must define $conn (mysqli)

// Charset & Collation
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

date_default_timezone_set('Asia/Kolkata');
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// Base URL for assets
$rootFolder = basename($_SERVER['DOCUMENT_ROOT']);
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://')
         . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));

// -------------------------- KPIs --------------------------
$totalLeads = 0;
$r = mysqli_query($conn, "SELECT COUNT(*) c FROM `leads`");
if($r){ $totalLeads = (int)mysqli_fetch_assoc($r)['c']; }

$totalDeals = 0;
$r = mysqli_query($conn, "SELECT COUNT(*) c FROM `leads` WHERE `opp_stage` IS NOT NULL AND `opp_stage`<>''");
if($r){ $totalDeals = (int)mysqli_fetch_assoc($r)['c']; }

$closedWon = 0;
$r = mysqli_query($conn, "SELECT COUNT(*) c FROM `leads` WHERE `opp_stage`='Prospecting' OR `opp_stage`='Qualification' OR `opp_stage`='Proposal' OR `opp_stage`='Negotiation' OR `opp_stage`='Closed Won'");
if($r){ /* not used */ }

$r = mysqli_query($conn, "SELECT COUNT(*) c FROM `leads` WHERE `opp_stage`='Closed Won'");
if($r){ $closedWon = (int)mysqli_fetch_assoc($r)['c']; }

$conversionRatio = ($totalDeals>0) ? round(($closedWon/$totalDeals)*100,2) : 0;

$totalCustomers = 0; // leads that became customers (Closed Won OR tag contains 'Customer')
$r = mysqli_query($conn, "SELECT COUNT(*) c FROM `leads` 
  WHERE `opp_stage`='Closed Won' OR FIND_IN_SET('Customer', REPLACE(COALESCE(`tags`,''),' ',''))");
if($r){ $totalCustomers = (int)mysqli_fetch_assoc($r)['c']; }

$totalRevenue = 0.00;
$r = mysqli_query($conn, "SELECT COALESCE(SUM(`opp_amount`),0) amt 
                          FROM `leads` WHERE `opp_stage`='Closed Won'");
if($r){ $totalRevenue = (float)mysqli_fetch_assoc($r)['amt']; }

// YoY deltas (Customers & Revenue)
$currYear = (int)date('Y');
$prevYear = $currYear-1;
$revPrev = 0.00;
$r = mysqli_query($conn, "SELECT COALESCE(SUM(`opp_amount`),0) amt
                          FROM `leads` 
                          WHERE `opp_stage`='Closed Won' AND YEAR(`opp_close_date`)={$prevYear}");
if($r){ $revPrev = (float)mysqli_fetch_assoc($r)['amt']; }
$revDeltaPct = ($revPrev>0) ? round((($totalRevenue-$revPrev)/$revPrev)*100,2) : 100.00;

$custPrev = 0;
$r = mysqli_query($conn, "SELECT COUNT(*) c FROM `leads` 
  WHERE (`opp_stage`='Closed Won' OR FIND_IN_SET('Customer', REPLACE(COALESCE(`tags`,''),' ','')))
    AND YEAR(`created_at`)={$prevYear}");
if($r){ $custPrev = (int)mysqli_fetch_assoc($r)['c']; }
$custDeltaPct = ($custPrev>0) ? round((($totalCustomers-$custPrev)/$custPrev)*100,2) : 100.00;

// -------------------- Revenue Analytics (Monthly) --------------------
$monthTotals = array_fill(1, 12, 0.0);
$q = "SELECT MONTH(`opp_close_date`) m, COALESCE(SUM(`opp_amount`),0) s
      FROM `leads`
      WHERE `opp_stage`='Closed Won' AND YEAR(`opp_close_date`)={$currYear}
      GROUP BY m";
$r = mysqli_query($conn, $q);
if($r){
  while($row = mysqli_fetch_assoc($r)){
    $m = (int)$row['m'];
    if($m>=1 && $m<=12) $monthTotals[$m] = (float)$row['s'];
  }
}

// --------------------- Leads By Source (Top 6) ----------------------
$sourceRows = [];
$r = mysqli_query($conn, "SELECT COALESCE(NULLIF(TRIM(`lead_source`),''),'Others') `src`, COUNT(*) c
                          FROM `leads`
                          GROUP BY `src`
                          ORDER BY c DESC, src ASC
                          LIMIT 6");
if($r){ while($row = mysqli_fetch_assoc($r)){ $sourceRows[] = $row; } }
$leadsSourceLabels = array_map(fn($x)=>$x['src'], $sourceRows);
$leadsSourceCounts = array_map(fn($x)=>(int)$x['c'], $sourceRows);

// --------------------- Deals Statistics (by Stage) -------------------
$stageMap = ['Prospecting'=>0,'Qualification'=>0,'Proposal'=>0,'Negotiation'=>0,'Closed Won'=>0];
$r = mysqli_query($conn, "SELECT `opp_stage` s, COUNT(*) c 
                          FROM `leads`
                          WHERE `opp_stage` IS NOT NULL AND `opp_stage`<>''
                          GROUP BY s");
if($r){
  while($row = mysqli_fetch_assoc($r)){
    $s = (string)$row['s'];
    $c = (int)$row['c'];
    if(isset($stageMap[$s])) $stageMap[$s] = $c;
  }
}

// ------------------- Tasks: Today & Upcoming (crm_activities) -------------------
$today = date('Y-m-d');
$tasksToday = [];
$r = mysqli_query($conn, "SELECT a.`id`, a.`title`, a.`status`, a.`date_start`, a.`assigned_user`,
                                 l.`contact_name`, l.`company`
                          FROM `crm_activities` a
                          JOIN `leads` l ON l.`id`=a.`lead_id`
                          WHERE a.`activity_type`='task' AND DATE(a.`date_start`)='{$today}'
                          ORDER BY a.`date_start` ASC
                          LIMIT 6");
if($r){ while($row = mysqli_fetch_assoc($r)){ $tasksToday[] = $row; } }

$tasksUpcoming = [];
$r = mysqli_query($conn, "SELECT a.`id`, a.`title`, a.`status`, a.`date_start`, a.`assigned_user`,
                                 l.`contact_name`, l.`company`
                          FROM `crm_activities` a
                          JOIN `leads` l ON l.`id`=a.`lead_id`
                          WHERE a.`activity_type`='task' AND DATE(a.`date_start`)>'{$today}'
                          ORDER BY a.`date_start` ASC
                          LIMIT 6");
if($r){ while($row = mysqli_fetch_assoc($r)){ $tasksUpcoming[] = $row; } }

// --------------------- Leads Overview (latest 5) ---------------------
$leadsOverview = [];
$r = mysqli_query($conn, "SELECT `id`,`contact_name`,`company`,`lead_status`,`lead_source`,`avatar`
                          FROM `leads`
                          ORDER BY `created_at` DESC
                          LIMIT 5");
if($r){ while($row = mysqli_fetch_assoc($r)){ $leadsOverview[] = $row; } }

// --------------------- Top Deals (by opp_amount) ---------------------
$topDeals = [];
$r = mysqli_query($conn, "SELECT `id`,`opp_title`,`company`,`opp_amount`,`opp_stage`,`opp_close_date`,`assigned_user`
                          FROM `leads`
                          WHERE `opp_amount` IS NOT NULL
                          ORDER BY `opp_amount` DESC
                          LIMIT 6");
if($r){ while($row = mysqli_fetch_assoc($r)){ $topDeals[] = $row; } }

// Progress widget (Total Leads vs Target)
$leadsTarget = max(1, (int)ceil($totalLeads * 1.5));
$progressPct  = min(100, (int)round(($totalLeads / $leadsTarget) * 100));
$progressRest = 100 - $progressPct;

// Helpers for UI
function statusBadge($s){
  $s = (string)$s;
  $map = [
    'New' => 'bg-primary',
    'Contacted' => 'bg-info',
    'Qualified' => 'bg-success',
    'Follow-up' => 'bg-warning',
    'Closed' => 'bg-secondary',
    'Disqualified' => 'bg-dark'
  ];
  return $map[$s] ?? 'bg-light text-dark';
}
function taskBadge($s){
  $map = [
    'Planned'   => 'bg-warning-transparent',
    'Completed' => 'bg-success-transparent',
    'Held'      => 'bg-success-transparent',
    'Not Held'  => 'bg-danger-transparent',
    'Cancelled' => 'bg-secondary-transparent'
  ];
  return $map[$s] ?? 'bg-primary-transparent';
}
function dealStageBadge($s){
  $map = [
    'Prospecting'  => 'bg-primary-transparent',
    'Qualification'=> 'bg-secondary-transparent',
    'Proposal'     => 'bg-warning-transparent',
    'Negotiation'  => 'bg-info-transparent',
    'Closed Won'   => 'bg-success-transparent'
  ];
  return $map[$s] ?? 'bg-light';
}
function derivedPriority($amount){
  $amt = (float)$amount;
  if($amt >= 75000) return 'High';
  if($amt >= 50000) return 'Medium';
  if($amt > 0) return 'Low';
  return '—';
}

// ---------------------- Styles (inline) ----------------------
ob_start(); ?>
<style>
.lead-title.total::before { content:"<?php echo h(number_format($totalLeads)); ?> • "; color:#999; }
.lead-title.target::after { content:" • Target <?php echo h(number_format($leadsTarget)); ?>"; color:#999; }
.task-list-tab li { padding:12px 16px; border-bottom:1px solid #eef1f5; }
.task-list-tab li:last-child { border-bottom:0; }
.crm-task-name { font-weight:600; }
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
	
<!-- Start::page-header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">CRM</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
      <li class="breadcrumb-item active" aria-current="page">CRM</li>
    </ol>
  </div>
</div>
<!-- End::page-header -->

<!-- Start:: row-1 -->
<div class="row">
  <div class="col-xxl-5">
    <div class="row">
      <div class="col-lg-6">
        <div class="card custom-card">
          <div class="card-body">
            <div class="d-flex align-items-center gap-3 flex-wrap">
              <div class="lh-1">
                <span class="avatar avatar-md bg-primary avatar-rounded">
                  <i class="ti ti-users fs-5"></i>
                </span> 
              </div>
              <div class="flex-fill">
                <span>Total Customers</span>
                <h5 class="d-block fw-semibold mb-0"><?php echo number_format($totalCustomers); ?></h5>
              </div>
              <div class="text-end">
                <div class="fw-semibold <?php echo $custDeltaPct>=0?'text-success':'text-danger'; ?>">
                  <i class="ti <?php echo $custDeltaPct>=0?'ti-arrow-narrow-up':'ti-arrow-narrow-down'; ?> me-1"></i><?php echo ($custDeltaPct>=0?'+':'').$custDeltaPct; ?>%
                </div>
                <div class="text-muted fs-12">vs last year</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="col-lg-6">
        <div class="card custom-card">
          <div class="card-body">
            <div class="d-flex align-items-center gap-3 flex-wrap">
              <div class="lh-1">
                <span class="avatar avatar-md bg-secondary avatar-rounded">
                  <i class="ti ti-briefcase fs-5"></i>
                </span> 
              </div>
              <div class="flex-fill">
                <span>Total Deals</span>
                <h5 class="d-block fw-semibold mb-0"><?php echo number_format($totalDeals); ?></h5>
              </div>
              <div class="text-end">
                <div class="fw-semibold text-muted"><i class="ti ti-circle-dot me-1"></i>Active</div>
                <div class="text-muted fs-12">This Year</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="col-lg-6">
        <div class="card custom-card">
          <div class="card-body">
            <div class="d-flex align-items-center gap-3 flex-wrap">
              <div class="lh-1">
                <span class="avatar avatar-md bg-success avatar-rounded">
                  <i class="ti ti-wave-square fs-5"></i>
                </span> 
              </div>
              <div class="flex-fill">
                <span>Conversion Ratio</span>
                <h5 class="d-block fw-semibold mb-0"><?php echo number_format($conversionRatio,2); ?>%</h5>
              </div>
              <div class="text-end">
                <div class="fw-semibold text-muted"><i class="ti ti-checks me-1"></i>Won</div>
                <div class="text-muted fs-12"><?php echo number_format($closedWon); ?> closed</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="col-lg-6">
        <div class="card custom-card">
          <div class="card-body">
            <div class="d-flex align-items-center gap-3 flex-wrap">
              <div class="lh-1">
                <span class="avatar avatar-md bg-warning avatar-rounded">
                  <i class="ti ti-wallet fs-5"></i>
                </span> 
              </div>
              <div class="flex-fill">
                <span>Total Revenue</span>
                <h5 class="d-block fw-semibold mb-0">₹<?php echo number_format($totalRevenue,2); ?></h5>
              </div>
              <div class="text-end">
                <div class="fw-semibold <?php echo $revDeltaPct>=0?'text-success':'text-danger'; ?>">
                  <i class="ti <?php echo $revDeltaPct>=0?'ti-arrow-narrow-up':'ti-arrow-narrow-down'; ?> me-1"></i><?php echo ($revDeltaPct>=0?'+':'').$revDeltaPct; ?>%
                </div>
                <div class="text-muted fs-12">vs last year</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="col-xl-12">
        <div class="card custom-card">
          <div class="card-header">
            <div class="card-title">Revenue Analytics (<?php echo (int)$currYear; ?>)</div>
          </div>
          <div class="card-body pb-0">
            <div id="crm-revenue-analytics"></div>
          </div>
        </div>
      </div>
    </div>
  </div>

  <!-- Tasks Column -->
  <div class="col-xxl-4">
    <div class="row">
      <div class="col-xl-12">
        <div class="card custom-card overflow-hidden">
          <div class="card-header justify-content-between">
            <div class="card-title">Tasks List</div>
            <ul class="nav nav-tabs justify-content-end nav-tabs-header card-headertabs" role="tablist">
              <li class="nav-item" role="presentation">
                <a class="nav-link <?php echo empty($tasksToday)?'':'active'; ?>" data-bs-toggle="tab" role="tab" href="#today" aria-selected="<?php echo empty($tasksToday)?'false':'true'; ?>">Today</a>
              </li>
              <li class="nav-item" role="presentation">
                <a class="nav-link <?php echo empty($tasksToday)?'active show':''; ?>" data-bs-toggle="tab" role="tab" href="#Upcoming" aria-selected="<?php echo empty($tasksToday)?'true':'false'; ?>">Upcoming</a>
              </li>
            </ul>
          </div>
          <div class="card-body todo-tab p-0">
            <div class="tab-content">
              <div class="tab-pane border-0 <?php echo empty($tasksToday)?'':'active show'; ?>" id="today" role="tabpanel">
                <ul class="list-unstyled task-list-tab">
                  <?php if(!$tasksToday): ?>
                    <li class="text-center text-muted py-3">No tasks for today.</li>
                  <?php else: foreach($tasksToday as $t): ?>
                    <li>
                      <div class="d-flex align-items-start gap-2 flex-wrap">
                        <div class="mb-3 form-check">
                          <input type="checkbox" class="form-check-input" <?php echo in_array($t['status'],['Completed','Held'])?'checked':''; ?>>
                        </div>
                        <div class="flex-fill">
                          <div class="d-flex align-items-center gap-2 mb-1">
                            <a href="javascript:void(0);" class="crm-task-name"><?php echo h($t['title'] ?: ('Task for '.($t['company']?:$t['contact_name']))); ?></a>
                            <span class="avatar avatar-xs avatar-rounded bg-light border text-muted fw-semibold flex-shrink-0" data-bs-toggle="tooltip" title="<?php echo h($t['status']); ?>">
                              <i class="ti ti-info-circle fs-11"></i>
                            </span>
                          </div>
                          <div class="d-flex align-items-center gap-2 lh-1">
                            <span class="fs-12"><?php echo h($t['contact_name']); ?></span>
                            <div class="vr"></div>
                            <span class="text-muted mb-0 fs-12"><i class="ti ti-user me-1 fw-medium"></i><?php echo h($t['assigned_user'] ?: 'Unassigned'); ?></span>
                            <div class="vr"></div>
                            <span class="badge <?php echo taskBadge($t['status']); ?>"><?php echo h($t['status']); ?></span>
                          </div>
                        </div>
                        <div class="text-end">
                          <span class="d-block fw-medium"><?php echo h(date('h:i A', strtotime($t['date_start']))); ?></span>
                          <span class="d-block fs-12 text-muted">Due Time</span>
                        </div>
                      </div>
                    </li>
                  <?php endforeach; endif; ?>
                </ul>
              </div>

              <div class="tab-pane border-0 <?php echo empty($tasksToday)?'active show':''; ?>" id="Upcoming" role="tabpanel">
                <ul class="list-unstyled task-list-tab">
                  <?php if(!$tasksUpcoming): ?>
                    <li class="text-center text-muted py-3">No upcoming tasks.</li>
                  <?php else: foreach($tasksUpcoming as $t): ?>
                    <li>
                      <div class="d-flex align-items-start gap-2 flex-wrap">
                        <div class="mb-3 form-check">
                          <input type="checkbox" class="form-check-input" <?php echo in_array($t['status'],['Completed','Held'])?'checked':''; ?>>
                        </div>
                        <div class="flex-fill">
                          <div class="d-flex align-items-center gap-2 mb-1">
                            <a href="javascript:void(0);" class="crm-task-name"><?php echo h($t['title'] ?: ('Task for '.($t['company']?:$t['contact_name']))); ?></a>
                            <span class="avatar avatar-xs avatar-rounded bg-light border text-muted fw-semibold flex-shrink-0" data-bs-toggle="tooltip" title="<?php echo h($t['status']); ?>">
                              <i class="ti ti-info-circle fs-11"></i>
                            </span>
                          </div>
                          <div class="d-flex align-items-center gap-2 lh-1">
                            <span class="fs-12"><?php echo h($t['contact_name']); ?></span>
                            <div class="vr"></div>
                            <span class="text-muted mb-0 fs-12"><i class="ti ti-user me-1 fw-medium"></i><?php echo h($t['assigned_user'] ?: 'Unassigned'); ?></span>
                            <div class="vr"></div>
                            <span class="badge <?php echo taskBadge($t['status']); ?>"><?php echo h($t['status']); ?></span>
                          </div>
                        </div>
                        <div class="text-end">
                          <span class="d-block fw-medium"><?php echo h(date('Y-m-d', strtotime($t['date_start']))); ?></span>
                          <span class="d-block fs-12 text-muted">Due Date</span>
                        </div>
                      </div>
                    </li>
                  <?php endforeach; endif; ?>
                </ul>
              </div>
            </div>
          </div>
        </div>
      </div>

      <!-- Leads Progress -->
      <div class="col-xl-12">
        <div class="card custom-card">
          <div class="card-body">
            <div class="d-flex align-items-center justify-content-between flex-wrap gap-1">
              <div class="lead-title total">Total Leads</div>
              <div class="lead-title target">Leads Target</div>
            </div>
            <div class="progress-stacked progress-animate progress-sm my-3"> 
              <div class="progress-bar" role="progressbar" style="width: <?php echo $progressPct; ?>%" aria-valuenow="<?php echo $progressPct; ?>" aria-valuemin="0" aria-valuemax="100"></div> 
              <div class="progress-bar bg-success" role="progressbar" style="width: <?php echo $progressRest; ?>%" aria-valuenow="<?php echo $progressRest; ?>" aria-valuemin="0" aria-valuemax="100"></div> 
            </div>
            <div class="d-flex align-items-center gap-3 justify-content-between flex-wrap">
              <div><span class="text-success fw-medium me-2"><i class="ti ti-arrow-up me-1"></i><?php echo $progressPct; ?>%</span> Achieved</div>
              <div><a href="<?php echo $baseUrl; ?>/pages/crm-leads.php" class="link-primary text-decoration-underline">Leads Report <i class="ti ti-arrow-narrow-right"></i></a></div>
            </div>
          </div>
        </div>
      </div>

    </div>
  </div>

  <!-- Leads Source Column -->
  <div class="col-xxl-3">
    <div class="card custom-card overflow-hidden">
      <div class="card-header">
        <div class="card-title">Leads By Source</div>
      </div>
      <div class="card-body">
        <div id="leads-source"></div>                                
      </div>
      <div class="card-footer p-0">
        <ul class="list-group list-group-flush leads-source-list">
          <?php if(!$sourceRows): ?>
            <li class="list-group-item"><div class="text-center text-muted">No data</div></li>
          <?php else: foreach($sourceRows as $sr): ?>
            <li class="list-group-item">
              <div class="d-flex align-items-center justify-content-between">
                <div class="fw-semibold"><?php echo h($sr['src']); ?></div>
                <div class="lh-1 text-end">
                  <span class="d-block fw-semibold mb-0"><?php echo (int)$sr['c']; ?></span>
                </div>
              </div>
            </li>
          <?php endforeach; endif; ?>
        </ul>
      </div>
    </div>
  </div>
</div>
<!-- End:: row-1 -->

<!-- Start:: row-2 -->
<div class="row">
  <div class="col-xxl-3">
    <div class="card custom-card">
      <div class="card-header">
        <div class="card-title">Top Deals (by Amount)</div>
      </div>
      <div class="card-body">
        <ul class="list-unstyled crm-top-deals">
          <?php if(!$topDeals): ?>
            <li class="text-center text-muted">No deals found.</li>
          <?php else: foreach($topDeals as $d): ?>
            <li>
              <div class="d-flex align-items-top flex-wrap">
                <div class="me-2">
                  <span class="avatar avatar-sm avatar-rounded bg-primary-transparent fw-semibold">
                    <?php echo strtoupper(substr((string)$d['company'],0,2) ?: 'DL'); ?>
                  </span>
                </div>
                <div class="flex-fill">
                  <p class="fw-semibold mb-0"><?php echo h($d['opp_title'] ?: 'Deal #'.$d['id']); ?></p>
                  <span class="text-muted fs-12"><?php echo h($d['company'] ?: '—'); ?></span>
                </div>
                <div class="fw-semibold fs-15">₹<?php echo number_format((float)$d['opp_amount'],2); ?></div>
              </div>
            </li>
          <?php endforeach; endif; ?>
        </ul>
      </div>
    </div>
  </div>

  <div class="col-xxl-3">
    <div class="card custom-card">
      <div class="card-header">
        <div class="card-title">Deals Statistics</div>
      </div>
      <div class="card-body py-0">
        <div id="deals-statistics"></div>
      </div>
    </div>
  </div>

  <div class="col-xxl-6">
    <div class="card custom-card overflow-hidden">
      <div class="card-header">
        <div class="card-title">Leads Overview</div>
      </div>
      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table text-nowrap">
            <thead>
              <tr>
                <th scope="col">Lead Name</th>
                <th scope="col">Company</th>
                <th scope="col">Status</th>
                <th scope="col">Source</th>
              </tr>
            </thead>
            <tbody>
              <?php if(!$leadsOverview): ?>
                <tr><td colspan="4" class="text-center text-muted py-3">No leads yet.</td></tr>
              <?php else: foreach($leadsOverview as $L):
                $avatar = $L['avatar'] ? ($baseUrl.'/'.ltrim($L['avatar'],'/')) : ($baseUrl.'/assets/images/faces/11.jpg'); ?>
                <tr>
                  <td>
                    <div class="d-flex align-items-center gap-2">
                      <div class="lh-1">
                        <span class="avatar avatar-sm avatar-rounded">
                          <img src="<?php echo h($avatar); ?>" alt="">
                        </span>
                      </div>
                      <div><?php echo h($L['contact_name']); ?></div>
                    </div>
                  </td>
                  <td><?php echo h($L['company'] ?: '—'); ?></td>
                  <td><span class="badge <?php echo statusBadge($L['lead_status']); ?>"><?php echo h($L['lead_status'] ?: 'New'); ?></span></td>
                  <td><?php echo h($L['lead_source'] ?: '—'); ?></td>
                </tr>
              <?php endforeach; endif; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>
  </div>
</div>
<!-- End:: row-2 -->

<!-- Start:: row-3 -->
<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card overflow-hidden">
      <div class="card-header">
        <div class="card-title">Top Deals</div>
      </div>
      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table text-nowrap">
            <thead>
              <tr>
                <th>Deal ID</th>
                <th>Deal Name</th>
                <th>Client Name</th>
                <th>Deal Amount</th>
                <th>Status</th>
                <th>Closing Date</th>
                <th>Sales Rep</th>
                <th>Priority</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              <?php if(!$topDeals): ?>
                <tr><td colspan="9" class="text-center text-muted py-3">No deals to display.</td></tr>
              <?php else: foreach($topDeals as $d): ?>
                <tr>
                  <td><a href="javascript:void(0);"><?php echo '#LEAD-'.(int)$d['id']; ?></a></td>
                  <td><?php echo h($d['opp_title'] ?: '—'); ?></td>
                  <td>
                    <div class="d-flex align-items-center gap-2 position-relative">
                      <a href="javascript:void(0);" class="stretched-link"></a>
                      <div class="lh-1">
                        <span class="avatar avatar-sm avatar-rounded bg-primary-transparent">
                          <?php echo strtoupper(substr((string)$d['company'],0,2) ?: 'CO'); ?>
                        </span>
                      </div>
                      <div><?php echo h($d['company'] ?: '—'); ?></div>
                    </div>
                  </td>
                  <td>₹<?php echo number_format((float)$d['opp_amount'],2); ?></td>
                  <td><span class="badge <?php echo dealStageBadge($d['opp_stage']); ?>"><?php echo h($d['opp_stage'] ?: '—'); ?></span></td>
                  <td><?php echo $d['opp_close_date'] ? h(date('Y-m-d', strtotime($d['opp_close_date']))) : '—'; ?></td>
                  <td><?php echo h($d['assigned_user'] ?: '—'); ?></td>
                  <td><?php echo h(derivedPriority($d['opp_amount'])); ?></td>
                  <td>
                    <div class="btn-list">
                      <button class="btn btn-icon btn-primary-light btn-sm"><i class="ti ti-edit"></i></button>
                      <button class="btn btn-icon btn-danger-light btn-sm"><i class="ti ti-trash"></i></button>
                    </div>
                  </td>
                </tr>
              <?php endforeach; endif; ?>
            </tbody>
          </table>
        </div>
      </div>
      <div class="card-footer">
        <div class="d-flex align-items-center flex-wrap"> 
          <div> Showing <?php echo count($topDeals); ?> Entries <i class="bi bi-arrow-right ms-2 fw-semibold"></i> </div> 
          <div class="ms-auto"> 
            <nav aria-label="Page navigation" class="pagination-style-2">
              <ul class="pagination mb-0 flex-wrap">
                <li class="page-item disabled"><a class="page-link" href="javascript:void(0);">Prev</a></li>
                <li class="page-item active"><a class="page-link" href="javascript:void(0);">1</a></li>
                <li class="page-item disabled"><a class="page-link text-primary" href="javascript:void(0);">next</a></li>
              </ul>
            </nav>
          </div> 
        </div> 
      </div>
    </div>
  </div>
</div>
<!-- End:: row-3 -->

<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>

<!-- Apex Charts JS -->
<script src="<?php echo $baseUrl; ?>/assets/libs/apexcharts/apexcharts.min.js"></script>

<script>
// -------- Revenue Analytics (Area) --------
(function(){
  var el = document.querySelector('#crm-revenue-analytics');
  if(!el || typeof ApexCharts==='undefined') return;

  var data = <?php echo json_encode(array_values($monthTotals), JSON_NUMERIC_CHECK); ?>;
  var options = {
    chart: { type:'area', height: 280, toolbar:{show:false} },
    series: [{ name:'Revenue', data: data }],
    dataLabels: { enabled:false },
    stroke: { curve:'smooth', width:2 },
    xaxis: { categories: ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'] },
    yaxis: { labels:{ formatter: function(v){ return v.toFixed(0); } } },
    fill: { opacity:0.25 }
  };
  new ApexCharts(el, options).render();
})();

// -------- Leads By Source (Donut) --------
(function(){
  var el = document.querySelector('#leads-source');
  if(!el || typeof ApexCharts==='undefined') return;

  var labels = <?php echo json_encode($leadsSourceLabels, JSON_UNESCAPED_UNICODE); ?>;
  var series = <?php echo json_encode($leadsSourceCounts, JSON_NUMERIC_CHECK); ?>;
  if(series.length===0){ el.innerHTML = '<div class="text-muted text-center">No data</div>'; return; }
  var options = {
    chart: { type: 'donut', height: 260 },
    series: series,
    labels: labels,
    legend: { position:'bottom' }
  };
  new ApexCharts(el, options).render();
})();

// -------- Deals Statistics (Bar by Stage) --------
(function(){
  var el = document.querySelector('#deals-statistics');
  if(!el || typeof ApexCharts==='undefined') return;

  var stages = ['Prospecting','Qualification','Proposal','Negotiation','Closed Won'];
  var counts = <?php echo json_encode(array_values($stageMap), JSON_NUMERIC_CHECK); ?>;
  var options = {
    chart: { type:'bar', height:260, toolbar:{show:false} },
    series: [{ name:'Deals', data: counts }],
    xaxis: { categories: stages },
    dataLabels: { enabled:false }
  };
  new ApexCharts(el, options).render();
})();
</script>

<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
