<?php
/* ============================================================================
 * File: roles.php
 * Purpose: Manage Roles & Permissions + Assign Users
 * Stack : Core PHP + MySQLi (NO PDO). DB connection: ../config.php
 * Rules : utf8mb4_general_ci
 * Notes : - Superuser rule: if $_SESSION['user_id1'] is missing/0 → allow all.
 *         - Permissions included: leads.create, leads.export, activities.log, roles.manage
 * ========================================================================== */

date_default_timezone_set('Asia/Kolkata');
@session_start();
require_once __DIR__ . '/../config.php';

mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function col_exists($conn,$table,$col){
  $t = mysqli_real_escape_string($conn,$table);
  $c = mysqli_real_escape_string($conn,$col);
  $q = mysqli_query($conn,"SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='{$t}' AND COLUMN_NAME='{$c}'");
  return ($q && mysqli_num_rows($q)>0);
}

$CURRENT_USER_ID = (int)($_SESSION['user_id1'] ?? 0);

/* ---------------------------- RBAC: schema & utils ---------------------------- */
function rbac_ensure_schema($conn){
  // roles
  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `roles` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `name` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
    `description` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `uniq_role_name` (`name`)
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

  // permissions registry (optional but useful for labels)
  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `permissions` (
    `code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
    `label` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
    `group_name` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
    PRIMARY KEY (`code`)
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

  // role → permissions (store by code so we don’t need FK to permissions)
  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `role_permissions` (
    `role_id` INT UNSIGNED NOT NULL,
    `perm_code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
    PRIMARY KEY (`role_id`,`perm_code`),
    KEY `idx_perm_code` (`perm_code`),
    CONSTRAINT `fk_role_permissions_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

  // user → roles mapping (no FK to users table to keep it generic)
  @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `user_roles` (
    `user_id` INT NOT NULL,
    `role_id` INT UNSIGNED NOT NULL,
    PRIMARY KEY (`user_id`,`role_id`),
    KEY `idx_role_id` (`role_id`),
    CONSTRAINT `fk_user_roles_role` FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
  ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
}

function rbac_seed_permissions($conn){
  $perms = [
    ['leads.create','Create leads','Leads'],
    ['leads.export','Export leads (CSV)','Leads'],
    ['activities.log','Log activities (call/mail/task/meeting)','Activities'],
    ['roles.manage','Manage roles & permissions','Admin']
  ];
  foreach($perms as $p){
    $code = mysqli_real_escape_string($conn,$p[0]);
    $label = mysqli_real_escape_string($conn,$p[1]);
    $grp = mysqli_real_escape_string($conn,$p[2]);
    @mysqli_query($conn, "INSERT IGNORE INTO `permissions` (`code`,`label`,`group_name`) VALUES ('{$code}','{$label}','{$grp}')");
  }
}

function rbac_has($conn,$uid,$code){
  if(!$uid){ return true; } // superuser mode if no auth wired yet
  $code = mysqli_real_escape_string($conn,$code);
  $uid  = (int)$uid;
  $q = mysqli_query($conn,"SELECT 1
                           FROM user_roles ur
                           JOIN role_permissions rp ON rp.role_id = ur.role_id
                           WHERE ur.user_id={$uid} AND rp.perm_code='{$code}'
                           LIMIT 1");
  return ($q && mysqli_num_rows($q)>0);
}

rbac_ensure_schema($conn);
rbac_seed_permissions($conn);

$CAN_MANAGE_ROLES = rbac_has($conn,$CURRENT_USER_ID,'roles.manage');

/* --------------------------------- Access gate -------------------------------- */
if(!$CAN_MANAGE_ROLES){
  http_response_code(403);
  echo '<!doctype html><html><head><meta charset="utf-8"><title>Forbidden</title></head><body style="font-family:system-ui;padding:24px"><h2>403 — Not authorized</h2><p>You do not have permission to manage roles.</p></body></html>';
  exit;
}

/* ---------------------------------- Actions ---------------------------------- */
$flash = '';
if($_SERVER['REQUEST_METHOD']==='POST'){
  $act = $_POST['action'] ?? '';

  if($act==='create_role'){
    $name = trim($_POST['name'] ?? '');
    $desc = trim($_POST['description'] ?? '');
    if($name!==''){
      $stmt = mysqli_prepare($conn,"INSERT INTO `roles` (`name`,`description`) VALUES (?,?)");
      mysqli_stmt_bind_param($stmt,'ss',$name,$desc);
      $ok = mysqli_stmt_execute($stmt);
      $roleId = $ok ? mysqli_insert_id($conn) : 0;
      mysqli_stmt_close($stmt);

      // permissions
      $perms = isset($_POST['perms']) ? (array)$_POST['perms'] : [];
      if($roleId && $perms){
        $values = [];
        foreach($perms as $pc){ $pc = mysqli_real_escape_string($conn,$pc); $values[] = "({$roleId},'{$pc}')"; }
        @mysqli_query($conn,"INSERT IGNORE INTO `role_permissions` (`role_id`,`perm_code`) VALUES ".implode(',',$values));
      }
      $flash = $ok ? 'Role created.' : 'Error creating role.';
    } else {
      $flash = 'Role name is required.';
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)); exit;
  }

  if($act==='update_role'){
    $roleId = (int)($_POST['id'] ?? 0);
    $name = trim($_POST['name'] ?? '');
    $desc = trim($_POST['description'] ?? '');
    if($roleId>0 && $name!==''){
      $stmt = mysqli_prepare($conn,"UPDATE `roles` SET `name`=?, `description`=? WHERE `id`=?");
      mysqli_stmt_bind_param($stmt,'ssi',$name,$desc,$roleId);
      $ok = mysqli_stmt_execute($stmt);
      mysqli_stmt_close($stmt);

      // reset permissions
      @mysqli_query($conn,"DELETE FROM `role_permissions` WHERE `role_id`={$roleId}");
      $perms = isset($_POST['perms']) ? (array)$_POST['perms'] : [];
      if($perms){
        $values = [];
        foreach($perms as $pc){ $pc = mysqli_real_escape_string($conn,$pc); $values[] = "({$roleId},'{$pc}')"; }
        @mysqli_query($conn,"INSERT IGNORE INTO `role_permissions` (`role_id`,`perm_code`) VALUES ".implode(',',$values));
      }
      $flash = $ok ? 'Role updated.' : 'Error updating role.';
    } else {
      $flash = 'Invalid role data.';
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)); exit;
  }

  if($act==='delete_role'){
    $roleId = (int)($_POST['id'] ?? 0);
    if($roleId>0){
      // delete cascades remove role_permissions & user_roles via FK on role_id
      @mysqli_query($conn,"DELETE FROM `roles` WHERE `id`={$roleId} LIMIT 1");
      $flash = 'Role deleted.';
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)); exit;
  }

  if($act==='assign_users'){
    $roleId = (int)($_POST['id'] ?? 0);
    $idsCsv = trim($_POST['user_ids'] ?? '');
    if($roleId>0 && $idsCsv!==''){
      $parts = preg_split('/[,\s]+/',$idsCsv);
      $values = [];
      foreach($parts as $raw){
        if($raw==='') continue;
        $uid = (int)$raw;
        if($uid<=0) continue;
        $values[] = "({$uid},{$roleId})";
      }
      if($values){
        @mysqli_query($conn,"INSERT IGNORE INTO `user_roles` (`user_id`,`role_id`) VALUES ".implode(',',$values));
        $flash = 'User(s) assigned to role.';
      }
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)."#r{$roleId}"); exit;
  }

  if($act==='remove_user'){
    $roleId = (int)($_POST['role_id'] ?? 0);
    $uid    = (int)($_POST['user_id'] ?? 0);
    if($roleId>0 && $uid>0){
      @mysqli_query($conn,"DELETE FROM `user_roles` WHERE `user_id`={$uid} AND `role_id`={$roleId} LIMIT 1");
      $flash = 'User removed from role.';
    }
    header("Location: ".$_SERVER['PHP_SELF']."?flash=".urlencode($flash)."#r{$roleId}"); exit;
  }
}

$flash = $_GET['flash'] ?? '';

/* ----------------------------------- Data ----------------------------------- */
$perms = [];
$pr = mysqli_query($conn,"SELECT code,label,group_name FROM permissions ORDER BY group_name, code");
if($pr){ while($p = mysqli_fetch_assoc($pr)){ $perms[] = $p; } }

$roles = [];
$res = mysqli_query($conn, "SELECT r.*,
         (SELECT GROUP_CONCAT(rp.perm_code ORDER BY rp.perm_code SEPARATOR ',') FROM role_permissions rp WHERE rp.role_id=r.id) AS perms_csv,
         (SELECT COUNT(*) FROM user_roles ur WHERE ur.role_id=r.id) AS user_count
         FROM roles r
         ORDER BY r.name");
if($res){ while($row=mysqli_fetch_assoc($res)){ $roles[]=$row; } }

$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));
?>
<?php ob_start(); ?>
<style>
.badge-soft { background: rgba(0,0,0,.04); color: #222; }
.permission-chip{display:inline-block;padding:.25rem .5rem;border:1px solid #e5e7eb;border-radius:999px;font-size:.8rem;margin:.15rem  .25rem .15rem 0;}
.role-card{border:1px solid #eee;border-radius:10px;padding:12px;margin-bottom:12px;background:#fff;}
.user-pill{display:inline-flex;align-items:center;gap:6px;border:1px solid #e5e7eb;border-radius:999px;padding:.2rem .45rem;margin:.17rem .25rem .17rem 0;}
.user-pill button{border:none;background:none;color:#c00;}
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>

<!-- Header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Roles & Permissions</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Settings</a></li>
      <li class="breadcrumb-item active" aria-current="page">Roles</li>
    </ol>
  </div>
</div>

<?php if($flash): ?><div class="alert alert-success py-2 px-3 mb-3"><?php echo h($flash); ?></div><?php endif; ?>

<div class="card custom-card">
  <div class="card-header d-flex align-items-center justify-content-between">
    <div class="card-title">Manage Roles</div>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createRoleModal"><i class="ri-add-line me-1"></i>Create Role</button>
  </div>
  <div class="card-body">
    <?php if(!$roles): ?>
      <div class="text-muted">No roles yet. Create your first role.</div>
    <?php else: ?>
      <?php foreach($roles as $R):
        $permsCsv = trim((string)$R['perms_csv']);
        $permsArr = $permsCsv ? explode(',',$permsCsv) : [];
      ?>
      <div class="role-card" id="r<?php echo (int)$R['id']; ?>">
        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
          <div>
            <div class="fw-bold fs-16"><?php echo h($R['name']); ?></div>
            <?php if($R['description']): ?><div class="text-muted"><?php echo h($R['description']); ?></div><?php endif; ?>
          </div>
          <div class="d-flex gap-2">
            <button class="btn btn-sm btn-info-light btn-edit-role"
              data-id="<?php echo (int)$R['id']; ?>"
              data-name="<?php echo h($R['name']); ?>"
              data-desc="<?php echo h($R['description']); ?>"
              data-perms="<?php echo h($permsCsv); ?>"
            ><i class="ri-pencil-line me-1"></i>Edit</button>

            <button class="btn btn-sm btn-secondary btn-assign-users"
              data-id="<?php echo (int)$R['id']; ?>"
              data-name="<?php echo h($R['name']); ?>"
            ><i class="ri-user-add-line me-1"></i>Assign Users</button>

            <form method="post" onsubmit="return confirm('Delete this role?');" class="d-inline">
              <input type="hidden" name="action" value="delete_role">
              <input type="hidden" name="id" value="<?php echo (int)$R['id']; ?>">
              <button class="btn btn-sm btn-danger-light"><i class="ri-delete-bin-line me-1"></i>Delete</button>
            </form>
          </div>
        </div>

        <!-- Permissions -->
        <div class="mt-2">
          <?php if(!$permsArr): ?>
            <span class="text-muted">No permissions assigned.</span>
          <?php else: foreach($permsArr as $pc): ?>
            <span class="permission-chip"><?php echo h($pc); ?></span>
          <?php endforeach; endif; ?>
        </div>

        <!-- Users with this role -->
        <div class="mt-3">
          <div class="text-muted mb-1">Users with this role (count: <?php echo (int)$R['user_count']; ?>):</div>
          <div class="d-flex flex-wrap">
            <?php
              $users = [];
              $ur = mysqli_query($conn,"SELECT user_id FROM user_roles WHERE role_id=".(int)$R['id']." ORDER BY user_id");
              if($ur){ while($u=mysqli_fetch_assoc($ur)){ $users[]=(int)$u['user_id']; } }
              if(!$users) echo '<span class="text-muted">— none —</span>';
              foreach($users as $uid):
            ?>
              <span class="user-pill">User #<?php echo (int)$uid; ?>
                <form method="post" class="d-inline">
                  <input type="hidden" name="action" value="remove_user">
                  <input type="hidden" name="role_id" value="<?php echo (int)$R['id']; ?>">
                  <input type="hidden" name="user_id" value="<?php echo (int)$uid; ?>">
                  <button title="Remove" aria-label="Remove"><i class="ri-close-line"></i></button>
                </form>
              </span>
            <?php endforeach; ?>
          </div>
        </div>
      </div>
      <?php endforeach; ?>
    <?php endif; ?>
  </div>
</div>

<!-- Create Role Modal -->
<div class="modal fade" id="createRoleModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="create_role">
      <div class="modal-header"><h6 class="modal-title">Create Role</h6><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
      <div class="modal-body">
        <div class="mb-2"><label class="form-label">Role Name</label><input type="text" class="form-control" name="name" required></div>
        <div class="mb-2"><label class="form-label">Description</label><input type="text" class="form-control" name="description"></div>
        <div class="mb-2"><label class="form-label">Permissions</label>
          <div class="row">
            <?php foreach($perms as $p): ?>
              <div class="col-12 col-md-6">
                <label class="form-check">
                  <input class="form-check-input" type="checkbox" name="perms[]" value="<?php echo h($p['code']); ?>">
                  <span class="form-check-label"><?php echo h(($p['group_name']?$p['group_name'].': ':'').($p['label'] ?: $p['code'])); ?></span>
                </label>
              </div>
            <?php endforeach; ?>
          </div>
        </div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button><button class="btn btn-primary">Create</button></div>
    </form>
  </div>
</div>

<!-- Edit Role Modal -->
<div class="modal fade" id="editRoleModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post" id="edit-role-form">
      <input type="hidden" name="action" value="update_role">
      <input type="hidden" name="id" id="er-id">
      <div class="modal-header"><h6 class="modal-title">Edit Role</h6><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
      <div class="modal-body">
        <div class="mb-2"><label class="form-label">Role Name</label><input type="text" class="form-control" name="name" id="er-name" required></div>
        <div class="mb-2"><label class="form-label">Description</label><input type="text" class="form-control" name="description" id="er-desc"></div>
        <div class="mb-2"><label class="form-label">Permissions</label>
          <div class="row" id="er-perms">
            <?php foreach($perms as $p): ?>
              <div class="col-12 col-md-6">
                <label class="form-check">
                  <input class="form-check-input er-perm" type="checkbox" name="perms[]" value="<?php echo h($p['code']); ?>">
                  <span class="form-check-label"><?php echo h(($p['group_name']?$p['group_name'].': ':'').($p['label'] ?: $p['code'])); ?></span>
                </label>
              </div>
            <?php endforeach; ?>
          </div>
        </div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button><button class="btn btn-primary">Save</button></div>
    </form>
  </div>
</div>

<!-- Assign Users Modal -->
<div class="modal fade" id="assignUsersModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="assign_users">
      <input type="hidden" name="id" id="au-role-id">
      <div class="modal-header"><h6 class="modal-title" id="au-title">Assign Users</h6><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
      <div class="modal-body">
        <div class="mb-2">
          <label class="form-label">User IDs (comma or space separated)</label>
          <input type="text" class="form-control" name="user_ids" placeholder="e.g. 1, 2 3">
        </div>
        <div class="small text-muted">Tip: if you don’t have a users table wired, just map your numeric IDs (the app reads <code>$_SESSION['user_id1']</code>).</div>
      </div>
      <div class="modal-footer"><button type="button" class="btn btn-light" data-bs-dismiss="modal">Close</button><button class="btn btn-primary">Assign</button></div>
    </form>
  </div>
</div>

<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
(function(){
  // Edit Role
  document.querySelectorAll('.btn-edit-role').forEach(btn=>{
    btn.addEventListener('click', function(){
      const d = this.dataset;
      document.getElementById('er-id').value = d.id||'';
      document.getElementById('er-name').value = d.name||'';
      document.getElementById('er-desc').value = d.desc||'';
      const selected = (d.perms||'').split(',').map(s=>s.trim());
      document.querySelectorAll('#er-perms .er-perm').forEach(ch => { ch.checked = selected.includes(ch.value); });
      new bootstrap.Modal(document.getElementById('editRoleModal')).show();
    });
  });

  // Assign Users
  document.querySelectorAll('.btn-assign-users').forEach(btn=>{
    btn.addEventListener('click', function(){
      const d = this.dataset;
      document.getElementById('au-role-id').value = d.id||'';
      document.getElementById('au-title').textContent = 'Assign Users to "' + (d.name||'Role') + '"';
      new bootstrap.Modal(document.getElementById('assignUsersModal')).show();
    });
  });
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
