<?php
/* =============================================================================
 * File: unit_form.php
 * Purpose: Themed Add/Edit Unit form using Melody layout slots ($styles/$content/$scripts)
 * Stack : Core PHP + MySQLi (NO PDO). DB in /broker-config.php. Collation: utf8mb4_general_ci
 * Save  : Posts to unit_save.php (existing server-side handler)
 * ============================================================================= */

date_default_timezone_set('Asia/Kolkata');
@session_start();

/* ---- DB config (ABSOLUTE from docroot) ---- */
require_once 'broker-config.php';
if (!isset($conn) || !($conn instanceof mysqli)) { die('DB connection missing.'); }

/* ---- Charset & Collation ---- */
mysqli_set_charset($conn,'utf8mb4');
@mysqli_query($conn,"SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn,"SET collation_connection='utf8mb4_general_ci'");

/* ---- Helpers ---- */
if (!function_exists('h')) { function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); } }

/* ---- Load Unit ---- */
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$unit = [
  'project_id'=>'','tower_id'=>'','unit_no'=>'','floor'=>'','unit_type'=>'2BHK',
  'saleable_area'=>'','carpet_area'=>'','plot_area'=>'',
  'facing'=>'','aspect'=>'NA','furnishing'=>'Bare',
  'bedrooms'=>'','bathrooms'=>'','balconies'=>'',
  'parking_slots'=>'0','base_rate'=>'','plc_amount'=>'0','other_charges'=>'0','gst_rate'=>'0',
  'total_price'=>'0','payment_plan'=>'','status'=>'Available',
  'hold_expires_at'=>'','hold_reason'=>'','booked_by_broker'=>'','booking_date'=>'',
  'customer_name'=>'','customer_mobile'=>'','token_amount'=>'0',
  'agreement_no'=>'','possession_date'=>'','remarks'=>''
];
if ($id) {
  $rs = mysqli_query($conn,"SELECT * FROM units WHERE id=$id");
  if ($rs && mysqli_num_rows($rs)) { $unit = mysqli_fetch_assoc($rs); }
}

/* ---- Projects ---- */
$projects = mysqli_query($conn,"SELECT id,name FROM projects WHERE status='Active' ORDER BY name");

/* ---- Towers (if project preselected) ---- */
$towers = [];
if (($id && !empty($unit['project_id'])) || (!empty($_GET['project_id']))) {
  $projIdForTowers = $id ? (int)$unit['project_id'] : (int)$_GET['project_id'];
  $towers_rs = mysqli_query($conn,"SELECT id,name FROM towers WHERE project_id=$projIdForTowers ORDER BY id");
  if($towers_rs){ while($t = mysqli_fetch_assoc($towers_rs)){ $towers[]=$t; } }
}

/* ---- Base URL for assets ---- */
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS']==='on' ? 'https://' : 'http://')
         . $_SERVER['HTTP_HOST']
         . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));

/* ================================ $styles ================================ */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.css">
<style>
  .form-section-title{ font-weight:600; margin:8px 0 14px; font-size:14px; text-transform:uppercase; letter-spacing:.4px; color:#64748b; }
  .calc-pill{display:inline-block;padding:.35rem .6rem;border-radius:999px;background:rgba(59,130,246,.08)}
  .calc-pill .v{font-weight:600}
</style>
<?php $styles = ob_get_clean();

/* =============================== $content =============================== */
ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0"><?php echo $id ? 'Edit Unit' : 'Add Unit'; ?></h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">Real Estate</a></li>
      <li class="breadcrumb-item active" aria-current="page"><?php echo $id ? 'Edit' : 'Create'; ?></li>
    </ol>
  </div>
</div>

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card">
      <div class="card-header justify-content-between">
        <div class="card-title"><i class="fa fa-door-open me-2"></i><?php echo $id ? 'Update' : 'Create'; ?> Unit</div>
        <div class="ms-auto">
          <a href="units.php" class="btn btn-outline-secondary btn-sm"><i class="fa fa-arrow-left me-2"></i>Back to Units</a>
        </div>
      </div>
      <div class="card-body">
        <form method="post" action="unit_save.php" id="unitForm" autocomplete="off">
          <input type="hidden" name="id" value="<?php echo (int)$id; ?>">

          <!-- Section: Mapping -->
          <div class="form-section-title">Mapping</div>
          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">Project <span class="text-danger">*</span></label>
              <select name="project_id" class="form-control" required>
                <option value="">-- Select --</option>
                <?php $selectedPid=(string)$unit['project_id']; while($p=mysqli_fetch_assoc($projects)): ?>
                  <option value="<?php echo (int)$p['id']; ?>" <?php echo ($selectedPid!=='' && (int)$selectedPid===(int)$p['id'])?'selected':''; ?>>
                    <?php echo h($p['name']); ?>
                  </option>
                <?php endwhile; ?>
              </select>
              <small class="text-muted">Associate the unit to a project.</small>
            </div>
            <div class="col-md-4">
              <label class="form-label">Tower / Block (optional)</label>
              <?php if(!empty($towers)): ?>
                <select name="tower_id" class="form-control">
                  <option value="">-- None --</option>
                  <?php foreach($towers as $t): ?>
                    <option value="<?php echo (int)$t['id']; ?>" <?php echo ((string)$unit['tower_id']!=='' && (int)$unit['tower_id']===(int)$t['id'])?'selected':''; ?>>
                      <?php echo h($t['name']); ?>
                    </option>
                  <?php endforeach; ?>
                </select>
              <?php else: ?>
                <input type="text" name="tower_id" class="form-control" placeholder="Leave blank if none" value="<?php echo h($unit['tower_id']); ?>">
              <?php endif; ?>
            </div>
            <div class="col-md-2">
              <label class="form-label">Unit No <span class="text-danger">*</span></label>
              <input type="text" name="unit_no" class="form-control" value="<?php echo h($unit['unit_no']); ?>" required>
            </div>
            <div class="col-md-2">
              <label class="form-label">Floor <span class="text-danger">*</span></label>
              <input type="number" name="floor" class="form-control" value="<?php echo h($unit['floor']); ?>" required>
            </div>
          </div>

          <!-- Section: Typology & Areas -->
          <div class="form-section-title">Typology & Areas</div>
          <div class="row g-3">
            <div class="col-md-3">
              <label class="form-label">Unit Type</label>
              <select name="unit_type" class="form-control">
                <?php foreach(['1BHK','2BHK','3BHK','4BHK','Shop','Office','Plot'] as $opt): ?>
                  <option <?php echo ($unit['unit_type']===$opt)?'selected':''; ?>><?php echo h($opt); ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-3">
              <label class="form-label">Saleable Area (sqft) <span class="text-danger">*</span></label>
              <input type="number" step="0.01" name="saleable_area" class="form-control calc" value="<?php echo h($unit['saleable_area']); ?>" required>
            </div>
            <div class="col-md-3">
              <label class="form-label">Carpet Area (sqft)</label>
              <input type="number" step="0.01" name="carpet_area" class="form-control" value="<?php echo h($unit['carpet_area']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Plot Area (sqyd)</label>
              <input type="number" step="0.01" name="plot_area" class="form-control" value="<?php echo h($unit['plot_area']); ?>">
            </div>
          </div>

          <!-- Section: Attributes -->
          <div class="form-section-title">Attributes</div>
          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">Facing (CSV)</label>
              <input type="text" name="facing" class="form-control" placeholder="e.g., Garden, Pool" value="<?php echo h($unit['facing']); ?>">
            </div>
            <div class="col-md-4">
              <label class="form-label">Aspect</label>
              <select name="aspect" class="form-control">
                <?php foreach(['NA','East','West','North','South'] as $a): ?>
                  <option <?php echo ($unit['aspect']===$a)?'selected':''; ?>><?php echo h($a); ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-4">
              <label class="form-label">Furnishing</label>
              <select name="furnishing" class="form-control">
                <?php foreach(['Bare','Semi','Full'] as $a): ?>
                  <option <?php echo ($unit['furnishing']===$a)?'selected':''; ?>><?php echo h($a); ?></option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>

          <!-- Section: Counts & Status -->
          <div class="form-section-title">Counts & Status</div>
          <div class="row g-3">
            <div class="col-md-2">
              <label class="form-label">Bedrooms</label>
              <input type="number" name="bedrooms" class="form-control" value="<?php echo h($unit['bedrooms']); ?>">
            </div>
            <div class="col-md-2">
              <label class="form-label">Bathrooms</label>
              <input type="number" name="bathrooms" class="form-control" value="<?php echo h($unit['bathrooms']); ?>">
            </div>
            <div class="col-md-2">
              <label class="form-label">Balconies</label>
              <input type="number" name="balconies" class="form-control" value="<?php echo h($unit['balconies']); ?>">
            </div>
            <div class="col-md-2">
              <label class="form-label">Parking Slots</label>
              <input type="number" name="parking_slots" class="form-control" value="<?php echo h($unit['parking_slots']); ?>">
            </div>
            <div class="col-md-2">
              <label class="form-label">Status</label>
              <select name="status" class="form-control">
                <?php foreach(['Available','Hold','Booked','Blocked','Unavailable'] as $st): ?>
                  <option <?php echo ($unit['status']===$st)?'selected':''; ?>><?php echo h($st); ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-2">
              <label class="form-label">Payment Plan</label>
              <input type="text" name="payment_plan" class="form-control" value="<?php echo h($unit['payment_plan']); ?>">
            </div>
          </div>

          <!-- Section: Pricing -->
          <div class="form-section-title">Pricing</div>
          <div class="row g-3">
            <div class="col-md-3">
              <label class="form-label">Base Rate (/sqft) <span class="text-danger">*</span></label>
              <input type="number" step="0.01" name="base_rate" class="form-control calc" value="<?php echo h($unit['base_rate']); ?>" required>
            </div>
            <div class="col-md-3">
              <label class="form-label">PLC Amount</label>
              <input type="number" step="0.01" name="plc_amount" class="form-control calc" value="<?php echo h($unit['plc_amount']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Other Charges</label>
              <input type="number" step="0.01" name="other_charges" class="form-control calc" value="<?php echo h($unit['other_charges']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">GST Rate (%)</label>
              <input type="number" step="0.01" name="gst_rate" class="form-control calc" value="<?php echo h($unit['gst_rate']); ?>">
            </div>
          </div>

          <!-- Inline calc summary (read-only display) -->
          <div class="mt-2">
            <span class="calc-pill">Base Total: <span class="v" id="vBase">₹ 0.00</span></span>
            <span class="calc-pill ms-2">GST Amt: <span class="v" id="vGst">₹ 0.00</span></span>
            <span class="calc-pill ms-2">Grand Total: <span class="v" id="vGrand">₹ 0.00</span></span>
          </div>

          <!-- Section: Booking (optional) -->
          <div class="form-section-title">Booking (Optional)</div>
          <div class="row g-3">
            <div class="col-md-3">
              <label class="form-label">Booking Date</label>
              <input type="date" name="booking_date" class="form-control" value="<?php echo h($unit['booking_date']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Customer Name</label>
              <input type="text" name="customer_name" class="form-control" value="<?php echo h($unit['customer_name']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Customer Mobile</label>
              <input type="text" name="customer_mobile" class="form-control" value="<?php echo h($unit['customer_mobile']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Token Amount</label>
              <input type="number" step="0.01" name="token_amount" class="form-control" value="<?php echo h($unit['token_amount']); ?>">
            </div>
          </div>

          <!-- Section: Agreement & Hold -->
          <div class="form-section-title">Agreement & Hold</div>
          <div class="row g-3">
            <div class="col-md-3">
              <label class="form-label">Agreement No</label>
              <input type="text" name="agreement_no" class="form-control" value="<?php echo h($unit['agreement_no']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Possession Date</label>
              <input type="date" name="possession_date" class="form-control" value="<?php echo h($unit['possession_date']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Hold Expires At</label>
              <input type="datetime-local" name="hold_expires_at" class="form-control" value="<?php echo h($unit['hold_expires_at']); ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">Hold Reason</label>
              <input type="text" name="hold_reason" class="form-control" value="<?php echo h($unit['hold_reason']); ?>">
            </div>
          </div>

          <!-- Section: Broker & Remarks -->
          <div class="form-section-title">Broker & Remarks</div>
          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">Booked By (Broker)</label>
              <input type="text" name="booked_by_broker" class="form-control" value="<?php echo h($unit['booked_by_broker']); ?>">
            </div>
            <div class="col-md-8">
              <label class="form-label">Remarks</label>
              <textarea name="remarks" class="form-control" rows="2"><?php echo h($unit['remarks']); ?></textarea>
            </div>
          </div>

          <div class="mt-4 d-flex gap-2">
            <button class="btn btn-primary"><i class="fa fa-save me-2"></i>Save Unit</button>
            <a href="units.php" class="btn btn-light">Cancel</a>
          </div>
        </form>
      </div>
    </div>
  </div>
</div>
<?php $content = ob_get_clean();

/* =============================== $scripts =============================== */
ob_start(); ?>
<script src="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.js"></script>
<script>
(function(){
  function toNum(v){ var n=parseFloat(v); return isNaN(n)?0:n; }
  function inr(n){
    n = Number(n||0);
    const sign = n<0?'-':'';
    n = Math.abs(n);
    const parts = n.toFixed(2).split('.');
    let s = parts[0], dec = parts[1];
    if (s.length > 3) {
      const last3 = s.slice(-3);
      let rest = s.slice(0,-3).replace(/\B(?=(\d{2})+(?!\d))/g, ',');
      s = rest + ',' + last3;
    }
    return '₹ ' + sign + s + '.' + dec;
  }
  function recalc(){
    const area = toNum(document.querySelector('[name="saleable_area"]').value);
    const rate = toNum(document.querySelector('[name="base_rate"]').value);
    const plc  = toNum(document.querySelector('[name="plc_amount"]').value);
    const oth  = toNum(document.querySelector('[name="other_charges"]').value);
    const gstR = toNum(document.querySelector('[name="gst_rate"]').value);
    const base = (rate*area)+plc+oth;
    const gst  = base * (gstR/100);
    const grand= base + gst;
    document.getElementById('vBase').textContent  = inr(base);
    document.getElementById('vGst').textContent   = inr(gst);
    document.getElementById('vGrand').textContent = inr(grand);
  }
  document.querySelectorAll('.calc').forEach(el=>{ el.addEventListener('input', recalc); });
  recalc();
})();
</script>
<?php $scripts = ob_get_clean();

/* ------------------------------ Render base ------------------------------ */
include 'layouts/base.php';
