<?php
/* File: ams_report_export_last_month.php
 * Purpose: Export AMS pivot for the PREVIOUS month to Excel (PhpSpreadsheet)
 * Notes  : - Uses ../config.php for MySQLi connection (NO PDO)
 *          - Ensures utf8mb4 charset / COLLATE=utf8mb4_general_ci on connection
 *          - Builds dynamic pivot columns for each day of the previous month
 *          - Requires PhpSpreadsheet (adjust path to vendor/autoload.php if needed)
 */

session_start();
date_default_timezone_set('Asia/Kolkata');

include('../config.php');                       // <-- must provide $conn_ic or $conn (mysqli)
require '../PhpSpreadsheet/vendor/autoload.php'; // adjust path if your vendor folder differs

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

/* ---------- Resolve mysqli connection (expecting $conn_ic or $conn) ---------- */
$db = null;
if (isset($conn_ic) && $conn_ic instanceof mysqli) {
    $db = $conn_ic;
} elseif (isset($conn) && $conn instanceof mysqli) {
    $db = $conn;
} else {
    http_response_code(500);
    echo "Database connection not found. Please ensure ../config.php defines \$conn_ic or \$conn (mysqli).";
    exit;
}

/* ---------- Ensure proper charset / collation on connection ---------- */
@mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");

/* ---------- Determine PREVIOUS month ---------- */
$firstOfPrevMonth = new DateTime('first day of last month');
$year  = $firstOfPrevMonth->format('Y');   // e.g. 2025
$month = $firstOfPrevMonth->format('m');   // e.g. 09
$monthName = $firstOfPrevMonth->format('F'); // e.g. September

$daysInMonth = (int)$firstOfPrevMonth->format('t'); // number of days in previous month

/* ---------- Build array of YYYY-MM-DD dates for previous month ---------- */
$dates = [];
for ($d = 1; $d <= $daysInMonth; $d++) {
    $dates[] = sprintf('%04d-%02d-%02d', $year, $month, $d);
}

/* ---------- Build dynamic SQL SELECT part (pivot) ---------- */
$selectPieces = [];
$selectPieces[] = 'af.empid';
foreach ($dates as $dt) {
    $safe_dt = mysqli_real_escape_string($db, $dt);
    // alias the column with full date to avoid collisions; we'll reference it by that key in PHP
    $selectPieces[] = "IFNULL(MAX(CASE WHEN OfficeDate = '" . $safe_dt . "' THEN FinalStatus END), 'A') AS `" . $safe_dt . "`";
}

$sql = "SELECT\n    " . implode(",\n    ", $selectPieces) . "\nFROM ams_final af\nGROUP BY af.empid\nORDER BY af.empid;";

/* ---------- Execute query ---------- */
$result = mysqli_query($db, $sql);
if ($result === false) {
    // helpful debug output (you may log this in production)
    $err = mysqli_error($db);
    echo "<pre>SQL error: " . htmlspecialchars($err) . "\n\nQuery:\n" . htmlspecialchars($sql) . "</pre>";
    exit;
}

/* ---------- Create spreadsheet and fill headers ---------- */
$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();

// Header row
$sheet->setCellValue('A1', 'S.No.');
$sheet->setCellValue('B1', 'Emp ID');

$colIndex = 3; // column C
foreach ($dates as $date) {
    // display header as dd-mm-YYYY (user-friendly)
    $sheet->setCellValueByColumnAndRow($colIndex++, 1, date('d-m-Y', strtotime($date)));
}

/* ---------- Fill data rows ---------- */
$rowNum = 2;
$sno = 1;
while ($row = mysqli_fetch_assoc($result)) {
    // S.No.
    $sheet->setCellValueByColumnAndRow(1, $rowNum, $sno);
    // Emp ID
    $sheet->setCellValueByColumnAndRow(2, $rowNum, $row['empid']);

    $colIndex = 3;
    foreach ($dates as $date) {
        // if column missing for some reason, fallback to 'A'
        $val = (isset($row[$date]) && $row[$date] !== null) ? $row[$date] : 'A';
        $sheet->setCellValueByColumnAndRow($colIndex++, $rowNum, $val);
    }

    $rowNum++;
    $sno++;
}
mysqli_free_result($result);

/* ---------- Output Excel file to browser ---------- */
$exportFilename = $monthName . "-export_" . date("Y-m-d") . ".xlsx";

header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment; filename="' . $exportFilename . '"');
header('Cache-Control: max-age=0');

$writer = new Xlsx($spreadsheet);
$writer->save('php://output');
exit;
?>
