<?php
/* File: ams_report_last_month.php
 * Purpose: AMS Report — auto-generates pivot for the LAST month (dynamic dates)
 * Notes  : - Uses ../config.php for DB connection (MySQLi; COLLATE=utf8mb4_general_ci; NO PDO)
 *          - Preserves original permission check and layout buffering
 *          - Query builds CASE expressions for each date of the previous month
 *          - Table header and body are generated dynamically based on number of days in last month
 */

session_start();
include('../config.php');
//require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
//checkPageAccessOrRedirect($currentPage, $conn_sa);

/* ---------- Determine previous month (server timezone) ---------- */
date_default_timezone_set('Asia/Kolkata'); // match project timezone
$firstOfPrevMonth = new DateTime('first day of last month');
$year  = $firstOfPrevMonth->format('Y');   // e.g. 2025
$month = $firstOfPrevMonth->format('m');   // e.g. 09
$daysInMonth = (int)$firstOfPrevMonth->format('t'); // 28..31

// build array of YYYY-MM-DD strings for each day of previous month
$dates = [];
for ($d = 1; $d <= $daysInMonth; $d++) {
    $dates[] = sprintf('%s-%s-%02d', $year, $month, $d);
}

/* ---------- Build dynamic SQL (pivot-like) ---------- */
$selectPieces = [];
$selectPieces[] = 'empid';
foreach ($dates as $dt) {
    // alias uses the full date string (safe); will be accessed by $row[$dt]
    $selectPieces[] = "IFNULL(MAX(CASE WHEN OfficeDate = '" . mysqli_real_escape_string($conn_ic, $dt) . "' THEN FinalStatus END), 'A') AS `" . $dt . "`";
}
$query = "SELECT\n    " . implode(",\n    ", $selectPieces) . "\nFROM ams_final\nGROUP BY empid\nORDER BY empid;";

/* ---------- Execute query ---------- */
if (isset($conn_ic) && $conn_ic instanceof mysqli) {
    @mysqli_set_charset($conn_ic, 'utf8mb4');
} else {
    // fallback if connection variable differs (keeps compatibility with different setups)
    if (isset($conn) && $conn instanceof mysqli) {
        $conn_ic = $conn;
        @mysqli_set_charset($conn_ic, 'utf8mb4');
    } else {
        http_response_code(500);
        echo "Database connection not found. Please ensure ../config.php defines \$conn_ic or \$conn (mysqli).";
        exit;
    }
}

$result = mysqli_query($conn_ic, $query);
if ($result === false) {
    // helpful debug for developers; in production you may want to log instead
    $err = mysqli_error($conn_ic);
    echo "<pre>SQL error: " . htmlspecialchars($err) . "\n\nQuery:\n" . htmlspecialchars($query) . "</pre>";
    exit;
}

/* ---------- Render page (keeps original buffering/layout pattern) ---------- */
?>
<?php ob_start(); ?>

<?php $styles = ob_get_clean(); ?>
<?php ob_start(); ?>
	
	<!-- Start::page-header -->
	<div class="page-header-breadcrumb mb-3">
		<div class="d-flex align-center justify-content-between flex-wrap">
			<h1 class="page-title fw-medium fs-18 mb-0">AMS Report — <?php echo htmlspecialchars($firstOfPrevMonth->format('F Y')); ?></h1>
			<ol class="breadcrumb mb-0">
				<li class="breadcrumb-item"><a href="javascript:void(0);">Pages</a></li>
				<li class="breadcrumb-item active" aria-current="page">AMS Report</li>
			</ol>
		</div>
	</div>
	<!-- End::page-header -->
	
	<!-- Start::row-1 -->
	<div class="row">
		<div class="col-xl-12">
			<div class="card custom-card">
				<div class="card-body table-responsive">
					<?php
					// Export link can accept year-month (e.g. 2025-09) so export page can pick correct month
					$ym = $year . '-' . $month;
					?>
					<a href="ams-report-ap-lm-export.php?ym=<?php echo urlencode($ym); ?>" class="btn btn-outline-secondary text-end">Export Data</a>

					<table class="table text-nowrap table-striped table-bordered bootstrap-datatable datatable responsive">
						<thead>
						<tr>
							<th>S.N.</th>
							<th>Emp Id</th>
							<?php
							// print day headers (01,02,...), based on last month days
							foreach ($dates as $dt) {
								$day = substr($dt, -2); // last two chars -> day number
								echo '<th>' . htmlspecialchars($day) . '</th>';
							}
							?>
						</tr>
						</thead>
						<tbody>
						<?php
						$sn = 0;
						while ($data = mysqli_fetch_assoc($result)) {
							$sn++;
							echo "<tr>\n";
							echo "<td>" . (int)$sn . "</td>\n";
							echo "<td>" . htmlspecialchars($data['empid']) . "</td>\n";

							// output each date column in the same order as header
							foreach ($dates as $dt) {
								$cell = isset($data[$dt]) ? $data[$dt] : 'A';
								// sanitize cell value before output
								echo '<td>' . htmlspecialchars($cell) . '</td>';
							}

							echo "\n</tr>\n";
						}
						// free result
						mysqli_free_result($result);
						?>
						</tbody>
					</table>
				</div>
			</div>
		</div>
	</div>
	<!--End::row-1 -->
<?php $content = ob_get_clean(); ?>
<?php ob_start(); ?>
<?php $scripts = ob_get_clean(); ?>
<?php include 'layouts/base.php'; ?>
