<?php
// ============================================================================
// File: applied_calling_form_applications.php
// Purpose:
//   - Show "Applied Calling Form Applications" for logged-in recruiter
//   - Theme & style same as Jobs List (checkboxes, colors, badges, search box)
//   - Auth via signin.php / HR sessions (empid / hr_empid)
//   - DB: MySQLi via ../config.php (NO PDO), COLLATE=utf8mb4_general_ci
// ============================================================================

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* -------------------------- Load central config -------------------------- */
/* IMPORTANT: we always use ../config.php and $conn */
$cfgPath = __DIR__ . '/../config.php';
if (!file_exists($cfgPath)) {
    http_response_code(500);
    die('config.php not found at ../config.php');
}
require_once $cfgPath;

if (!isset($conn) || !($conn instanceof mysqli)) {
    http_response_code(500);
    die('Database connection $conn not available');
}

/* Collation / Charset */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4'");
@mysqli_query($conn, "SET collation_connection='utf8mb4_general_ci'");

/* ------------------------------ AUTH / SESSION ------------------------------ */
/*
 * We support both session patterns:
 *  - New login (signin.php):  $_SESSION['empid'], $_SESSION['username']
 *  - Old HR login:            $_SESSION['hr_empid'], $_SESSION['namevfied1']
 */
$loggedEmpid = '';
$loggedName  = '';

if (!empty($_SESSION['user_empid'])) {
    $loggedEmpid = (string)$_SESSION['user_empid'];
} elseif (!empty($_SESSION['hr_empid'])) {
    $loggedEmpid = (string)$_SESSION['hr_empid'];
}

if (!empty($_SESSION['username'])) {
    $loggedName = (string)$_SESSION['username'];
} elseif (!empty($_SESSION['namevfied1'])) {
    $loggedName = (string)$_SESSION['namevfied1'];
}

/* If no empid at all, send to signin.php (or change to index.php if you want) */
if ($loggedEmpid === '') {
    header('Location: login.php');
    exit;
}

/* -------------------------- AUTO BASE URL -------------------------- */
$rootFolder = basename($_SERVER['DOCUMENT_ROOT']);
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on'
    ? 'https://' : 'http://')
    . $_SERVER['HTTP_HOST']
    . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));

// Helpers
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function buildQuery(array $override = []){
    $params = $_GET;
    foreach ($override as $k => $v) {
        if ($v === null) {
            unset($params[$k]);
        } else {
            $params[$k] = $v;
        }
    }
    return '?' . http_build_query($params);
}

/* -------------------------- FETCH CALLING DATA -------------------------- */
$rows = [];
$totalRows = 0;

$empEsc = mysqli_real_escape_string($conn, $loggedEmpid);

// Pagination + sorting (server-side)
$perPageOptions = [10, 25, 50, 100];
$perPage = isset($_GET['per_page']) ? (int)$_GET['per_page'] : 10;
if (!in_array($perPage, $perPageOptions, true)) {
    $perPage = 10;
}

$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$page = max(1, $page);

$sortParam = isset($_GET['sort']) ? (string)$_GET['sort'] : 'call_id';
$dirParam  = isset($_GET['dir']) ? strtolower((string)$_GET['dir']) : 'desc';
$dir       = $dirParam === 'asc' ? 'ASC' : 'DESC';

$allowedSorts = [
    'call_id'        => 'cf.callingid',
    'call_date'      => 'cf.calling_date',
    'interview_date' => 'cf.calling_interview_date',
    'status'         => 'cf.calling_remark',
    'login'          => 'emp.created_at',
    'pif'            => 'pif.created_at',
];

$sortColumn = isset($allowedSorts[$sortParam]) ? $allowedSorts[$sortParam] : $allowedSorts['call_id'];

// Count total
$countSql = "SELECT COUNT(*) AS cnt FROM calling_form cf WHERE cf.recruiter_name = '{$empEsc}'";
$countRes = mysqli_query($conn, $countSql);
if ($countRes && ($countRow = mysqli_fetch_assoc($countRes))) {
    $totalRows = (int)$countRow['cnt'];
}

$totalPages = max(1, (int)ceil($totalRows / $perPage));
$page = min($page, $totalPages);
$offset = ($page - 1) * $perPage;

// Main query with joins for Candidate Login & PIF Submit
$sql = "
    SELECT
        cf.callingid,
        cf.calling_date,
        cf.username,
        cf.phone,
        cf.email,
        cf.calling_remark,
        cf.calling_interview_date,
        cf.CallFrom,
        emp.created_at AS candidate_login_at,
        pif.created_at AS pif_created_at
    FROM calling_form cf
    LEFT JOIN employees emp ON emp.email = cf.email
    LEFT JOIN user_pif pif ON pif.email = cf.email
    WHERE cf.recruiter_name = '{$empEsc}'
    ORDER BY {$sortColumn} {$dir}
    LIMIT {$perPage} OFFSET {$offset}
";

$result = mysqli_query($conn, $sql);
if ($result) {
    while ($r = mysqli_fetch_assoc($result)) {
        $rows[] = $r;
    }
} else {
    // If there is an SQL error, show it once to help debug
    $errorMsg = mysqli_error($conn);
}

/* ======================= INTERNAL STYLES ($styles) ======================= */
ob_start();
?>
<style>
    .page-title {
        font-weight: 600;
    }
    .badge-status {
        font-size: 11px;
        padding: 4px 8px;
    }

    /* Blinking call-to-action buttons */
    .btn-blink {
        position: relative;
        animation: blinkPulse 1.2s infinite;
    }
    @keyframes blinkPulse {
        0%, 100% {
            box-shadow: 0 0 0 0 rgba(79,70,229,.45);
            transform: translateY(0);
        }
        50% {
            box-shadow: 0 0 0 8px rgba(79,70,229,0);
            transform: translateY(-1px);
        }
    }
</style>
<?php
$styles = ob_get_clean();

/* ========================= PAGE CONTENT ($content) ======================== */
ob_start();
?>

<!-- Start::page-header -->
<div class="page-header-breadcrumb mb-3">
    <div class="d-flex align-center justify-content-between flex-wrap">
        <h1 class="page-title fw-medium fs-18 mb-0">Applied Calling Form Applications</h1>
        <ol class="breadcrumb mb-0">
            <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
            <li class="breadcrumb-item"><a href="javascript:void(0);">HR</a></li>
            <li class="breadcrumb-item active" aria-current="page">Applied Calling</li>
        </ol>
    </div>
</div>
<!-- End::page-header -->

<!-- Start::row-1 -->
<div class="row">
    <div class="col-xl-12">
        <div class="card custom-card overflow-hidden">
            <div class="card-header justify-content-between">
                <div class="card-title">
                    Applied Calling Form Applications
                    <div class="small text-muted mt-1">
                        Welcome:
                        <strong><?php echo htmlspecialchars((string)$loggedName, ENT_QUOTES, 'UTF-8'); ?></strong>
                        (<?php echo htmlspecialchars((string)$loggedEmpid, ENT_QUOTES, 'UTF-8'); ?>)
                        &nbsp; | &nbsp;
                        <?php echo date('d-M-Y'); ?>
                        &nbsp; | &nbsp;
                        <span class="badge rounded-pill bg-primary-transparent">
                            Total Records: <?php echo (int)$totalRows; ?>
                        </span>
                    </div>
                </div>

                <div class="d-flex flex-wrap gap-2">
                    <!-- New Calling Form (blink) -->
                    <a href="calling_form_new_application.php"
                       class="btn btn-success btn-wave btn-blink">
                        <i class="ri-phone-line me-1 align-middle"></i>
                        New Calling Form
                    </a>

                    <!-- Direct Reference (blink) -->
                    <a href="direct_reference_application.php"
                       class="btn btn-warning btn-wave btn-blink">
                        <i class="ri-user-add-line me-1 align-middle"></i>
                        Direct Reference
                    </a>

                    <!-- Search box -->
                    <div>
                        <input class="form-control" type="text" placeholder="Search Here"
                               aria-label=".form-control-sm example" id="callSearch">
                    </div>

                    <!-- Sort dropdown -->
                    <div class="dropdown">
                        <a href="javascript:void(0);" class="btn btn-primary btn-wave"
                           data-bs-toggle="dropdown" aria-expanded="false">
                            Sort By<i class="ri-arrow-down-s-line align-middle ms-1 d-inline-block"></i>
                        </a>
                        <ul class="dropdown-menu" role="menu">
                            <?php
                            $sortOptions = [
                                'Newest Call'           => ['call_id', 'desc'],
                                'Oldest Call'           => ['call_id', 'asc'],
                                'Call Date'             => ['call_date', $sortParam==='call_date' && $dir==='ASC' ? 'desc' : 'asc'],
                                'Interview Date'        => ['interview_date', $sortParam==='interview_date' && $dir==='ASC' ? 'desc' : 'asc'],
                                'Status'                => ['status', $sortParam==='status' && $dir==='ASC' ? 'desc' : 'asc'],
                                'Candidate Login Date'  => ['login', $sortParam==='login' && $dir==='ASC' ? 'desc' : 'asc'],
                                'PIF Submitted Date'    => ['pif', $sortParam==='pif' && $dir==='ASC' ? 'desc' : 'asc'],
                            ];
                            foreach ($sortOptions as $label => $opt):
                                $url = buildQuery(['sort'=>$opt[0],'dir'=>$opt[1],'page'=>1]);
                            ?>
                                <li><a class="dropdown-item" href="<?= h($url); ?>"><?= h($label); ?></a></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>

                    <!-- Per page -->
                    <form method="get" class="d-flex align-items-center gap-2">
                        <input type="hidden" name="sort" value="<?= h($sortParam); ?>">
                        <input type="hidden" name="dir" value="<?= strtolower(h($dir)); ?>">
                        <label for="per_page" class="mb-0 small fw-medium">Per Page</label>
                        <select class="form-select" id="per_page" name="per_page" onchange="this.form.submit()">
                            <?php foreach ($perPageOptions as $opt): ?>
                                <option value="<?= (int)$opt; ?>" <?= $opt===$perPage ? 'selected' : ''; ?>><?= (int)$opt; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </form>
                </div>
            </div>

            <div class="card-body p-0">
                <?php if (!empty($errorMsg)): ?>
                    <div class="alert alert-danger m-3">
                        SQL Error: <?php echo htmlspecialchars((string)$errorMsg, ENT_QUOTES, 'UTF-8'); ?>
                    </div>
                <?php endif; ?>

                <div class="table-responsive">
                    <table class="table table-hover text-nowrap" id="CallingTable">
                        <thead>
                        <tr>
                            <th scope="row" class="ps-4">
                                <input class="form-check-input check-all" type="checkbox"
                                       id="checkboxAllCalls" value="" aria-label="Select all">
                            </th>
                            <th>Applicant / Call-ID</th>
                            <th>Form Type</th>
                            <th>Contact & Email</th>
                            <th>Status</th>
                            <th>Call Date</th>
                            <th>Candidate Login</th>
                            <th>PIF Submit</th>
                            <th>Interview Date</th>
                            <th>Action</th>
                        </tr>
                        </thead>
                        <tbody>
                        <?php if (!empty($rows)) : ?>
                            <?php
                            $avatarClasses = [
                                'bg-primary-transparent svg-primary',
                                'bg-secondary-transparent svg-secondary',
                                'bg-warning-transparent svg-warning',
                                'bg-info-transparent svg-info',
                                'bg-success-transparent svg-success',
                                'bg-danger-transparent svg-danger',
                                'bg-teal-transparent svg-teal',
                                'bg-pink-transparent svg-pink',
                                'bg-purple-transparent svg-purple'
                            ];
                            ?>
                            <?php foreach ($rows as $i => $data) : ?>
                                <?php
                                $callingid   = isset($data['callingid']) ? (string)$data['callingid'] : '';
                                $callingDate = isset($data['calling_date']) ? (string)$data['calling_date'] : '';
                                $username    = isset($data['username']) ? (string)$data['username'] : '';
                                $phone       = isset($data['phone']) ? (string)$data['phone'] : '';
                                $email       = isset($data['email']) ? (string)$data['email'] : '';
                                $remark      = isset($data['calling_remark']) ? trim((string)$data['calling_remark']) : '';
                                $intDate     = isset($data['calling_interview_date']) ? (string)$data['calling_interview_date'] : '';
                                $callFrom    = isset($data['CallFrom']) ? (string)$data['CallFrom'] : '';
                                $loginAt     = isset($data['candidate_login_at']) ? (string)$data['candidate_login_at'] : '';
                                $pifAt       = isset($data['pif_created_at']) ? (string)$data['pif_created_at'] : '';

                                // Status color mapping
                                $statusClass = 'bg-primary-transparent';
                                $remarkLower = strtolower($remark);
                                if (in_array($remarkLower, ['interested', 'applied', 'joined'])) {
                                    $statusClass = 'bg-success-transparent';
                                } elseif (in_array($remarkLower, ['not interested', 'rejected'])) {
                                    $statusClass = 'bg-danger-transparent';
                                } elseif (in_array($remarkLower, ['not picked', 'rnr', 'not reachable', 'switch off', 'busy'])) {
                                    $statusClass = 'bg-warning-transparent';
                                }

                                $avatarClass = $avatarClasses[$i % count($avatarClasses)];

                                $formLabel = ($callFrom === 'Reference') ? 'Direct Reference' : 'New Calling Form';
                                $formBadge = ($callFrom === 'Reference') ? 'bg-warning-transparent' : 'bg-success-transparent';
                                $formLink  = ($callFrom === 'Reference')
                                    ? 'direct_reference_application.php?callingid=' . urlencode($callingid)
                                    : 'calling_form_new_application.php?callingid=' . urlencode($callingid);

                                $loginText = $loginAt ? date('d-M-Y H:i', strtotime($loginAt)) : '-';
                                $pifText   = $pifAt ? date('d-M-Y H:i', strtotime($pifAt)) : '-';
                                ?>
                                <tr class="joblist-list">
                                    <!-- Checkbox -->
                                    <td class="ps-4 joblist-checkbox">
                                        <input class="form-check-input" type="checkbox"
                                               value="<?php echo htmlspecialchars($callingid, ENT_QUOTES, 'UTF-8'); ?>"
                                               aria-label="Select row">
                                    </td>

                                    <!-- Applicant + Call-ID -->
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="lh-1">
                                                <span class="avatar avatar-md avatar-rounded <?php echo $avatarClass; ?>">
                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 256">
                                                        <rect width="256" height="256" fill="none"/>
                                                        <path d="M44,64A20,20,0,0,1,64,44h24a20,20,0,0,1,20,20v24a12,12,0,0,1-3.51,8.49L90,111a120.12,120.12,0,0,0,55,55l14.54-14.54A12,12,0,0,1,168,148h24a20,20,0,0,1,20,20v24a20,20,0,0,1-20,20h-8A148,148,0,0,1,44,72Z"
                                                              fill="none" stroke="currentColor" stroke-linecap="round"
                                                              stroke-linejoin="round" stroke-width="16"/>
                                                    </svg>
                                                </span>
                                            </div>
                                            <div class="ms-2">
                                                <p class="fw-medium mb-0 d-flex align-items-center">
                                                    <?php echo htmlspecialchars($username, ENT_QUOTES, 'UTF-8'); ?>
                                                </p>
                                                <p class="fs-12 text-muted mb-0">
                                                    Call-ID: <?php echo htmlspecialchars($callingid, ENT_QUOTES, 'UTF-8'); ?>
                                                </p>
                                            </div>
                                        </div>
                                    </td>

                                    <!-- Form Type -->
                                    <td>
                                        <span class="badge rounded-pill <?php echo $formBadge; ?> badge-status">
                                            <?php echo htmlspecialchars($formLabel, ENT_QUOTES, 'UTF-8'); ?>
                                        </span>
                                    </td>

                                    <!-- Contact & Email -->
                                    <td>
                                        <?php echo htmlspecialchars($phone, ENT_QUOTES, 'UTF-8'); ?><br>
                                        <small class="text-muted"><?php echo htmlspecialchars($email, ENT_QUOTES, 'UTF-8'); ?></small>
                                    </td>

                                    <!-- Status -->
                                    <td>
                                        <span class="badge rounded-pill <?php echo $statusClass; ?> badge-status">
                                            <?php echo htmlspecialchars($remark, ENT_QUOTES, 'UTF-8'); ?>
                                        </span>
                                    </td>

                                    <!-- Call Date -->
                                    <td><?php echo htmlspecialchars($callingDate, ENT_QUOTES, 'UTF-8'); ?></td>

                                    <!-- Candidate Login -->
                                    <td><?php echo htmlspecialchars($loginText, ENT_QUOTES, 'UTF-8'); ?></td>

                                    <!-- PIF Submit -->
                                    <td><?php echo htmlspecialchars($pifText, ENT_QUOTES, 'UTF-8'); ?></td>

                                    <!-- Interview Date -->
                                    <td><?php echo htmlspecialchars($intDate, ENT_QUOTES, 'UTF-8'); ?></td>

                                    <!-- Action -->
                                    <td>
                                        <a href="<?php echo htmlspecialchars($formLink, ENT_QUOTES, 'UTF-8'); ?>"
                                           class="btn btn-icon btn-sm btn-primary-light btn-wave waves-effect waves-light"
                                           title="Open Application">
                                            <i class="ri-arrow-right-line"></i>
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <tr>
                                <td colspan="10" class="text-center text-muted">
                                    No calling form applications found for your Emp Code.
                                </td>
                            </tr>
                        <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <div class="card-footer border-top-0">
                <div class="d-flex align-items-center flex-wrap overflow-auto">
                    <?php
                        $showingFrom = ($totalRows === 0) ? 0 : ($offset + 1);
                        $showingTo   = min($offset + $perPage, $totalRows);
                        $prevDisabled = ($page <= 1) ? ' disabled' : '';
                        $nextDisabled = ($page >= $totalPages) ? ' disabled' : '';
                    ?>
                    <div class="mb-2 mb-sm-0">
                        Showing <b><?= h($showingFrom); ?></b> to <b><?= h($showingTo); ?></b> of <b><?= h($totalRows); ?></b> entries
                        <i class="bi bi-arrow-right ms-2 fw-medium"></i>
                    </div>
                    <div class="ms-auto">
                        <ul class="pagination mb-0 overflow-auto">
                            <li class="page-item<?= $prevDisabled; ?>">
                                <a class="page-link" href="<?= h(buildQuery(['page' => max(1, $page-1)])); ?>">Previous</a>
                            </li>
                            <?php
                            $window = 5;
                            $start = max(1, $page - (int)floor($window/2));
                            $end   = min($totalPages, $start + $window - 1);
                            $start = max(1, min($start, $end - $window + 1));
                            for ($p=$start; $p<=$end; $p++):
                                $active = ($p === $page) ? ' active' : '';
                            ?>
                                <li class="page-item<?= $active; ?>"><a class="page-link" href="<?= h(buildQuery(['page'=>$p])); ?>"><?= (int)$p; ?></a></li>
                            <?php endfor; ?>
                            <li class="page-item<?= $nextDisabled; ?>">
                                <a class="page-link" href="<?= h(buildQuery(['page' => min($totalPages, $page+1)])); ?>">Next</a>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<!--End::row-1 -->

<?php
$content = ob_get_clean();

/* ===================== INTERNAL SCRIPTS ($scripts) ===================== */
ob_start();
?>
<script src="<?php echo $baseUrl; ?>/assets/js/job-list.js"></script>

<!-- Simple client-side search -->
<script>
(function () {
    const searchInput = document.getElementById('callSearch');
    const table = document.getElementById('CallingTable');
    if (!searchInput || !table) return;

    searchInput.addEventListener('keyup', function () {
        const q = this.value.toLowerCase();
        const rows = table.querySelectorAll('tbody tr.joblist-list');
        rows.forEach(function (row) {
            const txt = row.textContent.toLowerCase();
            row.style.display = txt.indexOf(q) !== -1 ? '' : 'none';
        });
    });

    // "Select all" checkboxes (same as theme behaviour)
    const checkAll = document.querySelector('.check-all');
    if (checkAll) {
        checkAll.addEventListener('change', function () {
            const checked = this.checked;
            document.querySelectorAll('.joblist-checkbox .form-check-input')
                .forEach(function (cb) { cb.checked = checked; });
        });
    }
})();
</script>
<?php
$scripts = ob_get_clean();

/* ============================ RENDER LAYOUT ============================ */
include 'layouts/base.php';