<?php
/* =============================================================================
 * File   : signin.php
 * Goal   : Modern, accessible Sign In page (Core PHP + MySQLi, NO PDO)
 * Notes  : - Uses ../config.php for DB ($conn_sa or $conn)
 *          - Collation enforced: utf8mb4 / utf8mb4_general_ci
 *          - Renders via layouts/custom-base.php using $styles, $content, $scripts
 *          - Subtle glassmorphism card, gradient background, password toggle,
 *            caps-lock warning, submit spinner, responsive 2-panel layout.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ------------------------------ Config & DB ------------------------------ */
$cfgs = [
  __DIR__ . '/../config.php',
  __DIR__ . '/../../config.php',
  __DIR__ . '/config.php'
];
$cfgFound = false;
foreach ($cfgs as $c) {
  if (file_exists($c)) { require_once $c; $cfgFound = true; break; }
}
if (!$cfgFound) { http_response_code(500); die('config.php not found'); }

$DB = null;
if (isset($conn_sa) && $conn_sa instanceof mysqli) { $DB = $conn_sa; }
elseif (isset($conn) && $conn instanceof mysqli) { $DB = $conn; }
else { http_response_code(500); die('Database connection ($conn_sa/$conn) not found'); }

/* Collation / Charset */
mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* ------------------------------ Handle POST ----------------------------- */
$error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email    = isset($_POST['email']) ? filter_var(trim($_POST['email']), FILTER_VALIDATE_EMAIL) : false;
    $password = isset($_POST['password']) ? trim($_POST['password']) : '';

    if (!$email) {
        $error = 'Please enter a valid email.';
    } elseif ($password === '') {
        $error = 'Please enter your password.';
    } else {
        $sql = "SELECT id, email, password, role_id, first_name, last_name FROM users WHERE email = ?";
        if ($stmt = $DB->prepare($sql)) {
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $res = $stmt->get_result();
            if ($res && $res->num_rows === 1) {
                $user = $res->fetch_assoc();
                if (password_verify($password, $user['password'])) {
                    $_SESSION['user_id']    = $user['id'];
                    $_SESSION['user_email'] = $user['email'];
                    $_SESSION['role_id']    = $user['role_id'];
                    $_SESSION['username']   = trim(($user['first_name'] ?? '').' '.($user['last_name'] ?? ''));
                    session_write_close();
                    header("Location: onb-dashboard.php");
                    exit;
                } else {
                    $error = 'Incorrect password.';
                }
            } else {
                $error = 'Email not found.';
            }
            $stmt->close();
        } else {
            $error = 'Something went wrong. Please try again.';
        }
    }
}

/* Body class (consumed by layouts/custom-base.php if supported) */
$bodyClass = 'bg-auth';
?>

<?php ob_start(); ?>
<style>
    /* ---------------------- Page Background ---------------------- */
    .bg-auth {
        background: radial-gradient(1200px 600px at -10% -10%, rgba(99,102,241,.15), transparent 60%),
                    radial-gradient(900px 500px at 110% 10%, rgba(16,185,129,.18), transparent 60%),
                    linear-gradient(180deg, #f8fafc 0%, #ffffff 100%);
        min-height: 100vh;
    }

    /* ---------------------- Layout Wrapper ----------------------- */
    .auth-wrap {
        display: grid;
        grid-template-columns: 1fr;
        min-height: 100vh;
    }
    @media (min-width: 992px) {
        .auth-wrap { grid-template-columns: 1.1fr 1fr; }
    }

    /* ---------------------- Left Hero Pane ----------------------- */
    .auth-hero {
        display: none;
        padding: clamp(24px, 5vw, 48px);
        background:
          radial-gradient(800px 400px at 20% 0%, rgba(79,70,229,.15), transparent 60%),
          linear-gradient(135deg, #111827 0%, #1f2937 100%);
        color: #e5e7eb;
        align-items: center;
        justify-content: center;
    }
    .auth-hero-inner {
        max-width: 520px;
        margin: 0 auto;
    }
    .auth-badge {
        display: inline-flex;
        gap: .5rem;
        padding: .4rem .8rem;
        border: 1px solid rgba(255,255,255,.15);
        border-radius: 999px;
        font-size: .8rem;
        align-items: center;
        backdrop-filter: blur(6px);
        background: rgba(255,255,255,.06);
    }
    .auth-hero h1 {
        font-weight: 700;
        letter-spacing: -.02em;
        margin: 18px 0 8px;
        color: #fff;
    }
    .auth-hero p {
        color: #d1d5db;
        margin-bottom: 0;
    }
    @media (min-width: 992px) {
        .auth-hero { display: flex; }
    }

    /* ---------------------- Right Form Pane ---------------------- */
    .auth-pane {
        display: flex;
        align-items: center;
        justify-content: center;
        padding: clamp(24px, 6vw, 56px);
    }

    /* ----------------------- Glass Card -------------------------- */
    .login-card {
        width: 100%;
        max-width: 440px;
        position: relative;
        border: 1px solid rgba(0,0,0,.06);
        background: rgba(255,255,255,.72);
        backdrop-filter: blur(10px);
        -webkit-backdrop-filter: blur(10px);
        border-radius: 16px;
        box-shadow: 0 10px 30px rgba(2,6,23,.08);
        padding: clamp(20px, 4vw, 32px);
    }

    /* ----------------------- Branding ---------------------------- */
    .brand {
        display: flex;
        align-items: center;
        gap: 10px;
        margin-bottom: 18px;
    }
    .brand .logo {
        width: 44px; height: 44px;
        border-radius: 12px;
        background: linear-gradient(135deg, #4f46e5, #06b6d4);
        box-shadow: 0 8px 20px rgba(79,70,229,.35);
    }
    .brand span {
        font-weight: 700;
        letter-spacing: -.02em;
        font-size: 1.15rem;
    }

    /* ----------------------- Form Bits --------------------------- */
    .form-label { font-weight: 600; color: #111827; }
    .form-control {
        border-radius: 12px;
        border-color: #e5e7eb;
        padding: .7rem .9rem;
    }
    .form-control:focus {
        border-color: #4f46e5;
        box-shadow: 0 0 0 .2rem rgba(79,70,229,.15);
    }
    .input-icon {
        position: absolute;
        right: .75rem;
        top: 50%;
        transform: translateY(-50%);
        cursor: pointer;
        color: #6b7280;
        user-select: none;
    }
    .caps-hint {
        display: none;
        font-size: .8rem;
        color: #f59e0b;
        margin-top: .5rem;
    }
    .alert { border-radius: 12px; }

    .btn-primary {
        border-radius: 12px;
        font-weight: 600;
        padding: .75rem 1rem;
    }
    .btn-primary .spinner-border {
        width: 1rem; height: 1rem; border-width: .15rem;
    }

    /* ----------------------- Helpers ----------------------------- */
    .muted { color: #6b7280; }
    .divider {
        height: 1px;
        background: linear-gradient(90deg, transparent, #e5e7eb, transparent);
        margin: 18px 0;
    }
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
<div class="auth-wrap">
    <!-- Left: Brand / Copy -->
    <aside class="auth-hero">
        <div class="auth-hero-inner">
            <div class="auth-badge">
                <span>🔐</span><strong>Secure Access</strong><span>•</span><span>Admin Suite</span>
            </div>
            <h1>Welcome back to your dashboard</h1>
            <p>Sign in to manage enquiries, properties, assets, and more — all in one place.</p>
        </div>
    </aside>

    <!-- Right: Sign In -->
    <main class="auth-pane">
        <div class="login-card">
            <div class="brand">
                <div class="logo" aria-hidden="true"></div>
                <span>Sign in</span>
            </div>
            <p class="muted mb-3">Use your work email and password to continue.</p>

            <?php if (!empty($error)): ?>
                <div class="alert alert-danger" role="alert"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>

            <form method="POST" action="" id="signinForm" novalidate>
                <!-- Email -->
                <div class="mb-3">
                    <label for="signin-email" class="form-label">Email address</label>
                    <input
                        type="email"
                        class="form-control"
                        id="signin-email"
                        name="email"
                        placeholder="you@company.com"
                        required
                        value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>"
                        autocomplete="username"
                        autofocus
                    >
                </div>

                <!-- Password -->
                <div class="mb-1 position-relative">
                    <label for="signin-password" class="form-label">Password</label>
                    <input
                        type="password"
                        class="form-control"
                        id="signin-password"
                        name="password"
                        placeholder="Enter your password"
                        required
                        autocomplete="current-password"
                        aria-describedby="caps-hint"
                    >
                    <button type="button" class="btn btn-link p-0 input-icon" id="togglePwd" aria-label="Show password">
                        <i class="ri-eye-off-line" id="pwdIcon"></i>
                    </button>
                </div>
                <div id="caps-hint" class="caps-hint">Caps Lock is ON</div>

                <!-- Remember -->
                <div class="d-flex align-items-center justify-content-between mt-3 mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="remember-me" name="remember" checked>
                        <label class="form-check-label" for="remember-me">Remember me</label>
                    </div>
                    <!-- <a href="reset-password-basic.php" class="link-danger fw-semibold small">Forgot password?</a> -->
                </div>

                <!-- Submit -->
                <div class="d-grid gap-2">
                    <button type="submit" class="btn btn-primary" id="submitBtn">
                        <span class="submit-text">Sign In</span>
                        <span class="submit-wip d-none">
                            <span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
                            Signing in...
                        </span>
                    </button>
                </div>

                <div class="divider"></div>
                <div class="small muted">
                    By continuing, you agree to our Terms & Privacy Policy.
                </div>
            </form>
        </div>
    </main>
</div>
<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
/* ---------------- Password Toggle (no external dependency) ---------------- */
(function(){
    const pwd = document.getElementById('signin-password');
    const btn = document.getElementById('togglePwd');
    const icon = document.getElementById('pwdIcon');

    if (btn && pwd && icon) {
        btn.addEventListener('click', function(){
            const isPwd = pwd.getAttribute('type') === 'password';
            pwd.setAttribute('type', isPwd ? 'text' : 'password');
            // Toggle RemixIcon classes if available
            if (icon.classList) {
                icon.classList.toggle('ri-eye-off-line', !isPwd);
                icon.classList.toggle('ri-eye-line', isPwd);
            } else {
                btn.setAttribute('aria-label', isPwd ? 'Hide password' : 'Show password');
            }
            pwd.focus();
        });
    }

    // Caps Lock detection
    const hint = document.getElementById('caps-hint');
    if (pwd && hint) {
        pwd.addEventListener('keyup', function(e){
            if (e.getModifierState && e.getModifierState('CapsLock')) {
                hint.style.display = 'block';
            } else {
                hint.style.display = 'none';
            }
        });
        pwd.addEventListener('blur', function(){ hint.style.display = 'none'; });
    }

    // Submit spinner + prevention of double submit
    const form = document.getElementById('signinForm');
    const submitBtn = document.getElementById('submitBtn');
    if (form && submitBtn) {
        form.addEventListener('submit', function(){
            const text = submitBtn.querySelector('.submit-text');
            const wip  = submitBtn.querySelector('.submit-wip');
            if (text && wip) {
                text.classList.add('d-none');
                wip.classList.remove('d-none');
            }
            submitBtn.setAttribute('disabled', 'disabled');
        });
    }
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include __DIR__ . '/layouts/custom-base.php'; ?>
