<?php
/* =============================================================================
 * File   : broker-list.php
 * Purpose: Admin view for broker applications (filters, search, approve/reject
 *          with ZeptoMail email notifications + bulk CSV export)
 * Stack  : Core PHP + MySQLi (NO PDO). DB via ../config.php
 * DB     : utf8mb4 / COLLATE=utf8mb4_general_ci
 * Notes  : - Email & Phone are NOT shown in the table (only used internally)
 *          - Uses SENDGRID_* or ZEPTO_* constants (ZeptoMail key) from config.php
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* ------------------- Central config + permissions -------------------- */
require_once __DIR__ . '/../config.php';
if (file_exists(__DIR__ . '/includes/check-permission.php')) {
    require_once __DIR__ . '/includes/check-permission.php';
}

/* ZeptoMail template helper (portal) */
$zeptoHelperPath = __DIR__ . '/zeptomail_helper.php';
if (is_file($zeptoHelperPath)) {
    require_once $zeptoHelperPath;
}

if (!isset($conn) || !($conn instanceof mysqli)) {
    http_response_code(500);
    exit('Database connection ($conn) not found. Please ensure ../config.php defines $conn (mysqli).');
}

/* Permission guard (if available) */
$currentPage = basename(__FILE__);
if (function_exists('checkPageAccessOrRedirect')) {
    $permConn = isset($conn_sa) && $conn_sa instanceof mysqli ? $conn_sa : $conn;
    checkPageAccessOrRedirect($currentPage, $permConn);
}

/* ----------------- Charset & collation enforcement ------------------- */
mysqli_set_charset($conn, 'utf8mb4');
mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* --------------------------- Helpers --------------------------------- */
if (!function_exists('esc')) {
    function esc($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}
if (!function_exists('h')) {
    function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

function badge_class_from_status($s) {
    $s = strtolower(trim((string)$s));
    if ($s === 'approved' || $s === 'active')       return 'bg-primary-transparent text-primary';
    if ($s === 'rejected' || $s === 'declined' ||
        $s === 'blocked')                           return 'bg-danger-transparent text-danger';
    if ($s === 'pending' || $s === '' || $s === null) return 'bg-warning-transparent text-warning';
    return 'bg-secondary-transparent text-secondary';
}

/* ------------- Email sending is now handled via zepto_send_template()
 * ------------- in the Approve/Reject block using the template portal. */

/* --------------------------- Messages & input ------------------------- */
$msg    = '';
$errMsg = '';
$action = isset($_GET['action']) ? $_GET['action'] : '';
$id     = isset($_GET['id']) ? (int)$_GET['id'] : 0;

/* ----------------------- Bulk export (selected) ----------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_action']) && $_POST['bulk_action'] === 'export') {
    $ids = isset($_POST['ids']) && is_array($_POST['ids']) ? array_filter(array_map('intval', $_POST['ids'])) : [];
    if (!empty($ids)) {
        $idList = implode(',', $ids);
        $sqlExp = "SELECT id, broker_code, first_name, last_name, email, phone, firm_name, empid, status, created_at
                   FROM brokers
                   WHERE id IN ({$idList})
                   ORDER BY created_at DESC";
        $resExp = mysqli_query($conn, $sqlExp);

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=brokers_export_' . date('Ymd_His') . '.csv');

        $out = fopen('php://output', 'w');
        // CSV for admin; contains email/phone (UI does NOT show them)
        fputcsv($out, ['ID', 'Broker Code', 'First Name', 'Last Name', 'Email', 'Phone', 'Firm', 'EmpID', 'Status', 'Created At']);

        if ($resExp) {
            while ($row = mysqli_fetch_assoc($resExp)) {
                fputcsv($out, [
                    $row['id'],
                    $row['broker_code'],
                    $row['first_name'],
                    $row['last_name'],
                    $row['email'],
                    $row['phone'],
                    $row['firm_name'],
                    $row['empid'],
                    $row['status'],
                    $row['created_at']
                ]);
            }
        }
        fclose($out);
        exit;
    } else {
        $errMsg = 'No brokers selected for export.';
    }
}

/* -------------------------- Approve / Reject -------------------------- */
if (($action === 'approve' || $action === 'reject') && $id > 0) {
    $newStatus = ($action === 'approve') ? 'Approved' : 'Rejected';

    // Fetch broker entry
    $broRow = null;
    $resB = mysqli_query($conn, "SELECT broker_code, first_name, last_name, email FROM brokers WHERE id = " . (int)$id . " LIMIT 1");
    if ($resB) {
        $broRow = mysqli_fetch_assoc($resB);
    }

    // Update brokers.status
    $stmt = @mysqli_prepare($conn, "UPDATE brokers SET status = ? WHERE id = ?");
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, 'si', $newStatus, $id);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    } else {
        $escStatus = mysqli_real_escape_string($conn, $newStatus);
        mysqli_query($conn, "UPDATE brokers SET status = '{$escStatus}' WHERE id = " . (int)$id);
    }

    // If Approved → activate corresponding user (username = broker_code)
    if ($action === 'approve' && $broRow && !empty($broRow['broker_code'])) {
        $uname = mysqli_real_escape_string($conn, $broRow['broker_code']);
        $stmt2 = @mysqli_prepare($conn, "UPDATE users SET status = 'Active' WHERE username = ?");
        if ($stmt2) {
            mysqli_stmt_bind_param($stmt2, 's', $uname);
            mysqli_stmt_execute($stmt2);
            mysqli_stmt_close($stmt2);
        } else {
            mysqli_query($conn, "UPDATE users SET status = 'Active' WHERE username = '{$uname}'");
        }
    }

    // Send ZeptoMail email via template portal
    $mailErr = '';
    if ($broRow && !empty($broRow['email'])) {
        $fullName    = trim(($broRow['first_name'] ?? '') . ' ' . ($broRow['last_name'] ?? ''));
        $displayName = $fullName !== '' ? $fullName : ($broRow['broker_code'] ?? 'Broker');

        // Choose template based on action
        // Make sure these keys exist in email_templates portal:
        //   broker_approved
        //   broker_rejected
        $templateKey = ($action === 'approve') ? 'broker_approved' : 'broker_rejected';

        if (function_exists('zepto_send_template')) {
            $placeholders = array(
                '{{NAME}}'        => $displayName,
                '{{STATUS}}'      => $newStatus,
                '{{BROKER_CODE}}' => (string)($broRow['broker_code'] ?? ''),
            );

            $mailResult = zepto_send_template(
                $broRow['email'],
                $displayName,
                $templateKey,
                $placeholders
            );

            if (empty($mailResult['success'])) {
                $mailErr = $mailResult['error'] ?? 'Unknown error from ZeptoMail';
            }
        } else {
            $mailErr = 'Email helper zepto_send_template() not found.';
        }
    }

    $msgText = "Status updated to {$newStatus}.";
    if ($mailErr !== '') {
        $msgText .= ' Email error: ' . $mailErr;
    }

    header('Location: broker-list.php?msg=' . urlencode($msgText));
    exit;
}

/* ------------------ Set / Save credentials for broker ----------------- */
if ($action === 'setcred' && $id > 0 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = isset($_POST['username']) ? trim($_POST['username']) : '';
    $password = isset($_POST['password']) ? trim($_POST['password']) : '';

    if ($username === '' || $password === '') {
        $errMsg = 'Username and password are required.';
    } else {
        $hash = password_hash($password, PASSWORD_BCRYPT);
        $escUsername = mysqli_real_escape_string($conn, $username);

        $uRes = mysqli_query($conn, "SELECT id FROM users WHERE username = '{$escUsername}' LIMIT 1");
        $uRow = $uRes ? mysqli_fetch_assoc($uRes) : null;

        if ($uRow) {
            // Update existing user
            $stmt = @mysqli_prepare($conn, "UPDATE users SET password_hash = ?, role = 'Broker' WHERE id = ?");
            if ($stmt) {
                mysqli_stmt_bind_param($stmt, 'si', $hash, $uRow['id']);
                if (!mysqli_stmt_execute($stmt)) {
                    $errMsg = 'Failed to update credentials.';
                } else {
                    $msg = 'Credentials updated for existing user.';
                }
                mysqli_stmt_close($stmt);
            } else {
                $escHash = mysqli_real_escape_string($conn, $hash);
                if (!mysqli_query($conn, "UPDATE users SET password_hash = '{$escHash}', role = 'Broker' WHERE id = " . (int)$uRow['id'])) {
                    $errMsg = 'Failed to update credentials (fallback).';
                } else {
                    $msg = 'Credentials updated for existing user (fallback).';
                }
            }
        } else {
            // Create new user
            $role = 'Broker';
            $stmt = @mysqli_prepare($conn, "INSERT INTO users (username, password_hash, role) VALUES (?, ?, ?)");
            if ($stmt) {
                mysqli_stmt_bind_param($stmt, 'sss', $username, $hash, $role);
                if (!mysqli_stmt_execute($stmt)) {
                    $errMsg = 'Failed to save credentials.';
                } else {
                    $msg = 'Credentials saved.';
                }
                mysqli_stmt_close($stmt);
            } else {
                $escHash = mysqli_real_escape_string($conn, $hash);
                $escRole = mysqli_real_escape_string($conn, $role);
                if (!mysqli_query($conn, "INSERT INTO users (username, password_hash, role) VALUES ('{$escUsername}', '{$escHash}', '{$escRole}')")) {
                    $errMsg = 'Failed to save credentials (fallback).';
                } else {
                    $msg = 'Credentials saved (fallback).';
                }
            }
        }
    }
}

/* --------------- Which row is in "Set Credential" mode ---------------- */
$editCredId = ($action === 'cred' && $id > 0) ? (int)$id : 0;

/* ---------------------------- Filters --------------------------------- */
$filterSearch = isset($_GET['q'])      ? trim($_GET['q'])      : '';
$filterStatus = isset($_GET['status']) ? trim($_GET['status']) : '';
$filterFrom   = isset($_GET['from'])   ? trim($_GET['from'])   : '';
$filterTo     = isset($_GET['to'])     ? trim($_GET['to'])     : '';
$filterSort   = isset($_GET['sort'])   ? trim($_GET['sort'])   : 'latest';

$where = [];

// Search by code / name / firm
if ($filterSearch !== '') {
    $s = mysqli_real_escape_string($conn, $filterSearch);
    $where[] = "(broker_code LIKE '%{$s}%' OR first_name LIKE '%{$s}%' OR last_name LIKE '%{$s}%' OR firm_name LIKE '%{$s}%')";
}

// Status filter
if ($filterStatus !== '') {
    $st = mysqli_real_escape_string($conn, $filterStatus);
    $where[] = "status = '{$st}'";
}

// Date range on created_at
if ($filterFrom !== '') {
    $dFrom = mysqli_real_escape_string($conn, $filterFrom);
    $where[] = "DATE(created_at) >= '{$dFrom}'";
}
if ($filterTo !== '') {
    $dTo = mysqli_real_escape_string($conn, $filterTo);
    $where[] = "DATE(created_at) <= '{$dTo}'";
}

$sql = "SELECT id, broker_code, first_name, last_name, firm_name, empid, status, created_at
        FROM brokers";

if (!empty($where)) {
    $sql .= " WHERE " . implode(' AND ', $where);
}

// Sort
switch ($filterSort) {
    case 'oldest':
        $sql .= " ORDER BY created_at ASC";
        break;
    case 'name_az':
        $sql .= " ORDER BY first_name ASC, last_name ASC";
        break;
    case 'status':
        $sql .= " ORDER BY status ASC, created_at DESC";
        break;
    case 'latest':
    default:
        $sql .= " ORDER BY created_at DESC";
        break;
}

$brokers = [];
$rs = mysqli_query($conn, $sql);
if ($rs) {
    while ($row = mysqli_fetch_assoc($rs)) {
        $brokers[] = $row;
    }
}

/* -------------- Base URL fallback (for assets/links) ------------------ */
if (!isset($baseUrl)) {
    $scheme  = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https://' : 'http://';
    $host    = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $scriptDir = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/');
    $baseUrl = $scheme . $host . $scriptDir;
}

/* ------------------------------- Styles -------------------------------- */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo esc($baseUrl); ?>/assets/libs/sweetalert2/sweetalert2.min.css">
<style>
.page-title{font-weight:600;font-size:1.125rem;}
.card.custom-card{border-radius:.6rem;box-shadow:0 .25rem .5rem rgba(18,38,63,.04);}
.table thead th{font-weight:600;font-size:.95rem;white-space:nowrap;}
.table tbody td{font-size:.9rem;}
.badge{padding:.45em .65em;border-radius:.5rem;font-size:.8rem;}
.bg-primary-transparent{background:rgba(23,83,252,.08);color:#1753fc;}
.bg-warning-transparent{background:rgba(251,188,11,.08);color:#fbcc0b;}
.bg-danger-transparent{background:rgba(238,51,94,.08);color:#ee335e;}
.bg-secondary-transparent{background:rgba(99,102,241,.06);color:#6366f1;}
.action-btns .btn{margin-left:.2rem;margin-right:.2rem;margin-bottom:4px;}
.filters-row .form-label{font-size:.75rem;text-transform:uppercase;color:#6c757d;letter-spacing:.04em;}
.filters-row .form-control,.filters-row .form-select{font-size:.85rem;}
.bulk-actions .btn{font-size:.8rem;}
@media (max-width: 575px){
  .action-btns .btn{display:block;width:100%;}
}
</style>
<?php
$styles = ob_get_clean();

/* ------------------------------ Content -------------------------------- */
ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
    <h1 class="page-title mb-0">Brokers</h1>
    <a href="broker-onboarding.php" class="btn btn-primary">
      <i class="fa fa-plus me-1"></i> New Application
    </a>
  </div>
</div>

<?php if (!empty($_GET['msg'])): ?>
  <div class="alert alert-success py-2 mb-2"><?php echo h($_GET['msg']); ?></div>
<?php endif; ?>
<?php if (!empty($msg)): ?>
  <div class="alert alert-success py-2 mb-2"><?php echo h($msg); ?></div>
<?php endif; ?>
<?php if (!empty($errMsg)): ?>
  <div class="alert alert-danger py-2 mb-2"><?php echo h($errMsg); ?></div>
<?php endif; ?>

<div class="card custom-card mb-3">
  <div class="card-body">
    <div class="d-flex align-items-center justify-content-between mb-3">
      <h4 class="card-title mb-0">
        <i class="fas fa-users me-2"></i> All Applications
      </h4>
      <small class="text-muted">Total: <?php echo count($brokers); ?></small>
    </div>

    <!-- Filters -->
    <form method="get" class="row g-2 mb-3 filters-row">
      <div class="col-md-3 col-sm-6">
        <label class="form-label">Search</label>
        <input type="text" name="q" value="<?php echo h($filterSearch); ?>" class="form-control" placeholder="Code, name, firm">
      </div>
      <div class="col-md-2 col-sm-6">
        <label class="form-label">Status</label>
        <select name="status" class="form-select">
          <option value="">All</option>
          <option value="Pending"  <?php echo $filterStatus === 'Pending'  ? 'selected' : ''; ?>>Pending</option>
          <option value="Approved" <?php echo $filterStatus === 'Approved' ? 'selected' : ''; ?>>Approved</option>
          <option value="Rejected" <?php echo $filterStatus === 'Rejected' ? 'selected' : ''; ?>>Rejected</option>
        </select>
      </div>
      <div class="col-md-2 col-sm-6">
        <label class="form-label">From</label>
        <input type="date" name="from" value="<?php echo h($filterFrom); ?>" class="form-control">
      </div>
      <div class="col-md-2 col-sm-6">
        <label class="form-label">To</label>
        <input type="date" name="to" value="<?php echo h($filterTo); ?>" class="form-control">
      </div>
      <div class="col-md-2 col-sm-6">
        <label class="form-label">Sort By</label>
        <select name="sort" class="form-select">
          <option value="latest"  <?php echo $filterSort === 'latest'  ? 'selected' : ''; ?>>Newest first</option>
          <option value="oldest"  <?php echo $filterSort === 'oldest'  ? 'selected' : ''; ?>>Oldest first</option>
          <option value="name_az" <?php echo $filterSort === 'name_az' ? 'selected' : ''; ?>>Name A–Z</option>
          <option value="status"  <?php echo $filterSort === 'status'  ? 'selected' : ''; ?>>Status</option>
        </select>
      </div>
      <div class="col-md-1 col-sm-6 d-flex align-items-end">
        <button type="submit" class="btn btn-outline-primary w-100">Apply</button>
      </div>
    </form>

    <!-- Bulk actions -->
    <div class="d-flex justify-content-between align-items-center mb-2 bulk-actions">
      <div class="small text-muted">
        Tick rows to export selected brokers. (Email & Phone are hidden on this screen.)
      </div>
      <div>
        <button type="button" class="btn btn-sm btn-outline-secondary me-1" id="btnSelectAll">Select All</button>
        <button type="button" class="btn btn-sm btn-outline-secondary me-1" id="btnClearAll">Clear</button>
        <button type="button" class="btn btn-sm btn-success" id="btnExportSelected">
          <i class="fa fa-file-excel me-1"></i> Export Selected (CSV)
        </button>
      </div>
    </div>

    <form method="post" id="bulkForm">
      <input type="hidden" name="bulk_action" id="bulk_action" value="">
      <div class="table-responsive">
        <table class="table table-hover align-middle">
          <thead>
            <tr>
              <th style="width:40px;"><input type="checkbox" id="checkAll"></th>
              <th style="width:60px;">#</th>
              <th>Code</th>
              <th>Name</th>
              <th>Firm</th>
              <th>EmpID</th>
              <th>Created</th>
              <th>Status</th>
              <th class="text-end" style="width:340px;">Actions</th>
            </tr>
          </thead>
          <tbody>
          <?php if (empty($brokers)): ?>
            <tr><td colspan="9" class="text-muted">No applications found.</td></tr>
          <?php else: foreach ($brokers as $r): ?>
            <tr>
              <td>
                <input type="checkbox" class="row-check" name="ids[]" value="<?php echo (int)$r['id']; ?>">
              </td>
              <td><?php echo (int)$r['id']; ?></td>
              <td><?php echo h($r['broker_code']); ?></td>
              <td><?php echo h(trim(($r['first_name'] ?? '') . ' ' . ($r['last_name'] ?? ''))); ?></td>
              <td><?php echo h($r['firm_name']); ?></td>
              <td><?php echo h($r['empid']); ?></td>
              <td>
                <?php
                if (!empty($r['created_at'])) {
                    echo h(date('d-M-Y H:i', strtotime($r['created_at'])));
                } else {
                    echo '-';
                }
                ?>
              </td>
              <td>
                <?php $cls = badge_class_from_status($r['status']); ?>
                <span class="badge <?php echo h($cls); ?>">
                  <?php echo h($r['status'] !== '' ? $r['status'] : 'Pending'); ?>
                </span>
              </td>
              <td class="text-end action-btns">
                <a class="btn btn-sm btn-outline-info" href="broker-view.php?id=<?php echo (int)$r['id']; ?>">
                  <i class="fas fa-eye me-1"></i> View
                </a>
                <a class="btn btn-sm btn-outline-secondary" href="broker-list.php?action=cred&id=<?php echo (int)$r['id']; ?>">
                  <i class="fas fa-key me-1"></i> Set Credentials
                </a>
                <a class="btn btn-sm btn-outline-success"
                   href="broker-list.php?action=approve&id=<?php echo (int)$r['id']; ?>"
                   onclick="return confirm('Approve this broker?');">
                  <i class="fas fa-check me-1"></i> Approve
                </a>
                <a class="btn btn-sm btn-outline-danger"
                   href="broker-list.php?action=reject&id=<?php echo (int)$r['id']; ?>"
                   onclick="return confirm('Reject this broker?');">
                  <i class="fas fa-times me-1"></i> Reject
                </a>
              </td>
            </tr>

            <?php if ($editCredId === (int)$r['id']): ?>
            <tr>
              <td colspan="9">
                <form method="post" action="broker-list.php?action=setcred&id=<?php echo (int)$r['id']; ?>" class="d-flex flex-wrap gap-2 align-items-center">
                  <div class="input-group" style="min-width:230px;">
                    <span class="input-group-text">Username</span>
                    <input type="text" name="username" class="form-control" value="<?php echo h($r['broker_code']); ?>" required>
                  </div>
                  <div class="input-group" style="min-width:230px;">
                    <span class="input-group-text">Password</span>
                    <input type="text" name="password" class="form-control" required>
                  </div>
                  <div>
                    <button type="submit" class="btn btn-primary">
                      <i class="fa fa-save me-1"></i> Save
                    </button>
                    <a href="broker-list.php" class="btn btn-light">Cancel</a>
                  </div>
                </form>
              </td>
            </tr>
            <?php endif; ?>

          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </form>
  </div>
</div>
<?php
$content = ob_get_clean();

/* ------------------------------- Scripts ------------------------------- */
ob_start(); ?>
<script src="<?php echo esc($baseUrl); ?>/assets/libs/sweetalert2/sweetalert2.min.js"></script>
<script>
// Success toast from ?msg=
(function(){
  const urlParams = new URLSearchParams(window.location.search);
  if (urlParams.has('msg')) {
    const m = urlParams.get('msg');
    setTimeout(function(){
      Swal.fire({
        icon: 'success',
        title: 'Success',
        text: m,
        timer: 3200,
        showConfirmButton: false
      });
    }, 120);
  }
})();

// Bulk select & export
(function(){
  const checkAll   = document.getElementById('checkAll');
  const rowChecks  = document.querySelectorAll('.row-check');
  const btnSelectAll = document.getElementById('btnSelectAll');
  const btnClearAll  = document.getElementById('btnClearAll');
  const btnExport    = document.getElementById('btnExportSelected');
  const bulkForm     = document.getElementById('bulkForm');
  const bulkAction   = document.getElementById('bulk_action');

  if (checkAll) {
    checkAll.addEventListener('change', function(){
      rowChecks.forEach(function(cb){ cb.checked = checkAll.checked; });
    });
  }

  if (btnSelectAll) {
    btnSelectAll.addEventListener('click', function(){
      rowChecks.forEach(function(cb){ cb.checked = true; });
      if (checkAll) checkAll.checked = true;
    });
  }

  if (btnClearAll) {
    btnClearAll.addEventListener('click', function(){
      rowChecks.forEach(function(cb){ cb.checked = false; });
      if (checkAll) checkAll.checked = false;
    });
  }

  if (btnExport && bulkForm && bulkAction) {
    btnExport.addEventListener('click', function(){
      const anyChecked = Array.prototype.some.call(rowChecks, function(cb){ return cb.checked; });
      if (!anyChecked) {
        Swal.fire({
          icon: 'warning',
          title: 'No selection',
          text: 'Please select at least one broker to export.',
          timer: 2200,
          showConfirmButton: false
        });
        return;
      }
      bulkAction.value = 'export';
      bulkForm.submit();
    });
  }
})();
</script>
<?php
$scripts = ob_get_clean();

/* --------------------------- Render layout ---------------------------- */
/* layouts/base.php must echo $styles, $content, $scripts in proper slots */
include __DIR__ . '/layouts/base.php';
