<?php
/* =============================================================================
 * File: units.php
 * Purpose: Broker-visible Units listing with Melody theme blocks + Grid.js table
 * Roles  : Admin (Add/Edit/Block/Delete + Hold/Unhold/Book), Broker (View + Hold/Unhold/Book)
 * Stack  : Core PHP + MySQLi (NO PDO). DB in /broker-config.php. Collation: utf8mb4_general_ci
 * Notes  : - Uses layouts/base.php renderer with $styles, $content, $scripts slots
 *          - Lightweight JSON API in this file when ?api=grid
 *          - Action links: unit_view.php, unit_form.php, unit_hold_book.php, unit_status.php, unit_delete.php
 * ============================================================================= */

date_default_timezone_set('Asia/Kolkata');
@session_start();

/* ---- Load DB config (ABSOLUTE from docroot) ---- */
require_once 'broker-config.php';
if (!isset($conn) || !($conn instanceof mysqli)) { die('DB connection missing.'); }
include('../config.php');
require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);
/* ---- Charset & collation ---- */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection='utf8mb4_general_ci'");

/* ---- Helpers ---- */
if (!function_exists('h')) { function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); } }
function getv($k,$d=''){ return isset($_GET[$k]) ? trim($_GET[$k]) : $d; }

/* INR formatter (Indian grouping) */
if (!function_exists('format_inr')) {
  function format_inr($num) {
    $num = (float)$num; $sign = $num < 0 ? '-' : ''; $num = abs($num);
    $parts = explode('.', number_format($num, 2, '.', '')); $int = $parts[0]; $dec = $parts[1];
    if (strlen($int) <= 3) return $sign . $int . '.' . $dec;
    $last3 = substr($int, -3); $rest  = substr($int, 0, -3);
    $rest  = preg_replace('/\B(?=(\d{2})+(?!\d))/', ',', $rest);
    return $sign . $rest . ',' . $last3 . '.' . $dec;
  }
}
function inr_sym($amount){ return '₹ ' . format_inr($amount); }

/* ---- Role & permissions ---- */
$role    = isset($_SESSION['role']) ? (string)$_SESSION['role'] : '';
$isAdmin = (strcasecmp($role,'Admin') === 0);
$canAdd=$isAdmin; $canEdit=$isAdmin; $canBlock=$isAdmin; $canDelete=$isAdmin;
$canUnhold=true; $canHold=true; $canBook=true;

/* ---- Filters (HTML + JSON API) ---- */
$q   = getv('q'); $pid = getv('project_id'); $st  = getv('status');
$where = "WHERE 1=1";
if ($q !== '')  { $q2  = mysqli_real_escape_string($conn, $q);  $where .= " AND (u.unit_no LIKE '%$q2%' OR u.unit_type LIKE '%$q2%')"; }
if ($pid !== ''){ $where .= " AND u.project_id=".(int)$pid; }
if ($st !== '') { $st2 = mysqli_real_escape_string($conn, $st); $where .= " AND u.status='$st2'"; }

/* ---- Header counters (unfiltered) ---- */
$counts=['total'=>0,'available'=>0,'hold'=>0,'booked'=>0,'blocked'=>0];
$crs = mysqli_query($conn, "SELECT 
  COUNT(*) total,
  SUM(CASE WHEN status='Available' THEN 1 ELSE 0 END) available,
  SUM(CASE WHEN status='Hold' THEN 1 ELSE 0 END) hold,
  SUM(CASE WHEN status='Booked' THEN 1 ELSE 0 END) booked,
  SUM(CASE WHEN status='Blocked' THEN 1 ELSE 0 END) blocked
FROM units");
if ($crs && $row=mysqli_fetch_assoc($crs)) { $counts = array_map('intval',$row); }

/* ---- Projects for filter ---- */
$projects=[]; $prs=mysqli_query($conn,"SELECT id,name FROM projects ORDER BY name");
if($prs){ while($p=mysqli_fetch_assoc($prs)){ $projects[]=$p; } }

/* ---- JSON API for Grid.js ---- */
if (isset($_GET['api']) && $_GET['api']==='grid') {
  $rows=[]; $sql="SELECT u.*,p.name project_name FROM units u JOIN projects p ON p.id=u.project_id $where ORDER BY u.updated_at DESC";
  $rs=mysqli_query($conn,$sql);
  while($r=mysqli_fetch_assoc($rs)){
    $area=(float)$r['saleable_area']; $rate=(float)$r['base_rate']; $plc=(float)$r['plc_amount']; $other=(float)$r['other_charges']; $gstRate=(float)$r['gst_rate'];
    $baseTotal = ($rate*$area)+$plc+$other; $gstAmt=($baseTotal*$gstRate)/100.0;
    $badge='secondary'; if($r['status']==='Available')$badge='success'; elseif($r['status']==='Hold')$badge='warning'; elseif($r['status']==='Booked')$badge='danger'; elseif($r['status']==='Blocked')$badge='dark';
    $status_html = '<span class="badge bg-'.$badge.'">'.h($r['status']).'</span>';
    $acts=[];
    $acts[]='<a class="btn btn-sm btn-outline-secondary me-2" href="unit_view.php?id='.(int)$r['id'].'"><i class="fa fa-eye me-1"></i>View</a>';
    if($canEdit)  $acts[]='<a class="btn btn-sm btn-outline-primary me-2" href="unit_form.php?id='.(int)$r['id'].'"><i class="fa fa-edit me-1"></i>Edit</a>';
    if($canHold)  $acts[]='<a class="btn btn-sm btn-outline-warning me-2" href="unit_hold_book.php?id='.(int)$r['id'].'&action=hold"><i class="fa fa-hand-paper-o me-1"></i>Hold</a>';
    if($canUnhold)$acts[]='<a class="btn btn-sm btn-outline-success me-2" href="unit_status.php?id='.(int)$r['id'].'&to=Available" onclick="return confirm(\'Mark Available (Unhold)?\')"><i class="fa fa-undo me-1"></i>Unhold</a>';
    if($canBook)  $acts[]='<a class="btn btn-sm btn-outline-info me-2" href="unit_hold_book.php?id='.(int)$r['id'].'&action=book"><i class="fa fa-file-signature me-1"></i>Book</a>';
    if($canBlock) $acts[]='<a class="btn btn-sm btn-outline-dark me-2" href="unit_status.php?id='.(int)$r['id'].'&to=Blocked" onclick="return confirm(\'Block unit?\')"><i class="fa fa-ban me-1"></i>Block</a>';
    if($canDelete)$acts[]='<a class="btn btn-sm btn-outline-danger" href="unit_delete.php?id='.(int)$r['id'].'" onclick="return confirm(\'Delete unit?\')"><i class="fa fa-trash me-1"></i>Delete</a>';
    $rows[]=[
      'project'=>$r['project_name'],'unit_no'=>$r['unit_no'],'unit_type'=>$r['unit_type'],'facing'=>$r['facing'],
      'area'=>number_format($area,2).' sqft','rate'=>inr_sym($rate),'total'=>inr_sym((float)$r['total_price']),
      'gst_rate'=>number_format($gstRate,2).'%', 'gst_amt'=>inr_sym($gstAmt),'status'=>$status_html,'actions'=>implode('',$acts)
    ];
  }
  header('Content-Type: application/json; charset=utf-8'); echo json_encode(['rows'=>$rows], JSON_UNESCAPED_UNICODE); exit;
}

/* ---- Base URL for assets ---- */
// $baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS']==='on' ? 'https://' : 'http://').$_SERVER['HTTP_HOST'].str_replace('/pages','',dirname($_SERVER['SCRIPT_NAME']));

/* ---- Flash messages ---- */
$flash_error = !empty($_SESSION['flash_error']) ? (string)$_SESSION['flash_error'] : '';
$flash_success = !empty($_SESSION['flash_success']) ? (string)$_SESSION['flash_success'] : '';
unset($_SESSION['flash_error'], $_SESSION['flash_success']);

/* ================================ $styles ================================ */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/gridjs/theme/mermaid.min.css">
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.css">
<style>.grid-card-table .gridjs-container{border:0!important}.gridjs-th,.gridjs-td{white-space:nowrap}.dashboard-main-card .avatar svg{width:22px;height:22px}</style>
<?php $styles = ob_get_clean();

/* =============================== $content =============================== */
ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Units</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">Real Estate</a></li>
      <li class="breadcrumb-item active" aria-current="page">Units</li>
    </ol>
  </div>
</div>

<?php if ($flash_error): ?><div class="alert alert-danger"><?php echo h($flash_error); ?></div><?php endif; ?>
<?php if ($flash_success): ?><div class="alert alert-success"><?php echo h($flash_success); ?></div><?php endif; ?>

<div class="row row-cols-xxl-5 row-cols-md-3 row-cols-1">
  <div class="col"><div class="card custom-card dashboard-main-card primary"><div class="card-body"><div class="d-flex align-items-start gap-3 flex-wrap"><div class="flex-fill"><span class="fs-13 fw-medium">Total Units</span><h4 class="fw-semibold my-2 lh-1"><?php echo (int)$counts['total']; ?></h4><span class="d-block text-muted">All projects</span></div><div><span class="avatar avatar-md bg-primary-transparent svg-primary"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 256"><path d="M32 56a16 16 0 0 1 16-16h160a16 16 0 0 1 16 16v136a8 8 0 0 1-8 8H40a8 8 0 0 1-8-8Z"/></svg></span></div></div></div></div></div>
  <div class="col"><div class="card custom-card dashboard-main-card success"><div class="card-body"><div class="d-flex align-items-start gap-3 flex-wrap"><div class="flex-fill"><span class="fs-13 fw-medium">Available</span><h4 class="fw-semibold my-2 lh-1"><?php echo (int)$counts['available']; ?></h4><span class="d-block text-muted">Ready to book</span></div><div><span class="avatar avatar-md bg-success-transparent svg-success"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 256"><path d="M224 56v144a8 8 0 0 1-8 8H40a8 8 0 0 1-8-8V56Z"/></svg></span></div></div></div></div></div>
  <div class="col"><div class="card custom-card dashboard-main-card warning"><div class="card-body"><div class="d-flex align-items-start gap-3 flex-wrap"><div class="flex-fill"><span class="fs-13 fw-medium">On Hold</span><h4 class="fw-semibold my-2 lh-1"><?php echo (int)$counts['hold']; ?></h4><span class="d-block text-muted">Temporarily reserved</span></div><div><span class="avatar avatar-md bg-warning-transparent svg-warning"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 256"><path d="M32 88h192v112a8 8 0 0 1-8 8H40a8 8 0 0 1-8-8Z"/></svg></span></div></div></div></div></div>
  <div class="col"><div class="card custom-card dashboard-main-card secondary"><div class="card-body"><div class="d-flex align-items-start gap-3 flex-wrap"><div class="flex-fill"><span class="fs-13 fw-medium">Booked</span><h4 class="fw-semibold my-2 lh-1"><?php echo (int)$counts['booked']; ?></h4><span class="d-block text-muted">Under agreement</span></div><div><span class="avatar avatar-md bg-secondary-transparent svg-secondary"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 256"><path d="M48 40h160v80H48Z"/></svg></span></div></div></div></div></div>
  <div class="col"><div class="card custom-card dashboard-main-card info"><div class="card-body"><div class="d-flex align-items-start gap-3 flex-wrap"><div class="flex-fill"><span class="fs-13 fw-medium">Blocked</span><h4 class="fw-semibold my-2 lh-1"><?php echo (int)$counts['blocked']; ?></h4><span class="d-block text-muted">Not for sale</span></div><div><span class="avatar avatar-md bg-info-transparent svg-info"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 256 256"><path d="M40 80h176v96H40Z"/></svg></span></div></div></div></div></div>
</div>

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card products-filter">
      <div class="card-header justify-content-between border-bottom-0">
        <form class="row gy-2 w-100 align-items-end" method="get" action="">
          <div class="col-12 col-sm-4">
            <label class="form-label mb-1 text-muted small">Search (Unit No / Type)</label>
            <input class="form-control" type="search" name="q" value="<?php echo h($q); ?>" placeholder="e.g. A-1204 or 2BHK">
          </div>
          <div class="col-12 col-sm-4">
            <label class="form-label mb-1 text-muted small">Project</label>
            <select class="form-control" name="project_id">
              <option value="">All Projects</option>
              <?php foreach($projects as $p): ?>
                <option value="<?php echo (int)$p['id']; ?>" <?php echo ($pid!=='' && (int)$pid===(int)$p['id'])?'selected':''; ?>>
                  <?php echo h($p['name']); ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-12 col-sm-3">
            <label class="form-label mb-1 text-muted small">Status</label>
            <select class="form-control" name="status">
              <option value="">Any Status</option>
              <?php foreach(['Available','Hold','Booked','Blocked','Unavailable'] as $opt): ?>
                <option value="<?php echo h($opt); ?>" <?php echo ($st===$opt)?'selected':''; ?>><?php echo h($opt); ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-12 col-sm-1 d-grid">
            <button class="btn btn-primary"><i class="fa fa-filter me-1"></i>Go</button>
          </div>
        </form>
      </div>

      <div class="card-body">
        <div class="d-flex justify-content-between mb-2">
          <h5 class="mb-0">Unit Inventory</h5>
          <?php if ($canAdd): ?>
            <a class="btn btn-sm btn-primary" href="unit_form.php"><i class="fa fa-plus me-2"></i>Add Unit</a>
          <?php endif; ?>
        </div>
        <div id="units-table" class="grid-card-table"></div>
      </div>
    </div>
  </div>
</div>
<?php $content = ob_get_clean();

/* =============================== $scripts =============================== */
ob_start(); ?>
<script src="<?php echo $baseUrl; ?>/assets/libs/gridjs/gridjs.umd.js"></script>
<script src="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.js"></script>
<script>
(function(){
  const qs = new URLSearchParams(window.location.search);
  const apiUrl = window.location.pathname + '?api=grid&' + qs.toString();
  const grid = new gridjs.Grid({
    columns: [
      { id:'project', name:'Project' },
      { id:'unit_no', name:'Unit-No' },
      { id:'unit_type', name:'Type' },
      { id:'facing', name:'Facing' },
      { id:'area', name:'Area' },
      { id:'rate', name:'Rate' },
      { id:'total', name:'Total' },
      { id:'gst_rate', name:'GST %' },
      { id:'gst_amt', name:'GST Amt' },
      { id:'status', name:'Status', formatter:(c)=>gridjs.html(c) },
      { id:'actions', name:'Actions', formatter:(c)=>gridjs.html(c) }
    ],
    pagination:{ limit:10 }, sort:true, search:false, className:{ table:'table' },
    server:{ url: apiUrl, then: d => d.rows }
  }).render(document.getElementById('units-table'));

  <?php if ($flash_success): ?>Swal.fire({icon:'success',title:'Done',text:<?php echo json_encode($flash_success); ?>,timer:2000,showConfirmButton:false});<?php endif; ?>
  <?php if ($flash_error): ?>Swal.fire({icon:'error',title:'Error',text:<?php echo json_encode($flash_error); ?>});<?php endif; ?>
})();
</script>
<?php $scripts = ob_get_clean();

/* ------------------------------ Render base ------------------------------ */
include 'layouts/base.php';
