<?php
/* =============================================================================
 * File   : broker_edit.php
 * Purpose: View / Edit page to see & update a saved Broker Onboarding record
 * Stack  : Core PHP + MySQLi (NO PDO). DB via broker-config.php.
 * DB     : utf8mb4 / COLLATE=utf8mb4_general_ci
 * Notes  : - Fetch by ?id=123 OR ?code=BRK-YYYYMMDD-XXXX
 *          - Default = READ-ONLY; Edit button toggles edit mode.
 *          - On Save, updates brokers table via prepared statement.
 *          - Reuses Melody layout via layouts/base.php
 * ============================================================================= */

@session_start();
require_once __DIR__ . '/../config.php';
// require_once 'includes/check-permission.php';
// $currentPage = basename(__FILE__);
// checkPageAccessOrRedirect($currentPage, $conn_sa);

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ---- Required includes ---- */
require_once 'broker-config.php';
require_once 'functions.php';

/* ---- Ensure DB ready & charset/collation ---- */
if (!isset($conn) || !($conn instanceof mysqli)) {
    die('DB connection missing.');
}
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection='utf8mb4_general_ci'");

/* ---- Helpers ---- */
if (!function_exists('esc')) {
    function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

/* ---- Status badge helper ---- */
function status_badge_class($status){
    $status = strtolower((string)$status);
    if ($status === 'approved') return 'badge bg-success';
    if ($status === 'rejected') return 'badge bg-danger';
    return 'badge bg-warning text-dark'; // Pending/default
}

/* ---- Document link helper ---- */
function doc_link($baseUrl, $path){
    if (!$path) {
        return '<span class="text-muted">Not uploaded</span>';
    }
    $url = rtrim($baseUrl, '/') . '/' . ltrim($path, '/');
    return '<a href="'.esc($url).'" target="_blank" class="btn btn-sm btn-outline-primary">View</a>';
}

/* ---- Init vars ---- */
$error_msg   = '';
$success_msg = '';
$broker      = null;

/* ---- Resolve id/code from GET ---- */
$id   = isset($_GET['id'])   ? (int)$_GET['id'] : 0;
$code = isset($_GET['code']) ? trim((string)$_GET['code']) : '';

/* ========================== Handle Update (POST) ========================== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $posted_id = isset($_POST['broker_id']) ? (int)$_POST['broker_id'] : 0;

    if ($posted_id <= 0) {
        $error_msg = 'Invalid broker reference for update.';
    } else {
        // For subsequent SELECT, use the posted id
        $id   = $posted_id;
        $code = '';

        // Collect editable fields from POST
        $first_name          = trim($_POST['first_name'] ?? '');
        $last_name           = trim($_POST['last_name'] ?? '');
        $email               = trim($_POST['email'] ?? '');
        $phone               = trim($_POST['phone'] ?? '');
        $alt_phone           = trim($_POST['alt_phone'] ?? '');
        $whatsapp            = trim($_POST['whatsapp'] ?? '');
        $dob                 = trim($_POST['dob'] ?? '');
        $broker_code         = trim($_POST['broker_code'] ?? '');
        $firm_name           = trim($_POST['firm_name'] ?? '');
        $business_type       = trim($_POST['business_type'] ?? '');
        $rera_id             = trim($_POST['rera_id'] ?? '');
        $license_valid_till  = trim($_POST['license_valid_till'] ?? '');
        $gstin               = trim($_POST['gstin'] ?? '');
        $pan                 = trim($_POST['pan'] ?? '');
        $addr_line           = trim($_POST['addr_line'] ?? '');
        $city                = trim($_POST['city'] ?? '');
        $state               = trim($_POST['state'] ?? '');
        $pincode             = trim($_POST['pincode'] ?? '');
        $bank_account_name   = trim($_POST['bank_account_name'] ?? '');
        $bank_account_number = trim($_POST['bank_account_number'] ?? '');
        $bank_ifsc           = trim($_POST['bank_ifsc'] ?? '');
        $bank_name           = trim($_POST['bank_name'] ?? '');
        $bank_branch         = trim($_POST['bank_branch'] ?? '');
        $referral_source     = trim($_POST['referral_source'] ?? '');
        $notes               = trim($_POST['notes'] ?? '');
        $agreement_accept    = isset($_POST['agreement_accept']) ? 1 : 0;
        $status              = trim($_POST['status'] ?? '');
        if ($status === '') { $status = 'Pending'; }

        // Document paths (optional text edit)
        $doc_rera_cert        = trim($_POST['doc_rera_cert'] ?? '');
        $doc_pan_card         = trim($_POST['doc_pan_card'] ?? '');
        $doc_gst_cert         = trim($_POST['doc_gst_cert'] ?? '');
        $doc_cancelled_cheque = trim($_POST['doc_cancelled_cheque'] ?? '');
        $doc_address_proof    = trim($_POST['doc_address_proof'] ?? '');

        // Allow non-mandatory DATE fields to be NULL when left blank
        if ($dob === '') {
            $dob = null;                 // brokers.dob is DATE NULL
        }
        if ($license_valid_till === '') {
            $license_valid_till = null;  // brokers.license_valid_till is DATE NULL
        }

        $sqlUpdate = "UPDATE `brokers` SET
            `first_name` = ?,
            `last_name` = ?,
            `email` = ?,
            `phone` = ?,
            `alt_phone` = ?,
            `whatsapp` = ?,
            `dob` = ?,
            `broker_code` = ?,
            `firm_name` = ?,
            `business_type` = ?,
            `rera_id` = ?,
            `license_valid_till` = ?,
            `gstin` = ?,
            `pan` = ?,
            `addr_line` = ?,
            `city` = ?,
            `state` = ?,
            `pincode` = ?,
            `bank_account_name` = ?,
            `bank_account_number` = ?,
            `bank_ifsc` = ?,
            `bank_name` = ?,
            `bank_branch` = ?,
            `referral_source` = ?,
            `notes` = ?,
            `agreement_accept` = ?,
            `status` = ?,
            `doc_rera_cert` = ?,
            `doc_pan_card` = ?,
            `doc_gst_cert` = ?,
            `doc_cancelled_cheque` = ?,
            `doc_address_proof` = ?,
            `updated_at` = NOW()
        WHERE `id` = ? LIMIT 1";

        $stmtUpdate = mysqli_prepare($conn, $sqlUpdate);
        if ($stmtUpdate) {
            // 33 params total: 31 strings + 2 ints (agreement_accept, id)
            // 1-25  : s, 26 : i, 27-32 : s, 33 : i
            $types = str_repeat('s', 25) . 'i' . str_repeat('s', 6) . 'i';

            mysqli_stmt_bind_param(
                $stmtUpdate,
                $types,
                $first_name,
                $last_name,
                $email,
                $phone,
                $alt_phone,
                $whatsapp,
                $dob,
                $broker_code,
                $firm_name,
                $business_type,
                $rera_id,
                $license_valid_till,
                $gstin,
                $pan,
                $addr_line,
                $city,
                $state,
                $pincode,
                $bank_account_name,
                $bank_account_number,
                $bank_ifsc,
                $bank_name,
                $bank_branch,
                $referral_source,
                $notes,
                $agreement_accept,
                $status,
                $doc_rera_cert,
                $doc_pan_card,
                $doc_gst_cert,
                $doc_cancelled_cheque,
                $doc_address_proof,
                $posted_id
            );

            if (mysqli_stmt_execute($stmtUpdate)) {
                $success_msg = 'Broker details updated successfully.';
            } else {
                $error_msg = 'Failed to update broker record: ' . mysqli_stmt_error($stmtUpdate);
            }
            mysqli_stmt_close($stmtUpdate);
        } else {
            $error_msg = 'Failed to prepare update query: ' . mysqli_error($conn);
        }
    }
}

/* ============================ Load record (SELECT) ============================ */
if ($id <= 0 && $code === '') {
    if ($error_msg === '') {
        $error_msg = 'No broker selected. Please pass ?id=123 or ?code=BRK-... in URL.';
    }
} else {
    if ($id > 0) {
        $sql  = "SELECT * FROM brokers WHERE id = ? LIMIT 1";
        $stmt = mysqli_prepare($conn, $sql);
        if ($stmt) {
            mysqli_stmt_bind_param($stmt, 'i', $id);
        }
    } else {
        $sql  = "SELECT * FROM brokers WHERE broker_code = ? LIMIT 1";
        $stmt = mysqli_prepare($conn, $sql);
        if ($stmt) {
            mysqli_stmt_bind_param($stmt, 's', $code);
        }
    }

    if (!$stmt) {
        if ($error_msg === '') {
            $error_msg = 'Failed to prepare select query: ' . mysqli_error($conn);
        }
    } else {
        if (!mysqli_stmt_execute($stmt)) {
            if ($error_msg === '') {
                $error_msg = 'Failed to fetch broker record: ' . mysqli_stmt_error($stmt);
            }
        } else {
            $res = mysqli_stmt_get_result($stmt);
            if ($res && mysqli_num_rows($res) > 0) {
                $broker = mysqli_fetch_assoc($res);
            } else {
                if ($error_msg === '') {
                    $error_msg = 'Broker record not found.';
                }
            }
            if ($res) { mysqli_free_result($res); }
        }
        mysqli_stmt_close($stmt);
    }
}

/* ================================ $styles ================================ */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.css">
<style>
  .page-hint-badge{ font-size:.8rem; }
  .form-label { font-weight: 500; }
  .readonly-input[disabled],
  .readonly-input[readonly],
  .readonly-input:disabled{
    background-color:#f8fafc;
    color:#0f172a;
    opacity:1;
  }
</style>
<?php $styles = ob_get_clean();

/* =============================== $content =============================== */
ob_start(); ?>
<!-- Page Header (Melody) -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <div>
      <h1 class="page-title fw-medium fs-18 mb-1">
        Broker Details
        <?php if($broker): ?>
          <small class="text-muted fs-12 ms-2">(<?= esc($broker['broker_code']); ?>)</small>
        <?php endif; ?>
      </h1>
      <p class="text-muted mb-0 fs-12">View / Edit snapshot of an onboarded broker.</p>
    </div>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">Partners</a></li>
      <li class="breadcrumb-item active" aria-current="page">Broker Details</li>
    </ol>
  </div>
</div>

<?php if($error_msg): ?>
  <div class="alert alert-danger d-flex justify-content-between align-items-center">
    <span><?php echo esc($error_msg); ?></span>
    <button type="button" class="btn btn-sm btn-secondary" onclick="history.back();">
      <i class="fa fa-arrow-left me-1"></i> Back
    </button>
  </div>
<?php endif; ?>

<?php if($success_msg): ?>
  <div class="alert alert-success d-flex justify-content-between align-items-center">
    <span><?php echo esc($success_msg); ?></span>
    <button type="button" class="btn btn-sm btn-outline-success" onclick="window.location.reload();">
      <i class="fa fa-sync me-1"></i> Reload
    </button>
  </div>
<?php endif; ?>

<?php if($broker): ?>
  <div class="mb-3 d-flex justify-content-between align-items-center flex-wrap gap-2">
    <div>
      <span class="<?= status_badge_class($broker['status']); ?>">
        Status: <?= esc($broker['status']); ?>
      </span>
      <?php if(!empty($broker['agreement_accept'])): ?>
        <span class="badge bg-primary ms-2">Agreement Accepted</span>
      <?php else: ?>
        <span class="badge bg-secondary ms-2">Agreement Pending</span>
      <?php endif; ?>
    </div>
    <div class="text-end">
      <div class="small text-muted">
        Created: <?= $broker['created_at'] ? esc(date('d M Y, h:i A', strtotime($broker['created_at']))) : '-'; ?>
      </div>
      <div class="mt-1">
        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="history.back();">
          <i class="fa fa-arrow-left me-1"></i> Back
        </button>
        <button type="button" id="edit-toggle-btn" class="btn btn-sm btn-primary ms-2" onclick="toggleEditMode();">
          <i class="fa fa-edit me-1"></i> Edit
        </button>
      </div>
    </div>
  </div>

  <!-- Form wraps all cards: default READ-ONLY, Edit button toggles enable -->
  <form id="broker-view-form" method="post" action="">
    <!-- Personal -->
    <div class="card custom-card mb-3">
      <div class="card-header justify-content-between">
        <div class="card-title"><i class="fa fa-id-card me-2"></i> Personal Information</div>
        <span id="mode-badge" class="badge bg-light text-dark page-hint-badge">
          View only • Editing is disabled
        </span>
      </div>
      <div class="card-body">
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">First Name</label>
            <input type="text" name="first_name" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['first_name']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Last Name</label>
            <input type="text" name="last_name" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['last_name']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Email</label>
            <input type="text" name="email" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['email']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Phone</label>
            <input type="text" name="phone" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['phone']); ?>" disabled>
            <small class="text-muted">Primary contact number</small>
          </div>
          <div class="col-md-4">
            <label class="form-label">Alternate Phone</label>
            <input type="text" name="alt_phone" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['alt_phone']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">WhatsApp</label>
            <input type="text" name="whatsapp" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['whatsapp']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Date of Birth</label>
            <input type="text" name="dob" class="form-control readonly-input js-editable"
                   value="<?= $broker['dob'] ? esc($broker['dob']) : ''; ?>" disabled>
          </div>
        </div>
      </div>
    </div>

    <!-- Firm & Compliance -->
    <div class="card custom-card mb-3">
      <div class="card-header">
        <div class="card-title"><i class="fa fa-building me-2"></i> Firm & Compliance</div>
      </div>
      <div class="card-body">
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">Broker Code</label>
            <input type="text" name="broker_code" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['broker_code']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Firm/Company Name</label>
            <input type="text" name="firm_name" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['firm_name']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Business Type</label>
            <input type="text" name="business_type" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['business_type']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">RERA ID</label>
            <input type="text" name="rera_id" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['rera_id']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">License Valid Till</label>
            <input type="text" name="license_valid_till" class="form-control readonly-input js-editable"
                   value="<?= $broker['license_valid_till'] ? esc($broker['license_valid_till']) : ''; ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">GSTIN</label>
            <input type="text" name="gstin" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['gstin']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">PAN</label>
            <input type="text" name="pan" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['pan']); ?>" disabled>
          </div>
        </div>
      </div>
    </div>

    <!-- Address -->
    <div class="card custom-card mb-3">
      <div class="card-header">
        <div class="card-title"><i class="fa fa-map-marker me-2"></i> Address</div>
      </div>
      <div class="card-body">
        <div class="row g-3">
          <div class="col-md-6">
            <label class="form-label">Address Line</label>
            <input type="text" name="addr_line" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['addr_line']); ?>" disabled>
          </div>
          <div class="col-md-3">
            <label class="form-label">City</label>
            <input type="text" name="city" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['city']); ?>" disabled>
          </div>
          <div class="col-md-3">
            <label class="form-label">State</label>
            <input type="text" name="state" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['state']); ?>" disabled>
          </div>
          <div class="col-md-3">
            <label class="form-label">Pincode</label>
            <input type="text" name="pincode" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['pincode']); ?>" disabled>
          </div>
        </div>
      </div>
    </div>

    <!-- Bank -->
    <div class="card custom-card mb-3">
      <div class="card-header">
        <div class="card-title"><i class="fa fa-university me-2"></i> Bank Details</div>
      </div>
      <div class="card-body">
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">Account Holder Name</label>
            <input type="text" name="bank_account_name" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['bank_account_name']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Account Number</label>
            <input type="text" name="bank_account_number" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['bank_account_number']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">IFSC</label>
            <input type="text" name="bank_ifsc" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['bank_ifsc']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Bank Name</label>
            <input type="text" name="bank_name" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['bank_name']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Branch</label>
            <input type="text" name="bank_branch" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['bank_branch']); ?>" disabled>
          </div>
        </div>
      </div>
    </div>

    <!-- Documents -->
    <div class="card custom-card mb-3">
      <div class="card-header">
        <div class="card-title"><i class="fa fa-file-upload me-2"></i> Documents</div>
      </div>
      <div class="card-body">
        <p class="text-muted small mb-3">
          Click "View" to open document in a new tab. Paths below are editable as text in Edit mode.
        </p>
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">RERA Certificate</label><br>
            <?= doc_link($baseUrl, $broker['doc_rera_cert']); ?>
            <input type="text" name="doc_rera_cert"
                   class="form-control readonly-input js-editable mt-1"
                   value="<?= esc($broker['doc_rera_cert']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">PAN Card</label><br>
            <?= doc_link($baseUrl, $broker['doc_pan_card']); ?>
            <input type="text" name="doc_pan_card"
                   class="form-control readonly-input js-editable mt-1"
                   value="<?= esc($broker['doc_pan_card']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">GST Certificate</label><br>
            <?= doc_link($baseUrl, $broker['doc_gst_cert']); ?>
            <input type="text" name="doc_gst_cert"
                   class="form-control readonly-input js-editable mt-1"
                   value="<?= esc($broker['doc_gst_cert']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Cancelled Cheque</label><br>
            <?= doc_link($baseUrl, $broker['doc_cancelled_cheque']); ?>
            <input type="text" name="doc_cancelled_cheque"
                   class="form-control readonly-input js-editable mt-1"
                   value="<?= esc($broker['doc_cancelled_cheque']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Address Proof</label><br>
            <?= doc_link($baseUrl, $broker['doc_address_proof']); ?>
            <input type="text" name="doc_address_proof"
                   class="form-control readonly-input js-editable mt-1"
                   value="<?= esc($broker['doc_address_proof']); ?>" disabled>
          </div>
        </div>
      </div>
    </div>

    <!-- Additional -->
    <div class="card custom-card mb-3">
      <div class="card-header">
        <div class="card-title"><i class="fa fa-clipboard-list me-2"></i> Additional</div>
      </div>
      <div class="card-body">
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">Referral Source</label>
            <input type="text" name="referral_source" class="form-control readonly-input js-editable"
                   value="<?= esc($broker['referral_source']); ?>" disabled>
          </div>
          <div class="col-md-4">
            <label class="form-label">Status</label>
            <select name="status" class="form-select readonly-input js-editable" disabled>
              <?php
                $statusVal = trim((string)$broker['status']);
                $statuses = ['Pending', 'Approved', 'Rejected'];
                foreach ($statuses as $s) {
                    $sel = (strcasecmp($statusVal, $s) === 0) ? 'selected' : '';
                    echo '<option value="'.esc($s).'" '.$sel.'>'.esc($s).'</option>';
                }
              ?>
            </select>
          </div>
          <div class="col-md-12">
            <label class="form-label">Notes</label>
            <textarea name="notes" class="form-control readonly-input js-editable" rows="3" disabled><?= esc($broker['notes']); ?></textarea>
          </div>
          <div class="col-md-12">
            <label class="form-label">Agreement</label>
            <div class="form-check mt-1">
              <input type="checkbox" name="agreement_accept"
                     class="form-check-input js-editable" id="agree_view"
                     <?= !empty($broker['agreement_accept']) ? 'checked' : ''; ?> disabled>
              <label class="form-check-label" for="agree_view">
                I confirm the information is true and I accept the onboarding terms.
              </label>
            </div>
          </div>
        </div>

        <!-- Hidden primary key for POST -->
        <input type="hidden" name="broker_id" value="<?= (int)$broker['id']; ?>">

        <div class="mt-3 d-flex justify-content-between align-items-center">
          <button type="button" class="btn btn-outline-secondary" onclick="history.back();">
            <i class="fa fa-arrow-left me-1"></i> Back
          </button>
          <button type="submit" id="save-btn" class="btn btn-primary d-none">
            <i class="fa fa-save me-1"></i> Save Changes
          </button>
        </div>
      </div>
    </div>
  </form>
<?php endif; ?>
<?php $content = ob_get_clean();

/* =============================== $scripts =============================== */
ob_start(); ?>
<script src="<?php echo $baseUrl; ?>/assets/libs/sweetalert2/sweetalert2.min.js"></script>
<script>
(function(){
  let isEditing = false;

  window.toggleEditMode = function(){
    const form    = document.getElementById('broker-view-form');
    const badge   = document.getElementById('mode-badge');
    const editBtn = document.getElementById('edit-toggle-btn');
    const saveBtn = document.getElementById('save-btn');

    if (!form) return;

    const editables = form.querySelectorAll('.js-editable');

    isEditing = !isEditing;

    editables.forEach(function(el){
      if (isEditing) {
        el.removeAttribute('disabled');
      } else {
        el.setAttribute('disabled', 'disabled');
      }
    });

    if (badge) {
      if (isEditing) {
        badge.textContent = 'Edit mode • You can modify and save changes';
        badge.classList.remove('bg-light');
        badge.classList.add('bg-warning', 'text-dark');
      } else {
        badge.textContent = 'View only • Editing is disabled';
        badge.classList.remove('bg-warning');
        badge.classList.add('bg-light', 'text-dark');
      }
    }

    if (editBtn) {
      editBtn.innerHTML = isEditing
        ? '<i class="fa fa-times me-1"></i> Cancel'
        : '<i class="fa fa-edit me-1"></i> Edit';
    }

    if (saveBtn) {
      if (isEditing) {
        saveBtn.classList.remove('d-none');
      } else {
        saveBtn.classList.add('d-none');
      }
    }
  };
})();
</script>
<?php $scripts = ob_get_clean();

/* ------------------------------ Render base ------------------------------ */
include 'layouts/base.php';
