<?php
/* ============================================================================
 * File: activities-calls.php
 * Purpose: Dashboard listing all CALL activities from `crm_activities`
 * Stack : Core PHP + MySQLi (NO PDO). DB connection via ../config.php
 * Rules : Collation utf8mb4_general_ci
 * Layout: Includes ../layouts/base.php (RBAC/bootstrap/UI gating)
 * Notes : - Filters: date_from/date_to (date_start), q (title/desc/lead name/company)
 *         - Pagination: 10/25/50/500
 *         - Source Of is derived from saved_from: contains 'opportunit' => Opportunity else Lead
 *         - Sorting: click any header to sort ASC/DESC
 *         - Extra: Employee filter (assigned_user), checkboxes + export (selected/all)
 * ========================================================================== */

@session_start();
require_once __DIR__ . '/../config.php'; // MUST hold $conn (mysqli)
require_once 'includes/check-permission.php';

$role_id    = $_SESSION['role_id']   ?? 0;
$user_email = $_SESSION['user_email']?? '';
$isAdmin    = in_array((int)$role_id, [1, 2], true);
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

/* ---- Charset & collation enforcement ---- */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* ---------- Helper(s) ---------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function src_of($savedFrom){
    return (stripos((string)$savedFrom, 'opportunit') !== false) ? 'Opportunity' : 'Lead';
}

function status_badge($status){
    $status = trim((string)$status);
    $cls = 'badge-status-other';
    $lower = strtolower($status);

    if ($lower === 'planned' || strpos($lower, 'plan') !== false || strpos($lower, 'sched') !== false) {
        $cls = 'badge-status-planned';
    } elseif (
        in_array($lower, ['completed','held','done','closed','success'], true) ||
        strpos($lower, 'complete') !== false
    ) {
        $cls = 'badge-status-done';
    } elseif (strpos($lower, 'cancel') !== false || strpos($lower, 'not held') !== false || strpos($lower, 'failed') !== false) {
        $cls = 'badge-status-cancel';
    }

    return '<span class="badge badge-status '.$cls.'">'.h($status).'</span>';
}

function sort_link($label, $key, $curSort, $curDir, $baseParams){
    $params = $baseParams;
    $params['sort'] = $key;
    $params['dir']  = ($curSort === $key && $curDir === 'asc') ? 'desc' : 'asc';
    $params['page'] = 1;

    $href = htmlspecialchars($_SERVER['PHP_SELF'].'?'.http_build_query($params), ENT_QUOTES, 'UTF-8');
    $labelEsc = htmlspecialchars($label, ENT_QUOTES, 'UTF-8');

    $arrow = '';
    if ($curSort === $key) {
        $arrow = ($curDir === 'asc') ? ' ▲' : ' ▼';
    }

    return '<a href="'.$href.'" class="text-decoration-none text-body">'.$labelEsc.$arrow.'</a>';
}

date_default_timezone_set('Asia/Kolkata');

/* ---------- Request source (GET for list, POST for export) ---------- */
$request = ($_SERVER['REQUEST_METHOD'] === 'POST') ? $_POST : $_GET;

/* ---------- Filters ---------- */
$perPageOptions = [10, 25, 50, 500];
$perPage = (int)($request['per_page'] ?? 25);
if (!in_array($perPage, $perPageOptions, true)) {
    $perPage = 25;
}

$page = max(1, (int)($request['page'] ?? 1));
$offset = ($page - 1) * $perPage;

$q        = trim($request['q'] ?? '');
$df       = trim($request['date_from'] ?? ''); // YYYY-MM-DD
$dt       = trim($request['date_to'] ?? '');   // YYYY-MM-DD
$assigned = trim($request['assigned'] ?? '');  // filter for assigned_user (EMPID/email)

/* ---------- Sorting ---------- */
$allowedSorts = [
    'when'       => 'a.`date_start`',
    'contact'    => 'l.`contact_name`',
    'title'      => 'a.`title`',
    'status'     => 'a.`status`',
    'reminder'   => 'a.`reminder_at`',
    'saved_from' => 'a.`saved_from`',
    'source_of'  => 'a.`saved_from`', // same base as saved_from
    'assigned'   => 'a.`assigned_user`'
];

$sort = $request['sort'] ?? 'when';
if (!isset($allowedSorts[$sort])) {
    $sort = 'when';
}
$dir = strtolower($request['dir'] ?? 'desc');
$dirSql = ($dir === 'asc') ? 'ASC' : 'DESC';

/* ---------- WHERE conditions ---------- */
$where = ["a.`activity_type`='call'"];

if ($q !== '') {
    $qEsc = mysqli_real_escape_string($conn, $q);
    $where[] = "(a.`title` LIKE '%{$qEsc}%' OR a.`description` LIKE '%{$qEsc}%' OR l.`contact_name` LIKE '%{$qEsc}%' OR l.`company` LIKE '%{$qEsc}%')";
}

if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $df)) {
    $dfEsc = mysqli_real_escape_string($conn, $df);
    $where[] = "a.`date_start` >= '{$dfEsc} 00:00:00'";
}
if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dt)) {
    $dtEsc = mysqli_real_escape_string($conn, $dt);
    $where[] = "a.`date_start` <= '{$dtEsc} 23:59:59'";
}

if ($assigned !== '') {
    $assignedEsc = mysqli_real_escape_string($conn, $assigned);
    $where[] = "a.`assigned_user` = '{$assignedEsc}'";
}

if (!$isAdmin && $user_email !== '') {
    $emailEsc = mysqli_real_escape_string($conn, $user_email);
    // extra safety: non-admin always restricted to own records
    // $where[] = "a.`assigned_user` = '{$emailEsc}'";
}

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

/* ---------- Employee list for filter (assigned_user) ---------- */
$assignedList = [];
$empSql = "SELECT DISTINCT a.`assigned_user`
           FROM `crm_activities` a
           WHERE a.`activity_type`='call' AND a.`assigned_user` IS NOT NULL AND a.`assigned_user` <> ''
           ORDER BY a.`assigned_user`";
$empRes = mysqli_query($conn, $empSql);
if ($empRes) {
    while ($er = mysqli_fetch_assoc($empRes)) {
        $assignedList[] = $er['assigned_user'];
    }
}

if (!$isAdmin && $user_email !== '') {
    // For non-admins, keep dropdown scoped to their own email only
    $assignedList = [$user_email];
}

/* ---------- KPIs (Today / This Week / Overdue Reminders) ---------- */
$today = date('Y-m-d');
$weekStart = date('Y-m-d', strtotime('monday this week'));
$now = date('Y-m-d H:i:s');
$kpiToday = $kpiWeek = $kpiOverdue = 0;
$kpiWhere = ["`activity_type`='call'"];
if (!$isAdmin && $user_email !== '') {
    $emailEsc = mysqli_real_escape_string($conn, $user_email);
    $kpiWhere[] = "`assigned_user` = '{$emailEsc}'";
}
$kpiSqlWhere = 'WHERE ' . implode(' AND ', $kpiWhere);
$r = mysqli_query($conn, "SELECT 
  SUM(CASE WHEN DATE(`date_start`)='{$today}' THEN 1 ELSE 0 END) AS k1,
  SUM(CASE WHEN DATE(`date_start`) BETWEEN '{$weekStart}' AND '{$today}' THEN 1 ELSE 0 END) AS k2,
  SUM(CASE WHEN `remind`=1 AND `reminder_at` IS NOT NULL AND `reminder_at` < '{$now}' THEN 1 ELSE 0 END) AS k3
  FROM `crm_activities` {$kpiSqlWhere}");
if($r){
    $row = mysqli_fetch_assoc($r);
    $kpiToday   = (int)($row['k1'] ?? 0);
    $kpiWeek    = (int)($row['k2'] ?? 0);
    $kpiOverdue = (int)($row['k3'] ?? 0);
}

/* ---------- Export (selected / all with same filters) ---------- */
$isExport = ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export_action']));
if ($isExport) {
    $action = $_POST['export_action'];

    // If "selected", collect IDs; if none selected, fall back to "all"
    $ids = [];
    if ($action === 'selected' && !empty($_POST['selected_ids']) && is_array($_POST['selected_ids'])) {
        foreach ($_POST['selected_ids'] as $id) {
            $id = (int)$id;
            if ($id > 0) {
                $ids[] = $id;
            }
        }
    }
    if ($action === 'selected' && !$ids) {
        $action = 'all';
    }

    $exportWhereSql = $whereSql;
    if ($action === 'selected' && $ids) {
        $idList = implode(',', $ids);
        if ($exportWhereSql) {
            $exportWhereSql .= " AND a.`id` IN ({$idList})";
        } else {
            $exportWhereSql = "WHERE a.`id` IN ({$idList})";
        }
    }

    $exportSql = "SELECT 
            a.`id`,a.`lead_id`,a.`status`,a.`direction`,a.`title`,a.`description`,
            a.`date_start`,a.`date_end`,a.`duration_minutes`,a.`assigned_user`,
            a.`created_by`,a.`remind`,a.`reminder_at`,a.`saved_from`,
            l.`contact_name`,l.`company`,l.`phone`,l.`email`
        FROM `crm_activities` a
        JOIN `leads` l ON l.`id`=a.`lead_id`
        {$exportWhereSql}
        ORDER BY {$allowedSorts[$sort]} {$dirSql}, a.`id` DESC";

    $exportRes = mysqli_query($conn, $exportSql);

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="activities_calls_'.date('Ymd_His').'.csv"');

    $out = fopen('php://output', 'w');
    fputcsv($out, [
        'ID','Lead ID','Contact Name','Company','Phone','Email',
        'Title','Description','Call Status','Direction',
        'Date Start','Date End','Duration (mins)',
        'Remind','Reminder At','Saved From','Source Of','Assigned User','Created By'
    ]);

    if ($exportRes) {
        while ($erow = mysqli_fetch_assoc($exportRes)) {
            $sourceOf = src_of($erow['saved_from'] ?? '');
            fputcsv($out, [
                $erow['id'],
                $erow['lead_id'],
                $erow['contact_name'],
                $erow['company'],
                $erow['phone'],
                $erow['email'],
                $erow['title'],
                $erow['description'],
                $erow['status'],
                $erow['direction'],
                $erow['date_start'],
                $erow['date_end'],
                $erow['duration_minutes'],
                $erow['remind'],
                $erow['reminder_at'],
                $erow['saved_from'],
                $sourceOf,
                $erow['assigned_user'],
                $erow['created_by']
            ]);
        }
    }
    fclose($out);
    exit;
}

/* ---------- Count & Fetch ---------- */
$totalRows = 0;
$cntRes = mysqli_query($conn, "SELECT COUNT(*) AS c
  FROM `crm_activities` a
  JOIN `leads` l ON l.`id`=a.`lead_id`
  {$whereSql}");
if($cntRes){
    $cr = mysqli_fetch_assoc($cntRes);
    $totalRows = (int)($cr['c'] ?? 0);
}
$totalPages = max(1, (int)ceil($totalRows / $perPage));

$sql = "SELECT a.`id`,a.`lead_id`,a.`status`,a.`direction`,a.`title`,a.`description`,
               a.`date_start`,a.`date_end`,a.`duration_minutes`,a.`assigned_user`,
               a.`created_by`,a.`remind`,a.`reminder_at`,a.`saved_from`,
               l.`contact_name`,l.`company`,l.`phone`,l.`email`
        FROM `crm_activities` a
        JOIN `leads` l ON l.`id`=a.`lead_id`
        {$whereSql}
        ORDER BY {$allowedSorts[$sort]} {$dirSql}, a.`id` DESC
        LIMIT {$perPage} OFFSET {$offset}";
$rows = [];
$res = mysqli_query($conn, $sql);
if($res){
    while($row = mysqli_fetch_assoc($res)){
        $rows[] = $row;
    }
}

/* ---------- Base params for links (sorting/pagination) ---------- */
$baseParams = [
    'per_page'  => $perPage,
    'q'         => $q,
    'date_from' => $df,
    'date_to'   => $dt,
    'assigned'  => $assigned,
    'sort'      => $sort,
    'dir'       => $dir
];

/* ---------- Styles ---------- */
ob_start(); ?>
<style>
.kpi { display:flex; gap:12px; flex-wrap:wrap; margin-bottom:12px; }
.kpi .card { padding:10px 14px; border:1px solid #e9ecef; border-radius:10px; background:#fff; min-width:180px; }
.table td, .table th { vertical-align: middle; }
.badge-soft { background: rgba(0,0,0,.06); color:#222; }
.small-muted { font-size:12px;color:#777; }

/* sortable headers */
.table thead th a { font-weight:600; display:inline-flex; align-items:center; gap:4px; color:inherit; }
.table thead th a:hover { text-decoration: underline; }

/* status badges */
.badge-status { border-radius:999px; padding:3px 9px; font-size:12px; font-weight:500; }
.badge-status-planned { background:#e0f3ff; color:#055160; }
.badge-status-done { background:#e7f5e8; color:#0f5132; }
.badge-status-cancel { background:#f8d7da; color:#842029; }
.badge-status-other { background:#e2e3e5; color:#41464b; }

/* selection highlight */
.table-hover tbody tr.table-active td { background-color:#e9f5ff; }

/* buttons */
.btn-primary, .btn-success, .btn-outline-success {
    border-radius:999px;
}
.btn-primary {
    box-shadow:0 2px 6px rgba(13,110,253,0.35);
}
.btn-success, .btn-outline-success {
    box-shadow:0 2px 6px rgba(25,135,84,0.25);
}

/* checkbox column */
th.select-col, td.select-col {
    width:32px;
    text-align:center;
}
</style>
<?php $styles = ob_get_clean();

/* ---------- Content ---------- */
ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Activities — Calls</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Calls</li>
    </ol>
  </div>
</div>

<div class="kpi">
  <div class="card">
      <div class="text-muted">Today</div>
      <div class="fs-4 fw-bold"><?php echo (int)$kpiToday; ?></div>
  </div>
  <div class="card">
      <div class="text-muted">This Week</div>
      <div class="fs-4 fw-bold"><?php echo (int)$kpiWeek; ?></div>
  </div>
  <div class="card">
      <div class="text-muted">Overdue Reminders</div>
      <div class="fs-4 fw-bold"><?php echo (int)$kpiOverdue; ?></div>
  </div>
</div>

<div class="card custom-card">
  <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-3">
    <div class="card-title">All Calls</div>
    <form class="d-flex align-items-end flex-wrap gap-2" method="get">
      <div>
        <label class="form-label mb-1 small">Date From</label>
        <input type="date" class="form-control form-control-sm" name="date_from" value="<?php echo h($df); ?>">
      </div>
      <div>
        <label class="form-label mb-1 small">Date To</label>
        <input type="date" class="form-control form-control-sm" name="date_to" value="<?php echo h($dt); ?>">
      </div>
      <div>
        <label class="form-label mb-1 small">Search</label>
        <input type="text" class="form-control form-control-sm" name="q" placeholder="name, company, title..." value="<?php echo h($q); ?>">
      </div>
      <div>
        <label class="form-label mb-1 small">Employee</label>
        <select class="form-select form-select-sm" name="assigned" onchange="this.form.submit()">
          <option value="">All</option>
          <?php foreach ($assignedList as $email): ?>
            <option value="<?php echo h($email); ?>" <?php echo $email === $assigned ? 'selected' : ''; ?>>
              <?php echo h($email); ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>
      <div>
        <label class="form-label mb-1 small">Per Page</label>
        <select class="form-select form-select-sm" name="per_page" onchange="this.form.submit()">
          <?php foreach($perPageOptions as $opt): ?>
            <option value="<?php echo $opt; ?>" <?php echo $opt === $perPage ? 'selected' : ''; ?>><?php echo $opt; ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div>
        <button class="btn btn-sm btn-primary">Filter</button>
      </div>
    </form>
  </div>

  <form method="post" id="export-form">
    <input type="hidden" name="q" value="<?php echo h($q); ?>">
    <input type="hidden" name="date_from" value="<?php echo h($df); ?>">
    <input type="hidden" name="date_to" value="<?php echo h($dt); ?>">
    <input type="hidden" name="per_page" value="<?php echo (int)$perPage; ?>">
    <input type="hidden" name="page" value="<?php echo (int)$page; ?>">
    <input type="hidden" name="sort" value="<?php echo h($sort); ?>">
    <input type="hidden" name="dir" value="<?php echo h($dir); ?>">
    <input type="hidden" name="assigned" value="<?php echo h($assigned); ?>">

    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table table-hover text-nowrap align-middle">
          <thead>
            <tr>
              <th class="select-col">
                <input type="checkbox" id="select-all">
              </th>
              <th><?php echo sort_link('When', 'when', $sort, $dir, $baseParams); ?></th>
              <th><?php echo sort_link('Contact', 'contact', $sort, $dir, $baseParams); ?></th>
              <th><?php echo sort_link('Title', 'title', $sort, $dir, $baseParams); ?></th>
              <th><?php echo sort_link('Call Status', 'status', $sort, $dir, $baseParams); ?></th>
              <th><?php echo sort_link('Reminder', 'reminder', $sort, $dir, $baseParams); ?></th>
              <th><?php echo sort_link('Saved From', 'saved_from', $sort, $dir, $baseParams); ?></th>
              <th><?php echo sort_link('Source Of', 'source_of', $sort, $dir, $baseParams); ?></th>
              <th><?php echo sort_link('Assigned', 'assigned', $sort, $dir, $baseParams); ?></th>
            </tr>
          </thead>
          <tbody>
            <?php if(!$rows): ?>
              <tr><td colspan="9" class="text-center py-4">No call activities found.</td></tr>
            <?php else: foreach($rows as $r): ?>
              <tr>
                <td class="select-col">
                  <input type="checkbox" class="row-check" name="selected_ids[]" value="<?php echo (int)$r['id']; ?>">
                </td>
                <td>
                  <div class="fw-medium">
                    <?php echo h($r['date_start'] ? date('Y-m-d H:i', strtotime($r['date_start'])) : '—'); ?>
                  </div>
                  <?php if($r['date_end']): ?>
                    <div class="small-muted">
                      to <?php echo h(date('Y-m-d H:i', strtotime($r['date_end']))); ?>
                    </div>
                  <?php endif; ?>
                </td>
                <td><?php echo h($r['contact_name'] ?: '—'); ?></td>
                <td>
                  <div class="fw-medium"><?php echo h($r['title'] ?: 'Call'); ?></div>
                  <?php if(!empty($r['description'])): ?>
                    <div class="small text-muted">
                      <?php echo h(mb_strimwidth($r['description'], 0, 80, '…', 'UTF-8')); ?>
                    </div>
                  <?php endif; ?>
                </td>
                <td><?php echo status_badge($r['status']); ?></td>
                <td>
                  <?php
                  if ($r['remind'] && $r['reminder_at']) {
                      echo h(date('Y-m-d H:i', strtotime($r['reminder_at'])));
                  } else {
                      echo '—';
                  }
                  ?>
                </td>
                <td><span class="badge badge-soft"><?php echo h($r['saved_from'] ?: '—'); ?></span></td>
                <td><?php echo h(src_of($r['saved_from'])); ?></td>
                <td><?php echo h($r['assigned_user'] ?: '—'); ?></td>
              </tr>
            <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>

    <div class="card-footer border-top-0">
      <?php
        $commonQuery = [
            'per_page'  => $perPage,
            'q'         => $q,
            'date_from' => $df,
            'date_to'   => $dt,
            'assigned'  => $assigned,
            'sort'      => $sort,
            'dir'       => $dir
        ];
      ?>
      <div class="d-flex align-items-center flex-wrap w-100 gap-2">
        <div class="d-flex align-items-center gap-2">
          <button type="submit" name="export_action" value="selected" class="btn btn-sm btn-success">
            Export Selected
          </button>
          <button type="submit" name="export_action" value="all" class="btn btn-sm btn-outline-success">
            Export All (Filtered)
          </button>
        </div>
        <div class="ms-0 ms-md-3">
          Showing <?php echo count($rows); ?> of <?php echo (int)$totalRows; ?> entries
        </div>
        <div class="ms-auto">
          <nav aria-label="Page navigation" class="pagination-style-4">
            <ul class="pagination mb-0">
              <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                <a class="page-link" href="<?php
                  if ($page > 1) {
                      $paramsPrev = $commonQuery;
                      $paramsPrev['page'] = $page - 1;
                      echo h($_SERVER['PHP_SELF']).'?'.http_build_query($paramsPrev);
                  } else {
                      echo 'javascript:void(0);';
                  }
                ?>">Prev</a>
              </li>
              <?php
                $start = max(1, $page - 2);
                $end   = min($totalPages, $page + 2);
                for ($p = $start; $p <= $end; $p++):
                    $paramsPage = $commonQuery;
                    $paramsPage['page'] = $p;
              ?>
                <li class="page-item <?php echo $p === $page ? 'active' : ''; ?>">
                  <a class="page-link" href="<?php echo h($_SERVER['PHP_SELF']).'?'.http_build_query($paramsPage); ?>">
                    <?php echo $p; ?>
                  </a>
                </li>
              <?php endfor; ?>
              <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                <a class="page-link" href="<?php
                  if ($page < $totalPages) {
                      $paramsNext = $commonQuery;
                      $paramsNext['page'] = $page + 1;
                      echo h($_SERVER['PHP_SELF']).'?'.http_build_query($paramsNext);
                  } else {
                      echo 'javascript:void(0);';
                  }
                ?>">Next</a>
              </li>
            </ul>
          </nav>
        </div>
      </div>
    </div>
  </form>
</div>
<?php $content = ob_get_clean();

/* ---------- Scripts ---------- */
ob_start(); ?>
<script>
document.addEventListener('DOMContentLoaded', function () {
    var selectAll = document.getElementById('select-all');
    var rowChecks = document.querySelectorAll('.row-check');

    function toggleRowHighlight(cb) {
        var tr = cb.closest('tr');
        if (!tr) return;
        tr.classList.toggle('table-active', cb.checked);
    }

    if (selectAll) {
        selectAll.addEventListener('change', function () {
            var checked = this.checked;
            rowChecks.forEach(function (cb) {
                cb.checked = checked;
                toggleRowHighlight(cb);
            });
        });
    }

    rowChecks.forEach(function (cb) {
        cb.addEventListener('change', function () {
            toggleRowHighlight(cb);
            if (selectAll && !this.checked) {
                selectAll.checked = false;
            }
        });
        if (cb.checked) {
            toggleRowHighlight(cb);
        }
    });
});
</script>
<?php $scripts = ob_get_clean();

/* ---------- Render via layout ---------- */
include 'layouts/base.php';
