<?php
/* ============================================================================
 * File: opportunities.php
 * Purpose: Opportunities page with compact Kanban/List + Activity logger
 * Stack : Core PHP + MySQLi (NO PDO)
 * Notes :
 *   - Shows only leads where pp_status = 'Prospect' (if column exists),
 *     else falls back to tags containing "Prospect".
 *   - RBAC permission "activities.log" gates activity buttons + save.
 *   - saved_from = "<StageLabel> Opportunities"
 *   - Board:
 *       • Compact 5-column Kanban so all stages fit on one screen
 *       • Drag & drop between stages (AJAX endpoint)
 *       • Card footer shows Assigned + Date, and below that
 *         a single row of “count + icon” buttons (Call/Mail/Task/Meeting)
 *   - List:
 *       • Columns: Contact Name, Stage, Assigned, Lead Source, Activities
 *       • Activities column = single row of “count + icon” buttons
 *       • Company column removed
 *   - Closed Won column visually highlighted
 * Charset : utf8mb4 / COLLATE=utf8mb4_general_ci
 * ========================================================================== */

date_default_timezone_set('Asia/Kolkata');
@session_start();
require_once __DIR__ . '/../config.php';

require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

/* charset & collation */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

function h($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}
function col_exists($conn, $table, $col) {
    $t = mysqli_real_escape_string($conn, $table);
    $c = mysqli_real_escape_string($conn, $col);
    $q = mysqli_query(
        $conn,
        "SELECT 1
         FROM INFORMATION_SCHEMA.COLUMNS
         WHERE TABLE_SCHEMA = DATABASE()
           AND TABLE_NAME='{$t}'
           AND COLUMN_NAME='{$c}'"
    );
    return ($q && mysqli_num_rows($q) > 0);
}

/* ---------------------------- RBAC: schema & utils ---------------------------- */
$CURRENT_USER_ID = (int)($_SESSION['user_id1'] ?? 0);

function rbac_ensure_schema($conn) {
    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `roles` (
      `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
      `name` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `description` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      UNIQUE KEY `uniq_role_name` (`name`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `permissions` (
      `code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `label` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `group_name` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
      PRIMARY KEY (`code`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `role_permissions` (
      `role_id` INT UNSIGNED NOT NULL,
      `perm_code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      PRIMARY KEY (`role_id`,`perm_code`),
      KEY `idx_perm_code` (`perm_code`),
      CONSTRAINT `fk_role_permissions_role`
        FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `user_roles` (
      `user_id` INT NOT NULL,
      `role_id` INT UNSIGNED NOT NULL,
      PRIMARY KEY (`user_id`,`role_id`),
      KEY `idx_role_id` (`role_id`),
      CONSTRAINT `fk_user_roles_role`
        FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
}

function rbac_seed_permissions($conn) {
    $perms = [
        ['activities.log', 'Log activities',      'Activities'],
        ['leads.create',   'Create leads',        'Leads'],
        ['leads.export',   'Export leads (CSV)',  'Leads'],
        ['roles.manage',   'Manage roles',        'Admin'],
    ];
    foreach ($perms as $p) {
        @mysqli_query(
            $conn,
            "INSERT IGNORE INTO `permissions` (`code`,`label`,`group_name`)
             VALUES ('" . mysqli_real_escape_string($conn, $p[0]) . "',
                     '" . mysqli_real_escape_string($conn, $p[1]) . "',
                     '" . mysqli_real_escape_string($conn, $p[2]) . "')"
        );
    }
}

function rbac_has($conn, $uid, $code) {
    if (!$uid) return true; // treat missing session as superuser in this app
    $uid  = (int)$uid;
    $code = mysqli_real_escape_string($conn, $code);
    $q = mysqli_query(
        $conn,
        "SELECT 1
         FROM user_roles ur
         JOIN role_permissions rp ON rp.role_id = ur.role_id
         WHERE ur.user_id = {$uid}
           AND rp.perm_code = '{$code}'
         LIMIT 1"
    );
    return ($q && mysqli_num_rows($q) > 0);
}

rbac_ensure_schema($conn);
rbac_seed_permissions($conn);

$CAN_LOG_ACTIVITIES = rbac_has($conn, $CURRENT_USER_ID, 'activities.log');

/* ---------------------------- Constants / helpers ---------------------------- */
$TAG_STATUS = ['Pending','Assigned','In Process','Prospect','Recycled','Dead'];
$STAGES     = ['Prospecting','Qualification','Proposal','Negotiation','Closed Won'];

function stage_to_label($s) {
    return $s === 'Prospecting' ? 'Prospect' : $s;
}

/* ---- Leads table (ensure minimal columns if table missing) ---- */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `leads` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `contact_name` VARCHAR(150) COLLATE utf8mb4_general_ci NOT NULL,
  `email` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `phone` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_status` ENUM('New','Follow-up','Closed','Contacted','Disqualified','Qualified')
      COLLATE utf8mb4_general_ci DEFAULT 'New',
  `company` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_source` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `tags` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `avatar` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `opp_title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `opp_amount` DECIMAL(12,2) DEFAULT NULL,
  `opp_close_date` DATE DEFAULT NULL,
  `opp_stage` ENUM('Prospecting','Qualification','Proposal','Negotiation','Closed Won')
      COLLATE utf8mb4_general_ci DEFAULT 'Prospecting',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

if (!col_exists($conn, 'leads', 'assigned_user')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
      ADD COLUMN `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL");
}
if (!col_exists($conn, 'leads', 'opp_title')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
      ADD COLUMN `opp_title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL");
}
if (!col_exists($conn, 'leads', 'opp_amount')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
      ADD COLUMN `opp_amount` DECIMAL(12,2) DEFAULT NULL");
}
if (!col_exists($conn, 'leads', 'opp_close_date')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
      ADD COLUMN `opp_close_date` DATE DEFAULT NULL");
}
if (!col_exists($conn, 'leads', 'opp_stage')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
      ADD COLUMN `opp_stage` ENUM('Prospecting','Qualification','Proposal','Negotiation','Closed Won')
      COLLATE utf8mb4_general_ci DEFAULT 'Prospecting'");
}

/* ---- Activities table (ensure columns) ---- */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `crm_activities` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_id` INT UNSIGNED NOT NULL,
  `activity_type` ENUM('call','meeting','task','email')
      COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'task',
  `status` ENUM('Planned','Held','Not Held','Completed','Cancelled')
      COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'Planned',
  `direction` ENUM('Inbound','Outbound','N/A')
      COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'N/A',
  `title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `description` TEXT COLLATE utf8mb4_general_ci,
  `date_start` DATETIME DEFAULT NULL,
  `date_end` DATETIME DEFAULT NULL,
  `duration_minutes` INT DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_by` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `remind` TINYINT(1) NOT NULL DEFAULT 0,
  `reminder_at` DATETIME DEFAULT NULL,
  `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY `idx_lead_id` (`lead_id`),
  KEY `idx_reminder_at` (`reminder_at`),
  KEY `idx_saved_from` (`saved_from`),
  CONSTRAINT `fk_crm_activities_lead_id`
    FOREIGN KEY (`lead_id`) REFERENCES `leads`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

if (!col_exists($conn, 'crm_activities', 'remind')) {
    @mysqli_query($conn, "ALTER TABLE `crm_activities`
      ADD COLUMN `remind` TINYINT(1) NOT NULL DEFAULT 0 AFTER `created_by`");
}
if (!col_exists($conn, 'crm_activities', 'reminder_at')) {
    @mysqli_query($conn, "ALTER TABLE `crm_activities`
      ADD COLUMN `reminder_at` DATETIME DEFAULT NULL AFTER `remind`");
    @mysqli_query($conn, "CREATE INDEX `idx_reminder_at` ON `crm_activities` (`reminder_at`)");
}
if (!col_exists($conn, 'crm_activities', 'saved_from')) {
    @mysqli_query($conn, "ALTER TABLE `crm_activities`
      ADD COLUMN `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL AFTER `reminder_at`");
    @mysqli_query($conn, "CREATE INDEX `idx_saved_from` ON `crm_activities` (`saved_from`)");
}

/* ------------------------------- AJAX: save_activity ------------------------------- */
if (isset($_POST['ajax']) && $_POST['ajax'] === 'save_activity') {
    header('Content-Type: application/json');

    if (!$CAN_LOG_ACTIVITIES) {
        echo json_encode(['ok' => false, 'msg' => 'Not authorized to log activities']);
        exit;
    }

    $lead_id = (int)($_POST['id'] ?? 0);
    if ($lead_id <= 0) {
        echo json_encode(['ok' => false, 'msg' => 'Invalid lead']);
        exit;
    }

    $type = strtolower(trim($_POST['type'] ?? 'task'));
    if (!in_array($type, ['call','meeting','task','email'], true)) {
        $type = 'task';
    }

    $status = trim($_POST['status'] ?? 'Planned');
    if (!in_array($status, ['Planned','Held','Not Held','Completed','Cancelled'], true)) {
        $status = 'Planned';
    }

    $direction = trim($_POST['direction'] ?? 'N/A');
    if (!in_array($direction, ['Inbound','Outbound','N/A'], true)) {
        $direction = 'N/A';
    }

    $duration_minutes = max(0, (int)($_POST['duration_minutes'] ?? 5));
    $date_end_raw     = trim($_POST['date_end'] ?? '');
    $date_end         = $date_end_raw ? date('Y-m-d H:i:00', strtotime($date_end_raw))
                                      : date('Y-m-d H:i:00');
    $date_start       = date('Y-m-d H:i:00', strtotime($date_end) - ($duration_minutes * 60));

    $title       = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');

    $remind          = isset($_POST['remind']) ? (int)$_POST['remind'] : 0;
    $reminder_at_raw = trim($_POST['reminder_at'] ?? '');
    $reminder_at     = ($remind && $reminder_at_raw)
                     ? date('Y-m-d H:i:00', strtotime($reminder_at_raw))
                     : null;

    // lead data
    $assigned_user = null;
    $lead_name     = '';
    $stage         = 'Prospecting';

    $qr = mysqli_query(
        $conn,
        "SELECT assigned_user, contact_name, opp_stage
         FROM leads
         WHERE id = {$lead_id}
         LIMIT 1"
    );
    if ($qr && mysqli_num_rows($qr)) {
        $lr            = mysqli_fetch_assoc($qr);
        $assigned_user = $lr['assigned_user'];
        $lead_name     = $lr['contact_name'];
        if (!empty($lr['opp_stage'])) {
            $stage = $lr['opp_stage'];
        }
    }

    if ($title === '') {
        $title = ucfirst($type) . ' - ' . $lead_name;
    }

    $saved_from = stage_to_label($stage) . ' Opportunities';
    $created_by = isset($_SESSION['namevfied1'])
                ? (string)$_SESSION['namevfied1']
                : (isset($_SESSION['user_id1']) ? (string)$_SESSION['user_id1'] : null);

    $sql = "INSERT INTO `crm_activities`
      (`lead_id`,`activity_type`,`status`,`direction`,`title`,`description`,
       `date_start`,`date_end`,`duration_minutes`,
       `assigned_user`,`created_by`,
       `remind`,`reminder_at`,`saved_from`)
      VALUES (
        " . (int)$lead_id . ",
        '" . mysqli_real_escape_string($conn, $type) . "',
        '" . mysqli_real_escape_string($conn, $status) . "',
        '" . mysqli_real_escape_string($conn, $direction) . "',
        '" . mysqli_real_escape_string($conn, $title) . "',
        '" . mysqli_real_escape_string($conn, $description) . "',
        '" . mysqli_real_escape_string($conn, $date_start) . "',
        '" . mysqli_real_escape_string($conn, $date_end) . "',
        " . (int)$duration_minutes . ",
        " . ($assigned_user === null ? "NULL" : "'" . mysqli_real_escape_string($conn, $assigned_user) . "'") . ",
        " . ($created_by   === null ? "NULL" : "'" . mysqli_real_escape_string($conn, $created_by)   . "'") . ",
        " . (int)$remind . ",
        " . ($reminder_at === null ? "NULL" : "'" . mysqli_real_escape_string($conn, $reminder_at) . "'") . ",
        '" . mysqli_real_escape_string($conn, $saved_from) . "'
      )";

    $ok = mysqli_query($conn, $sql);
    if ($ok) {
        echo json_encode(['ok' => true, 'id' => mysqli_insert_id($conn)]);
    } else {
        echo json_encode([
            'ok'  => false,
            'msg' => 'DB error',
            'err' => mysqli_error($conn),
            'sql' => $sql,
        ]);
    }
    exit;
}

/* --------------------------- AJAX: update stage (Drag) --------------------------- */
if (isset($_POST['ajax']) && $_POST['ajax'] === 'update_stage') {
    header('Content-Type: application/json');

    $id    = (int)($_POST['id'] ?? 0);
    $stage = trim($_POST['stage'] ?? '');

    global $STAGES;
    if ($id <= 0 || !in_array($stage, $STAGES, true)) {
        echo json_encode(['ok' => false, 'msg' => 'Invalid data']);
        exit;
    }

    $stmt = mysqli_prepare($conn, "UPDATE `leads` SET `opp_stage` = ? WHERE `id` = ?");
    mysqli_stmt_bind_param($stmt, 'si', $stage, $id);
    $ok = mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);

    echo json_encode(['ok' => $ok ? true : false]);
    exit;
}

/* --------------------------- Stage form (POST non-AJAX) --------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'update_stage_form') {
    $id    = (int)($_POST['id'] ?? 0);
    $stage = trim($_POST['opp_stage'] ?? '');

    if ($id > 0 && in_array($stage, $STAGES, true)) {
        $stmt = mysqli_prepare($conn, "UPDATE `leads` SET `opp_stage` = ? WHERE `id` = ?");
        mysqli_stmt_bind_param($stmt, 'si', $stage, $id);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }

    header("Location: " . $_SERVER['PHP_SELF'] . "?mode=" . urlencode($_GET['mode'] ?? 'list'));
    exit;
}

/* ------------------------------ Tags form (POST) ------------------------------ */
$flash = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST'
    && empty($_POST['ajax'])
    && ($_POST['action'] ?? '') === 'update'
) {
    $id      = (int)($_POST['id'] ?? 0);
    $tagsArr = isset($_POST['tags_full']) ? (array)$_POST['tags_full'] : [];
    $tagsCsv = implode(',', array_map('trim', $tagsArr));

    if ($id > 0) {
        $stmt = mysqli_prepare($conn, "UPDATE `leads` SET `tags` = ? WHERE `id` = ?");
        mysqli_stmt_bind_param($stmt, 'si', $tagsCsv, $id);
        $flash = mysqli_stmt_execute($stmt) ? 'Tags updated.' : 'Error updating tags.';
        mysqli_stmt_close($stmt);
    }

    header(
        "Location: " . $_SERVER['PHP_SELF'] .
        "?flash=" . urlencode($flash) .
        "&mode=" . urlencode($_GET['mode'] ?? 'list')
    );
    exit;
}

/* ------------------------------ Mode & fetching ------------------------------- */
$ua        = $_SERVER['HTTP_USER_AGENT'] ?? '';
$isMobile  = (bool)preg_match('/Mobile|Android|iPhone|iPad|iPod|IEMobile|Opera Mini/i', $ua);
$mode      = ($_GET['mode'] ?? 'board') === 'list' ? 'list' : 'board';
if ($isMobile && $mode === 'board') {
    $mode = 'list';
}

$perPageOptions = [10, 25, 50, 500];
$perPage        = (int)($_GET['per_page'] ?? 25);
if (!in_array($perPage, $perPageOptions, true)) {
    $perPage = 25;
}
$page   = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $perPage;

/* ---- Filter: all pp_status = 'Prospect' (if column exists), else tags ---- */
$whereProspect = "1";
if (col_exists($conn, 'leads', 'pp_status')) {
    $whereProspect = "pp_status = 'Prospect'";
} else {
    $whereProspect =
        "FIND_IN_SET('Prospect', REPLACE(COALESCE(`tags`,''), ' ', ''))";
}

/* Limit to assigned_user for non-admin roles */
$roleId = (int)($_SESSION['role_id'] ?? 0);
if (!in_array($roleId, [1, 2], true)) {
    $userEmail = mysqli_real_escape_string($conn, $_SESSION['user_email'] ?? '');
    if ($userEmail !== '') {
        $whereProspect .= " AND assigned_user = '{$userEmail}'";
    }
}

$orderBy = "ORDER BY `updated_at` DESC";

/* Count total rows */
$totalRows = 0;
$cntRes = mysqli_query(
    $conn,
    "SELECT COUNT(*) AS c
     FROM `leads`
     WHERE {$whereProspect}"
);
if ($cntRes) {
    $r = mysqli_fetch_assoc($cntRes);
    $totalRows = (int)$r['c'];
}

/* Fetch rows with activity counts */
$rows = [];
$sql  = "SELECT
            l.`id`,
            l.`contact_name`,
            l.`email`,
            l.`phone`,
            l.`lead_status`,
            l.`company`,
            l.`lead_source`,
            l.`tags`,
            l.`avatar`,
            l.`assigned_user`,
            l.`opp_title`,
            l.`opp_amount`,
            l.`opp_close_date`,
            l.`opp_stage`,
            l.`created_at`,
            l.`updated_at`,
            (SELECT COUNT(*) FROM `crm_activities` ca
               WHERE ca.`lead_id` = l.`id` AND ca.`activity_type` = 'call')    AS cnt_call,
            (SELECT COUNT(*) FROM `crm_activities` ca
               WHERE ca.`lead_id` = l.`id` AND ca.`activity_type` = 'email')   AS cnt_email,
            (SELECT COUNT(*) FROM `crm_activities` ca
               WHERE ca.`lead_id` = l.`id` AND ca.`activity_type` = 'task')    AS cnt_task,
            (SELECT COUNT(*) FROM `crm_activities` ca
               WHERE ca.`lead_id` = l.`id` AND ca.`activity_type` = 'meeting') AS cnt_meeting
        FROM `leads` l
        WHERE {$whereProspect}
        {$orderBy}" .
        ($mode === 'list' ? " LIMIT {$perPage} OFFSET {$offset}" : "");

$res = mysqli_query($conn, $sql);
if ($res) {
    while ($row = mysqli_fetch_assoc($res)) {
        $rows[] = $row;
    }
}
$totalPages = max(1, (int)ceil($totalRows / $perPage));

/* Group by stage for board view */
$byStage = [
    'Prospecting'  => [],
    'Qualification'=> [],
    'Proposal'     => [],
    'Negotiation'  => [],
    'Closed Won'   => [],
];
foreach ($rows as $r) {
    $stage = $r['opp_stage'] ?: 'Prospecting';
    if (!isset($byStage[$stage])) {
        $byStage[$stage] = [];
    }
    $byStage[$stage][] = $r;
}

$flash = $_GET['flash'] ?? $flash;
?>
<?php ob_start(); ?>
<link rel="stylesheet" href="<?php echo $baseUrl; ?>/assets/libs/dragula/dragula.min.css">
<style>
/* Hide any stray "New Lead" buttons on this page */
.new-lead,
.btn-new-lead {
    display: none !important;
}
.page-header-breadcrumb .btn[href*="lead"],
.page-header-breadcrumb .btn:has(i.ri-add-line) {
    display: none !important;
}

/* Small quick action icons (used in older layout – kept for consistency) */
.quick-action-icon{
    width: 28px;
    height: 28px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    border-radius: 7px;
}
.quick-action-icon i{
    font-size: 14px;
}

/* Stage chips */
.stage-pill{
    display:inline-block;
    padding:.25rem .55rem;
    border-radius:10px;
    color:#fff;
    font-weight:600;
    margin-right:.35rem;
    margin-bottom:.35rem;
    opacity:.95;
    font-size:.8rem;
}

/* Stage colors */
:root{
  --stg-prospect:#7c3aed1a;     --stg-prospect-b:#7c3aed;
  --stg-qual:#0891b21a;         --stg-qual-b:#0891b2;
  --stg-proposal:#2563eb1a;     --stg-proposal-b:#2563eb;
  --stg-neg:#d977061a;          --stg-neg-b:#d97706;
  --stg-won:#10b9811a;          --stg-won-b:#10b981;
}
.stage-Prospecting   { background:var(--stg-prospect-b); }
.stage-Qualification { background:var(--stg-qual-b); }
.stage-Proposal      { background:var(--stg-proposal-b); }
.stage-Negotiation   { background:var(--stg-neg-b); }

/* Highlight Closed Won chip more prominently */
.stage-Closed\ Won{
    background:#4f46e5;
    color:#fff;
    box-shadow:0 0 0 2px #6366f1;
}

/* Kanban board layout */
.board{
    display:flex;
    gap:10px;
    align-items:flex-start;
    flex-wrap:nowrap;
}
@media (min-width: 992px){
    .board{ overflow-x:hidden; }
}
.board-col{
    flex:1 1 calc(20% - 10px);
    min-width:210px;
}
.board-head{
    border-radius:8px;
    padding:.3rem .5rem;
    font-weight:700;
    color:#0f172a;
    background:#f8fafc;
    border:1px dashed #e5e7eb;
    display:flex;
    align-items:center;
    justify-content:space-between;
    font-size:.9rem;
}
.board-body{
    background:#fbfbfd;
    border:1px solid #eef0f4;
    border-radius:10px;
    padding:6px;
    min-height:120px;
    max-height:68vh;
    overflow:auto;
}

/* Per-column tint */
.board-Prospecting   .card-op{ background:var(--stg-prospect); border-left:3px solid var(--stg-prospect-b); }
.board-Qualification .card-op{ background:var(--stg-qual);     border-left:3px solid var(--stg-qual-b); }
.board-Proposal      .card-op{ background:var(--stg-proposal); border-left:3px solid var(--stg-proposal-b); }
.board-Negotiation   .card-op{ background:var(--stg-neg);      border-left:3px solid var(--stg-neg-b); }
.board-Closed\ Won   .card-op{ background:var(--stg-won);      border-left:3px solid var(--stg-won-b); }

.board-Prospecting   .board-head{ border-color:var(--stg-prospect-b); }
.board-Qualification .board-head{ border-color:var(--stg-qual-b); }
.board-Proposal      .board-head{ border-color:var(--stg-proposal-b); }
.board-Negotiation   .board-head{ border-color:var(--stg-neg-b); }

/* Strong visual highlight for Closed Won column */
.board-Closed\ Won .board-head{
    border-color:#2563eb;
    background:#eff6ff;
    box-shadow:0 0 0 1px #2563eb;
}
.board-Closed\ Won .board-body{
    background:#f8fafc;
    border-color:#bfdbfe;
}

/* Card */
.card-op{
    border:1px solid #e6e8ef;
    border-radius:10px;
    padding:10px 10px;
    background:#ffffff;
    margin-bottom:8px;
    box-shadow:0 1px 0 rgba(0,0,0,.03);
    cursor:grab;
    transition:box-shadow .2s, transform .05s;
}
.card-op:hover{
    box-shadow:0 6px 18px rgba(0,0,0,.06);
}
.card-op .title{
    font-weight:700;
    font-size:.95rem;
}
.card-op .company{
    color:#475569;
    font-size:.78rem;
}
.card-op .footer{
    color:#64748b;
    font-size:.75rem;
    display:flex;
    justify-content:space-between;
    margin-top:6px;
}
.card-op .badges .badge{
    font-size:.68rem;
    padding:.15rem .4rem;
    border-radius:.35rem;
    background:#eef2ff;
    color:#3730a3;
}

/* Table compactness */
.table td,
.table th{
    white-space:nowrap;
}
.table-responsive{
    overflow-x:auto;
}
.view-more-button{
    display:none !important;
}

/* Activity “counts + icons” row (used both in list & board) */
.logs-group{
    display:flex;
    align-items:center;
    gap:.75rem;
}
.log-chip{
    display:inline-flex;
    align-items:center;
    gap:.25rem;
    font-size:12px;
}
.log-count{
    min-width:1ch;
    text-align:right;
    font-weight:600;
}
.log-icon-btn{
    width:32px;
    height:32px;
    padding:0;
    border-radius:10px;
    border:0;
    background:#f7f6ff;
    display:inline-flex;
    align-items:center;
    justify-content:center;
}
.log-icon-btn i{
    font-size:16px;
    color:#001b5f;
}
.op-activity-row{
    display:flex;
    align-items:center;
    gap:.75rem;
    margin-top:6px;
}

@media (min-width: 1200px){
    .content,
    .main-content,
    .content-body,
    .page{
        padding-right:8px;
        padding-left:8px;
    }
}
@media (max-width: 1199.98px){
    .board{ overflow-x:auto; }
}
@media (max-width: 991.98px){
    .app-sidebar,
    .main-sidebar,
    .sidemenu,
    [class*="sidebar"]{
        transform: translateX(-100%) !important;
        visibility: hidden !important;
        position: fixed !important;
        left: 0;
        top: 0;
        bottom: 0;
        z-index: 100;
    }
    .content,
    .main-content,
    .content-body,
    .page{
        margin-left: 0 !important;
    }
    .btn-mode-board{
        display:none !important;
    }
}
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>

<!-- Header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Opportunities — Kanban Board</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Opportunities</li>
    </ol>
  </div>
</div>

<?php if ($flash): ?>
  <div class="alert alert-success py-2 px-3 mb-3"><?php echo h($flash); ?></div>
<?php endif; ?>

<!-- Stage chips + mode toggle -->
<div class="d-flex align-items-center justify-content-between flex-wrap gap-2 mb-3">
  <div class="d-flex flex-wrap">
    <?php foreach ($STAGES as $s): ?>
      <span class="stage-pill stage-<?php echo str_replace(' ', '\ ', $s); ?>">
        <?php echo h($s); ?>
      </span>
    <?php endforeach; ?>
  </div>
  <div class="btn-group">
    <a class="btn btn-outline-primary btn-mode-board <?php echo $mode === 'board' ? 'active' : ''; ?>"
       href="<?php echo h($_SERVER['PHP_SELF']); ?>?mode=board">
       Board
    </a>
    <a class="btn btn-outline-primary <?php echo $mode === 'list' ? 'active' : ''; ?>"
       href="<?php echo h($_SERVER['PHP_SELF']); ?>?mode=list">
       List
    </a>
  </div>
</div>

<?php if ($mode === 'board'): ?>

  <!-- Kanban board -->
  <div class="board">
    <?php foreach ($STAGES as $stage): ?>
      <div class="board-col board-<?php echo h($stage); ?>">
        <div class="board-head">
          <span><?php echo h($stage); ?></span>
          <span class="small opacity-75"><?php echo count($byStage[$stage]); ?></span>
        </div>
        <div class="board-body droppable" data-stage="<?php echo h($stage); ?>">
          <?php foreach ($byStage[$stage] as $r):
            $avatar = $r['avatar']
                ? ($baseUrl . '/' . ltrim($r['avatar'], '/'))
                : ($baseUrl . '/assets/images/faces/9.jpg');

            $date = $r['opp_close_date'] ?: substr($r['created_at'], 0, 10);

            $tagsSmall = [];
            if (!empty($r['tags'])) {
                foreach (explode(',', $r['tags']) as $tg) {
                    $tg = trim($tg);
                    if ($tg !== '') $tagsSmall[] = $tg;
                }
            }

            $cntCall    = (int)($r['cnt_call'] ?? 0);
            $cntEmail   = (int)($r['cnt_email'] ?? 0);
            $cntTask    = (int)($r['cnt_task'] ?? 0);
            $cntMeeting = (int)($r['cnt_meeting'] ?? 0);
          ?>
            <div class="card-op draggable" data-id="<?php echo (int)$r['id']; ?>">
              <div class="d-flex align-items-center justify-content-between">
                <div class="badges">
                  <?php if ($r['opp_title']): ?>
                    <span class="badge">
                      #<?php echo h(substr(preg_replace('/\s+/', '-', $r['opp_title']), 0, 10)); ?>
                    </span>
                  <?php endif; ?>
                  <?php if ($tagsSmall): ?>
                    <span class="badge ms-1"><?php echo h($tagsSmall[0]); ?></span>
                  <?php endif; ?>
                </div>
                <div class="text-muted small">
                  <i class="ri-time-line me-1"></i>
                  <?php echo h('Created - ' . date('d M', strtotime($r['created_at']))); ?>
                </div>
              </div>

              <div class="mt-2">
                <a href="javascript:void(0);"
                   class="opp-name lead-full"
                   data-id="<?php echo (int)$r['id']; ?>"
                   data-name="<?php echo h($r['contact_name']); ?>"
                   data-email="<?php echo h($r['email']); ?>"
                   data-phone="<?php echo h($r['phone']); ?>"
                   data-status="<?php echo h($r['lead_status']); ?>"
                   data-company="<?php echo h($r['company']); ?>"
                   data-source="<?php echo h($r['lead_source']); ?>"
                   data-tags="<?php echo h($r['tags']); ?>"
                   data-avatar="<?php echo h($avatar); ?>"
                   data-assigned="<?php echo h($r['assigned_user']); ?>"
                   data-oppstage="<?php echo h($r['opp_stage']); ?>">
                  <div class="title">
                    <?php echo h($r['contact_name'] ?: ($r['opp_title'] ?: '—')); ?>
                  </div>
                </a>
                <div class="company">
                  <i class="ri-building-line me-1"></i>
                  <?php echo h($r['company'] ?: '—'); ?>
                </div>
              </div>

              <div class="footer">
                <span>
                  <i class="ri-user-line me-1"></i>
                  <?php echo h($r['assigned_user'] ?: 'Unassigned'); ?>
                </span>
                <span><?php echo h($date); ?></span>
              </div>

              <!-- Single row: counts + clickable icons (green style) -->
              <?php if ($CAN_LOG_ACTIVITIES): ?>
                <div class="op-activity-row">
                  <div class="logs-group">
                    <div class="log-chip">
                      <span class="log-count"><?php echo $cntCall; ?></span>
                      <button type="button"
                              class="btn btn-sm log-icon-btn btn-quick-action"
                              data-action="call"
                              data-id="<?php echo (int)$r['id']; ?>"
                              data-name="<?php echo h($r['contact_name']); ?>"
                              data-email="<?php echo h($r['email']); ?>"
                              data-phone="<?php echo h($r['phone']); ?>"
                              data-company="<?php echo h($r['company']); ?>"
                              title="Call">
                        <i class="ri-phone-fill"></i>
                      </button>
                    </div>

                    <div class="log-chip">
                      <span class="log-count"><?php echo $cntEmail; ?></span>
                      <button type="button"
                              class="btn btn-sm log-icon-btn btn-quick-action"
                              data-action="mail"
                              data-id="<?php echo (int)$r['id']; ?>"
                              data-name="<?php echo h($r['contact_name']); ?>"
                              data-email="<?php echo h($r['email']); ?>"
                              data-phone="<?php echo h($r['phone']); ?>"
                              data-company="<?php echo h($r['company']); ?>"
                              title="Email">
                        <i class="ri-mail-send-line"></i>
                      </button>
                    </div>

                    <div class="log-chip">
                      <span class="log-count"><?php echo $cntTask; ?></span>
                      <button type="button"
                              class="btn btn-sm log-icon-btn btn-quick-action"
                              data-action="task"
                              data-id="<?php echo (int)$r['id']; ?>"
                              data-name="<?php echo h($r['contact_name']); ?>"
                              data-email="<?php echo h($r['email']); ?>"
                              data-phone="<?php echo h($r['phone']); ?>"
                              data-company="<?php echo h($r['company']); ?>"
                              title="Task">
                        <i class="ri-task-line"></i>
                      </button>
                    </div>

                    <div class="log-chip">
                      <span class="log-count"><?php echo $cntMeeting; ?></span>
                      <button type="button"
                              class="btn btn-sm log-icon-btn btn-quick-action"
                              data-action="meeting"
                              data-id="<?php echo (int)$r['id']; ?>"
                              data-name="<?php echo h($r['contact_name']); ?>"
                              data-email="<?php echo h($r['email']); ?>"
                              data-phone="<?php echo h($r['phone']); ?>"
                              data-company="<?php echo h($r['company']); ?>"
                              title="Meeting">
                        <i class="ri-calendar-event-line"></i>
                      </button>
                    </div>
                  </div>
                </div>
              <?php endif; ?>
            </div>
          <?php endforeach; ?>
        </div>
      </div>
    <?php endforeach; ?>
  </div>

<?php else: ?>

  <!-- List mode -->
  <div class="card custom-card">
    <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-3">
      <div class="card-title">
        Prospect Opportunities (<?php echo (int)$totalRows; ?>)
      </div>
      <form method="get" class="d-flex align-items-center gap-2">
        <input type="hidden" name="mode" value="list">
        <select name="per_page"
                class="form-select form-select-sm"
                onchange="this.form.submit()">
          <?php foreach ($perPageOptions as $opt): ?>
            <option value="<?php echo $opt; ?>"
              <?php echo $opt === $perPage ? 'selected' : ''; ?>>
              <?php echo $opt; ?> / page
            </option>
          <?php endforeach; ?>
        </select>
      </form>
    </div>

    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table align-middle text-nowrap">
          <thead>
            <tr>
              <th>Contact Name</th>
              <th>Stage</th>
              <th>Assigned</th>
              <th>Lead Source</th>
              <th class="text-end pe-4">Activities</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!$rows): ?>
              <tr>
                <td colspan="5" class="text-center py-4">
                  No opportunities found.
                </td>
              </tr>
            <?php else: ?>
              <?php foreach ($rows as $r):
                $avatar = $r['avatar']
                    ? ($baseUrl . '/' . ltrim($r['avatar'], '/'))
                    : ($baseUrl . '/assets/images/faces/9.jpg');

                $cntCall    = (int)($r['cnt_call'] ?? 0);
                $cntEmail   = (int)($r['cnt_email'] ?? 0);
                $cntTask    = (int)($r['cnt_task'] ?? 0);
                $cntMeeting = (int)($r['cnt_meeting'] ?? 0);
              ?>
                <tr>
                  <td>
                    <div class="d-flex align-items-center gap-2">
                      <span class="avatar avatar-sm avatar-rounded">
                        <img src="<?php echo h($avatar); ?>" alt="">
                      </span>
                      <a href="javascript:void(0);"
                         class="fw-medium lead-full"
                         data-id="<?php echo (int)$r['id']; ?>"
                         data-name="<?php echo h($r['contact_name']); ?>"
                         data-email="<?php echo h($r['email']); ?>"
                         data-phone="<?php echo h($r['phone']); ?>"
                         data-status="<?php echo h($r['lead_status']); ?>"
                         data-company="<?php echo h($r['company']); ?>"
                         data-source="<?php echo h($r['lead_source']); ?>"
                         data-tags="<?php echo h($r['tags']); ?>"
                         data-avatar="<?php echo h($avatar); ?>"
                         data-assigned="<?php echo h($r['assigned_user']); ?>"
                         data-oppstage="<?php echo h($r['opp_stage']); ?>">
                        <?php echo h($r['contact_name']); ?>
                      </a>
                    </div>
                  </td>

                  <td>
                    <a href="javascript:void(0);"
                       class="badge bg-light text-dark stage-trigger"
                       data-id="<?php echo (int)$r['id']; ?>"
                       data-stage="<?php echo h($r['opp_stage']); ?>">
                      <?php echo h($r['opp_stage']); ?>
                    </a>
                  </td>

                  <td><?php echo h($r['assigned_user'] ?: '—'); ?></td>
                  <td><?php echo h($r['lead_source'] ?: '—'); ?></td>

                  <!-- Single Activities column: counts + clickable icons -->
                  <td class="text-end pe-4">
                    <?php if ($CAN_LOG_ACTIVITIES): ?>
                      <div class="logs-group justify-content-end">
                        <div class="log-chip">
                          <span class="log-count"><?php echo $cntCall; ?></span>
                          <button type="button"
                                  class="btn btn-sm log-icon-btn btn-quick-action"
                                  data-action="call"
                                  data-id="<?php echo (int)$r['id']; ?>"
                                  data-name="<?php echo h($r['contact_name']); ?>"
                                  data-email="<?php echo h($r['email']); ?>"
                                  data-phone="<?php echo h($r['phone']); ?>"
                                  data-company="<?php echo h($r['company']); ?>"
                                  title="Call">
                            <i class="ri-phone-fill"></i>
                          </button>
                        </div>

                        <div class="log-chip">
                          <span class="log-count"><?php echo $cntEmail; ?></span>
                          <button type="button"
                                  class="btn btn-sm log-icon-btn btn-quick-action"
                                  data-action="mail"
                                  data-id="<?php echo (int)$r['id']; ?>"
                                  data-name="<?php echo h($r['contact_name']); ?>"
                                  data-email="<?php echo h($r['email']); ?>"
                                  data-phone="<?php echo h($r['phone']); ?>"
                                  data-company="<?php echo h($r['company']); ?>"
                                  title="Email">
                            <i class="ri-mail-send-line"></i>
                          </button>
                        </div>

                        <div class="log-chip">
                          <span class="log-count"><?php echo $cntTask; ?></span>
                          <button type="button"
                                  class="btn btn-sm log-icon-btn btn-quick-action"
                                  data-action="task"
                                  data-id="<?php echo (int)$r['id']; ?>"
                                  data-name="<?php echo h($r['contact_name']); ?>"
                                  data-email="<?php echo h($r['email']); ?>"
                                  data-phone="<?php echo h($r['phone']); ?>"
                                  data-company="<?php echo h($r['company']); ?>"
                                  title="Task">
                            <i class="ri-task-line"></i>
                          </button>
                        </div>

                        <div class="log-chip">
                          <span class="log-count"><?php echo $cntMeeting; ?></span>
                          <button type="button"
                                  class="btn btn-sm log-icon-btn btn-quick-action"
                                  data-action="meeting"
                                  data-id="<?php echo (int)$r['id']; ?>"
                                  data-name="<?php echo h($r['contact_name']); ?>"
                                  data-email="<?php echo h($r['email']); ?>"
                                  data-phone="<?php echo h($r['phone']); ?>"
                                  data-company="<?php echo h($r['company']); ?>"
                                  title="Meeting">
                            <i class="ri-calendar-event-line"></i>
                          </button>
                        </div>
                      </div>
                    <?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>

    <div class="card-footer border-top-0">
      <div class="d-flex align-items-center flex-wrap w-100">
        <div>
          Showing <?php echo count($rows); ?> of <?php echo (int)$totalRows; ?> entries
        </div>
        <div class="ms-auto">
          <nav aria-label="Page navigation" class="pagination-style-4">
            <ul class="pagination mb-0">
              <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                <a class="page-link"
                   href="<?php
                     if ($page > 1) {
                       echo h($_SERVER['PHP_SELF']) . '?mode=list&page=' . ($page - 1) .
                            '&per_page=' . $perPage;
                     } else {
                       echo 'javascript:void(0);';
                     }
                   ?>">
                  Prev
                </a>
              </li>
              <?php
                $start = max(1, $page - 2);
                $end   = min($totalPages, $page + 2);
                for ($p = $start; $p <= $end; $p++):
              ?>
                <li class="page-item <?php echo $p === $page ? 'active' : ''; ?>">
                  <a class="page-link"
                     href="<?php echo h($_SERVER['PHP_SELF']) .
                                  '?mode=list&page=' . $p .
                                  '&per_page=' . $perPage; ?>">
                    <?php echo $p; ?>
                  </a>
                </li>
              <?php endfor; ?>
              <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                <a class="page-link"
                   href="<?php
                     if ($page < $totalPages) {
                       echo h($_SERVER['PHP_SELF']) . '?mode=list&page=' . ($page + 1) .
                            '&per_page=' . $perPage;
                     } else {
                       echo 'javascript:void(0);';
                     }
                   ?>">
                  Next
                </a>
              </li>
            </ul>
          </nav>
        </div>
      </div>
    </div>

  </div>
<?php endif; ?>

<!-- Stage Change Modal -->
<div class="modal fade" id="stageManageModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="update_stage_form">
      <input type="hidden" name="id" id="stage-id">
      <div class="modal-header">
        <h6 class="modal-title">Change Stage</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body px-4">
        <?php foreach ($STAGES as $s): ?>
          <label class="form-check mb-2">
            <input class="form-check-input stage-radio"
                   type="radio"
                   name="opp_stage"
                   value="<?php echo h($s); ?>">
            <span class="form-check-label"><?php echo h($s); ?></span>
          </label>
        <?php endforeach; ?>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
        <button type="submit" class="btn btn-primary">Save</button>
      </div>
    </form>
  </div>
</div>

<!-- Activity Modal -->
<div class="modal fade" id="activityModal" tabindex="-1"
     aria-labelledby="activityModal" data-bs-keyboard="false" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h6 class="modal-title">Log Activity</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body" id="activity-body">
        <div class="text-center text-muted">Select an action from the list/board.</div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary"
                data-bs-dismiss="modal">Close</button>
        <button type="button" class="btn btn-primary" id="activity-save">
          Save Activity
        </button>
      </div>
    </div>
  </div>
</div>

<!-- Lead Full Modal (tags editable) -->
<div class="modal fade" id="leadFullModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-dialog-scrollable">
    <form class="modal-content" method="post">
      <input type="hidden" name="action" value="update">
      <input type="hidden" name="id" id="full-id">
      <div class="modal-header">
        <h6 class="modal-title">Lead Details</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body px-4">
        <div class="row gy-3">
          <div class="col-md-6">
            <label class="form-label">Contact Name</label>
            <div id="v-name" class="form-control-plaintext border rounded px-2 py-1"></div>
          </div>
          <div class="col-md-6">
            <label class="form-label">Assigned User</label>
            <div id="v-assigned" class="form-control-plaintext border rounded px-2 py-1"></div>
          </div>
          <div class="col-md-6">
            <label class="form-label">Email</label>
            <div id="v-email" class="form-control-plaintext border rounded px-2 py-1"></div>
          </div>
          <div class="col-md-6">
            <label class="form-label">Phone</label>
            <div id="v-phone" class="form-control-plaintext border rounded px-2 py-1"></div>
          </div>
          <div class="col-md-6">
            <label class="form-label">Company</label>
            <div id="v-company" class="form-control-plaintext border rounded px-2 py-1"></div>
          </div>
          <div class="col-md-6">
            <label class="form-label">Lead Source</label>
            <div id="v-source" class="form-control-plaintext border rounded px-2 py-1"></div>
          </div>
          <div class="col-md-6">
            <label class="form-label">Lead Status</label>
            <div id="v-status" class="form-control-plaintext border rounded px-2 py-1"></div>
          </div>
          <div class="col-12">
            <label class="form-label">Tag Status (tick to apply)</label>
            <div class="row">
              <?php foreach ($TAG_STATUS as $tg): ?>
                <div class="col-6 col-md-4 mb-2">
                  <label class="form-check">
                    <input class="form-check-input full-tags-check"
                           type="checkbox"
                           name="tags_full[]"
                           value="<?php echo h($tg); ?>">
                    <span class="form-check-label"><?php echo h($tg); ?></span>
                  </label>
                </div>
              <?php endforeach; ?>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-light"
                data-bs-dismiss="modal">Close</button>
        <button type="submit" class="btn btn-primary">Save</button>
      </div>
    </form>
  </div>
</div>

<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script src="<?php echo $baseUrl; ?>/assets/libs/dragula/dragula.min.js"></script>
<script>
(function(){
  // Hide any stray button that literally says "New Lead"
  document.addEventListener('DOMContentLoaded', function(){
    document.querySelectorAll('button, a').forEach(function(el){
      if ((el.textContent || '').trim().toLowerCase() === 'new lead') {
        el.style.display = 'none';
        el.classList.add('btn-new-lead');
      }
    });
  });

  // Lead details modal (view + tags)
  const fullEl = document.getElementById('leadFullModal');
  document.querySelectorAll('.lead-full').forEach(function(a){
    a.addEventListener('click', function(){
      const d = this.dataset;
      document.getElementById('full-id').value     = d.id || '';
      document.getElementById('v-name').textContent     = d.name     || '';
      document.getElementById('v-assigned').textContent = d.assigned || '—';
      document.getElementById('v-email').textContent    = d.email    || '—';
      document.getElementById('v-phone').textContent    = d.phone    || '—';
      document.getElementById('v-company').textContent  = d.company  || '—';
      document.getElementById('v-source').textContent   = d.source   || '—';
      document.getElementById('v-status').textContent   = d.status   || 'New';

      const sel = (d.tags || '').split(',').map(function(s){ return s.trim(); });
      fullEl.querySelectorAll('.full-tags-check').forEach(function(ch){
        ch.checked = sel.includes(ch.value);
      });

      new bootstrap.Modal(fullEl).show();
    });
  });

  // Stage change modal
  const stageModalEl = document.getElementById('stageManageModal');
  document.querySelectorAll('.stage-trigger').forEach(function(a){
    a.addEventListener('click', function(){
      const id    = this.getAttribute('data-id');
      const stage = this.getAttribute('data-stage') || '';
      document.getElementById('stage-id').value = id || '';
      stageModalEl.querySelectorAll('.stage-radio').forEach(function(r){
        r.checked = (r.value === stage);
      });
      new bootstrap.Modal(stageModalEl).show();
    });
  });

  // Activity modal (unified form + reminder)
  const activityModalEl = document.getElementById('activityModal');
  const activityBody    = document.getElementById('activity-body');
  const activitySaveBtn = document.getElementById('activity-save');
  let currentActivity   = {type:null, id:null};

  function defaultEndISO(min){
    const t  = new Date(Date.now() + min * 60000);
    const y  = t.getFullYear();
    const m  = String(t.getMonth() + 1).padStart(2, '0');
    const d  = String(t.getDate()).padStart(2, '0');
    const hh = String(t.getHours()).padStart(2, '0');
    const mm = String(t.getMinutes()).padStart(2, '0');
    return `${y}-${m}-${d}T${hh}:${mm}`;
  }

  function buildActivityForm(type, d){
    const isCall = (type === 'call');
    const pretty =
      type === 'meeting' ? 'Meeting' :
      type === 'mail'    ? 'Email'   :
      type === 'call'    ? 'Call'    : 'Task';

    const endDefault = defaultEndISO(5);

    return `
      <div class="row g-2">
        <div class="col-md-6">
          <label class="form-label">Status</label>
          <select class="form-select" id="act-status">
            <option value="Planned">Planned</option>
            <option value="Held">Held</option>
            <option value="Not Held">Not Held</option>
            <option value="Completed">Completed</option>
            <option value="Cancelled">Cancelled</option>
          </select>
        </div>
        <div class="col-md-6">
          <label class="form-label">Direction</label>
          <select class="form-select" id="act-direction" ${isCall ? '' : 'disabled'}>
            ${isCall
              ? '<option value="Outbound" selected>Outbound</option><option value="Inbound">Inbound</option><option value="N/A">N/A</option>'
              : '<option value="N/A" selected>N/A</option>'}
          </select>
          ${isCall ? '' : '<div class="small text-muted mt-1">Direction applies to Calls only.</div>'}
        </div>
        <div class="col-md-6">
          <label class="form-label">Duration</label>
          <select class="form-select" id="act-duration">
            <option value="5">5m</option>
            <option value="15">15m</option>
            <option value="30">30m</option>
            <option value="45">45m</option>
            <option value="60">1h</option>
            <option value="90">1h 30m</option>
            <option value="120">2h</option>
          </select>
        </div>
        <div class="col-md-6">
          <label class="form-label">Date End *</label>
          <input type="datetime-local" class="form-control" id="act-dateend" value="${endDefault}">
        </div>
        <div class="col-12">
          <label class="form-label">Title</label>
          <input type="text" class="form-control" id="act-title"
                 value="${pretty + (d.name ? (' - ' + d.name) : '')}">
        </div>
        <div class="col-12">
          <label class="form-label">Description</label>
          <textarea class="form-control" id="act-desc" rows="4"
            placeholder="${pretty === 'Meeting' ? 'Agenda / notes' : 'Notes'}"></textarea>
        </div>
        <div class="col-12"><hr class="my-2"></div>
        <div class="col-md-6">
          <label class="form-label d-flex align-items-center gap-2">
            <input type="checkbox" id="act-remind"> Set Reminder
          </label>
        </div>
        <div class="col-md-6">
          <label class="form-label">Reminder At</label>
          <input type="datetime-local" class="form-control" id="act-remind-at"
                 value="${endDefault}" disabled>
        </div>
      </div>
    `;
  }

  document.addEventListener('click', function(ev){
    const btn = ev.target.closest('.btn-quick-action');
    if (!btn) return;

    const d = btn.dataset;
    currentActivity = {
      type: (d.action || 'task'),
      id:   (d.id     || null)
    };

    const titleText =
      currentActivity.type === 'call'    ? 'Call'    :
      currentActivity.type === 'meeting' ? 'Meeting' :
      currentActivity.type === 'mail'    ? 'Email'   : 'Task';

    activityModalEl.querySelector('.modal-title').textContent =
      'Log ' + titleText;

    activityBody.innerHTML = buildActivityForm(currentActivity.type, d);

    const cb = document.getElementById('act-remind');
    const at = document.getElementById('act-remind-at');
    if (cb && at) {
      at.disabled = !cb.checked;
      cb.addEventListener('change', function(){
        at.disabled = !cb.checked;
      });
    }

    new bootstrap.Modal(activityModalEl).show();
  });

  if (activitySaveBtn) {
    activitySaveBtn.addEventListener('click', function(){
      if (!currentActivity.id) {
        alert('Missing lead');
        return;
      }

      const status    = document.getElementById('act-status').value;
      const dirEl     = document.getElementById('act-direction');
      const direction = dirEl ? (dirEl.disabled ? 'N/A' : dirEl.value) : 'N/A';
      const duration  = parseInt(document.getElementById('act-duration').value || '5', 10);
      const dateend   = document.getElementById('act-dateend').value;
      const desc      = document.getElementById('act-desc').value  || '';
      const title     = document.getElementById('act-title').value || '';
      const remind    = document.getElementById('act-remind').checked ? 1 : 0;
      const remindAt  = document.getElementById('act-remind-at').value || '';

      const form = new URLSearchParams();
      form.append('ajax', 'save_activity');
      form.append('id', currentActivity.id);
      form.append('type', currentActivity.type);
      form.append('status', status);
      form.append('direction', direction);
      form.append('duration_minutes', isNaN(duration) ? 5 : duration);
      form.append('date_end', dateend ? dateend.replace('T', ' ') : '');
      form.append('description', desc);
      form.append('title', title);
      form.append('remind', String(remind));
      form.append('reminder_at', remindAt ? remindAt.replace('T', ' ') : '');

      fetch('<?php echo h($_SERVER["PHP_SELF"]); ?>', {
        method: 'POST',
        headers: {'Content-Type':'application/x-www-form-urlencoded'},
        body: form.toString()
      })
      .then(async function(r){
        const t = await r.text();
        let js = null;
        try { js = JSON.parse(t); } catch(e){}
        return {ok:r.ok, text:t, json:js};
      })
      .then(function(res){
        if (res.json && res.json.ok) {
          (bootstrap.Modal.getInstance(activityModalEl)
            || new bootstrap.Modal(activityModalEl)).hide();
          alert('Activity saved.');
        } else {
          alert(
            (res.json && (res.json.err || res.json.msg))
              ? (res.json.msg + (res.json.err ? ' — ' + res.json.err : ''))
              : 'Save failed.'
          );
          console.log('RAW:', res.text);
          if (res.json && res.json.sql) console.log('SQL:', res.json.sql);
        }
      })
      .catch(function(){
        alert('Network error');
      });
    });
  }

  // Drag & drop between columns (desktop only)
  const cols = Array.prototype.slice.call(
    document.querySelectorAll('.board-body.droppable')
  );
  if (cols.length && window.innerWidth >= 992) {
    const drake = dragula(cols, {revertOnSpill:true});
    drake.on('drop', function(el, target){
      const id    = el.getAttribute('data-id');
      const stage = target.getAttribute('data-stage');
      if (!id || !stage) return;

      const body = new URLSearchParams({
        ajax: 'update_stage',
        id:   id,
        stage: stage
      }).toString();

      fetch('<?php echo h($_SERVER["PHP_SELF"]); ?>', {
        method: 'POST',
        headers: {'Content-Type':'application/x-www-form-urlencoded'},
        body: body
      })
      .then(function(r){ return r.json(); })
      .then(function(js){
        if (!js.ok) {
          alert('Failed to update stage');
        }
      })
      .catch(function(){
        alert('Network error');
      });
    });
  }
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
