<?php
/* =============================================================================
 * File   : email_templates.php
 * Purpose: GUI-based portal to create and manage ZeptoMail templates.
 *          - Stores templates in `email_templates` table (utf8mb4_general_ci).
 *          - Each template gets a unique `template_key`.
 *          - Supports editable Header HTML + Footer HTML + Plain Body Text
 *            which is auto-converted into full HTML email layout.
 *          - Shows ready-to-copy PHP script using zeptomail_helper.php.
 * Stack  : Core PHP + MySQLi (NO PDO). DB connection via config.php.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

date_default_timezone_set('Asia/Kolkata');

/* ------------------------- Centralized config include ------------------------ */
$config_paths = array(
    __DIR__ . '/config.php',
    __DIR__ . '/../config.php',
    __DIR__ . '/../../config.php'
);

$config_found = false;
foreach ($config_paths as $cp) {
    if (is_file($cp)) {
        require_once $cp;
        $config_found = true;
        break;
    }
}

if (!$config_found) {
    die('config.php not found for email templates portal.');
}

/* ------------------------------ DB connection -------------------------------- */
if (!isset($conn) || !($conn instanceof mysqli)) {
    die('MySQLi connection $conn not found. Please ensure config.php sets $conn.');
}

mysqli_set_charset($conn, 'utf8mb4');
mysqli_query($conn, "SET collation_connection='utf8mb4_general_ci'");

/* ----------------------------- Helper functions ------------------------------ */

/**
 * Check if a column exists in a table.
 */
function et_column_exists(mysqli $conn, $table, $column)
{
    $table  = $conn->real_escape_string($table);
    $column = $conn->real_escape_string($column);
    $sql    = "SHOW COLUMNS FROM `{$table}` LIKE '{$column}'";
    if ($res = $conn->query($sql)) {
        $exists = ($res->num_rows > 0);
        $res->free();
        return $exists;
    }
    return false;
}

/**
 * Ensure new columns exist in email_templates table.
 * Adds:
 *   header_html LONGTEXT
 *   footer_html LONGTEXT
 */
function et_ensure_template_columns(mysqli $conn)
{
    if (!et_column_exists($conn, 'email_templates', 'header_html')) {
        $conn->query("ALTER TABLE `email_templates` ADD COLUMN `header_html` LONGTEXT NULL AFTER `subject`");
    }
    if (!et_column_exists($conn, 'email_templates', 'footer_html')) {
        $conn->query("ALTER TABLE `email_templates` ADD COLUMN `footer_html` LONGTEXT NULL AFTER `html_body`");
    }
}

et_ensure_template_columns($conn);

/**
 * Default Broriser header HTML (matches your screenshot).
 */
function et_default_header_html()
{
    return ''
        . '<div style="background:#0d152d;padding:25px 30px;border-top-left-radius:12px;'
        . 'border-top-right-radius:12px;">'
        . '  <table width="100%" cellspacing="0" cellpadding="0">'
        . '    <tr>'
        . '      <td style="color:#ffffff;font-family:Arial,sans-serif;font-size:22px;font-weight:700;">'
        . '        Broriser<br>'
        . '        <span style="font-size:13px;font-weight:500;">Employee Onboarding</span>'
        . '      </td>'
        . '      <td align="right">'
        . '        <div style="border:2px solid #23b2ff;border-radius:6px;padding:10px 18px;'
        . '                    color:#ffffff;font-size:14px;font-weight:600;">'
        . '          LOGO image'
        . '        </div>'
        . '      </td>'
        . '    </tr>'
        . '  </table>'
        . '</div>'
        . '<div style="height:6px;background:linear-gradient(90deg,#d4ff00,#00ead9);"></div>';
}

/**
 * Default Broriser footer HTML (matches your screenshot).
 */
function et_default_footer_html()
{
    $year = date('Y');
    return ''
        . '<div style="height:6px;background:linear-gradient(90deg,#d4ff00,#00ead9);margin-top:25px;"></div>'
        . '<div style="padding:10px 0;text-align:center;color:#6b7280;'
        . 'font-family:Arial,sans-serif;font-size:13px;">'
        . '© ' . $year . ' Broriser'
        . '</div>';
}

/**
 * Convert plain text body into HTML paragraphs with basic styling.
 * - Double line breaks => new paragraph
 * - Single line breaks => <br>
 * - Keeps placeholders like {{NAME}} as-is.
 */
function et_plain_to_html_body($plain)
{
    $plain = (string)$plain;
    $plain = str_replace(array("\r\n", "\r"), "\n", $plain);
    $plain = trim($plain);
    if ($plain === '') {
        return '';
    }

    $blocks = preg_split('/\n{2,}/', $plain);
    $html   = '';

    foreach ($blocks as $block) {
        $block = trim($block);
        if ($block === '') {
            continue;
        }
        $block = nl2br($block, false);
        $html .= '<p style="margin:0 0 10px 0;line-height:1.5;font-size:14px;color:#0f172a;">'
              . $block
              . '</p>';
    }

    return $html;
}

/**
 * Build full HTML email: wrapper + header_html + body_html + footer_html.
 */
function et_build_full_html_email($header_html, $body_plain, $footer_html)
{
    $header_html = trim((string)$header_html);
    $footer_html = trim((string)$footer_html);
    $body_html   = et_plain_to_html_body($body_plain);

    $html  = '<div style="font-family:Inter,Arial,sans-serif;background:#f6f7fb;';
    $html .= 'padding:28px 0;color:#0f172a;">';
    $html .= '<table role="presentation" cellpadding="0" cellspacing="0" align="center" width="680" ';
    $html .= 'style="background:#ffffff;border-radius:16px;overflow:hidden;border:1px solid #e8ecf3">';

    if ($header_html !== '') {
        $html .= '<tr><td style="padding:0">' . $header_html . '</td></tr>';
    }

    $html .= '<tr><td style="padding:22px">';
    $html .= $body_html !== '' ? $body_html
        : '<p style="margin:0 0 10px 0;line-height:1.5;font-size:14px;color:#0f172a;">&nbsp;</p>';
    $html .= '</td></tr>';

    if ($footer_html !== '') {
        $html .= '<tr><td style="padding:0">' . $footer_html . '</td></tr>';
    }

    $html .= '</table></div>';

    return $html;
}

/**
 * Extracts {{PLACEHOLDER}} tokens from text (subject + HTML body).
 */
function et_extract_placeholders($text)
{
    $out = array();
    if (preg_match_all('/\{\{([A-Z0-9_]+)\}\}/i', $text, $matches)) {
        foreach ($matches[1] as $name) {
            $name = strtoupper(trim($name));
            if ($name !== '' && !in_array($name, $out, true)) {
                $out[] = $name;
            }
        }
    }
    return $out;
}

/* --------------------------- Init view variables ---------------------------- */
$message      = '';
$error        = '';
$editing      = false;
$edit_row     = null;
$snippet_code = '';
$redirect_id  = 0;

/* ------------------------------ Handle POST save ---------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_template'])) {
    $id            = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $template_name = isset($_POST['template_name']) ? trim($_POST['template_name']) : '';
    $template_key  = isset($_POST['template_key']) ? trim($_POST['template_key']) : '';
    $subject       = isset($_POST['subject']) ? trim($_POST['subject']) : '';

    // New fields: header + plain body + footer
    $header_html   = isset($_POST['header_html']) ? (string)$_POST['header_html'] : '';
    $body_plain    = isset($_POST['body_plain']) ? (string)$_POST['body_plain'] : '';
    $text_body     = $body_plain; // store plain text as text version
    $footer_html   = isset($_POST['footer_html']) ? (string)$_POST['footer_html'] : '';

    $default_cc    = isset($_POST['default_cc']) ? trim($_POST['default_cc']) : '';
    $default_bcc   = isset($_POST['default_bcc']) ? trim($_POST['default_bcc']) : '';
    $track_clicks  = isset($_POST['track_clicks']) ? 1 : 0;
    $track_opens   = isset($_POST['track_opens']) ? 1 : 0;
    $is_active     = isset($_POST['is_active']) ? 1 : 0;

    // Normalize template_key as slug: a-z0-9_ only
    $template_key = strtolower($template_key);
    $template_key = preg_replace('/[^a-z0-9_]+/', '_', $template_key);
    $template_key = trim($template_key, '_');

    // Apply defaults for brand header/footer/body when creating a NEW template
    if ($id <= 0) {
        if (trim($header_html) === '') {
            $header_html = et_default_header_html();
        }
        if (trim($footer_html) === '') {
            $footer_html = et_default_footer_html();
        }
        if (trim($body_plain) === '') {
            $body_plain = "Hello {{NAME}},\n\n"
                        . "Your Employee onboarding application has been {{STATUS}}.\n\n"
                        . "For any clarification, you may contact your Broriser HR Deptt.\n\n"
                        . "Regards,\nBroriser Team";
            $text_body  = $body_plain;
        }
    }

    // Build HTML body from header + plain body + footer
    $html_body = et_build_full_html_email($header_html, $body_plain, $footer_html);

    if ($template_name === '' || $template_key === '' || $subject === '' || trim($html_body) === '') {
        $error = 'Template Name, Template Key, Subject, and Body are required.';
    } else {
        if ($id > 0) {
            // Update existing template
            $sql = "UPDATE email_templates
                    SET template_name = ?, template_key = ?, subject = ?,
                        header_html = ?, html_body = ?, footer_html = ?,
                        text_body = ?, default_cc = ?, default_bcc = ?,
                        track_clicks = ?, track_opens = ?, is_active = ?, updated_at = NOW()
                    WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                $error = 'Failed to prepare UPDATE: ' . $conn->error;
            } else {
                $stmt->bind_param(
                    'sssssssssiiii',
                    $template_name,
                    $template_key,
                    $subject,
                    $header_html,
                    $html_body,
                    $footer_html,
                    $text_body,
                    $default_cc,
                    $default_bcc,
                    $track_clicks,
                    $track_opens,
                    $is_active,
                    $id
                );
                if ($stmt->execute()) {
                    $message     = 'Template updated successfully.';
                    $redirect_id = $id;
                } else {
                    $error = 'Failed to execute UPDATE: ' . $stmt->error;
                }
                $stmt->close();
            }
        } else {
            // Insert new template
            $sql = "INSERT INTO email_templates
                        (template_name, template_key, subject,
                         header_html, html_body, footer_html, text_body,
                         default_cc, default_bcc, track_clicks, track_opens, is_active,
                         created_at, updated_at)
                    VALUES (?,?,?,?,?,?,?,?,?,?,?,?,NOW(),NOW())";
            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                $error = 'Failed to prepare INSERT: ' . $conn->error;
            } else {
                $stmt->bind_param(
                    'sssssssssiii',
                    $template_name,
                    $template_key,
                    $subject,
                    $header_html,
                    $html_body,
                    $footer_html,
                    $text_body,
                    $default_cc,
                    $default_bcc,
                    $track_clicks,
                    $track_opens,
                    $is_active
                );
                if ($stmt->execute()) {
                    $newId       = $stmt->insert_id;
                    $message     = 'Template created successfully.';
                    $redirect_id = $newId;
                } else {
                    $error = 'Failed to execute INSERT: ' . $stmt->error;
                }
                $stmt->close();
            }
        }
    }

    // Redirect after successful save to avoid form resubmission
    if ($redirect_id > 0 && $error === '') {
        $loc = $_SERVER['PHP_SELF'] . '?edit=' . $redirect_id . '&saved=1';
        header('Location: ' . $loc);
        exit;
    }
}

/* ------------------------------ Handle GET edit ----------------------------- */
if (isset($_GET['edit'])) {
    $edit_id = (int)$_GET['edit'];
    if ($edit_id > 0) {
        $sql = "SELECT * FROM email_templates WHERE id = ? LIMIT 1";
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param('i', $edit_id);
            $stmt->execute();
            $res = $stmt->get_result();
            if ($res && $res->num_rows > 0) {
                $edit_row = $res->fetch_assoc();
                $editing  = true;
            }
            $stmt->close();
        }
    }
}

/* ------------------------------ Saved message ------------------------------- */
if (isset($_GET['saved']) && $_GET['saved'] == '1' && $message === '') {
    $message = 'Template saved successfully.';
}

/* ----------------------------- Fetch all templates -------------------------- */
$templates = array();
$res = $conn->query("SELECT id, template_name, template_key, subject, is_active, created_at, updated_at
                     FROM email_templates
                     ORDER BY template_name ASC");
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $templates[] = $row;
    }
    $res->free();
}

/* ---------------------------- Prepare form values --------------------------- */
$form_id            = 0;
$form_name          = '';
$form_key           = '';
$form_subject       = '';
$form_header_html   = '';
$form_html_body     = '';
$form_footer_html   = '';
$form_text_body     = '';
$form_default_cc    = '';
$form_default_bcc   = '';
$form_track_clicks  = 1;
$form_track_opens   = 1;
$form_is_active     = 1;

if ($editing && $edit_row) {
    $form_id            = (int)$edit_row['id'];
    $form_name          = $edit_row['template_name'];
    $form_key           = $edit_row['template_key'];
    $form_subject       = $edit_row['subject'];
    $form_header_html   = isset($edit_row['header_html']) ? $edit_row['header_html'] : '';
    $form_html_body     = $edit_row['html_body'];
    $form_footer_html   = isset($edit_row['footer_html']) ? $edit_row['footer_html'] : '';
    $form_text_body     = $edit_row['text_body'];
    $form_default_cc    = $edit_row['default_cc'];
    $form_default_bcc   = $edit_row['default_bcc'];
    $form_track_clicks  = (int)$edit_row['track_clicks'];
    $form_track_opens   = (int)$edit_row['track_opens'];
    $form_is_active     = (int)$edit_row['is_active'];

    // Generate sample PHP snippet for this template
    $placeholders = et_extract_placeholders($form_subject . ' ' . $form_html_body);
    $snippet_code  = "<?php\n";
    $snippet_code .= "require_once __DIR__ . '/zeptomail_helper.php';\n\n";
    $snippet_code .= "\$toEmail = 'user@example.com';\n";
    $snippet_code .= "\$toName  = 'User Name';\n\n";

    if (!empty($placeholders)) {
        $snippet_code .= "\$placeholders = [\n";
        foreach ($placeholders as $ph) {
            $snippet_code .= "    '{{" . $ph . "}}' => 'VALUE_FOR_" . $ph . "',\n";
        }
        $snippet_code .= "];\n\n";
    } else {
        $snippet_code .= "\$placeholders = [];\n\n";
    }

    $snippet_code .= "\$result = zepto_send_template(\n";
    $snippet_code .= "    \$toEmail,\n";
    $snippet_code .= "    \$toName,\n";
    $snippet_code .= "    '" . $form_key . "', // template_key\n";
    $snippet_code .= "    \$placeholders\n";
    $snippet_code .= ");\n\n";
    $snippet_code .= "if (!empty(\$result['success'])) {\n";
    $snippet_code .= "    // Email sent successfully\n";
    $snippet_code .= "    // You can show a success message or log the result\n";
    $snippet_code .= "} else {\n";
    $snippet_code .= "    // Failed to send email, handle error\n";
    $snippet_code .= "    error_log('ZeptoMail error: ' . (\$result['error'] ?? 'unknown error'));\n";
    $snippet_code .= "}\n";
    $snippet_code .= "?>\n";
}

/* If not editing and POST failed, keep last entered data for user convenience */
if (!$editing && $_SERVER['REQUEST_METHOD'] === 'POST' && $error !== '') {
    $form_id            = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $form_name          = isset($_POST['template_name']) ? $_POST['template_name'] : '';
    $form_key           = isset($_POST['template_key']) ? $_POST['template_key'] : '';
    $form_subject       = isset($_POST['subject']) ? $_POST['subject'] : '';
    $form_header_html   = isset($_POST['header_html']) ? $_POST['header_html'] : '';
    $form_text_body     = isset($_POST['body_plain']) ? $_POST['body_plain'] : '';
    $form_footer_html   = isset($_POST['footer_html']) ? $_POST['footer_html'] : '';
    $form_default_cc    = isset($_POST['default_cc']) ? $_POST['default_cc'] : '';
    $form_default_bcc   = isset($_POST['default_bcc']) ? $_POST['default_bcc'] : '';
    $form_track_clicks  = isset($_POST['track_clicks']) ? 1 : 0;
    $form_track_opens   = isset($_POST['track_opens']) ? 1 : 0;
    $form_is_active     = isset($_POST['is_active']) ? 1 : 0;
    $form_html_body     = et_build_full_html_email(
        $form_header_html !== '' ? $form_header_html : et_default_header_html(),
        $form_text_body,
        $form_footer_html !== '' ? $form_footer_html : et_default_footer_html()
    );
}

/* New template default on first load */
if (!$editing && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    $form_text_body   = "Hello {{NAME}},\n\n"
        . "Your Employee onboarding application has been {{STATUS}}.\n\n"
        . "For any clarification, you may contact your Broriser HR Deptt.\n\n"
        . "Regards,\nBroriser Team";
    $form_header_html = et_default_header_html();
    $form_footer_html = et_default_footer_html();
    $form_html_body   = et_build_full_html_email($form_header_html, $form_text_body, $form_footer_html);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Email Templates Manager (ZeptoMail)</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body {
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            font-size: 14px;
            margin: 0;
            padding: 0;
            background: #f4f5f7;
        }
        .et-wrapper {
            display: flex;
            min-height: 100vh;
        }
        .et-sidebar {
            width: 30%;
            max-width: 380px;
            background: #111827;
            color: #e5e7eb;
            padding: 16px;
            box-sizing: border-box;
        }
        .et-main {
            flex: 1;
            padding: 16px 24px;
            box-sizing: border-box;
        }
        h1, h2, h3 {
            margin-top: 0;
        }
        .et-logo {
            font-weight: 600;
            margin-bottom: 16px;
        }
        .et-template-list {
            list-style: none;
            padding: 0;
            margin: 0;
            max-height: calc(100vh - 140px);
            overflow-y: auto;
        }
        .et-template-list li {
            margin-bottom: 8px;
        }
        .et-template-link {
            display: block;
            padding: 8px 10px;
            background: #111827;
            border-radius: 6px;
            text-decoration: none;
            color: #e5e7eb;
            border: 1px solid #374151;
        }
        .et-template-link.active {
            background: #2563eb;
            border-color: #2563eb;
        }
        .et-template-link span.key {
            display: block;
            font-size: 11px;
            color: #9ca3af;
        }
        .et-badge {
            font-size: 11px;
            padding: 2px 6px;
            border-radius: 999px;
            border: 1px solid rgba(255,255,255,0.3);
            margin-left: 8px;
        }
        .et-badge.inactive {
            border-color: #f97316;
            color: #fed7aa;
        }
        .et-badge.active {
            border-color: #22c55e;
            color: #bbf7d0;
        }
        .et-btn {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 6px;
            border: 1px solid transparent;
            background: #2563eb;
            color: #fff;
            font-size: 13px;
            cursor: pointer;
            text-decoration: none;
            margin-right: 6px;
        }
        .et-btn.secondary {
            background: #111827;
            border-color: #4b5563;
        }
        .et-btn.small {
            padding: 4px 8px;
            font-size: 12px;
        }
        .et-section {
            background: #ffffff;
            border-radius: 12px;
            padding: 16px 18px;
            margin-bottom: 16px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }
        label {
            display: block;
            font-size: 13px;
            margin-bottom: 4px;
            font-weight: 500;
        }
        input[type="text"],
        textarea {
            width: 100%;
            padding: 7px 10px;
            border-radius: 6px;
            border: 1px solid #d1d5db;
            font-size: 13px;
            box-sizing: border-box;
        }
        textarea {
            min-height: 80px;
            resize: vertical;
        }
        .et-grid {
            display: grid;
            grid-gap: 12px;
        }
        @media (min-width: 900px) {
            .et-grid-2 {
                grid-template-columns: repeat(2, minmax(0, 1fr));
            }
        }
        .et-row {
            margin-bottom: 10px;
        }
        .et-flags label {
            display: inline-flex;
            align-items: center;
            margin-right: 14px;
            font-weight: 400;
        }
        .et-flags input[type="checkbox"] {
            margin-right: 6px;
        }
        .et-alert {
            padding: 8px 10px;
            border-radius: 6px;
            margin-bottom: 10px;
            font-size: 13px;
        }
        .et-alert.success {
            background: #ecfdf3;
            border: 1px solid #22c55e;
            color: #166534;
        }
        .et-alert.error {
            background: #fef2f2;
            border: 1px solid #ef4444;
            color: #b91c1c;
        }
        .et-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 12px;
        }
        .et-table th,
        .et-table td {
            border-bottom: 1px solid #e5e7eb;
            padding: 6px 4px;
            text-align: left;
        }
        .et-table th {
            font-weight: 600;
            color: #4b5563;
        }
        .et-code {
            width: 100%;
            min-height: 200px;
            font-family: "SFMono-Regular", Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
            font-size: 12px;
            padding: 8px 10px;
            border-radius: 8px;
            border: 1px solid #d1d5db;
            background: #f9fafb;
            box-sizing: border-box;
        }
        .et-header-row {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 10px;
        }
        .et-header-row h2,
        .et-header-row h3 {
            margin: 0;
            font-size: 16px;
        }
        .et-note {
            font-size: 12px;
            color: #6b7280;
        }
        .et-preview-box {
            background:#e5e7eb;
            padding:12px;
            border-radius:10px;
            max-height:420px;
            overflow:auto;
        }
    </style>
</head>
<body>
<div class="et-wrapper">
    <aside class="et-sidebar">
        <div class="et-logo">Email Templates (ZeptoMail)</div>
        <div style="margin-bottom: 10px;">
            <a href="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" class="et-btn small">+ New Template</a>
        </div>
        <ul class="et-template-list">
            <?php if (!empty($templates)): ?>
                <?php foreach ($templates as $tpl): ?>
                    <?php
                    $isActive  = (int)$tpl['is_active'] === 1;
                    $isCurrent = ($editing && $form_id === (int)$tpl['id']);
                    ?>
                    <li>
                        <a class="et-template-link<?php echo $isCurrent ? ' active' : ''; ?>"
                           href="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>?edit=<?php echo (int)$tpl['id']; ?>">
                            <strong><?php echo htmlspecialchars($tpl['template_name']); ?></strong>
                            <span class="key"><?php echo htmlspecialchars($tpl['template_key']); ?></span>
                            <span class="et-badge <?php echo $isActive ? 'active' : 'inactive'; ?>">
                                <?php echo $isActive ? 'Active' : 'Inactive'; ?>
                            </span>
                        </a>
                    </li>
                <?php endforeach; ?>
            <?php else: ?>
                <li>No templates created yet.</li>
            <?php endif; ?>
        </ul>
    </aside>
    <main class="et-main">
        <div class="et-section">
            <div class="et-header-row">
                <h2><?php echo $editing ? 'Edit Template' : 'Create New Template'; ?></h2>
                <span class="et-note">
                    Use <code>{{PLACEHOLDER}}</code> tokens in Subject / Body for dynamic values
                    (e.g. {{NAME}}, {{STATUS}}).
                </span>
            </div>

            <?php if ($message !== ''): ?>
                <div class="et-alert success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>

            <?php if ($error !== ''): ?>
                <div class="et-alert error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>

            <form method="post" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>">
                <input type="hidden" name="id" value="<?php echo (int)$form_id; ?>">

                <div class="et-grid et-grid-2">
                    <div class="et-row">
                        <label for="template_name">Template Name *</label>
                        <input type="text" name="template_name" id="template_name"
                               value="<?php echo htmlspecialchars($form_name); ?>"
                               placeholder="Onboarding Rejected, Onboarding Approved, etc.">
                    </div>
                    <div class="et-row">
                        <label for="template_key">Template Key (used in script) *</label>
                        <input type="text" name="template_key" id="template_key"
                               value="<?php echo htmlspecialchars($form_key); ?>"
                               placeholder="employee_onb_rejected">
                        <div class="et-note">
                            Only letters, numbers, and underscore will be kept. Eg: <code>employee_onb_rejected</code>.
                        </div>
                    </div>
                </div>

                <div class="et-row">
                    <label for="subject">Email Subject *</label>
                    <input type="text" name="subject" id="subject"
                           value="<?php echo htmlspecialchars($form_subject); ?>"
                           placeholder="Employee Onboarding Status – {{STATUS}}">
                </div>

                <div class="et-row">
                    <label for="header_html">Header HTML (blue bar + logo) – optional</label>
                    <textarea name="header_html" id="header_html"
                              placeholder="Custom header HTML. Leave blank for default."><?php
                        echo htmlspecialchars($form_header_html);
                    ?></textarea>
                    <div class="et-note">
                        This appears at the very top of the email (Broriser logo bar).  
                        You can keep the default or paste your own HTML with colors, background, and &lt;img&gt; logo.
                    </div>
                </div>

                <div class="et-row">
                    <label for="body_plain">Email Body (plain text, auto-converted to HTML) *</label>
                    <textarea name="body_plain" id="body_plain"
                              placeholder="Hello {{NAME}},&#10;&#10;Your Employee onboarding application has been {{STATUS}}.&#10;&#10;Regards,&#10;Broriser Team"><?php
                        echo htmlspecialchars($form_text_body);
                    ?></textarea>
                    <div class="et-note">
                        Write normal text here. Line breaks and paragraphs will be converted into HTML automatically.
                    </div>
                </div>

                <div class="et-row">
                    <label for="footer_html">Footer HTML (neon line + © text) – optional</label>
                    <textarea name="footer_html" id="footer_html"
                              placeholder="Custom footer HTML. Leave blank for default."><?php
                        echo htmlspecialchars($form_footer_html);
                    ?></textarea>
                    <div class="et-note">
                        This appears at the bottom of the email.  
                        You can keep the default neon strip and © text or paste custom HTML.
                    </div>
                </div>

                <div class="et-row">
                    <label for="html_body">Generated HTML (preview, read-only)</label>
                    <textarea id="html_body" readonly><?php echo htmlspecialchars($form_html_body); ?></textarea>
                    <div class="et-note">
                        This is the final HTML that will be sent to ZeptoMail when you use this template.
                    </div>
                </div>

                <div class="et-grid et-grid-2">
                    <div class="et-row">
                        <label for="default_cc">Default CC (comma-separated emails)</label>
                        <input type="text" name="default_cc" id="default_cc"
                               value="<?php echo htmlspecialchars($form_default_cc); ?>"
                               placeholder="manager@yourdomain.com,support@yourdomain.com">
                    </div>
                    <div class="et-row">
                        <label for="default_bcc">Default BCC (comma-separated emails)</label>
                        <input type="text" name="default_bcc" id="default_bcc"
                               value="<?php echo htmlspecialchars($form_default_bcc); ?>"
                               placeholder="audit@yourdomain.com">
                    </div>
                </div>

                <div class="et-row et-flags">
                    <label>
                        <input type="checkbox" name="track_clicks" value="1" <?php echo ($form_track_clicks ? 'checked' : ''); ?>>
                        Track Clicks
                    </label>
                    <label>
                        <input type="checkbox" name="track_opens" value="1" <?php echo ($form_track_opens ? 'checked' : ''); ?>>
                        Track Opens
                    </label>
                    <label>
                        <input type="checkbox" name="is_active" value="1" <?php echo ($form_is_active ? 'checked' : ''); ?>>
                        Active
                    </label>
                </div>

                <div class="et-row">
                    <button type="submit" name="save_template" value="1" class="et-btn">
                        <?php echo $editing ? 'Update Template' : 'Create Template'; ?>
                    </button>
                    <?php if ($editing): ?>
                        <a href="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" class="et-btn secondary">
                            + New Template
                        </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <div class="et-section">
            <div class="et-header-row">
                <h3>Live Email Preview</h3>
                <span class="et-note">
                    This shows how your Header + Body + Footer will look with current settings.
                </span>
            </div>
            <div id="et-live-preview" class="et-preview-box"></div>
        </div>

        <div class="et-section">
            <div class="et-header-row">
                <h3>Generated PHP Script</h3>
                <span class="et-note">
                    Copy &amp; paste this into any action to send email using this template via zeptomail_helper.php.
                </span>
            </div>
            <?php if ($editing && $snippet_code !== ''): ?>
                <textarea class="et-code" readonly><?php echo htmlspecialchars($snippet_code); ?></textarea>
            <?php else: ?>
                <p class="et-note">
                    After you save a template, open it from the left list to get a ready-to-use PHP script here.
                </p>
            <?php endif; ?>
        </div>

        <div class="et-section">
            <h3>Templates Overview</h3>
            <table class="et-table">
                <thead>
                <tr>
                    <th>ID</th>
                    <th>Name</th>
                    <th>Key</th>
                    <th>Subject</th>
                    <th>Status</th>
                    <th>Updated</th>
                    <th>Open</th>
                </tr>
                </thead>
                <tbody>
                <?php if (!empty($templates)): ?>
                    <?php foreach ($templates as $tpl): ?>
                        <tr>
                            <td><?php echo (int)$tpl['id']; ?></td>
                            <td><?php echo htmlspecialchars($tpl['template_name']); ?></td>
                            <td><?php echo htmlspecialchars($tpl['template_key']); ?></td>
                            <td><?php echo htmlspecialchars($tpl['subject']); ?></td>
                            <td><?php echo ((int)$tpl['is_active'] === 1 ? 'Active' : 'Inactive'); ?></td>
                            <td><?php echo htmlspecialchars($tpl['updated_at']); ?></td>
                            <td>
                                <a href="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>?edit=<?php echo (int)$tpl['id']; ?>"
                                   class="et-btn small">Edit</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr><td colspan="7">No templates created yet.</td></tr>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
    </main>
</div>

<script>
// Live preview on the page (colors, background, images).
(function () {
    function plainToHtml(text) {
        text = text.replace(/\r\n/g, "\n").replace(/\r/g, "\n");
        text = text.trim();
        if (!text) return "";
        var blocks = text.split(/\n{2,}/);
        var html = "";
        blocks.forEach(function (block) {
            block = block.trim();
            if (!block) return;
            block = block.replace(/\n/g, "<br>");
            html += '<p style="margin:0 0 10px 0;line-height:1.5;font-size:14px;color:#0f172a;">'
                 + block + '</p>';
        });
        return html;
    }

    var defaultHeader = <?php echo json_encode(et_default_header_html()); ?>;
    var defaultFooter = <?php echo json_encode(et_default_footer_html()); ?>;

    var headerField = document.getElementById('header_html');
    var bodyField   = document.getElementById('body_plain');
    var footerField = document.getElementById('footer_html');
    var previewBox  = document.getElementById('et-live-preview');

    function buildPreview() {
        if (!previewBox) return;
        var header = (headerField && headerField.value.trim()) || defaultHeader;
        var bodyPlain = (bodyField && bodyField.value) || "";
        var footer = (footerField && footerField.value.trim()) || defaultFooter;
        var bodyHtml = plainToHtml(bodyPlain);

        var full = ''
            + '<div style="font-family:Inter,Arial,sans-serif;background:#f6f7fb;padding:28px 0;color:#0f172a;">'
            + '<table role="presentation" cellpadding="0" cellspacing="0" align="center" width="680" '
            + 'style="background:#ffffff;border-radius:16px;overflow:hidden;border:1px solid #e8ecf3">';
        if (header) {
            full += '<tr><td style="padding:0">' + header + '</td></tr>';
        }
        full += '<tr><td style="padding:22px">' + (bodyHtml || '&nbsp;') + '</td></tr>';
        if (footer) {
            full += '<tr><td style="padding:0">' + footer + '</td></tr>';
        }
        full += '</table></div>';

        previewBox.innerHTML = full;
    }

    ['input', 'keyup', 'change'].forEach(function (evt) {
        if (headerField) headerField.addEventListener(evt, buildPreview);
        if (bodyField)   bodyField.addEventListener(evt, buildPreview);
        if (footerField) footerField.addEventListener(evt, buildPreview);
    });

    buildPreview();
})();
</script>
</body>
</html>
