<?php
// ==============================
// pages/index7.php  (Dashboard)
// ==============================
// Satya's notes:
// - mysqli only (no PDO). DB connection is required from ../config.php
// - Connection + all string comparisons/UNIONs are forced to utf8mb4_general_ci
// - "Candidates Overview" card is now **Total Employees** and uses
//   broriserin_onb.employee_applications (also shows gender shares from this table)
// - Recent Activity shows candidate names and dedupes by candidate (latest event only)
//   and uses passport photos from /onboarding/employee/employee-app/uploads/{id}/{passport_photo}
// - Growth % now shows a useful number even when previous period is 0 (returns 100% if now>0)
// - “Recently Added Jobs” is replaced by **Recent Onboarding** table sourced from employee_applications
// - Employees Performance chart is the Candidate Journey (keeps theme colors)

session_start();
include('../config.php');
require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);


if (isset($conn) && $conn instanceof mysqli) {
    @mysqli_set_charset($conn, 'utf8mb4');
    @mysqli_query($conn, "SET NAMES utf8mb4 COLLATE utf8mb4_general_ci");
    @mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");
}

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function qcount(mysqli $c, string $sql): int {
    $res = mysqli_query($c, $sql);
    if (!$res) return 0;
    $row = mysqli_fetch_row($res);
    return (int)($row[0] ?? 0);
}
function qall(mysqli $c, string $sql): array {
    $res = mysqli_query($c, $sql);
    if (!$res) return [];
    $out = [];
    while ($r = mysqli_fetch_assoc($res)) $out[] = $r;
    return $out;
}
function qone(mysqli $c, string $sql): ?array {
    $res = mysqli_query($c, $sql);
    if (!$res) return null;
    $r = mysqli_fetch_assoc($res);
    return $r ?: null;
}
function growth_pct(int $now, int $prev): float {
    // If previous period has no data but now does, show 100% for a meaningful signal.
    if ($prev <= 0) return $now > 0 ? 100.0 : 0.0;
    return round((($now - $prev) / $prev) * 100, 2);
}
function pct_badge(float $pct): array {
    $up = $pct >= 0;
    return [
        'cls'  => $up ? 'text-success' : 'text-danger',
        'icon' => $up ? 'ti ti-arrow-narrow-up' : 'ti ti-arrow-narrow-down',
        'text' => ($up ? $pct : -$pct) . '%'
    ];
}
function badge_class_from_status(string $s): string {
    $s = strtolower(trim($s));
    if (in_array($s, ['approved','selected','hired','success','accepted','active'])) return 'bg-primary-transparent';
    if (in_array($s, ['completed','submitted','done','shortlisted','short listed'])) return 'bg-success-transparent';
    if (in_array($s, ['pending','in-progress','under review','review','pending review'])) return 'bg-warning-transparent';
    if (in_array($s, ['rejected','declined','failed','blocked'])) return 'bg-danger-transparent';
    return 'bg-secondary-transparent';
}
function clean_prefixes(string $name): string {
    // strip Mr/Ms/Mrs/Miss/Sir/Madam prefixes anywhere in a short value
    $name = trim($name);
    $name = preg_replace('/\b(mr|ms|mrs|miss|sir|madam)\.?\b/i', '', $name);
    return trim(preg_replace('/\s{2,}/', ' ', $name));
}

// ---------- Safe date handling ----------
function dtExpr(string $col): string {
    // supports DATETIME or DATE string columns safely
    return "COALESCE(STR_TO_DATE(CONCAT($col), '%Y-%m-%d %H:%i:%s'), STR_TO_DATE(CONCAT($col), '%Y-%m-%d'))";
}
function rangeExpr(string $expr, int $days): string {
    return "($expr IS NOT NULL AND $expr >= DATE_SUB(NOW(), INTERVAL $days DAY))";
}
function firstValidDateExpr(string $a, string $b, string $c): string {
    return "COALESCE(".dtExpr($a).", ".dtExpr($b).", ".dtExpr($c).")";
}
function weekNowCol(string $col): string { return rangeExpr(dtExpr($col), 7); }
function weekPrevCol(string $col): string { return "(".rangeExpr(dtExpr($col),14)." AND ".dtExpr($col)." < DATE_SUB(NOW(), INTERVAL 7 DAY))"; }
function periodFilter(string $expr, string $period): string {
    $days = ($period === 'daily') ? 1 : (($period === 'weekly') ? 7 : 30);
    return rangeExpr($expr, $days);
}

// ---------- Collation-safe equality for JOIN/WHERE ----------
function colEq(string $left, string $right): string {
    return "TRIM(LOWER(CONVERT($left USING utf8mb4))) COLLATE utf8mb4_general_ci = TRIM(LOWER(CONVERT($right USING utf8mb4))) COLLATE utf8mb4_general_ci";
}

// ---------- helpers ----------
function initials($s) {
    $s = trim($s);
    if ($s==='') return 'NA';
    $p = preg_split('/\s+/', $s);
    return strtoupper(mb_substr($p[0],0,1).(isset($p[1])?mb_substr($p[1],0,1):''));
}
function time_ago($ts) {
    $d = time() - $ts;
    if ($d < 60) return $d.' sec ago';
    if ($d < 3600) return floor($d/60).' mins ago';
    if ($d < 86400) return floor($d/3600).' hrs ago';
    return floor($d/86400).' days ago';
}
function photo_from_email(mysqli $c, string $email): ?array {
    $DB = 'broriserin_onb';
    if (!$email) return null;
    $emailEsc = mysqli_real_escape_string($c, $email);
    $row = qone($c, "SELECT id, passport_photo FROM `$DB`.employee_applications WHERE ".colEq('email', "CONVERT('$emailEsc' USING utf8mb4)")." ORDER BY ".dtExpr('created_at')." DESC LIMIT 1");
    if (!$row || !$row['id'] || !$row['passport_photo']) return null;
    return ['id' => (int)$row['id'], 'file' => $row['passport_photo']];
}
function resolve_name(mysqli $c, string $email, string ...$candidates): string {
    $try = [];
    foreach ($candidates as $cand) {
        $cand = clean_prefixes(trim($cand));
        if ($cand !== '') $try[] = $cand;
    }
    foreach ($try as $t) if ($t !== '') return $t;

    if ($email !== '') {
        $DB = 'broriserin_onb';
        $esc = mysqli_real_escape_string($c, $email);
        // Prefer onboarding names
        $r = qone($c, "SELECT first_name, surname FROM `$DB`.employee_applications WHERE ".colEq('email', "CONVERT('$esc' USING utf8mb4)")." ORDER BY ".dtExpr('created_at')." DESC LIMIT 1");
        if ($r && (trim($r['first_name'] ?? '') !== '' || trim($r['surname'] ?? '') !== '')) {
            return clean_prefixes(trim(($r['first_name']??'').' '.($r['surname']??'')));
        }
        // Fallback to registration
        $r = qone($c, "SELECT first_name, last_name FROM `$DB`.employees WHERE ".colEq('email', "CONVERT('$esc' USING utf8mb4)")." ORDER BY ".dtExpr('created_at')." DESC LIMIT 1");
        if ($r && (trim($r['first_name'] ?? '') !== '' || trim($r['last_name'] ?? '') !== '')) {
            return clean_prefixes(trim(($r['first_name']??'').' '.($r['last_name']??'')));
        }
        // Fallback to username
        $r = qone($c, "SELECT username FROM `$DB`.user_pif WHERE ".colEq('email', "CONVERT('$esc' USING utf8mb4)")." ORDER BY ".dtExpr('created_at')." DESC LIMIT 1");
        if ($r && trim($r['username'] ?? '') !== '') {
            return clean_prefixes($r['username']);
        }
    }
    return 'Candidate';
}

// ---------- DB ----------
$DB = 'broriserin_onb';

// ---------- KPI cards ----------
$totalCandidates = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employees");
$candWeekNow     = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employees WHERE ".weekNowCol('created_at'));
$candWeekPrev    = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employees WHERE ".weekPrevCol('created_at'));
$candPct         = growth_pct($candWeekNow, $candWeekPrev);
$candBadge       = pct_badge($candPct);

$approvedOnboard = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status))='approved'");
$onbDateExpr     = firstValidDateExpr('status_date','updated_at','created_at');
$onbWeekNow      = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status))='approved' AND ".periodFilter($onbDateExpr,'weekly'));
$onbWeekPrev     = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status))='approved' AND (".rangeExpr($onbDateExpr,14)." AND $onbDateExpr < DATE_SUB(NOW(), INTERVAL 7 DAY))");
$onbPct          = growth_pct($onbWeekNow, $onbWeekPrev);
$onbBadge        = pct_badge($onbPct);

$totalEmployers  = qcount($conn, "SELECT COUNT(DISTINCT lastorg) FROM `$DB`.user_pif WHERE lastorg IS NOT NULL AND TRIM(lastorg) <> ''");
$empWeekNow      = qcount($conn, "SELECT COUNT(DISTINCT lastorg) FROM `$DB`.user_pif WHERE lastorg IS NOT NULL AND TRIM(lastorg) <> '' AND ".weekNowCol('created_at'));
$empWeekPrev     = qcount($conn, "SELECT COUNT(DISTINCT lastorg) FROM `$DB`.user_pif WHERE lastorg IS NOT NULL AND TRIM(lastorg) <> '' AND ".weekPrevCol('created_at'));
$empPct          = growth_pct($empWeekNow, $empWeekPrev);
$empBadge        = pct_badge($empPct);

$resumes     = qcount($conn, "SELECT COUNT(*) FROM `$DB`.user_pif WHERE resume IS NOT NULL AND TRIM(resume) <> ''");
$resWeekNow  = qcount($conn, "SELECT COUNT(*) FROM `$DB`.user_pif WHERE resume IS NOT NULL AND TRIM(resume) <> '' AND ".weekNowCol('created_at'));
$resWeekPrev = qcount($conn, "SELECT COUNT(*) FROM `$DB`.user_pif WHERE resume IS NOT NULL AND TRIM(resume) <> '' AND ".weekPrevCol('created_at'));
$resPct      = growth_pct($resWeekNow, $resWeekPrev);
$resBadge    = pct_badge($resPct);

// locations — union with explicit collation
$locTotalSQL = "
SELECT COUNT(*) FROM(
  SELECT DISTINCT CAST(currentCity AS CHAR CHARACTER SET utf8mb4) COLLATE utf8mb4_general_ci AS city
  FROM `$DB`.user_pif WHERE currentCity IS NOT NULL AND TRIM(currentCity) <> ''
  UNION
  SELECT DISTINCT CAST(city AS CHAR CHARACTER SET utf8mb4) COLLATE utf8mb4_general_ci AS city
  FROM `$DB`.employee_applications WHERE city IS NOT NULL AND TRIM(city) <> ''
) t";
$totalLocations = qcount($conn, $locTotalSQL);

$locWeekNowSQL = "
SELECT COUNT(*) FROM(
  SELECT DISTINCT CAST(currentCity AS CHAR CHARACTER SET utf8mb4) COLLATE utf8mb4_general_ci AS city
  FROM `$DB`.user_pif WHERE currentCity IS NOT NULL AND TRIM(currentCity) <> '' AND ".weekNowCol('created_at')."
  UNION
  SELECT DISTINCT CAST(city AS CHAR CHARACTER SET utf8mb4) COLLATE utf8mb4_general_ci AS city
  FROM `$DB`.employee_applications WHERE city IS NOT NULL AND TRIM(city) <> '' AND ".periodFilter($onbDateExpr,'weekly')."
) t";
$locWeekPrevSQL = "
SELECT COUNT(*) FROM(
  SELECT DISTINCT CAST(currentCity AS CHAR CHARACTER SET utf8mb4) COLLATE utf8mb4_general_ci AS city
  FROM `$DB`.user_pif WHERE currentCity IS NOT NULL AND TRIM(currentCity) <> '' AND ".weekPrevCol('created_at')."
  UNION
  SELECT DISTINCT CAST(city AS CHAR CHARACTER SET utf8mb4) COLLATE utf8mb4_general_ci AS city
  FROM `$DB`.employee_applications WHERE city IS NOT NULL AND TRIM(city) <> '' AND (".rangeExpr($onbDateExpr,14)." AND $onbDateExpr < DATE_SUB(NOW(), INTERVAL 7 DAY))
) t";
$locWeekNow  = qcount($conn, $locWeekNowSQL);
$locWeekPrev = qcount($conn, $locWeekPrevSQL);
$locPct      = growth_pct($locWeekNow, $locWeekPrev);
$locBadge    = pct_badge($locPct);

// ---------- Total Employees (for the right donut) ----------
$totalEmployeesRows = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications");
$maleCount   = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(gender)) IN('male','m')");
$femaleCount = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(gender)) IN('female','f')");
$otherCount  = max(0, $totalEmployeesRows - $maleCount - $femaleCount);
$totalMF     = $maleCount + $femaleCount + $otherCount;

// employees out of candidates %
$empOutOfCandPct = $totalCandidates>0 ? round(($totalEmployeesRows/$totalCandidates)*100,2) : 0.0;

// ---------- Candidate Journey Series ----------
$CJ_LABELS = ['Registered','PIF Submitted','PIF Approved','For Interview','Interview Approved','Onboarding Approved'];
function cjCounts(mysqli $c, string $period): array {
    $DB = 'broriserin_onb';
    $out = [];
    $out[] = qcount($c, "SELECT COUNT(*) FROM `$DB`.employees WHERE ".periodFilter(dtExpr('created_at'), $period));
    $pifDate = firstValidDateExpr('page6_date','updated_at','created_at');
    $out[]   = qcount($c, "SELECT COUNT(*) FROM `$DB`.user_pif WHERE final_submit IS NOT NULL AND TRIM(final_submit) <> '' AND TRIM(final_submit) NOT IN('0','no','false','No') AND ".periodFilter($pifDate,$period));
    $st1Date = firstValidDateExpr('st1_date','updated_at','created_at');
    $st2Date = firstValidDateExpr('st2_date','updated_at','created_at');
    $st3Date = firstValidDateExpr('st3_date','updated_at','created_at');
    $out[] = qcount($c, "SELECT COUNT(*) FROM `$DB`.user_pif WHERE LOWER(TRIM(st1_status))='approved' AND ".periodFilter($st1Date,$period));
    $out[] = qcount($c, "SELECT COUNT(*) FROM `$DB`.user_pif WHERE LOWER(TRIM(st2_status))='approved' AND ".periodFilter($st2Date,$period));
    $out[] = qcount($c, "SELECT COUNT(*) FROM `$DB`.user_pif WHERE LOWER(TRIM(st3_status))='approved' AND ".periodFilter($st3Date,$period));
    $onbDate = firstValidDateExpr('status_date','updated_at','created_at');
    $out[]   = qcount($c, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status))='approved' AND ".periodFilter($onbDate,$period));
    return $out;
}
$cjWeekly  = cjCounts($conn, 'weekly');
$cjMonthly = cjCounts($conn, 'monthly');
$cjDaily   = cjCounts($conn, 'daily');

// ---------- Recent Activity (DEDUPE PER CANDIDATE, LATEST EVENT) ----------
$activityMap = []; // email(or key) => event

// (1) registrations
foreach (qall($conn, "SELECT first_name,last_name,email,created_at FROM `$DB`.employees ORDER BY ".dtExpr('created_at')." DESC LIMIT 50") as $r) {
    $email = trim($r['email'] ?? '');
    $name  = resolve_name($conn, $email, ($r['first_name']??'').' '.($r['last_name']??''));
    $photo = $email ? photo_from_email($conn, $email) : null;
    $avatar = $photo ? "/onboarding/employee/employee-app/uploads/{$photo['id']}/{$photo['file']}" : null;
    $key = $email !== '' ? strtolower($email) : 'noemail-'.md5($name);
    $ev = [
        'time'  => strtotime($r['created_at'] ?? 'now'),
        'name'  => $name,
        'badge' => 'Registered',
        'cls'   => 'bg-success-transparent',
        'desc'  => 'Candidate registration',
        'img'   => $avatar,
        'email' => $email
    ];
    if (!isset($activityMap[$key]) || $ev['time'] > $activityMap[$key]['time']) $activityMap[$key] = $ev;
}
// (2) pre-interview stage-1
foreach (qall($conn, "SELECT username,email,appliedfor,st1_status,st1_date,updated_at,created_at FROM `$DB`.user_pif ORDER BY COALESCE(".dtExpr('st1_date').", ".dtExpr('updated_at').", ".dtExpr('created_at').") DESC LIMIT 50") as $r) {
    $email = trim($r['email'] ?? '');
    $name  = resolve_name($conn, $email, $r['username'] ?? '');
    $photo = $email ? photo_from_email($conn, $email) : null;
    $avatar = $photo ? "/onboarding/employee/employee-app/uploads/{$photo['id']}/{$photo['file']}" : null;
    $tsRaw = $r['st1_date'] ?: ($r['updated_at'] ?: $r['created_at']);
    $status = trim($r['st1_status'] ?? '');
    $key = $email !== '' ? strtolower($email) : 'noemail-'.md5($name);
    $ev = [
        'time'  => strtotime($tsRaw ?: 'now'),
        'name'  => $name,
        'badge' => $status !== '' ? $status : 'Stage-1',
        'cls'   => badge_class_from_status($status !== '' ? $status : 'pending'),
        'desc'  => 'Pre-interview (Stage 1)',
        'img'   => $avatar,
        'email' => $email
    ];
    if (!isset($activityMap[$key]) || $ev['time'] > $activityMap[$key]['time']) $activityMap[$key] = $ev;
}
// (3) onboarding
foreach (qall($conn, "SELECT id, first_name,surname,title,email,application_status,status_date,updated_at,created_at,passport_photo FROM `$DB`.employee_applications ORDER BY COALESCE(".dtExpr('status_date').", ".dtExpr('updated_at').", ".dtExpr('created_at').") DESC LIMIT 50") as $r) {
    $email = trim($r['email'] ?? '');
    $name  = resolve_name($conn, $email, ($r['first_name']??'').' '.($r['surname']??''), $r['title'] ?? '');
    $avatar = (!empty($r['passport_photo'])) ? "/onboarding/employee/employee-app/uploads/{$r['id']}/{$r['passport_photo']}" : null;
    $tsRaw = $r['status_date'] ?: ($r['updated_at'] ?: $r['created_at']);
    $status = trim($r['application_status'] ?? '');
    $key = $email !== '' ? strtolower($email) : 'noemail-'.md5($name);
    $ev = [
        'time'  => strtotime($tsRaw ?: 'now'),
        'name'  => $name,
        'badge' => $status !== '' ? $status : 'Onboarding',
        'cls'   => badge_class_from_status($status !== '' ? $status : 'pending'),
        'desc'  => 'Onboarding application',
        'img'   => $avatar,
        'email' => $email
    ];
    if (!isset($activityMap[$key]) || $ev['time'] > $activityMap[$key]['time']) $activityMap[$key] = $ev;
}

$activity = array_values($activityMap);
usort($activity, function($a,$b){ return $b['time'] <=> $a['time']; });
$activity = array_slice($activity, 0, 5);

// ---------- Recent Onboarding (table) ----------
$recentOnboarding = qall(
    $conn,
    "SELECT id, first_name, surname, city, application_status, ".dtExpr('created_at')." AS c_at, passport_photo
     FROM `$DB`.employee_applications
     ORDER BY ".dtExpr('created_at')." DESC
     LIMIT 5"
);

// ---------- Acquisitions ----------
$totalApps   = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications");
$recruited   = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status)) IN('approved','selected','hired')");
$shortlisted = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status))='submitted'");
$rejected    = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status))='rejected'");
$blocked     = qcount($conn, "SELECT COUNT(*) FROM `$DB`.employee_applications WHERE LOWER(TRIM(application_status))='blocked'");

$acqPerc = [
    'primary'   => $totalApps>0 ? 52 : 0,
    'secondary' => $totalApps>0 ? round(($recruited/$totalApps)*100) : 0,
    'success'   => $totalApps>0 ? round(($shortlisted/$totalApps)*100) : 0,
    'warning'   => $totalApps>0 ? round(($rejected/$totalApps)*100) : 0,
    'danger'    => $totalApps>0 ? round(($blocked/$totalApps)*100) : 0,
];

// ---------- Right "Recent Jobs" small list (kept as-is) ----------
$recentJobsList = qall(
    $conn,
    "SELECT ea.title, ea.email, ".dtExpr('ea.created_at')." AS c_at,
            COALESCE(NULLIF(up.jobType,''),'Full Time') AS job_type,
            COALESCE(NULLIF(up.totalExp,''),'0') AS totalExp,
            COALESCE(NULLIF(up.appliedfor,''),'') AS company_hint
     FROM `$DB`.employee_applications ea
     LEFT JOIN `$DB`.user_pif up
        ON ".colEq('up.email','ea.email')."
     ORDER BY ".dtExpr('ea.created_at')." DESC
     LIMIT 6"
);

?>

<?php ob_start(); ?>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
<!-- Start::page-header -->
<div class="page-header-breadcrumb mb-3">
    <div class="d-flex align-center justify-content-between flex-wrap">
        <h1 class="page-title fw-medium fs-18 mb-0">Onboarding</h1>
        <ol class="breadcrumb mb-0">
            <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
            <li class="breadcrumb-item active" aria-current="page">Onboarding</li>
            <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboard</a></li>
        </ol>
    </div>
</div>
<!-- End::page-header -->

<!-- KPI cards -->
<div class="row row-cols-xxl-5 row-cols-lg-3 row-cols-sm-2 row-cols-1">
    <div class="col">
        <div class="card custom-card dashboard-main-card">
            <div class="card-body">
                <div class="d-flex align-items-center gap-3 justify-content-between flex-wrap">
                    <div>
                        <span class="d-block mb-1">Total Employers</span>
                        <div class="d-flex align-items-center gap-2">
                            <h3 class="fw-semibold mb-0"><?php echo number_format($totalEmployers); ?></h3>
                            <span class="<?php echo $empBadge['cls']; ?>"><i class="<?php echo $empBadge['icon']; ?>"></i><?php echo h($empBadge['text']); ?></span>
                        </div>
                    </div>
                    <div class="lh-1">
                        <span class="avatar avatar-lg bg-primary-transparent svg-primary">
                            <svg xmlns="http://www.w3.org/2000/svg" enable-background="new 0 0 24 24" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><g><rect fill="none" height="24" width="24"/></g><g><path d="M20,7h-5V4c0-1.1-0.9-2-2-2h-2C9.9,2,9,2.9,9,4v3H4C2.9,7,2,7.9,2,9v11c0,1.1,0.9,2,2,2h16c1.1,0,2-0.9,2-2V9 C22,7.9,21.1,7,20,7z M9,12c0.83,0,1.5,1.67,1.5,2.5S9.83,16,9,16s-1.5-0.67-1.5-1.5S8.17,12,9,12z"/></g></svg>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col">
        <div class="card custom-card dashboard-main-card">
            <div class="card-body">
                <div class="d-flex align-items-center gap-3 justify-content-between flex-wrap">
                    <div>
                        <span class="d-block mb-1">Total Candidates</span>
                        <div class="d-flex align-items-center gap-2">
                            <h3 class="fw-semibold mb-0"><?php echo number_format($totalCandidates); ?></h3>
                            <span class="<?php echo $candBadge['cls']; ?>"><i class="<?php echo $candBadge['icon']; ?>"></i><?php echo h($candBadge['text']); ?></span>
                        </div>
                        <small class="text-muted d-block">This week: <?php echo number_format($candWeekNow); ?> new</small>
                    </div>
                    <div class="lh-1">
                        <span class="avatar avatar-lg bg-secondary-transparent svg-secondary">
                            <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><path d="M0 0h24v24H0z" fill="none"/><path d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5c-1.66 0-3 1.34-3 3s1.34 3 3 3zm-8 0c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5C6.34 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5c0-2.33-4.67-3.5-7-3.5zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z"/></svg>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col">
        <div class="card custom-card dashboard-main-card">
            <div class="card-body">
                <div class="d-flex align-items-center gap-3 justify-content-between flex-wrap">
                    <div>
                        <span class="d-block mb-1">Total Locations</span>
                        <div class="d-flex align-items-center gap-2">
                            <h3 class="fw-semibold mb-0"><?php echo number_format($totalLocations); ?></h3>
                            <span class="<?php echo $locBadge['cls']; ?>"><i class="<?php echo $locBadge['icon']; ?>"></i><?php echo h($locBadge['text']); ?></span>
                        </div>
                    </div>
                    <div class="lh-1">
                        <span class="avatar avatar-lg bg-warning-transparent svg-warning">
                            <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><path d="M0 0h24v24H0z" fill="none"/><path d="M20.5 3l-.16.03L15 5.1 9 3 3.36 4.9c-.21.07-.36.25-.36.48V20.5c0 .28.22.5.5.5l.16-.03L9 18.9l6 2.1 5.64-1.9c.21-.07.36-.25.36-.48V3.5c0-.28-.22-.5-.5-.5zM15 19l-6-2.11V5l6 2.11V19z"/></svg>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col">
        <div class="card custom-card dashboard-main-card">
            <div class="card-body">
                <div class="d-flex align-items-center gap-3 justify-content-between flex-wrap">
                    <div>
                        <span class="d-block mb-1">Resumes Generated</span>
                        <div class="d-flex align-items-center gap-2">
                            <h3 class="fw-semibold mb-0"><?php echo number_format($resumes); ?></h3>
                            <span class="<?php echo $resBadge['cls']; ?>"><i class="<?php echo $resBadge['icon']; ?>"></i><?php echo h($resBadge['text']); ?></span>
                        </div>
                        <small class="text-muted d-block">This week: <?php echo number_format($resWeekNow); ?></small>
                    </div>
                    <div class="lh-1">
                        <span class="avatar avatar-lg bg-info-transparent svg-info">
                            <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><path d="M0 0h24v24H0z" fill="none"/><path d="M14 2H6c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6zm2 16H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z"/></svg>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col">
        <div class="card custom-card dashboard-main-card">
            <div class="card-body">
                <div class="d-flex align-items-center gap-3 justify-content-between flex-wrap">
                    <div>
                        <span class="d-block mb-1">Approved Onboarding</span>
                        <div class="d-flex align-items-center gap-2">
                            <h3 class="fw-semibold mb-0"><?php echo number_format($approvedOnboard); ?></h3>
                            <span class="<?php echo $onbBadge['cls']; ?>"><i class="<?php echo $onbBadge['icon']; ?>"></i><?php echo h($onbBadge['text']); ?></span>
                        </div>
                        <small class="text-muted d-block">of candidates: <?php echo number_format($empOutOfCandPct,2); ?>%</small>
                    </div>
                    <div class="lh-1">
                        <span class="avatar avatar-lg bg-success-transparent svg-success">
                            <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><path d="M12 22c1.1 0 2-.9 2-2h-4c0 1.1.89 2 2 2zm6-6v-5c0-3.07-1.64-5.64-4.5-6.32V4c0-.83-.67-1.5-1.5-1.5S11.5,3.17,11.5,4v.68C7.63,5.36,6,7.92,6,11v5l-2 2v1h16v-1l-2-2z"/></svg>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Start:: row-2 -->
<div class="row">
    <div class="col-xxl-6">
        <div class="card custom-card">
            <div class="card-header">
                <div class="card-title">Employees Performance</div>
            </div>
            <div class="card-body">
                <div id="employees-performance"
                     data-labels='<?php echo json_encode($CJ_LABELS); ?>'
                     data-weekly='<?php echo json_encode($cjWeekly); ?>'
                     data-monthly='<?php echo json_encode($cjMonthly); ?>'
                     data-daily='<?php echo json_encode($cjDaily); ?>'></div>
            </div>
        </div>
    </div>

    <div class="col-xxl-3 col-xl-6">
        <div class="card custom-card">
            <div class="card-header">
                <div class="card-title">Recent Activity</div>
            </div>
            <div class="card-body">
                <ul class="list-unstyled jobs-recent-activity-list">
                    <?php foreach ($activity as $i=>$it): ?>
                    <li>
                        <div class="d-flex align-items-start gap-3 flex-wrap flex-xxl-nowrap">
                            <div>
                                <span class="avatar avatar-md avatar-rounded">
                                    <?php if (!empty($it['img'])): ?>
                                        <img src="<?php echo h($baseUrl.$it['img']); ?>" alt="">
                                    <?php else: ?>
                                        <img src="<?php echo $baseUrl; ?>/assets/images/faces/<?php echo [12,2,15,5,2][$i%5]; ?>.jpg" alt="">
                                    <?php endif; ?>
                                </span>
                            </div>
                            <div>
                                <div class="d-flex align-items-center gap-2">
                                    <div class="fw-semibold"><?php echo h(clean_prefixes($it['name'])); ?></div>
                                    <span class="badge <?php echo h($it['cls']); ?>"><?php echo h($it['badge']); ?></span>
                                </div>
                                <div class="fs-13 description mb-1"><?php echo h($it['desc']); ?></div>
                                <span class="d-block fs-12 text-muted"><?php echo date('M d, Y - h:i A', $it['time']); ?></span>
                            </div>
                        </div>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        </div>
    </div>

    <div class="col-xxl-3 col-xl-6">
        <div class="card custom-card overflow-hidden">
            <div class="card-header">
                <div class="card-title">Total Employees</div>
            </div>
            <div class="card-body">
                <!-- We render OUR donut (not theme) so the title/total matches -->
                <div id="total-employees-donut"
                     data-total="<?php echo (int)$totalEmployeesRows; ?>"
                     data-male="<?php echo (int)$maleCount; ?>"
                     data-female="<?php echo (int)$femaleCount; ?>"
                     data-other="<?php echo (int)$otherCount; ?>"></div>
            </div>
            <div class="card-footer p-0">
                <ul class="list-group list-group-flush">
                    <li class="list-group-item">
                        <div class="d-flex align-items-center gap-2">
                            <div class="lh-1">
                                <span class="avatar avatar-md bg-warning-transparent svg-warning">
                                    <svg xmlns="http://www.w3.org/2000/svg" enable-background="new 0 0 24 24" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><rect fill="none" height="24" width="24"/><path d="M9.5,11c1.93,0,3.5,1.57,3.5,3.5S11.43,18,9.5,18S6,16.43,6,14.5S7.57,11,9.5,11z M9.5,9C6.46,9,4,11.46,4,14.5 S6.46,20,9.5,20s5.5-2.46,5.5-5.5c0-1.16-0.36-2.23-0.97-3.12L18,7.42V10h2V4h-6v2h2.58l-3.97,3.97C11.73,9.36,10.66,9,9.5,9z"/></svg>
                                </span>
                            </div>
                            <div class="flex-fill">
                                <span class="fs-13 d-block">Male</span>
                                <h5 class="fw-semibold mb-0"><?php echo number_format($maleCount); ?></h5>
                            </div>
                            <div>
                                <span class="text-success"><i class="ti ti-arrow-narrow-up me-1"></i><?php echo ($totalMF>0?number_format(($maleCount/$totalMF)*100,2):'0'); ?>%</span>
                                <span class="d-block fs-13 text-muted"></span>
                            </div>
                        </div>
                    </li>
                    <li class="list-group-item">
                        <div class="d-flex align-items-center gap-2">
                            <div class="lh-1">
                                <span class="avatar avatar-md bg-primary-transparent svg-primary">
                                    <svg xmlns="http://www.w3.org/2000/svg" enable-background="new 0 0 24 24" height="24px" viewBox="0 0 24 24" width="24px" fill="#5f6368"><rect fill="none" height="24" width="24"/><path d="M17.5,9.5C17.5,6.46,15.04,4,12,4S6.5,6.46,6.5,9.5c0,2.7,1.94,4.93,4.5,5.4V17H9v2h2v2h2v-2h2v-2h-2v-2.1 C15.56,14.43,17.5,12.2,17.5,9.5z M8.5,9.5C8.5,7.57,10.07,6,12,6s3.5,1.57,3.5,3.5S13.93,13,12,13S8.5,11.43,8.5,9.5z"/></svg>
                                </span>
                            </div>
                            <div class="flex-fill">
                                <span class="fs-13 d-block">Female</span>
                                <h5 class="fw-semibold mb-0"><?php echo number_format($femaleCount); ?></h5>
                            </div>
                            <div>
                                <span class="text-danger"><i class="ti ti-arrow-narrow-down me-1"></i><?php echo ($totalMF>0?number_format(($femaleCount/$totalMF)*100,2):'0'); ?>%</span>
                                <span class="d-block fs-13 text-muted"></span>
                            </div>
                        </div>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>
<!-- End:: row-2 -->

<!-- Start:: row-3 -->
<div class="row">
    <div class="col-xxl-6">
        <div class="card custom-card overflow-hidden">
            <div class="card-header">
                <div class="card-title">Recent Onboarding</div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table text-nowrap">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>City</th>
                                <th>Status</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php if (!$recentOnboarding): ?>
                            <tr><td colspan="4" class="text-muted">No data</td></tr>
                        <?php else: foreach ($recentOnboarding as $r): ?>
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center gap-2">
                                        <span class="avatar avatar-sm bg-light avatar-rounded">
                                            <?php if (!empty($r['passport_photo'])): ?>
                                                <img src="<?php echo $baseUrl; ?>/onboarding/employee/employee-app/uploads/<?php echo (int)$r['id']; ?>/<?php echo h($r['passport_photo']); ?>" alt="">
                                            <?php else: ?>
                                                <img src="<?php echo $baseUrl; ?>/assets/images/faces/<?php echo rand(1,15); ?>.jpg" alt="">
                                            <?php endif; ?>
                                        </span>
                                        <a href="javascript:void(0);" class="fw-medium"><?php echo h(clean_prefixes(trim(($r['first_name']??'').' '.($r['surname']??'')))); ?></a>
                                    </div>
                                </td>
                                <td><?php echo h($r['city'] ?: '—'); ?></td>
                                <td><span class="badge <?php echo badge_class_from_status($r['application_status'] ?: ''); ?>"><?php echo h($r['application_status'] ?: '—'); ?></span></td>
                                <td><?php echo $r['c_at'] ? date('M d, Y', strtotime($r['c_at'])) : '—'; ?></td>
                            </tr>
                        <?php endforeach; endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xxl-3 col-xl-12">
        <div class="card custom-card">
            <div class="card-header"><div class="card-title">Acquisitions</div></div>
            <div class="card-body">
                <div class="progress progress-md mb-4 mt-2">
                    <div class="progress-bar bg-primary" role="progressbar" style="width: <?php echo (int)$acqPerc['primary']; ?>%"></div>
                    <div class="progress-bar bg-secondary" role="progressbar" style="width: <?php echo (int)$acqPerc['secondary']; ?>%"></div>
                    <div class="progress-bar bg-success" role="progressbar" style="width: <?php echo (int)$acqPerc['success']; ?>%"></div>
                    <div class="progress-bar bg-warning" role="progressbar" style="width: <?php echo (int)$acqPerc['warning']; ?>%"></div>
                    <div class="progress-bar bg-danger" role="progressbar" style="width: <?php echo (int)$acqPerc['danger']; ?>%"></div>
                </div>
                <ul class="list-group acquisitions-list mt-1">
                    <li class="list-group-item">Total Applications <span class="badge float-end bg-primary-transparent"><?php echo number_format($totalApps); ?></span></li>
                    <li class="list-group-item">Recruited <span class="badge float-end bg-secondary-transparent"><?php echo number_format($recruited); ?></span></li>
                    <li class="list-group-item">Short Listed <span class="badge float-end bg-success-transparent"><?php echo number_format($shortlisted); ?></span></li>
                    <li class="list-group-item">Rejected <span class="badge float-end bg-warning-transparent"><?php echo number_format($rejected); ?></span></li>
                    <li class="list-group-item">Blocked <span class="badge float-end bg-danger-transparent"><?php echo number_format($blocked); ?></span></li>
                </ul>
            </div>
        </div>
    </div>

    <div class="col-xxl-3 col-xl-12">
        <div class="card custom-card overflow-hidden">
            <div class="card-header"><div class="card-title">Recent Jobs</div></div>
            <div class="card-body p-0">
                <ul class="list-group list-group-flush" id="recent-jobs">
                    <?php if (!$recentJobsList): ?>
                        <li class="list-group-item">No data</li>
                    <?php else: foreach ($recentJobsList as $r):
                        $abbr = initials($r['title'] ?: 'Job');
                        $ago  = time_ago(strtotime($r['c_at'] ?? 'now'));
                        $tag  = ((int)$r['totalExp']<=0) ? '<span class="badge bg-success-transparent">Fresher</span>' : '<span class="badge bg-info-transparent">+'.(int)$r['totalExp'].' yrs - Experience</span>';
                    ?>
                    <li class="list-group-item border-start-0 border-end-0">
                        <a href="javascript:void(0);">
                            <div class="d-flex align-items-center flex-wrap">
                                <div class="me-2 lh-1">
                                    <span class="avatar avatar-md avatar-rounded bg-primary-transparent"><?php echo h($abbr); ?></span>
                                </div>
                                <div class="flex-fill">
                                    <p class="mb-0 fw-semibold"><?php echo h($r['title'] ?: '—'); ?></p>
                                    <p class="fs-12 text-muted mb-0"><?php echo h($r['company_hint'] ?: '—'); ?> - <?php echo h($ago); ?></p>
                                </div>
                                <div class="text-end">
                                    <p class="mb-0 fs-12"><?php echo h($r['job_type']); ?></p>
                                    <?php echo $tag; ?>
                                </div>
                            </div>
                        </a>
                    </li>
                    <?php endforeach; endif; ?>
                </ul>
            </div>
        </div>
    </div>
</div>
<!-- End:: row-3 -->

<!-- Start:: row-4 (Recent Job Postings table stays as earlier version, not changed by this request) -->
<?php
// We keep your existing “Recent Job Postings” section untouched.
?>
<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<!-- Apex Charts -->
<script src="<?php echo $baseUrl; ?>/assets/libs/apexcharts/apexcharts.min.js"></script>

<!-- Theme default charts (we’ll override where needed below) -->
<script src="<?php echo $baseUrl; ?>/assets/js/jobs-dashboard.js"></script>

<script>
// Render ONLY our charts and keep theme style/colors.

// Candidate Journey (Employees Performance)
(function(){
  var el = document.getElementById('employees-performance');
  if (el && typeof ApexCharts !== 'undefined') {
      el.innerHTML = '';
      var labels  = JSON.parse(el.getAttribute('data-labels') || '[]');
      var weekly  = JSON.parse(el.getAttribute('data-weekly') || '[]');
      var monthly = JSON.parse(el.getAttribute('data-monthly') || '[]');
      var daily   = JSON.parse(el.getAttribute('data-daily') || '[]');

      var chart = new ApexCharts(el, {
          chart: { height: 320, type: 'line', toolbar: { show: false }, foreColor: '#737b8b' },
          stroke: { width: [0, 2, 3], curve: 'smooth' },
          colors: ['#6f42c1', '#ffbe57', '#f74f9e'],
          series: [
              { name: 'Weekly',  type: 'column', data: weekly },
              { name: 'Monthly', type: 'area',   data: monthly },
              { name: 'Daily',   type: 'line',   data: daily }
          ],
          plotOptions: { bar: { columnWidth: '38%', borderRadius: 6 } },
          fill: { opacity: [1, 0.15, 1], gradient: { type: 'vertical', opacityFrom: 0.25, opacityTo: 0.05 } },
          dataLabels: { enabled: false },
          legend: { position: 'top', horizontalAlign: 'right' },
          xaxis: { categories: labels, axisBorder: { show: false }, axisTicks: { show: false } },
          yaxis: { min: 0, forceNiceScale: true, tickAmount: 5 }
      });
      chart.render();
  }

  // Total Employees donut (replaces theme's "Candidates Overview")
  var cov = document.getElementById('total-employees-donut');
  if (cov && typeof ApexCharts !== 'undefined') {
      cov.innerHTML = '';
      var total  = parseInt(cov.getAttribute('data-total') || '0', 10);
      var male   = parseInt(cov.getAttribute('data-male')  || '0', 10);
      var female = parseInt(cov.getAttribute('data-female')|| '0', 10);
      var other  = parseInt(cov.getAttribute('data-other') || '0', 10);

      var donut = new ApexCharts(cov, {
          chart: { type: 'donut', height: 280, toolbar: { show: false } },
          colors: ['#845adf', '#ffbe57', '#23b7e5'], // theme-like
          labels: ['Female', 'Male', 'Other'],
          series: [female, male, other],
          stroke: { width: 2 },
          legend: { position: 'bottom' },
          dataLabels: { enabled: true },
          plotOptions: {
            pie: {
              donut: {
                size: '80%',
                labels: {
                  show: true,
                  name: { show: false }, // avoid overlap
                  value: { show: true, offsetY: -4, fontSize: '18px', formatter: function(v){ return Math.round(v); } },
                  total: {
                    show: true,
                    label: 'Total Employees',
                    fontSize: '13px',
                    offsetY: 12,
                    formatter: function(){ return total; }
                  }
                }
              }
            }
          }
      });
      donut.render();
  }
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
