<?php
/* =============================================================================
 * File   : pages/leads_distribution_dashboard.php
 * Purpose: Dashboard for uploaded leads distribution:
 *          - Leads count grouped by Date, Lead Type, Project, EmpID
 *          - Date range filter
 *          - Lead Type filter dropdown
 *          - Search (project / empid / type)
 *          - Pagination & per-page selector
 *          - Export/Download CSV
 *          - Manage employee pools (Fresh / Resycled) from hr_dump:
 *              * Status='Active' only
 *              * Add / Remove empid for Fresh/Resycled pools
 *
 * Data source : broriserin_onb.leads
 *   - lead_type           (Fresh / Resycled)
 *   - project_id/name/title
 *   - Functional_Reporting (for Fresh)
 *   - empid               (for Resycled)
 *
 * Employee pool : broriserin_ic_attendance.hr_dump
 *   - For Fresh    → fresh_leads='yes' AND Status='Active'
 *   - For Resycled → recycle_leads='yes' AND Status='Active'
 * ============================================================================= */

date_default_timezone_set('Asia/Kolkata');
@session_start();

require_once __DIR__ . '/../config.php';

/* Charset / collation */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

if (!function_exists('h')) {
    function h($s){
        return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }
}

/* --------------------- Manage lead employee pools (POST) --------------------- */
$manage_msg = '';
$manage_err = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['manage_pool_action'])) {
    $action           = isset($_POST['manage_pool_action']) ? trim($_POST['manage_pool_action']) : '';
    $lead_type_manage = isset($_POST['lead_type_manage']) ? trim($_POST['lead_type_manage']) : '';
    $empid_manage     = isset($_POST['empid_manage']) ? trim($_POST['empid_manage']) : '';

    if (!isset($conn_ic) || !$conn_ic) {
        $manage_err = "IC Attendance database connection (\$conn_ic) not available.";
    } elseif ($lead_type_manage !== 'Fresh' && $lead_type_manage !== 'Resycled') {
        $manage_err = "Invalid Lead Type for employee pool management.";
    } elseif ($empid_manage === '') {
        $manage_err = "Please provide an EmpID.";
    } else {
        $empEsc = mysqli_real_escape_string($conn_ic, $empid_manage);
        $chkSql = "SELECT * FROM hr_dump WHERE empid='{$empEsc}' AND Status='Active' LIMIT 1";
        $chkRes = mysqli_query($conn_ic, $chkSql);

        if (!$chkRes || mysqli_num_rows($chkRes) === 0) {
            $manage_err = "EmpID not found or not Active in hr_dump.";
        } else {
            $col = ($lead_type_manage === 'Fresh') ? 'fresh_leads' : 'recycle_leads';

            if ($action === 'add') {
                $sql = "UPDATE hr_dump SET {$col}='yes' WHERE empid='{$empEsc}' AND Status='Active'";
            } elseif ($action === 'remove') {
                $sql = "UPDATE hr_dump SET {$col}='no' WHERE empid='{$empEsc}'";
            } else {
                $sql = '';
                $manage_err = "Invalid action.";
            }

            if ($sql !== '') {
                if (mysqli_query($conn_ic, $sql)) {
                    if ($action === 'add') {
                        $manage_msg = "EmpID {$empid_manage} added to {$lead_type_manage} pool.";
                    } elseif ($action === 'remove') {
                        $manage_msg = "EmpID {$empid_manage} removed from {$lead_type_manage} pool.";
                    }
                } else {
                    $manage_err = "Database error while updating pool: " . h(mysqli_error($conn_ic));
                }
            }
        }

        if ($chkRes) {
            mysqli_free_result($chkRes);
        }
    }
}

/* ----------------- Load current pools from hr_dump ----------------- */
$fresh_pool    = [];
$resycled_pool = [];
$pool_err      = '';

if (!isset($conn_ic) || !$conn_ic) {
    $pool_err = "IC Attendance database connection (\$conn_ic) not available.";
} else {
    /* Use SELECT * to avoid column-name mismatches; we only read empid + name safely */
    $fSql = "
        SELECT *
        FROM hr_dump
        WHERE Status='Active' AND fresh_leads='yes'
        ORDER BY empid
    ";
    $fRes = mysqli_query($conn_ic, $fSql);
    if ($fRes) {
        while ($row = mysqli_fetch_assoc($fRes)) {
            $name = '';
            if (isset($row['Employee_Name'])) {
                $name = $row['Employee_Name'];
            } elseif (isset($row['employee_name'])) {
                $name = $row['employee_name'];
            }
            $fresh_pool[] = [
                'empid' => $row['empid'],
                'name'  => $name
            ];
        }
        mysqli_free_result($fRes);
    }

    $rSql = "
        SELECT *
        FROM hr_dump
        WHERE Status='Active' AND recycle_leads='yes'
        ORDER BY empid
    ";
    $rRes = mysqli_query($conn_ic, $rSql);
    if ($rRes) {
        while ($row = mysqli_fetch_assoc($rRes)) {
            $name = '';
            if (isset($row['Employee_Name'])) {
                $name = $row['Employee_Name'];
            } elseif (isset($row['employee_name'])) {
                $name = $row['employee_name'];
            }
            $resycled_pool[] = [
                'empid' => $row['empid'],
                'name'  => $name
            ];
        }
        mysqli_free_result($rRes);
    }
}

/* ----------------- Filters: date range, search, lead_type, pagination ----------------- */
$today        = date('Y-m-d');
$firstOfMonth = date('Y-m-01');

$from   = isset($_GET['from']) && $_GET['from'] !== '' ? $_GET['from'] : $firstOfMonth;
$to     = isset($_GET['to'])   && $_GET['to']   !== '' ? $_GET['to']   : $today;
$search = isset($_GET['search']) ? trim($_GET['search']) : '';

$lead_type_filter = isset($_GET['lead_type_filter']) ? trim($_GET['lead_type_filter']) : '';
if ($lead_type_filter !== 'Fresh' && $lead_type_filter !== 'Resycled') {
    $lead_type_filter = ''; // All
}

$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
if ($page < 1) $page = 1;

$per_page          = isset($_GET['per_page']) ? (int)$_GET['per_page'] : 25;
$allowed_per_page  = [10,25,50,100,500];
if (!in_array($per_page, $allowed_per_page, true)) {
    $per_page = 25;
}
$offset = ($page - 1) * $per_page;

/* Build WHERE clause */
$where = "1=1";
if ($from !== '') {
    $from_esc = mysqli_real_escape_string($conn, $from);
    $where   .= " AND DATE(created_at) >= '{$from_esc}'";
}
if ($to !== '') {
    $to_esc = mysqli_real_escape_string($conn, $to);
    $where .= " AND DATE(created_at) <= '{$to_esc}'";
}
if ($lead_type_filter !== '') {
    $ltf_esc = mysqli_real_escape_string($conn, $lead_type_filter);
    $where  .= " AND lead_type = '{$ltf_esc}'";
}
if ($search !== '') {
    $s_esc = mysqli_real_escape_string($conn, '%' . $search . '%');
    $where .= " AND (
        project_name         LIKE '{$s_esc}' OR
        project_title        LIKE '{$s_esc}' OR
        project_city         LIKE '{$s_esc}' OR
        project_region       LIKE '{$s_esc}' OR
        project_location     LIKE '{$s_esc}' OR
        empid                LIKE '{$s_esc}' OR
        Functional_Reporting LIKE '{$s_esc}' OR
        assigned_user        LIKE '{$s_esc}' OR
        lead_type            LIKE '{$s_esc}'
    )";
}

/* ----------------- Export CSV (before normal rendering) ----------------- */
if (isset($_GET['export']) && $_GET['export'] === '1') {
    $exp_sql = "
        SELECT
            DATE(created_at) AS created_date,
            lead_type,
            project_id,
            project_name,
            project_title,
            IF(lead_type = 'Fresh', Functional_Reporting, empid) AS dist_empid,
            COUNT(*) AS lead_count
        FROM leads
        WHERE {$where}
        GROUP BY
            DATE(created_at),
            lead_type,
            project_id,
            project_name,
            project_title,
            dist_empid
        ORDER BY created_date DESC, project_name ASC, dist_empid ASC
    ";
    $exp_res = mysqli_query($conn, $exp_sql);

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=leads_distribution_' . date('Ymd_His') . '.csv');

    $out = fopen('php://output', 'w');
    fputcsv($out, ['Date','Lead Type','Project ID','Project Name','Project Title','EmpID','Leads Count']);

    if ($exp_res) {
        while ($row = mysqli_fetch_assoc($exp_res)) {
            fputcsv($out, [
                $row['created_date'],
                $row['lead_type'],
                $row['project_id'],
                $row['project_name'],
                $row['project_title'],
                $row['dist_empid'],
                $row['lead_count']
            ]);
        }
        mysqli_free_result($exp_res);
    }
    fclose($out);
    exit;
}

/* ----------------- Summary counts ----------------- */
$total_leads    = 0;
$fresh_leads    = 0;
$resycled_leads = 0;

$summary_sql = "SELECT lead_type, COUNT(*) AS cnt FROM leads WHERE {$where} GROUP BY lead_type";
$summary_res = mysqli_query($conn, $summary_sql);
if ($summary_res) {
    while ($row = mysqli_fetch_assoc($summary_res)) {
        $cnt = (int)$row['cnt'];
        $total_leads += $cnt;
        if ($row['lead_type'] === 'Fresh') {
            $fresh_leads = $cnt;
        } elseif ($row['lead_type'] === 'Resycled') {
            $resycled_leads = $cnt;
        }
    }
    mysqli_free_result($summary_res);
}

/* ----------------- Per-empid total counts (for header text) ----------------- */
$emp_totals_text = '';
$emp_total_sql = "
    SELECT
        IF(lead_type = 'Fresh', Functional_Reporting, empid) AS dist_empid,
        COUNT(*) AS total_leads
    FROM leads
    WHERE {$where}
    GROUP BY dist_empid
    HAVING dist_empid IS NOT NULL AND dist_empid <> ''
    ORDER BY dist_empid
";
$emp_total_res = mysqli_query($conn, $emp_total_sql);
$emp_parts = [];
if ($emp_total_res) {
    while ($row = mysqli_fetch_assoc($emp_total_res)) {
        $empid = $row['dist_empid'];
        $cnt   = (int)$row['total_leads'];
        $emp_parts[] = $empid . ' - ' . $cnt . ' Leads';
    }
    mysqli_free_result($emp_total_res);
}
if (!empty($emp_parts)) {
    $emp_totals_text = implode(', ', $emp_parts);
}

/* ----------------- Total groups for pagination ----------------- */
$total_groups = 0;
$count_sql = "
    SELECT COUNT(*) AS total_groups
    FROM (
        SELECT
            DATE(created_at) AS created_date,
            lead_type,
            project_id,
            project_name,
            project_title,
            IF(lead_type = 'Fresh', Functional_Reporting, empid) AS dist_empid
        FROM leads
        WHERE {$where}
        GROUP BY
            DATE(created_at),
            lead_type,
            project_id,
            project_name,
            project_title,
            dist_empid
    ) AS t
";
$count_res = mysqli_query($conn, $count_sql);
if ($count_res && $row = mysqli_fetch_assoc($count_res)) {
    $total_groups = (int)$row['total_groups'];
}
if ($count_res) {
    mysqli_free_result($count_res);
}

$total_pages = ($total_groups > 0) ? (int)ceil($total_groups / $per_page) : 1;
if ($page > $total_pages) $page = $total_pages;
$offset = ($page - 1) * $per_page;

/* ----------------- Data query ----------------- */
$data = [];
$data_sql = "
    SELECT
        DATE(created_at) AS created_date,
        lead_type,
        project_id,
        project_name,
        project_title,
        IF(lead_type = 'Fresh', Functional_Reporting, empid) AS dist_empid,
        COUNT(*) AS lead_count
    FROM leads
    WHERE {$where}
    GROUP BY
        DATE(created_at),
        lead_type,
        project_id,
        project_name,
        project_title,
        dist_empid
    ORDER BY created_date DESC, project_name ASC, dist_empid ASC
    LIMIT {$offset}, {$per_page}
";
$data_res = mysqli_query($conn, $data_sql);
if ($data_res) {
    while ($row = mysqli_fetch_assoc($data_res)) {
        $data[] = $row;
    }
    mysqli_free_result($data_res);
}

/* ============================== THEME SECTIONS ============================== */
ob_start(); ?>
<style>
.leads-dash .card-title { font-weight: 600; }
.leads-dash .muted { color:#6c757d; font-size:.9rem; }
.leads-dash .summary-card {
    border-radius: .75rem;
    border: 1px solid rgba(15,23,42,0.06);
}
.leads-dash .summary-card .value {
    font-size: 1.4rem;
    font-weight: 700;
}
.leads-dash .summary-card .label {
    font-size: .85rem;
    text-transform: uppercase;
    letter-spacing: .04em;
}
.leads-dash .filters .form-label {
    font-size: .85rem;
    font-weight: 500;
}
.leads-dash .pagination {
    margin-bottom: 0;
}
.leads-pools .table-sm td,
.leads-pools .table-sm th {
    padding-top: .35rem;
    padding-bottom: .35rem;
}
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Leads Distribution Dashboard</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Leads Distribution</li>
    </ol>
  </div>
</div>

<div class="row leads-dash">
  <!-- Filters + Manage Button -->
  <div class="col-12 mb-3">
    <div class="card custom-card">
      <div class="card-body filters">
        <div class="d-flex justify-content-between align-items-center mb-3 flex-wrap gap-2">
          <h6 class="mb-0">Filters</h6>
          <button type="button"
                  id="btnTogglePools"
                  class="btn btn-outline-secondary btn-sm">
            Manage Lead Employees
          </button>
        </div>
        <form method="get" class="row g-3 align-items-end">
          <div class="col-12 col-md-3">
            <label class="form-label">From Date</label>
            <input type="date" name="from" value="<?php echo h($from); ?>" class="form-control">
          </div>
          <div class="col-12 col-md-3">
            <label class="form-label">To Date</label>
            <input type="date" name="to" value="<?php echo h($to); ?>" class="form-control">
          </div>
          <div class="col-12 col-md-3">
            <label class="form-label">Lead Type</label>
            <select name="lead_type_filter" class="form-select">
              <option value="">All</option>
              <option value="Fresh"    <?php echo ($lead_type_filter === 'Fresh'    ? 'selected' : ''); ?>>Fresh</option>
              <option value="Resycled" <?php echo ($lead_type_filter === 'Resycled' ? 'selected' : ''); ?>>Resycled</option>
            </select>
          </div>
          <div class="col-12 col-md-3">
            <label class="form-label">Search (Project / EmpID / Type)</label>
            <input type="text"
                   name="search"
                   value="<?php echo h($search); ?>"
                   class="form-control"
                   placeholder="Project name, empid, Fresh/Resycled">
          </div>
          <div class="col-6 col-md-2">
            <label class="form-label">Per Page</label>
            <select name="per_page" class="form-select">
              <?php foreach ($allowed_per_page as $pp): ?>
                <option value="<?php echo $pp; ?>" <?php echo ($pp === $per_page ? 'selected' : ''); ?>>
                  <?php echo $pp; ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-6 col-md-2 d-grid">
            <button type="submit" class="btn btn-primary btn-wave">Apply</button>
          </div>
          <div class="col-12 col-md-2 d-grid">
            <?php
              $export_url = $_SERVER['PHP_SELF'] . '?' .
                'from=' . urlencode($from) .
                '&to=' . urlencode($to) .
                '&lead_type_filter=' . urlencode($lead_type_filter) .
                '&search=' . urlencode($search) .
                '&export=1';
            ?>
            <a href="<?php echo h($export_url); ?>" class="btn btn-success btn-wave">
              Export CSV
            </a>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Lead Employee Pools (hidden by default, toggled by button) -->
  <div class="col-12 mb-3">
    <div id="leadPools" class="card custom-card leads-pools d-none">
      <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
        <div class="card-title mb-0">Lead Employee Pools (Status = Active)</div>
        <span class="muted small">Source: broriserin_ic_attendance.hr_dump</span>
      </div>
      <div class="card-body">
        <?php if ($pool_err): ?>
          <div class="alert alert-warning"><?php echo h($pool_err); ?></div>
        <?php endif; ?>

        <?php if ($manage_err): ?>
          <div class="alert alert-danger"><?php echo h($manage_err); ?></div>
        <?php elseif ($manage_msg): ?>
          <div class="alert alert-success"><?php echo h($manage_msg); ?></div>
        <?php endif; ?>

        <div class="row g-4">
          <!-- Fresh Pool -->
          <div class="col-md-6">
            <h6 class="mb-2">Fresh Leads Pool</h6>
            <div class="table-responsive">
              <table class="table table-sm table-hover align-middle">
                <thead class="table-light">
                  <tr>
                    <th>EmpID</th>
                    <th>Employee Name</th>
                    <th style="width:80px;">Action</th>
                  </tr>
                </thead>
                <tbody>
                  <?php if (empty($fresh_pool)): ?>
                    <tr>
                      <td colspan="3" class="text-center muted">No employees configured for Fresh leads.</td>
                    </tr>
                  <?php else: ?>
                    <?php foreach ($fresh_pool as $row): ?>
                      <tr>
                        <td><?php echo h($row['empid']); ?></td>
                        <td><?php echo h($row['name']); ?></td>
                        <td>
                          <form method="post" class="d-inline">
                            <input type="hidden" name="manage_pool_action" value="remove">
                            <input type="hidden" name="lead_type_manage" value="Fresh">
                            <input type="hidden" name="empid_manage" value="<?php echo h($row['empid']); ?>">
                            <button type="submit" class="btn btn-outline-danger btn-xs btn-sm">
                              Remove
                            </button>
                          </form>
                        </td>
                      </tr>
                    <?php endforeach; ?>
                  <?php endif; ?>
                </tbody>
              </table>
            </div>
            <form method="post" class="row g-2 mt-2">
              <input type="hidden" name="manage_pool_action" value="add">
              <input type="hidden" name="lead_type_manage" value="Fresh">
              <div class="col-8">
                <input type="text"
                       name="empid_manage"
                       class="form-control form-control-sm"
                       placeholder="Add EmpID (Active)">
              </div>
              <div class="col-4 d-grid">
                <button type="submit" class="btn btn-outline-primary btn-sm">
                  Add to Fresh
                </button>
              </div>
            </form>
          </div>

          <!-- Resycled Pool -->
          <div class="col-md-6">
            <h6 class="mb-2">Resycled Leads Pool</h6>
            <div class="table-responsive">
              <table class="table table-sm table-hover align-middle">
                <thead class="table-light">
                  <tr>
                    <th>EmpID</th>
                    <th>Employee Name</th>
                    <th style="width:80px;">Action</th>
                  </tr>
                </thead>
                <tbody>
                  <?php if (empty($resycled_pool)): ?>
                    <tr>
                      <td colspan="3" class="text-center muted">No employees configured for Resycled leads.</td>
                    </tr>
                  <?php else: ?>
                    <?php foreach ($resycled_pool as $row): ?>
                      <tr>
                        <td><?php echo h($row['empid']); ?></td>
                        <td><?php echo h($row['name']); ?></td>
                        <td>
                          <form method="post" class="d-inline">
                            <input type="hidden" name="manage_pool_action" value="remove">
                            <input type="hidden" name="lead_type_manage" value="Resycled">
                            <input type="hidden" name="empid_manage" value="<?php echo h($row['empid']); ?>">
                            <button type="submit" class="btn btn-outline-danger btn-xs btn-sm">
                              Remove
                            </button>
                          </form>
                        </td>
                      </tr>
                    <?php endforeach; ?>
                  <?php endif; ?>
                </tbody>
              </table>
            </div>
            <form method="post" class="row g-2 mt-2">
              <input type="hidden" name="manage_pool_action" value="add">
              <input type="hidden" name="lead_type_manage" value="Resycled">
              <div class="col-8">
                <input type="text"
                       name="empid_manage"
                       class="form-control form-control-sm"
                       placeholder="Add EmpID (Active)">
              </div>
              <div class="col-4 d-grid">
                <button type="submit" class="btn btn-outline-primary btn-sm">
                  Add to Resycled
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  </div>

  <!-- Summary Cards -->
  <div class="col-md-4 col-sm-12 mb-3">
    <div class="card summary-card h-100">
      <div class="card-body">
        <div class="label text-muted mb-1">Total Leads</div>
        <div class="value mb-1"><?php echo (int)$total_leads; ?></div>
        <div class="muted">Filtered range: <?php echo h($from); ?> to <?php echo h($to); ?></div>
      </div>
    </div>
  </div>
  <div class="col-md-4 col-sm-12 mb-3">
    <div class="card summary-card h-100">
      <div class="card-body">
        <div class="label text-muted mb-1">Fresh Leads</div>
        <div class="value mb-1"><?php echo (int)$fresh_leads; ?></div>
        <div class="muted">Assigned via <code>Functional_Reporting</code></div>
      </div>
    </div>
  </div>
  <div class="col-md-4 col-sm-12 mb-3">
    <div class="card summary-card h-100">
      <div class="card-body">
        <div class="label text-muted mb-1">Resycled Leads</div>
        <div class="value mb-1"><?php echo (int)$resycled_leads; ?></div>
        <div class="muted">Assigned via <code>empid / assigned_user</code></div>
      </div>
    </div>
  </div>

  <!-- Distribution table -->
  <div class="col-12">
    <div class="card custom-card">
      <div class="card-header d-flex align-items-start justify-content-between flex-wrap gap-2">
        <div class="card-title">
          Leads Distribution (Date × Type × Project × EmpID)
          <?php if ($emp_totals_text !== ''): ?>
            <div class="muted small mt-1">
              <?php echo h($emp_totals_text); ?>
            </div>
          <?php endif; ?>
        </div>
        <div class="muted">
          Showing <?php echo count($data); ?> group(s) of <?php echo (int)$total_groups; ?>
        </div>
      </div>
      <div class="card-body">
        <?php if (empty($data)): ?>
          <div class="alert alert-info mb-0">
            No leads found for the selected filters.
          </div>
        <?php else: ?>
          <div class="table-responsive">
            <table class="table table-sm table-hover align-middle">
              <thead class="table-light">
                <tr>
                  <th>Date</th>
                  <th>Lead Type</th>
                  <th>Project</th>
                  <th>EmpID</th>
                  <th class="text-end">Leads Count</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($data as $row): ?>
                  <tr>
                    <td><?php echo h($row['created_date']); ?></td>
                    <td><?php echo h($row['lead_type']); ?></td>
                    <td>
                      <div class="fw-semibold"><?php echo h($row['project_name']); ?></div>
                      <div class="muted small">
                        ID: <?php echo h($row['project_id']); ?> &bull;
                        <?php echo h($row['project_title']); ?>
                      </div>
                    </td>
                    <td><?php echo h($row['dist_empid']); ?></td>
                    <td class="text-end fw-semibold"><?php echo (int)$row['lead_count']; ?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>

          <!-- Pagination -->
          <div class="d-flex justify-content-between align-items-center mt-3 flex-wrap gap-2">
            <div class="muted">
              Page <?php echo (int)$page; ?> of <?php echo (int)$total_pages; ?>
            </div>
            <nav>
              <ul class="pagination pagination-sm mb-0">
                <?php
                  $base_url = $_SERVER['PHP_SELF'] . '?' .
                    'from=' . urlencode($from) .
                    '&to=' . urlencode($to) .
                    '&lead_type_filter=' . urlencode($lead_type_filter) .
                    '&search=' . urlencode($search) .
                    '&per_page=' . urlencode($per_page) .
                    '&page=';
                ?>
                <li class="page-item <?php echo ($page <= 1 ? 'disabled' : ''); ?>">
                  <a class="page-link" href="<?php echo ($page <= 1 ? '#' : $base_url . ($page-1)); ?>">&laquo;</a>
                </li>
                <?php
                  $start_p = max(1, $page - 2);
                  $end_p   = min($total_pages, $page + 2);
                  for ($p = $start_p; $p <= $end_p; $p++):
                ?>
                  <li class="page-item <?php echo ($p === $page ? 'active' : ''); ?>">
                    <a class="page-link" href="<?php echo $base_url . $p; ?>"><?php echo $p; ?></a>
                  </li>
                <?php endfor; ?>
                <li class="page-item <?php echo ($page >= $total_pages ? 'disabled' : ''); ?>">
                  <a class="page-link" href="<?php echo ($page >= $total_pages ? '#' : $base_url . ($page+1)); ?>">&raquo;</a>
                </li>
              </ul>
            </nav>
          </div>
        <?php endif; ?>
      </div>
    </div>
  </div>
</div>
<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
document.addEventListener('DOMContentLoaded', function () {
    var btn = document.getElementById('btnTogglePools');
    var box = document.getElementById('leadPools');
    if (btn && box) {
        btn.addEventListener('click', function () {
            box.classList.toggle('d-none');
        });
    }
});
</script>
<?php $scripts = ob_get_clean(); ?>

<?php
include 'layouts/base.php';
