<?php
/* ============================================================================
 * File: leads_employee_summary.php
 * Purpose:
 *   - Employee-wise summary of Leads & Activities
 *   - Shows, for each Assigned User (employee) based on LEAD DISTRIBUTION DATE
 *       • Total leads allotted (by created_at date → Today / Yesterday / All)
 *       • Leads with NO activities (call/mail/meeting/task, ever)
 *       • Leads WITH activity (ever) (GREEN)
 *       • Total activities + per-type counts
 *   - Default sort by Leads With Activity (top → low)
 *   - Click employee row → show all leads for that employee (detail table)
 *   - Click lead row in detail table → popup of activities (Title, Status, Remarks, etc.)
 *   - Export (CSV) for summary + per-employee detail
 *   - Charts (Bar, Doughnut, Lead Status wise) on top in a single row
 *   - Date filter: Today / Yesterday / All (default = Today) on LEAD created_at
 * Stack : Core PHP + MySQLi (NO PDO). DB connection in ../config.php
 * Rules : utf8mb4_general_ci
 * ========================================================================== */

date_default_timezone_set('Asia/Kolkata');
@session_start();

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/includes/check-permission.php';

$currentPage = basename(__FILE__);
if (function_exists('checkPageAccessOrRedirect')) {
    checkPageAccessOrRedirect($currentPage, $conn_sa);
}

/* ---- DB handle (force ONB schema like leads.php) ---- */
$DB = null;
if (isset($conn) && $conn instanceof mysqli) {
    $DB = $conn;
} elseif (isset($conn_crm) && $conn_crm instanceof mysqli) {
    $DB = $conn_crm;
} elseif (isset($DB) && $DB instanceof mysqli) {
    $DB = $DB;
}

if (!($DB instanceof mysqli)) {
    http_response_code(500);
    exit('DB connection missing.');
}

/* Use CRM DB */
@mysqli_query($DB, "USE `broriserin_onb`");
@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* ---- Helpers ---- */
function h($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}
function q_all($db, $sql) {
    $out = [];
    if ($rs = @mysqli_query($db, $sql)) {
        while ($r = mysqli_fetch_assoc($rs)) {
            $out[] = $r;
        }
        @mysqli_free_result($rs);
    }
    return $out;
}
function col_exists($conn, $table, $col) {
    $t = mysqli_real_escape_string($conn, $table);
    $c = mysqli_real_escape_string($conn, $col);
    $q = mysqli_query(
        $conn,
        "SELECT 1
         FROM INFORMATION_SCHEMA.COLUMNS
         WHERE TABLE_SCHEMA = DATABASE()
           AND TABLE_NAME='{$t}'
           AND COLUMN_NAME='{$c}'"
    );
    return ($q && mysqli_num_rows($q) > 0);
}

/* ---------------- RBAC: re-use same permissions as leads.php ---------------- */
$CURRENT_USER_ID = (int)($_SESSION['user_id1'] ?? 0);

function rbac_ensure_schema_emp_summary($conn) {
    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `roles` (
      `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
      `name` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `description` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      UNIQUE KEY `uniq_role_name` (`name`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `permissions` (
      `code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      `label` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `group_name` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
      PRIMARY KEY (`code`)
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `role_permissions` (
      `role_id` INT UNSIGNED NOT NULL,
      `perm_code` VARCHAR(100) COLLATE utf8mb4_general_ci NOT NULL,
      PRIMARY KEY (`role_id`,`perm_code`),
      KEY `idx_perm_code` (`perm_code`),
      CONSTRAINT `fk_role_permissions_role`
        FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

    @mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `user_roles` (
      `user_id` INT NOT NULL,
      `role_id` INT UNSIGNED NOT NULL,
      PRIMARY KEY (`user_id`,`role_id`),
      KEY `idx_role_id` (`role_id`),
      CONSTRAINT `fk_user_roles_role`
        FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");
}
function rbac_seed_permissions_emp_summary($conn) {
    $perms = [
        ['leads.create',    'Create leads',          'Leads'],
        ['leads.export',    'Export leads (CSV)',    'Leads'],
        ['activities.log',  'Log activities',        'Activities'],
        ['roles.manage',    'Manage roles',          'Admin'],
    ];
    foreach ($perms as $p) {
        @mysqli_query(
            $conn,
            "INSERT IGNORE INTO `permissions` (`code`,`label`,`group_name`)
             VALUES ('" . mysqli_real_escape_string($conn, $p[0]) . "',
                     '" . mysqli_real_escape_string($conn, $p[1]) . "',
                     '" . mysqli_real_escape_string($conn, $p[2]) . "')"
        );
    }
}
function rbac_has_emp_summary($conn, $uid, $code) {
    if (!$uid) return true;
    $uid  = (int)$uid;
    $code = mysqli_real_escape_string($conn, $code);
    $q = mysqli_query(
        $conn,
        "SELECT 1
         FROM user_roles ur
         JOIN role_permissions rp ON rp.role_id = ur.role_id
         WHERE ur.user_id = {$uid}
           AND rp.perm_code = '{$code}'
         LIMIT 1"
    );
    return ($q && mysqli_num_rows($q) > 0);
}

rbac_ensure_schema_emp_summary($DB);
rbac_seed_permissions_emp_summary($DB);

$CAN_EXPORT_LEADS = rbac_has_emp_summary($DB, $CURRENT_USER_ID, 'leads.export');

/* -------------------------- Ensure base tables exist ------------------------ */
@mysqli_query($DB, "CREATE TABLE IF NOT EXISTS `leads` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `contact_name` VARCHAR(150) COLLATE utf8mb4_general_ci NOT NULL,
  `email` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `phone_number` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_status` ENUM('New','Follow-up','Closed','Contacted','Disqualified','Qualified') COLLATE utf8mb4_general_ci DEFAULT 'New',
  `company` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_source` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `tags` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `pp_status` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `avatar` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

@mysqli_query($DB, "CREATE TABLE IF NOT EXISTS `crm_activities` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_id` INT UNSIGNED NOT NULL,
  `activity_type` ENUM('call','meeting','task','email') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'task',
  `status` ENUM('Planned','Held','Not Held','Completed','Cancelled') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'Planned',
  `direction` ENUM('Inbound','Outbound','N/A') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'N/A',
  `title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `description` TEXT COLLATE utf8mb4_general_ci,
  `date_start` DATETIME DEFAULT NULL,
  `date_end` DATETIME DEFAULT NULL,
  `duration_minutes` INT DEFAULT NULL,
  `assigned_user` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_by` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `remind` TINYINT(1) NOT NULL DEFAULT 0,
  `reminder_at` DATETIME DEFAULT NULL,
  `saved_from` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `location_id` INT DEFAULT NULL,
  `budget_id` INT DEFAULT NULL,
  `project_id` INT DEFAULT NULL,
  `lead_type` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY `idx_lead_id` (`lead_id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

/* -------------------------- Date filter (Today/Yesterday/All) --------------- */
/* NOTE: Date filter is on LEAD created_at (distribution date). Activities are lifetime. */
$allowedDateModes = ['today', 'yesterday', 'all'];
$dateMode = isset($_GET['date_mode']) ? strtolower(trim($_GET['date_mode'])) : 'today';
if (!in_array($dateMode, $allowedDateModes, true)) {
    $dateMode = 'today';
}

$todayHuman     = date('d M Y');
$yesterdayHuman = date('d M Y', strtotime('-1 day'));

$selectedDate = null;
if ($dateMode === 'today') {
    $selectedDate = date('Y-m-d');
} elseif ($dateMode === 'yesterday') {
    $selectedDate = date('Y-m-d', strtotime('-1 day'));
}

$leadDateClause = '1=1';

if ($selectedDate !== null) {
    $safeDate = mysqli_real_escape_string($DB, $selectedDate);
    $leadDateClause = "DATE(l.`created_at`) = '{$safeDate}'";
}

if ($dateMode === 'today') {
    $dateLabel = 'Today (' . $todayHuman . ')';
} elseif ($dateMode === 'yesterday') {
    $dateLabel = 'Yesterday (' . $yesterdayHuman . ')';
} else {
    $dateLabel = 'All Dates';
}

/* Filter based purely on distribution date (created_at) */
$filterClause = $leadDateClause;

/* ------------------------- AJAX: lead activities popup ---------------------- */
if (isset($_POST['ajax']) && $_POST['ajax'] === 'lead_activities') {
    header('Content-Type: application/json; charset=utf-8');

    $leadId = isset($_POST['lead_id']) ? (int)$_POST['lead_id'] : 0;
    if ($leadId <= 0) {
        echo json_encode(['ok' => false, 'msg' => 'Invalid lead id.']);
        exit;
    }

    // Lifetime activities for that lead
    $sql = "
        SELECT
            id,
            activity_type,
            status,
            direction,
            title,
            description,
            date_start,
            date_end,
            created_at
        FROM crm_activities
        WHERE lead_id = {$leadId}
        ORDER BY COALESCE(date_start, date_end, created_at) DESC, id DESC
    ";
    $rows = q_all($DB, $sql);

    echo json_encode(['ok' => true, 'activities' => $rows]);
    exit;
}

/* ----------------------------- EXPORT HANDLERS ----------------------------- */
if (isset($_GET['export']) && $_GET['export'] === 'summary') {
    if (!$CAN_EXPORT_LEADS) {
        http_response_code(403);
        exit('Not authorized to export.');
    }

    $sql = "
        SELECT
            TRIM(COALESCE(l.assigned_user,'')) AS assigned_user,
            COUNT(DISTINCT l.id) AS total_leads,
            SUM(CASE WHEN a.lead_id IS NULL THEN 1 ELSE 0 END) AS leads_no_activity,
            SUM(CASE WHEN a.lead_id IS NOT NULL THEN 1 ELSE 0 END) AS leads_with_activity,
            COALESCE(SUM(a.total_activities),0) AS total_activities,
            COALESCE(SUM(a.cnt_call),0)    AS total_calls,
            COALESCE(SUM(a.cnt_email),0)   AS total_emails,
            COALESCE(SUM(a.cnt_task),0)    AS total_tasks,
            COALESCE(SUM(a.cnt_meeting),0) AS total_meetings
        FROM leads l
        LEFT JOIN (
            SELECT
                lead_id,
                COUNT(*) AS total_activities,
                SUM(activity_type='call')    AS cnt_call,
                SUM(activity_type='email')   AS cnt_email,
                SUM(activity_type='task')    AS cnt_task,
                SUM(activity_type='meeting') AS cnt_meeting
            FROM crm_activities
            GROUP BY lead_id
        ) a ON a.lead_id = l.id
        WHERE {$filterClause}
        GROUP BY assigned_user
        ORDER BY leads_with_activity DESC, total_leads DESC, assigned_user ASC
    ";
    $rows = q_all($DB, $sql);

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=employee_leads_summary_' . date('Ymd_His') . '.csv');

    $out = fopen('php://output', 'w');
    fputcsv($out, [
        'Employee / Assigned User',
        'Total Leads (by Created Date)',
        'Leads With Activity',
        'Leads With No Activity',
        'Total Activities',
        'Total Calls',
        'Total Emails',
        'Total Tasks',
        'Total Meetings'
    ]);
    foreach ($rows as $r) {
        fputcsv($out, [
            $r['assigned_user'] !== '' ? $r['assigned_user'] : 'Unassigned',
            (int)$r['total_leads'],
            (int)$r['leads_with_activity'],
            (int)$r['leads_no_activity'],
            (int)$r['total_activities'],
            (int)$r['total_calls'],
            (int)$r['total_emails'],
            (int)$r['total_tasks'],
            (int)$r['total_meetings'],
        ]);
    }
    fclose($out);
    exit;
}

if (isset($_GET['export']) && $_GET['export'] === 'detail') {
    if (!$CAN_EXPORT_LEADS) {
        http_response_code(403);
        exit('Not authorized to export.');
    }
    $empKey = isset($_GET['emp']) ? trim($_GET['emp']) : (isset($_GET['empid']) ? trim($_GET['empid']) : '');
    if ($empKey === '') {
        http_response_code(400);
        exit('Missing employee key.');
    }

    $isUnassigned = ($empKey === '__UNASSIGNED__');
    if ($isUnassigned) {
        $empFilter = "(l.`assigned_user` IS NULL OR TRIM(l.`assigned_user`)='')";
    } else {
        $safeEmp = mysqli_real_escape_string($DB, $empKey);
        $empFilter = "TRIM(l.`assigned_user`) = '{$safeEmp}'";
    }

    $sql = "
        SELECT
          l.`id`,
          l.`contact_name`,
          l.`email`,
          l.`phone_number`,
          l.`lead_status`,
          l.`pp_status`,
          l.`company`,
          l.`lead_source`,
          l.`tags`,
          l.`assigned_user`,
          l.`created_at`,
          l.`updated_at`,
          (SELECT MAX(COALESCE(ca.`date_start`, ca.`date_end`, ca.`created_at`))
             FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id`) AS last_activity_at,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='call')    AS cnt_call,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='email')   AS cnt_email,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='task')    AS cnt_task,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='meeting') AS cnt_meeting
        FROM `leads` l
        WHERE {$filterClause} AND {$empFilter}
        ORDER BY l.`updated_at` DESC, l.`id` DESC
    ";
    $rows = q_all($DB, $sql);

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=employee_leads_detail_' . date('Ymd_His') . '.csv');

    $out = fopen('php://output', 'w');
    fputcsv($out, [
        'Lead ID','Contact Name','Email','Phone',
        'Lead Status (from PP Status)',
        'Company','Lead Source','Tags',
        'Assigned User','Created At',
        'Activity Date (Recent)',
        'Call Count','Email Count','Task Count','Meeting Count'
    ]);
    foreach ($rows as $r) {
        $leadStatusFromPP = $r['pp_status'];

        fputcsv($out, [
            (int)$r['id'],
            $r['contact_name'],
            $r['email'],
            $r['phone_number'],
            $leadStatusFromPP,
            $r['company'],
            $r['lead_source'],
            $r['tags'],
            $r['assigned_user'],
            $r['created_at'],
            $r['last_activity_at'],
            (int)$r['cnt_call'],
            (int)$r['cnt_email'],
            (int)$r['cnt_task'],
            (int)$r['cnt_meeting'],
        ]);
    }
    fclose($out);
    exit;
}

/* ---------------------- Fetch summary data for the page --------------------- */
$perPageOptions = [10, 25, 50, 500];
$perPage = (int)($_GET['per_page'] ?? 25);
if (!in_array($perPage, $perPageOptions, true)) {
    $perPage = 25;
}
$page   = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $perPage;

/* Summary (lifetime activities; leads filtered by created_at) */
$summarySql = "
    SELECT
        TRIM(COALESCE(l.assigned_user,'')) AS assigned_user,
        COUNT(DISTINCT l.id) AS total_leads,
        SUM(CASE WHEN a.lead_id IS NULL THEN 1 ELSE 0 END) AS leads_no_activity,
        SUM(CASE WHEN a.lead_id IS NOT NULL THEN 1 ELSE 0 END) AS leads_with_activity,
        COALESCE(SUM(a.total_activities),0) AS total_activities,
        COALESCE(SUM(a.cnt_call),0)    AS total_calls,
        COALESCE(SUM(a.cnt_email),0)   AS total_emails,
        COALESCE(SUM(a.cnt_task),0)    AS total_tasks,
        COALESCE(SUM(a.cnt_meeting),0) AS total_meetings
    FROM leads l
    LEFT JOIN (
        SELECT
            lead_id,
            COUNT(*) AS total_activities,
            SUM(activity_type='call')    AS cnt_call,
            SUM(activity_type='email')   AS cnt_email,
            SUM(activity_type='task')    AS cnt_task,
            SUM(activity_type='meeting') AS cnt_meeting
        FROM crm_activities
        GROUP BY lead_id
    ) a ON a.lead_id = l.id
    WHERE {$filterClause}
    GROUP BY assigned_user
    ORDER BY leads_with_activity DESC, total_leads DESC, assigned_user ASC
";

$allSummaryRows = q_all($DB, $summarySql);
$totalEmployees = count($allSummaryRows);

/* Global counters for charts */
$globalTotalLeads        = 0;
$globalLeadsNoActivity   = 0;
$globalLeadsWithActivity = 0;
$globalTotalActivities   = 0;
$globalCalls             = 0;
$globalEmails            = 0;
$globalTasks             = 0;
$globalMeetings          = 0;

$chartLabels            = [];
$chartTotalLeads        = [];
$chartLeadsNoActivity   = [];
$chartTotalActivities   = [];
$chartLeadsWithActivity = [];

$summaryRowsForPage = [];

/* Helper for per-emp donut chart: leads with vs without activity */
$withWithoutChartDataByEmp = [];

foreach ($allSummaryRows as $idx => $row) {
    $assignedRaw = trim($row['assigned_user'] ?? '');
    $displayName = $assignedRaw !== '' ? $assignedRaw : 'Unassigned';
    $assignedKey = $assignedRaw !== '' ? $assignedRaw : '__UNASSIGNED__';

    $totalLeads   = (int)$row['total_leads'];
    $noActivity   = (int)$row['leads_no_activity'];
    $withActivity = (int)$row['leads_with_activity'];
    $totalActs    = (int)$row['total_activities'];
    $totalCalls   = (int)$row['total_calls'];
    $totalEmails  = (int)$row['total_emails'];
    $totalTasks   = (int)$row['total_tasks'];
    $totalMeetings= (int)$row['total_meetings'];

    $globalTotalLeads        += $totalLeads;
    $globalLeadsNoActivity   += $noActivity;
    $globalLeadsWithActivity += $withActivity;
    $globalTotalActivities   += $totalActs;
    $globalCalls             += $totalCalls;
    $globalEmails            += $totalEmails;
    $globalTasks             += $totalTasks;
    $globalMeetings          += $totalMeetings;

    $chartLabels[]            = $displayName;
    $chartTotalLeads[]        = $totalLeads;
    $chartLeadsNoActivity[]   = $noActivity;
    $chartLeadsWithActivity[] = $withActivity;
    $chartTotalActivities[]   = $totalActs;

    $row['display_name']        = $displayName;
    $row['assigned_key']        = $assignedKey;
    $row['leads_with_activity'] = $withActivity;

    $allSummaryRows[$idx] = $row;

    $withWithoutChartDataByEmp[$assignedKey] = [
        'labels' => ['With Activity', 'No Activity'],
        'data'   => [$withActivity, $noActivity],
    ];
}

/* Donut chart dataset (global + per-emp) */
$withWithoutChartData = [];
$withWithoutChartData['__ALL__'] = [
    'labels' => ['With Activity', 'No Activity'],
    'data'   => [$globalLeadsWithActivity, $globalLeadsNoActivity],
];
foreach ($withWithoutChartDataByEmp as $empKey => $data) {
    $withWithoutChartData[$empKey] = $data;
}

/* Slice for current page */
if ($totalEmployees > 0) {
    $summaryRowsForPage = array_slice($allSummaryRows, $offset, $perPage);
}
$totalPages = max(1, (int)ceil($totalEmployees / $perPage));

/* ---------------------- Drill-down: leads for one employee ------------------ */
$selectedEmpKey   = isset($_GET['emp']) ? trim($_GET['emp']) : (isset($_GET['empid']) ? trim($_GET['empid']) : '');
$selectedEmpLabel = '';
$detailLeads      = [];

if ($selectedEmpKey !== '') {
    $isUnassigned = ($selectedEmpKey === '__UNASSIGNED__');
    if ($isUnassigned) {
        $empFilter = "(l.`assigned_user` IS NULL OR TRIM(l.`assigned_user`)='')";
        $selectedEmpLabel = 'Unassigned';
    } else {
        $safeEmp = mysqli_real_escape_string($DB, $selectedEmpKey);
        $empFilter = "TRIM(l.`assigned_user`) = '{$safeEmp}'";
        $selectedEmpLabel = $selectedEmpKey;
    }

    $detailSql = "
        SELECT
          l.`id`,
          l.`contact_name`,
          l.`email`,
          l.`phone_number`,
          l.`lead_status`,
          l.`pp_status`,
          l.`company`,
          l.`lead_source`,
          l.`tags`,
          l.`assigned_user`,
          l.`created_at`,
          l.`updated_at`,
          (SELECT MAX(COALESCE(ca.`date_start`, ca.`date_end`, ca.`created_at`))
             FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id`) AS last_activity_at,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='call')    AS cnt_call,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='email')   AS cnt_email,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='task')    AS cnt_task,
          (SELECT COUNT(*) FROM `crm_activities` ca
             WHERE ca.`lead_id` = l.`id` AND ca.`activity_type`='meeting') AS cnt_meeting
        FROM `leads` l
        WHERE {$filterClause} AND {$empFilter}
        ORDER BY l.`updated_at` DESC, l.`id` DESC
    ";
    $detailLeads = q_all($DB, $detailSql);
}

/* ---------------------- Lead Status (PP Status) chart data ------------------ */
/* Global + per-emp lead status (pp_status shown as Lead Status) */
$statusSql = "
    SELECT
      TRIM(COALESCE(l.assigned_user,'')) AS assigned_user,
      COALESCE(l.pp_status, '') AS pp_status,
      COUNT(*) AS cnt
    FROM leads l
    WHERE {$filterClause}
    GROUP BY assigned_user, pp_status
";
$statusRows = q_all($DB, $statusSql);

$globalStatusCounts  = [];
$statusCountsByEmp   = [];

foreach ($statusRows as $r) {
    $assignedRaw = trim($r['assigned_user'] ?? '');
    $assignedKey = $assignedRaw !== '' ? $assignedRaw : '__UNASSIGNED__';

    $statusLabel = trim($r['pp_status'] ?? '');
    if ($statusLabel === '') {
        $statusLabel = 'No Status';
    }

    $cnt = (int)$r['cnt'];

    if (!isset($globalStatusCounts[$statusLabel])) {
        $globalStatusCounts[$statusLabel] = 0;
    }
    $globalStatusCounts[$statusLabel] += $cnt;

    if (!isset($statusCountsByEmp[$assignedKey])) {
        $statusCountsByEmp[$assignedKey] = [];
    }
    if (!isset($statusCountsByEmp[$assignedKey][$statusLabel])) {
        $statusCountsByEmp[$assignedKey][$statusLabel] = 0;
    }
    $statusCountsByEmp[$assignedKey][$statusLabel] += $cnt;
}

$leadStatusChartData = [];

/* Global (__ALL__) */
if (!empty($globalStatusCounts)) {
    $leadStatusChartData['__ALL__'] = [
        'labels' => array_keys($globalStatusCounts),
        'data'   => array_values($globalStatusCounts),
    ];
} else {
    $leadStatusChartData['__ALL__'] = [
        'labels' => [],
        'data'   => [],
    ];
}

/* Per-emp */
foreach ($statusCountsByEmp as $empKey => $stCounts) {
    $leadStatusChartData[$empKey] = [
        'labels' => array_keys($stCounts),
        'data'   => array_values($stCounts),
    ];
}

/* ------------------------------- Page styles -------------------------------- */
ob_start();
?>
<style>
.employee-summary-table .table > :not(caption) > * > * {
    padding-top: .55rem;
    padding-bottom: .55rem;
}
.employee-summary-table tbody tr.employee-row {
    cursor: pointer;
}
.employee-summary-table tbody tr.employee-row:hover {
    background-color: rgba(59,130,246,.06);
}
.employee-summary-table .stat-pill {
    display:inline-block;
    font-size:11px;
    padding:2px 8px;
    border-radius:999px;
    background:#eef2ff;
    color:#312e81;
}
.employee-summary-table .stat-pill-green {
    background:#dcfce7;
    color:#166534;
}
.employee-summary-table .text-muted-small {
    font-size:11px;
    color:#6b7280 !important;
}
.chart-card canvas {
    width: 100%;
    height: 260px;
}
.employee-summary-table .badge-unassigned {
    background:#fee2e2;
    color:#b91c1c;
    padding:2px 8px;
    border-radius:999px;
    font-size:11px;
}
.leads-detail-table .table > :not(caption) > * > * {
    padding-top: .5rem;
    padding-bottom: .5rem;
}
.employee-summary-table th.sortable {
    cursor:pointer;
    user-select:none;
    white-space:nowrap;
}
.employee-summary-table th.sortable .sort-indicator {
    font-size:11px;
    margin-left:4px;
    opacity:.7;
}
#leadActivitiesModal .table > :not(caption) > * > * {
    padding-top:.45rem;
    padding-bottom:.45rem;
    font-size: 13px;
}
</style>
<?php
$styles = ob_get_clean();

/* ------------------------------ Page content -------------------------------- */
ob_start();
?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Employee Lead Activity Summary</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Employee Lead Activity</li>
    </ol>
  </div>
</div>

<!-- Date filter -->
<div class="mb-3 d-flex align-items-center justify-content-between flex-wrap gap-2">
  <div>
    <span class="me-2 fw-semibold">Distribution Date (Created At):</span>
    <span class="badge bg-light text-primary border"><?php echo h($dateLabel); ?></span>
  </div>
  <div class="btn-group btn-group-sm" role="group" aria-label="Date filter">
    <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?date_mode=today"
       class="btn btn-outline-primary <?php echo $dateMode === 'today' ? 'active' : ''; ?>">
      Today (<?php echo h($todayHuman); ?>)
    </a>
    <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?date_mode=yesterday"
       class="btn btn-outline-primary <?php echo $dateMode === 'yesterday' ? 'active' : ''; ?>">
      Yesterday (<?php echo h($yesterdayHuman); ?>)
    </a>
    <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?date_mode=all"
       class="btn btn-outline-primary <?php echo $dateMode === 'all' ? 'active' : ''; ?>">
      All
    </a>
  </div>
</div>

<!-- Charts row -->
<div class="row mb-3">
  <div class="col-xl-4">
    <div class="card custom-card chart-card mb-3 mb-xl-0">
      <div class="card-header">
        <div class="card-title mb-0">Leads per Employee (Bar)</div>
      </div>
      <div class="card-body">
        <canvas id="chartLeadsByEmployee"></canvas>
      </div>
    </div>
  </div>
  <div class="col-xl-4">
    <div class="card custom-card chart-card mb-3 mb-xl-0">
      <div class="card-header">
        <div class="card-title mb-0">Leads With vs Without Activity</div>
      </div>
      <div class="card-body">
        <canvas id="chartLeadsActivitySplit"></canvas>
      </div>
    </div>
  </div>
  <div class="col-xl-4">
    <div class="card custom-card chart-card">
      <div class="card-header">
        <div class="card-title mb-0">Lead Status wise</div>
      </div>
      <div class="card-body">
        <canvas id="chartLeadStatusWise"></canvas>
      </div>
    </div>
  </div>
</div>

<!-- Summary table -->
<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card employee-summary-table">
      <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-2">
        <div class="card-title mb-0">Employees vs Leads &amp; Activities</div>
        <div class="d-flex align-items-center gap-2 flex-wrap">
          <form method="get" class="d-flex align-items-center gap-2">
            <?php if ($selectedEmpKey !== ''): ?>
              <input type="hidden" name="emp" value="<?php echo h($selectedEmpKey); ?>">
            <?php endif; ?>
            <input type="hidden" name="date_mode" value="<?php echo h($dateMode); ?>">
            <select name="per_page" class="form-select form-select-sm" onchange="this.form.submit()">
              <?php foreach ($perPageOptions as $opt): ?>
                <option value="<?php echo (int)$opt; ?>" <?php echo $opt === $perPage ? 'selected' : ''; ?>>
                  <?php echo (int)$opt; ?> / page
                </option>
              <?php endforeach; ?>
            </select>
          </form>

          <?php if ($CAN_EXPORT_LEADS): ?>
          <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?export=summary&amp;date_mode=<?php echo h($dateMode); ?>"
             class="btn btn-outline-success btn-sm">
            <i class="ri-download-2-line me-1"></i>Export Summary
          </a>
          <?php endif; ?>
        </div>
      </div>
      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table text-nowrap align-middle mb-0">
            <thead>
              <tr>
                <th class="sortable" data-sort-key="seq" data-sort-type="number">
                  #
                  <span class="sort-indicator"></span>
                </th>
                <th class="sortable" data-sort-key="employee" data-sort-type="string">
                  Employee / Assigned User
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="totalLeads" data-sort-type="number">
                  Total Leads (Created Date)
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="withAct" data-sort-type="number">
                  Leads With Activity
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="noAct" data-sort-type="number">
                  Leads No Activity
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="totalActs" data-sort-type="number">
                  Total Activities
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="calls" data-sort-type="number">
                  Call
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="mails" data-sort-type="number">
                  Mail
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="tasks" data-sort-type="number">
                  Task
                  <span class="sort-indicator"></span>
                </th>
                <th class="text-center sortable" data-sort-key="meetings" data-sort-type="number">
                  Meeting
                  <span class="sort-indicator"></span>
                </th>
              </tr>
            </thead>
            <tbody>
              <?php if (!$summaryRowsForPage): ?>
                <tr><td colspan="10" class="text-center py-4">No data found.</td></tr>
              <?php else: ?>
                <?php foreach ($summaryRowsForPage as $i => $row): ?>
                  <?php
                    $displayName = $row['display_name'];
                    $assignedKey = $row['assigned_key'];
                    $totalLeads  = (int)$row['total_leads'];
                    $noActivity  = (int)$row['leads_no_activity'];
                    $withAct     = (int)$row['leads_with_activity'];
                    $totalActs   = (int)$row['total_activities'];
                    $idxNumber   = $offset + $i + 1;
                  ?>
                  <tr class="employee-row"
                      data-assigned="<?php echo h($assignedKey); ?>"
                      data-seq="<?php echo (int)$idxNumber; ?>"
                      data-employee="<?php echo h(mb_strtolower($displayName, 'UTF-8')); ?>"
                      data-total-leads="<?php echo (int)$totalLeads; ?>"
                      data-with-act="<?php echo (int)$withAct; ?>"
                      data-no-act="<?php echo (int)$noActivity; ?>"
                      data-total-acts="<?php echo (int)$totalActs; ?>"
                      data-calls="<?php echo (int)$row['total_calls']; ?>"
                      data-mails="<?php echo (int)$row['total_emails']; ?>"
                      data-tasks="<?php echo (int)$row['total_tasks']; ?>"
                      data-meetings="<?php echo (int)$row['total_meetings']; ?>">
                    <td><?php echo (int)$idxNumber; ?></td>
                    <td>
                      <?php if ($assignedKey === '__UNASSIGNED__'): ?>
                        <span class="badge-unassigned">Unassigned Leads</span>
                      <?php else: ?>
                        <span class="fw-semibold"><?php echo h($displayName); ?></span>
                      <?php endif; ?>
                      <div class="text-muted-small">
                        Click row to view all leads for this employee
                      </div>
                    </td>
                    <td class="text-center">
                      <span class="fw-semibold"><?php echo (int)$totalLeads; ?></span>
                    </td>
                    <td class="text-center">
                      <span class="stat-pill stat-pill-green"><?php echo (int)$withAct; ?></span>
                    </td>
                    <td class="text-center">
                      <span class="stat-pill"><?php echo (int)$noActivity; ?></span>
                    </td>
                    <td class="text-center">
                      <span class="fw-semibold"><?php echo (int)$totalActs; ?></span>
                    </td>
                    <td class="text-center"><?php echo (int)$row['total_calls']; ?></td>
                    <td class="text-center"><?php echo (int)$row['total_emails']; ?></td>
                    <td class="text-center"><?php echo (int)$row['total_tasks']; ?></td>
                    <td class="text-center"><?php echo (int)$row['total_meetings']; ?></td>
                  </tr>
                <?php endforeach; ?>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </div>
      <div class="card-footer border-top-0">
        <div class="d-flex align-items-center flex-wrap w-100">
          <div>Showing <?php echo count($summaryRowsForPage); ?> of <?php echo (int)$totalEmployees; ?> employees</div>
          <div class="ms-auto">
            <nav aria-label="Page navigation" class="pagination-style-4">
              <ul class="pagination mb-0">
                <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                  <a class="page-link"
                     href="<?php
                       if ($page > 1) {
                         $q = $_GET;
                         $q['page'] = $page - 1;
                         $q['per_page'] = $perPage;
                         echo h($_SERVER['PHP_SELF'] . '?' . http_build_query($q));
                       } else {
                         echo 'javascript:void(0);';
                       } ?>">Prev</a>
                </li>
                <?php
                  $start = max(1, $page - 2);
                  $end   = min($totalPages, $page + 2);
                  for ($p = $start; $p <= $end; $p++):
                    $q = $_GET;
                    $q['page'] = $p;
                    $q['per_page'] = $perPage;
                ?>
                  <li class="page-item <?php echo $p === $page ? 'active' : ''; ?>">
                    <a class="page-link"
                       href="<?php echo h($_SERVER['PHP_SELF'] . '?' . http_build_query($q)); ?>">
                      <?php echo $p; ?>
                    </a>
                  </li>
                <?php endfor; ?>
                <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                  <a class="page-link text-primary"
                     href="<?php
                       if ($page < $totalPages) {
                         $q = $_GET;
                         $q['page'] = $page + 1;
                         $q['per_page'] = $perPage;
                         echo h($_SERVER['PHP_SELF'] . '?' . http_build_query($q));
                       } else {
                         echo 'javascript:void(0);';
                       } ?>">Next</a>
                </li>
              </ul>
            </nav>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<?php if ($selectedEmpKey !== '' && $detailLeads): ?>
<div class="row mt-3">
  <div class="col-xl-12">
    <div class="card custom-card leads-detail-table">
      <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-2">
        <div>
          <div class="card-title mb-0">
            Leads for: <span class="fw-semibold"><?php echo h($selectedEmpLabel); ?></span>
          </div>
          <div class="text-muted-small">
            Total leads: <?php echo count($detailLeads); ?> (click a lead row to view activities)
          </div>
        </div>
        <div class="d-flex align-items-center gap-2 flex-wrap">
          <?php if ($CAN_EXPORT_LEADS): ?>
          <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?export=detail&amp;emp=<?php echo h($selectedEmpKey); ?>&amp;date_mode=<?php echo h($dateMode); ?>"
             class="btn btn-outline-success btn-sm">
            <i class="ri-file-excel-2-line me-1"></i>Export Leads
          </a>
          <?php endif; ?>
          <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?date_mode=<?php echo h($dateMode); ?>" class="btn btn-light btn-sm">
            Clear Selection
          </a>
        </div>
      </div>
      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table text-nowrap align-middle mb-0">
            <thead>
              <tr>
                <th>Lead ID</th>
                <th>Contact</th>
                <th>Source</th>
                <!-- Old lead_status column removed -->
                <th>Lead Status</th>
                <th>Created</th>
                <th>Activity Date (Recent)</th>
                <th class="text-center">Call</th>
                <th class="text-center">Mail</th>
                <th class="text-center">Task</th>
                <th class="text-center">Meeting</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($detailLeads as $L): ?>
                <?php
                  $cntCall    = (int)($L['cnt_call'] ?? 0);
                  $cntEmail   = (int)($L['cnt_email'] ?? 0);
                  $cntTask    = (int)($L['cnt_task'] ?? 0);
                  $cntMeeting = (int)($L['cnt_meeting'] ?? 0);
                ?>
                <tr class="lead-row" data-lead-id="<?php echo (int)$L['id']; ?>">
                  <td><?php echo (int)$L['id']; ?></td>
                  <td>
                    <div class="fw-semibold"><?php echo h($L['contact_name']); ?></div>
                    <div class="text-muted-small">
                      <?php echo h($L['phone_number']); ?>
                      <?php if (!empty($L['email'])): ?>
                        · <?php echo h($L['email']); ?>
                      <?php endif; ?>
                    </div>
                  </td>
                  <td><?php echo h($L['lead_source']); ?></td>
                  <td><?php echo h($L['pp_status']); ?></td>
                  <td><?php echo h($L['created_at']); ?></td>
                  <td><?php echo h($L['last_activity_at']); ?></td>
                  <td class="text-center"><?php echo $cntCall; ?></td>
                  <td class="text-center"><?php echo $cntEmail; ?></td>
                  <td class="text-center"><?php echo $cntTask; ?></td>
                  <td class="text-center"><?php echo $cntMeeting; ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>
  </div>
</div>
<?php endif; ?>

<!-- Lead Activities Modal -->
<div class="modal fade" id="leadActivitiesModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h6 class="modal-title">Lead Activities</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div id="lead-activities-empty" class="text-muted text-center my-3" style="display:none;">
          No activities found for this lead.
        </div>
        <div class="table-responsive" id="lead-activities-table-wrapper" style="display:none;">
          <table class="table table-sm table-bordered mb-0">
            <thead>
              <tr>
                <th style="width: 130px;">Date / Time</th>
                <th style="width: 80px;">Type</th>
                <th style="width: 110px;">Status</th>
                <th style="width: 90px;">Direction</th>
                <th>Title</th>
                <th>Remarks</th>
              </tr>
            </thead>
            <tbody id="lead-activities-body">
            </tbody>
          </table>
        </div>
      </div>
      <div class="modal-footer py-2">
        <button type="button" class="btn btn-light btn-sm" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<?php
$content = ob_get_clean();

/* ------------------------------ Page scripts ------------------------------- */
ob_start();
?>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
(function(){
  var currentDateMode = <?php echo json_encode($dateMode, JSON_UNESCAPED_UNICODE); ?>;
  var currentEmpKey   = <?php echo json_encode($selectedEmpKey, JSON_UNESCAPED_UNICODE); ?> || '';

  // PHP → JS datasets
  var chartLabels            = <?php echo json_encode($chartLabels, JSON_UNESCAPED_UNICODE); ?>;
  var chartTotalLeads        = <?php echo json_encode($chartTotalLeads); ?>;
  var chartLeadsWithActivity = <?php echo json_encode($chartLeadsWithActivity); ?>;

  var withWithoutChartData = <?php echo json_encode($withWithoutChartData, JSON_UNESCAPED_UNICODE); ?>;
  var leadStatusChartData  = <?php echo json_encode($leadStatusChartData, JSON_UNESCAPED_UNICODE); ?>;

  var globalTotalLeads        = <?php echo (int)$globalTotalLeads; ?>;
  var globalLeadsNoActivity   = <?php echo (int)$globalLeadsNoActivity; ?>;
  var globalLeadsWithActivity = <?php echo (int)$globalLeadsWithActivity; ?>;
  var globalTotalActivities   = <?php echo (int)$globalTotalActivities; ?>;
  var globalCalls             = <?php echo (int)$globalCalls; ?>;
  var globalEmails            = <?php echo (int)$globalEmails; ?>;
  var globalTasks             = <?php echo (int)$globalTasks; ?>;
  var globalMeetings          = <?php echo (int)$globalMeetings; ?>;

  // Row click → drill-down by employee
  document.querySelectorAll('.employee-summary-table .employee-row').forEach(function(tr){
    tr.addEventListener('click', function(){
      var key = this.getAttribute('data-assigned') || '';
      if (!key) return;
      var url = new URL(window.location.href);
      url.searchParams.set('emp', key);
      url.searchParams.set('date_mode', currentDateMode);
      url.searchParams.delete('page');
      window.location.href = url.toString();
    });
  });

  // Sort summary headers (client-side for current page)
  (function(){
    var table = document.querySelector('.employee-summary-table table');
    if (!table) return;

    var tbody = table.querySelector('tbody');
    var headers = table.querySelectorAll('th.sortable');

    headers.forEach(function(th){
      th.addEventListener('click', function(){
        if (!tbody) return;

        var key = th.getAttribute('data-sort-key');
        var type = th.getAttribute('data-sort-type') || 'number';
        if (!key) return;

        var currentDir = th.getAttribute('data-sort-dir') === 'asc' ? 'desc' : 'asc';

        headers.forEach(function(h){
          if (h !== th) {
            h.removeAttribute('data-sort-dir');
            var s = h.querySelector('.sort-indicator');
            if (s) s.textContent = '';
          }
        });

        th.setAttribute('data-sort-dir', currentDir);
        var indicator = th.querySelector('.sort-indicator');
        if (indicator) indicator.textContent = currentDir === 'asc' ? '▲' : '▼';

        var rows = Array.prototype.slice.call(tbody.querySelectorAll('tr.employee-row'));

        rows.sort(function(a,b){
          var va, vb;

          switch (key) {
            case 'seq':
              va = parseInt(a.getAttribute('data-seq') || '0', 10);
              vb = parseInt(b.getAttribute('data-seq') || '0', 10);
              break;
            case 'employee':
              va = (a.getAttribute('data-employee') || '').toString();
              vb = (b.getAttribute('data-employee') || '').toString();
              break;
            case 'totalLeads':
              va = parseInt(a.getAttribute('data-total-leads') || '0', 10);
              vb = parseInt(b.getAttribute('data-total-leads') || '0', 10);
              break;
            case 'withAct':
              va = parseInt(a.getAttribute('data-with-act') || '0', 10);
              vb = parseInt(b.getAttribute('data-with-act') || '0', 10);
              break;
            case 'noAct':
              va = parseInt(a.getAttribute('data-no-act') || '0', 10);
              vb = parseInt(b.getAttribute('data-no-act') || '0', 10);
              break;
            case 'totalActs':
              va = parseInt(a.getAttribute('data-total-acts') || '0', 10);
              vb = parseInt(b.getAttribute('data-total-acts') || '0', 10);
              break;
            case 'calls':
              va = parseInt(a.getAttribute('data-calls') || '0', 10);
              vb = parseInt(b.getAttribute('data-calls') || '0', 10);
              break;
            case 'mails':
              va = parseInt(a.getAttribute('data-mails') || '0', 10);
              vb = parseInt(b.getAttribute('data-mails') || '0', 10);
              break;
            case 'tasks':
              va = parseInt(a.getAttribute('data-tasks') || '0', 10);
              vb = parseInt(b.getAttribute('data-tasks') || '0', 10);
              break;
            case 'meetings':
              va = parseInt(a.getAttribute('data-meetings') || '0', 10);
              vb = parseInt(b.getAttribute('data-meetings') || '0', 10);
              break;
            default:
              va = 0;
              vb = 0;
          }

          if (type === 'string') {
            va = va.toString();
            vb = vb.toString();
            if (va < vb) return currentDir === 'asc' ? -1 : 1;
            if (va > vb) return currentDir === 'asc' ? 1 : -1;
            return 0;
          } else {
            va = Number(va) || 0;
            vb = Number(vb) || 0;
            if (va < vb) return currentDir === 'asc' ? -1 : 1;
            if (va > vb) return currentDir === 'asc' ? 1 : -1;
            return 0;
          }
        });

        rows.forEach(function(r){ tbody.appendChild(r); });
      });
    });
  })();

  // Bar: Leads per Employee (Leads With Activity) - always across all employees
  (function(){
    var maxBars = 10;
    var barLabels = chartLabels.slice(0, maxBars);
    var barData   = chartLeadsWithActivity.slice(0, maxBars);
    var ctxBar = document.getElementById('chartLeadsByEmployee');
    if (ctxBar && barLabels.length) {
      new Chart(ctxBar, {
        type: 'bar',
        data: {
          labels: barLabels,
          datasets: [{
            label: 'Leads With Activity',
            data: barData
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: {
            legend: { display: false }
          },
          scales: {
            x: {
              ticks: {
                maxRotation: 45,
                minRotation: 0,
                autoSkip: true
              }
            },
            y: {
              beginAtZero: true,
              ticks: { precision: 0 }
            }
          }
        }
      });
    }
  })();

  // Donut: Leads With vs Without Activity (global or selected emp)
  (function(){
    var ctxDonut = document.getElementById('chartLeadsActivitySplit');
    if (!ctxDonut || !withWithoutChartData) return;

    var key = currentEmpKey && withWithoutChartData[currentEmpKey] ? currentEmpKey : '__ALL__';
    var d = withWithoutChartData[key];

    if (!d || !d.labels || !d.labels.length) return;

    new Chart(ctxDonut, {
      type: 'doughnut',
      data: {
        labels: d.labels,
        datasets: [{
          data: d.data
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: { position: 'bottom' }
        }
      }
    });
  })();

  // Pie: Lead Status wise (PP Status) (global or selected emp)
  (function(){
    var ctxStatus = document.getElementById('chartLeadStatusWise');
    if (!ctxStatus || !leadStatusChartData) return;

    var key = currentEmpKey && leadStatusChartData[currentEmpKey] ? currentEmpKey : '__ALL__';
    var d = leadStatusChartData[key];

    if (!d || !d.labels || !d.labels.length) return;

    // Color mapping: Dead = red, Prospect = green (always)
    var bgColors = (d.labels || []).map(function(lbl, idx){
      var v = (lbl || '').toString().toLowerCase().trim();
      if (v === 'dead') {
        return '#ef4444'; // red
      }
      if (v === 'prospect' || v === 'prospecting') {
        return '#22c55e'; // green
      }
      // fallback palette for other statuses
      var palette = ['#3b82f6','#f97316','#a855f7','#06b6d4','#84cc16','#eab308','#6366f1'];
      return palette[idx % palette.length];
    });

    new Chart(ctxStatus, {
      type: 'pie',
      data: {
        labels: d.labels,
        datasets: [{
          data: d.data,
          backgroundColor: bgColors
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: { position: 'bottom' }
        }
      }
    });
  })();

  // Lead detail row click → activities popup
  (function(){
    var leadRows = document.querySelectorAll('.leads-detail-table .lead-row');
    if (!leadRows.length) return;

    var modalEl = document.getElementById('leadActivitiesModal');
    if (!modalEl) return;

    var modal = null;
    var bodyContainer = document.getElementById('lead-activities-body');
    var wrapper = document.getElementById('lead-activities-table-wrapper');
    var emptyMsg = document.getElementById('lead-activities-empty');

    leadRows.forEach(function(tr){
      tr.addEventListener('click', function(){
        var leadId = this.getAttribute('data-lead-id') || '';
        if (!leadId) return;

        if (bodyContainer) bodyContainer.innerHTML = '';
        if (wrapper) wrapper.style.display = 'none';
        if (emptyMsg) emptyMsg.style.display = 'none';

        var form = new URLSearchParams();
        form.append('ajax','lead_activities');
        form.append('lead_id', leadId);

        fetch('<?php echo h($_SERVER["PHP_SELF"]); ?>?date_mode=' + encodeURIComponent(currentDateMode), {
          method: 'POST',
          headers: {'Content-Type': 'application/x-www-form-urlencoded'},
          body: form.toString()
        })
        .then(function(r){ return r.json().catch(function(){ return null; }); })
        .then(function(res){
          if (!res || !res.ok) {
            if (emptyMsg) {
              emptyMsg.textContent = (res && res.msg) ? res.msg : 'Unable to load activities.';
              emptyMsg.style.display = 'block';
            }
          } else {
            var list = Array.isArray(res.activities) ? res.activities : [];
            if (!list.length) {
              if (emptyMsg) {
                emptyMsg.textContent = 'No activities found for this lead.';
                emptyMsg.style.display = 'block';
              }
            } else if (bodyContainer) {
              list.forEach(function(a){
                var tr = document.createElement('tr');

                var dtVal = a.date_start || a.date_end || a.created_at || '';
                var cellDate = document.createElement('td');
                cellDate.textContent = dtVal || '';
                tr.appendChild(cellDate);

                var cellType = document.createElement('td');
                cellType.textContent = (a.activity_type || '').toUpperCase();
                tr.appendChild(cellType);

                var cellStatus = document.createElement('td');
                cellStatus.textContent = a.status || '';
                tr.appendChild(cellStatus);

                var cellDir = document.createElement('td');
                cellDir.textContent = a.direction || '';
                tr.appendChild(cellDir);

                var cellTitle = document.createElement('td');
                cellTitle.textContent = a.title || '';
                tr.appendChild(cellTitle);

                var cellRemarks = document.createElement('td');
                cellRemarks.textContent = a.description || '';
                tr.appendChild(cellRemarks);

                bodyContainer.appendChild(tr);
              });
              if (wrapper) wrapper.style.display = 'block';
            }
          }

          if (!modal) {
            modal = new bootstrap.Modal(modalEl);
          }
          modal.show();
        })
        .catch(function(){
          if (emptyMsg) {
            emptyMsg.textContent = 'Network error while loading activities.';
            emptyMsg.style.display = 'block';
          }
          if (!modal) {
            modal = new bootstrap.Modal(modalEl);
          }
          modal.show();
        });
      });
    });
  })();
})();
</script>
<?php
$scripts = ob_get_clean();

include __DIR__ . '/layouts/base.php';
?>
