<?php
// ============================================================================
// File   : leads_pp_status_list.php
// Purpose:
//   - List leads from broriserin_onb.leads based on pp_status
//   - Columns:
//       contact_name, lead_source, project_title,
//       created_at, updated_at, status (pp_status),
//       assigned_user (EmpID), Employee Name (hr_dump.Employee_Full_Name),
//       opp_stage, claim_date
//   - Search + pp_status filter + pagination + CSV export
//   - Employee Name fetched from broriserin_ic_attendance.hr_dump using empid
// Stack  : Core PHP + MySQLi (NO PDO). DB connections in ../config.php
// Collate: utf8mb4_general_ci
// ============================================================================

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
date_default_timezone_set('Asia/Kolkata');

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/includes/check-permission.php';

$currentPage = basename(__FILE__);
if (function_exists('checkPageAccessOrRedirect')) {
    checkPageAccessOrRedirect($currentPage, $conn_sa);
}

/* ----------------------------- DB handles ---------------------------------- */
// ONB leads connection (same as leads_upload.php)
if (!isset($conn) || !($conn instanceof mysqli)) {
    http_response_code(500);
    exit('DB connection $conn not available.');
}
$DB_LEADS = $conn;

// HR / Attendance connection (for hr_dump)
$DB_HR = (isset($conn_ic) && $conn_ic instanceof mysqli) ? $conn_ic : null;

// Charset / collation
mysqli_set_charset($DB_LEADS, 'utf8mb4');
@mysqli_query($DB_LEADS, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($DB_LEADS, "SET collation_connection = 'utf8mb4_general_ci'");

/* ----------------------------- Helpers ------------------------------------- */
function h($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

/* ----------------------------- Filters ------------------------------------- */
$search   = isset($_GET['q']) ? trim($_GET['q']) : '';
$ppStatus = isset($_GET['pp_status']) ? trim($_GET['pp_status']) : 'all';

$statusOptions = [
    'Pending',
    'Assigned',
    'In Process',
    'Prospect',
    'Recycled',
    'Dead'
];

// Pagination
$perPageOptions = [10, 25, 50, 500];
$perPage = isset($_GET['per_page']) ? (int)$_GET['per_page'] : 25;
if (!in_array($perPage, $perPageOptions, true)) {
    $perPage = 25;
}
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
if ($page < 1) $page = 1;
$offset = ($page - 1) * $perPage;

/* ------------------------ WHERE clause for leads --------------------------- */
$whereParts = [];
$whereParts[] = "1=1";

// pp_status filter
if ($ppStatus !== '' && $ppStatus !== 'all') {
    $ppEsc = mysqli_real_escape_string($DB_LEADS, $ppStatus);
    $whereParts[] = "COALESCE(pp_status,'') = '{$ppEsc}'";
}

// Search filter (name/source/project/user)
if ($search !== '') {
    $sEsc = mysqli_real_escape_string($DB_LEADS, $search);
    $like = "%{$sEsc}%";
    $whereParts[] = "(
        contact_name  LIKE '{$like}' OR
        lead_source   LIKE '{$like}' OR
        project_title LIKE '{$like}' OR
        assigned_user LIKE '{$like}'
    )";
}

$whereSql = implode(' AND ', $whereParts);

/* ------------------------------ Export CSV --------------------------------- */
if (isset($_GET['export']) && $_GET['export'] === '1') {

    $exportSql = "
        SELECT
            id,
            contact_name,
            lead_source,
            project_title,
            created_at,
            updated_at,
            pp_status,
            assigned_user,
            empid,
            opp_stage,
            claim_date
        FROM leads
        WHERE {$whereSql}
        ORDER BY created_at DESC, id DESC
    ";

    $resExp = mysqli_query($DB_LEADS, $exportSql);
    $rowsExp = [];
    if ($resExp) {
        while ($r = mysqli_fetch_assoc($resExp)) {
            $rowsExp[] = $r;
        }
        mysqli_free_result($resExp);
    }

    // Build empid list and fetch Employee_Full_Name from hr_dump
    $empNames = [];
    if ($DB_HR && $rowsExp) {
        $empIds = [];
        foreach ($rowsExp as $r) {
            if (!empty($r['empid'])) {
                $empIds[] = $r['empid'];
            }
        }
        $empIds = array_values(array_unique(array_filter($empIds)));
        if ($empIds) {
            $parts = [];
            foreach ($empIds as $e) {
                $parts[] = "'" . mysqli_real_escape_string($DB_HR, $e) . "'";
            }
            $inEmp = implode(',', $parts);
            $sqlHr = "
                SELECT empid, Employee_Full_Name
                FROM hr_dump
                WHERE empid IN ({$inEmp})
            ";
            $resHr = mysqli_query($DB_HR, $sqlHr);
            if ($resHr) {
                while ($hr = mysqli_fetch_assoc($resHr)) {
                    $empNames[$hr['empid']] = $hr['Employee_Full_Name'];
                }
                mysqli_free_result($resHr);
            }
        }
    }

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=leads_pp_status_' . date('Ymd_His') . '.csv');

    $out = fopen('php://output', 'w');
    fputcsv($out, [
        'Contact Name',
        'Lead Source',
        'Project Title',
        'Assigned Date',
        'Status Date',
        'Status ',
        'Assigned User (EmpID)',
        'Employee Name',
        'Opp Stage',
        'Claim Date'
    ]);

    foreach ($rowsExp as $r) {
        $empid   = (string)($r['empid'] ?? '');
        $empName = $empNames[$empid] ?? '';
        fputcsv($out, [
            $r['contact_name'],
            $r['lead_source'],
            $r['project_title'],
            $r['created_at'],
            $r['updated_at'],
            $r['pp_status'],
            $r['assigned_user'],
            $empName,
            $r['opp_stage'],
            $r['claim_date'],
        ]);
    }
    fclose($out);
    exit;
}

/* ------------------- Count + fetch paginated leads ------------------------- */
$dbError = '';

$countSql = "
    SELECT COUNT(*) AS cnt
    FROM leads
    WHERE {$whereSql}
";
$totalRows = 0;
$resCount = mysqli_query($DB_LEADS, $countSql);
if ($resCount) {
    $rowC = mysqli_fetch_assoc($resCount);
    if ($rowC) {
        $totalRows = (int)$rowC['cnt'];
    }
    mysqli_free_result($resCount);
} else {
    $dbError = mysqli_error($DB_LEADS);
}

$totalPages = $totalRows > 0 ? (int)ceil($totalRows / $perPage) : 1;
if ($totalPages < 1) $totalPages = 1;
if ($page > $totalPages) {
    $page   = $totalPages;
    $offset = ($page - 1) * $perPage;
    if ($offset < 0) $offset = 0;
}

$listSql = "
    SELECT
        id,
        contact_name,
        lead_source,
        project_title,
        created_at,
        updated_at,
        pp_status,
        assigned_user,
        empid,
        opp_stage,
        claim_date
    FROM leads
    WHERE {$whereSql}
    ORDER BY created_at DESC, id DESC
    LIMIT {$perPage} OFFSET {$offset}
";

$rows = [];
$resList = mysqli_query($DB_LEADS, $listSql);
if ($resList) {
    while ($r = mysqli_fetch_assoc($resList)) {
        $rows[] = $r;
    }
    mysqli_free_result($resList);
} else {
    $dbError = mysqli_error($DB_LEADS);
}

/* ------------------ Fetch Employee names from hr_dump ---------------------- */
$empNameMap = [];
if ($DB_HR && $rows) {
    $empIds = [];
    foreach ($rows as $r) {
        if (!empty($r['empid'])) {
            $empIds[] = $r['empid'];
        }
    }
    $empIds = array_values(array_unique(array_filter($empIds)));
    if ($empIds) {
        $parts = [];
        foreach ($empIds as $e) {
            $parts[] = "'" . mysqli_real_escape_string($DB_HR, $e) . "'";
        }
        $inEmp = implode(',', $parts);
        $sqlHr = "
            SELECT empid, Employee_Full_Name
            FROM hr_dump
            WHERE empid IN ({$inEmp})
        ";
        $resHr = mysqli_query($DB_HR, $sqlHr);
        if ($resHr) {
            while ($hr = mysqli_fetch_assoc($resHr)) {
                $empNameMap[$hr['empid']] = $hr['Employee_Full_Name'];
            }
            mysqli_free_result($resHr);
        }
    }
}

/* -------------------------------- Styles ----------------------------------- */
ob_start();
?>
<style>
.leads-pp-status-table .table > :not(caption) > * > * {
    padding-top: .55rem;
    padding-bottom: .55rem;
}
.leads-pp-status-table th {
    white-space: nowrap;
}
.leads-pp-status-table .text-muted-small {
    font-size: 11px;
    color: #6b7280 !important;
}
.leads-pp-status-table .status-pill {
    display: inline-block;
    padding: 2px 8px;
    border-radius: 999px;
    font-size: 11px;
    background: #eef2ff;
    color: #312e81;
}
.leads-pp-status-table .status-pill.in-process {
    background: #dcfce7;
    color: #166534;
}
.leads-pp-status-table .status-pill.prospect {
    background: #fef3c7;
    color: #92400e;
}
.leads-pp-status-table .status-pill.dead {
    background: #fee2e2;
    color: #b91c1c;
}
</style>
<?php
$styles = ob_get_clean();

/* ------------------------------ Page content -------------------------------- */
ob_start();
?>
<div class="page-header-breadcrumb mb-3">
    <div class="d-flex align-center justify-content-between flex-wrap">
        <h1 class="page-title fw-medium fs-18 mb-0">Leads by PP Status</h1>
        <ol class="breadcrumb mb-0">
            <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
            <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
            <li class="breadcrumb-item active" aria-current="page">Leads by PP Status</li>
        </ol>
    </div>
</div>

<div class="row">
    <div class="col-xl-12">
        <div class="card custom-card leads-pp-status-table">
            <div class="card-header border-bottom-0 pb-0">
                <div class="d-flex flex-column flex-lg-row align-items-lg-center justify-content-between w-100 gap-2">
                    <div>
                        <div class="card-title mb-1">Leads by PP Status</div>
                        <div class="text-muted-small">
                          
                        </div>
                    </div>
                    <div class="d-flex align-items-center gap-2 flex-wrap">
                        <form method="get" class="d-flex align-items-center gap-2 flex-wrap" action="<?php echo h($_SERVER['PHP_SELF']); ?>">
                            <div class="input-group input-group-sm">
                                <span class="input-group-text border-end-0">
                                    <i class="ri-search-line"></i>
                                </span>
                                <input type="text"
                                       name="q"
                                       class="form-control form-control-sm border-start-0"
                                       placeholder="Search name / source / project / user"
                                       value="<?php echo h($search); ?>">
                            </div>

                            <select name="pp_status" class="form-select form-select-sm">
                                <option value="all"<?php echo ($ppStatus === 'all' ? ' selected' : ''); ?>>All Status</option>
                                <?php foreach ($statusOptions as $st): ?>
                                    <option value="<?php echo h($st); ?>"<?php echo ($ppStatus === $st ? ' selected' : ''); ?>>
                                        <?php echo h($st); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>

                            <select name="per_page" class="form-select form-select-sm">
                                <?php foreach ($perPageOptions as $opt): ?>
                                    <option value="<?php echo (int)$opt; ?>"<?php echo ($perPage === $opt ? ' selected' : ''); ?>>
                                        <?php echo (int)$opt; ?> / page
                                    </option>
                                <?php endforeach; ?>
                            </select>

                            <button type="submit" class="btn btn-primary btn-sm">
                                <i class="ri-filter-3-line me-1"></i>Apply
                            </button>
                            <a href="<?php echo h($_SERVER['PHP_SELF']); ?>" class="btn btn-outline-secondary btn-sm">
                                Clear
                            </a>
                        </form>

                        <a href="<?php
                            $q = $_GET;
                            $q['export'] = 1;
                            echo h($_SERVER['PHP_SELF'] . '?' . http_build_query($q));
                        ?>" class="btn btn-outline-success btn-sm">
                            <i class="ri-download-2-line me-1"></i>Export
                        </a>
                    </div>
                </div>
                <div class="mt-2 text-muted-small">
                    Filter &nbsp;|&nbsp; Total matching leads: <?php echo (int)$totalRows; ?>
                </div>
                <?php if ($dbError !== ''): ?>
                    <div class="mt-1 text-danger text-muted-small">
                        DB Error: <?php echo h($dbError); ?>
                    </div>
                <?php endif; ?>
            </div>

            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table text-nowrap mb-0">
                        <thead>
                            <tr>
                                <th style="width:60px;">#</th>
                                <th>Contact Name</th>
                                <th>Lead Source</th>
                                <th>Project Title</th>
                                <th>Created At</th>
                                <th>Updated At</th>
                                <th>Status</th>
                                <th>Assigned User (EmpID)</th>
                                <th>Employee Name</th>
                                <th>Opp Stage</th>
                                <th>Claim Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!$rows): ?>
                                <tr>
                                    <td colspan="11" class="text-center py-4">
                                        No leads found for selected filters.
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php
                                $startIndex = $offset + 1;
                                foreach ($rows as $i => $r):
                                    $rowNum = $startIndex + $i;
                                    $status = (string)($r['pp_status'] ?? '');
                                    $statusClass = 'status-pill';
                                    $statusLower = strtolower($status);
                                    if ($statusLower === 'in process') {
                                        $statusClass .= ' in-process';
                                    } elseif ($statusLower === 'prospect') {
                                        $statusClass .= ' prospect';
                                    } elseif ($statusLower === 'dead') {
                                        $statusClass .= ' dead';
                                    }
                                    $empid   = (string)($r['empid'] ?? '');
                                    $empName = $empNameMap[$empid] ?? '';
                                ?>
                                    <tr>
                                        <td><?php echo (int)$rowNum; ?></td>
                                        <td><?php echo h($r['contact_name']); ?></td>
                                        <td><?php echo h($r['lead_source']); ?></td>
                                        <td><?php echo h($r['project_title']); ?></td>
                                        <td><?php echo h($r['created_at']); ?></td>
                                        <td><?php echo h($r['updated_at']); ?></td>
                                        <td>
                                            <?php if ($status !== ''): ?>
                                                <span class="<?php echo h($statusClass); ?>">
                                                    <?php echo h($status); ?>
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo h($r['assigned_user']); ?></td>
                                        <td><?php echo h($empName); ?></td>
                                        <td><?php echo h($r['opp_stage']); ?></td>
                                        <td><?php echo h($r['claim_date']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <div class="card-footer border-top-0">
                <div class="d-flex align-items-center flex-wrap w-100">
                    <div>
                        Showing <?php echo count($rows); ?> of <?php echo (int)$totalRows; ?> leads
                    </div>
                    <div class="ms-auto">
                        <nav aria-label="Page navigation" class="pagination-style-4">
                            <ul class="pagination mb-0">
                                <?php
                                $baseParams = $_GET;
                                unset($baseParams['page']);
                                $baseParams['per_page'] = $perPage;
                                ?>
                                <li class="page-item <?php echo ($page <= 1 ? 'disabled' : ''); ?>">
                                    <a class="page-link"
                                       href="<?php
                                           if ($page > 1) {
                                               $baseParams['page'] = $page - 1;
                                               echo h($_SERVER['PHP_SELF'] . '?' . http_build_query($baseParams));
                                           } else {
                                               echo 'javascript:void(0);';
                                           }
                                       ?>">Prev</a>
                                </li>
                                <?php
                                $start = max(1, $page - 2);
                                $end   = min($totalPages, $page + 2);
                                for ($p = $start; $p <= $end; $p++):
                                    $baseParams['page'] = $p;
                                ?>
                                    <li class="page-item <?php echo ($p === $page ? 'active' : ''); ?>">
                                        <a class="page-link"
                                           href="<?php echo h($_SERVER['PHP_SELF'] . '?' . http_build_query($baseParams)); ?>">
                                            <?php echo $p; ?>
                                        </a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo ($page >= $totalPages ? 'disabled' : ''); ?>">
                                    <a class="page-link"
                                       href="<?php
                                           if ($page < $totalPages) {
                                               $baseParams['page'] = $page + 1;
                                               echo h($_SERVER['PHP_SELF'] . '?' . http_build_query($baseParams));
                                           } else {
                                               echo 'javascript:void(0);';
                                           }
                                       ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    </div>
                </div>
            </div>

        </div>
    </div>
</div>
<?php
$content = ob_get_clean();

/* ------------------------------ Scripts ------------------------------------ */
ob_start();
// no extra JS for now
$scripts = ob_get_clean();

/* --------------------------- Render with layout ---------------------------- */
include __DIR__ . '/layouts/base.php';
?>
