<?php
/* =============================================================================
 * File   : pages/leads_upload.php
 * Purpose: THEME-INTEGRATED CSV uploader for broriserin_onb.leads
 *          + Attach all uploaded leads to a SELECTED project from
 *            CRM Projects master (via $conn_crmplus)
 *          + Distribute leads to employees from IC Attendance master
 *            based on Lead Type (Fresh / Resycled), with ACTIVE status filter
 *            and persistent round-robin state across uploads.
 *
 * Lead Type logic:
 *   - Lead Type = "Fresh"
 *       * Method "Team Leader"
 *           · Use ACTIVE team leaders configured for Fresh leads
 *           · Equally distribute uploaded leads across these employees (round-robin)
 *           · For each inserted lead:
 *                 leads.Functional_Reporting = selected empid
 *           · leads.assigned_user / leads.empid remain NULL
 *
 *       * Method "Via Claim"
 *           · NO auto-distribution happens
 *           · For each inserted lead:
 *                 Functional_Reporting, assigned_user, empid all remain NULL
 *           · Leads are later claimed/assigned from CRM
 *
 *   - Lead Type = "Resyled"
 *       * Use ACTIVE employees configured for Resycled leads
 *       * Equally distribute uploaded leads across these employees (round-robin)
 *       * For each inserted lead:
 *             leads.assigned_user = empid
 *             leads.empid         = empid
 *             leads.claim_date    = created_at (IST)
 *
 * Round-robin persistence:
 *   - If 10 empid are eligible and only 4 leads uploaded,
 *     remaining 6 empid are remembered using table lead_distribution_state.
 *   - Next upload will start from empid index where last upload stopped.
 *
 * Extra rules:
 *   - On upload of ANY type, set leads.pp_status = 'Pending'
 *     and DO NOT set opp_stage (kept blank/default).
 *   - Only projects with is_active = 1 are shown.
 *
 * Stack  : Core PHP + MySQLi (NO PDO). DB connections in ../config.php
 * Rules  : COLLATE=utf8mb4_general_ci
 * ============================================================================= */

date_default_timezone_set('Asia/Kolkata');
@session_start();

/* --- Central config (defines $conn, $conn_crmplus, $conn_ic, etc.) --- */
require_once __DIR__ . '/../config.php';

/* Charset / collation for ONB (leads table lives here) */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* Helpers */
if (!function_exists('h')) {
    function h($s){
        return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    }
}
function col_exists($conn,$table,$col){
    $t = mysqli_real_escape_string($conn,$table);
    $c = mysqli_real_escape_string($conn,$col);
    $q = mysqli_query(
        $conn,
        "SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS
         WHERE TABLE_SCHEMA = DATABASE()
           AND TABLE_NAME='{$t}'
           AND COLUMN_NAME='{$c}'"
    );
    return ($q && mysqli_num_rows($q) > 0);
}

/* Ensure leads table & columns exist (ONB DB via $conn, idempotent) */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `leads` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `contact_name` VARCHAR(150) COLLATE utf8mb4_general_ci NOT NULL,
  `phone_number` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `email` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_source` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT '',
  `project_id` INT UNSIGNED DEFAULT NULL,
  `project_title` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `project_listing_type` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `project_tag` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `project_name` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `project_city` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `project_region` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `project_location` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `project_address` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `Functional_Reporting` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `assigned_user` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `empid` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_type` VARCHAR(20) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `pp_status` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT 'Pending',
  `opp_stage` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `claim_date` DATETIME NULL DEFAULT NULL,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

if (!col_exists($conn,'leads','phone_number')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `phone_number` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `contact_name`");
}
if (!col_exists($conn,'leads','lead_source')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `lead_source` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT ''
        AFTER `email`");
}
if (!col_exists($conn,'leads','project_id')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_id` INT UNSIGNED DEFAULT NULL
        AFTER `lead_source`");
}
if (!col_exists($conn,'leads','project_title')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_title` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_id`");
}
if (!col_exists($conn,'leads','project_listing_type')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_listing_type` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_title`");
}
if (!col_exists($conn,'leads','project_tag')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_tag` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_listing_type`");
}
if (!col_exists($conn,'leads','project_name')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_name` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_tag`");
}
if (!col_exists($conn,'leads','project_city')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_city` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_name`");
}
if (!col_exists($conn,'leads','project_region')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_region` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_city`");
}
if (!col_exists($conn,'leads','project_location')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_location` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_region`");
}
if (!col_exists($conn,'leads','project_address')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `project_address` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_location`");
}
if (!col_exists($conn,'leads','Functional_Reporting')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `Functional_Reporting` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `project_address`");
}
if (!col_exists($conn,'leads','assigned_user')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `assigned_user` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `Functional_Reporting`");
}
if (!col_exists($conn,'leads','empid')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `empid` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `assigned_user`");
}
if (!col_exists($conn,'leads','lead_type')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `lead_type` VARCHAR(20) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `empid`");
}
if (!col_exists($conn,'leads','pp_status')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `pp_status` VARCHAR(50) COLLATE utf8mb4_general_ci DEFAULT 'Pending'
        AFTER `lead_type`");
}
if (!col_exists($conn,'leads','opp_stage')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `opp_stage` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL
        AFTER `pp_status`");
}
if (!col_exists($conn,'leads','claim_date')) {
    @mysqli_query($conn, "ALTER TABLE `leads`
        ADD COLUMN `claim_date` DATETIME NULL DEFAULT NULL
        AFTER `created_at`");
}

/* Table to store round-robin state per lead_type (Fresh / Resycled) */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `lead_distribution_state` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_type` VARCHAR(20) COLLATE utf8mb4_general_ci NOT NULL,
  `current_index` INT UNSIGNED NOT NULL DEFAULT 0,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_lead_type` (`lead_type`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

/* ----------------- Load projects from CRMPLUS DB (Projects master) ----------------- */
$projects     = [];
$projects_err = null;

if (!isset($conn_crmplus) || !$conn_crmplus) {
    $projects_err = "CRM Projects database connection is not available.";
} else {
    $proj_sql = "
        SELECT
            id,
            title,
            listing_type,
            project_tag,
            project_name,
            city,
            region,
            location,
            address
        FROM `projects`
        WHERE is_active = 1
        ORDER BY id DESC
    ";
    $proj_res = @mysqli_query($conn_crmplus, $proj_sql);
    if ($proj_res) {
        while ($row = mysqli_fetch_assoc($proj_res)) {
            $projects[] = $row;
        }
        mysqli_free_result($proj_res);
    } else {
        $projects_err = "Could not load active projects from CRM.";
    }
}

/* Sample CSV download */
if (isset($_GET['sample']) && $_GET['sample'] === '1') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=leads_sample.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['contact_name','phone_number','email','lead_source']);
    fputcsv($out, ['John Doe', '+919876543210', 'john@example.com', 'Website']);
    fputcsv($out, ['Jane Roe', '9876543210', 'jane@example.com', 'Referral']);
    fclose($out);
    exit;
}

/* Handle upload */
$errors              = [];
$success             = null;
$inserted            = 0;
$skipped             = 0;
$rows_total          = 0;
$selected_project_id = 0;
$lead_type           = '';   // Fresh / Resycled for sticky form
$fresh_method        = '';   // team_leader / via_claim for Fresh

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $selected_project_id = isset($_POST['project_id']) ? (int)$_POST['project_id'] : 0;
    $lead_type           = isset($_POST['lead_type']) ? trim($_POST['lead_type']) : '';
    $fresh_method        = isset($_POST['fresh_method']) ? trim($_POST['fresh_method']) : '';
    $proj_selected       = null;

    /* Validate Lead Type */
    if ($lead_type !== 'Fresh' && $lead_type !== 'Resycled') {
        $errors[] = "Please choose a valid Lead Type (Fresh / Resycled).";
    }

    /* For Fresh, ensure method is selected (Team Leader / Via Claim) */
    if ($lead_type === 'Fresh' && $fresh_method !== '') {
        $fresh_method = ($fresh_method === 'team_leader' || $fresh_method === 'via_claim')
            ? $fresh_method
            : '';
    }
    if ($lead_type === 'Fresh' && $fresh_method === '') {
        $errors[] = "Please choose the method for Fresh leads (Team Leader / Via Claim).";
    }

    /* Validate Project selection */
    if ($selected_project_id <= 0) {
        $errors[] = "Please select a project before uploading leads.";
    } else {
        foreach ($projects as $p) {
            if ((int)$p['id'] === $selected_project_id) {
                $proj_selected = $p;
                break;
            }
        }
        if (!$proj_selected) {
            $errors[] = "Selected project was not found in the project list.";
        }
    }

    /* Validate file presence */
    if (empty($_FILES['leads_csv']) || $_FILES['leads_csv']['error'] !== UPLOAD_ERR_OK) {
        $errors[] = "Please choose a CSV file to upload (ensure size within server limits).";
    }

    /* Decide if we need auto-distribution */
    $assign_pool        = [];
    $assign_count       = 0;
    $emp_index          = 0;
    $needs_distribution = false;

    if ($lead_type === 'Resycled') {
        $needs_distribution = true;
    } elseif ($lead_type === 'Fresh' && $fresh_method === 'team_leader') {
        $needs_distribution = true;
    }

    /* Build employee pool only when distribution is needed */
    if (empty($errors) && $needs_distribution) {
        if (!isset($conn_ic) || !$conn_ic) {
            $errors[] = "Attendance/HR database connection is not available.";
        } else {
            if ($lead_type === 'Fresh') {
                $emp_q = mysqli_query(
                    $conn_ic,
                    "SELECT empid FROM hr_dump
                     WHERE fresh_leads = 'yes'
                       AND Status = 'Active'"
                );
            } else { // Resycled
                $emp_q = mysqli_query(
                    $conn_ic,
                    "SELECT empid FROM hr_dump
                     WHERE recycle_leads = 'yes'
                       AND Status = 'Active'"
                );
            }

            if ($emp_q && mysqli_num_rows($emp_q) > 0) {
                while ($er = mysqli_fetch_assoc($emp_q)) {
                    if (!empty($er['empid'])) {
                        $assign_pool[] = $er['empid'];
                    }
                }
                mysqli_free_result($emp_q);
                $assign_count = count($assign_pool);
            }

            if ($assign_count === 0) {
                if ($lead_type === 'Fresh') {
                    $errors[] = "No ACTIVE team leaders are configured for Fresh lead distribution.";
                } else {
                    $errors[] = "No ACTIVE employees are configured for Resycled lead distribution.";
                }
            }
        }
    }

    /* Load round-robin pointer from lead_distribution_state (only if distributing) */
    $current_index = 0;
    if (empty($errors) && $assign_count > 0 && $needs_distribution) {
        $lt_esc = mysqli_real_escape_string($conn, $lead_type);
        $s_sql  = "SELECT current_index FROM lead_distribution_state
                   WHERE lead_type = '{$lt_esc}'
                   LIMIT 1";
        $s_res  = mysqli_query($conn, $s_sql);
        if ($s_res && $s_row = mysqli_fetch_assoc($s_res)) {
            $current_index = (int)$s_row['current_index'];
        }
        if ($s_res) {
            mysqli_free_result($s_res);
        }
        if ($current_index < 0) {
            $current_index = 0;
        }
        if ($assign_count > 0) {
            $current_index = $current_index % $assign_count;
        }
        $emp_index = $current_index; // start assigning from this index
    }

    if (empty($errors)) {
        $fileTmp = $_FILES['leads_csv']['tmp_name'];
        if (($handle = fopen($fileTmp, 'r')) === false) {
            $errors[] = "Failed to open uploaded file.";
        } else {
            $header = fgetcsv($handle);
            if ($header === false) {
                $errors[] = "CSV appears to be empty.";
                fclose($handle);
            } else {
                /* Flexible header mapping */
                $map = [];
                foreach ($header as $i => $col) {
                    $colname = strtolower(trim((string)$col));
                    if (in_array($colname, ['contact_name','name','contact'], true)) {
                        $map['contact_name'] = $i;
                    } elseif (in_array($colname, ['phone_number','phone','mobile','msisdn'], true)) {
                        $map['phone_number'] = $i;
                    } elseif (in_array($colname, ['email','e-mail','mail'], true)) {
                        $map['email'] = $i;
                    } elseif (in_array($colname, ['lead_source','source','leadsource'], true)) {
                        $map['lead_source'] = $i;
                    }
                }

                $use_positional = (!isset($map['contact_name']) || !isset($map['phone_number']) || !isset($map['email']));
                if ($use_positional) {
                    rewind($handle);
                }

                /* Prepared insert with project + distribution columns (into ONB.leads) */
                $stmt = $conn->prepare("
                    INSERT INTO `leads`
                        (`contact_name`,
                         `phone_number`,
                         `email`,
                         `lead_source`,
                         `project_id`,
                         `project_title`,
                         `project_listing_type`,
                         `project_tag`,
                         `project_name`,
                         `project_city`,
                         `project_region`,
                         `project_location`,
                         `project_address`,
                         `Functional_Reporting`,
                         `assigned_user`,
                         `empid`,
                         `lead_type`,
                         `pp_status`,
                         `created_at`,
                         `claim_date`)
                    VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
                ");

                if ($stmt === false) {
                    $errors[] = "Prepare failed: " . h($conn->error);
                    fclose($handle);
                } else {
                    $conn->begin_transaction();

                    /* Static project fields (applied to every imported lead) */
                    $pid        = isset($proj_selected['id'])           ? (string)$proj_selected['id']           : '';
                    $ptitle     = mb_substr((string)($proj_selected['title']         ?? ''), 0, 191);
                    $plisting   = mb_substr((string)($proj_selected['listing_type']  ?? ''), 0, 100);
                    $ptag       = mb_substr((string)($proj_selected['project_tag']   ?? ''), 0, 100);
                    $pprojname  = mb_substr((string)($proj_selected['project_name']  ?? ''), 0, 191);
                    $pcity      = mb_substr((string)($proj_selected['city']          ?? ''), 0, 100);
                    $pregion    = mb_substr((string)($proj_selected['region']        ?? ''), 0, 100);
                    $plocation  = mb_substr((string)($proj_selected['location']      ?? ''), 0, 191);
                    $paddress   = mb_substr((string)($proj_selected['address']       ?? ''), 0, 255);

                    while (($row = fgetcsv($handle)) !== false) {
                        $rows_total++;

                        if ($use_positional) {
                            $name  = isset($row[0]) ? trim($row[0]) : '';
                            $phone = isset($row[1]) ? trim($row[1]) : '';
                            $email = isset($row[2]) ? trim($row[2]) : '';
                            $src   = isset($row[3]) ? trim($row[3]) : '';
                        } else {
                            $name  = isset($map['contact_name']) && isset($row[$map['contact_name']])
                                    ? trim($row[$map['contact_name']])
                                    : '';
                            $phone = isset($map['phone_number']) && isset($row[$map['phone_number']])
                                    ? trim($row[$map['phone_number']])
                                    : '';
                            $email = isset($map['email']) && isset($row[$map['email']])
                                    ? trim($row[$map['email']])
                                    : '';
                            $src   = (isset($map['lead_source']) && isset($row[$map['lead_source']]))
                                    ? trim($row[$map['lead_source']])
                                    : '';
                        }

                        /* Skip completely empty lines */
                        if ($name === '' && $phone === '' && $email === '' && $src === '') {
                            $skipped++;
                            continue;
                        }

                        /* Normalize + validate */
                        $phone_normalized = preg_replace('/[^\d+]/', '', $phone);
                        if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                            $skipped++;
                            continue;
                        }
                        $src = mb_substr($src ?? '', 0, 191);

                        /* Choose employee from pool (round-robin, only when we actually insert and need distribution) */
                        $distEmp = null;
                        if ($assign_count > 0 && $needs_distribution) {
                            $distEmp = $assign_pool[$emp_index % $assign_count];
                        }

                        /* Fill Functional_Reporting / assigned_user / empid as per Lead Type + method */
                        $Functional_Reporting = null;
                        $assigned_user_val     = null;
                        $empid_val             = null;

                        if ($lead_type === 'Fresh' && $fresh_method === 'team_leader' && $distEmp !== null) {
                            $Functional_Reporting = $distEmp;
                        } elseif ($lead_type === 'Resycled' && $distEmp !== null) {
                            $assigned_user_val = $distEmp;
                            $empid_val         = $distEmp;
                        }
                        // For Fresh + Via Claim: all three stay NULL by design.

                        $lead_type_val = $lead_type;
                        $pp_status_val = 'Pending'; // always Pending on upload

                        // Created/claim date (IST). For Resycled leads, claim_date = created_at
                        $created_at_val = date('Y-m-d H:i:s');
                        $claim_date_val = ($lead_type === 'Resycled') ? $created_at_val : null;

                        $stmt->bind_param(
                            'ssssssssssssssssssss',
                            $name,
                            $phone_normalized,
                            $email,
                            $src,
                            $pid,
                            $ptitle,
                            $plisting,
                            $ptag,
                            $pprojname,
                            $pcity,
                            $pregion,
                            $plocation,
                            $paddress,
                            $Functional_Reporting,
                            $assigned_user_val,
                            $empid_val,
                            $lead_type_val,
                            $pp_status_val,
                            $created_at_val,
                            $claim_date_val
                        );

                        $ok = $stmt->execute();
                        if ($ok) {
                            $inserted++;
                            if ($needs_distribution && $assign_count > 0) {
                                $emp_index++; // rotate pool only for successfully inserted rows when distributing
                            }
                        } else {
                            $skipped++;
                        }
                    }

                    $conn->commit();
                    $stmt->close();
                    fclose($handle);

                    /* Save new round-robin position for this lead_type, only if distribution was used */
                    if ($needs_distribution && $assign_count > 0) {
                        $new_index = $emp_index % $assign_count;
                        $lt_esc    = mysqli_real_escape_string($conn, $lead_type);
                        $ni        = (int)$new_index;
                        mysqli_query(
                            $conn,
                            "INSERT INTO lead_distribution_state (lead_type, current_index)
                             VALUES ('{$lt_esc}', {$ni})
                             ON DUPLICATE KEY UPDATE
                               current_index = {$ni},
                               updated_at = NOW()"
                        );
                    }

                    $success = "{$inserted} lead(s) inserted. {$skipped} row(s) skipped. {$rows_total} row(s) processed.";
                }
            }
        }
    }
}

/* Build summary/ warning text for Step 2 (shown before Step 3) */
$lead_summary_heading = '';
$lead_summary_body    = '';
$lead_summary_warn    = '';

if ($lead_type === 'Fresh') {
    if ($fresh_method === 'team_leader') {
        $lead_summary_heading = 'Lead Type: Fresh — Method: Team Leader';
        $lead_summary_body    = "Leads will be equally distributed among eligible ACTIVE team leaders configured for Fresh leads. For each lead, Functional_Reporting will be set to the selected empid.";
        $lead_summary_warn    = "Please confirm this before proceeding to Step 3 (Select CSV file).";
    } elseif ($fresh_method === 'via_claim') {
        $lead_summary_heading = 'Lead Type: Fresh — Method: Via Claim';
        $lead_summary_body    = "All leads will be inserted without auto-assignment. Functional_Reporting, assigned_user and empid will remain empty.";
        $lead_summary_warn    = "You must manually claim/assign these leads later in CRM. Confirm this before Step 3 (Select CSV file).";
    }
} elseif ($lead_type === 'Resycled') {
    $lead_summary_heading = 'Lead Type: Resycled';
    $lead_summary_body    = "Leads will be equally distributed among eligible ACTIVE employees configured for Resycled leads. For each lead, assigned_user and empid will be set to the selected empid, and claim_date will be set equal to created_at.";
    $lead_summary_warn    = "Please confirm this before proceeding to Step 3 (Select CSV file).";
}

/* ============================== THEME SECTIONS ============================== */
ob_start(); ?>
<style>
/* Page-specific tweaks that fit your CRM theme */
.csv-card .card-title { font-weight: 600; }
.csv-card .muted { color:#6c757d; font-size:.9rem; }
.csv-card .sample-box {
    background: var(--bs-light, #f6f7f9);
    border-radius: .5rem;
    padding: .75rem;
    font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace;
    white-space: pre-wrap;
}
.csv-card .divider {
    height:1px;
    background: rgba(0,0,0,.08);
    margin: .75rem 0;
}
.csv-card .projects-table-wrapper {
    max-height: 320px;
    overflow: auto;
    border: 1px solid rgba(0,0,0,.05);
    border-radius: .5rem;
}
.csv-card .lead-summary-alert .fw-semibold {
    font-size: .92rem;
}

/* Make all radio buttons darker */
.csv-card .form-check-input[type="radio"],
.csv-card input[type="radio"].form-check-input {
    accent-color: #343a40;      /* dark grey */
    border-color: #343a40;
}
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
<!-- Header / Breadcrumb -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Leads CSV Upload</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Upload</li>
    </ol>
  </div>
</div>

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card csv-card">
      <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-2">
        <div class="card-title">Import Leads (CSV) Against Project</div>
        <div class="d-flex gap-2">
          <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?sample=1" class="btn btn-light btn-wave">Download Sample</a>
        </div>
      </div>

      <div class="card-body">
        <?php if (!empty($errors)): ?>
          <div class="alert alert-danger">
            <strong>Upload Error</strong>
            <ul class="mb-0">
              <?php foreach ($errors as $e): ?>
                <li><?php echo h($e); ?></li>
              <?php endforeach; ?>
            </ul>
          </div>
        <?php endif; ?>

        <?php if ($success !== null): ?>
          <div class="alert alert-success">
            <strong>Result:</strong> <?php echo h($success); ?>
          </div>
        <?php endif; ?>

        <form method="post" enctype="multipart/form-data" class="row g-3">

          <!-- ===================== Step 1: Select Project ===================== -->
          <div class="col-12">
            <h6 class="mb-2">Step 1: Select Project</h6>
            <p class="mb-2 muted">
              First select the project from the Projects list.
              All uploaded leads will be linked to this project.
            </p>

            <?php if ($projects_err): ?>
              <div class="alert alert-warning mb-2">
                <?php echo h($projects_err); ?>
              </div>
            <?php elseif (empty($projects)): ?>
              <div class="alert alert-info mb-2">
                No active projects were found.
              </div>
            <?php else: ?>
              <div class="projects-table-wrapper mb-2">
                <table class="table table-sm table-hover align-middle mb-0">
                  <thead class="table-light">
                    <tr>
                      <th style="width:40px;">Select</th>
                      <th>ID</th>
                      <th>Title</th>
                      <th>Listing Type</th>
                      <th>Project Tag</th>
                      <th>Project Name</th>
                      <th>City</th>
                      <th>Region</th>
                      <th>Location</th>
                      <th>Address</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php foreach ($projects as $p): ?>
                      <?php
                        $pid = (int)$p['id'];
                        $checked = ($selected_project_id === $pid) ? 'checked' : '';
                      ?>
                      <tr>
                        <td>
                          <!-- Single selection (one project) using radio -->
                          <input type="radio"
                                 class="form-check-input"
                                 name="project_id"
                                 value="<?php echo $pid; ?>"
                                 <?php echo $checked; ?>>
                        </td>
                        <td><?php echo h($p['id']); ?></td>
                        <td><?php echo h($p['title']); ?></td>
                        <td><?php echo h($p['listing_type']); ?></td>
                        <td><?php echo h($p['project_tag']); ?></td>
                        <td><?php echo h($p['project_name']); ?></td>
                        <td><?php echo h($p['city']); ?></td>
                        <td><?php echo h($p['region']); ?></td>
                        <td><?php echo h($p['location']); ?></td>
                        <td><?php echo h($p['address']); ?></td>
                      </tr>
                    <?php endforeach; ?>
                  </tbody>
                </table>
              </div>
              <p class="mb-0 muted">
                Only one project can be selected. All rows in the CSV will be attached to this selected project.
              </p>
            <?php endif; ?>
          </div>

          <div class="col-12"><div class="divider"></div></div>

          <!-- ===================== Step 2: Lead Type ===================== -->
          <div class="col-12 col-md-4">
            <label class="form-label">Step 2: Lead Type</label>
            <select name="lead_type" class="form-select" required>
              <option value="">-- Select --</option>
              <option value="Fresh"    <?php echo ($lead_type === 'Fresh'    ? 'selected' : ''); ?>>Fresh</option>
              <option value="Resycled" <?php echo ($lead_type === 'Resycled' ? 'selected' : ''); ?>>Resycled</option>
            </select>
            <div class="form-text">
              Behaviour depends on Lead Type and, for Fresh leads, the method selected below.
            </div>

            <!-- Fresh sub-options: Team Leader / Via Claim -->
            <div id="fresh-method-wrapper" class="mt-2 <?php echo ($lead_type === 'Fresh' ? '' : 'd-none'); ?>">
              <label class="form-label mb-1">Fresh Lead Assignment</label>
              <div class="form-check">
                <input class="form-check-input"
                       type="radio"
                       name="fresh_method"
                       id="fresh_team_leader"
                       value="team_leader"
                       <?php echo ($lead_type === 'Fresh' && $fresh_method === 'team_leader') ? 'checked' : ''; ?>>
                <label class="form-check-label" for="fresh_team_leader">
                  Team Leader &mdash; auto distribute to eligible ACTIVE team leaders and set <code>Functional_Reporting</code>.
                </label>
              </div>
              <div class="form-check mt-1">
                <input class="form-check-input"
                       type="radio"
                       name="fresh_method"
                       id="fresh_via_claim"
                       value="via_claim"
                       <?php echo ($lead_type === 'Fresh' && $fresh_method === 'via_claim') ? 'checked' : ''; ?>>
                <label class="form-check-label" for="fresh_via_claim">
                  Via Claim &mdash; insert leads without any assignment. <code>Functional_Reporting</code>, <code>assigned_user</code> and <code>empid</code> remain empty.
                </label>
              </div>
              <div class="form-text mt-1">
                Choose one method for Fresh leads before moving to Step 3.
              </div>
            </div>

            <!-- Summary + warning (always before Step 3) -->
            <?php if ($lead_summary_heading !== ''): ?>
              <div class="alert alert-warning lead-summary-alert mt-3 mb-0 py-2 px-3">
                <div class="fw-semibold">
                  <?php echo h($lead_summary_heading); ?>
                </div>
                <?php if ($lead_summary_body !== ''): ?>
                  <div class="small mb-1">
                    <?php echo h($lead_summary_body); ?>
                  </div>
                <?php endif; ?>
                <?php if ($lead_summary_warn !== ''): ?>
                  <div class="small text-danger fw-semibold mb-0">
                    <?php echo h($lead_summary_warn); ?>
                  </div>
                <?php endif; ?>
              </div>
            <?php endif; ?>
          </div>

          <!-- ===================== Step 3: Upload CSV ===================== -->
          <div class="col-12 col-md-4">
            <label class="form-label">Step 3: Select CSV file</label>
            <input type="file" class="form-control" name="leads_csv" accept=".csv,text/csv" required>
            <div class="form-text">
              Allowed: .csv — UTF-8 recommended. Ensure Step 2 (Lead Type &amp; method) is correctly selected before upload.
            </div>
          </div>

          <div class="col-12 col-md-4 d-flex align-items-end">
            <button type="submit" class="btn btn-primary btn-wave w-100">
              Upload &amp; Insert Leads
            </button>
          </div>
        </form>

        <div class="csv-help mt-3">
          <div class="divider"></div>
          <div class="row g-3">
            <div class="col-12 col-lg-6">
              <h6 class="mb-2">Accepted Columns</h6>
              <p class="mb-2 muted">
                <code>contact_name</code>, <code>phone_number</code>, <code>email</code>, <code>lead_source</code>
              </p>
              <p class="mb-0 muted">
                Header names are flexible (e.g., <em>name</em>, <em>phone</em>, <em>source</em>) or omit header and use positional order.
                All imported rows will be stored against the selected project and distributed (or left unassigned) as per Lead Type and selected method.
              </p>
            </div>
            <div class="col-12 col-lg-6">
              <h6 class="mb-2">Example CSV</h6>
              <div class="sample-box">
contact_name,phone_number,email,lead_source
John Doe,+919876543210,john@example.com,Website
Jane Roe,9876543210,jane@example.com,Referral
              </div>
            </div>
          </div>

          <div class="divider"></div>
          <ul class="mb-0 muted">
            <li>Phone numbers are normalized to digits with optional leading <code>+</code>.</li>
            <li>Invalid emails (if present) are skipped; blank lines are ignored.</li>
            <li>Connection collation is enforced as <code>utf8mb4_general_ci</code>.</li>
          </ul>
        </div>
      </div>
    </div>
  </div>
</div>
<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
document.addEventListener('DOMContentLoaded', function () {
  var leadTypeSelect   = document.querySelector('select[name="lead_type"]');
  var freshMethodWrap  = document.getElementById('fresh-method-wrapper');

  function syncFreshMethodVisibility() {
    if (!leadTypeSelect || !freshMethodWrap) return;
    if (leadTypeSelect.value === 'Fresh') {
      freshMethodWrap.classList.remove('d-none');
    } else {
      freshMethodWrap.classList.add('d-none');
      var radios = freshMethodWrap.querySelectorAll('input[name="fresh_method"]');
      radios.forEach(function (r) { r.checked = false; });
    }
  }

  if (leadTypeSelect && freshMethodWrap) {
    leadTypeSelect.addEventListener('change', syncFreshMethodVisibility);
    syncFreshMethodVisibility();
  }
});
</script>
<?php $scripts = ob_get_clean(); ?>

<?php
/* Render using the same base layout as other CRM pages (ensures sidebar shows) */
include 'layouts/base.php';
