<?php
/* ========================================================================
 * File   : mail_integration_builder.php
 * Purpose: Generate CLEAN & SAFE ZeptoMail snippet for ANY PHP page/button.
 * Notes:
 *   - Auto-suggest email/name expressions from base PHP code.
 *   - Supports custom ZeptoMail templates.
 *   - Generates perfect placeholder mapping without breaking PHP.
 *   - Prevents ALL previous errors (NULL warnings, undefined $status, etc.)
 * ====================================================================== */

@session_start();
mb_internal_encoding('UTF-8');

/* ---------- simple HTML escaper ---------- */
function h($v) {
    return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8');
}

/* ---------- read POST data ---------- */
$base_code       = isset($_POST['base_code'])       ? (string)$_POST['base_code']       : '';
$to_email_expr   = isset($_POST['to_email_expr'])   ? trim((string)$_POST['to_email_expr'])   : '';
$to_name_expr    = isset($_POST['to_name_expr'])    ? trim((string)$_POST['to_name_expr'])    : '';
$template_key    = isset($_POST['template_key'])    ? trim((string)$_POST['template_key'])    : 'employee_onb_rejected';
$placeholders_in = isset($_POST['tpl_placeholders'])? (string)$_POST['tpl_placeholders']      : '';

$generated_snippet = '';
$include_line      = "require_once __DIR__ . '/zeptomail_helper.php';";

/* ---------- auto-suggest extraction from base code ---------- */
$emailSuggestions = [];
$nameSuggestions  = [];

if ($base_code !== '') {

    // EMAIL patterns
    $patternsEmail = [
        '/\$[A-Za-z_][A-Za-z0-9_]*\[[\'"](email|Email|EMAIL)[\'"]\]/',
        '/\$[A-Za-z_][A-Za-z0-9_]*email\b/i',
        '/\$[A-Za-z_][A-Za-z0-9_]*mail\b/i',
    ];
    foreach ($patternsEmail as $re) {
        if (preg_match_all($re, $base_code, $m)) {
            foreach ($m[0] as $expr) {
                $expr = trim($expr);
                if ($expr !== '' && !in_array($expr, $emailSuggestions, true)) {
                    $emailSuggestions[] = $expr;
                }
            }
        }
    }

    // NAME patterns
    $patternsName = [
        '/\$[A-Za-z_][A-Za-z0-9_]*\[[\'"](first_name|firstname|name|full_name|fullname|surname)[\'"]\]/i',
        '/\$[A-Za-z_][A-Za-z0-9_]*name\b/i',
    ];
    foreach ($patternsName as $re) {
        if (preg_match_all($re, $base_code, $m)) {
            foreach ($m[0] as $expr) {
                $expr = trim($expr);
                if ($expr !== '' && !in_array($expr, $nameSuggestions, true)) {
                    $nameSuggestions[] = $expr;
                }
            }
        }
    }
}

/* ---------- build snippet when requested ---------- */
if (!empty($_POST['build'])) {

    if ($to_email_expr === '' && !empty($emailSuggestions)) {
        $to_email_expr = $emailSuggestions[0];
    }
    if ($to_name_expr === '' && !empty($nameSuggestions)) {
        $to_name_expr = $nameSuggestions[0];
    }

    if ($to_email_expr === '') {
        $to_email_expr = "\$app['email']";
    }
    if ($to_name_expr === '') {
        $to_name_expr = "trim((\$app['first_name'] ?? '') . ' ' . (\$app['surname'] ?? ''))";
    }
    if ($template_key === '') {
        $template_key = 'employee_onb_rejected';
    }

    /* ---------- Parse placeholders ---------- */
    $placeholderLines = preg_split('/\R+/', $placeholders_in);
    $parsedPlaceholders = [];

    if ($placeholders_in === '') {
        // Default placeholders
        $parsedPlaceholders[] = ['{{NAME}}', "\$app['first_name']"];
        $parsedPlaceholders[] = ['{{STATUS}}', '$status'];
    } else {
        foreach ($placeholderLines as $line) {
            $line = trim($line);
            if ($line === '') continue;

            if (strpos($line, '=>') !== false) {
                [$k, $expr] = explode('=>', $line, 2);
            } elseif (strpos($line, '=') !== false) {
                [$k, $expr] = explode('=', $line, 2);
            } else {
                $k = $line;
                // auto-generate variable
                $expr = '$' . preg_replace('/[^A-Za-z0-9_]/', '_', $line);
            }

            $k    = trim($k);
            $expr = trim($expr);

            // Remove quotes around key
            if ((str_starts_with($k, "'") && str_ends_with($k, "'")) ||
                (str_starts_with($k, '"') && str_ends_with($k, '"'))) {
                $k = substr($k, 1, -1);
            }

            // skip NULL / blank
            if ($k === '' || $expr === '' || $k === null || $expr === null) {
                continue;
            }

            $parsedPlaceholders[] = [$k, $expr];
        }
    }

    /* ---------- Build clean ZeptoMail snippet ---------- */
    $out  = "";
    $out .= "// 1) Add this near the top of your PHP file (after config include)\n";
    $out .= $include_line . ";\n\n";

    $out .= "// 2) Paste this block INSIDE your button action\n";
    $out .= "//    e.g. inside:  if (\$act === 'reject') { ... }\n\n";

    // Prevent undefined variable
    $out .= "\$status = \$status ?? 'Pending';\n\n";

    $out .= "\$__z_placeholders = [\n";
    foreach ($parsedPlaceholders as [$k, $expr]) {
        $safeKey = str_replace("'", "\\'", $k);
        $out .= "    '{$safeKey}' => {$expr},\n";
    }
    $out .= "];\n\n";

    $out .= "\$__z_result = zepto_send_template(\n";
    $out .= "    {$to_email_expr}, // To Email\n";
    $out .= "    {$to_name_expr},  // To Name\n";
    $out .= "    '" . str_replace("'", "\\'", $template_key) . "', // Template key\n";
    $out .= "    \$__z_placeholders\n";
    $out .= ");\n\n";

    $out .= "if (empty(\$__z_result['success'])) {\n";
    $out .= "    // error_log('ZeptoMail error: ' . (\$__z_result['error'] ?? 'unknown'));\n";
    $out .= "}\n";

    $generated_snippet = $out;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Mail Integration Builder – ZeptoMail Snippet</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body{font-family:system-ui,-apple-system,BlinkMacSystemFont,"Segoe UI",sans-serif;background:#f4f5fb;margin:0;padding:0}
        .wrap{max-width:1100px;margin:24px auto;padding:0 16px 32px}
        h1{margin:0 0 12px;font-size:22px}
        .card{background:#fff;border-radius:12px;box-shadow:0 10px 30px rgba(15,23,42,.08);padding:16px 18px;margin-bottom:16px}
        textarea,input{font-family:inherit;font-size:13px}
        textarea{width:100%;min-height:160px;border-radius:8px;border:1px solid #d4d7e5;padding:8px 10px;resize:vertical}
        input[type="text"]{width:100%;border-radius:8px;border:1px solid #d4d7e5;padding:7px 9px}
        label{font-size:13px;font-weight:600;margin-bottom:4px;display:block;color:#111827}
        .btn-primary{border:0;border-radius:999px;background:linear-gradient(90deg,#6366F1,#22C55E);color:#fff;padding:8px 18px;font-weight:700;font-size:13px;cursor:pointer}
        pre{background:#020617;color:#e5e7eb;border-radius:10px;padding:10px 12px;overflow:auto;font-size:12px}
        .mono{font-family:Menlo,Monaco,Consolas,"Courier New",monospace}
        .chips{margin-top:4px;display:flex;flex-wrap:wrap;gap:6px}
        .chip{border:0;border-radius:999px;background:#e5e7ff;color:#111827;font-size:11px;padding:4px 9px;cursor:pointer}
    </style>
</head>
<body>
<div class="wrap">
    <h1>Mail Integration Builder <span class="small">– ZeptoMail</span></h1>

    <form method="post" autocomplete="off">

        <!-- STEP 1 -->
        <div class="card">
            <strong>STEP 1 — Paste Base PHP Code (optional)</strong>
            <textarea name="base_code" placeholder="$app = mysqli_fetch_assoc($q); ..."><?php echo h($base_code); ?></textarea>
        </div>

        <!-- STEP 2 -->
        <div class="card">
            <strong>STEP 2 — Configure ZeptoMail</strong>

            <label>To Email (PHP Expression)</label>
            <input type="text" id="to_email_expr" name="to_email_expr" value="<?php echo h($to_email_expr); ?>">
            <?php if ($emailSuggestions): ?>
                <div class="chips">
                    <?php foreach ($emailSuggestions as $s): ?>
                        <button type="button" class="chip" data-target="to_email_expr"><?= h($s) ?></button>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <label style="margin-top:12px;">To Name (PHP Expression)</label>
            <input type="text" id="to_name_expr" name="to_name_expr" value="<?php echo h($to_name_expr); ?>">
            <?php if ($nameSuggestions): ?>
                <div class="chips">
                    <?php foreach ($nameSuggestions as $s): ?>
                        <button type="button" class="chip" data-target="to_name_expr"><?= h($s) ?></button>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <label style="margin-top:12px;">Template Key</label>
            <input type="text" id="template_key" name="template_key" value="<?php echo h($template_key); ?>">

            <label style="margin-top:12px;">Placeholders</label>
            <textarea name="tpl_placeholders" placeholder="{{NAME}} => $app['first_name']&#10;{{STATUS}} => $status"><?php echo h($placeholders_in); ?></textarea>

            <button type="submit" name="build" value="1" class="btn-primary" style="margin-top:14px;">Build Snippet</button>
        </div>

        <!-- STEP 3 -->
        <div class="card">
            <strong>STEP 3 — Copy Generated PHP Code</strong>
            <pre class="mono"><?php echo $generated_snippet ? h($generated_snippet) : "// Fill STEP 2 and click Build."; ?></pre>
        </div>
    </form>
</div>

<script>
document.querySelectorAll('.chip').forEach(chip => {
    chip.onclick = () => {
        let t = chip.getAttribute('data-target');
        let input = document.getElementById(t);
        input.value = chip.textContent.trim();
        input.focus();
    };
});
</script>

</body>
</html>
