<?php

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

include('../config.php'); // mysqli connections (including $conn_sa)
require_once 'includes/check-permission.php';

$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_user);

/* -----------------------------------------------------------------------------
 * AJAX HANDLER (Add / Update / Delete)
 * -------------------------------------------------------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax']) && $_POST['ajax'] === '1') {
    header('Content-Type: application/json; charset=utf-8');

    $action    = isset($_POST['action']) ? trim($_POST['action']) : '';
    $id        = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $mainTitle = isset($_POST['main_title']) ? trim($_POST['main_title']) : '';
    $pageName  = isset($_POST['page_name']) ? trim($_POST['page_name']) : '';
    $label     = isset($_POST['label']) ? trim($_POST['label']) : '';
    $sortOrder = isset($_POST['sort_order']) ? (int)$_POST['sort_order'] : 0;

    // Basic validation
    if (in_array($action, ['add', 'update'], true)) {
        if ($mainTitle === '' || $pageName === '' || $label === '') {
            echo json_encode([
                'status'  => 'error',
                'message' => 'Main-Title, Page and Label are required.'
            ]);
            exit;
        }
    }

    if ($action === 'add') {
        $sql = "INSERT INTO role_permission_master (main_title, page_name, label, sort_order, is_active)
                VALUES (?, ?, ?, ?, 1)";
        $stmt = $conn_user->prepare($sql);
        if (!$stmt) {
            echo json_encode(['status' => 'error', 'message' => $conn_user->error]);
            exit;
        }
        $stmt->bind_param('sssi', $mainTitle, $pageName, $label, $sortOrder);
        if ($stmt->execute()) {
            echo json_encode(['status' => 'success', 'message' => 'Permission added successfully.']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Insert failed.']);
        }
        $stmt->close();
        exit;
    }

    if ($action === 'update') {
        if ($id <= 0) {
            echo json_encode(['status' => 'error', 'message' => 'Invalid record ID.']);
            exit;
        }
        $sql = "UPDATE role_permission_master
                SET main_title = ?, page_name = ?, label = ?, sort_order = ?
                WHERE id = ?";
        $stmt = $conn_user->prepare($sql);
        if (!$stmt) {
            echo json_encode(['status' => 'error', 'message' => $conn_user->error]);
            exit;
        }
        $stmt->bind_param('sssii', $mainTitle, $pageName, $label, $sortOrder, $id);
        if ($stmt->execute()) {
            echo json_encode(['status' => 'success', 'message' => 'Permission updated successfully.']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Update failed.']);
        }
        $stmt->close();
        exit;
    }

    if ($action === 'delete') {
        if ($id <= 0) {
            echo json_encode(['status' => 'error', 'message' => 'Invalid record ID.']);
            exit;
        }
        $sql = "DELETE FROM role_permission_master WHERE id = ?";
        $stmt = $conn_user->prepare($sql);
        if (!$stmt) {
            echo json_encode(['status' => 'error', 'message' => $conn_user->error]);
            exit;
        }
        $stmt->bind_param('i', $id);
        if ($stmt->execute()) {
            echo json_encode(['status' => 'success', 'message' => 'Permission deleted successfully.']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Delete failed.']);
        }
        $stmt->close();
        exit;
    }

    echo json_encode(['status' => 'error', 'message' => 'Invalid action.']);
    exit;
}

/* -----------------------------------------------------------------------------
 * FETCH DATA FOR LIST
 * -------------------------------------------------------------------------- */
$permRows = [];
$sqlList = "SELECT id, main_title, page_name, label, sort_order, is_active
            FROM role_permission_master
            ORDER BY main_title ASC, sort_order ASC, label ASC";
$res = $conn_user->query($sqlList);
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $permRows[] = $row;
    }
}
?>

<?php ob_start(); ?>
<link rel="stylesheet" href="<?= $baseUrl; ?>/assets/libs/prismjs/themes/prism-coy.min.css">
<style>
    .badge-main-title {
        font-size: 0.775rem;
        padding: 0.35rem 0.6rem;
        border-radius: 999px;
    }
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
<div class="container py-5">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="mb-0">Permission Master</h1>
        <button type="button" class="btn btn-primary" id="addPermissionBtn">
            + Add Permission
        </button>
    </div>

    <div class="card">
        <div class="card-body p-0">
            <table class="table table-hover table-bordered mb-0 align-middle">
                <thead class="table-light">
                    <tr>
                        <th style="width:60px;">#</th>
                        <th>Main-Title</th>
                        <th>Assigned Page (PHP)</th>
                        <th>Label</th>
                        <th style="width:100px;">Sort</th>
                        <th style="width:150px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                <?php if (!empty($permRows)): ?>
                    <?php $i = 1; foreach ($permRows as $r): ?>
                        <tr>
                            <td><?= $i++; ?></td>
                            <td>
                                <span class="badge bg-secondary badge-main-title">
                                    <?= htmlspecialchars($r['main_title']); ?>
                                </span>
                            </td>
                            <td><code><?= htmlspecialchars($r['page_name']); ?></code></td>
                            <td><?= htmlspecialchars($r['label']); ?></td>
                            <td><?= (int)$r['sort_order']; ?></td>
                            <td>
                                <button
                                    type="button"
                                    class="btn btn-sm btn-outline-primary me-1 btn-edit-perm"
                                    data-id="<?= (int)$r['id']; ?>"
                                    data-main-title="<?= htmlspecialchars($r['main_title'], ENT_QUOTES); ?>"
                                    data-page-name="<?= htmlspecialchars($r['page_name'], ENT_QUOTES); ?>"
                                    data-label="<?= htmlspecialchars($r['label'], ENT_QUOTES); ?>"
                                    data-sort-order="<?= (int)$r['sort_order']; ?>"
                                >Edit</button>
                                <button
                                    type="button"
                                    class="btn btn-sm btn-outline-danger btn-delete-perm"
                                    data-id="<?= (int)$r['id']; ?>"
                                >Delete</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="6" class="text-center text-muted py-4">
                            No permission mappings found. Click "Add Permission" to create one.
                        </td>
                    </tr>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal: Add / Edit Permission -->
<div class="modal fade" id="permModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <form class="modal-content" id="permForm">
            <div class="modal-header">
                <h5 class="modal-title" id="permModalTitle">Add Permission</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" name="ajax" value="1">
                <input type="hidden" name="action" id="permAction" value="add">
                <input type="hidden" name="id" id="permId" value="0">

                <div class="mb-3">
                    <label class="form-label">Main-Title (Section)</label>
                    <input type="text" class="form-control" name="main_title" id="mainTitleInput"
                           placeholder="e.g., Onboarding, AMS, DST" required>
                    <div class="form-text">
                        This groups pages like:
                        <strong>Onboarding</strong> =&gt; onb-dashboard.php, onb-approved-application.php, etc.
                    </div>
                </div>

                <div class="mb-3">
                    <label class="form-label">Assigned Page (PHP File)</label>
                    <input type="text" class="form-control" name="page_name" id="pageNameInput"
                           placeholder="e.g., onb-dashboard.php" required>
                    <div class="form-text">
                        Use only the file name, same as used in roles permissions JSON.
                    </div>
                </div>

                <div class="mb-3">
                    <label class="form-label">Label (Shown in UI)</label>
                    <input type="text" class="form-control" name="label" id="labelInput"
                           placeholder="e.g., Dashboard" required>
                </div>

                <div class="mb-3">
                    <label class="form-label">Sort Order (Optional)</label>
                    <input type="number" class="form-control" name="sort_order" id="sortOrderInput"
                           placeholder="0" value="0">
                    <div class="form-text">
                        Lower numbers appear first within the same Main-Title.
                    </div>
                </div>

            </div>
            <div class="modal-footer">
                <button type="submit" class="btn btn-success">Save</button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            </div>
        </form>
    </div>
</div>
<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const permModalEl   = document.getElementById('permModal');
    const permModal     = new bootstrap.Modal(permModalEl);
    const permForm      = document.getElementById('permForm');
    const permActionEl  = document.getElementById('permAction');
    const permIdEl      = document.getElementById('permId');
    const mainTitleEl   = document.getElementById('mainTitleInput');
    const pageNameEl    = document.getElementById('pageNameInput');
    const labelEl       = document.getElementById('labelInput');
    const sortOrderEl   = document.getElementById('sortOrderInput');
    const permModalTitleEl = document.getElementById('permModalTitle');

    // Add new
    document.getElementById('addPermissionBtn').addEventListener('click', function () {
        permActionEl.value = 'add';
        permIdEl.value     = '0';
        permModalTitleEl.textContent = 'Add Permission';

        mainTitleEl.value = '';
        pageNameEl.value  = '';
        labelEl.value     = '';
        sortOrderEl.value = '0';

        permModal.show();
    });

    // Edit existing
    document.querySelectorAll('.btn-edit-perm').forEach(function (btn) {
        btn.addEventListener('click', function () {
            const id         = this.getAttribute('data-id');
            const mainTitle  = this.getAttribute('data-main-title');
            const pageName   = this.getAttribute('data-page-name');
            const label      = this.getAttribute('data-label');
            const sortOrder  = this.getAttribute('data-sort-order') || '0';

            permActionEl.value = 'update';
            permIdEl.value     = id;
            permModalTitleEl.textContent = 'Edit Permission';

            mainTitleEl.value = mainTitle || '';
            pageNameEl.value  = pageName || '';
            labelEl.value     = label || '';
            sortOrderEl.value = sortOrder;

            permModal.show();
        });
    });

    // Delete
    document.querySelectorAll('.btn-delete-perm').forEach(function (btn) {
        btn.addEventListener('click', function () {
            const id = this.getAttribute('data-id');
            if (!id) return;

            if (!confirm('Are you sure you want to delete this permission mapping?')) {
                return;
            }

            const fd = new FormData();
            fd.append('ajax', '1');
            fd.append('action', 'delete');
            fd.append('id', id);

            fetch('permission-master.php', {
                method: 'POST',
                body: fd
            })
            .then(res => res.json())
            .then(res => {
                if (res.status === 'success') {
                    location.reload();
                } else {
                    alert(res.message || 'Delete failed');
                }
            })
            .catch(err => {
                console.error(err);
                alert('Error deleting record');
            });
        });
    });

    // Save (Add/Update)
    permForm.addEventListener('submit', function (e) {
        e.preventDefault();

        const fd = new FormData(permForm);

        fetch('permission-master.php', {
            method: 'POST',
            body: fd
        })
        .then(res => res.json())
        .then(res => {
            if (res.status === 'success') {
                permModal.hide();
                location.reload();
            } else {
                alert(res.message || 'Save failed');
            }
        })
        .catch(err => {
            console.error(err);
            alert('Error saving data');
        });
    });
});
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
