<?php
/* =============================================================================
 * File   : pif-reviewer-interview.php
 * Goal   : Reviewer Interview Page (Decision + Remarks Only)
 * Stack  : Core PHP + MySQLi, DB via ../config.php, COLLATE=utf8mb4_general_ci
 *
 * URL    : pif-reviewer-interview.php?id={idpif}&empid={empid}
 *
 * Logic  :
 *  - Uses central config.php
 *      * $conn = intbroriser_onboarding (user_pif)
 *      * $hrd  = intbroriser_hrd (hr_dump)
 *  - Valid link only for 3 days from user_pif.st1_date.
 *  - Finds reviewer in user_pif.reviewer_details (JSON) by EmpID.
 *      * If found -> update that reviewer.
 *      * If not found -> create a new reviewer entry using hr_dump.
 *  - Reviewer can only enter Decision + Remarks.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/../config.php';

/* Ensure charset on main connection */
if (isset($conn) && $conn instanceof mysqli) {
    mysqli_set_charset($conn, 'utf8mb4');
    @mysqli_query($conn, "SET NAMES 'utf8mb4'");
    @mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");
} else {
    http_response_code(500);
    die('Database connection ($conn) not found. Check config.php');
}

/* Helper: safe output */
function e($str)
{
    return htmlspecialchars((string)$str, ENT_QUOTES, 'UTF-8');
}

/* --------------------------------------------------------------------------
 * Read parameters
 * ----------------------------------------------------------------------- */
$idpif = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$empid = isset($_GET['empid']) ? trim($_GET['empid']) : '';

if ($idpif <= 0 || $empid === '') {
    http_response_code(400);
    die('Invalid link. Missing PIF ID or EmpID.');
}

/* --------------------------------------------------------------------------
 * Fetch PIF row
 * ----------------------------------------------------------------------- */
$pif = null;
$sqlPif = "SELECT * FROM user_pif WHERE idpif = ? LIMIT 1";

if ($stmt = $conn->prepare($sqlPif)) {
    $stmt->bind_param('i', $idpif);
    $stmt->execute();
    $res = $stmt->get_result();
    $pif = $res->fetch_assoc();
    $stmt->close();
}

if (!$pif) {
    http_response_code(404);
    die('PIF record not found.');
}

// Fallback to query parameters when HR fields are still unsaved in DB
$appointedOverride = isset($_GET['appointed_as']) ? trim($_GET['appointed_as']) : '';
$evalOverride      = isset($_GET['eval_r1_hr']) ? trim($_GET['eval_r1_hr']) : '';

if ($appointedOverride !== '' && empty($pif['appointed_as'])) {
    $pif['appointed_as'] = $appointedOverride;
}

if ($evalOverride !== '' && empty($pif['eval_r1_hr'])) {
    $pif['eval_r1_hr'] = $evalOverride;
}

/* --------------------------------------------------------------------------
 * 3-day validity based on st1_date (HR evaluation date)
 * ----------------------------------------------------------------------- */
$linkExpired = false;
if (!empty($pif['st1_date'])) {
    $st1Ts = strtotime($pif['st1_date']);
    if ($st1Ts !== false) {
        $nowTs      = time();
        $diffSecs   = $nowTs - $st1Ts;
        $threeDays  = 3 * 24 * 60 * 60;
        if ($diffSecs > $threeDays) {
            $linkExpired = true;
        }
    }
}

if ($linkExpired) {
    echo 'Interview link expired (more than 3 days from HR evaluation date).';
    exit;
}

/* --------------------------------------------------------------------------
 * Decode reviewer_details JSON & locate reviewer by EmpID
 * ----------------------------------------------------------------------- */
$reviewerDetails = [];
$reviewerIndex   = null;

if (!empty($pif['reviewer_details'])) {
    $tmp = json_decode($pif['reviewer_details'], true);
    if (is_array($tmp)) {
        $reviewerDetails = $tmp;
    }
}

if (!empty($reviewerDetails)) {
    foreach ($reviewerDetails as $idx => $rev) {
        // Compare as string to avoid type mismatch
        $revEmp = isset($rev['empid']) ? (string)$rev['empid'] : '';
        if ($revEmp !== '' && strcasecmp($revEmp, $empid) === 0) {
            $reviewerIndex = $idx;
            break;
        }
    }
}

/* --------------------------------------------------------------------------
 * If reviewer not present in JSON, try to build from hr_dump (HR DB)
 * ----------------------------------------------------------------------- */
$reviewerName = '';
$reviewerTeam = '';
$reviewerDecision = '';
$reviewerRemarks  = '';

if ($reviewerIndex !== null) {
    // Existing reviewer entry
    $rev = $reviewerDetails[$reviewerIndex];
    $reviewerName     = isset($rev['name']) ? (string)$rev['name'] : '';
    $reviewerTeam     = isset($rev['team']) ? (string)$rev['team'] : '';
    $reviewerDecision = isset($rev['decision']) ? (string)$rev['decision'] : '';
    $reviewerRemarks  = isset($rev['remarks']) ? (string)$rev['remarks'] : '';
} else {
    // Build from HR dump via $hrd (preferred) or $conn as fallback
    $dbHr = null;
    if (isset($hrd) && $hrd instanceof mysqli) {
        $dbHr = $hrd;
    } else {
        $dbHr = $conn;
    }

    if ($dbHr instanceof mysqli) {
        $sqlHr = "
            SELECT Employee_Full_Name, Team_name
            FROM hr_dump
            WHERE empid = ?
            LIMIT 1
        ";
        if ($stmt = $dbHr->prepare($sqlHr)) {
            $stmt->bind_param('s', $empid);
            $stmt->execute();
            $resHr = $stmt->get_result();
            if ($rowHr = $resHr->fetch_assoc()) {
                $reviewerName = (string)$rowHr['Employee_Full_Name'];
                $reviewerTeam = (string)$rowHr['Team_name'];
            }
            $stmt->close();
        }
    }

    // If still empty, fallback to EmpID only
    if ($reviewerName === '') {
        $reviewerName = 'Reviewer EmpID ' . $empid;
    }
}

/* --------------------------------------------------------------------------
 * Handle POST (Reviewer Decision + Remarks)
 * ----------------------------------------------------------------------- */
$successMsg = '';
$errorMsg   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $decision = isset($_POST['decision']) ? trim($_POST['decision']) : '';
    $remarks  = isset($_POST['remarks']) ? trim($_POST['remarks']) : '';

    if ($decision === '') {
        $errorMsg = 'Please select your decision.';
    }

    if ($errorMsg === '') {
        // If reviewer entry did not exist earlier, append a new one
        if ($reviewerIndex === null) {
            $reviewerDetails[] = [
                'empid'    => $empid,
                'name'     => $reviewerName,
                'team'     => $reviewerTeam,
                'decision' => $decision,
                'remarks'  => $remarks,
            ];
        } else {
            // Update existing entry
            $reviewerDetails[$reviewerIndex]['empid']    = $empid;
            $reviewerDetails[$reviewerIndex]['name']     = $reviewerName;
            $reviewerDetails[$reviewerIndex]['team']     = $reviewerTeam;
            $reviewerDetails[$reviewerIndex]['decision'] = $decision;
            $reviewerDetails[$reviewerIndex]['remarks']  = $remarks;
        }

        $json = json_encode($reviewerDetails);

        $updSql = "UPDATE user_pif SET reviewer_details = ? WHERE idpif = ? LIMIT 1";
        if ($stmt = $conn->prepare($updSql)) {
            $stmt->bind_param('si', $json, $idpif);
            if ($stmt->execute()) {
                $successMsg = 'Your decision has been submitted successfully.';
                $reviewerDecision = $decision;
                $reviewerRemarks  = $remarks;
            } else {
                $errorMsg = 'Database error while saving your decision. (' . $stmt->error . ')';
            }
            $stmt->close();
        } else {
            $errorMsg = 'Unable to prepare update statement. (' . $conn->error . ')';
        }
    }
}

/* --------------------------------------------------------------------------
 * Build base URL for display
 * ----------------------------------------------------------------------- */
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://')
    . $_SERVER['HTTP_HOST']
    . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));

$resumeBasePath = $baseUrl . '/pif-uploads';
$resumeUrl      = '';

if (!empty($pif['resume'])) {
    $resumeUrl = $resumeBasePath . '/' . rawurlencode($pif['resume']);
}

$ratingsMap = [
    'r1_qual'       => 'Qualifications',
    'r1_knowledge'  => 'Industry Knowledge',
    'r1_experience' => 'Relevant Experience',
    'r1_record'     => 'Past Track Record',
    'r1_initiative' => 'Initiative / Drive',
    'r1_tplayer'    => 'Team Player',
    'r1_leadership' => 'Leadership',
    'r1_stability'  => 'Job Stability',
    'r1_comskill'   => 'Communication Skills',
    'r1_ratings'    => 'Final Ratings',
];

?>
<?php ob_start(); ?>
<style>
    .review-page-wrapper {
        max-width: 900px;
        margin: 30px auto;
    }
    .review-card {
        border-radius: 14px;
        border: 1px solid #e5e7eb;
        background: #ffffff;
        box-shadow: 0 10px 25px rgba(15, 23, 42, 0.08);
        padding: 20px 24px;
    }
    .review-header-title {
        font-size: 18px;
        font-weight: 600;
    }
    .review-subtitle {
        font-size: 13px;
        color: #6b7280;
    }
    .pif-summary-label {
        font-size: 12px;
        font-weight: 600;
        color: #6b7280;
        margin-bottom: 2px;
    }
    .pif-summary-value {
        background-color: #f3f4f6;
        border-radius: 6px;
        padding: 6px 8px;
        font-size: 13px;
        color: #111827;
    }
    .section-card {
        background: #f8fafc;
        border: 1px solid #e5e7eb;
        box-shadow: inset 0 1px 0 rgba(255, 255, 255, 0.7);
    }

    .section-card.section-even {
        background: #eef2ff;
    }

    .section-card.section-odd {
        background: #f8fafc;
    }
    .section-title-sm {
        font-size: 14px;
        font-weight: 600;
        margin-bottom: 6px;
    }
</style>
<?php
$styles = ob_get_clean();
?>

<?php ob_start(); ?>
<div class="review-page-wrapper">
    <div class="review-card">
        <div class="mb-2">
            <div class="review-header-title">
                Interview Evaluation – PIF #<?php echo (int)$pif['idpif']; ?>
            </div>
            <div class="review-subtitle">
                Applicant: <strong><?php echo e($pif['username']); ?></strong>
                &nbsp; | &nbsp; Position Applied: <strong><?php echo e($pif['appliedfor']); ?></strong>
            </div>
            <div class="review-subtitle">
                Reviewer: <strong><?php echo e($reviewerName); ?></strong>
                (EmpID: <?php echo e($empid); ?>, Team: <?php echo e($reviewerTeam); ?>)
            </div>
        </div>

        <?php if ($successMsg): ?>
            <div class="alert alert-success" role="alert">
                <?php echo e($successMsg); ?>
            </div>
        <?php endif; ?>

        <?php if ($errorMsg): ?>
            <div class="alert alert-danger" role="alert">
                <?php echo e($errorMsg); ?>
            </div>
        <?php endif; ?>

        <!-- Applicant snapshot -->
        <div class="border rounded-3 p-3 mb-3 section-card">
            <div class="section-title-sm">Applicant Details : Pre - Interview Form (PIF)</div>
            <div class="row g-2">
                <div class="col-md-3"><div class="pif-summary-label">Applicant Name</div><div class="pif-summary-value"><?php echo e($pif['username']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Contact Number</div><div class="pif-summary-value"><?php echo e($pif['phone']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Email</div><div class="pif-summary-value"><?php echo e($pif['email']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Gender</div><div class="pif-summary-value"><?php echo e($pif['gender']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Marital Status</div><div class="pif-summary-value"><?php echo e($pif['marital_status']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Date of Birth</div><div class="pif-summary-value"><?php echo e($pif['dob']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Current Address</div><div class="pif-summary-value"><?php echo e($pif['currentAddress']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Current City</div><div class="pif-summary-value"><?php echo e($pif['currentCity']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Conveyance</div><div class="pif-summary-value"><?php echo e($pif['conveyance'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">If Yes</div><div class="pif-summary-value"><?php echo e($pif['convyes'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Position Applied For</div><div class="pif-summary-value"><?php echo e($pif['appliedfor']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Current Salary (Per Month)</div><div class="pif-summary-value"><?php echo e($pif['current_salary']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Expected Salary (Per Month)</div><div class="pif-summary-value"><?php echo e($pif['expected_salary']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Notice Period (Days)</div><div class="pif-summary-value"><?php echo e($pif['notice_period']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Qualification</div><div class="pif-summary-value"><?php echo e($pif['qualification']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Current / Last Org</div><div class="pif-summary-value"><?php echo e($pif['lastorg']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Current / Last Designation</div><div class="pif-summary-value"><?php echo e($pif['last_designation']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Job Type</div><div class="pif-summary-value"><?php echo e($pif['jobType']); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Total Work Experience</div><div class="pif-summary-value"><?php echo e($pif['totalExp']); ?></div></div>
                <div class="col-md-6"><div class="pif-summary-label">Reason for Leaving</div><div class="pif-summary-value"><?php echo e($pif['reason']); ?></div></div>
                <div class="col-md-3">
                    <div class="pif-summary-label">Resume</div>
                    <div class="pif-summary-value">
                        <?php if ($resumeUrl): ?>
                            <a class="btn btn-outline-primary btn-sm" href="<?php echo e($resumeUrl); ?>" target="_blank">Click to View</a>
                        <?php else: ?>
                            Not Uploaded
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <div class="border rounded-3 p-3 mb-3 section-card">
            <div class="section-title-sm">To Be Appointed As (For Official Use Only)</div>
            <div class="row g-2">
                <div class="col-md-3"><div class="pif-summary-label">To Be Appointed As</div><div class="pif-summary-value"><?php echo e($pif['appointed_as'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Fresh / Rehire</div><div class="pif-summary-value"><?php echo e($pif['fresh_hire'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">AD / Director EmpId</div><div class="pif-summary-value"><?php echo e($pif['ad_director_id'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">AD / Director Name</div><div class="pif-summary-value"><?php echo e($pif['ad_director_name'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Team Name</div><div class="pif-summary-value"><?php echo e($pif['team_name'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Department</div><div class="pif-summary-value"><?php echo e($pif['Department'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Location of Work</div><div class="pif-summary-value"><?php echo e($pif['Location_of_Work'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Designation</div><div class="pif-summary-value"><?php echo e($pif['designation'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">CTC per month</div><div class="pif-summary-value"><?php echo e($pif['ctc_per_month'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Expected DOJ</div><div class="pif-summary-value"><?php echo e($pif['expected_doj'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Actual DOJ</div><div class="pif-summary-value"><?php echo e($pif['actual_doj'] ?? ''); ?></div></div>
                <div class="col-md-4"><div class="pif-summary-label">Immediate Reporting Manager</div><div class="pif-summary-value"><?php echo e($pif['reporting_manager'] ?? ''); ?></div></div>
                <div class="col-md-2"><div class="pif-summary-label">RM Code</div><div class="pif-summary-value"><?php echo e($pif['rm_code'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Human Resource (TA)</div><div class="pif-summary-value"><?php echo e($pif['hr_ta'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">HEAD / DGM HR</div><div class="pif-summary-value"><?php echo e($pif['hr_head_dgm'] ?? ''); ?></div></div>
                <div class="col-md-3"><div class="pif-summary-label">Interview Date</div><div class="pif-summary-value"><?php echo e($pif['interview_date'] ?? ''); ?></div></div>
            </div>
        </div>

        <div class="border rounded-3 p-3 mb-3 section-card">
            <div class="section-title-sm">Evaluation Criteria: Reviewer 1 (HR)</div>
            <p class="text-muted fs-12">Rating Scale : Excellent = A, Very Good = B+, Good = B, Average = C, Poor = D</p>
            <div class="row g-2">
                <?php foreach ($ratingsMap as $field => $label): ?>
                    <div class="col-md-3">
                        <div class="pif-summary-label"><?php echo e($label); ?></div>
                        <div class="pif-summary-value"><?php echo e($pif[$field] ?? ''); ?></div>
                    </div>
                <?php endforeach; ?>
                <div class="col-md-3">
                    <div class="pif-summary-label">Reviewer 1 Decision</div>
                    <div class="pif-summary-value"><?php echo e($pif['st1_status']); ?></div>
                </div>
                <div class="col-md-3">
                    <div class="pif-summary-label">Reviewer 1 Remarks</div>
                    <div class="pif-summary-value"><?php echo e($pif['st1_feedback']); ?></div>
                </div>
                <div class="col-md-6">
                    <div class="pif-summary-label">Evaluation Criteria / Notes</div>
                    <div class="pif-summary-value"><?php echo e($pif['eval_r1_hr']); ?></div>
                </div>
            </div>
        </div>

        <!-- Reviewer decision form -->
        <form method="post" action="">
            <div class="border rounded-3 p-3 mb-2 section-card">
                <div class="section-title-sm">Your Evaluation (Decision &amp; Remarks)</div>

                <div class="mb-3">
                    <label class="form-label">Decision*</label>
                    <div class="row">
                        <div class="col-auto">
                            <div class="form-check">
                                <input class="form-check-input"
                                       type="radio"
                                       name="decision"
                                       id="decApprove"
                                       value="Approve"
                                    <?php echo (strcasecmp($reviewerDecision, 'Approve') === 0) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="decApprove">
                                    Approve
                                </label>
                            </div>
                        </div>
                        <div class="col-auto">
                            <div class="form-check">
                                <input class="form-check-input"
                                       type="radio"
                                       name="decision"
                                       id="decReject"
                                       value="Reject"
                                    <?php echo (strcasecmp($reviewerDecision, 'Reject') === 0) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="decReject">
                                    Reject
                                </label>
                            </div>
                        </div>
                        <div class="col-auto">
                            <div class="form-check">
                                <input class="form-check-input"
                                       type="radio"
                                       name="decision"
                                       id="decHold"
                                       value="Hold"
                                    <?php echo (strcasecmp($reviewerDecision, 'Hold') === 0) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="decHold">
                                    Hold
                                </label>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="mb-0">
                    <label class="form-label">Remarks</label>
                    <textarea
                        name="remarks"
                        class="form-control"
                        rows="3"
                        placeholder="Write your specific feedback / comments..."
                    ><?php echo e($reviewerRemarks); ?></textarea>
                </div>
            </div>

            <div class="d-flex justify-content-end">
                <button type="submit" class="btn btn-primary">
                    Submit Decision
                </button>
            </div>
        </form>
    </div>
</div>
<?php
$content = ob_get_clean();
?>

<?php ob_start(); ?>
<script>
    (function () {
        const cards = document.querySelectorAll('.section-card');

        cards.forEach(function (card, idx) {
            card.classList.remove('section-odd', 'section-even');
            card.classList.add(idx % 2 === 0 ? 'section-odd' : 'section-even');
        });
    })();
</script>
<?php
$scripts = ob_get_clean();

include __DIR__ . '/layouts/base.php';