<?php
/* =============================================================================
 * File   : project_form.php  (Melody admin theme)
 * Goal   : Create / Edit Project using centralized config.php (MySQLi, NO PDO)
 * Notes  : - Uses ONLY previews outside FilePond (no preloading in pond)
 *          - FilePond has NO type validation and NO image transforms (no blur)
 *          - Delete buttons for cover + gallery + floor plans
 *            (project_cover_delete.php, project_media_delete.php)
 *          - Property Type select, Amenities (pivot),
 *            Property Video upload/URL, Floor Plans (image+title),
 *            Nearby (schools/food) saved in features_json
 *          - Listing Type (Sell/Rent)
 *          - Property Type includes "Plots"
 *          - BHK-wise configurations in border blocks with "Add More":
 *              Bedrooms, Type, Bathrooms, Floors, Area, Carpet Area,
 *              Price Min, Price Text
 *          - Behaviour by Property Type:
 *              • House / Villa / Apartment / others:
 *                    show Bedrooms + Bathrooms + Floors
 *              • Office Desk / Plots:
 *                    hide Bedrooms & Bathrooms, show Floors
 *          - Configs saved as JSON string in bhk_config_json
 *            (plus first row mirrored into legacy columns)
 *          - NEW (this edit):
 *              • "Properties" dropdown (Exclusive / Featured) mapped to
 *                column project_tag, required, default = Exclusive,
 *                styled as bold/colored, placed before Title.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ------------------------- Centralized config ------------------------- */
$cfgs = [
  __DIR__ . '/../config.php',
  __DIR__ . '/../../config.php',
  __DIR__ . '/config.php'
];
$cfgFound = false;
foreach ($cfgs as $c) {
  if (file_exists($c)) {
    require_once $c;
    $cfgFound = true;
    break;
  }
}
if (!$cfgFound) die('config.php not found');

/* Pick mysqli */
$DB = null;
if (isset($conn_crm)     && $conn_crm     instanceof mysqli) $DB = $conn_crm;
elseif (isset($conn_crmplus) && $conn_crmplus instanceof mysqli) $DB = $conn_crmplus;
elseif (isset($conn)     && $conn     instanceof mysqli) $DB = $conn;
elseif (isset($conn_sa)  && $conn_sa  instanceof mysqli) $DB = $conn_sa;
if (!$DB instanceof mysqli) {
  http_response_code(500);
  exit('DB connection not found - check config.php');
}

/* Charset / Collation */
@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* ---------------------------- Helpers ---------------------------- */
if (!function_exists('h')) {
  function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); }
}
function q($s){ global $DB; return mysqli_real_escape_string($DB,(string)$s); }
function base_url(){
  if (!empty($GLOBALS['baseUrl'])) return rtrim($GLOBALS['baseUrl'],'/');
  $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS']!=='off') ? 'https' : 'http';
  $host  = $_SERVER['HTTP_HOST'] ?? ($_SERVER['SERVER_NAME'] ?? 'localhost');
  $dir   = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? '/'),'/');
  if (substr($dir, -6) === '/pages') $dir = rtrim(substr($dir, 0, -6), '/');
  return rtrim("$proto://$host$dir",'/');
}
function table_exists($name){
  global $DB;
  $name = q($name);
  $r = @mysqli_query($DB,"SHOW TABLES LIKE '{$name}'");
  $ok = ($r && mysqli_num_rows($r)>0);
  if ($r) @mysqli_free_result($r);
  return $ok;
}
function col_exists($tbl,$col){
  global $DB;
  $tbl = q($tbl); $col = q($col);
  $r = @mysqli_query($DB,"SHOW COLUMNS FROM `{$tbl}` LIKE '{$col}'");
  $ok = ($r && mysqli_num_rows($r)>0);
  if ($r) @mysqli_free_result($r);
  return $ok;
}

/* ------------------------- Resolve tables ------------------------ */
$PROJECTS_TABLE = null;
foreach (['projects','project','tbl_projects','pp_projects'] as $t) {
  if (table_exists($t)) { $PROJECTS_TABLE = $t; break; }
}
if (!$PROJECTS_TABLE) { die('Projects table not found.'); }
$MEDIA_TABLE = table_exists('project_media') ? 'project_media' : null;

/* ------------------------- Load Project -------------------------- */
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$project = [];
if ($id > 0) {
  $rs = mysqli_query($DB,"SELECT * FROM `{$PROJECTS_TABLE}` WHERE id={$id} LIMIT 1");
  if ($rs && mysqli_num_rows($rs)) { $project = mysqli_fetch_assoc($rs); }
  if ($rs) @mysqli_free_result($rs);
}
function pv($key,$default=''){
  global $project,$PROJECTS_TABLE;
  return (isset($project[$key]) && col_exists($PROJECTS_TABLE,$key)) ? $project[$key] : $default;
}

/* ------------------------- Existing media ------------------------ */
$existing = ['images'=>[], 'floors'=>[]];
if ($id && $MEDIA_TABLE) {
  $urlCol   = col_exists($MEDIA_TABLE,'url') ? 'url' : (col_exists($MEDIA_TABLE,'file_url') ? 'file_url' : null);
  $titleCol = col_exists($MEDIA_TABLE,'title') ? 'title' : null;
  $sortCol  = col_exists($MEDIA_TABLE,'sort_order') ? 'sort_order' : 'id';
  $isActCol = col_exists($MEDIA_TABLE,'is_active') ? 'COALESCE(is_active,1)=1 AND ' : '';
  if ($urlCol) {
    $sel = "id, media_type, {$urlCol} AS file_url".($titleCol?(", {$titleCol}"): '');
    $sql = "SELECT {$sel} FROM `{$MEDIA_TABLE}` WHERE {$isActCol} project_id={$id} ORDER BY ".($sortCol?$sortCol.', ':'')."id";
    if ($mrs = mysqli_query($DB,$sql)) {
      while($row = mysqli_fetch_assoc($mrs)){
        $mt = strtolower((string)$row['media_type']);
        if ($mt === 'image') { $existing['images'][] = $row; }
        if ($mt === 'floor') { $existing['floors'][] = $row; }
      }
      @mysqli_free_result($mrs);
    }
  }
}

/* -------------------- Amenities (master & pivot) ----------------- */
$amenities = [];
$amenityTableExists = table_exists('property_ameneties');
$pivotAmenityExists = table_exists('project_ameneties');
if ($amenityTableExists) {
  $rsA = mysqli_query($DB,"SELECT id,amenity_name FROM property_ameneties WHERE is_active=1 ORDER BY sort_order,amenity_name");
  if ($rsA) {
    while($r=mysqli_fetch_assoc($rsA)) $amenities[]=$r;
    mysqli_free_result($rsA);
  }
}
$checkedAmenityIds = [];
if ($id && $pivotAmenityExists) {
  $rsC = mysqli_query($DB,"SELECT amenity_id FROM project_ameneties WHERE project_id=".(int)$id);
  if ($rsC) {
    while($r=mysqli_fetch_assoc($rsC)) $checkedAmenityIds[]=(int)$r['amenity_id'];
    mysqli_free_result($rsC);
  }
}

/* --------------- Nearby (prefill from features_json) ------------- */
$nearbySchools = $nearbyFood = [];
$featRaw = pv('features_json','');
if ($featRaw) {
  $feat = json_decode($featRaw, true);
  if (isset($feat['nearby']['schools']) && is_array($feat['nearby']['schools'])) $nearbySchools = $feat['nearby']['schools'];
  if (isset($feat['nearby']['food'])    && is_array($feat['nearby']['food']))    $nearbyFood    = $feat['nearby']['food'];
}
$prefillNearby = function($arr,$i,$k,$d=''){ return isset($arr[$i][$k]) ? h($arr[$i][$k]) : $d; };

/* --------------- BHK configs from bhk_config_json or legacy ------- */
$bhkConfigs = [];
$bhkRaw = '';
if (col_exists($PROJECTS_TABLE,'bhk_config_json')) {
  $bhkRaw = pv('bhk_config_json','');
  if ($bhkRaw) {
    $tmp = json_decode($bhkRaw, true);
    if (is_array($tmp)) $bhkConfigs = $tmp;
  }
}
if (empty($bhkConfigs)) {
  // Fallback: build one config from legacy single columns
  $bhkConfigs[] = [
    'bedrooms'    => pv('bedrooms',''),
    'type'        => pv('usage_type',''),
    'bathrooms'   => pv('bathrooms',''),
    'floors'      => pv('floors',''),
    'area_sqft'   => pv('area_sqft',''),
    'carpet_area' => pv('carpet_area',''),
    'price_min'   => pv('price_min',''),
    'price_text'  => pv('price_text',''),
  ];
}

/* ------------------------- Theme blocks -------------------------- */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo base_url(); ?>/assets/libs/filepond/filepond.min.css">
<link rel="stylesheet" href="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-preview/filepond-plugin-image-preview.min.css">
<style>
  .pill      { display:inline-flex; align-items:center; gap:.5rem; padding:.45rem .75rem; border:1px solid #e4e7eb; border-radius:999px; background:#f8fafc; }
  .pill input{ margin-top:1px; }
  .thumb     { width:160px; height:110px; object-fit:cover; border-radius:10px; border:1px solid #e6e8ef; }
  .muted     { color:#6b7380; }
  .thumb-grid{ display:flex; flex-wrap:wrap; gap:.75rem; }
  .thumb-card{ width:180px; border:1px solid #e9ecf2; border-radius:12px; overflow:hidden; background:#fff; }
  .thumb-card .body{ padding:.5rem; text-align:center; }
  .form-label.required::after{
    content:" *";
    margin-left: 2px;
    color:#dc3545;
    font-weight:600;
  }
  /* Properties (project_tag) styling */
  .project-tag-label{
    font-weight:700;
    color:#7c3aed; /* violet tone */
  }
  .project-tag-select{
    font-weight:600;
    background:#f3e8ff;
    border-color:#c4b5fd;
  }
</style>
<?php $styles = ob_get_clean();

ob_start(); ?>

<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0"><?php echo $id ? 'Edit Project' : 'Create Project'; ?></h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="projects-list.php">Projects List</a></li>
      <li class="breadcrumb-item active" aria-current="page"><?php echo $id ? 'Edit' : 'Create'; ?></li>
    </ol>
  </div>
</div>

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card">
      <div class="card-header">
        <div class="card-title"><?php echo $id ? 'Update Project' : 'Create Project'; ?></div>
        <div class="ms-auto">
          <a href="projects-list.php" class="btn btn-outline-secondary btn-sm">
            <i class="ri-arrow-go-back-line me-1"></i>Projects List
          </a>
        </div>
      </div>

      <div class="card-body">
        <form method="post" action="project_save.php" enctype="multipart/form-data" class="row gy-3">
          <input type="hidden" name="id" value="<?php echo (int)$id; ?>">
          <input type="hidden" name="features_json_existing" value="<?php echo h($featRaw); ?>">
          <input type="hidden" name="bhk_config_json" id="bhkConfigJson" value="<?php echo h($bhkRaw); ?>">

          <!-- BASIC -->

          <?php if (col_exists($PROJECTS_TABLE,'project_tag')): ?>
            <div class="col-xl-3">
              <label class="form-label project-tag-label">Properties</label>
              <?php $tagVal = pv('project_tag','Exclusive'); ?>
              <select name="project_tag"
                      class="form-select project-tag-select"
                      required>
                <option value="Exclusive" <?php echo ($tagVal==='Exclusive'?'selected':''); ?>>
                  Exclusive
                </option>
                <option value="Featured" <?php echo ($tagVal==='Featured'?'selected':''); ?>>
                  Featured
                </option>
              </select>
            </div>
          <?php endif; ?>

          <?php if (col_exists($PROJECTS_TABLE,'title')): ?>
            <div class="col-xl-5">
              <label class="form-label">Title</label>
              <input type="text" name="title" class="form-control"
                     value="<?php echo h(pv('title')); ?>" required>
            </div>
          <?php endif; ?>

          <?php if (col_exists($PROJECTS_TABLE,'city')): ?>
            <div class="col-xl-2">
              <label class="form-label">City</label>
              <input type="text" name="city" class="form-control"
                     value="<?php echo h(pv('city')); ?>">
            </div>
          <?php endif; ?>

          <?php if (col_exists($PROJECTS_TABLE,'address')): ?>
            <div class="col-xl-2">
              <label class="form-label">Address</label>
              <input type="text" name="address" class="form-control"
                     value="<?php echo h(pv('address')); ?>">
            </div>
          <?php endif; ?>

          <!-- LISTING / PROPERTY / URL / GEO -->
          <?php if (col_exists($PROJECTS_TABLE,'listing_type')): ?>
            <div class="col-xl-3">
              <label class="form-label">Listing Type</label>
              <select name="listing_type" class="form-select">
                <?php $lt = pv('listing_type','Sell'); ?>
                <option value="Sell" <?php echo ($lt==='Sell'?'selected':''); ?>>Sell</option>
                <option value="Rent" <?php echo ($lt==='Rent'?'selected':''); ?>>Rent</option>
              </select>
            </div>
          <?php endif; ?>

          <?php
            // Property types (Plots added)
            $propTypeOpts = ['House','Office Desk','Villa','Apartment','Condo','Denver','Studio','Plots'];
            if (col_exists($PROJECTS_TABLE,'property_type')):
          ?>
            <div class="col-xl-3">
              <label class="form-label">Property Type</label>
              <select name="property_type" class="form-select" id="propertyTypeSelect">
                <option value="">-- Select --</option>
                <?php foreach ($propTypeOpts as $opt): ?>
                  <option value="<?php echo h($opt); ?>"
                    <?php echo (pv('property_type')===$opt?'selected':''); ?>>
                    <?php echo h($opt); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          <?php endif; ?>

          <?php if (col_exists($PROJECTS_TABLE,'website_url')): ?>
            <div class="col-xl-6">
              <label class="form-label">Website URL</label>
              <input type="text" name="website_url" class="form-control"
                     value="<?php echo h(pv('website_url')); ?>" placeholder="https://">
            </div>
          <?php endif; ?>

          <div class="col-xl-2">
            <label class="form-label">Latitude</label>
            <input type="text" name="lat" class="form-control"
                   value="<?php echo h(pv('lat')); ?>">
          </div>

          <div class="col-xl-2">
            <label class="form-label">Longitude</label>
            <input type="text" name="lng" class="form-control"
                   value="<?php echo h(pv('lng')); ?>">
          </div>

          <div class="col-12"><hr></div>

          <!-- BHK CONFIGURATIONS (border blocks + Add More) -->
          <div class="col-xl-12">
            <div class="d-flex align-items-center justify-content-between mb-1">
              <label class="form-label mb-0">Configurations (per unit)</label>
              <button type="button" class="btn btn-outline-primary btn-sm" id="addConfigRow">
                <i class="ri-add-line me-1"></i>Add More
              </button>
            </div>
            <small class="muted d-block mb-2">
              Add multiple configurations (2BHK, 3BHK, etc.). Fields show/hide automatically based on Property Type.
            </small>

            <div id="bhkConfigRows">
              <?php
              $bedMapStatic = [
                1 => '01 Bedroom',
                2 => '02 Bedrooms',
                3 => '03 Bedrooms',
                4 => '04 Bedrooms',
                5 => '05 Bedrooms',
                6 => '06+ Bedrooms'
              ];
              foreach ($bhkConfigs as $cfgIndex => $cfg):
                $cfgBedrooms    = isset($cfg['bedrooms'])    ? (int)$cfg['bedrooms']    : 0;
                $cfgType        = isset($cfg['type'])        ? (string)$cfg['type']     : '';
                $cfgBathrooms   = isset($cfg['bathrooms'])   ? (string)$cfg['bathrooms']   : '';
                $cfgFloors      = isset($cfg['floors'])      ? (string)$cfg['floors']      : '';
                $cfgArea        = isset($cfg['area_sqft'])   ? (string)$cfg['area_sqft']   : '';
                $cfgCarpet      = isset($cfg['carpet_area']) ? (string)$cfg['carpet_area'] : '';
                $cfgPriceMin    = isset($cfg['price_min'])   ? (string)$cfg['price_min']   : '';
                $cfgPriceText   = isset($cfg['price_text'])  ? (string)$cfg['price_text']  : '';
              ?>
              <div class="border rounded p-3 mb-2 bhk-config-row">
                <div class="row g-2 align-items-end">
                  <?php if (col_exists($PROJECTS_TABLE,'bedrooms')): ?>
                    <div class="col-xl-2 col-md-3" data-role="bedrooms">
                      <label class="form-label">Bedrooms</label>
                      <select name="config_bedrooms[]" class="form-select">
                        <option value="">-- Select --</option>
                        <?php foreach ($bedMapStatic as $v=>$label): ?>
                          <option value="<?php echo $v; ?>"
                            <?php echo ($cfgBedrooms===$v?'selected':''); ?>>
                            <?php echo h($label); ?>
                          </option>
                        <?php endforeach; ?>
                      </select>
                    </div>
                  <?php endif; ?>

                  <div class="col-xl-2 col-md-3">
                    <label class="form-label">Type</label>
                    <select name="config_type[]" class="form-select">
                      <option value="">-- Select --</option>
                      <option value="Residential" <?php echo ($cfgType==='Residential'?'selected':''); ?>>Residential</option>
                      <option value="Commercial" <?php echo ($cfgType==='Commercial'?'selected':''); ?>>Commercial</option>
                    </select>
                  </div>

                  <?php if (col_exists($PROJECTS_TABLE,'bathrooms')): ?>
                    <div class="col-xl-2 col-md-3" data-role="bathrooms">
                      <label class="form-label">Bathrooms</label>
                      <input type="number" name="config_bathrooms[]" class="form-control"
                             value="<?php echo h($cfgBathrooms); ?>">
                    </div>
                  <?php endif; ?>

                  <?php if (col_exists($PROJECTS_TABLE,'floors')): ?>
                    <div class="col-xl-2 col-md-3">
                      <label class="form-label">Floors</label>
                      <input type="number" name="config_floors[]" class="form-control"
                             value="<?php echo h($cfgFloors); ?>">
                    </div>
                  <?php endif; ?>

                  <?php if (col_exists($PROJECTS_TABLE,'area_sqft')): ?>
                    <div class="col-xl-2 col-md-3">
                      <label class="form-label">Area (sqft)</label>
                      <input type="number" step="0.01" name="config_area_sqft[]" class="form-control"
                             value="<?php echo h($cfgArea); ?>">
                    </div>
                  <?php endif; ?>

                  <?php if (col_exists($PROJECTS_TABLE,'carpet_area')): ?>
                    <div class="col-xl-2 col-md-3">
                      <label class="form-label">Carpet Area</label>
                      <input type="text" name="config_carpet_area[]" class="form-control"
                             value="<?php echo h($cfgCarpet); ?>">
                    </div>
                  <?php endif; ?>

                  <?php if (col_exists($PROJECTS_TABLE,'price_min')): ?>
                    <div class="col-xl-2 col-md-3">
                      <label class="form-label">Price Min (numeric)</label>
                      <input type="number" step="0.01" name="config_price_min[]" class="form-control"
                             value="<?php echo h($cfgPriceMin); ?>">
                    </div>
                  <?php endif; ?>

                  <?php if (col_exists($PROJECTS_TABLE,'price_text')): ?>
                    <div class="col-xl-3 col-md-4">
                      <label class="form-label">Price Text</label>
                      <input type="text" name="config_price_text[]" class="form-control"
                             value="<?php echo h($cfgPriceText); ?>" placeholder="₹ 60L - ₹ 1.2Cr">
                    </div>
                  <?php endif; ?>

                  <div class="col-xl-1 col-md-2">
                    <button type="button" class="btn btn-outline-danger w-100 remove-config-row mt-4">
                      &times;
                    </button>
                  </div>
                </div>
              </div>
              <?php endforeach; ?>
            </div>
          </div>

          <!-- Hidden legacy singles: will be set from FIRST config on submit -->
          <?php if (col_exists($PROJECTS_TABLE,'bedrooms')): ?>
            <input type="hidden" name="bedrooms" id="primary_bedrooms" value="<?php echo h(pv('bedrooms')); ?>">
          <?php endif; ?>
          <input type="hidden" name="usage_type" id="primary_usage_type" value="<?php echo h(pv('usage_type')); ?>">
          <?php if (col_exists($PROJECTS_TABLE,'bathrooms')): ?>
            <input type="hidden" name="bathrooms" id="primary_bathrooms" value="<?php echo h(pv('bathrooms')); ?>">
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'floors')): ?>
            <input type="hidden" name="floors" id="primary_floors" value="<?php echo h(pv('floors')); ?>">
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'area_sqft')): ?>
            <input type="hidden" name="area_sqft" id="primary_area_sqft" value="<?php echo h(pv('area_sqft')); ?>">
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'carpet_area')): ?>
            <input type="hidden" name="carpet_area" id="primary_carpet_area" value="<?php echo h(pv('carpet_area')); ?>">
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'price_min')): ?>
            <input type="hidden" name="price_min" id="primary_price_min" value="<?php echo h(pv('price_min')); ?>">
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'price_text')): ?>
            <input type="hidden" name="price_text" id="primary_price_text" value="<?php echo h(pv('price_text')); ?>">
          <?php endif; ?>

          <div class="col-12"><hr></div>

          <!-- AREA UNIT / CONFIG / SORT / ACTIVE -->
          <?php if (col_exists($PROJECTS_TABLE,'area_unit')): ?>
            <div class="col-xl-3">
              <label class="form-label">Area Unit</label>
              <input type="text" name="area_unit" class="form-control"
                     value="<?php echo h(pv('area_unit','ft2')); ?>">
            </div>
          <?php endif; ?>

          <?php if (col_exists($PROJECTS_TABLE,'configuration')): ?>
            <div class="col-xl-9">
              <label class="form-label">Configurations</label>
              <input type="text" name="configuration" class="form-control"
                     placeholder="e.g., East Facing, Corner, Sea View, Servant Room"
                     value="<?php echo h(pv('configuration')); ?>">
            </div>
          <?php endif; ?>

          <div class="col-xl-2">
            <label class="form-label">Sort Order</label>
            <input type="number" name="sort_order" class="form-control"
                   value="<?php echo h(pv('sort_order',0)); ?>">
          </div>

          <div class="col-xl-12">
            <label class="form-check mt-2">
              <input type="checkbox" class="form-check-input" name="is_active" value="1"
                <?php echo (pv('is_active','1')=='1'?'checked':''); ?>>
              <span class="ms-1">Active</span>
            </label>
          </div>

          <div class="col-12"><hr></div>

          <!-- DESCRIPTION -->
          <div class="col-xl-12">
            <label class="form-label">Description</label>
            <textarea name="description" class="form-control" rows="4"
                      placeholder="Project overview, highlights, nearby landmarks…"><?php
              echo h(pv('description'));
            ?></textarea>
          </div>

          <div class="col-12"><hr></div>

          <!-- AMENITIES (MASTER + PIVOT) -->
          <?php if ($amenityTableExists && !empty($amenities)): ?>
            <div class="col-xl-12">
              <label class="form-label d-block mb-2">Amenities</label>
              <div class="thumb-grid">
                <?php foreach ($amenities as $am): ?>
                  <label class="pill">
                    <input type="checkbox" name="amenity_ids[]"
                           value="<?php echo (int)$am['id']; ?>"
                           <?php echo in_array((int)$am['id'],$checkedAmenityIds,true)?'checked':''; ?>>
                    <span><?php echo h($am['amenity_name']); ?></span>
                  </label>
                <?php endforeach; ?>
              </div>
              <small class="muted d-block mt-2">
                Selections saved in <code>project_ameneties</code>
                (handled in <code>project_save.php</code>).
              </small>
            </div>
          <?php endif; ?>

          <!-- (Optional legacy) Amenities CSV column if exists -->
          <?php if (col_exists($PROJECTS_TABLE,'amenities')): ?>
            <div class="col-xl-12">
              <label class="form-label">Amenities &amp; Features (CSV)</label>
              <input type="text" name="amenities" class="form-control"
                     value="<?php echo h(pv('amenities')); ?>"
                     placeholder="Parking,Pool,Garden">
            </div>
          <?php endif; ?>

          <div class="col-12"><hr></div>

          <!-- COVER IMAGE -->
          <div class="col-xl-12">
            <h6 class="mb-2 d-flex align-items-center justify-content-between">
              <span>Cover Image</span>
              <?php if (!empty($project['cover_image']) && $id): ?>
                <a class="btn btn-sm btn-outline-danger"
                   href="project_cover_delete.php?pid=<?php echo (int)$id; ?>"
                   onclick="return confirm('Remove cover image?');">
                  <i class="ri-delete-bin-line me-1"></i>Remove
                </a>
              <?php endif; ?>
            </h6>
            <div class="row g-3 align-items-end">
              <div class="col-xl-6">
                <label class="form-label">Upload Cover</label>
                <input id="coverPond" type="file" name="cover_image_upload" class="form-control">
              </div>
              <div class="col-xl-6">
                <label class="form-label">OR Cover Image URL</label>
                <input type="text" name="cover_image_url" class="form-control"
                       value="<?php echo h(pv('cover_image')); ?>"
                       placeholder="https://.../cover.webp">
              </div>
              <?php if (!empty($project['cover_image'])): ?>
                <div class="col-12">
                  <small class="muted d-block mb-1">Current cover preview:</small>
                  <img src="<?php echo h($project['cover_image']); ?>"
                       class="thumb" alt="Cover">
                </div>
              <?php endif; ?>
            </div>
          </div>

          <div class="col-12"><hr></div>

          <!-- GALLERY IMAGES -->
          <div class="col-xl-12">
            <h6 class="mb-2 d-flex align-items-center justify-content-between">
              <span>Gallery Images</span>
            </h6>
            <input id="galleryPond" type="file" name="project_images[]" class="form-control" multiple>
            <small class="muted">
              Uploads go to <code>uploads/projects/{id}/images/</code> and are saved
              in <code>project_media</code> as <em>image</em>.
            </small>

            <?php if (!empty($existing['images'])): ?>
              <div class="thumb-grid mt-3">
                <?php foreach ($existing['images'] as $img): ?>
                  <div class="thumb-card">
                    <img src="<?php echo h($img['file_url']); ?>" class="w-100"
                         style="height:110px; object-fit:cover;" alt="Image">
                    <div class="body">
                      <a class="btn btn-xs btn-outline-danger"
                         href="project_media_delete.php?id=<?php echo (int)$img['id']; ?>&amp;pid=<?php echo (int)$id; ?>"
                         onclick="return confirm('Delete this image?');">
                        <i class="ri-delete-bin-line me-1"></i>Remove
                      </a>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>
            <?php endif; ?>
          </div>

          <div class="col-12"><hr></div>

          <!-- PROPERTY VIDEO -->
          <?php if (col_exists($PROJECTS_TABLE,'property_video')): ?>
            <div class="col-xl-12">
              <h6 class="mb-2">Property Video</h6>
              <div class="row g-3 align-items-end">
                <div class="col-xl-6">
                  <label class="form-label">Upload Video</label>
                  <input id="videoInput" type="file" accept="video/*"
                         name="property_video_upload" class="form-control">
                  <small class="muted">
                    MP4 / WebM recommended. Max size as per your server limits.
                  </small>
                </div>
                <div class="col-xl-6">
                  <label class="form-label">OR Video URL</label>
                  <input type="text" name="property_video_url" class="form-control"
                         value="<?php echo h(pv('property_video')); ?>"
                         placeholder="https:// (YouTube/Vimeo or direct mp4)">
                </div>
              </div>
              <?php if (!empty($project['property_video'])): ?>
                <div class="mt-2">
                  <small class="muted d-block mb-1">Current video:</small>
                  <video controls
                         style="max-width:340px;border-radius:10px;border:1px solid #e6e8ef">
                    <source src="<?php echo h($project['property_video']); ?>">
                  </video>
                </div>
              <?php endif; ?>
            </div>
          <?php endif; ?>

          <div class="col-12"><hr></div>

          <!-- FLOOR PLANS -->
          <div class="col-xl-12">
            <h6 class="mb-2 d-flex align-items-center justify-content-between">
              <span>Floor Plans</span>
            </h6>

            <div id="floorPlansWrap" class="row gy-2">
              <div class="col-xl-4">
                <label class="form-label">Plan Title</label>
                <input type="text" name="floor_plan_titles[]" class="form-control"
                       placeholder="e.g., 2BHK - Type A">
              </div>
              <div class="col-xl-4">
                <label class="form-label">Upload Image</label>
                <input type="file" name="floor_plan_images[]" class="form-control">
              </div>
              <div class="col-xl-4">
                <label class="form-label">OR Image URL</label>
                <input type="text" name="floor_plan_urls[]" class="form-control"
                       placeholder="https://.../plan-a.jpg">
              </div>
            </div>

            <button type="button" class="btn btn-outline-primary btn-sm mt-2" id="addFloorRow">
              <i class="ri-add-line me-1"></i>Add another
            </button>

            <?php if (!empty($existing['floors'])): ?>
              <div class="thumb-grid mt-3">
                <?php foreach ($existing['floors'] as $fl): ?>
                  <div class="thumb-card">
                    <img src="<?php echo h($fl['file_url']); ?>" class="w-100"
                         style="height:110px;object-fit:cover" alt="Floor">
                    <div class="body">
                      <div class="small">
                        <?php echo h($fl['title'] ?? 'Floor Plan'); ?>
                      </div>
                      <a class="btn btn-xs btn-outline-danger mt-1"
                         href="project_media_delete.php?id=<?php echo (int)$fl['id']; ?>&amp;pid=<?php echo (int)$id; ?>"
                         onclick="return confirm('Delete this floor plan?');">
                        <i class="ri-delete-bin-line me-1"></i>Remove
                      </a>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>
            <?php endif; ?>
          </div>

          <div class="col-12"><hr></div>

          <!-- NEARBY -->
          <div class="col-xl-12">
            <h6 class="mb-2">Nearby</h6>

            <!-- Schools -->
            <div class="border rounded p-3 mb-3">
              <div class="d-flex align-items-center justify-content-between">
                <h6 class="mb-0">Schools Around</h6>
                <button class="btn btn-outline-primary btn-sm" type="button" id="addSchoolRow">
                  <i class="ri-add-line me-1"></i>Add
                </button>
              </div>
              <div id="schoolsRows" class="mt-2">
                <?php for($i=0;$i<max(1,count($nearbySchools));$i++): ?>
                  <div class="row g-2 align-items-end mb-2">
                    <div class="col-md-4">
                      <label class="form-label">Name</label>
                      <input type="text" name="nearby_schools[name][]" class="form-control"
                             value="<?php echo $prefillNearby($nearbySchools,$i,'name'); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Distance (mi)</label>
                      <input type="text" name="nearby_schools[distance][]" class="form-control"
                             placeholder="e.g., 3.52"
                             value="<?php echo $prefillNearby($nearbySchools,$i,'distance'); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Rating (0-5)</label>
                      <input type="number" step="0.1" min="0" max="5"
                             name="nearby_schools[rating][]" class="form-control"
                             value="<?php echo $prefillNearby($nearbySchools,$i,'rating'); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Reviews</label>
                      <input type="number" name="nearby_schools[reviews][]" class="form-control"
                             value="<?php echo $prefillNearby($nearbySchools,$i,'reviews'); ?>">
                    </div>
                    <div class="col-md-2">
                      <button type="button" class="btn btn-outline-danger w-100 remove-row">
                        Remove
                      </button>
                    </div>
                  </div>
                <?php endfor; ?>
              </div>
            </div>

            <!-- Food -->
            <div class="border rounded p-3">
              <div class="d-flex align-items-center justify-content-between">
                <h6 class="mb-0">Food Around</h6>
                <button class="btn btn-outline-primary btn-sm" type="button" id="addFoodRow">
                  <i class="ri-add-line me-1"></i>Add
                </button>
              </div>
              <div id="foodRows" class="mt-2">
                <?php for($i=0;$i<max(1,count($nearbyFood));$i++): ?>
                  <div class="row g-2 align-items-end mb-2">
                    <div class="col-md-4">
                      <label class="form-label">Name</label>
                      <input type="text" name="nearby_food[name][]" class="form-control"
                             value="<?php echo $prefillNearby($nearbyFood,$i,'name'); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Distance (mi)</label>
                      <input type="text" name="nearby_food[distance][]" class="form-control"
                             value="<?php echo $prefillNearby($nearbyFood,$i,'distance'); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Rating (0-5)</label>
                      <input type="number" step="0.1" min="0" max="5"
                             name="nearby_food[rating][]" class="form-control"
                             value="<?php echo $prefillNearby($nearbyFood,$i,'rating'); ?>">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Reviews</label>
                      <input type="number" name="nearby_food[reviews][]" class="form-control"
                             value="<?php echo $prefillNearby($nearbyFood,$i,'reviews'); ?>">
                    </div>
                    <div class="col-md-2">
                      <button type="button" class="btn btn-outline-danger w-100 remove-row">
                        Remove
                      </button>
                    </div>
                  </div>
                <?php endfor; ?>
              </div>
            </div>
          </div>

          <div class="col-12"><hr></div>

          <!-- DOCUMENTS -->
          <div class="col-xl-12">
            <h6 class="mb-2">Documents</h6>
            <input type="file" name="project_docs[]" class="form-control" multiple>
            <small class="muted">
              Stored under <code>uploads/projects/{id}/docs/</code>
              (saved as <em>doc</em> in <code>project_media</code>).
            </small>
          </div>

          <div class="col-12"><hr></div>

          <div class="card-footer bg-transparent px-0">
            <button class="btn btn-primary-light btn-wave ms-auto float-end" type="submit">
              <i class="ri-save-3-line me-1"></i>
              <?php echo $id ? 'Update Project' : 'Create Project'; ?>
            </button>
          </div>

        </form>
      </div>
    </div>
  </div>
</div>

<?php $content = ob_get_clean();

/* ----------------------------- Scripts ---------------------------- */
ob_start(); ?>
<script src="<?php echo base_url(); ?>/assets/libs/filepond/filepond.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-preview/filepond-plugin-image-preview.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-exif-orientation/filepond-plugin-image-exif-orientation.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-file-validate-size/filepond-plugin-file-validate-size.min.js"></script>
<script>
/* FilePond: no type validation, no transforms */
FilePond.registerPlugin(
  FilePondPluginImagePreview,
  FilePondPluginImageExifOrientation,
  FilePondPluginFileValidateSize
);
const COMMON_OPTS = {
  storeAsFile: true,
  allowFileTypeValidation: false,
  allowImagePreview: true,
  allowReorder: true,
  allowMultiple: true,
  allowImageTransform: false
};
const coverInput = document.getElementById('coverPond');
if (coverInput) FilePond.create(coverInput, Object.assign({}, COMMON_OPTS, {allowMultiple:false}));
const galleryInput = document.getElementById('galleryPond');
if (galleryInput) FilePond.create(galleryInput, COMMON_OPTS);

/* Floor Plans: Add row */
const addFloorBtn = document.getElementById('addFloorRow');
if (addFloorBtn) addFloorBtn.addEventListener('click', function(){
  const row = document.createElement('div');
  row.className = 'row gy-2 mt-2';
  row.innerHTML = `
    <div class="col-xl-4">
      <input type="text" name="floor_plan_titles[]" class="form-control" placeholder="Plan Title">
    </div>
    <div class="col-xl-4">
      <input type="file" name="floor_plan_images[]" class="form-control">
    </div>
    <div class="col-xl-4">
      <input type="text" name="floor_plan_urls[]" class="form-control" placeholder="https://.../plan.jpg">
    </div>`;
  document.getElementById('floorPlansWrap').appendChild(row);
});

/* Nearby: add/remove rows (schools/food) */
function addNearbyRow(containerId, key){
  const wrap = document.getElementById(containerId);
  const row = document.createElement('div');
  row.className = 'row g-2 align-items-end mb-2';
  row.innerHTML = `
    <div class="col-md-4">
      <input type="text" name="nearby_${key}[name][]" class="form-control" placeholder="Name">
    </div>
    <div class="col-md-2">
      <input type="text" name="nearby_${key}[distance][]" class="form-control" placeholder="e.g., 0.52">
    </div>
    <div class="col-md-2">
      <input type="number" step="0.1" min="0" max="5"
             name="nearby_${key}[rating][]" class="form-control" placeholder="4.5">
    </div>
    <div class="col-md-2">
      <input type="number" name="nearby_${key}[reviews][]" class="form-control" placeholder="470">
    </div>
    <div class="col-md-2">
      <button type="button" class="btn btn-outline-danger w-100 remove-row">Remove</button>
    </div>`;
  wrap.appendChild(row);
}
const addSchoolRowBtn = document.getElementById('addSchoolRow');
if (addSchoolRowBtn) addSchoolRowBtn.addEventListener('click', ()=>addNearbyRow('schoolsRows','schools'));
const addFoodRowBtn = document.getElementById('addFoodRow');
if (addFoodRowBtn) addFoodRowBtn.addEventListener('click', ()=>addNearbyRow('foodRows','food'));

document.addEventListener('click', function(e){
  if (e.target.classList.contains('remove-row')) {
    const row = e.target.closest('.row');
    if (row) row.remove();
  }
});

/* BHK config blocks: visibility & add/remove */
function updateBhkVisibility(){
  const propSel = document.getElementById('propertyTypeSelect');
  const val = propSel ? propSel.value : '';
  const special = (val === 'Office Desk' || val === 'Plots'); // bedrooms + bathrooms hide

  document.querySelectorAll('.bhk-config-row').forEach(function(row){
    const bedCol  = row.querySelector('[data-role="bedrooms"]');
    const bathCol = row.querySelector('[data-role="bathrooms"]');

    if (bedCol) {
      if (special) {
        bedCol.style.display = 'none';
        const sel = bedCol.querySelector('select');
        if (sel) sel.selectedIndex = 0;
      } else {
        bedCol.style.display = '';
      }
    }
    if (bathCol) {
      if (special) {
        bathCol.style.display = 'none';
        const inp = bathCol.querySelector('input');
        if (inp) inp.value = '';
      } else {
        bathCol.style.display = '';
      }
    }
  });
}

(function(){
  const propSel = document.getElementById('propertyTypeSelect');
  if (propSel) {
    propSel.addEventListener('change', updateBhkVisibility);
  }
  updateBhkVisibility();

  const rowsWrap = document.getElementById('bhkConfigRows');
  const addBtn   = document.getElementById('addConfigRow');

  if (rowsWrap && addBtn) {
    addBtn.addEventListener('click', function(){
      const firstRow = rowsWrap.querySelector('.bhk-config-row');
      if (!firstRow) return;
      const clone = firstRow.cloneNode(true);

      clone.querySelectorAll('input').forEach(function(inp){
        if (inp.type === 'number' || inp.type === 'text') inp.value = '';
      });
      clone.querySelectorAll('select').forEach(function(sel){
        sel.selectedIndex = 0;
      });

      rowsWrap.appendChild(clone);
      updateBhkVisibility();
    });
  }

  document.addEventListener('click', function(e){
    if (e.target.classList.contains('remove-config-row')) {
      const row  = e.target.closest('.bhk-config-row');
      const rows = document.querySelectorAll('.bhk-config-row');
      if (!row) return;
      if (rows.length <= 1) {
        // keep at least one row, just clear values
        row.querySelectorAll('input').forEach(function(inp){
          if (inp.type === 'number' || inp.type === 'text') inp.value = '';
        });
        row.querySelectorAll('select').forEach(function(sel){
          sel.selectedIndex = 0;
        });
      } else {
        row.remove();
      }
    }
  });

  /* On submit:
     - Build bhk_config_json from all config rows
     - Mirror FIRST config into legacy single columns
     - Normalize lat/lng/sort_order numeric fields
  */
  document.addEventListener('submit', function(e){
    const form = e.target;
    if (!form || !form.matches('form[action="project_save.php"]')) return;

    const configs = [];
    const rows = form.querySelectorAll('.bhk-config-row');

    rows.forEach(function(row){
      const cfg = {};
      const bedSel    = row.querySelector('[data-role="bedrooms"] select');
      const typeSel   = row.querySelector('select[name="config_type[]"]');
      const bathInp   = row.querySelector('[data-role="bathrooms"] input[name="config_bathrooms[]"]');
      const floorInp  = row.querySelector('input[name="config_floors[]"]');
      const areaInp   = row.querySelector('input[name="config_area_sqft[]"]');
      const carpetInp = row.querySelector('input[name="config_carpet_area[]"]');
      const pminInp   = row.querySelector('input[name="config_price_min[]"]');
      const ptxtInp   = row.querySelector('input[name="config_price_text[]"]');

      cfg.bedrooms    = bedSel   ? bedSel.value : '';
      cfg.type        = typeSel  ? typeSel.value : '';
      cfg.bathrooms   = bathInp  ? bathInp.value : '';
      cfg.floors      = floorInp ? floorInp.value : '';
      cfg.area_sqft   = areaInp  ? areaInp.value : '';
      cfg.carpet_area = carpetInp? carpetInp.value : '';
      cfg.price_min   = pminInp  ? pminInp.value : '';
      cfg.price_text  = ptxtInp  ? ptxtInp.value : '';

      const hasValue =
        cfg.bedrooms || cfg.type || cfg.bathrooms || cfg.floors ||
        cfg.area_sqft || cfg.carpet_area || cfg.price_min || cfg.price_text;

      if (hasValue) configs.push(cfg);
    });

    const jsonField = form.querySelector('#bhkConfigJson');
    if (jsonField) {
      jsonField.value = JSON.stringify(configs);
    }

    if (configs.length > 0) {
      const first = configs[0];
      const pb     = form.querySelector('#primary_bedrooms');
      const pt     = form.querySelector('#primary_usage_type');
      const pba    = form.querySelector('#primary_bathrooms');
      const pf     = form.querySelector('#primary_floors');
      const pa     = form.querySelector('#primary_area_sqft');
      const pc     = form.querySelector('#primary_carpet_area');
      const ppmin  = form.querySelector('#primary_price_min');
      const pptxt  = form.querySelector('#primary_price_text');

      if (pb)    pb.value    = first.bedrooms || '';
      if (pt)    pt.value    = first.type || '';
      if (pba)   pba.value   = first.bathrooms || '';
      if (pf)    pf.value    = first.floors || '';
      if (pa)    pa.value    = first.area_sqft || '';
      if (pc)    pc.value    = first.carpet_area || '';
      if (ppmin) ppmin.value = first.price_min || '';
      if (pptxt) pptxt.value = first.price_text || '';
    }

    const numericFields = ['lat','lng','sort_order'];
    numericFields.forEach(function(name){
      const f = form.querySelector('input[name="'+name+'"]');
      if (f && f.value.trim() === '') {
        f.value = '0';
      }
    });
  });
})();

/* Mark required field labels with * automatically */
(function(){
  var fields = document.querySelectorAll('input[required], select[required], textarea[required]');
  fields.forEach(function(field){
    var wrapper = field.closest('.col-xl-12, .col-xl-6, .col-xl-5, .col-xl-4, .col-xl-3, .col-xl-2, .col-md-4, .col-md-3, .col-md-2');
    if (!wrapper) return;
    var label = wrapper.querySelector('label.form-label');
    if (label && !label.classList.contains('required')) {
      label.classList.add('required');
    }
  });
})();
</script>
<?php $scripts = ob_get_clean();

/* ----------------------------- Render base ----------------------------- */
$baseLayout = __DIR__ . '/layouts/base.php';
if (file_exists($baseLayout)) {
  include $baseLayout;
} else {
  ?><!doctype html>
  <html>
  <head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>Project Form</title>
    <?php echo $styles; ?>
  </head>
  <body>
    <div class="container p-3"><?php echo $content; ?></div>
    <?php echo $scripts; ?>
  </body>
  </html><?php
}
