<?php
/* =============================================================================
 * File   : reels_backend.php
 * Goal   : Upload/manage reels + track hearts/shares; Melody admin theme.
 * Stack  : Core PHP + MySQLi (NO PDO). Uses central config + layouts/base.php.
 * CRUD   : Add (upload), Edit title/short_desc, Toggle active, Delete.
 * Notes  : Upload UI restricted via $isAdmin. AJAX hearts/shares usable
 *          from frontend reel viewer / microsite pages.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ---- login guard (same as other pages) ---- */
if (empty($_SESSION['auth']) || !is_array($_SESSION['auth'])) {
  header('Location: login.php');
  exit;
}

/* ------------------------- Centralized config ------------------------- */
if (file_exists(__DIR__ . '/../config.php'))      require_once __DIR__ . '/../config.php';
elseif (file_exists(__DIR__ . '/config.php'))     require_once __DIR__ . '/config.php';
else { http_response_code(500); exit('Missing config.php'); }

/* ------------------------- DB pick (Melody-style) --------------------- */
$DB = null;
if (isset($conn_crm) && $conn_crm instanceof mysqli)              $DB = $conn_crm;
elseif (isset($conn_crmplus) && $conn_crmplus instanceof mysqli)  $DB = $conn_crmplus;
elseif (isset($conn) && $conn instanceof mysqli)                  $DB = $conn;
elseif (isset($conn_sa) && $conn_sa instanceof mysqli)            $DB = $conn_sa;

if (!$DB instanceof mysqli) { http_response_code(500); exit('DB connection not found - check config.php'); }

@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* Prefer `broriserin_onb`, fallback to `broriser_onb` */
if (!@mysqli_query($DB, "USE `broriserin_onb`")) {
  @mysqli_query($DB, "USE `broriser_onb`");
}

/* --------------------------------- Helpers -------------------------------- */
if (!function_exists('h')) {
  function h($s){
    return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
  }
}

/* local, unique helpers so we don't clash with config.php */
if (!function_exists('reel_q')) {
  function reel_q($sql){
    global $DB;
    return @mysqli_query($DB, $sql);
  }
}

if (!function_exists('reel_row')) {
  function reel_row($sql){
    $row = null;
    if ($res = reel_q($sql)) {
      $row = mysqli_fetch_assoc($res);
      mysqli_free_result($res);
    }
    return $row;
  }
}

if (!function_exists('reel_all')) {
  function reel_all($sql){
    $out = [];
    if ($res = reel_q($sql)) {
      while ($r = mysqli_fetch_assoc($res)) $out[] = $r;
      mysqli_free_result($res);
    }
    return $out;
  }
}

if (!function_exists('reel_col_exists')) {
  function reel_col_exists($tbl, $col){
    global $DB;
    $tbl = mysqli_real_escape_string($DB, (string)$tbl);
    $col = mysqli_real_escape_string($DB, (string)$col);
    $res = reel_q("SHOW COLUMNS FROM `{$tbl}` LIKE '{$col}'");
    $ok  = $res && mysqli_num_rows($res) > 0;
    if ($res) mysqli_free_result($res);
    return $ok;
  }
}

if (!function_exists('bytes_from_ini')) {
  function bytes_from_ini($val){
    $val = trim((string)$val);
    if ($val === '') return 0;
    $n = (float)$val;
    $u = strtolower(substr($val, -1));
    switch ($u) {
      case 'g': return (int)($n * 1024 * 1024 * 1024);
      case 'm': return (int)($n * 1024 * 1024);
      case 'k': return (int)($n * 1024);
      default:  return (int)$n;
    }
  }
}

if (!function_exists('human_bytes')) {
  function human_bytes($b){
    $b = (float)$b;
    if ($b >= 1073741824) return number_format($b / 1073741824, 2) . ' GB';
    if ($b >= 1048576)    return number_format($b / 1048576, 2) . ' MB';
    if ($b >= 1024)       return number_format($b / 1024, 2) . ' KB';
    return (int)$b . ' B';
  }
}

if (!function_exists('asset_base')) {
  function asset_base(){
    if (!empty($GLOBALS['baseUrl'])) return rtrim($GLOBALS['baseUrl'], '/');
    $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host  = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $dir   = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? '/'), '/');
    return rtrim("$proto://$host$dir", '/');
  }
}

/* --------------------------- Current user/session --------------------------- */
$ME        = $_SESSION['auth'] ?? [];
$ROLE_NAME = strtolower($ME['role_name'] ?? 'user');
$ME_ID     = (int)($ME['id'] ?? 0);
$ME_MAIL   = trim((string)($ME['email'] ?? ''));
$ME_EMPID  = trim((string)($ME['empid'] ?? ''));

/* Resolve EMPID if missing (same pattern as other pages) */
if ($ME_EMPID === '') {
  $cond = '';
  if ($ME_ID) {
    $cond = "id=" . (int)$ME_ID;
  } elseif ($ME_MAIL !== '') {
    $cond = "email='" . mysqli_real_escape_string($DB, $ME_MAIL) . "'";
  }
  if ($cond !== '') {
    $u = reel_row("SELECT empid FROM broriserin_superadmin.users WHERE {$cond} LIMIT 1");
    if (!empty($u['empid'])) {
      $ME_EMPID = trim($u['empid']);
      $_SESSION['auth']['empid'] = $ME_EMPID;
    }
  }
}

/* ---------------------- Ensure our tables exist ---------------------------- */
reel_q("CREATE TABLE IF NOT EXISTS `reels_videos`(
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `title` VARCHAR(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `short_desc` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `file_url` VARCHAR(255) COLLATE utf8mb4_general_ci NOT NULL,
  `thumb_url` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_by_empid` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_by_email` VARCHAR(190) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `is_active` TINYINT(1) NOT NULL DEFAULT 1,
  `likes_count` INT UNSIGNED NOT NULL DEFAULT 0,
  `shares_count` INT UNSIGNED NOT NULL DEFAULT 0,
  `width` INT DEFAULT NULL,
  `height` INT DEFAULT NULL,
  `duration_seconds` INT DEFAULT NULL,
  PRIMARY KEY (`id`),
  KEY `idx_created` (`created_at`),
  KEY `idx_active` (`is_active`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

/* Add short_desc if table existed without it */
if (!reel_col_exists('reels_videos', 'short_desc')) {
  reel_q("ALTER TABLE reels_videos ADD COLUMN `short_desc` VARCHAR(255) COLLATE utf8mb4_general_ci NULL AFTER `title`");
}

reel_q("CREATE TABLE IF NOT EXISTS `reels_hearts`(
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `reel_id` INT UNSIGNED NOT NULL,
  `user_empid` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `user_email` VARCHAR(190) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uniq_like` (`reel_id`,`user_empid`),
  KEY `idx_reel` (`reel_id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

reel_q("CREATE TABLE IF NOT EXISTS `reels_shares`(
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `reel_id` INT UNSIGNED NOT NULL,
  `user_empid` VARCHAR(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `user_email` VARCHAR(190) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `channel` ENUM('whatsapp','system','copylink','other') NOT NULL DEFAULT 'other',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_reel` (`reel_id`),
  KEY `idx_channel` (`channel`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

/* =============================== AJAX API ================================== */
if (isset($_POST['ajax'])) {
  header('Content-Type: application/json; charset=utf-8');
  $a = $_POST['ajax'];

  if ($a === 'toggle_active') {
    $id  = (int)($_POST['id'] ?? 0);
    $val = (int)($_POST['val'] ?? 0);
    $ok  = $id > 0 ? reel_q("UPDATE reels_videos SET is_active={$val} WHERE id={$id} LIMIT 1") : false;
    echo json_encode(['ok' => (bool)$ok]);
    exit;
  }

  if ($a === 'update_meta') {
    $id    = (int)($_POST['id'] ?? 0);
    $title = mysqli_real_escape_string($DB, trim((string)($_POST['title'] ?? '')));
    $sd    = mysqli_real_escape_string($DB, trim((string)($_POST['short_desc'] ?? '')));
    $ok    = $id > 0 ? reel_q("UPDATE reels_videos SET title='{$title}', short_desc='{$sd}' WHERE id={$id} LIMIT 1") : false;
    echo json_encode(['ok' => (bool)$ok]);
    exit;
  }

  if ($a === 'delete') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id <= 0) {
      echo json_encode(['ok' => false]);
      exit;
    }
    $r  = reel_row("SELECT file_url FROM reels_videos WHERE id={$id} LIMIT 1");
    $ok = reel_q("DELETE FROM reels_videos WHERE id={$id} LIMIT 1");
    if ($ok && $r && !empty($r['file_url'])) {
      $abs = __DIR__ . '/' . ltrim($r['file_url'], '/');
      if (is_file($abs)) { @unlink($abs); }
    }
    echo json_encode(['ok' => (bool)$ok]);
    exit;
  }

  if ($a === 'heart_toggle') {
    $rid = (int)($_POST['id'] ?? 0);
    if ($rid <= 0) {
      echo json_encode(['ok' => false]);
      exit;
    }
    $emp = mysqli_real_escape_string($DB, $ME_EMPID);
    $eml = mysqli_real_escape_string($DB, $ME_MAIL);
    $has = reel_row("SELECT id FROM reels_hearts WHERE reel_id={$rid} AND user_empid='{$emp}' LIMIT 1");
    if ($has) {
      reel_q("DELETE FROM reels_hearts WHERE id=" . (int)$has['id'] . " LIMIT 1");
      reel_q("UPDATE reels_videos SET likes_count=GREATEST(likes_count-1,0) WHERE id={$rid}");
      $liked = false;
    } else {
      reel_q("INSERT INTO reels_hearts (reel_id,user_empid,user_email) VALUES ({$rid},'{$emp}','{$eml}')");
      reel_q("UPDATE reels_videos SET likes_count=likes_count+1 WHERE id={$rid}");
      $liked = true;
    }
    $counts = reel_row("SELECT likes_count, shares_count FROM reels_videos WHERE id={$rid} LIMIT 1") ?: ['likes_count' => 0, 'shares_count' => 0];
    echo json_encode([
      'ok'     => true,
      'liked'  => $liked,
      'likes'  => (int)$counts['likes_count'],
      'shares' => (int)$counts['shares_count']
    ]);
    exit;
  }

  if ($a === 'share') {
    $rid     = (int)($_POST['id'] ?? 0);
    $channel = strtolower(trim((string)($_POST['channel'] ?? 'other')));
    if (!in_array($channel, ['whatsapp','system','copylink','other'], true)) $channel = 'other';
    if ($rid <= 0) {
      echo json_encode(['ok' => false]);
      exit;
    }
    $emp = mysqli_real_escape_string($DB, $ME_EMPID);
    $eml = mysqli_real_escape_string($DB, $ME_MAIL);
    $ok  = reel_q("INSERT INTO reels_shares (reel_id,user_empid,user_email,channel) VALUES ({$rid},'{$emp}','{$eml}','{$channel}')");
    if ($ok) {
      reel_q("UPDATE reels_videos SET shares_count=shares_count+1 WHERE id={$rid}");
    }
    $counts = reel_row("SELECT likes_count, shares_count FROM reels_videos WHERE id={$rid} LIMIT 1") ?: ['likes_count' => 0, 'shares_count' => 0];
    echo json_encode([
      'ok'     => true,
      'likes'  => (int)$counts['likes_count'],
      'shares' => (int)$counts['shares_count']
    ]);
    exit;
  }

  echo json_encode(['ok' => false, 'msg' => 'unknown']);
  exit;
}

/* ============================ Page (Melody UI) ============================ */

/* Upload permissions: restrict upload/list to specific roles */
$isAdmin = in_array($ROLE_NAME, ['superadmin','employee'], true);

/* Server limits (for friendly errors + client-side hint) */
$limit_up   = bytes_from_ini(ini_get('upload_max_filesize'));
$limit_post = bytes_from_ini(ini_get('post_max_size'));
$max_bytes  = min($limit_up ?: PHP_INT_MAX, $limit_post ?: PHP_INT_MAX);

/* Handle upload */
$msg = '';
if ($isAdmin && ($_POST['action'] ?? '') === 'upload' && isset($_FILES['video'])) {
  $file  = $_FILES['video'];
  $title = trim((string)($_POST['title'] ?? ''));
  $short = trim((string)($_POST['short_desc'] ?? ''));

  if ($file['error'] !== UPLOAD_ERR_OK) {
    $errors = [
      UPLOAD_ERR_INI_SIZE   => 'File is larger than server limit (upload_max_filesize).',
      UPLOAD_ERR_FORM_SIZE  => 'File is larger than the form limit (MAX_FILE_SIZE).',
      UPLOAD_ERR_PARTIAL    => 'File was only partially uploaded.',
      UPLOAD_ERR_NO_FILE    => 'No file was uploaded.',
      UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder on server.',
      UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk.',
      UPLOAD_ERR_EXTENSION  => 'Upload stopped by a PHP extension.'
    ];
    $msg = 'Upload error: ' . $file['error'] . '. ' . ($errors[$file['error']] ?? 'Unknown error.');
  } elseif ($max_bytes && $file['size'] > $max_bytes) {
    $msg = 'File size ' . human_bytes($file['size']) . ' exceeds server limit ' . human_bytes($max_bytes) . '.';
  } else {
    $allowedExt = ['mp4','webm','mov','m4v','ogg'];
    $ext        = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, $allowedExt, true)) {
      $msg = 'Only MP4 / WEBM / MOV / M4V / OGG allowed.';
    } else {
      $mime = @mime_content_type($file['tmp_name']);
      if ($mime && strpos($mime, 'video/') !== 0) {
        $msg = 'File is not a video (mime: ' . $mime . ').';
      } else {
        $targetDirRel = 'xhtml-assets/images/reels';
        $targetDirAbs = rtrim(__DIR__ . '/' . $targetDirRel, '/');
        if (!is_dir($targetDirAbs)) {
          @mkdir($targetDirAbs, 0775, true);
        }
        $safeBase = preg_replace('~[^a-zA-Z0-9_\-]~', '_', pathinfo($file['name'], PATHINFO_FILENAME));
        $newName  = 'reel_' . date('Ymd_His') . '_' . mt_rand(1000, 9999) . '_' . $safeBase . '.' . $ext;
        $absPath  = $targetDirAbs . '/' . $newName;
        $relUrl   = $targetDirRel . '/' . $newName;

        if (move_uploaded_file($file['tmp_name'], $absPath)) {
          $emp = mysqli_real_escape_string($DB, $ME_EMPID);
          $eml = mysqli_real_escape_string($DB, $ME_MAIL);
          $ttl = mysqli_real_escape_string($DB, $title);
          $sdc = mysqli_real_escape_string($DB, $short);
          $url = mysqli_real_escape_string($DB, $relUrl);
          $ok  = reel_q("INSERT INTO reels_videos (title,short_desc,file_url,created_by_empid,created_by_email,is_active)
                         VALUES ('{$ttl}','{$sdc}','{$url}','{$emp}','{$eml}',1)");
          $msg = $ok ? 'Uploaded successfully.' : ('DB error: ' . mysqli_error($DB));
        } else {
          $msg = 'Failed to move uploaded file.';
        }
      }
    }
  }
}

/* Fetch list for right panel */
$reels = $isAdmin ? reel_all("SELECT * FROM reels_videos ORDER BY id DESC LIMIT 200") : [];

/* ========================== $styles ========================== */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo asset_base(); ?>/assets/libs/sweetalert2/sweetalert2.min.css">
<style>
  .reels-wrapper{
    max-width: 1200px;
    margin: 0 auto;
  }
  .reels-grid{
    display: grid;
    grid-template-columns: minmax(0, 340px) minmax(0, 1fr);
    gap: 1.25rem;
  }
  @media (max-width: 992px){
    .reels-grid{
      grid-template-columns: minmax(0,1fr);
    }
  }
  .reels-panel{
    background-color: #f9fafb;
    border-radius: 0.75rem;
    padding: 1rem;
    border: 1px solid #e5e7eb;
  }
  .reels-panel-header{
    font-weight: 600;
    font-size: 0.95rem;
    margin-bottom: 0.25rem;
  }
  .reels-panel-sub{
    font-size: 0.78rem;
    color: #6b7280;
  }
  .reels-limits{
    font-size: 0.78rem;
    color: #6b7280;
  }
  .reel-item{
    background-color: #ffffff;
    border-radius: 0.75rem;
    border: 1px solid #e5e7eb;
    padding: 0.6rem;
    margin-bottom: 0.7rem;
  }
  .reel-item-body{
    display: grid;
    grid-template-columns: 120px minmax(0,1fr) auto;
    gap: 0.75rem;
    align-items: center;
  }
  @media (max-width: 768px){
    .reel-item-body{
      grid-template-columns: minmax(0,1fr);
    }
  }
  .reel-thumb video{
    width: 110px;
    height: 180px;
    object-fit: cover;
    background: #000;
    border-radius: 0.75rem;
  }
  @media (max-width: 768px){
    .reel-thumb video{
      width: 100%;
      height: 260px;
    }
  }
  .reel-meta{
    display: grid;
    gap: 0.35rem;
  }
  .reel-meta-two{
    display: grid;
    grid-template-columns: minmax(0,1fr) minmax(0,1fr);
    gap: 0.5rem;
  }
  @media (max-width: 520px){
    .reel-meta-two{
      grid-template-columns: minmax(0,1fr);
    }
  }
  .reel-url{
    font-size: 0.78rem;
    color: #4b5563;
  }
  .reel-url input{
    font-size: 0.78rem;
  }
  .reel-chips{
    display: flex;
    flex-wrap: wrap;
    gap: 0.35rem;
  }
  .reel-chip{
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.15rem 0.5rem;
    font-size: 0.76rem;
    border-radius: 999px;
    background-color: #eef2ff;
    font-weight: 600;
  }
  .reel-chip--shares{
    background-color: #ecfeff;
  }
  .reel-footnote{
    font-size: 0.74rem;
    color: #6b7280;
  }
  .reel-actions{
    display: flex;
    flex-direction: column;
    align-items: flex-end;
    gap: 0.35rem;
  }
  @media (max-width: 768px){
    .reel-actions{
      flex-direction: row;
      justify-content: flex-start;
    }
  }
</style>
<?php $styles = ob_get_clean();

/* ========================== $content ========================= */
ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap gap-2">
    <h1 class="page-title fw-medium fs-18 mb-0">Reels Manager</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">Marketing</a></li>
      <li class="breadcrumb-item active">Reels</li>
    </ol>
  </div>
</div>

<div class="reels-wrapper">
  <div class="card custom-card">
    <div class="card-header justify-content-between border-bottom-0">
      <div>
        <div class="card-title">Project Reels — Upload &amp; Manage</div>
        <div class="text-muted small">
          Files are stored under <code>/brocrm/mobile/xhtml-assets/images/reels/</code> and surfaced on the frontend reels viewer.
        </div>
      </div>
    </div>
    <div class="card-body">
      <?php if (!$isAdmin): ?>
        <div class="alert alert-danger mb-0">
          403 — Only admins can access the reels upload interface.
        </div>
      <?php else: ?>
        <?php if ($msg !== ''): ?>
          <?php $isOk = stripos($msg, 'success') !== false; ?>
          <div class="alert <?php echo $isOk ? 'alert-success' : 'alert-danger'; ?> mb-3">
            <?php echo h($msg); ?>
          </div>
        <?php endif; ?>

        <div class="reels-limits mb-3">
          Server limits:
          upload_max_filesize <strong><?php echo h(ini_get('upload_max_filesize')); ?></strong>,
          post_max_size <strong><?php echo h(ini_get('post_max_size')); ?></strong>
          <?php if ($max_bytes): ?>
            (effective max ~<?php echo h(human_bytes($max_bytes)); ?>).
          <?php endif; ?>
        </div>

        <div class="reels-grid">
          <!-- Left: upload -->
          <div class="reels-panel">
            <div class="reels-panel-header">Add new reel</div>
            <div class="reels-panel-sub mb-2">
              Title &amp; Short Description are optional and are overlaid on the reel on frontend.
            </div>
            <form method="post" enctype="multipart/form-data" id="uploadForm">
              <input type="hidden" name="action" value="upload">
              <div class="mb-2">
                <label class="form-label small mb-1">Title</label>
                <input class="form-control form-control-sm" type="text" name="title" placeholder="Title (optional)">
              </div>
              <div class="mb-2">
                <label class="form-label small mb-1">Short description</label>
                <textarea class="form-control form-control-sm" name="short_desc" rows="2" placeholder="Short description (optional)"></textarea>
              </div>
              <div class="mb-1">
                <label class="form-label small mb-1">Video file</label>
                <input class="form-control form-control-sm" type="file" id="fileInput" name="video" accept="video/*" required>
                <div class="reels-limits mt-1" id="sizeHint"></div>
              </div>
              <div class="mt-2">
                <button class="btn btn-primary btn-sm" type="submit">
                  <i class="fa fa-upload me-1"></i>Upload
                </button>
              </div>
            </form>
            <div class="reels-limits mt-3">
              If you get <code>Upload error: 1</code>, increase limits with <code>.htaccess</code> under
              <code>/brocrm/mobile/</code>:
<pre class="mt-1 mb-0" style="background:#f3f4f6;padding:8px;border-radius:0.5rem;font-size:11px;">
php_value upload_max_filesize 64M
php_value post_max_size 64M
php_value max_file_uploads 20
php_value memory_limit 256M
php_value max_execution_time 120
php_value max_input_time 120</pre>
            </div>
          </div>

          <!-- Right: list/manage -->
          <div class="reels-panel">
            <div class="d-flex justify-content-between align-items-center mb-2">
              <div class="reels-panel-header mb-0">Manage reels</div>
              <?php if ($reels): ?>
                <span class="badge bg-light text-muted fs-11">Total: <?php echo count($reels); ?></span>
              <?php endif; ?>
            </div>
            <?php if (!$reels): ?>
              <div class="text-muted small">No reels uploaded yet.</div>
            <?php else: ?>
              <div id="reelsList">
                <?php foreach ($reels as $r): ?>
                  <div class="reel-item" data-id="<?php echo (int)$r['id']; ?>">
                    <div class="reel-item-body">
                      <div class="reel-thumb">
                        <video src="<?php echo h($r['file_url']); ?>" muted playsinline></video>
                      </div>
                      <div class="reel-meta">
                        <div class="reel-meta-two">
                          <div>
                            <label class="form-label small mb-1">Title</label>
                            <input class="form-control form-control-sm js-ttl" type="text" placeholder="Title" value="<?php echo h($r['title']); ?>">
                          </div>
                          <div>
                            <label class="form-label small mb-1">Short description</label>
                            <input class="form-control form-control-sm js-sdesc" type="text" placeholder="Short description" value="<?php echo h($r['short_desc']); ?>">
                          </div>
                        </div>
                        <div class="reel-url">
                          <label class="form-label small mb-1 mb-0">File URL</label>
                          <input class="form-control form-control-sm" value="<?php echo h($r['file_url']); ?>" readonly>
                        </div>
                        <div class="reel-chips">
                          <div class="reel-chip">
                            <i class="fa fa-heart text-danger"></i>
                            Likes: <?php echo (int)$r['likes_count']; ?>
                          </div>
                          <div class="reel-chip reel-chip--shares">
                            <i class="fa fa-share-alt"></i>
                            Shares: <?php echo (int)$r['shares_count']; ?>
                          </div>
                        </div>
                        <div class="reel-footnote">
                          Uploaded <?php echo h(date('d M Y, h:i A', strtotime($r['created_at']))); ?>
                        </div>
                      </div>
                      <div class="reel-actions">
                        <div class="form-check form-switch mb-0">
                          <input class="form-check-input js-tg" type="checkbox" role="switch" <?php echo $r['is_active'] ? 'checked' : ''; ?>>
                        </div>
                        <button type="button" class="btn btn-outline-secondary btn-xs js-save">
                          <i class="fa fa-save me-1"></i>Save
                        </button>
                        <button type="button" class="btn btn-outline-danger btn-xs js-del">
                          <i class="fa fa-trash me-1"></i>Delete
                        </button>
                      </div>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>
            <?php endif; ?>
          </div>
        </div>
      <?php endif; ?>
    </div>
  </div>
</div>

<?php $content = ob_get_clean();

/* ========================== $scripts ========================= */
ob_start(); ?>
<script src="<?php echo asset_base(); ?>/assets/libs/sweetalert2/sweetalert2.min.js"></script>
<script>
(function(){
  var maxBytes = <?php echo (int)$max_bytes; ?>;
  var hint     = document.getElementById('sizeHint');
  if (hint && maxBytes) {
    hint.textContent = 'Max upload size: ~' + formatBytes(maxBytes) + '.';
  }
  var fi = document.getElementById('fileInput');
  if (fi && maxBytes) {
    fi.addEventListener('change', function(){
      var f = this.files && this.files[0];
      if (!f) return;
      if (f.size > maxBytes) {
        alert('Selected file is ' + formatBytes(f.size) + ', which exceeds server limit of ~' + formatBytes(maxBytes) + '.\nPlease choose a smaller file or increase server limits.');
        this.value = '';
      }
    });
  }
  function formatBytes(b){
    if (b >= 1073741824) return (b/1073741824).toFixed(2) + ' GB';
    if (b >= 1048576)    return (b/1048576).toFixed(2) + ' MB';
    if (b >= 1024)       return (b/1024).toFixed(2) + ' KB';
    return b + ' B';
  }

  /* Toggle active */
  document.querySelectorAll('.reel-item .js-tg').forEach(function(chk){
    chk.addEventListener('change', function(){
      var row = this.closest('.reel-item');
      if (!row) return;
      var id = row.getAttribute('data-id');
      fetch('reels_backend.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: new URLSearchParams({ajax:'toggle_active', id:id, val:this.checked ? 1 : 0})
      });
    }, {passive:true});
  });

  /* Save meta */
  document.querySelectorAll('.reel-item .js-save').forEach(function(btn){
    btn.addEventListener('click', function(){
      var row = this.closest('.reel-item');
      if (!row) return;
      var id  = row.getAttribute('data-id');
      var ttl = row.querySelector('.js-ttl') ? row.querySelector('.js-ttl').value : '';
      var sd  = row.querySelector('.js-sdesc') ? row.querySelector('.js-sdesc').value : '';
      fetch('reels_backend.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: new URLSearchParams({ajax:'update_meta', id:id, title:ttl, short_desc:sd})
      }).then(function(r){ return r.json(); }).then(function(j){
        if (j && j.ok) {
          var old = btn.innerHTML;
          btn.innerHTML = '<i class="fa fa-check me-1"></i>Saved';
          setTimeout(function(){ btn.innerHTML = old; }, 900);
        }
      });
    });
  });

  /* Delete reel (with SweetAlert if available) */
  document.querySelectorAll('.reel-item .js-del').forEach(function(btn){
    btn.addEventListener('click', function(){
      var row = this.closest('.reel-item');
      if (!row) return;
      var id = row.getAttribute('data-id');

      function doDelete(){
        fetch('reels_backend.php', {
          method: 'POST',
          headers: {'Content-Type': 'application/x-www-form-urlencoded'},
          body: new URLSearchParams({ajax:'delete', id:id})
        }).then(function(r){ return r.json(); }).then(function(j){
          if (j && j.ok) {
            row.remove();
          }
        });
      }

      if (typeof Swal !== 'undefined') {
        Swal.fire({
          title: 'Delete this reel?',
          text: 'This action cannot be undone.',
          icon: 'warning',
          showCancelButton: true,
          confirmButtonText: 'Yes, delete',
          cancelButtonText: 'Cancel'
        }).then(function(result){
          if (result.isConfirmed) {
            doDelete();
          }
        });
      } else {
        if (confirm('Delete this reel? This action cannot be undone.')) {
          doDelete();
        }
      }
    });
  });
})();
</script>
<?php $scripts = ob_get_clean();

/* render */
include __DIR__ . '/layouts/base.php';
