<?php
/* =============================================================================
 * File   : st2-pif-proceed.php
 * Goal   : Stage-2 PIF Proceed (HR Evaluation + Reviewers) in Soziety theme
 * Stack  : Core PHP + MySQLi, DB via ../config.php, COLLATE=utf8mb4_general_ci
 *
 * URL    : st2-pif-proceed.php?id={idpif}
 * Notes  :
 *  - Applicant details (PIF) are fetched from user_pif.
 *  - New fields stored in user_pif:
 *        appointed_as, eval_r1_hr, reviewer_details (JSON),
 *        module_name, final_decision, final_decision_remark
 *  - Reviewer 1 (HR) decision  -> st1_status
 *    Reviewer 1 (HR) remarks   -> st1_feedback
 *    Reviewer 1 date           -> st1_date (now)
 *  - Additional reviewers are stored as JSON in reviewer_details.
 *  - AJAX endpoint: ?ajax=empinfo&empid=XXXX
 *    (returns name + team from intbroriser_hrd.hr_dump via $hrd)
 *  - For each Reviewer, this page now:
 *      * Captures EmpID (then auto-fills Name & Team).
 *      * Shows a "Create Interview Link" button which opens
 *        pif-reviewer-interview.php in a new tab.
 *      * Decision & Remarks are NOT taken on this page; they are
 *        entered by the reviewer in the separate page.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/../config.php';

/* Charset / collation for main ONB connection ($conn) */
if (isset($conn) && $conn instanceof mysqli) {
    mysqli_set_charset($conn, 'utf8mb4');
    @mysqli_query($conn, "SET NAMES 'utf8mb4'");
    @mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");
} else {
    http_response_code(500);
    die('Database connection ($conn) not found. Check config.php');
}

/**
 * Cache table columns to allow conditional updates (e.g., skip columns
 * that are missing in certain environments). This prevents SQL errors like
 * "Unknown column 'fresh_hire' in 'field list'" when the database schema
 * lags behind the code.
 */
function getTableColumns(mysqli $conn, string $table): array
{
    static $cache = [];

    if (isset($cache[$table])) {
        return $cache[$table];
    }

    $columns  = [];
    $tableEsc = $conn->real_escape_string($table);
    $sql      = "SHOW COLUMNS FROM `{$tableEsc}`";

    if ($result = $conn->query($sql)) {
        while ($row = $result->fetch_assoc()) {
            if (isset($row['Field'])) {
                $columns[$row['Field']] = true;
            }
        }
        $result->close();
    }

    $cache[$table] = $columns;

    return $columns;
}

function hasTableColumn(mysqli $conn, string $table, string $column): bool
{
    $cols = getTableColumns($conn, $table);
    return isset($cols[$column]);
}

/* --------------------------------------------------------------------------
 * AJAX: Get Employee Name + Team from intbroriser_hrd.hr_dump by EmpID
 * Uses central HR connection $hrd from config.php
 * ----------------------------------------------------------------------- */
if (isset($_GET['ajax']) && $_GET['ajax'] === 'empinfo') {
    header('Content-Type: application/json; charset=utf-8');

    $empid = isset($_GET['empid']) ? trim($_GET['empid']) : '';
    $resp  = ['name' => '', 'team' => ''];

    // Prefer dedicated HR connection; fallback to $conn if needed
    $dbHr = null;
    if (isset($hrd) && $hrd instanceof mysqli) {
        $dbHr = $hrd;
    } elseif (isset($conn) && $conn instanceof mysqli) {
        $dbHr = $conn;
    }

    if ($empid !== '' && $dbHr instanceof mysqli) {
        $sql = "
            SELECT Employee_Full_Name, Team_name
            FROM hr_dump
            WHERE empid = ?
            LIMIT 1
        ";

        if ($stmt = $dbHr->prepare($sql)) {
            $stmt->bind_param('s', $empid);
            $stmt->execute();
            $res = $stmt->get_result();

            if ($row = $res->fetch_assoc()) {
                $resp['name'] = (string) $row['Employee_Full_Name'];
                $resp['team'] = (string) $row['Team_name'];
            }

            $stmt->close();
        }
    }

    echo json_encode($resp);
    exit;
}

/* --------------------------------------------------------------------------
 * Resolve logged-in EMPID (used for interview_update_EMPID / audit)
 * ----------------------------------------------------------------------- */
$loggedEmpid = '';

if (!empty($_SESSION['empid'])) {
    $loggedEmpid = (string) $_SESSION['empid'];
} elseif (!empty($_SESSION['hr_empid'])) {
    $loggedEmpid = (string) $_SESSION['hr_empid'];
}

/* --------------------------------------------------------------------------
 * Get PIF ID
 * ----------------------------------------------------------------------- */
$idpif = isset($_GET['id']) ? (int) $_GET['id'] : 0;

if ($idpif <= 0) {
    http_response_code(400);
    die('Missing or invalid PIF ID.');
}

/* --------------------------------------------------------------------------
 * Fetch PIF row
 * ----------------------------------------------------------------------- */
$pif    = null;
$sqlPif = "SELECT * FROM user_pif WHERE idpif = ? LIMIT 1";

if ($stmt = $conn->prepare($sqlPif)) {
    $stmt->bind_param('i', $idpif);
    $stmt->execute();
    $res = $stmt->get_result();
    $pif = $res->fetch_assoc();
    $stmt->close();
}

if (!$pif) {
    http_response_code(404);
    die('PIF record not found.');
}

/* --------------------------------------------------------------------------
 * AJAX: Save only HR sections (To Be Appointed As + Evaluation Criteria)
 * Triggered before creating interviewer link to avoid blank values.
 * ----------------------------------------------------------------------- */
if (
    $_SERVER['REQUEST_METHOD'] === 'POST'
    && isset($_POST['ajax'])
    && $_POST['ajax'] === 'save_hr_sections'
) {
    header('Content-Type: application/json; charset=utf-8');

    // Sanitize incoming fields
    $appointed_as     = trim($_POST['appointed_as'] ?? '');
    $fresh_hire       = trim($_POST['fresh_hire'] ?? '');
    $ad_director_id   = trim($_POST['ad_director_id'] ?? '');
    $ad_director_name = trim($_POST['ad_director_name'] ?? '');
    $team_name        = trim($_POST['team_name'] ?? '');
    $Department       = trim($_POST['Department'] ?? '');
    $Location_of_Work = trim($_POST['Location_of_Work'] ?? '');
    $designation      = trim($_POST['designation'] ?? '');
    $ctc_per_month    = trim($_POST['ctc_per_month'] ?? '');
    $expected_doj     = trim($_POST['expected_doj'] ?? '');
    $actual_doj       = trim($_POST['actual_doj'] ?? '');
    $reporting_manager = trim($_POST['reporting_manager'] ?? '');
    $rm_code           = trim($_POST['rm_code'] ?? '');
    $hr_ta             = trim($_POST['hr_ta'] ?? '');
    $hr_head_dgm       = trim($_POST['hr_head_dgm'] ?? '');

    $r1_qual       = trim($_POST['r1_qual'] ?? '');
    $r1_knowledge  = trim($_POST['r1_knowledge'] ?? '');
    $r1_experience = trim($_POST['r1_experience'] ?? '');
    $r1_record     = trim($_POST['r1_record'] ?? '');
    $r1_initiative = trim($_POST['r1_initiative'] ?? '');
    $r1_tplayer    = trim($_POST['r1_tplayer'] ?? '');
    $r1_leadership = trim($_POST['r1_leadership'] ?? '');
    $r1_stability  = trim($_POST['r1_stability'] ?? '');
    $r1_comskill   = trim($_POST['r1_comskill'] ?? '');
    $r1_ratings    = trim($_POST['r1_ratings'] ?? '');

    $interview_date = trim($_POST['interview_date'] ?? '');
    $r1_decision    = trim($_POST['r1_decision'] ?? '');
    $r1_remarks     = trim($_POST['r1_remarks'] ?? '');
    $eval_r1_hr     = trim($_POST['eval_r1_hr'] ?? '');

    if ($appointed_as === '' || $eval_r1_hr === '') {
        echo json_encode([
            'success' => false,
            'message' => 'Please complete "To Be Appointed As" and "Evaluation Criteria" before creating the link.'
        ]);
        exit;
    }

    $st1_date = date('Y-m-d H:i:s');
    $auditEmp = $loggedEmpid !== '' ? $loggedEmpid : null;

    $tableCols = getTableColumns($conn, 'user_pif');
    $types     = '';
    $values    = [];
    $setParts  = [];

    $setParts[] = 'appointed_as           = ?';
    $types     .= 's';
    $values[]   = $appointed_as;

    if (isset($tableCols['fresh_hire'])) {
        $setParts[] = 'fresh_hire             = ?';
        $types     .= 's';
        $values[]   = $fresh_hire;
    }

    $hasAdDirectorId   = isset($tableCols['ad_director_id']);
    $hasAdDirectorName = isset($tableCols['ad_director_name']);

    if ($hasAdDirectorId) {
        $setParts[] = 'ad_director_id         = ?';
        $types     .= 's';
        $values[]   = $ad_director_id;
    }

    if ($hasAdDirectorName) {
        $setParts[] = 'ad_director_name       = ?';
        $types     .= 's';
        $values[]   = $ad_director_name;
    }

    $setParts[] = 'team_name              = ?';
    $types     .= 's';
    $values[]   = $team_name;

    $setParts[] = 'Department             = ?';
    $types     .= 's';
    $values[]   = $Department;

    $setParts[] = 'Location_of_Work       = ?';
    $types     .= 's';
    $values[]   = $Location_of_Work;

    $setParts[] = 'designation            = ?';
    $types     .= 's';
    $values[]   = $designation;

    $setParts[] = 'ctc_per_month          = ?';
    $types     .= 's';
    $values[]   = $ctc_per_month;

    $setParts[] = 'expected_doj           = ?';
    $types     .= 's';
    $values[]   = $expected_doj;

    $setParts[] = 'actual_doj             = ?';
    $types     .= 's';
    $values[]   = $actual_doj;

    $setParts[] = 'reporting_manager      = ?';
    $types     .= 's';
    $values[]   = $reporting_manager;

    $setParts[] = 'rm_code                = ?';
    $types     .= 's';
    $values[]   = $rm_code;

    $setParts[] = 'hr_ta                  = ?';
    $types     .= 's';
    $values[]   = $hr_ta;

    $setParts[] = 'hr_head_dgm            = ?';
    $types     .= 's';
    $values[]   = $hr_head_dgm;

    $setParts[] = 'r1_qual                = ?';
    $types     .= 's';
    $values[]   = $r1_qual;

    $setParts[] = 'r1_knowledge           = ?';
    $types     .= 's';
    $values[]   = $r1_knowledge;

    $setParts[] = 'r1_experience          = ?';
    $types     .= 's';
    $values[]   = $r1_experience;

    $setParts[] = 'r1_record              = ?';
    $types     .= 's';
    $values[]   = $r1_record;

    $setParts[] = 'r1_initiative          = ?';
    $types     .= 's';
    $values[]   = $r1_initiative;

    $setParts[] = 'r1_tplayer             = ?';
    $types     .= 's';
    $values[]   = $r1_tplayer;

    $setParts[] = 'r1_leadership          = ?';
    $types     .= 's';
    $values[]   = $r1_leadership;

    $setParts[] = 'r1_stability           = ?';
    $types     .= 's';
    $values[]   = $r1_stability;

    $setParts[] = 'r1_comskill            = ?';
    $types     .= 's';
    $values[]   = $r1_comskill;

    $setParts[] = 'r1_ratings             = ?';
    $types     .= 's';
    $values[]   = $r1_ratings;

    $setParts[] = 'interview_date         = ?';
    $types     .= 's';
    $values[]   = $interview_date;

    $setParts[] = 'eval_r1_hr             = ?';
    $types     .= 's';
    $values[]   = $eval_r1_hr;

    $setParts[] = 'st1_status             = ?';
    $types     .= 's';
    $values[]   = $r1_decision;

    $setParts[] = 'st1_feedback           = ?';
    $types     .= 's';
    $values[]   = $r1_remarks;

    $setParts[] = 'st1_date               = ?';
    $types     .= 's';
    $values[]   = $st1_date;

    $setParts[] = 'interview_update_EMPID = ?';
    $types     .= 's';
    $values[]   = $auditEmp;

    $types   .= 'i';
    $values[] = $idpif;

    $sql = "
        UPDATE user_pif
           SET " . implode(",\n               ", $setParts) . "
         WHERE idpif = ?
         LIMIT 1
    ";

    if ($stmt = $conn->prepare($sql)) {
        $bindParams = array_merge([$types], $values);

        foreach ($bindParams as $idx => $val) {
            $bindParams[$idx] = &$bindParams[$idx];
        }

        call_user_func_array([$stmt, 'bind_param'], $bindParams);

        if ($stmt->execute()) {
            echo json_encode([
                'success' => true,
                'message' => 'HR details saved. Creating interview link now.'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Could not save HR details. ' . $stmt->error
            ]);
        }

        $stmt->close();
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Could not prepare save statement. ' . $conn->error
        ]);
    }

    exit;
}

/* Decode existing reviewer_details JSON */
$existingReviewers = [];

if (!empty($pif['reviewer_details'])) {
    $tmp = json_decode($pif['reviewer_details'], true);
    if (is_array($tmp)) {
        $existingReviewers = $tmp;
    }
}

if (empty($existingReviewers)) {
    // At least one reviewer block (Reviewer 2)
    $existingReviewers[] = [
        'empid'    => '',
        'name'     => '',
        'team'     => '',
        'decision' => '',
        'remarks'  => '',
    ];
}

/* --------------------------------------------------------------------------
 * Handle POST (Save Evaluation)
 * ----------------------------------------------------------------------- */
$successMsg = '';
$errorMsg   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // Basic sanitization
    $appointed_as          = trim($_POST['appointed_as'] ?? '');
    $fresh_hire            = trim($_POST['fresh_hire'] ?? '');
    $ad_director_id        = trim($_POST['ad_director_id'] ?? '');
    $ad_director_name      = trim($_POST['ad_director_name'] ?? '');
    $team_name             = trim($_POST['team_name'] ?? '');
    $Department            = trim($_POST['Department'] ?? '');
    $Location_of_Work      = trim($_POST['Location_of_Work'] ?? '');
    $designation           = trim($_POST['designation'] ?? '');
    $ctc_per_month         = trim($_POST['ctc_per_month'] ?? '');
    $expected_doj          = trim($_POST['expected_doj'] ?? '');
    $actual_doj            = trim($_POST['actual_doj'] ?? '');
    $reporting_manager     = trim($_POST['reporting_manager'] ?? '');
    $rm_code               = trim($_POST['rm_code'] ?? '');
    $hr_ta                 = trim($_POST['hr_ta'] ?? '');
    $hr_head_dgm           = trim($_POST['hr_head_dgm'] ?? '');
    $r1_qual               = trim($_POST['r1_qual'] ?? '');
    $r1_knowledge          = trim($_POST['r1_knowledge'] ?? '');
    $r1_experience         = trim($_POST['r1_experience'] ?? '');
    $r1_record             = trim($_POST['r1_record'] ?? '');
    $r1_initiative         = trim($_POST['r1_initiative'] ?? '');
    $r1_tplayer            = trim($_POST['r1_tplayer'] ?? '');
    $r1_leadership         = trim($_POST['r1_leadership'] ?? '');
    $r1_stability          = trim($_POST['r1_stability'] ?? '');
    $r1_comskill           = trim($_POST['r1_comskill'] ?? '');
    $r1_ratings            = trim($_POST['r1_ratings'] ?? '');
    $interview_date        = trim($_POST['interview_date'] ?? '');
    $r1_decision           = trim($_POST['r1_decision'] ?? '');
    $r1_remarks            = trim($_POST['r1_remarks'] ?? '');
    $eval_r1_hr            = trim($_POST['eval_r1_hr'] ?? '');
    $module_final          = trim($_POST['module_final'] ?? '');
    $module_type           = trim($_POST['module_type'] ?? '');
    $module_name           = $module_final !== '' ? $module_final : trim($_POST['module_name'] ?? '');
    $final_decision        = trim($_POST['final_decision'] ?? '');
    $final_decision_remark = trim($_POST['final_decision_remark'] ?? '');

    // Reviewers (2,3,4,...) as array
    $reviewers      = $_POST['reviewers'] ?? [];
    $cleanReviewers = [];

    if (is_array($reviewers)) {
        foreach ($reviewers as $row) {
            $empid    = isset($row['empid']) ? trim($row['empid']) : '';
            $name     = isset($row['name']) ? trim($row['name']) : '';
            $team     = isset($row['team']) ? trim($row['team']) : '';

            // Decision & Remarks are now captured in the separate
            // reviewer page; keep them blank here (for backward
            // compatibility we still read keys if present).
            $decision = isset($row['decision']) ? trim($row['decision']) : '';
            $remarks  = isset($row['remarks']) ? trim($row['remarks']) : '';

            // Skip completely empty rows
            if ($empid === '' && $name === '' && $team === '') {
                continue;
            }

            $cleanReviewers[] = [
                'empid'    => $empid,
                'name'     => $name,
                'team'     => $team,
                'decision' => $decision,
                'remarks'  => $remarks,
            ];
        }
    }

    $reviewerJson = $cleanReviewers ? json_encode($cleanReviewers) : null;

    // Minimal validation
    if ($appointed_as === '') {
        $errorMsg = 'Please fill "To Be Appointed As".';
    } elseif ($fresh_hire === '' || $ad_director_id === '' || $ad_director_name === '') {
        $errorMsg = 'Please complete all mandatory "To Be Appointed As" fields.';
    } elseif ($r1_decision === '') {
        $errorMsg = 'Please select Reviewer 1 (HR) decision.';
    } elseif ($module_final === '' || $module_type === '' || $final_decision === '') {
        $errorMsg = 'Please fill "Module / Final Decision".';
    } elseif ($interview_date === '') {
        $errorMsg = 'Please choose the interview date.';
    }

    if ($errorMsg === '') {
        $st1_status   = $r1_decision;
        $st1_feedback = $r1_remarks;
        $st1_date     = date('Y-m-d H:i:s');

        $tableCols = getTableColumns($conn, 'user_pif');
        $types     = '';
        $values    = [];
        $setParts  = [];

        $setParts[] = 'appointed_as           = ?';
        $types     .= 's';
        $values[]   = $appointed_as;

        if (isset($tableCols['fresh_hire'])) {
            $setParts[] = 'fresh_hire             = ?';
            $types     .= 's';
            $values[]   = $fresh_hire;
        }

        $hasAdDirectorId   = isset($tableCols['ad_director_id']);
        $hasAdDirectorName = isset($tableCols['ad_director_name']);

        if ($hasAdDirectorId) {
            $setParts[] = 'ad_director_id         = ?';
            $types     .= 's';
            $values[]   = $ad_director_id;
        }

        if ($hasAdDirectorName) {
            $setParts[] = 'ad_director_name       = ?';
            $types     .= 's';
            $values[]   = $ad_director_name;
        }

        $setParts[] = 'team_name              = ?';
        $types     .= 's';
        $values[]   = $team_name;

        $setParts[] = 'Department             = ?';
        $types     .= 's';
        $values[]   = $Department;

        $setParts[] = 'Location_of_Work       = ?';
        $types     .= 's';
        $values[]   = $Location_of_Work;

        $setParts[] = 'designation            = ?';
        $types     .= 's';
        $values[]   = $designation;

        $setParts[] = 'ctc_per_month          = ?';
        $types     .= 's';
        $values[]   = $ctc_per_month;

        $setParts[] = 'expected_doj           = ?';
        $types     .= 's';
        $values[]   = $expected_doj;

        $setParts[] = 'actual_doj             = ?';
        $types     .= 's';
        $values[]   = $actual_doj;

        $setParts[] = 'reporting_manager      = ?';
        $types     .= 's';
        $values[]   = $reporting_manager;

        $setParts[] = 'rm_code                = ?';
        $types     .= 's';
        $values[]   = $rm_code;

        $setParts[] = 'hr_ta                  = ?';
        $types     .= 's';
        $values[]   = $hr_ta;

        $setParts[] = 'hr_head_dgm            = ?';
        $types     .= 's';
        $values[]   = $hr_head_dgm;

        $setParts[] = 'r1_qual                = ?';
        $types     .= 's';
        $values[]   = $r1_qual;

        $setParts[] = 'r1_knowledge           = ?';
        $types     .= 's';
        $values[]   = $r1_knowledge;

        $setParts[] = 'r1_experience          = ?';
        $types     .= 's';
        $values[]   = $r1_experience;

        $setParts[] = 'r1_record              = ?';
        $types     .= 's';
        $values[]   = $r1_record;

        $setParts[] = 'r1_initiative          = ?';
        $types     .= 's';
        $values[]   = $r1_initiative;

        $setParts[] = 'r1_tplayer             = ?';
        $types     .= 's';
        $values[]   = $r1_tplayer;

        $setParts[] = 'r1_leadership          = ?';
        $types     .= 's';
        $values[]   = $r1_leadership;

        $setParts[] = 'r1_stability           = ?';
        $types     .= 's';
        $values[]   = $r1_stability;

        $setParts[] = 'r1_comskill            = ?';
        $types     .= 's';
        $values[]   = $r1_comskill;

        $setParts[] = 'r1_ratings             = ?';
        $types     .= 's';
        $values[]   = $r1_ratings;

        $setParts[] = 'interview_date         = ?';
        $types     .= 's';
        $values[]   = $interview_date;

        $setParts[] = 'eval_r1_hr             = ?';
        $types     .= 's';
        $values[]   = $eval_r1_hr;

        $setParts[] = 'st1_status             = ?';
        $types     .= 's';
        $values[]   = $st1_status;

        $setParts[] = 'st1_feedback           = ?';
        $types     .= 's';
        $values[]   = $st1_feedback;

        $setParts[] = 'st1_date               = ?';
        $types     .= 's';
        $values[]   = $st1_date;

        $setParts[] = 'reviewer_details       = ?';
        $types     .= 's';
        $values[]   = $reviewerJson;

        $setParts[] = 'module_final           = ?';
        $types     .= 's';
        $values[]   = $module_final;

        $setParts[] = 'module_type            = ?';
        $types     .= 's';
        $values[]   = $module_type;

        $setParts[] = 'module_name            = ?';
        $types     .= 's';
        $values[]   = $module_name;

        $setParts[] = 'final_decision         = ?';
        $types     .= 's';
        $values[]   = $final_decision;

        $setParts[] = 'final_decision_remark  = ?';
        $types     .= 's';
        $values[]   = $final_decision_remark;

        $setParts[] = 'interview_update_EMPID = ?';
        $types     .= 's';
        $values[]   = $loggedEmpid !== '' ? $loggedEmpid : null;

        $types   .= 'i';
        $values[] = $idpif;

        $updateSql = "
            UPDATE user_pif
               SET " . implode(",\n                   ", $setParts) . "
             WHERE idpif = ?
        ";

        if ($stmt = $conn->prepare($updateSql)) {
            $bindParams = array_merge([$types], $values);

            foreach ($bindParams as $idx => $val) {
                $bindParams[$idx] = &$bindParams[$idx];
            }

            call_user_func_array([$stmt, 'bind_param'], $bindParams);

            if ($stmt->execute()) {
                $successMsg = 'PIF evaluation saved successfully.';

                // Refresh $pif for display after save
                $sqlReload = "SELECT * FROM user_pif WHERE idpif = ? LIMIT 1";
                if ($stmt2 = $conn->prepare($sqlReload)) {
                    $stmt2->bind_param('i', $idpif);
                    $stmt2->execute();
                    $res2 = $stmt2->get_result();
                    if ($row2 = $res2->fetch_assoc()) {
                        $pif = $row2;
                    }
                    $stmt2->close();
                }

                // Re-decode reviewers
                $existingReviewers = [];
                if (!empty($pif['reviewer_details'])) {
                    $tmp = json_decode($pif['reviewer_details'], true);
                    if (is_array($tmp)) {
                        $existingReviewers = $tmp;
                    }
                }

                if (empty($existingReviewers)) {
                    $existingReviewers[] = [
                        'empid'    => '',
                        'name'     => '',
                        'team'     => '',
                        'decision' => '',
                        'remarks'  => '',
                    ];
                }
            } else {
                $errorMsg = 'DB error while saving evaluation. (' . $stmt->error . ')';
            }

            $stmt->close();
        } else {
            $errorMsg = 'Unable to prepare update statement. (' . $conn->error . ')';
        }
    }
}

/* --------------------------------------------------------------------------
 * Base URL & resume path
 * ----------------------------------------------------------------------- */
$rootFolder = basename($_SERVER['DOCUMENT_ROOT']);

$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on'
        ? 'https://'
        : 'http://')
    . $_SERVER['HTTP_HOST']
    . str_replace(
        '/admin/applications',
        '',
        str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']))
    );

// Adjust this if your resume upload folder is different:
$resumeBasePath = $baseUrl . '/pif-uploads';

$resumeUrl = '';
if (!empty($pif['resume'])) {
    $resumeUrl = $resumeBasePath . '/' . rawurlencode($pif['resume']);
}

/* Lookup dropdown options */
$teamOptions       = [];
$departmentOptions = [];
$locationOptions   = [];

if ($conn instanceof mysqli) {
    if ($res = $conn->query("SELECT team_name FROM ic_team_name WHERE status='Active' ORDER BY team_name ASC")) {
        while ($row = $res->fetch_assoc()) {
            $teamOptions[] = (string) $row['team_name'];
        }
        $res->close();
    }

    if ($res = $conn->query("SELECT Department FROM ic_department ORDER BY Department ASC")) {
        while ($row = $res->fetch_assoc()) {
            $departmentOptions[] = (string) $row['Department'];
        }
        $res->close();
    }

    if ($res = $conn->query("SELECT Location_of_Work FROM ic_location ORDER BY Location_of_Work ASC")) {
        while ($row = $res->fetch_assoc()) {
            $locationOptions[] = (string) $row['Location_of_Work'];
        }
        $res->close();
    }
}

$ratingOptions = ['A', 'B+', 'B', 'C', 'D'];

$hrTaValue   = (string) ($pif['hr_ta'] ?? '');
$hrHeadValue = (string) ($pif['hr_head_dgm'] ?? 'Sonali Heera');

if ($hrTaValue === '' && !empty($_SESSION['namevfied1'])) {
    $hrTaValue = (string) $_SESSION['namevfied1'];
}

/* Short helper for safe output */
function e($str)
{
    return htmlspecialchars((string) $str, ENT_QUOTES, 'UTF-8');
}

?>
<?php ob_start(); ?>

<style>
    /* -------------------- Local Styles (Soziety-like) -------------------- */
    .pif-summary-label {
        font-size: 13px;
        font-weight: 600;
        color: #6b7280;
        margin-bottom: 4px;
    }

    .pif-summary-value {
        background-color: #f3f4f6;
        border-radius: 6px;
        padding: 8px 10px;
        font-size: 14px;
        color: #111827;
    }

    .section-title-sm {
        font-size: 15px;
        font-weight: 600;
        margin-bottom: 10px;
    }

    .badge-status {
        padding: 0.25rem 0.6rem;
        border-radius: 999px;
        font-size: 11px;
    }

    .badge-status.approve {
        background: rgba(34, 197, 94, 0.08);
        color: #16a34a;
    }

    .badge-status.reject {
        background: rgba(239, 68, 68, 0.08);
        color: #dc2626;
    }

    .badge-status.hold {
        background: rgba(234, 179, 8, 0.08);
        color: #a16207;
    }

.reviewer-block {
        border-radius: 10px;
        border: 1px dashed #e5e7eb;
        padding: 12px 12px 6px;
        margin-bottom: 10px;
        background: #fafafa;
    }

    .section-card {
        background: #f8fafc;
        border: 1px solid #e5e7eb;
        box-shadow: inset 0 1px 0 rgba(255, 255, 255, 0.7);
    }

    .section-card.section-even {
        background: #eef2ff;
    }

    .section-card.section-odd {
        background: #f8fafc;
    }

    .reviewer-block-title {
        font-size: 13px;
        font-weight: 600;
        color: #4b5563;
    }

    .btn-add-reviewer {
        animation: blink-btn 1.2s linear infinite;
    }

    @keyframes blink-btn {
        0%, 100% {
            box-shadow: 0 0 0 0 rgba(79, 70, 229, 0.5);
        }
        50% {
            box-shadow: 0 0 0 6px rgba(79, 70, 229, 0);
        }
    }

    .reviewer-link-input {
        font-size: 11px;
    }

    .reviewer-link-help {
        font-size: 11px;
        color: #6b7280;
        margin-top: 4px;
    }
</style>

<?php
$styles = ob_get_clean();
?>

<?php ob_start(); ?>

<!-- Page Header -->
<div class="page-header-breadcrumb mb-3">
    <div class="d-flex align-center justify-content-between flex-wrap">
        <h1 class="page-title fw-medium fs-18 mb-0">PIF Evaluation</h1>
        <ol class="breadcrumb mb-0">
            <li class="breadcrumb-item">
                <a href="javascript:void(0);">Dashboards</a>
            </li>
            <li class="breadcrumb-item">
                <a href="javascript:void(0);">HR</a>
            </li>
            <li class="breadcrumb-item active" aria-current="page">
                Stage-2 PIF Proceed
            </li>
        </ol>
    </div>
</div>

<!-- Alerts -->
<?php if ($successMsg): ?>
    <div class="alert alert-success" role="alert">
        <?php echo e($successMsg); ?>
    </div>
<?php endif; ?>

<?php if ($errorMsg): ?>
    <div class="alert alert-danger" role="alert">
        <?php echo e($errorMsg); ?>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-xl-12">
        <div class="card custom-card">
            <div class="card-header justify-content-between">
                <div>
                    <div class="card-title">
                        <?php echo e($pif['username']); ?> : For Interview
                    </div>
                    <div class="text-muted fs-12">
                        PIF ID:
                        <?php echo (int) $pif['idpif']; ?>
                        &nbsp;|&nbsp;
                        Contact:
                        <?php echo e($pif['phone']); ?>
                        &nbsp;|&nbsp;
                        Email:
                        <?php echo e($pif['email']); ?>
                    </div>
                </div>
                <div class="text-end">
                    <?php if ($resumeUrl): ?>
                        <a href="<?php echo e($resumeUrl); ?>" target="_blank"
                           class="btn btn-outline-primary btn-sm">
                            <i class="ri-file-3-line me-1"></i>
                            Click to View Resume
                        </a>
                    <?php endif; ?>
                </div>
            </div>

            <div class="card-body">

                <!-- Applicant Details: Pre - Interview Form (PIF) -->
                <div class="mb-4">
                    <div class="section-title-sm mb-2">
                        Applicant Details : Pre - Interview Form (PIF)
                    </div>

                    <div class="row g-3">
                        <div class="col-md-3">
                            <div class="pif-summary-label">Applicant Name</div>
                            <div class="pif-summary-value"><?php echo e($pif['username']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Contact Number</div>
                            <div class="pif-summary-value"><?php echo e($pif['phone']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Email</div>
                            <div class="pif-summary-value"><?php echo e($pif['email']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Gender</div>
                            <div class="pif-summary-value"><?php echo e($pif['gender']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Marital Status</div>
                            <div class="pif-summary-value"><?php echo e($pif['marital_status']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Date of Birth</div>
                            <div class="pif-summary-value"><?php echo e($pif['dob']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Current Address</div>
                            <div class="pif-summary-value"><?php echo e($pif['currentAddress']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Current City</div>
                            <div class="pif-summary-value"><?php echo e($pif['currentCity']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Conveyance</div>
                            <div class="pif-summary-value"><?php echo e($pif['conveyance']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">If Yes</div>
                            <div class="pif-summary-value"><?php echo e($pif['convyes']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Position Applied For</div>
                            <div class="pif-summary-value"><?php echo e($pif['appliedfor']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Current Salary (Per Month)</div>
                            <div class="pif-summary-value"><?php echo e($pif['current_salary']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Expected Salary (Per Month)</div>
                            <div class="pif-summary-value"><?php echo e($pif['expected_salary']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Notice Period (In Days)</div>
                            <div class="pif-summary-value"><?php echo e($pif['notice_period']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Qualification</div>
                            <div class="pif-summary-value"><?php echo e($pif['qualification']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Current / Last Org Name</div>
                            <div class="pif-summary-value"><?php echo e($pif['lastorg']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Current / Last Designation</div>
                            <div class="pif-summary-value"><?php echo e($pif['last_designation']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Job Type</div>
                            <div class="pif-summary-value"><?php echo e($pif['jobType']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Total Work Experience</div>
                            <div class="pif-summary-value"><?php echo e($pif['totalExp']); ?></div>
                        </div>
                        <div class="col-md-6">
                            <div class="pif-summary-label">Reason for leaving</div>
                            <div class="pif-summary-value"><?php echo e($pif['reason']); ?></div>
                        </div>
                        <div class="col-md-3">
                            <div class="pif-summary-label">Resume</div>
                            <div class="pif-summary-value">
                                <?php if ($resumeUrl): ?>
                                    <a class="btn btn-outline-primary btn-sm" href="<?php echo e($resumeUrl); ?>" target="_blank">
                                        Click to View
                                    </a>
                                <?php else: ?>
                                    Not Uploaded
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Evaluation Form -->
                <form method="post" action="" id="pifEvalForm" autocomplete="off">

                    <!-- To Be Appointed As -->
                    <div class="border rounded-3 p-3 mb-4 section-card">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <div class="section-title-sm mb-0">
                                To Be Appointed As (For Official Use Only)
                            </div>

                            <?php if (!empty($pif['st1_status'])): ?>
                                <span class="badge-status
                                    <?php
                                    echo strtolower($pif['st1_status']) === 'approve'
                                        ? 'approve'
                                        : (strtolower($pif['st1_status']) === 'reject'
                                            ? 'reject'
                                            : 'hold');
                                    ?>">
                                    Current: <?php echo e($pif['st1_status']); ?>
                                </span>
                            <?php endif; ?>
                        </div>

                        <div class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">To Be Appointed As*</label>
                                <input
                                    type="text"
                                    name="appointed_as"
                                    class="form-control"
                                    value="<?php echo e($pif['appointed_as'] ?? ''); ?>"
                                    required
                                >
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Fresh / Rehire*</label>
                                <select name="fresh_hire" class="form-select" required>
                                    <option value="">-Select-</option>
                                    <?php
                                    $freshOpts = ['Fresh', 'Rehire'];
                                    $currFresh = (string) ($pif['fresh_hire'] ?? '');
                                    foreach ($freshOpts as $opt) {
                                        $sel = (strcasecmp($currFresh, $opt) === 0) ? 'selected' : '';
                                        echo '<option value="' . e($opt) . '" ' . $sel . '>' . e($opt) . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">AD / Director EmpId*</label>
                                <input
                                    class="form-control"
                                    type="text"
                                    name="ad_director_id"
                                    value="<?php echo e($pif['ad_director_id'] ?? ''); ?>"
                                    placeholder="Emp. ID"
                                    required
                                >
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">AD / Director Name*</label>
                                <input
                                    class="form-control"
                                    type="text"
                                    name="ad_director_name"
                                    value="<?php echo e($pif['ad_director_name'] ?? ''); ?>"
                                    placeholder="Name"
                                    required
                                >
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Team Name</label>
                                <select class="form-select" name="team_name">
                                    <option value="">- Select -</option>
                                    <?php
                                    $selectedTeam = (string) ($pif['team_name'] ?? '');
                                    foreach ($teamOptions as $team) {
                                        $sel = (strcasecmp($selectedTeam, $team) === 0) ? 'selected' : '';
                                        echo '<option value="' . e($team) . '" ' . $sel . '>' . e($team) . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Department</label>
                                <select class="form-select" name="Department">
                                    <option value="">- Select -</option>
                                    <?php
                                    $selectedDept = (string) ($pif['Department'] ?? '');
                                    foreach ($departmentOptions as $dept) {
                                        $sel = (strcasecmp($selectedDept, $dept) === 0) ? 'selected' : '';
                                        echo '<option value="' . e($dept) . '" ' . $sel . '>' . e($dept) . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Location of Work</label>
                                <select class="form-select" name="Location_of_Work">
                                    <option value="">- Select -</option>
                                    <?php
                                    $selectedLoc = (string) ($pif['Location_of_Work'] ?? '');
                                    foreach ($locationOptions as $loc) {
                                        $sel = (strcasecmp($selectedLoc, $loc) === 0) ? 'selected' : '';
                                        echo '<option value="' . e($loc) . '" ' . $sel . '>' . e($loc) . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Designation</label>
                                <input
                                    class="form-control"
                                    type="text"
                                    name="designation"
                                    value="<?php echo e($pif['designation'] ?? ''); ?>"
                                >
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">CTC per month</label>
                                <input
                                    class="form-control"
                                    type="text"
                                    name="ctc_per_month"
                                    value="<?php echo e($pif['ctc_per_month'] ?? ''); ?>"
                                >
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Expected DOJ</label>
                                <input
                                    class="form-control"
                                    type="date"
                                    name="expected_doj"
                                    value="<?php echo e($pif['expected_doj'] ?? ''); ?>"
                                >
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Actual DOJ</label>
                                <input
                                    class="form-control"
                                    type="date"
                                    name="actual_doj"
                                    value="<?php echo e($pif['actual_doj'] ?? ''); ?>"
                                >
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Immediate Reporting Manager Name</label>
                                <input
                                    class="form-control"
                                    type="text"
                                    name="reporting_manager"
                                    value="<?php echo e($pif['reporting_manager'] ?? ''); ?>"
                                    placeholder="Manager Name"
                                >
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">RM Code</label>
                                <input
                                    class="form-control"
                                    type="text"
                                    name="rm_code"
                                    value="<?php echo e($pif['rm_code'] ?? ''); ?>"
                                    placeholder="Code"
                                >
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Human Resource (TA)</label>
                                <input type="hidden" name="hr_ta" value="<?php echo e($hrTaValue); ?>">
                                <input class="form-control" value="<?php echo e($hrTaValue); ?>" disabled>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">HEAD / DGM HR</label>
                                <input type="hidden" name="hr_head_dgm" value="<?php echo e($hrHeadValue); ?>">
                                <input class="form-control" value="<?php echo e($hrHeadValue); ?>" disabled>
                            </div>
                        </div>
                    </div>

                    <!-- Evaluation Criteria: Reviewer 1 (HR) -->
                    <div class="border rounded-3 p-3 mb-4 section-card">
                        <div class="section-title-sm mb-2 text-center">
                            Evaluation Criteria: Reviewer 1 (HR)
                        </div>
                        <p class="text-center text-muted fs-12">Rating Scale : Excellent = A, Very Good = B+, Good = B, Average = C, Poor = D</p>

                        <div class="row g-3">
                            <?php
                            $ratingsMap = [
                                'r1_qual'       => 'Qualifications',
                                'r1_knowledge'  => 'Industry Knowledge',
                                'r1_experience' => 'Relevant Experience',
                                'r1_record'     => 'Past Track Record',
                                'r1_initiative' => 'Initiative / Drive',
                                'r1_tplayer'    => 'Team Player',
                                'r1_leadership' => 'Leadership',
                                'r1_stability'  => 'Job Stability',
                                'r1_comskill'   => 'Communication Skills',
                                'r1_ratings'    => 'Final Ratings',
                            ];
                            foreach ($ratingsMap as $field => $label):
                                $currVal = (string) ($pif[$field] ?? '');
                            ?>
                                <div class="col-md-3">
                                    <label class="form-label mb-1"><?php echo e($label); ?></label><br>
                                    <?php foreach ($ratingOptions as $opt): ?>
                                        <label class="me-2 mb-0">
                                            <input
                                                type="radio"
                                                name="<?php echo e($field); ?>"
                                                value="<?php echo e($opt); ?>"
                                                <?php echo (strcasecmp($currVal, $opt) === 0) ? 'checked' : ''; ?>
                                            >
                                            <?php echo e($opt); ?>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>

                        <div class="row g-3 mt-2">
                            <div class="col-md-3">
                                <label class="form-label">Reviewer 1 (HR) Decision*</label>
                                <select name="r1_decision" class="form-select" required>
                                    <option value="">-- Select --</option>
                                    <?php
                                    $optR1     = ['Approve', 'Reject', 'Hold'];
                                    $currentR1 = (string) $pif['st1_status'];

                                    foreach ($optR1 as $opt) {
                                        $sel = (strcasecmp($currentR1, $opt) === 0)
                                            ? 'selected'
                                            : '';
                                        echo '<option value="' . e($opt) . '" ' . $sel . '>'
                                            . e($opt)
                                            . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>

                            <div class="col-md-3">
                                <label class="form-label">Reviewer 1 (HR) Remarks</label>
                                <input
                                    type="text"
                                    name="r1_remarks"
                                    class="form-control"
                                    value="<?php echo e($pif['st1_feedback']); ?>"
                                >
                            </div>

                            <div class="col-md-3">
                                <label class="form-label">Interview Date*</label>
                                <input
                                    type="date"
                                    name="interview_date"
                                    class="form-control"
                                    value="<?php echo e($pif['interview_date'] ?? ''); ?>"
                                    required
                                >
                            </div>

                            <div class="col-12">
                                <label class="form-label">
                                    Evaluation Criteria / Notes (Reviewer 1 - HR)
                                </label>
                                <textarea
                                    name="eval_r1_hr"
                                    class="form-control"
                                    rows="2"
                                    placeholder="Key points used while evaluating this applicant..."
                                ><?php echo e($pif['eval_r1_hr']); ?></textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Reviewer Details (2,3,...) -->                    <!-- Reviewer Details (2,3,...) -->
                    <div class="border rounded-3 p-3 mb-4 section-card">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <div class="section-title-sm mb-0">
                                Reviewer Details (Add More)
                            </div>
                            <button
                                type="button"
                                class="btn btn-primary btn-sm btn-add-reviewer"
                                id="btnAddReviewer"
                            >
                                <i class="ri-user-add-line me-1"></i>
                                Add More Reviewer
                            </button>
                        </div>

                        <p class="text-muted fs-12 mb-3">
                            Reviewer numbering starts from 2. Enter Reviewer Emp ID;
                            Name &amp; Team will auto-fill from
                            <code>hr_dump (HR DB)</code>. Decision &amp; Remarks will be
                            filled by the reviewer on the Interview page (link valid for 3 days).
                        </p>

                        <div id="reviewerContainer">
                            <?php foreach ($existingReviewers as $idx => $rev): ?>
                                <?php $indexHuman = $idx + 2; // Reviewer 2,3,... ?>
                                <div class="reviewer-block" data-index="<?php echo (int) $idx; ?>">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span class="reviewer-block-title">
                                            Reviewer <?php echo $indexHuman; ?>
                                        </span>
                                        <button
                                            type="button"
                                            class="btn btn-outline-danger btn-sm btn-remove-reviewer d-none"
                                        >
                                            <i class="ri-close-line"></i>
                                        </button>
                                    </div>

                                    <div class="row g-2 align-items-center">
                                        <div class="col-md-2">
                                            <label class="form-label mb-1">Reviewer Emp ID</label>
                                            <input
                                                type="text"
                                                name="reviewers[<?php echo (int) $idx; ?>][empid]"
                                                class="form-control reviewer-empid"
                                                value="<?php echo e($rev['empid'] ?? ''); ?>"
                                                placeholder="e.g. 20831"
                                            >
                                        </div>

                                        <div class="col-md-3">
                                            <label class="form-label mb-1">Reviewer Name</label>
                                            <input
                                                type="text"
                                                name="reviewers[<?php echo (int) $idx; ?>][name]"
                                                class="form-control reviewer-name"
                                                value="<?php echo e($rev['name'] ?? ''); ?>"
                                                readonly
                                            >
                                        </div>

                                        <div class="col-md-3">
                                            <label class="form-label mb-1">Team Name</label>
                                            <input
                                                type="text"
                                                name="reviewers[<?php echo (int) $idx; ?>][team]"
                                                class="form-control reviewer-team"
                                                value="<?php echo e($rev['team'] ?? ''); ?>"
                                            >
                                        </div>

                                        <div class="col-md-4">
                                            <label class="form-label mb-1">Interview Link (valid for 3 days)</label>
                                            <div class="input-group input-group-sm">
                                                <button
                                                    type="button"
                                                    class="btn btn-outline-primary btn-create-link"
                                                >
                                                    Create Interview Link
                                                </button>
                                                <input
                                                    type="text"
                                                    class="form-control reviewer-link reviewer-link-input"
                                                    placeholder="Link will appear here"
                                                    readonly
                                                >
                                            </div>
                                            <div class="reviewer-link-help">
                                                After entering Emp ID, click on "Create Interview Link".
                                                A new tab will open with the Reviewer page that can be
                                                shared with the respective Reviewer.
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Module / Final Decision -->
                    <div class="border rounded-3 p-3 mb-3 section-card">
                        <div class="section-title-sm mb-2">
                            Module / Final Decision*
                        </div>

                        <div class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">Module*</label>
                                <select name="module_final" class="form-select" required>
                                    <option value="">--Select--</option>
                                    <?php
                                    $moduleOptions = ['On Roll', 'Off Roll'];
                                    $currModule    = (string) ($pif['module_final'] ?? $pif['module_name']);
                                    foreach ($moduleOptions as $opt) {
                                        $sel = (strcasecmp($currModule, $opt) === 0) ? 'selected' : '';
                                        echo '<option value="' . e($opt) . '" ' . $sel . '>' . e($opt) . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Module Type*</label>
                                <select name="module_type" class="form-select" required>
                                    <option value="">--Select--</option>
                                    <?php
                                    $moduleTypeOptions = ['Salaried', 'ER', 'OPC', 'Retainer', 'ER-PM', 'Intern'];
                                    $currType          = (string) ($pif['module_type'] ?? '');
                                    foreach ($moduleTypeOptions as $opt) {
                                        $sel = (strcasecmp($currType, $opt) === 0) ? 'selected' : '';
                                        echo '<option value="' . e($opt) . '" ' . $sel . '>' . e($opt) . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Final Decision*</label>
                                <select name="final_decision" class="form-select" required>
                                    <option value="<?php echo e($pif['st1_status']); ?>"><?php echo e($pif['st1_status']); ?></option>
                                    <option value="">--Select--</option>
                                    <?php
                                    $finalOpts = ['Selected', 'Hold', 'Rejected', 'Denied By Candidate'];
                                    $currFinal = (string) $pif['final_decision'];

                                    foreach ($finalOpts as $opt) {
                                        $sel = (strcasecmp($currFinal, $opt) === 0) ? 'selected' : '';
                                        echo '<option value="' . e($opt) . '" ' . $sel . '>'
                                            . e($opt)
                                            . '</option>';
                                    }
                                    ?>
                                </select>
                            </div>

                            <div class="col-md-3">
                                <label class="form-label">Final Decision Remarks</label>
                                <textarea
                                    name="final_decision_remark"
                                    class="form-control"
                                    rows="2"
                                    placeholder="Summary of overall evaluation, offer conditions, etc."
                                ><?php echo e($pif['final_decision_remark']); ?></textarea>
                            </div>
                        </div>
                    </div>

                    <div class="d-flex justify-content-end">
                        <button type="submit" class="btn btn-primary">
                            <i class="ri-save-3-line me-1"></i>
                            Save Evaluation
                        </button>
                    </div>

                </form>
            </div>
        </div>
    </div>
</div>

<?php
$content = ob_get_clean();
?>

<?php ob_start(); ?>

<script>
// ---------------------- Reviewer Dynamic JS -------------------------
(function () {
    const container        = document.getElementById('reviewerContainer');
    const btnAdd           = document.getElementById('btnAddReviewer');
    const baseUrl          = <?php echo json_encode($baseUrl, JSON_UNESCAPED_SLASHES); ?>;
    const reviewerPageBase = <?php echo json_encode($baseUrl . '/pages/pif-reviewer-interview.php', JSON_UNESCAPED_SLASHES); ?>;
    const pifId            = <?php echo (int) $idpif; ?>;
    const evalForm         = document.getElementById('pifEvalForm');

    // AJAX endpoint for Emp Info (same file)
    const ajaxEmpUrl       = <?php echo json_encode($_SERVER['PHP_SELF'], JSON_UNESCAPED_SLASHES); ?>;

    // HR mandatory fields for link creation
    const appointedInput   = document.querySelector('input[name="appointed_as"]');
    const evalTextarea     = document.querySelector('textarea[name="eval_r1_hr"]');

    function applySectionStriping() {
        const cards = document.querySelectorAll('.section-card');
        cards.forEach(function (card, idx) {
            card.classList.remove('section-odd', 'section-even');
            card.classList.add(idx % 2 === 0 ? 'section-odd' : 'section-even');
        });
    }

    applySectionStriping();

    if (!container || !btnAdd) return;

    async function saveHrSections() {
        if (!evalForm) return true;

        const formData = new FormData(evalForm);
        formData.append('ajax', 'save_hr_sections');

        try {
            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData,
            });

            const data = await response.json();

            if (data && data.success) {
                return true;
            }

            alert((data && data.message) ? data.message : 'Unable to save HR details.');
            return false;
        } catch (err) {
            console.error('HR save error', err);
            alert('Network error while saving HR details. Please try again.');
            return false;
        }
    }

    function refreshRemoveButtons() {
        const blocks = container.querySelectorAll('.reviewer-block');
        blocks.forEach(function (block) {
            const btnRemove = block.querySelector('.btn-remove-reviewer');
            if (!btnRemove) return;

            if (blocks.length > 1) {
                btnRemove.classList.remove('d-none');
            } else {
                btnRemove.classList.add('d-none');
            }
        });
    }

    // ---- EmpID → Name & Team auto-fill via AJAX ----
    function bindEmpLookup(block) {
        const empInput  = block.querySelector('.reviewer-empid');
        const nameInput = block.querySelector('.reviewer-name');
        const teamInput = block.querySelector('.reviewer-team');

        if (!empInput || !nameInput) return;

        let timer = null;

        empInput.addEventListener('keyup', function () {
            const empid = empInput.value.trim();
            clearTimeout(timer);

            if (!empid) {
                nameInput.value = '';
                if (teamInput) teamInput.value = '';
                return;
            }

            timer = setTimeout(function () {
                const url = ajaxEmpUrl + '?ajax=empinfo&empid=' + encodeURIComponent(empid);
                console.log('EmpInfo URL:', url);

                fetch(url)
                    .then(function (r) { return r.json(); })
                    .then(function (data) {
                        nameInput.value = data.name || '';
                        if (teamInput) teamInput.value = data.team || '';
                    })
                    .catch(function (err) {
                        console.error('EmpInfo AJAX error:', err);
                    });
            }, 300);
        });
    }

    // ---- Create Interview Link button ----
    function bindCreateLink(block) {
        const btnCreate = block.querySelector('.btn-create-link');
        const empInput  = block.querySelector('.reviewer-empid');
        const linkInput = block.querySelector('.reviewer-link');

        if (!btnCreate || !empInput) return;

        btnCreate.addEventListener('click', async function () {
            // 1) Check HR mandatory fields first
            if (!appointedInput || !evalTextarea) {
                alert('Configuration error: HR fields not found on page.');
                return;
            }

            const appointedVal = appointedInput.value.trim();
            const evalVal      = evalTextarea.value.trim();

            if (!appointedVal || !evalVal) {
                alert('Please fill "To Be Appointed As" and "Evaluation Criteria / Notes (Reviewer 1 - HR)" before creating interview link.');
                if (!appointedVal) {
                    appointedInput.focus();
                } else {
                evalTextarea.focus();
            }
            return;
        }

            const confirmSave = window.confirm(
                'Save HR details before creating interview link? This ensures the reviewer sees your latest inputs.'
            );

            if (!confirmSave) {
                return;
            }

            const saved = await saveHrSections();

            if (!saved) {
                return;
            }

            // 2) Then check reviewer EmpID
            const empid = empInput.value.trim();
            if (!empid) {
                alert('Please enter Reviewer Emp ID first.');
                empInput.focus();
                return;
            }

            const link = reviewerPageBase + '?id=' + encodeURIComponent(pifId) +
                '&empid=' + encodeURIComponent(empid) +
                '&appointed_as=' + encodeURIComponent(appointedVal) +
                '&eval_r1_hr=' + encodeURIComponent(evalVal);

            if (linkInput) {
                linkInput.value = link;
            }

            alert('HR details saved. Opening interview link in a new tab.');

            // Open the reviewer page in a new tab
            window.open(link, '_blank');
        });
    }

    function initBlock(block) {
        bindEmpLookup(block);
        bindCreateLink(block);
    }

    function reindexBlocks() {
        const blocks = container.querySelectorAll('.reviewer-block');

        blocks.forEach(function (block, idx) {
            const indexHuman = idx + 2;
            block.dataset.index = idx;

            const titleEl = block.querySelector('.reviewer-block-title');
            if (titleEl) {
                titleEl.textContent = 'Reviewer ' + indexHuman;
            }

            const inputs = block.querySelectorAll('input[name], select[name], textarea[name]');
            inputs.forEach(function (input) {
                const name = input.getAttribute('name');
                if (!name) return;
                const newName = name.replace(/reviewers\[\d+\]/, 'reviewers[' + idx + ']');
                input.setAttribute('name', newName);
            });
        });
    }

    // Init: bind for existing reviewer blocks
    container.querySelectorAll('.reviewer-block').forEach(function (block) {
        initBlock(block);
    });
    refreshRemoveButtons();

    // ---- Add More Reviewer ----
    btnAdd.addEventListener('click', function () {
        const blocks    = container.querySelectorAll('.reviewer-block');
        const nextIndex = blocks.length;

        const template = document.createElement('div');
        template.className   = 'reviewer-block';
        template.dataset.index = String(nextIndex);

        template.innerHTML = `
            <div class="d-flex justify-content-between align-items-center mb-2">
                <span class="reviewer-block-title">Reviewer ${nextIndex + 2}</span>
                <button type="button" class="btn btn-outline-danger btn-sm btn-remove-reviewer">
                    <i class="ri-close-line"></i>
                </button>
            </div>
            <div class="row g-2 align-items-center">
                <div class="col-md-2">
                    <label class="form-label mb-1">Reviewer Emp ID</label>
                    <input
                        type="text"
                        name="reviewers[${nextIndex}][empid]"
                        class="form-control reviewer-empid"
                        placeholder="e.g. 20831"
                    >
                </div>
                <div class="col-md-3">
                    <label class="form-label mb-1">Reviewer Name</label>
                    <input
                        type="text"
                        name="reviewers[${nextIndex}][name]"
                        class="form-control reviewer-name"
                        readonly
                    >
                </div>
                <div class="col-md-3">
                    <label class="form-label mb-1">Team Name</label>
                    <input
                        type="text"
                        name="reviewers[${nextIndex}][team]"
                        class="form-control reviewer-team"
                    >
                </div>
                <div class="col-md-4">
                    <label class="form-label mb-1">Interview Link (valid for 3 days)</label>
                    <div class="input-group input-group-sm">
                        <button
                            type="button"
                            class="btn btn-outline-primary btn-create-link"
                        >
                            Create Interview Link
                        </button>
                        <input
                            type="text"
                            class="form-control reviewer-link reviewer-link-input"
                            placeholder="Link will appear here"
                            readonly
                        >
                    </div>
                    <div class="reviewer-link-help">
                        After entering Emp ID, click on "Create Interview Link".
                        A new tab will open with the Reviewer page that can be shared.
                    </div>
                </div>
            </div>
        `;

        container.appendChild(template);
        initBlock(template);
        refreshRemoveButtons();
        reindexBlocks();
    });

    // ---- Remove Reviewer ----
    container.addEventListener('click', function (e) {
        const btn = e.target.closest('.btn-remove-reviewer');
        if (!btn) return;

        const block = btn.closest('.reviewer-block');
        if (!block) return;

        block.remove();
        reindexBlocks();
        refreshRemoveButtons();
    });
})();
</script>

<?php
$scripts = ob_get_clean();

include __DIR__ . '/layouts/base.php';