<?php
/* =============================================================================
 * File   : user-roles.php
 * Purpose: Role Management + Permissions (DB + Hardcoded hybrid)
 * Notes  :
 *   - Loads permissions from role_permission_master (is_active=1)
 *   - Also includes original hardcoded permissions as fallback
 *   - New DB entries show along with old ones
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

include('../config.php');
require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

/* =============================================================================
 * AJAX: Save role permissions
 * ============================================================================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($_POST['ajax'])) {
    $roleId      = isset($_POST['role_id']) ? (int)$_POST['role_id'] : 0;
    $permissions = isset($_POST['permissions']) ? $_POST['permissions'] : [];
    if (!is_array($permissions)) {
        $permissions = [];
    }
    $permissionsJson = json_encode(array_values($permissions)); // ensure numeric array

    $stmt = $conn_user->prepare("UPDATE roles SET permissions = ? WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("si", $permissionsJson, $roleId);
        if ($stmt->execute()) {
            echo json_encode(['status' => 'success']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Update failed']);
        }
        $stmt->close();
    } else {
        echo json_encode(['status' => 'error', 'message' => $conn_user->error]);
    }
    exit;
}

/* =============================================================================
 * PERMISSION MASTER: DB + HARDCODED MERGE
 * ============================================================================= */

/* 1) Base from hardcoded list (original options) */
$allPerms = [];

$hardcoded = [
    'Calling From' => [
        'applied_calling_form_applications.php'             => 'Calling From',
    ],
    'Onboarding' => [
        'onb-dashboard.php'             => 'Dashboard',
        'onb-submitted-application.php' => 'Submitted',
        'onb-approved-application.php'  => 'Approved',
        'onb-inprogress-application.php'=> 'In-Progress',
        'onb-rejected-application.php'  => 'Rejected',
        'onb-without-applications.php'  => 'Without Applications',
    ],
    'AMS' => [
        'ams-dasboard.php'      => 'Dashboard',
        'ams-employee-add.php'  => 'Employee Add',
        'ams-employee-list.php' => 'Employee List',
        'ams-tracker.php'       => 'Office In/Out (Tracker)',
        'ams-final-cm.php'      => 'Report Final',
        'ams-report-ap-cm.php'  => 'Current Month A/P Format',
        'ams-report-ap-lm.php'  => 'Last Month A/P Format',
        'ams-leaves.php'        => 'Leaves',
        'ams-locations.php'     => 'Locations',
    ],
    'PIF Applications' => [
        'pif-submitted.php'          => 'PIF Submitted',
        'pif-approved.php'           => 'PIF Approved',
        'pif-rejected.php'           => 'PIF Rejected',
    ],
    'DST' => [
        'dst-dashboard.php'          => 'Dashboard',
        'dst-deals.php'              => 'DST Opportunities',
        'dst-leads.php'              => 'DST Leads',
        'dst-leads-upload.php'       => 'DST Leads Upload',
        'dst-activities-calendar.php'=> 'Activities Calendar',
        'dst-activities-calls.php'   => 'Activities Calls',
        'dst-activities-meetings.php'=> 'Meetings',
        'dst-activities-tasks.php'   => 'Tasks',
    ],
    'Broker Suite' => [
        'br-suite-dashboard.php'          => 'Dashboard',
        'br-suite-deals.php'              => 'Broker Opportunities',
        'br-suite-leads.php'              => 'Broker Leads',
        'br-suite-activities-calendar.php'=> 'Activities Calendar',
        'br-suite-activities-calls.php'   => 'Activities Calls',
        'br-suite-activities-meetings.php'=> 'Meetings',
        'br-suite-activities-tasks.php'   => 'Tasks',
    ],
    'Broker Onboarding' => [
        'broker-dashboard.php'   => 'Dashboard',
        'broker-list.php'        => 'List',
        'broker-projects.php'    => 'Projects',
        'broker-units.php'       => 'Units',
        'broker-onboarding.php'  => 'Broker Onboarding',
        'cart.php'               => 'Cart',
        'checkout.php'           => 'Checkout',
        'customers-list.php'     => 'Customers',
        'orders.php'             => 'Orders',
        'orders-details.php'     => 'Order Details',
        'add-product.php'        => 'Add Product',
    ],
    'Projects' => [
        'projects-dashboard.php' => 'Dashboard',
        'projects-list.php'      => 'Projects List',
        'projects-overview.php'  => 'Project Overview',
        'project_form.php'       => 'Create Project',
    ],
    'Sales' => [
        'index.php' => 'Sales',
    ],
    'Analytics' => [
        'index1.php' => 'Analytics',
    ],
    'Courses' => [
        'index9.php' => 'Courses',
    ],
    'Other Activities' => [
        'calendar.php'          => 'Calendar',
        'activities-calls.php'  => 'Calls',
        'activities-meetings.php'=> 'Meetings',
        'activities-tasks.php'  => 'Tasks',
    ],
    'User Tables' => [
        'user-roles.php'  => 'User Roles',
        'user-list.php'   => 'User List',
        'user-create.php' => 'Create User',
        'grid-tables.php' => 'Grid JS Tables',
        'data-tables.php' => 'Data Tables',
    ],
];

/* Copy hardcoded into $allPerms first */
foreach ($hardcoded as $section => $pages) {
    if (!isset($allPerms[$section])) {
        $allPerms[$section] = [];
    }
    foreach ($pages as $pg => $lbl) {
        $allPerms[$section][$pg] = $lbl;
    }
}

/* 2) Overlay with DB entries from role_permission_master
 *    - If same Main-Title + page_name exists => DB label overrides
 *    - If new Main-Title or new page_name => appended
 */
$sqlPerms = "SELECT main_title, page_name, label
             FROM role_permission_master
             WHERE is_active = 1
             ORDER BY main_title, sort_order, label";

$resPerms = $conn_user->query($sqlPerms);

if ($resPerms) {
    while ($p = $resPerms->fetch_assoc()) {
        $mt  = trim($p['main_title']);
        $pg  = trim($p['page_name']);
        $lbl = trim($p['label']);

        if ($mt === '' || $pg === '' || $lbl === '') {
            continue;
        }

        if (!isset($allPerms[$mt])) {
            $allPerms[$mt] = [];
        }

        // DB wins over hardcoded for same page
        $allPerms[$mt][$pg] = $lbl;
    }
}

/* Optional: sort sections alphabetically by title */
ksort($allPerms);

/* =============================================================================
 * VIEW: ROLE LIST + MODAL
 * ============================================================================= */

ob_start();
?>
<link rel="stylesheet" href="<?= $baseUrl; ?>/assets/libs/prismjs/themes/prism-coy.min.css">
<style>
    .badge-custom { margin-right: 0.25rem; }
</style>
<?php
$styles = ob_get_clean();

ob_start();
?>
<div class="container py-5">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="mb-0">Role Management</h1>
        <a href="user_role_create.php" class="btn btn-primary">
            <i class="fa fa-plus"></i> Add New Role
        </a>
    </div>
    <div class="card mb-4">
        <div class="card-body p-0">
            <table class="table table-hover table-bordered mb-0">
                <thead class="table-light">
                    <tr>
                        <th style="width:60px;">#</th>
                        <th>Role</th>
                        <th>Permissions</th>
                        <th style="width:120px;">Edit</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $res = $conn_user->query("SELECT id, role_name, permissions FROM roles");
                    $counter = 1;
                    while ($row = $res->fetch_assoc()):
                        if ((int)$row['id'] === 1) {
                            // skip super admin role if required
                            continue;
                        }
                        $perms = [];
                        if (!empty($row['permissions'])) {
                            $tmp = json_decode($row['permissions'], true);
                            if (is_array($tmp)) {
                                $perms = $tmp;
                            }
                        }
                    ?>
                        <tr>
                            <td><?= $counter++; ?></td>
                            <td><?= htmlspecialchars($row['role_name']); ?></td>
                            <td>
                                <?php if (!empty($perms)): ?>
                                    <?php foreach ($perms as $p): ?>
                                        <span class="badge bg-info badge-custom">
                                            <?php
                                            $label = '';
                                            foreach ($allPerms as $section => $pages) {
                                                if (isset($pages[$p])) {
                                                    $label = $pages[$p];
                                                    break;
                                                }
                                            }
                                            echo htmlspecialchars($label !== '' ? $label : ucfirst($p));
                                            ?>
                                        </span>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <span class="text-muted">None</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <button
                                    class="btn btn-sm btn-outline-primary edit-role-btn"
                                    type="button"
                                    data-role-id="<?= (int)$row['id']; ?>"
                                    data-role-name="<?= htmlspecialchars($row['role_name'], ENT_QUOTES); ?>"
                                    data-permissions='<?= json_encode(array_values($perms)); ?>'
                                >
                                    Edit
                                </button>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal: Edit Role Permissions -->
<div class="modal fade" id="editRoleModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <form id="rolePermissionForm" class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Permissions - <span id="modalRoleName"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" name="role_id" id="modalRoleId">
                <label class="form-label">Permissions</label>

                <?php foreach ($allPerms as $section => $permList): ?>
                    <h6 class="mt-3 mb-2"><?= htmlspecialchars($section); ?> Section</h6>
                    <?php foreach ($permList as $key => $label): ?>
                        <div class="form-check">
                            <input
                                class="form-check-input perm-checkbox"
                                type="checkbox"
                                name="permissions[]"
                                value="<?= htmlspecialchars($key); ?>"
                                id="perm<?= md5($section . '|' . $key); ?>"
                            >
                            <label class="form-check-label" for="perm<?= md5($section . '|' . $key); ?>">
                                <?= htmlspecialchars($label); ?>
                                <span class="text-muted small">(<?= htmlspecialchars($key); ?>)</span>
                            </label>
                        </div>
                    <?php endforeach; ?>
                <?php endforeach; ?>

            </div>
            <div class="modal-footer">
                <button type="submit" class="btn btn-success">Save</button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            </div>
        </form>
    </div>
</div>
<?php
$content = ob_get_clean();

ob_start();
?>
<script>
document.addEventListener("DOMContentLoaded", () => {
    const modalEl  = document.getElementById('editRoleModal');
    const modal    = new bootstrap.Modal(modalEl);
    const formEl   = document.getElementById('rolePermissionForm');
    const roleNameSpan = document.getElementById('modalRoleName');
    const roleIdInput  = document.getElementById('modalRoleId');

    // Open modal + pre-select permissions
    document.querySelectorAll('.edit-role-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            let perms = [];
            try {
                const raw = btn.getAttribute('data-permissions') || '[]';
                const parsed = JSON.parse(raw);
                if (Array.isArray(parsed)) {
                    perms = parsed;
                }
            } catch (e) {
                console.error('Invalid permissions JSON:', e);
            }

            roleNameSpan.textContent = btn.getAttribute('data-role-name') || '';
            roleIdInput.value        = btn.getAttribute('data-role-id') || '';

            document.querySelectorAll('.perm-checkbox').forEach(cb => {
                cb.checked = perms.includes(cb.value);
            });

            modal.show();
        });
    });

    // Save
    formEl.addEventListener('submit', e => {
        e.preventDefault();
        const data = new FormData(formEl);

        fetch('', {
            method: 'POST',
            body: data
        })
        .then(res => res.json())
        .then(res => {
            if (res.status === 'success') {
                modal.hide();
                location.reload();
            } else {
                alert(res.message || 'Update failed');
            }
        })
        .catch(err => {
            console.error(err);
            alert('Error saving permissions');
        });
    });
});
</script>
<?php
$scripts = ob_get_clean();

include 'layouts/base.php';
?>
