<?php
/* =============================================================================
 * File   : zeptomail_helper.php
 * Purpose: Common helper to send transactional emails via ZeptoMail
 *          using templates stored in `email_templates` table.
 * Usage  : require_once 'zeptomail_helper.php';
 *          $res = zepto_send_template($toEmail, $toName, 'welcome_email', $placeholders);
 * Notes  :
 *   - Uses DB connection from config.php (MySQLi, no PDO).
 *   - Make sure ZeptoMail constants are defined in config.php:
 *        ZEPTO_API_URL   (optional, default used if missing)
 *        ZEPTO_API_KEY   (Send Mail Token, mandatory)
 *        ZEPTO_FROM_EMAIL
 *        ZEPTO_FROM_NAME
 *   - Database must use utf8mb4_general_ci collation.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

date_default_timezone_set('Asia/Kolkata');

/* ------------------------- Centralized config include ------------------------ */
$config_paths = array(
    __DIR__ . '/config.php',
    __DIR__ . '/../config.php',
    __DIR__ . '/../../config.php'
);

$config_found = false;
foreach ($config_paths as $cp) {
    if (is_file($cp)) {
        require_once $cp;
        $config_found = true;
        break;
    }
}

if (!$config_found) {
    die('config.php not found for ZeptoMail helper.');
}

/* ------------------------------ DB connection -------------------------------- */
if (!isset($conn) || !($conn instanceof mysqli)) {
    die('MySQLi connection $conn not found. Please ensure config.php sets $conn.');
}

mysqli_set_charset($conn, 'utf8mb4');
mysqli_query($conn, "SET collation_connection='utf8mb4_general_ci'");

/* --------------------------- ZeptoMail constants ----------------------------- */
/* You can define these in config.php. If missing, defaults below are used.    */
if (!defined('ZEPTO_API_URL')) {
    define('ZEPTO_API_URL', 'https://api.zeptomail.in/v1.1/email');
}

if (!defined('ZEPTO_API_KEY')) {
    // IMPORTANT: Replace in config.php with your real Send Mail Token
    define('ZEPTO_API_KEY', 'YOUR_ZEPTO_SEND_MAIL_TOKEN_HERE');
}

if (!defined('ZEPTO_FROM_EMAIL')) {
    define('ZEPTO_FROM_EMAIL', 'noreply@example.com');
}

if (!defined('ZEPTO_FROM_NAME')) {
    define('ZEPTO_FROM_NAME', 'Your App Name');
}

/**
 * zepto_send_template
 *
 * @param string $toEmail        Recipient email address
 * @param string $toName         Recipient name
 * @param string $templateKey    email_templates.template_key
 * @param array  $placeholders   Key-value pairs like ['{{NAME}}' => 'Satya']
 * @param array  $extraCc        Extra CC emails (array of strings)
 * @param array  $extraBcc       Extra BCC emails (array of strings)
 * @param array  $attachments    Array of attachments:
 *                               [
 *                                 ['file_cache_key' => 'xxx'],
 *                                 ['content' => 'BASE64', 'mime_type' => 'text/plain', 'name' => 'readme.txt']
 *                               ]
 * @param string $clientReference Optional client reference string
 *
 * @return array [
 *   'success'       => bool,
 *   'http_code'     => int|null,
 *   'response_raw'  => string|null,
 *   'response_json' => array|null,
 *   'error'         => string|null
 * ]
 */
function zepto_send_template(
    $toEmail,
    $toName,
    $templateKey,
    $placeholders = array(),
    $extraCc = array(),
    $extraBcc = array(),
    $attachments = array(),
    $clientReference = ''
) {
    global $conn;

    $result = array(
        'success'       => false,
        'http_code'     => null,
        'response_raw'  => null,
        'response_json' => null,
        'error'         => null
    );

    /* --------------------------- Basic validations -------------------------- */
    $toEmail = trim($toEmail);
    $toName  = trim($toName);
    $templateKey = trim($templateKey);

    if (!filter_var($toEmail, FILTER_VALIDATE_EMAIL)) {
        $result['error'] = 'Invalid recipient email address.';
        return $result;
    }

    if ($templateKey === '') {
        $result['error'] = 'Template key is required.';
        return $result;
    }

    /* ------------------------ Fetch template from DB ------------------------ */
    $sql = "SELECT id, template_name, template_key, subject, html_body, text_body,
                   default_cc, default_bcc, track_clicks, track_opens
            FROM email_templates
            WHERE template_key = ? AND is_active = 1
            LIMIT 1";

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        $result['error'] = 'Failed to prepare template query: ' . $conn->error;
        return $result;
    }

    $stmt->bind_param('s', $templateKey);
    $stmt->execute();
    $res = $stmt->get_result();
    if (!$res || $res->num_rows === 0) {
        $result['error'] = 'Template not found or inactive for key: ' . $templateKey;
        $stmt->close();
        return $result;
    }

    $tpl = $res->fetch_assoc();
    $stmt->close();

    $subject  = $tpl['subject'];
    $htmlBody = $tpl['html_body'];
    $textBody = $tpl['text_body'];

    /* --------------------------- Placeholder merge -------------------------- */
    if (!empty($placeholders) && is_array($placeholders)) {
        foreach ($placeholders as $search => $replace) {
            $subject  = str_replace($search, $replace, $subject);
            $htmlBody = str_replace($search, $replace, $htmlBody);
            if (!is_null($textBody)) {
                $textBody = str_replace($search, $replace, $textBody);
            }
        }
    }

    /* --------------------------- Recipients setup --------------------------- */
    $to = array(
        array(
            'email_address' => array(
                'address' => $toEmail,
                'name'    => $toName
            )
        )
    );

    $cc  = array();
    $bcc = array();

    // Default CC from template (comma-separated)
    if (!empty($tpl['default_cc'])) {
        $parts = explode(',', $tpl['default_cc']);
        foreach ($parts as $addr) {
            $addr = trim($addr);
            if ($addr !== '' && filter_var($addr, FILTER_VALIDATE_EMAIL)) {
                $cc[] = array(
                    'email_address' => array(
                        'address' => $addr,
                        'name'    => ''
                    )
                );
            }
        }
    }

    // Default BCC from template (comma-separated)
    if (!empty($tpl['default_bcc'])) {
        $parts = explode(',', $tpl['default_bcc']);
        foreach ($parts as $addr) {
            $addr = trim($addr);
            if ($addr !== '' && filter_var($addr, FILTER_VALIDATE_EMAIL)) {
                $bcc[] = array(
                    'email_address' => array(
                        'address' => $addr,
                        'name'    => ''
                    )
                );
            }
        }
    }

    // Extra CC passed from function
    if (!empty($extraCc) && is_array($extraCc)) {
        foreach ($extraCc as $addr) {
            $addr = trim($addr);
            if ($addr !== '' && filter_var($addr, FILTER_VALIDATE_EMAIL)) {
                $cc[] = array(
                    'email_address' => array(
                        'address' => $addr,
                        'name'    => ''
                    )
                );
            }
        }
    }

    // Extra BCC passed from function
    if (!empty($extraBcc) && is_array($extraBcc)) {
        foreach ($extraBcc as $addr) {
            $addr = trim($addr);
            if ($addr !== '' && filter_var($addr, FILTER_VALIDATE_EMAIL)) {
                $bcc[] = array(
                    'email_address' => array(
                        'address' => $addr,
                        'name'    => ''
                    )
                );
            }
        }
    }

    /* ---------------------------- Payload build ----------------------------- */
    $payload = array(
        'from' => array(
            'address' => ZEPTO_FROM_EMAIL,
            'name'    => ZEPTO_FROM_NAME
        ),
        'to'     => $to,
        'subject'=> $subject,
        'track_clicks' => (bool)$tpl['track_clicks'],
        'track_opens'  => (bool)$tpl['track_opens']
    );

    if (!empty($htmlBody)) {
        $payload['htmlbody'] = $htmlBody;
    } elseif (!empty($textBody)) {
        $payload['textbody'] = $textBody;
    }

    if (!empty($cc)) {
        $payload['cc'] = $cc;
    }
    if (!empty($bcc)) {
        $payload['bcc'] = $bcc;
    }
    if (!empty($clientReference)) {
        $payload['client_reference'] = $clientReference;
    }

    // Attachments (file_cache_key OR base64 content)
    if (!empty($attachments) && is_array($attachments)) {
        $payload['attachments'] = array();
        foreach ($attachments as $att) {
            if (!is_array($att)) {
                continue;
            }
            $attFiltered = array();

            if (!empty($att['file_cache_key'])) {
                $attFiltered['file_cache_key'] = $att['file_cache_key'];
            }

            if (!empty($att['content']) && !empty($att['mime_type']) && !empty($att['name'])) {
                $attFiltered['content']   = $att['content'];   // Already base64 encoded
                $attFiltered['mime_type'] = $att['mime_type'];
                $attFiltered['name']      = $att['name'];
            }

            if (!empty($attFiltered)) {
                $payload['attachments'][] = $attFiltered;
            }
        }

        if (empty($payload['attachments'])) {
            unset($payload['attachments']);
        }
    }

    /* ------------------------------ cURL request ---------------------------- */
    $ch = curl_init();
    curl_setopt_array($ch, array(
        CURLOPT_URL            => ZEPTO_API_URL,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING       => '',
        CURLOPT_MAXREDIRS      => 10,
        CURLOPT_TIMEOUT        => 30,
        CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST  => 'POST',
        CURLOPT_HTTPHEADER     => array(
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization: Zoho-enczapikey ' . ZEPTO_API_KEY
        ),
        CURLOPT_POSTFIELDS     => json_encode($payload)
    ));

    $response = curl_exec($ch);
    $err      = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $result['http_code']    = $httpCode;
    $result['response_raw'] = $response;

    if ($err) {
        $result['error'] = 'cURL error: ' . $err;
        return $result;
    }

    $json = json_decode($response, true);
    $result['response_json'] = $json;

    if ($httpCode >= 200 && $httpCode < 300 && isset($json['data'])) {
        $result['success'] = true;
    } else {
        if (isset($json['error']['message'])) {
            $result['error'] = $json['error']['message'];
        } elseif (isset($json['message'])) {
            $result['error'] = $json['message'];
        } else {
            $result['error'] = 'Unknown error from ZeptoMail.';
        }
    }

    return $result;
}
?>
