<?php
session_start();
include('../config.php');
require_once 'includes/check-permission.php';

$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

// Helper function for escaping output
function h($str) {
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

// --------------------
// GET Parameters
// --------------------
$search = trim($_GET['search'] ?? '');

// Handle per_page safely
$allowed_per_page = [10, 25, 50, 500];
$per_page_input = $_GET['per_page'] ?? 10;
$per_page = in_array((int)$per_page_input, $allowed_per_page) ? (int)$per_page_input : 10;
$per_page = max(1, $per_page); // Prevent division by zero

// Pagination
$page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $per_page;

// Sorting
$sort = $_GET['sort'] ?? 'SNo_desc';
$orderClause = match($sort) {
    'SNo_asc' => 'ORDER BY SNo ASC',
    'SNo_desc' => 'ORDER BY SNo DESC',
    'created_asc' => 'ORDER BY COALESCE(updated_at, created_at) ASC',
    'created_desc' => 'ORDER BY COALESCE(updated_at, created_at) DESC',
    'status' => 'ORDER BY Status ASC',
    'name' => 'ORDER BY Employee_Full_Name ASC',
    default => 'ORDER BY SNo DESC',
};

// --------------------
// WHERE clause for search
// --------------------
$whereClause = '';
$searchParams = [];
$types = '';
if (!empty($search)) {
    $whereClause = "WHERE empid LIKE ? OR Employee_Full_Name LIKE ? OR Status LIKE ?";
    $searchParams = ["%$search%", "%$search%", "%$search%"];
    $types = str_repeat('s', count($searchParams));
}

// --------------------
// Count total rows
// --------------------
$countSql = "SELECT COUNT(*) FROM hr_dump $whereClause";
$countStmt = mysqli_prepare($conn_ic, $countSql);
if (!empty($searchParams)) {
    mysqli_stmt_bind_param($countStmt, $types, ...$searchParams);
}
mysqli_stmt_execute($countStmt);
$countResult = mysqli_stmt_get_result($countStmt);
$totalRows = mysqli_fetch_row($countResult)[0] ?? 0;
$totalPages = ceil($totalRows / $per_page);

// --------------------
// Fetch paginated data
// --------------------
$dataSql = "SELECT SNo, empid, Status, Employee_Full_Name, Member_Type, Date_of_Joining, Team_Name, Role, Department, Functional_Reporting, Functional_Reporting_Name, created_at
            FROM hr_dump
            $whereClause
            $orderClause
            LIMIT ?, ?";
$dataStmt = mysqli_prepare($conn_ic, $dataSql);

// Bind params dynamically
if (!empty($searchParams)) {
    $types .= 'ii'; // Add integer types for offset & limit
    mysqli_stmt_bind_param($dataStmt, $types, ...[...$searchParams, $offset, $per_page]);
} else {
    mysqli_stmt_bind_param($dataStmt, 'ii', $offset, $per_page);
}

mysqli_stmt_execute($dataStmt);
$dataResult = mysqli_stmt_get_result($dataStmt);
?>



<?php ob_start(); ?>
<?php $styles = ob_get_clean(); ?>
<?php ob_start(); ?>
<!-- Page Header -->
<div class="page-header-breadcrumb mb-3">
    <div class="d-flex align-center justify-content-between flex-wrap">
        <h1 class="page-title fw-medium fs-18 mb-0">Employee List</h1>
        <ol class="breadcrumb mb-0">
            <li class="breadcrumb-item"><a href="#">Pages</a></li>
            <li class="breadcrumb-item active">Employee List</li>
        </ol>
    </div>
</div>

<!-- Filters -->
<form method="GET" class="row g-2 align-items-end mb-3">
    <div class="col-md-3">
        <label class="form-label">Search</label>
        <input type="text" name="search" class="form-control" placeholder="Search any keyword..." value="<?= h($search ?? '') ?>">
    </div>
    <div class="col-md-3">
        <label class="form-label">Sort By</label>
        <select name="sort" class="form-select">
            <option value="SNo_desc" <?= ($sort ?? '') === 'SNo_desc' ? 'selected' : '' ?>>SNo (Latest First)</option>
            <option value="SNo_asc" <?= ($sort ?? '') === 'SNo_asc' ? 'selected' : '' ?>>SNo (Oldest First)</option>
            <option value="created_desc" <?= ($sort ?? '') === 'created_desc' ? 'selected' : '' ?>>Created (Newest)</option>
            <option value="created_asc" <?= ($sort ?? '') === 'created_asc' ? 'selected' : '' ?>>Created (Oldest)</option>
            <option value="status" <?= ($sort ?? '') === 'status' ? 'selected' : '' ?>>Status</option>
            <option value="name" <?= ($sort ?? '') === 'name' ? 'selected' : '' ?>>Employee Name</option>
        </select>
    </div>
    <div class="col-md-2">
        <label class="form-label">Rows</label>
        <select name="per_page" class="form-select">
            <?php foreach ([10, 25, 50, 500] as $opt): ?>
                <option value="<?= $opt ?>" <?= ($per_page ?? 10) == $opt ? 'selected' : '' ?>><?= $opt ?></option>
            <?php endforeach; ?>
        </select>
    </div>
    <div class="col-md-auto">
        <label class="form-label d-block">&nbsp;</label>
        <div class="d-flex gap-2">
            <button type="submit" class="btn btn-primary">Apply</button>
            <a href="ams-employee-list.php" class="btn btn-outline-secondary">Reset</a>
        </div>
    </div>
</form>

<div class="card custom-card">
    <div class="card-body table-responsive">
        <table class="table table-bordered table-hover text-nowrap">
            <thead>
                <tr>
                    <th>SNo</th>
                    <th>Emp ID</th>
                    <th>Status</th>
                    <th>Employee Name</th>
                    <th>Member Type</th>
                    <th>Joining Date</th>
                    <th>Team</th>
                    <th>Role</th>
                    <th>Department</th>
                    <th>Functional Reporting</th>
                    <th>Reporting Name</th>
                    <th>Created At</th>
                    <th>Edit</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($dataResult && mysqli_num_rows($dataResult) > 0): ?>
                    <?php while ($row = mysqli_fetch_assoc($dataResult)): ?>
                        <tr>
                            <td><?= h($row['SNo']) ?></td>
                            <td><?= h($row['empid']) ?></td>
                            <td><?= h($row['Status']) ?></td>
                            <td><?= h($row['Employee_Full_Name']) ?></td>
                            <td><?= h($row['Member_Type']) ?></td>
                            <td><?= !empty($row['Date_of_Joining']) ? date('d M Y', strtotime($row['Date_of_Joining'])) : 'NA' ?></td>
                            <td><?= h($row['Team_Name']) ?></td>
                            <td><?= h($row['Role']) ?></td>
                            <td><?= h($row['Department']) ?></td>
                            <td><?= h($row['Functional_Reporting']) ?></td>
                            <td><?= h($row['Functional_Reporting_Name']) ?></td>
                            <td><?= !empty($row['created_at']) ? date('d M Y H:i', strtotime($row['created_at'])) : 'NA' ?></td>
							<td><a href="ams-employee-edit.php?SNo=<?= htmlspecialchars(urlencode($row['SNo'])) ?>" class="btn btn-warning btn-sm">Edit</a></td>
                        </tr>
                    <?php endwhile; ?>
                <?php else: ?>
                    <tr><td colspan="12" class="text-center text-muted">No records found.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>
<!-- Pagination -->
<?php if ($totalPages > 1): ?>
<nav class="mt-3">
    <ul class="pagination justify-content-end">
        <?php if ($page > 1): ?>
            <li class="page-item">
                <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">Previous</a>
            </li>
        <?php endif; ?>

        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
            <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>"><?= $i ?></a>
            </li>
        <?php endfor; ?>

        <?php if ($page < $totalPages): ?>
            <li class="page-item">
                <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">Next</a>
            </li>
        <?php endif; ?>
    </ul>
</nav>
<?php endif; ?>
<?php $content = ob_get_clean(); ?>
<?php ob_start(); ?>
<?php $scripts = ob_get_clean(); ?>
<?php include 'layouts/base.php'; ?>