<?php
/**
 * ams-report-ap-cm.php
 *
 * - Uses config.php for DB (MySQLi; no PDO). Ensure config.php defines at least one mysqli link
 *   such as $conn_ic, $conn, $conn_crm, or $conn_crmplus.
 * - Auto-detects current calendar month and renders pivot table with headers like "01 Oct".
 * - Aggressive inline CSS + JS fallback to ensure month label is visible despite theme overrides.
 *
 * Full file from 1st line to end line.
 */

session_start();
date_default_timezone_set('Asia/Kolkata');

require_once __DIR__ . '/config.php';
require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

/* pick mysqli connection */
$DB = null;
if (isset($conn_ic) && $conn_ic instanceof mysqli)              { $DB = $conn_ic; }
elseif (isset($conn) && $conn instanceof mysqli)                { $DB = $conn; }
elseif (isset($conn_crm) && $conn_crm instanceof mysqli)        { $DB = $conn_crm; }
elseif (isset($conn_crmplus) && $conn_crmplus instanceof mysqli){ $DB = $conn_crmplus; }

if (!($DB instanceof mysqli)) {
    http_response_code(500);
    echo "Database connection (mysqli) not found. Please ensure config.php sets a mysqli connection.";
    exit;
}

/* ensure proper charset / collation */
@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");

/* ------------------------- Determine date range (current calendar month) ------------------------- */
$start_date = date('Y-m-01'); // first day of current month
$end_date   = date('Y-m-t');  // last day of current month

// Optional override via URL params (only if both from & to are provided and valid YYYY-MM-DD)
if (!empty($_GET['from']) && !empty($_GET['to'])) {
    $from = trim($_GET['from']);
    $to   = trim($_GET['to']);
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $from) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $to)) {
        $start_date = $from;
        $end_date = $to;
    }
}

/* ------------------------- Build list of dates between start and end ------------------------- */
$dates = [];
$headers = [];   // human-friendly header labels (e.g. "01 Oct")
$aliases = [];   // column aliases used in SQL & fetch (e.g. d_2025_10_01)

$period_start = strtotime($start_date);
$period_end   = strtotime($end_date);
for ($d = $period_start; $d <= $period_end; $d += 86400) {
    $ymd = date('Y-m-d', $d);
    $dates[] = $ymd;
    $headers[] = date('d M', $d);
    $aliases[] = 'd_' . str_replace('-', '_', $ymd);
}

/* ------------------------- Build dynamic SELECT (pivot) ------------------------- */
$selectParts = [];
$selectParts[] = "empid";
foreach ($dates as $idx => $dt) {
    $escaped = $DB->real_escape_string($dt);
    $alias = $aliases[$idx];
    $selectParts[] = "IFNULL(MAX(CASE WHEN OfficeDate = '{$escaped}' THEN FinalStatus END), 'A') AS `{$alias}`";
}
$selectSQL = implode(",\n       ", $selectParts);

$sql = "SELECT
       {$selectSQL}
FROM ams_final
WHERE OfficeDate BETWEEN '" . $DB->real_escape_string($start_date) . "' AND '" . $DB->real_escape_string($end_date) . "'
GROUP BY empid
ORDER BY empid ASC";

/* Execute query */
$res = $DB->query($sql);
if ($res === false) {
    echo "<pre>Query error: " . htmlspecialchars($DB->error, ENT_QUOTES, 'UTF-8') . "\n\n$sql</pre>";
    exit;
}

/* ------------------------- Output page ------------------------- */
?>
<?php ob_start(); ?>

<!-- Aggressive inline CSS to defeat theme overrides that hid the month part -->
<style>
/* Force headers to allow wrap and visible overflow */
.pivot-table thead th {
    white-space: normal !important;
    overflow: visible !important;
    text-overflow: clip !important;
    height: auto !important;
    max-height: none !important;
    min-width: 56px !important;     /* give a little width so month can fit on second line */
    padding: 8px !important;
    text-align: center !important;
    vertical-align: middle !important;
    display: table-cell !important; /* override flex/table mixing rules */
    line-height: 1.05 !important;
    word-break: normal !important;
}

/* Ensure header content is not hidden by ::after or child overflow */
.pivot-table thead th * {
    overflow: visible !important;
    display: inline-block !important;
}

/* smaller font for month if wrapped (optional) */
.pivot-table thead th .month {
    display: inline-block;
    font-size: 11px;
    color: #6c757d;
    margin-left: 4px;
}

/* Pivot body cells */
.pivot-table tbody td {
    text-align: center;
    vertical-align: middle;
    padding: 10px 6px;
    font-size: 13px;
    white-space: nowrap;
}

/* Responsive wrapper */
.pivot-wrapper {
    overflow-x: auto;
}

/* If container has forced overflow hidden at ancestor, try to allow visible (best-effort) */
.pivot-wrapper, .pivot-wrapper * {
    overflow-x: visible !important;
    overflow-y: visible !important;
}
</style>

<?php $styles = ob_get_clean(); ?>
<?php ob_start(); ?>

	<!-- Start::page-header -->
	<div class="page-header-breadcrumb mb-3">
		<div class="d-flex align-center justify-content-between flex-wrap">
			<h1 class="page-title fw-medium fs-18 mb-0">AMS Pivot — <?php echo htmlspecialchars(date('F Y', strtotime($start_date)), ENT_QUOTES, 'UTF-8'); ?></h1>
			<ol class="breadcrumb mb-0">
				<li class="breadcrumb-item"><a href="javascript:void(0);">Pages</a></li>
				<li class="breadcrumb-item active" aria-current="page">AMS Final Report</li>
			</ol>
		</div>
	</div>
	<!-- End::page-header -->

	<!-- Controls -->
	<div class="mb-2 d-flex align-items-center gap-2">
		<span class="badge bg-light text-dark">Range: <?php echo htmlspecialchars($start_date . ' → ' . $end_date, ENT_QUOTES, 'UTF-8'); ?></span>
		<a class="btn btn-sm btn-outline-secondary" href="ams-final-cm-export.php?from=<?php echo urlencode($start_date); ?>&to=<?php echo urlencode($end_date); ?>">Export Data</a>
	</div>

	<!-- Start::row-1 -->
	<div class="row">
		<div class="col-12">
			<div class="card custom-card">
				<div class="card-body">
					<div class="pivot-wrapper">
						<table id="hrTable" class="table table-striped table-bordered table-sm pivot-table">
							<thead>
							<tr>
								<th>S.N.</th>
								<th>Emp Id</th>
								<?php foreach ($dates as $i => $dt):
								       // header as single string "01 Oct" and also provide data attribute for JS fallback
								       $fmt = date('d M', strtotime($dt));
								?>
									<th data-dt="<?php echo htmlspecialchars($dt, ENT_QUOTES, 'UTF-8'); ?>" data-dt-format="<?php echo htmlspecialchars($fmt, ENT_QUOTES, 'UTF-8'); ?>">
										<?php echo htmlspecialchars($fmt, ENT_QUOTES, 'UTF-8'); ?>
									</th>
								<?php endforeach; ?>
							</tr>
							</thead>
							<tbody>
							<?php
							if ($res->num_rows === 0) {
							    echo '<tr><td colspan="' . (2 + count($dates)) . '" class="text-center">No records found for the selected range.</td></tr>';
							} else {
							    $sn = 1;
							    while ($row = $res->fetch_assoc()) {
							        echo '<tr>';
							        echo '<td>' . $sn++ . '</td>';
							        echo '<td>' . htmlspecialchars($row['empid'], ENT_QUOTES, 'UTF-8') . '</td>';
							        foreach ($aliases as $alias) {
							            $val = isset($row[$alias]) ? $row[$alias] : 'A';
							            echo '<td>' . htmlspecialchars($val, ENT_QUOTES, 'UTF-8') . '</td>';
							        }
							        echo '</tr>';
							    }
							}
							// free result
							$res->free();
							?>
							</tbody>
						</table>
					</div> <!-- .pivot-wrapper -->
				</div>
			</div>
		</div>
	</div>
	<!--End::row-1 -->
<?php $content = ob_get_clean(); ?>
<?php ob_start(); ?>
<!-- JS fallback: ensure header text is exactly the formatted date (overrides theme mutations) -->
<script>
(function(){
    try {
        var ths = document.querySelectorAll('.pivot-table thead th[data-dt-format]');
        ths.forEach(function(th){
            var txt = th.getAttribute('data-dt-format') || th.textContent.trim();
            // set plain text (no child spans) so CSS can't mask month via child rules
            th.textContent = txt;
            // also set title for hover clarity
            var dt = th.getAttribute('data-dt');
            if (dt) th.setAttribute('title', dt);
        });
    } catch(e) {
        // silent
        console && console.log && console.log('pivot header fallback error', e);
    }
})();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
