<?php
session_start();
include('../config.php');
require_once 'includes/check-permission.php';

$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

// Helpers
function h($string) {
    return htmlspecialchars($string ?? 'NA', ENT_QUOTES, 'UTF-8');
}

// Handle pagination, search, and sorting
$per_page = isset($_GET['per_page']) ? (int)$_GET['per_page'] : 10;
$per_page = in_array($per_page, [10, 25, 50, 500]) ? $per_page : 10;

$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$sort = $_GET['sort'] ?? 'created_desc';

$offset = ($page - 1) * $per_page;

// Sorting logic
$sort = $_GET['sort'] ?? 'id_desc';

$orderClause = match($sort) {
    'id_asc' => 'ORDER BY id ASC',
    'id_desc' => 'ORDER BY id DESC',
    'created_asc' => 'ORDER BY COALESCE(updated_at, created_at) ASC',
    'created_desc' => 'ORDER BY COALESCE(updated_at, created_at) DESC',
    'status' => 'ORDER BY status ASC',
    'name' => 'ORDER BY Employee_Full_Name ASC',
    default => 'ORDER BY id DESC',
};


// Build WHERE clause
$whereClause = '';
$params = [];
$paramTypes = '';

if (!empty($search)) {
    $whereClause = "WHERE Employee_Full_Name LIKE ? OR empid LIKE ?";
    $params[] = '%' . $search . '%';
    $params[] = '%' . $search . '%';
    $paramTypes .= 'ss';
}

// Count total records
$countSql = "SELECT COUNT(*) as total FROM tracker $whereClause";
$countStmt = mysqli_prepare($conn_ic, $countSql);
if (!empty($whereClause)) {
    mysqli_stmt_bind_param($countStmt, $paramTypes, ...$params);
}
mysqli_stmt_execute($countStmt);
$countResult = mysqli_stmt_get_result($countStmt);
$totalRows = mysqli_fetch_assoc($countResult)['total'] ?? 0;
$totalPages = ceil($totalRows / $per_page);

// Main data query
$sql = "SELECT * FROM tracker $whereClause $orderClause LIMIT ?, ?";
$params[] = $offset;
$params[] = $per_page;
$paramTypes .= 'ii';

$stmt = mysqli_prepare($conn_ic, $sql);
mysqli_stmt_bind_param($stmt, $paramTypes, ...$params);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
?>

<!-- Layout include -->
<?php ob_start(); ?>
<!-- Optional styles -->
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
<!-- Page Header -->
<div class="page-header-breadcrumb mb-3">
    <div class="d-flex align-center justify-content-between flex-wrap">
        <h1 class="page-title fw-medium fs-18 mb-0">AMS Tracker</h1>
        <ol class="breadcrumb mb-0">
            <li class="breadcrumb-item"><a href="#">Pages</a></li>
            <li class="breadcrumb-item active">AMS Tracker</li>
        </ol>
    </div>
</div>

<!-- Search, sort, pagination controls -->
<form method="GET" class="row g-2 align-items-center mb-3">
    <div class="col-md-3">
        <label class="form-label">Search</label>
		<input type="text" name="search" class="form-control" placeholder="Search by name or ID..." value="<?= h($search) ?>">
    </div>

    <div class="col-md-2">
        <label class="form-label">Sort By</label>
		<select name="sort" class="form-select">
			<option value="id_desc" <?= $sort === 'id_desc' ? 'selected' : '' ?>>ID (Latest First)</option>
			<option value="id_asc" <?= $sort === 'id_asc' ? 'selected' : '' ?>>ID (Oldest First)</option>
			<option value="created_desc" <?= $sort === 'created_desc' ? 'selected' : '' ?>>Created (Newest)</option>
			<option value="created_asc" <?= $sort === 'created_asc' ? 'selected' : '' ?>>Created (Oldest)</option>
			<option value="status" <?= $sort === 'status' ? 'selected' : '' ?>>Status</option>
			<option value="name" <?= $sort === 'name' ? 'selected' : '' ?>>Employee Name</option>
		</select>

    </div>

    <div class="col-md-2">
        <label class="form-label">Rows</label>
		<select name="per_page" class="form-select">
            <?php foreach ([10, 25, 50, 500] as $opt): ?>
                <option value="<?= $opt ?>" <?= $opt == $per_page ? 'selected' : '' ?>><?= $opt ?> rows</option>
            <?php endforeach; ?>
        </select>
    </div>

    <div class="col-md-auto">
        <label class="form-label d-block">&nbsp;</label>
		<button type="submit" class="btn btn-primary">Apply</button>
        <a href="ams-tracker.php" class="btn btn-outline-secondary">Reset</a>
    </div>
</form>




<!-- Table -->
<div class="card custom-card">
    <div class="col-xl-12">
        <a href="ams-tracker-export.php" class="btn btn-outline-secondary">Export Tracker Data</a>
    </div>
    <div class="card-body table-responsive">
        <table class="table table-bordered table-hover">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Emp ID</th>
                    <th>Full Name</th>
                    <th>Team</th>
                    <th>Fn. Reporting</th>
                    <th>In Location</th>
                    <th>Out Location</th>
                    <th>Status</th>
                    <th>Office Date</th>
                    <th>In Time</th>
                    <th>Late</th>
                    <th>Early</th>
                    <th>Out Time</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($result && mysqli_num_rows($result) > 0): ?>
                    <?php while ($row = mysqli_fetch_assoc($result)): ?>
                        <tr>
                            <td><?= h($row['id']) ?></td>
                            <td><?= h($row['empid']) ?></td>
                            <td><?= h($row['Employee_Full_Name']) ?></td>
                            <td><?= h($row['Team_Name']) ?></td>
                            <td><?= h($row['Functional_Reporting']) ?></td>
                            <td><?= h($row['location']) ?></td>
                            <td><?= h($row['out_location']) ?></td>
                            <td><?= h($row['status']) ?></td>
                            <td><?= !empty($row['OfficeDate']) ? date('d M Y', strtotime($row['OfficeDate'])) : 'NA' ?></td>
                            <td><?= h($row['OfficeIn']) ?></td>
                            <td><?= h($row['OfficeLate']) ?></td>
                            <td><?= h($row['OfficeEarly']) ?></td>
                            <td><?= h($row['OfficeOut']) ?></td>
                        </tr>
                    <?php endwhile; ?>
                <?php else: ?>
                    <tr><td colspan="13" class="text-muted text-center">No records found.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Pagination -->
<nav class="mt-3">
    <ul class="pagination justify-content-end">
        <?php if ($page > 1): ?>
            <li class="page-item">
                <a class="page-link" href="?<?= http_build_query(['search' => $search, 'sort' => $sort, 'per_page' => $per_page, 'page' => $page - 1]) ?>">Previous</a>
            </li>
        <?php endif; ?>

        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
            <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                <a class="page-link" href="?<?= http_build_query(['search' => $search, 'sort' => $sort, 'per_page' => $per_page, 'page' => $i]) ?>"><?= $i ?></a>
            </li>
        <?php endfor; ?>

        <?php if ($page < $totalPages): ?>
            <li class="page-item">
                <a class="page-link" href="?<?= http_build_query(['search' => $search, 'sort' => $sort, 'per_page' => $per_page, 'page' => $page + 1]) ?>">Next</a>
            </li>
        <?php endif; ?>
    </ul>
</nav>
<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<!-- Optional scripts -->
<?php $scripts = ob_get_clean(); ?>

<?php include 'layouts/base.php'; ?>
