<?php
// File: broker_list.php
/* =============================================================================
 * File: broker_list.php
 * Purpose: Admin view for broker applications (styled/structured like broker_onboarding.php / onb-dashboard)
 * Stack : Core PHP + MySQLi (NO PDO). DB via ../config.php. Collation: utf8mb4_general_ci
 * Notes : - Full file from first to last line as requested
 *         - Uses layout slots ($styles, $content, $scripts) and includes 'layouts/base.php' at end
 *         - Expects ../config.php to provide $conn and optionally $conn_sa, $baseUrl
 * ============================================================================= */

@session_start();
ini_set('display_errors',1); ini_set('display_startup_errors',1); error_reporting(E_ALL);
date_default_timezone_set('Asia/Kolkata');

/* ------------------- Required includes & permission check ------------------- */
require_once __DIR__ . '/../config.php';            // must provide $conn (mysqli) and optionally $baseUrl

require_once __DIR__ . '/broker-config.php';
require_once __DIR__ . '/functions.php';

$currentPage = basename(__FILE__);
if (function_exists('checkPageAccessOrRedirect')) {
    checkPageAccessOrRedirect($currentPage, $conn_sa ?? null);
}

/* -------------------- DB charset / collation enforcement ------------------- */
if (!isset($conn) || !($conn instanceof mysqli)) {
    http_response_code(500);
    exit('Database connection ($conn) missing. Please ensure ../config.php defines $conn as mysqli instance.');
}
@mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* -------------------------- Small helpers --------------------------------- */
if (!function_exists('esc')) { function esc($s){ return htmlspecialchars((string)$s,ENT_QUOTES,'UTF-8'); } }
if (!function_exists('h'))   { function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); } }
if (!function_exists('v'))   { function v($k,$d=''){ return isset($_POST[$k]) ? trim((string)$_POST[$k]) : $d; } }
function badge_class_from_status($s){
    $s = strtolower(trim((string)$s));
    if ($s === 'approved' || $s === 'active') return 'bg-primary-transparent text-primary';
    if ($s === 'rejected' || $s === 'blocked' || $s === 'declined') return 'bg-danger-transparent text-danger';
    if ($s === 'pending' || $s === '' || $s === null) return 'bg-warning-transparent text-warning';
    return 'bg-secondary-transparent text-secondary';
}

/* -------------------------- Messages / Input ------------------------------- */
$msg = $err = '';
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

/* ---------------- Approve / Reject action --------------------------------- */
if (($action === 'approve' || $action === 'reject') && $id > 0) {
    $new = ($action === 'approve') ? 'Approved' : 'Rejected';

    $stmt = @mysqli_prepare($conn, "UPDATE brokers SET status = ? WHERE id = ?");
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, 'si', $new, $id);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    } else {
        $escNew = mysqli_real_escape_string($conn, $new);
        mysqli_query($conn, "UPDATE brokers SET status = '{$escNew}' WHERE id = " . (int)$id);
    }

    /* If approved, try to activate corresponding user by broker_code */
    if ($action === 'approve') {
        $b = null;
        $res = mysqli_query($conn, "SELECT broker_code FROM brokers WHERE id = ".(int)$id." LIMIT 1");
        if ($res) $b = mysqli_fetch_assoc($res);
        if ($b && !empty($b['broker_code'])) {
            $uname = mysqli_real_escape_string($conn, $b['broker_code']);
            $stmt2 = @mysqli_prepare($conn, "UPDATE users SET status = 'Active' WHERE username = ?");
            if ($stmt2) {
                mysqli_stmt_bind_param($stmt2, 's', $uname);
                mysqli_stmt_execute($stmt2);
                mysqli_stmt_close($stmt2);
            } else {
                mysqli_query($conn, "UPDATE users SET status='Active' WHERE username='{$uname}'");
            }
        }
    }

    header('Location: broker_list.php?msg=' . urlencode("Status updated to $new."));
    exit;
}

/* ---------------- Set / Save Credentials for a broker (form POST) --------- */
if ($action === 'setcred' && $id > 0 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = trim($_POST['password'] ?? '');

    if ($username === '' || $password === '') {
        $err = "Username and password are required.";
    } else {
        $hash = password_hash($password, PASSWORD_BCRYPT);

        $escUsername = mysqli_real_escape_string($conn, $username);
        $uRes = mysqli_query($conn, "SELECT id FROM users WHERE username = '{$escUsername}' LIMIT 1");
        $u = $uRes ? mysqli_fetch_assoc($uRes) : null;

        if ($u) {
            $stmt = @mysqli_prepare($conn, "UPDATE users SET password_hash = ?, role = 'Broker' WHERE id = ?");
            if ($stmt) {
                mysqli_stmt_bind_param($stmt, 'si', $hash, $u['id']);
                if (!mysqli_stmt_execute($stmt)) $err = "Failed to update credentials.";
                else $msg = "Credentials updated for existing user.";
                mysqli_stmt_close($stmt);
            } else {
                $escHash = mysqli_real_escape_string($conn, $hash);
                if (!mysqli_query($conn, "UPDATE users SET password_hash='{$escHash}', role='Broker' WHERE id=".(int)$u['id'])) {
                    $err = "Failed to update credentials.";
                } else $msg = "Credentials updated for existing user (fallback).";
            }
        } else {
            $role = 'Broker';
            $stmt = @mysqli_prepare($conn, "INSERT INTO users (username, password_hash, role) VALUES (?, ?, ?)");
            if ($stmt) {
                mysqli_stmt_bind_param($stmt, 'sss', $username, $hash, $role);
                if (!mysqli_stmt_execute($stmt)) {
                    $err = "Failed to save credentials.";
                } else {
                    $msg = "Credentials saved.";
                }
                mysqli_stmt_close($stmt);
            } else {
                $escHash = mysqli_real_escape_string($conn, $hash);
                $escRole = mysqli_real_escape_string($conn, $role);
                if (!mysqli_query($conn, "INSERT INTO users (username,password_hash,role) VALUES ('{$escUsername}','{$escHash}','{$escRole}')")) {
                    $err = "Failed to save credentials (fallback).";
                } else {
                    $msg = "Credentials saved (fallback).";
                }
            }
        }
    }
}

/* ---------------- Determine credential edit row --------------------------- */
$editCred = ($action === 'cred' && $id > 0) ? (int)$id : 0;

/* ---------------- Fetch broker rows -------------------------------------- */
$brokers = [];
$rs = mysqli_query($conn, "SELECT id, broker_code, first_name, last_name, email, phone, firm_name, status, created_at FROM brokers ORDER BY created_at DESC");
if ($rs) {
    while ($row = mysqli_fetch_assoc($rs)) $brokers[] = $row;
}

/* ---------------- baseUrl fallback (if config.php didn't provide) --------- */
$baseUrl = $baseUrl ?? ((isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\'));

/* ================================ $styles ================================= */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo esc($baseUrl); ?>/assets/libs/sweetalert2/sweetalert2.min.css">
<style>
  .page-title{ font-weight:600; font-size:1.125rem; }
  .card.custom-card { border-radius: .6rem; box-shadow: 0 0.25rem 0.5rem rgba(18,38,63,0.04); }
  .table thead th { font-weight:600; font-size:.95rem; }
  .badge { padding:.5em .6em; border-radius:.5rem; font-size:.825rem; }
  .bg-primary-transparent { background-color: rgba(23,83,252,0.08); color:#1753fc; }
  .bg-warning-transparent { background-color: rgba(251,188,11,0.08); color:#fbcc0b; }
  .bg-danger-transparent { background-color: rgba(238,51,94,0.08); color:#ee335e; }
  .bg-secondary-transparent { background-color: rgba(99,102,241,0.06); color:#6366f1; }
  .action-btns .btn { margin-left:.35rem; margin-right:.35rem; }
  @media (max-width: 575px) {
    .action-btns .btn { display:block; margin:6px 0; width:100%; }
  }
</style>
<?php $styles = ob_get_clean();

/* =============================== $content ================================= */
ob_start(); ?>
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title mb-0">Brokers</h1>
    <div class="d-flex gap-2 align-items-center">
      <a href="broker_onboarding.php" class="btn btn-primary"><i class="fa fa-plus me-1"></i> New Application</a>
    </div>
  </div>
</div>

<?php if (!empty($_GET['msg'])): ?><div class="alert alert-success"><?php echo h($_GET['msg']); ?></div><?php endif; ?>
<?php if ($msg): ?><div class="alert alert-success"><?php echo h($msg); ?></div><?php endif; ?>
<?php if ($err): ?><div class="alert alert-danger"><?php echo h($err); ?></div><?php endif; ?>

<div class="card custom-card mb-3">
  <div class="card-body">
    <div class="d-flex align-items-center justify-content-between mb-3">
      <h4 class="card-title mb-0"><i class="fas fa-users me-2"></i> All Applications</h4>
      <small class="text-muted">Total: <?php echo count($brokers); ?></small>
    </div>

    <div class="table-responsive">
      <table class="table table-hover align-middle">
        <thead>
          <tr>
            <th style="width:60px;">#</th>
            <th>Code</th>
            <th>Name</th>
            <th>Email</th>
            <th>Phone</th>
            <th>Firm</th>
            <th>Status</th>
            <th class="text-end" style="width:340px;">Actions</th>
          </tr>
        </thead>
        <tbody>
        <?php if (empty($brokers)): ?>
            <tr><td colspan="8" class="text-muted">No applications found.</td></tr>
        <?php else: foreach ($brokers as $r): ?>
          <tr>
            <td><?php echo (int)$r['id']; ?></td>
            <td><?php echo h($r['broker_code']); ?></td>
            <td><?php echo h(trim(($r['first_name'] ?? '') . ' ' . ($r['last_name'] ?? ''))); ?></td>
            <td><?php echo h($r['email']); ?></td>
            <td><?php echo h($r['phone']); ?></td>
            <td><?php echo h($r['firm_name']); ?></td>
            <td>
              <?php $b = $r['status']; $cls = badge_class_from_status($b); ?>
              <span class="badge <?php echo h($cls); ?>"><?php echo h($b ?: 'Pending'); ?></span>
            </td>
            <td class="text-end action-btns">
              <a class="btn btn-sm btn-outline-info" href="broker_view.php?id=<?php echo (int)$r['id']; ?>"><i class="fas fa-eye me-1"></i> View</a>
              <a class="btn btn-sm btn-outline-secondary" href="broker_list.php?action=cred&id=<?php echo (int)$r['id']; ?>"><i class="fas fa-key me-1"></i> Set Credentials</a>
              <a class="btn btn-sm btn-outline-success" href="broker_list.php?action=approve&id=<?php echo (int)$r['id']; ?>" onclick="return confirm('Approve this broker?')"><i class="fas fa-check me-1"></i> Approve</a>
              <a class="btn btn-sm btn-outline-danger" href="broker_list.php?action=reject&id=<?php echo (int)$r['id']; ?>" onclick="return confirm('Reject this broker?')"><i class="fas fa-times me-1"></i> Reject</a>
            </td>
          </tr>

          <?php if ($editCred === (int)$r['id']): ?>
          <tr>
            <td colspan="8">
              <form method="post" action="broker_list.php?action=setcred&id=<?php echo (int)$r['id']; ?>" class="d-flex flex-wrap gap-2 align-items-center">
                <div class="input-group" style="min-width:230px;">
                  <span class="input-group-text">Username</span>
                  <input type="text" name="username" class="form-control" value="<?php echo h($r['broker_code']); ?>" required>
                </div>

                <div class="input-group" style="min-width:230px;">
                  <span class="input-group-text">Password</span>
                  <input type="text" name="password" class="form-control" required>
                </div>

                <div>
                  <button class="btn btn-primary" type="submit"><i class="fa fa-save me-1"></i> Save</button>
                  <a class="btn btn-light" href="broker_list.php">Cancel</a>
                </div>
              </form>
            </td>
          </tr>
          <?php endif; ?>

        <?php endforeach; endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>
<?php $content = ob_get_clean(); /* end $content */ ?>

/* =============================== $scripts ================================= */
<?php ob_start(); ?>
<script src="<?php echo esc($baseUrl); ?>/assets/libs/sweetalert2/sweetalert2.min.js"></script>
<script>
  // Optional: show a nicer toast when redirected with msg
  (function(){
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('msg')) {
      const msg = urlParams.get('msg');
      setTimeout(function(){
        Swal.fire({icon:'success', title:'Success', text: msg, timer:2200, showConfirmButton:false});
      }, 80);
    }
  })();
</script>
<?php $scripts = ob_get_clean(); ?>

<?php
/* ------------------------------ Render base ------------------------------ */
/* layouts/base.php is expected to render $styles, $content and $scripts */
include __DIR__ . '/layouts/base.php';
