<?php
/* =============================================================================
 * File   : pages/leads_upload.php
 * Purpose: THEME-INTEGRATED CSV uploader for broriserin_onb.leads
 * Stack  : Core PHP + MySQLi (NO PDO). DB connection in ../config.php
 * Rules  : COLLATE=utf8mb4_general_ci
 * Notes  :
 *   - Accepts CSV with columns: contact_name, phone_number, email, lead_source
 *   - Header-less CSV supported in positional order: name, phone, email, source
 *   - Uses the SAME layout/includes pattern as leads.php so the sidebar shows.
 * ============================================================================= */

date_default_timezone_set('Asia/Kolkata');
@session_start();

/* --- IMPORTANT: use the same include locations as leads.php so layout (sidebar) loads --- */
require_once __DIR__ . '/../config.php';
require_once 'includes/check-permission.php';
$currentPage = basename(__FILE__);
checkPageAccessOrRedirect($currentPage, $conn_sa);

/* Charset / collation (no PDO) */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* Helpers */
if (!function_exists('h')) { function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); } }
function col_exists($conn,$table,$col){
  $t = mysqli_real_escape_string($conn,$table);
  $c = mysqli_real_escape_string($conn,$col);
  $q = mysqli_query($conn,"SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='{$t}' AND COLUMN_NAME='{$c}'");
  return ($q && mysqli_num_rows($q) > 0);
}

/* Ensure table & columns exist (idempotent) */
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS `leads` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `contact_name` VARCHAR(150) COLLATE utf8mb4_general_ci NOT NULL,
  `phone_number` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `email` VARCHAR(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `lead_source` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT '',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB COLLATE=utf8mb4_general_ci");

if (!col_exists($conn,'leads','phone_number')) {
  @mysqli_query($conn, "ALTER TABLE `leads` ADD COLUMN `phone_number` VARCHAR(30) COLLATE utf8mb4_general_ci DEFAULT NULL AFTER `contact_name`");
}
if (!col_exists($conn,'leads','lead_source')) {
  @mysqli_query($conn, "ALTER TABLE `leads` ADD COLUMN `lead_source` VARCHAR(191) COLLATE utf8mb4_general_ci DEFAULT '' AFTER `email`");
}

/* Sample CSV download (helpful button in UI) */
if (isset($_GET['sample']) && $_GET['sample'] === '1') {
  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename=leads_sample.csv');
  $out = fopen('php://output', 'w');
  fputcsv($out, ['contact_name','phone_number','email','lead_source']);
  fputcsv($out, ['John Doe', '+919876543210', 'john@example.com', 'Website']);
  fputcsv($out, ['Jane Roe', '9876543210', 'jane@example.com', 'Referral']);
  fclose($out);
  exit;
}

/* Handle upload */
$errors = [];
$success = null;
$inserted = 0;
$skipped  = 0;
$rows_total = 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (empty($_FILES['leads_csv']) || $_FILES['leads_csv']['error'] !== UPLOAD_ERR_OK) {
    $errors[] = "Please choose a CSV file to upload (ensure size within server limits).";
  } else {
    $fileTmp = $_FILES['leads_csv']['tmp_name'];
    if (($handle = fopen($fileTmp, 'r')) === false) {
      $errors[] = "Failed to open uploaded file.";
    } else {
      $header = fgetcsv($handle);
      if ($header === false) {
        $errors[] = "CSV appears to be empty.";
      } else {
        /* Flexible header mapping */
        $map = [];
        foreach ($header as $i => $col) {
          $colname = strtolower(trim((string)$col));
          if (in_array($colname, ['contact_name','name','contact'], true)) $map['contact_name'] = $i;
          elseif (in_array($colname, ['phone_number','phone','mobile','msisdn'], true)) $map['phone_number'] = $i;
          elseif (in_array($colname, ['email','e-mail','mail'], true)) $map['email'] = $i;
          elseif (in_array($colname, ['lead_source','source','leadsource'], true)) $map['lead_source'] = $i;
        }

        $use_positional = (!isset($map['contact_name']) || !isset($map['phone_number']) || !isset($map['email']));
        if ($use_positional) rewind($handle);

        /* Prepared insert */
        $stmt = $conn->prepare("INSERT INTO `leads` (`contact_name`,`phone_number`,`email`,`lead_source`) VALUES (?,?,?,?)");
        if ($stmt === false) {
          $errors[] = "Prepare failed: " . h($conn->error);
          fclose($handle);
        } else {
          $conn->begin_transaction();

          while (($row = fgetcsv($handle)) !== false) {
            $rows_total++;
            if ($use_positional) {
              $name  = isset($row[0]) ? trim($row[0]) : '';
              $phone = isset($row[1]) ? trim($row[1]) : '';
              $email = isset($row[2]) ? trim($row[2]) : '';
              $src   = isset($row[3]) ? trim($row[3]) : '';
            } else {
              $name  = isset($row[$map['contact_name']]) ? trim($row[$map['contact_name']]) : '';
              $phone = isset($row[$map['phone_number']]) ? trim($row[$map['phone_number']]) : '';
              $email = isset($row[$map['email']]) ? trim($row[$map['email']]) : '';
              $src   = (isset($map['lead_source']) && isset($row[$map['lead_source']])) ? trim($row[$map['lead_source']]) : '';
            }

            /* Skip empty */
            if ($name === '' && $phone === '' && $email === '' && $src === '') { $skipped++; continue; }

            /* Normalize + validate */
            $phone_normalized = preg_replace('/[^\d+]/', '', $phone);
            if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) { $skipped++; continue; }
            $src = mb_substr($src ?? '', 0, 191);

            $stmt->bind_param('ssss', $name, $phone_normalized, $email, $src);
            $ok = $stmt->execute();
            if ($ok) $inserted++; else $skipped++;
          }

          $conn->commit();
          $stmt->close();
          fclose($handle);
          $success = "{$inserted} lead(s) inserted. {$skipped} row(s) skipped. {$rows_total} row(s) processed.";
        }
      }
    }
  }
}

/* ============================== THEME SECTIONS ============================== */
ob_start(); ?>
<style>
/* Page-specific tweaks that fit your CRM theme */
.csv-card .card-title { font-weight: 600; }
.csv-card .muted { color:#6c757d; font-size:.9rem; }
.csv-card .sample-box { background: var(--bs-light, #f6f7f9); border-radius: .5rem; padding: .75rem; font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace; white-space: pre-wrap; }
.csv-card .divider { height:1px; background: rgba(0,0,0,.08); margin: .75rem 0; }
</style>
<?php $styles = ob_get_clean(); ?>

<?php ob_start(); ?>
<!-- Header / Breadcrumb (same structure as leads.php so sidebar/layout stays consistent) -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Leads CSV Upload</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Apps</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">CRM</a></li>
      <li class="breadcrumb-item active" aria-current="page">Upload</li>
    </ol>
  </div>
</div>

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card csv-card">
      <div class="card-header d-flex align-items-center justify-content-between flex-wrap gap-2">
        <div class="card-title">Import Leads (CSV)</div>
        <div class="d-flex gap-2">
          <a href="<?php echo h($_SERVER['PHP_SELF']); ?>?sample=1" class="btn btn-light btn-wave">Download Sample</a>
        </div>
      </div>

      <div class="card-body">
        <?php if (!empty($errors)): ?>
          <div class="alert alert-danger">
            <strong>Upload Error</strong>
            <ul class="mb-0"><?php foreach ($errors as $e): ?><li><?php echo h($e); ?></li><?php endforeach; ?></ul>
          </div>
        <?php endif; ?>

        <?php if ($success !== null): ?>
          <div class="alert alert-success"><strong>Result:</strong> <?php echo h($success); ?></div>
        <?php endif; ?>

        <form method="post" enctype="multipart/form-data" class="row g-3">
          <div class="col-12 col-md-6">
            <label class="form-label">Select CSV file</label>
            <input type="file" class="form-control" name="leads_csv" accept=".csv,text/csv" required>
            <div class="form-text">Allowed: .csv — UTF-8 recommended.</div>
          </div>
          <div class="col-12 col-md-6 d-flex align-items-end">
            <button type="submit" class="btn btn-primary btn-wave">Upload & Insert</button>
          </div>
        </form>

        <div class="csv-help mt-3">
          <div class="divider"></div>
          <div class="row g-3">
            <div class="col-12 col-lg-6">
              <h6 class="mb-2">Accepted Columns</h6>
              <p class="mb-2 muted">
                <code>contact_name</code>, <code>phone_number</code>, <code>email</code>, <code>lead_source</code>
              </p>
              <p class="mb-0 muted">Header names are flexible (e.g., <em>name</em>, <em>phone</em>, <em>source</em>) or omit header and use positional order.</p>
            </div>
            <div class="col-12 col-lg-6">
              <h6 class="mb-2">Example CSV</h6>
              <div class="sample-box">
contact_name,phone_number,email,lead_source
John Doe,+919876543210,john@example.com,Website
Jane Roe,9876543210,jane@example.com,Referral
              </div>
            </div>
          </div>

          <div class="divider"></div>
          <ul class="mb-0 muted">
            <li>Phone numbers are normalized to digits with optional leading <code>+</code>.</li>
            <li>Invalid emails (if present) are skipped; blank lines are ignored.</li>
            <li>Connection collation is enforced as <code>utf8mb4_general_ci</code>.</li>
          </ul>
        </div>
      </div>
    </div>
  </div>
</div>
<?php $content = ob_get_clean(); ?>

<?php ob_start(); ?>
<script>
// (Reserved for future page-specific enhancements)
</script>
<?php $scripts = ob_get_clean(); ?>

<?php
/* Render using the same base layout as other CRM pages (ensures sidebar shows) */
include 'layouts/base.php';
