<?php
/* admin/employees/without-applications.php
 * Purpose: Employees who DO NOT have any application yet (collation-safe), styled like your Jobs List pages
 * Stack: Core PHP + MySQLi (no PDO). DB connection in ../config.php. Collation utf8mb4_general_ci.
 * Pattern: ($styles, $content, $scripts) + include 'layouts/base.php' so theme header/sidebar apply automatically.
 */

session_start();
require_once __DIR__ . '/../config.php';

/* Ensure charset + collation */
mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET collation_connection = 'utf8mb4_general_ci'");

/* Auth */
// if (!isset($_SESSION['admin']) || !isset($_SESSION['username'])) {
//     header("Location: " . BASE_URL . "admin/login");
//     exit;
// }

/* Helpers */
function h($v){ return htmlspecialchars((string)($v??''), ENT_QUOTES, 'UTF-8'); }

/* Admin (optional) */
$username = mysqli_real_escape_string($conn, $_SESSION['username']);
$admin_q  = mysqli_query($conn, "SELECT * FROM admins WHERE username='{$username}' LIMIT 1");
$admin    = $admin_q ? mysqli_fetch_assoc($admin_q) : null;

/* Base URL used by theme assets (same pattern as other Jobs List pages) */
// $rootFolder = basename($_SERVER['DOCUMENT_ROOT']);
// $baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . str_replace('/pages', '', dirname($_SERVER['SCRIPT_NAME']));

/* Data: Employees missing any application (collation-aligned on email) */
$sql = "
  SELECT DISTINCT s.id, s.first_name, s.last_name, s.country, s.phone, s.email, s.created_at
  FROM employees s
  WHERE NOT EXISTS (
    SELECT 1
    FROM employee_applications a
    WHERE a.email COLLATE utf8mb4_general_ci = s.email COLLATE utf8mb4_general_ci
  )
  ORDER BY s.created_at DESC, s.id DESC
";
$rows_q = mysqli_query($conn, $sql);
$total_without_app = $rows_q ? mysqli_num_rows($rows_q) : 0;

/* Context counts (chips) */
$all_employees_q = mysqli_query($conn, "SELECT COUNT(*) c FROM employees");
$all_employees   = $all_employees_q ? (int)mysqli_fetch_assoc($all_employees_q)['c'] : 0;
$all_apps_q      = mysqli_query($conn, "SELECT COUNT(*) c FROM employee_applications");
$all_apps        = $all_apps_q ? (int)mysqli_fetch_assoc($all_apps_q)['c'] : 0;

/* ---------- Internal styles (match the table style you showed) ---------- */
ob_start();
?>
<style>
/* from rejected-list table style */
.avatar-initial {
    width: 40px; height: 40px; border-radius: 50%;
    display: inline-grid; place-items: center;
    font-weight: 700; color: #fff; background: var(--primary03, #0d6efd);
}
.joblist-list .text-sub { font-size: 12px; color: var(--default-text-color, #6c757d); margin: 0; }
.joblist-list .fw-medium a { text-decoration: none; }
.badge-submitted  { background: rgba(13,110,253,.12); color:#0d6efd; border:1px solid rgba(13,110,253,.2); }
.badge-inprogress { background: rgba(255,193,7,.15); color:#b58100; border:1px solid rgba(255,193,7,.25); }
.badge-rejected   { background: rgba(255,193,7,.35); color:red; border:1px solid rgba(13,110,253,.2); }
.badge-other      { background: rgba(108,117,125,.15); color:#6c757d; border:1px solid rgba(108,117,125,.25); }
.table-hover tbody tr:hover { background: rgba(13,110,253,.04); }

/* lightweight header chips + toolbar */
.page-title{display:flex;align-items:center;gap:.75rem}
.page-title .dot{width:10px;height:10px;border-radius:50%;background:var(--bs-primary)}
.card-soft{border:0;box-shadow:0 4px 18px rgba(0,0,0,.08);border-radius:1rem}
.toolbar{display:flex;flex-wrap:wrap;gap:.75rem;align-items:center}
.search-wrap{position:relative;min-width:240px}
.search-wrap input{padding-left:2.25rem}
.search-wrap .bi{position:absolute;left:.65rem;top:50%;transform:translateY(-50%);opacity:.6}
.chip{display:inline-flex;align-items:center;gap:.5rem;border:1px solid var(--bs-border-color);padding:.35rem .65rem;border-radius:999px;background:#fff}
.chip .bi{opacity:.7}
.btn-icon{display:inline-flex;align-items:center;gap:.35rem}
.small-muted{font-size:.875rem;color:#6c757d}
@media (max-width: 575.98px){.toolbar{flex-direction:column;align-items:stretch}.search-wrap{min-width:100%}}
</style>
<?php
$styles = ob_get_clean();

/* ---------- Page content (Jobs List look & table structure like rejected-list) ---------- */
ob_start();
?>
<!-- Start::page-header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0">Employees Without Applications</h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="javascript:void(0);">Dashboards</a></li>
      <li class="breadcrumb-item"><a href="javascript:void(0);">Employees</a></li>
      <li class="breadcrumb-item active" aria-current="page">No Applications</li>
    </ol>
  </div>
</div>
<!-- End::page-header -->

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card overflow-hidden">
      <div class="card-header justify-content-between">
        <div class="card-title">All Employees Missing Applications</div>
        <div class="d-flex flex-wrap gap-2">
          <div class="search-wrap">
            <i class="bi bi-search"></i>
            <input id="tableSearch" type="text" class="form-control" placeholder="Search name, email, country, phone...">
          </div>
          <div class="d-flex align-items-center gap-2">
            <label for="rowsPerPage" class="text-muted small mb-0">Rows:</label>
            <select id="rowsPerPage" class="form-select form-select-sm" style="width:auto">
              <option value="10">10</option><option value="25">25</option><option value="50">50</option><option value="500">500</option><option value="all">All</option>
            </select>
          </div>
          <div class="ms-auto d-flex gap-2">
            <button id="btnRefresh" class="btn btn-light btn-icon"><i class="bi bi-arrow-clockwise"></i><span>Refresh</span></button>
            <button id="btnExport" class="btn btn-outline-primary btn-icon"><i class="bi bi-filetype-csv"></i><span>Export CSV</span></button>
          </div>
        </div>
      </div>

      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table table-hover text-nowrap" id="appsTable">
            <thead>
              <tr>
                <th class="ps-4"><input class="form-check-input check-all" type="checkbox" id="checkAllRows" aria-label="..."></th>
                <th>Employee</th>
                <th>Email</th>
                <th>Phone</th>
                <th>Country</th>
                <th>Registered</th>
                <th>Action</th>
              </tr>
            </thead>
            <tbody>
              <?php if ($rows_q && mysqli_num_rows($rows_q)>0): ?>
                <?php while($r = mysqli_fetch_assoc($rows_q)):
                  $first = trim((string)($r['first_name'] ?? ''));
                  $last  = trim((string)($r['last_name'] ?? ''));
                  $name  = trim($first.' '.$last);
                  $name  = $name !== '' ? $name : 'Unknown';
                  $initials = strtoupper(mb_substr($first ?: 'U',0,1).mb_substr($last ?: 'N',0,1));
                  $country  = $r['country'] ?: '-';
                  $email    = $r['email'] ?: '-';
                  $phone    = $r['phone'] ?: '-';
                  $regAt    = !empty($r['created_at']) ? date('M d, Y', strtotime($r['created_at'])) : '-';
                ?>
                <tr class="joblist-list">
                  <td class="ps-4"><input class="form-check-input row-check" type="checkbox" value="<?= (int)$r['id'] ?>"></td>
                  <td>
                    <div class="d-flex align-items-center">
                      <div class="lh-1">
                        <span class="avatar avatar-md avatar-rounded">
                          <span class="avatar-initial"><?= h($initials) ?></span>
                        </span>
                      </div>
                      <div class="ms-2">
                        <p class="fw-medium mb-0 d-flex align-items-center"><a href="javascript:void(0)"><?= h($name) ?></a></p>
                        <p class="text-sub"><?= h($country) ?></p>
                      </div>
                    </div>
                  </td>
                  <td><a href="mailto:<?= h($email) ?>"><?= h($email) ?></a></td>
                  <td><a href="tel:<?= h($phone) ?>" class="text-decoration-none"><?= h($phone) ?></a></td>
                  <td><?= h($country) ?></td>
                  <td><?= h($regAt) ?></td>
                  <td>
                    <a href="mailto:<?= h($email) ?>" class="btn btn-icon btn-sm btn-primary-light btn-wave waves-effect waves-light" title="Email">
                      <i class="ri-mail-line"></i>
                    </a>
                  </td>
                </tr>
                <?php endwhile; ?>
              <?php else: ?>
                <tr><td colspan="7" class="text-center text-muted py-5">No records found.</td></tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </div>

      <div class="card-footer border-top-0">
        <div class="d-flex align-items-center flex-wrap overflow-auto">
          <div class="mb-2 mb-sm-0">
            Showing <b>1</b> to <b><?= h(max(1, $total_without_app)) ?></b> of <b><?= h($total_without_app) ?></b> entries
            <i class="bi bi-arrow-right ms-2 fw-medium"></i>
          </div>
          <div class="ms-auto d-flex flex-wrap gap-2">
            <span class="chip"><i class="bi bi-people"></i>Total Employees: <strong><?= number_format($all_employees) ?></strong></span>
            <span class="chip"><i class="bi bi-inbox"></i>Total Applications: <strong><?= number_format($all_apps) ?></strong></span>
            <span class="chip"><i class="bi bi-envelope-exclamation"></i>Without Application: <strong><?= number_format($total_without_app) ?></strong></span>
          </div>
        </div>
      </div>

    </div>
  </div>
</div>
<?php
$content = ob_get_clean();

/* ---------- Internal scripts ---------- */
ob_start();
?>
<script src="<?php echo $baseUrl; ?>/assets/js/job-list.js"></script>
<script>
(function(){
  const table = document.getElementById('appsTable');
  const tbody = table ? table.querySelector('tbody') : null;
  const rows  = tbody ? Array.from(tbody.querySelectorAll('tr.joblist-list')) : [];
  const searchInput = document.getElementById('tableSearch');
  const rowsPerPageSel = document.getElementById('rowsPerPage');
  const refreshBtn = document.getElementById('btnRefresh');
  const exportBtn  = document.getElementById('btnExport');
  const checkAll   = document.getElementById('checkAllRows');

  function norm(s){ return (s||'').toString().toLowerCase().trim(); }

  // Search (client-side)
  if (searchInput && rows.length){
    searchInput.addEventListener('input', function(){
      const q = norm(this.value);
      rows.forEach(tr=>{
        const show = !q || norm(tr.innerText).includes(q);
        tr.style.display = show ? '' : 'none';
      });
      // optional: reset "rows per page" view by simply not paginating server-side here
    });
  }

  // Rows per page (simple client-side show/hide)
  if (rowsPerPageSel && rows.length){
    rowsPerPageSel.addEventListener('change', function(){
      const val = this.value;
      const perPage = (val==='all') ? rows.length : parseInt(val,10);
      rows.forEach((tr,idx)=>{ tr.style.display = (idx < perPage) ? '' : 'none'; });
    });
    rowsPerPageSel.dispatchEvent(new Event('change'));
  }

  // Check-all
  if (checkAll && rows.length){
    checkAll.addEventListener('change', function(){
      tbody.querySelectorAll('.row-check').forEach(cb => { cb.checked = checkAll.checked; });
    });
  }

  // Refresh
  if (refreshBtn){ refreshBtn.addEventListener('click', ()=>location.reload()); }

  // Export CSV (visible rows)
  if (exportBtn){
    exportBtn.addEventListener('click', function(){
      const headers = ['Employee','Email','Phone','Country','Registered'];
      const lines = [headers.join(',')];
      const visible = rows.filter(tr => tr.style.display !== 'none');
      visible.forEach(tr=>{
        const tds = tr.querySelectorAll('td');
        const employee = tds[1].innerText.replace(/\r?\n|\r/g,' ').trim();
        const email    = tds[2].innerText.trim();
        const phone    = tds[3].innerText.trim();
        const country  = tds[4].innerText.trim();
        const reg      = tds[5].innerText.trim();
        const arr = [employee,email,phone,country,reg].map(v=>`"${v.replace(/"/g,'""')}"`);
        lines.push(arr.join(','));
      });
      const blob = new Blob([lines.join('\n')], {type:'text/csv;charset=utf-8;'});
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url; a.download = 'employees_without_applications.csv';
      document.body.appendChild(a); a.click(); document.body.removeChild(a);
      URL.revokeObjectURL(url);
    });
  }
})();
</script>
<?php
$scripts = ob_get_clean();

/* ---------- Render with theme base (header/sidebar included there) ---------- */
include 'layouts/base.php';
