<?php
/* =============================================================================
 * File   : project_form.php  (Melody admin theme)
 * Goal   : Create / Edit Project using centralized config.php (MySQLi, NO PDO)
 * Uses   : layouts/base.php ($styles, $content, $scripts)
 * Notes  : - Detects columns safely; renders only what exists
 *          - Fetches existing gallery images from project_media (url/file_url)
 *          - Shows "Remove" buttons for fetched images & floor plans
 * DB     : utf8mb4_general_ci
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ------------------------- Centralized config ------------------------- */
$cfgs = [
  __DIR__ . '/../config.php',
  __DIR__ . '/../../config.php',
  __DIR__ . '/config.php'
];
$cfgFound = false;
foreach ($cfgs as $c) {
  if (file_exists($c)) { require_once $c; $cfgFound = true; break; }
}
if (!$cfgFound) die('config.php not found');

/* Pick a mysqli handle (same logic as project_add.php / project_details.php) */
$DB = null;
if (isset($conn_crm)     && $conn_crm     instanceof mysqli) $DB = $conn_crm;
elseif (isset($conn_crmplus) && $conn_crmplus instanceof mysqli) $DB = $conn_crmplus;
elseif (isset($conn)     && $conn     instanceof mysqli) $DB = $conn;
elseif (isset($conn_sa)  && $conn_sa  instanceof mysqli) $DB = $conn_sa;

if (!$DB instanceof mysqli) { http_response_code(500); exit('DB connection not found - check config.php'); }

/* Charset / Collation */
@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* ---------------------------- Helpers ---------------------------- */
if (!function_exists('h')) { function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); } }
function q($s){ global $DB; return mysqli_real_escape_string($DB,(string)$s); }
function base_url(){
  if (!empty($GLOBALS['baseUrl'])) return rtrim($GLOBALS['baseUrl'],'/');
  $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS']!=='off') ? 'https' : 'http';
  $host  = $_SERVER['HTTP_HOST'] ?? ($_SERVER['SERVER_NAME'] ?? 'localhost');
  $dir   = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? '/'),'/');
  return rtrim("$proto://$host$dir",'/');
}
function table_exists($name){ global $DB; $name = q($name); $r = @mysqli_query($DB,"SHOW TABLES LIKE '{$name}'"); $ok = ($r && mysqli_num_rows($r)>0); if($r) @mysqli_free_result($r); return $ok; }
function col_exists($tbl,$col){ global $DB; $tbl = q($tbl); $col = q($col); $r = @mysqli_query($DB,"SHOW COLUMNS FROM `{$tbl}` LIKE '{$col}'"); $ok = ($r && mysqli_num_rows($r)>0); if($r) @mysqli_free_result($r); return $ok; }

/* ------------------------- Resolve projects/media table -------------------- */
$PROJECTS_TABLE = null;
foreach (['projects','project','tbl_projects','pp_projects'] as $t) {
  if (table_exists($t)) { $PROJECTS_TABLE = $t; break; }
}
if (!$PROJECTS_TABLE) { die('Projects table not found.'); }

$MEDIA_TABLE = table_exists('project_media') ? 'project_media' : null;

/* ------------------------- Load Project (defaults) ------------------------ */
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$project = [];
if ($id > 0) {
  $rs = mysqli_query($DB,"SELECT * FROM `{$PROJECTS_TABLE}` WHERE id={$id} LIMIT 1");
  if ($rs && mysqli_num_rows($rs)) { $project = mysqli_fetch_assoc($rs); }
  if ($rs) @mysqli_free_result($rs);
}

/* Helper: get a value only if column exists, else '' */
function pv($key,$default=''){
  global $project,$PROJECTS_TABLE;
  return (isset($project[$key]) && col_exists($PROJECTS_TABLE,$key)) ? $project[$key] : $default;
}

/* ------------------------- Fetch existing media -------------------------- */
$existing = ['images'=>[], 'docs'=>[], 'floors'=>[]];
if ($id && $MEDIA_TABLE) {
  $urlCol   = col_exists($MEDIA_TABLE,'url') ? 'url' : (col_exists($MEDIA_TABLE,'file_url') ? 'file_url' : null);
  $titleCol = col_exists($MEDIA_TABLE,'title') ? 'title' : null;
  $sortCol  = col_exists($MEDIA_TABLE,'sort_order') ? 'sort_order' : 'id';
  $isActCol = col_exists($MEDIA_TABLE,'is_active') ? 'COALESCE(is_active,1)=1 AND ' : '';
  if ($urlCol) {
    $extraCols = [];
    if ($titleCol) $extraCols[] = $titleCol;
    if ($sortCol)  $extraCols[] = $sortCol;
    $sel = "id, media_type, {$urlCol} AS file_url".($extraCols?(', '.implode(',',$extraCols)):'');
    $sql = "SELECT {$sel} FROM `{$MEDIA_TABLE}` WHERE {$isActCol} project_id={$id} ORDER BY ".($sortCol?$sortCol.', ':'')."id";
    if ($mrs = mysqli_query($DB,$sql)) {
      while($row = mysqli_fetch_assoc($mrs)){
        $mt = strtolower(trim((string)($row['media_type'] ?? '')));
        if ($mt === 'image') { $existing['images'][] = $row; }
        elseif ($mt === 'doc') { $existing['docs'][] = $row; }
        elseif ($mt === 'floor' || $mt === 'floorplan') { $existing['floors'][] = $row; }
      }
      @mysqli_free_result($mrs);
    }
  }
}

/* ------------------------- Melody Theme Blocks --------------------------- */

/* ========== $styles ========== */
ob_start(); ?>
<link rel="stylesheet" href="<?php echo base_url(); ?>/assets/libs/filepond/filepond.min.css">
<link rel="stylesheet" href="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-preview/filepond-plugin-image-preview.min.css">
<style>
  .pill      { display:inline-flex; align-items:center; gap:.5rem; padding:.45rem .75rem; border:1px solid #e4e7eb; border-radius:999px; background:#f8fafc; }
  .pill input{ margin-top:1px; }
  .fp-row    { border:1px dashed #d8dde6; border-radius:10px; padding:12px; margin-bottom:10px; background:#fafbff; }
  .thumb     { width:160px; height:110px; object-fit:cover; border-radius:10px; border:1px solid #e6e8ef; }
  .muted     { color:#6b7380; }
  .thumb-grid{ display:flex; flex-wrap:wrap; gap:.75rem; }
  .thumb-card{ width:180px; border:1px solid #e9ecf2; border-radius:12px; overflow:hidden; background:#fff; }
  .thumb-card .body{ padding:.5rem; text-align:center; }
</style>
<?php $styles = ob_get_clean();

/* ========== $content ========== */
ob_start(); ?>

<!-- Page Header -->
<div class="page-header-breadcrumb mb-3">
  <div class="d-flex align-center justify-content-between flex-wrap">
    <h1 class="page-title fw-medium fs-18 mb-0"><?php echo $id ? 'Edit Project' : 'Create Project'; ?></h1>
    <ol class="breadcrumb mb-0">
      <li class="breadcrumb-item"><a href="projects-list.php">Projects List</a></li>
      <li class="breadcrumb-item active" aria-current="page"><?php echo $id ? 'Edit' : 'Create'; ?></li>
    </ol>
  </div>
</div>

<div class="row">
  <div class="col-xl-12">
    <div class="card custom-card">
      <div class="card-header">
        <div class="card-title"><?php echo $id ? 'Update Project' : 'Create Project'; ?></div>
        <div class="ms-auto">
          <a href="projects-list.php" class="btn btn-outline-secondary btn-sm"><i class="ri-arrow-go-back-line me-1"></i>Projects List</a>
        </div>
      </div>

      <div class="card-body">
        <form method="post" action="project_save.php" enctype="multipart/form-data" class="row gy-3">
          <input type="hidden" name="id" value="<?php echo (int)$id; ?>">

          <!-- BASIC -->
          <?php if (col_exists($PROJECTS_TABLE,'title')): ?>
            <div class="col-xl-6">
              <label class="form-label">Title *</label>
              <input type="text" name="title" class="form-control" value="<?php echo h(pv('title')); ?>" required>
            </div>
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'city')): ?>
            <div class="col-xl-3">
              <label class="form-label">City</label>
              <input type="text" name="city" class="form-control" value="<?php echo h(pv('city')); ?>">
            </div>
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'address')): ?>
            <div class="col-xl-3">
              <label class="form-label">Address</label>
              <input type="text" name="address" class="form-control" value="<?php echo h(pv('address')); ?>">
            </div>
          <?php endif; ?>

          <div class="col-12"><hr></div>

          <!-- CONFIG / AREA / PRICE -->
          <?php if (col_exists($PROJECTS_TABLE,'configuration')): ?>
            <div class="col-xl-4">
              <label class="form-label">Configuration <span class="muted">(e.g. 1BHK, 2BHK)</span></label>
              <input type="text" name="configuration" class="form-control" value="<?php echo h(pv('configuration')); ?>">
            </div>
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'carpet_area')): ?>
            <div class="col-xl-4">
              <label class="form-label">Carpet Area</label>
              <input type="text" name="carpet_area" class="form-control" value="<?php echo h(pv('carpet_area')); ?>">
            </div>
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'price_text')): ?>
            <div class="col-xl-4">
              <label class="form-label">Price Text</label>
              <input type="text" name="price_text" class="form-control" value="<?php echo h(pv('price_text')); ?>" placeholder="₹ 60L - ₹ 1.2Cr">
            </div>
          <?php endif; ?>
          <?php if (col_exists($PROJECTS_TABLE,'price_min')): ?>
            <div class="col-xl-3">
              <label class="form-label">Price Min (numeric)</label>
              <input type="number" step="0.01" name="price_min" class="form-control" value="<?php echo h(pv('price_min')); ?>">
            </div>
          <?php endif; ?>

          <!-- New numeric fields (will save if present in DB) -->
          <div class="col-xl-2">
            <label class="form-label">Bedrooms</label>
            <input type="number" name="bedrooms" class="form-control" value="<?php echo h(pv('bedrooms')); ?>">
          </div>
          <div class="col-xl-2">
            <label class="form-label">Bathrooms</label>
            <input type="number" name="bathrooms" class="form-control" value="<?php echo h(pv('bathrooms')); ?>">
          </div>
          <div class="col-xl-2">
            <label class="form-label">Floors</label>
            <input type="number" name="floors" class="form-control" value="<?php echo h(pv('floors')); ?>">
          </div>
          <div class="col-xl-2">
            <label class="form-label">Area (sqft)</label>
            <input type="number" step="0.01" name="area_sqft" class="form-control" value="<?php echo h(pv('area_sqft')); ?>">
          </div>
          <div class="col-xl-3">
            <label class="form-label">Area Unit</label>
            <input type="text" name="area_unit" class="form-control" value="<?php echo h(pv('area_unit','ft2')); ?>">
          </div>

          <!-- WEBSITE / GEO / ORDER / ACTIVE -->
          <?php if (col_exists($PROJECTS_TABLE,'website_url')): ?>
            <div class="col-xl-6">
              <label class="form-label">Website URL</label>
              <input type="text" name="website_url" class="form-control" value="<?php echo h(pv('website_url')); ?>" placeholder="https://">
            </div>
          <?php endif; ?>
          <div class="col-xl-2">
            <label class="form-label">Latitude</label>
            <input type="text" name="lat" class="form-control" value="<?php echo h(pv('lat')); ?>">
          </div>
          <div class="col-xl-2">
            <label class="form-label">Longitude</label>
            <input type="text" name="lng" class="form-control" value="<?php echo h(pv('lng')); ?>">
          </div>
          <div class="col-xl-2">
            <label class="form-label">Sort Order</label>
            <input type="number" name="sort_order" class="form-control" value="<?php echo h(pv('sort_order',0)); ?>">
          </div>
          <div class="col-xl-12">
            <label class="form-check mt-2">
              <input type="checkbox" class="form-check-input" name="is_active" value="1" <?php echo (pv('is_active','1')=='1'?'checked':''); ?>>
              <span class="ms-1">Active</span>
            </label>
          </div>

          <div class="col-12"><hr></div>

          <!-- DESCRIPTION -->
          <div class="col-xl-12">
            <label class="form-label">Description</label>
            <textarea name="description" class="form-control" rows="4" placeholder="Project overview, highlights, nearby landmarks…"><?php echo h(pv('description')); ?></textarea>
          </div>

          <div class="col-12"><hr></div>

          <!-- AMENITIES (checkboxes + optional CSV field if exists) -->
          <div class="col-xl-12">
            <label class="form-label">Amenities & Features</label>
            <div class="d-flex flex-wrap gap-2">
              <?php
              // map label => db column (boolean tinyint 0/1)
              $AMENITIES_MASTER = [
                'Parking'         => 'amen_parking',
                'Swimming pool'   => 'amen_pool',
                'Balcony'         => 'amen_balcony',
                'Garden'          => 'amen_garden',
                'Security'        => 'amen_security',
                'Fitness center'  => 'amen_fitness',
                'Air Conditioning'=> 'amen_ac',
                'Laundry Room'    => 'amen_laundry',
                'Spa & Massage'   => 'amen_spa',
              ];
              foreach($AMENITIES_MASTER as $label=>$col):
                $checked = (pv($col,'0')=='1') ? 'checked' : '';
              ?>
                <label class="pill">
                  <input type="checkbox" class="form-check-input" name="<?php echo 'amen_'.$col; ?>" value="1" <?php echo $checked; ?>>
                  <span><?php echo h($label); ?></span>
                </label>
              <?php endforeach; ?>
            </div>
            <?php if (col_exists($PROJECTS_TABLE,'amenities')): ?>
              <div class="mt-2">
                <small class="muted">Additionally, CSV in <code>amenities</code> column (optional).</small>
                <input type="text" name="amenities" class="form-control mt-1" value="<?php echo h(pv('amenities')); ?>" placeholder="Pool,Gym,Park">
              </div>
            <?php endif; ?>
          </div>

          <div class="col-12"><hr></div>

          <!-- COVER IMAGE -->
          <div class="col-xl-12">
            <h6 class="mb-2">Cover Image</h6>
            <div class="row g-3 align-items-end">
              <div class="col-xl-4">
                <label class="form-label">Upload Cover</label>
                <input type="file" name="cover_image_upload" accept="image/*" class="form-control filepond-single">
              </div>
              <div class="col-xl-6">
                <label class="form-label">OR Cover Image URL</label>
                <input type="text" name="cover_image_url" class="form-control" value="<?php echo h(pv('cover_image')); ?>" placeholder="https://.../cover.webp">
              </div>
              <?php if (!empty($project['cover_image'])): ?>
                <div class="col-xl-2 text-center">
                  <img src="<?php echo h($project['cover_image']); ?>" class="thumb" alt="Cover">
                </div>
              <?php endif; ?>
            </div>
          </div>

          <div class="col-12"><hr></div>

          <!-- GALLERY IMAGES -->
          <div class="col-xl-12">
            <h6 class="mb-2 d-flex align-items-center justify-content-between">
              <span>Gallery Images</span>
            </h6>
            <input type="file" name="project_images[]" accept=".jpg,.jpeg,.png,.webp" class="form-control multiple-filepond" multiple>
            <small class="muted">Uploads go to <code>uploads/projects/{id}/images/</code> and are saved in <code>project_media</code> as <em>image</em>.</small>

            <?php if (!empty($existing['images'])): ?>
              <div class="thumb-grid mt-3">
                <?php foreach ($existing['images'] as $img): ?>
                  <div class="thumb-card">
                    <img src="<?php echo h($img['file_url']); ?>" class="w-100" style="height:110px; object-fit:cover;" alt="Image">
                    <div class="body">
                      <a class="btn btn-xs btn-outline-danger" href="project_media_delete.php?id=<?php echo (int)$img['id']; ?>&pid=<?php echo (int)$id; ?>" onclick="return confirm('Delete this image?')">
                        <i class="ri-delete-bin-line me-1"></i>Remove
                      </a>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>
            <?php endif; ?>
          </div>

          <div class="col-12"><hr></div>

          <!-- DOCUMENTS -->
          <div class="col-xl-12">
            <h6 class="mb-2">Documents</h6>
            <input type="file" name="project_docs[]" class="form-control" multiple>
            <small class="muted">Stored under <code>uploads/projects/{id}/docs/</code> (saved as <em>doc</em> in <code>project_media</code>).</small>
          </div>

          <div class="col-12"><hr></div>

          <!-- FLOOR PLANS (Multi-row, upload OR URL) -->
          <div class="col-xl-12">
            <div class="d-flex align-items-center justify-content-between">
              <h6 class="mb-2">Floor Plans</h6>
              <button class="btn btn-sm btn-primary" type="button" onclick="addPlanRow();"><i class="ri-add-line me-1"></i>Add Plan</button>
            </div>

            <!-- Existing floor plans (thumbs + remove) -->
            <?php if (!empty($existing['floors'])): ?>
              <div class="thumb-grid mb-3">
                <?php foreach ($existing['floors'] as $fp): ?>
                  <div class="thumb-card">
                    <img src="<?php echo h($fp['file_url']); ?>" class="w-100" style="height:110px; object-fit:cover;" alt="<?php echo h($fp['title'] ?? 'Floor Plan'); ?>">
                    <div class="body">
                      <div class="small text-truncate" title="<?php echo h($fp['title'] ?? ''); ?>"><?php echo h($fp['title'] ?? ''); ?></div>
                      <a class="btn btn-xs btn-outline-danger mt-1" href="project_media_delete.php?id=<?php echo (int)$fp['id']; ?>&pid=<?php echo (int)$id; ?>" onclick="return confirm('Delete this floor plan?')">
                        <i class="ri-delete-bin-line me-1"></i>Remove
                      </a>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>
            <?php endif; ?>

            <!-- Add / Edit rows -->
            <div id="floorplans-wrap">
              <!-- Template (cloned by JS) -->
              <template id="tpl-plan-row">
                <div class="fp-row">
                  <div class="row g-2 align-items-end">
                    <div class="col-md-4">
                      <label class="form-label">Plan Title</label>
                      <input type="text" name="plan_title[]" class="form-control" placeholder="2 BHK - Type 1">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Bedrooms</label>
                      <input type="number" name="plan_bedrooms[]" class="form-control" placeholder="2">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Bathrooms</label>
                      <input type="number" name="plan_bathrooms[]" class="form-control" placeholder="2">
                    </div>
                    <div class="col-md-2">
                      <label class="form-label">Sort</label>
                      <input type="number" name="plan_sort[]" class="form-control" placeholder="1">
                    </div>
                    <div class="col-md-6">
                      <label class="form-label">Notes (pipe-separated)</label>
                      <input type="text" name="plan_notes[]" class="form-control" placeholder="Living|Kitchen|Balcony">
                    </div>
                    <div class="col-md-3">
                      <label class="form-label">Image (upload)</label>
                      <input type="file" name="plan_image_file[]" accept="image/*" class="form-control">
                    </div>
                    <div class="col-md-3">
                      <label class="form-label">OR Image URL</label>
                      <input type="text" name="plan_image_url[]" class="form-control" placeholder="https://...">
                    </div>
                    <div class="col-md-12 text-end">
                      <button class="btn btn-sm btn-outline-danger" type="button" onclick="this.closest('.fp-row').remove();"><i class="ri-delete-bin-line me-1"></i>Remove Row</button>
                    </div>
                  </div>
                </div>
              </template>
            </div>
          </div>

          <div class="col-12"><hr></div>

          <div class="card-footer bg-transparent px-0">
            <button class="btn btn-primary-light btn-wave ms-auto float-end" type="submit">
              <i class="ri-save-3-line me-1"></i> <?php echo $id ? 'Update Project' : 'Create Project'; ?>
            </button>
          </div>

        </form>
      </div>
    </div>
  </div>
</div>

<?php $content = ob_get_clean();

/* ========== $scripts ========== */
ob_start(); ?>
<script src="<?php echo base_url(); ?>/assets/libs/filepond/filepond.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-preview/filepond-plugin-image-preview.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-exif-orientation/filepond-plugin-image-exif-orientation.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-file-validate-size/filepond-plugin-file-validate-size.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-file-validate-type/filepond-plugin-file-validate-type.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-crop/filepond-plugin-image-crop.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-resize/filepond-plugin-image-resize.min.js"></script>
<script src="<?php echo base_url(); ?>/assets/libs/filepond-plugin-image-transform/filepond-plugin-image-transform.min.js"></script>

<script>
/* FilePond init */
FilePond.registerPlugin(
  FilePondPluginImagePreview,
  FilePondPluginImageExifOrientation,
  FilePondPluginFileValidateSize,
  FilePondPluginFileValidateType,
  FilePondPluginImageCrop,
  FilePondPluginImageResize,
  FilePondPluginImageTransform
);
document.querySelectorAll('.multiple-filepond').forEach(function(input){
  FilePond.create(input, { allowMultiple:true, storeAsFile:true });
});
document.querySelectorAll('.filepond-single').forEach(function(input){
  FilePond.create(input, { allowMultiple:false, storeAsFile:true });
});

/* Floorplan rows */
function addPlanRow(){
  var tpl = document.getElementById('tpl-plan-row');
  if (!tpl) return;
  var node = tpl.content.cloneNode(true);
  document.getElementById('floorplans-wrap').appendChild(node);
}
</script>
<?php $scripts = ob_get_clean();

/* ----------------------------- Render base ----------------------------- */
$baseLayout = __DIR__ . '/layouts/base.php';
if (file_exists($baseLayout)) {
  include $baseLayout;
} else {
  // Minimal fallback if Melody base missing
  ?><!doctype html><html><head>
  <meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Project Form</title><?php echo $styles; ?></head><body>
  <div class="container p-3"><?php echo $content; ?></div>
  <?php echo $scripts; ?></body></html><?php
}
