<?php
/* =============================================================================
 * File   : project_save.php
 * Role   : Handle Create / Update from project_form.php
 * DB     : MySQLi only (NO PDO), utf8mb4_general_ci
 * Media  : /uploads/projects/{id}/images , /docs
 * Notes  : - Writes floor plans into project_media (media_type='floor')
 *          - Creates/adjusts project_media if needed (adds 'floor' to ENUM)
 *          - Only reads/saves columns that exist in the DB
 * Redirects back to project_form.php with flash message.
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ------------------------- Centralized config ------------------------- */
$cfgs = [
  __DIR__ . '/../config.php',
  __DIR__ . '/../../config.php',
  __DIR__ . '/config.php'
];
$cfgFound = false;
foreach ($cfgs as $c) {
  if (file_exists($c)) { require_once $c; $cfgFound = true; break; }
}
if (!$cfgFound) { http_response_code(500); die('config.php not found'); }

/* ------------------------- Pick mysqli handle ------------------------- */
$DB = null;
if (isset($conn_crm)     && $conn_crm     instanceof mysqli) $DB = $conn_crm;
elseif (isset($conn_crmplus) && $conn_crmplus instanceof mysqli) $DB = $conn_crmplus;
elseif (isset($conn)     && $conn     instanceof mysqli) $DB = $conn;
elseif (isset($conn_sa)  && $conn_sa  instanceof mysqli) $DB = $conn_sa;

if (!$DB instanceof mysqli) { http_response_code(500); exit('DB connection not found - check config.php'); }

/* Charset / Collation */
@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* ------------------------- Helpers ------------------------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); }
function q($s){ global $DB; return mysqli_real_escape_string($DB,(string)$s); }
function base_url(){
  if (!empty($GLOBALS['baseUrl'])) return rtrim($GLOBALS['baseUrl'],'/');
  $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS']!=='off') ? 'https' : 'http';
  $host  = $_SERVER['HTTP_HOST'] ?? ($_SERVER['SERVER_NAME'] ?? 'localhost');
  $dir   = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? '/'),'/');
  return rtrim("$proto://$host$dir",'/');
}
function ensure_dir($path){
  if(!is_dir($path)) @mkdir($path, 0775, true);
  @chmod($path, 0775);
  return is_dir($path) && is_writable($path);
}
function table_exists($name){ global $DB; $name = q($name); $r = @mysqli_query($DB,"SHOW TABLES LIKE '{$name}'"); $ok = ($r && mysqli_num_rows($r)>0); if($r) @mysqli_free_result($r); return $ok; }
function col_exists($tbl,$col){ global $DB; $tbl = q($tbl); $col = q($col); $r = @mysqli_query($DB,"SHOW COLUMNS FROM `{$tbl}` LIKE '{$col}'"); $ok = ($r && mysqli_num_rows($r)>0); if($r) @mysqli_free_result($r); return $ok; }

/* Save ONE uploaded file and return URL (or '') */
function save_one_upload($field, $projectId, $subdir){
  if (empty($_FILES[$field]) || !is_uploaded_file($_FILES[$field]['tmp_name'])) return '';
  if ((int)$_FILES[$field]['error'] !== UPLOAD_ERR_OK) return '';
  $root = __DIR__ . "/uploads/projects/".intval($projectId)."/".trim($subdir,'/');
  if (!ensure_dir($root)) return '';
  $name = preg_replace('~[^A-Za-z0-9._-]+~','-', $_FILES[$field]['name']);
  $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  $fname = uniqid(date('Ymd').'-', true) . ($ext?'.'.$ext:'');
  $dest = $root . "/".$fname;
  if (@move_uploaded_file($_FILES[$field]['tmp_name'], $dest)) {
    @chmod($dest, 0644);
    return rtrim(base_url(),'/').'/uploads/projects/'.intval($projectId).'/'.trim($subdir,'/').'/'.$fname;
  }
  return '';
}

/* Save MULTIPLE uploads (array field) and return list of URLs */
function save_multi_uploads($field, $projectId, $subdir){
  $out = [];
  if (empty($_FILES[$field]) || !is_array($_FILES[$field]['name'])) return $out;
  $root = __DIR__ . "/uploads/projects/".intval($projectId)."/".trim($subdir,'/');
  if (!ensure_dir($root)) return $out;
  $count = count($_FILES[$field]['name']);
  for ($i=0; $i<$count; $i++){
    if (!isset($_FILES[$field]['error'][$i]) || $_FILES[$field]['error'][$i] !== UPLOAD_ERR_OK) continue;
    $tmp  = $_FILES[$field]['tmp_name'][$i];
    if (!is_uploaded_file($tmp)) continue;
    $name = preg_replace('~[^A-Za-z0-9._-]+~','-', $_FILES[$field]['name'][$i]);
    $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    $fname= uniqid(date('Ymd').'-', true) . ($ext?'.'.$ext:'');
    $dest = $root . "/".$fname;
    if (@move_uploaded_file($tmp, $dest)) {
      @chmod($dest, 0644);
      $out[] = rtrim(base_url(),'/').'/uploads/projects/'.intval($projectId).'/'.trim($subdir,'/').'/'.$fname;
    }
  }
  return $out;
}

/* ------------------------- Resolve projects/media table -------------------- */
$PROJECTS_TABLE = null;
foreach (['projects','project','tbl_projects','pp_projects'] as $t) {
  if (table_exists($t)) { $PROJECTS_TABLE = $t; break; }
}
if (!$PROJECTS_TABLE) { $_SESSION['flash_error'] = 'Projects table not found.'; header('Location: project_form.php'); exit; }

/* Ensure project_media exists; if enum missing "floor", add it. */
if (!table_exists('project_media')) {
  @mysqli_query($DB, "CREATE TABLE IF NOT EXISTS `project_media` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `project_id` INT UNSIGNED NOT NULL,
    `media_type` ENUM('image','doc','link','floor') COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'image',
    `title` VARCHAR(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
    `url` VARCHAR(1024) COLLATE utf8mb4_general_ci NOT NULL,
    `sort_order` INT DEFAULT 0,
    `is_active` TINYINT(1) NOT NULL DEFAULT 1,
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_pid` (`project_id`)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;");
} else {
  // Add 'floor' to enum if needed
  if ($rc = @mysqli_query($DB, "SHOW COLUMNS FROM `project_media` LIKE 'media_type'")) {
    if ($row = mysqli_fetch_assoc($rc)) {
      $type = strtolower((string)$row['Type']);
      if (strpos($type, "'floor'") === false) {
        @mysqli_query($DB, "ALTER TABLE `project_media`
          MODIFY `media_type` ENUM('image','doc','link','floor')
          COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'image'");
      }
    }
    @mysqli_free_result($rc);
  }
}

/* ------------------------- Read inputs ------------------------- */
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;

/* Map posted -> DB fields (only save if the column exists) */
$posted = [
  'title'         => trim($_POST['title'] ?? ''),
  'city'          => trim($_POST['city'] ?? ''),
  'address'       => trim($_POST['address'] ?? ''),
  'configuration' => trim($_POST['configuration'] ?? ''),
  'carpet_area'   => trim($_POST['carpet_area'] ?? ''),
  'price_text'    => trim($_POST['price_text'] ?? ''),
  'price_min'     => trim($_POST['price_min'] ?? ''),
  'bedrooms'      => trim($_POST['bedrooms'] ?? ''),
  'bathrooms'     => trim($_POST['bathrooms'] ?? ''),
  'floors'        => trim($_POST['floors'] ?? ''),
  'area_sqft'     => trim($_POST['area_sqft'] ?? ''),
  'area_unit'     => trim($_POST['area_unit'] ?? ''),
  'website_url'   => trim($_POST['website_url'] ?? ''),
  'lat'           => trim($_POST['lat'] ?? ''),
  'lng'           => trim($_POST['lng'] ?? ''),
  'sort_order'    => trim($_POST['sort_order'] ?? ''),
  'is_active'     => isset($_POST['is_active']) ? 1 : 0,
];

/* Description: save to description if exists, else description_long if that exists */
$desc = trim($_POST['description'] ?? '');
$descCol = null;
if (col_exists($PROJECTS_TABLE,'description'))      $descCol = 'description';
elseif (col_exists($PROJECTS_TABLE,'description_long')) $descCol = 'description_long';
if ($descCol !== null) { $posted[$descCol] = $desc; }

/* Optional amenities CSV if exists */
if (col_exists($PROJECTS_TABLE,'amenities')) {
  $posted['amenities'] = trim($_POST['amenities'] ?? '');
}

/* Amenity toggles (robust to name="amen_parking" or "amen_amen_parking") */
$AMENITIES_MASTER = [
  'amen_parking',
  'amen_pool',
  'amen_balcony',
  'amen_garden',
  'amen_security',
  'amen_fitness',
  'amen_ac',
  'amen_laundry',
  'amen_spa',
];
foreach ($AMENITIES_MASTER as $acol) {
  if (!col_exists($PROJECTS_TABLE,$acol)) continue;
  $k1 = $acol;                // e.g., amen_parking
  $k2 = 'amen_'.$acol;        // form had name="amen_amen_parking" (defensive)
  $val = (isset($_POST[$k1]) || isset($_POST[$k2])) ? 1 : 0;
  $posted[$acol] = $val;
}

/* ------------------------- Build SQL for insert/update --------------------- */
$setParts = [];
$cols = []; $vals = [];

foreach ($posted as $k=>$v) {
  if (!col_exists($PROJECTS_TABLE,$k)) continue;

  if (in_array($k, ['price_min','area_sqft'], true)) {
    $expr = ($v!=='' && is_numeric($v)) ? (string)((float)$v) : "NULL";
  } elseif (in_array($k, ['bedrooms','bathrooms','floors','sort_order'], true)) {
    $expr = ($v!=='' ? (string)intval($v) : "NULL");
  } elseif ($k === 'is_active') {
    $expr = $v ? "1" : "0";
  } else {
    $expr = "'".q($v)."'";
  }

  if ($id > 0) {
    $setParts[] = "`$k` = $expr";
  } else {
    $cols[] = "`$k`";
    $vals[] = $expr;
  }
}

/* Must have title on insert */
if ($id === 0 && !in_array('`title`', $cols, true)) {
  $_SESSION['flash_error'] = 'Title is required.';
  header('Location: project_form.php'); exit;
}

/* Do write */
if ($id > 0) {
  if ($setParts) {
    $sql = "UPDATE `{$PROJECTS_TABLE}` SET ".implode(',', $setParts)." WHERE id={$id} LIMIT 1";
    if (!@mysqli_query($DB,$sql)) {
      $_SESSION['flash_error'] = 'Update failed: '.h(mysqli_error($DB));
      header('Location: project_form.php?id='.$id); exit;
    }
  }
  $pid = $id;
} else {
  $sql = "INSERT INTO `{$PROJECTS_TABLE}` (".implode(',', $cols).") VALUES (".implode(',', $vals).")";
  if (!@mysqli_query($DB,$sql)) {
    $_SESSION['flash_error'] = 'Insert failed: '.h(mysqli_error($DB));
    header('Location: project_form.php'); exit;
  }
  $pid = (int)mysqli_insert_id($DB);
}

/* ------------------------- Cover image (URL or upload) --------------------- */
if (col_exists($PROJECTS_TABLE,'cover_image')) {
  $coverUrl = trim($_POST['cover_image_url'] ?? '');
  if ($coverUrl !== '') {
    @mysqli_query($DB, "UPDATE `{$PROJECTS_TABLE}` SET `cover_image`='".q($coverUrl)."' WHERE id={$pid} LIMIT 1");
  } else {
    $coverUpload = save_one_upload('cover_image_upload', $pid, 'images');
    if ($coverUpload !== '') {
      @mysqli_query($DB, "UPDATE `{$PROJECTS_TABLE}` SET `cover_image`='".q($coverUpload)."' WHERE id={$pid} LIMIT 1");
    }
  }
}

/* ------------------------- Gallery images -> project_media ------------------ */
if (table_exists('project_media')) {
  $imgUrls = save_multi_uploads('project_images', $pid, 'images'); // from form field
  $sort = 10;
  foreach ($imgUrls as $u) {
    $ttl = basename(parse_url($u, PHP_URL_PATH)) ?: 'Image';
    @mysqli_query($DB, "INSERT INTO `project_media`(project_id,media_type,title,`url`,sort_order,is_active)
                        VALUES ($pid,'image','".q($ttl)."','".q($u)."',".$sort.",1)");
    $sort++;
  }
}

/* ------------------------- Documents -> project_media ---------------------- */
if (table_exists('project_media')) {
  $docUrls = save_multi_uploads('project_docs', $pid, 'docs'); // from form field
  $sort = 100;
  foreach ($docUrls as $u) {
    $ttl = basename(parse_url($u, PHP_URL_PATH)) ?: 'Document';
    @mysqli_query($DB, "INSERT INTO `project_media`(project_id,media_type,title,`url`,sort_order,is_active)
                        VALUES ($pid,'doc','".q($ttl)."','".q($u)."',".$sort.",1)");
    $sort++;
  }
}

/* ------------------------- Floor plans -> project_media (media_type=floor) - */
if (table_exists('project_media')) {
  // Gather arrays; may be missing if no rows were added
  $plan_titles = isset($_POST['plan_title']) ? (array)$_POST['plan_title'] : [];
  $plan_beds   = isset($_POST['plan_bedrooms']) ? (array)$_POST['plan_bedrooms'] : [];
  $plan_baths  = isset($_POST['plan_bathrooms']) ? (array)$_POST['plan_bathrooms'] : [];
  $plan_sort   = isset($_POST['plan_sort']) ? (array)$_POST['plan_sort'] : [];
  $plan_notes  = isset($_POST['plan_notes']) ? (array)$_POST['plan_notes'] : [];
  $plan_urls   = isset($_POST['plan_image_url']) ? (array)$_POST['plan_image_url'] : [];

  // Handle uploaded images; map index => url
  $upload_map = [];
  if (!empty($_FILES['plan_image_file']) && is_array($_FILES['plan_image_file']['name'])) {
    // emulate save_multi_uploads but keep indices
    $root = __DIR__ . "/uploads/projects/".intval($pid)."/images";
    ensure_dir($root);
    $count = count($_FILES['plan_image_file']['name']);
    for ($i=0; $i<$count; $i++){
      if (!isset($_FILES['plan_image_file']['error'][$i]) || $_FILES['plan_image_file']['error'][$i] !== UPLOAD_ERR_OK) continue;
      $tmp  = $_FILES['plan_image_file']['tmp_name'][$i];
      if (!is_uploaded_file($tmp)) continue;
      $name = preg_replace('~[^A-Za-z0-9._-]+~','-', $_FILES['plan_image_file']['name'][$i]);
      $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));
      $fname= uniqid('plan-'.date('Ymd').'-', true) . ($ext?'.'.$ext:'');
      $dest = $root . "/".$fname;
      if (@move_uploaded_file($tmp, $dest)) {
        @chmod($dest, 0644);
        $upload_map[$i] = rtrim(base_url(),'/').'/uploads/projects/'.intval($pid).'/images/'.$fname;
      }
    }
  }

  // feature columns in project_media (optional)
  $hasDesc  = col_exists('project_media','description');
  $hasBeds  = col_exists('project_media','bedrooms');
  $hasBaths = col_exists('project_media','bathrooms');

  $rows = max(count($plan_titles), count($plan_beds), count($plan_baths), count($plan_sort), count($plan_notes), count($plan_urls), count($upload_map));
  for ($i=0; $i<$rows; $i++){
    $title = trim($plan_titles[$i] ?? '');
    $beds  = trim($plan_beds[$i] ?? '');
    $baths = trim($plan_baths[$i] ?? '');
    $notes = trim($plan_notes[$i] ?? '');
    $sort  = (int)($plan_sort[$i] ?? (300 + $i));
    $url   = trim($plan_urls[$i] ?? '');
    if (isset($upload_map[$i]) && $upload_map[$i] !== '') $url = $upload_map[$i];

    if ($title==='' && $url==='') continue;

    // Build insert, with optional cols
    $cols = "project_id, media_type, title, `url`, sort_order, is_active";
    $vals = "$pid, 'floor', '".q($title ?: 'Floor Plan')."', '".q($url)."', $sort, 1";
    if ($hasDesc)  { $cols .= ", description"; $vals .= ", ".($notes!=='' ? "'".q($notes)."'" : "NULL"); }
    if ($hasBeds)  { $cols .= ", bedrooms";    $vals .= ", ".($beds!=='' ? (int)$beds : "NULL"); }
    if ($hasBaths) { $cols .= ", bathrooms";   $vals .= ", ".($baths!=='' ? (int)$baths : "NULL"); }

    @mysqli_query($DB, "INSERT INTO `project_media` ($cols) VALUES ($vals)");
  }
}

/* ------------------------- Redirect back with flash ------------------------ */
$_SESSION['flash_success'] = ($id>0) ? 'Project updated successfully.' : 'Project created successfully.';
header('Location: project_form.php?id='.$pid);
exit;
