<?php
/* =============================================================================
 * File   : /brocrm/mobile/footer_mobile.php
 * Goal   : Mobile Bottom Dock with role-aware redirects (AMS/CRM/FAB)
 * Stack  : Core PHP + MySQLi (NO PDO)
 * DB     : Connection provided by config.php; force utf8mb4_general_ci if present
 * Rules  :
 *   - If NOT logged in  -> AMS links to grid-layout-with-sidebar.php (label = "Property Search")
 *   - If role_id = 4    -> AMS -> /brocrm/ams/user/dashboard.php (label = "AMS"); CRM -> dst.php; FAB -> /brocrm/mobile/broker_onboarding.php
 *   - If role_id = 6    -> CRM -> crm_mobile.php
 *   - Others/default    -> CRM defaults; AMS -> grid-layout-with-sidebar.php (label = "Property Search")
 * Notes  : Uses the same session structure as dst.php/footer_mobile.php ($_SESSION['auth'])
 * ============================================================================= */

if (session_status() === PHP_SESSION_NONE) { session_start(); }
date_default_timezone_set('Asia/Kolkata');

/* ------------------------- Centralized config ------------------------- */
$cfgs = [
  __DIR__ . '/config.php',
  __DIR__ . '/../config.php',
  __DIR__ . '/../../config.php'
];
$cfgFound = false;
foreach ($cfgs as $c) { if (file_exists($c)) { require_once $c; $cfgFound = true; break; } }

/* ------------------------------ DB collation ------------------------------ */
$DB = null;
if (isset($conn) && $conn instanceof mysqli)                    { $DB = $conn; }
elseif (isset($conn_crm) && $conn_crm instanceof mysqli)         { $DB = $conn_crm; }
elseif (isset($conn_crmplus) && $conn_crmplus instanceof mysqli) { $DB = $conn_crmplus; }
if ($DB instanceof mysqli) {
  @mysqli_query($DB, "SET NAMES 'utf8mb4' COLLATE 'utf8mb4_general_ci'");
  @mysqli_set_charset($DB, 'utf8mb4');
  @mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");
}

/* --------------------------- Current user/session -------------------------- */
$ME        = $_SESSION['auth'] ?? [];
$ROLE_ID   = isset($ME['role_id']) ? (int)$ME['role_id'] : 0;
$ROLE_NAME = strtolower($ME['role_name'] ?? 'user');
$ME_ID     = (int)($ME['id'] ?? 0);
$ME_MAIL   = trim((string)($ME['email'] ?? ''));
$ME_EMPID  = trim((string)($ME['empid'] ?? ''));

$isLogged = (($ME_ID > 0 || $ME_MAIL !== '' || $ME_EMPID !== '')&& in_array($ROLE_ID, [2, 6]));

/* ------------------------------ URL helpers -------------------------------- */
if (!isset($baseUrl)) { $baseUrl = ''; } // avoid undefined in Home link

// AMS link rules
$AMS_DESKTOP = $AMS_MOBILE = '';
if (!$isLogged) {
  $AMS_DESKTOP = $AMS_MOBILE = '/mobile/grid-layout-with-sidebar.php';
} else {
  if ($ROLE_ID === 4) {
    $AMS_DESKTOP = $AMS_MOBILE = '/ams/user/dashboard.php';
  } else {
    $AMS_DESKTOP = $AMS_MOBILE = '/mobile/grid-layout-with-sidebar.php';
  }
}

// Derive AMS visible label (if points to grid-layout-with-sidebar.php => "Property Search")
$gridSlug   = 'grid-layout-with-sidebar.php';
$AMS_LABEL  = (strpos($AMS_DESKTOP, $gridSlug) !== false || strpos($AMS_MOBILE, $gridSlug) !== false)
              ? 'Property Search'
              : 'AMS';

// CRM link rules
$CRM_DESKTOP = '/crmplus/pages/dst-leads.php';
$CRM_MOBILE  = 'dst.php';
if (!$isLogged) {
  $CRM_DESKTOP = $CRM_MOBILE = '/mobile/login.php';
} else {
  if ($ROLE_ID === 4) {
    $CRM_DESKTOP = $CRM_MOBILE = 'dst.php';
  } elseif ($ROLE_ID === 6) {
    $CRM_DESKTOP = $CRM_MOBILE = 'crm_mobile.php';
  } else {
    $CRM_DESKTOP = '/crmplus/pages/dst-leads.php';
    $CRM_MOBILE  = 'dst.php';
  }
}

// FAB link rules
$FAB_DESKTOP = $FAB_MOBILE = '/crmplus/pages/crm_add.php';
if ($ROLE_ID === 4) {
  $FAB_DESKTOP = $FAB_MOBILE = '/mobile/broker_onboarding.php';
}
?>
<?php if ($isLogged): ?>
<!-- Mobile Bottom Dock (modern, colorful) -->
<nav class="mobile-dock d-md-none" role="navigation" aria-label="Mobile dock">
  <!-- Home -->
  <a class="dock-item smart-link"
     href="index.php"
     data-href-desktop="<?php echo htmlspecialchars($baseUrl, ENT_QUOTES); ?>/"
     data-href-mobile="index.php"
     aria-label="Home" title="Home">
    <svg viewBox="0 0 24 24" width="26" height="26" aria-hidden="true">
      <path d="M3 10.5 12 3l9 7.5v8.25A1.25 1.25 0 0 1 19.75 20H16v-5a2 2 0 0 0-2-2h-4a2 2 0 0 0-2 2v5H4.25A1.25 1.25 0 0 1 3 18.75Z"/>
    </svg>
    <span class="dock-label">Home</span>
  </a>

  <!-- AMS (role-aware, label toggles to "Property Search" when pointing to the grid page) -->
  <a class="dock-item dock-ams smart-link"
     href="<?php echo htmlspecialchars($AMS_MOBILE, ENT_QUOTES); ?>"
     data-href-desktop="<?php echo htmlspecialchars($AMS_DESKTOP, ENT_QUOTES); ?>"
     data-href-mobile="<?php echo htmlspecialchars($AMS_MOBILE, ENT_QUOTES); ?>"
     aria-label="<?php echo htmlspecialchars($AMS_LABEL, ENT_QUOTES); ?>"
     title="<?php echo htmlspecialchars($AMS_LABEL, ENT_QUOTES); ?>">
    <svg viewBox="0 0 24 24" width="26" height="26" aria-hidden="true">
      <circle cx="12" cy="12" r="9" fill="none" stroke="currentColor" stroke-width="2"/>
      <path d="M12 7v5l3 2.2" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" fill="none"/>
    </svg>
    <span class="dock-label"><?php echo htmlspecialchars($AMS_LABEL, ENT_QUOTES); ?></span>
  </a>

  <!-- Spacer under center FAB (keeps 5-column balance) -->
  <div class="dock-item dock-spacer" aria-hidden="true"></div>

  <!-- DST/CRM (role-aware) -->
  <a class="dock-item smart-link"
     href="<?php echo htmlspecialchars($CRM_MOBILE, ENT_QUOTES); ?>"
     data-href-desktop="<?php echo htmlspecialchars($CRM_DESKTOP, ENT_QUOTES); ?>"
     data-href-mobile="<?php echo htmlspecialchars($CRM_MOBILE, ENT_QUOTES); ?>"
     aria-label="CRM" title="CRM">
    <svg viewBox="0 0 24 24" width="26" height="26" aria-hidden="true">
      <path d="M21 12a8.5 8.5 0 0 1-8.5 8.5c-1.25 0-2.44-.24-3.52-.68L3 21l1.18-4.03A8.5 8.5 0 1 1 21 12Z"/>
      <path d="M7.5 11.5h9M7.5 8.5h9M7.5 14.5h6" fill="none" stroke="currentColor" stroke-width="1.7" stroke-linecap="round"/>
    </svg>
    <span class="dock-label">CRM</span>
  </a>

  <!-- Profile -->
  <a class="dock-item smart-link"
     href="user-add.php"
     data-href-desktop="/user-add.php"
     data-href-mobile="user-add.php"
     aria-label="Profile" title="Profile">
    <svg viewBox="0 0 24 24" width="26" height="26" aria-hidden="true">
      <circle cx="12" cy="8" r="4"></circle>
      <path d="M5 20a7 7 0 0 1 14 0"/>
    </svg>
    <span class="dock-label">Profile</span>
  </a>

  <!-- Center FAB (+ Add) (role-aware) -->
  <a class="fab-add smart-link"
     href="<?php echo htmlspecialchars($FAB_MOBILE, ENT_QUOTES); ?>"
     data-href-desktop="<?php echo htmlspecialchars($FAB_DESKTOP, ENT_QUOTES); ?>"
     data-href-mobile="<?php echo htmlspecialchars($FAB_MOBILE, ENT_QUOTES); ?>"
     aria-label="Add" title="Add">
    <svg viewBox="0 0 24 24" width="26" height="26" aria-hidden="true">
      <path d="M12 5v14M5 12h14" stroke="#fff" stroke-width="2" stroke-linecap="round"/>
    </svg>
  </a>
</nav>

<style>
  :root{
    --dock-bg: rgba(255,255,255,0.9);
    --dock-border: rgba(0,0,0,0.08);
    --dock-shadow: 0 10px 30px rgba(0,0,0,0.12);
    --brand-primary:#2563eb;   /* Home icon */
    --brand-success:#16a34a;   /* AMS icon */
    --brand-amber:#f59e0b;     /* DST icon */
    --brand-cyan:#06b6d4;      /* Profile icon */
    --text-muted:#6b7280;
  }
  @media (prefers-color-scheme: dark){
    :root{
      --dock-bg: rgba(20,20,24,0.9);
      --dock-border: rgba(255,255,255,0.08);
      --dock-shadow: 0 10px 30px rgba(0,0,0,0.55);
      --text-muted: #a3a3a3;
    }
  }

  .mobile-dock{
    position: fixed; left: 0; right: 0; bottom: 0;
    z-index: 10050; /* above fireworks (9998/9999) */
    display: grid; grid-template-columns: repeat(5,1fr);
    align-items: end; gap: .25rem;
    padding: .4rem .6rem max(.4rem, env(safe-area-inset-bottom));
    background: var(--dock-bg); backdrop-filter: blur(12px);
    border-top: 1px solid var(--dock-border); box-shadow: var(--dock-shadow);
  }

  .dock-item{
    display: grid; place-items: center; gap: .15rem;
    min-height: 58px; padding: .35rem 0 .25rem;
    text-decoration: none; border-radius: 14px;
    color: var(--text-muted);
    font: 600 11px/1 system-ui,-apple-system,Segoe UI,Roboto,Arial;
    transition: transform .15s ease, color .15s ease;
    -webkit-tap-highlight-color: transparent;
  }
  .dock-item:active{ transform: translateY(1px) scale(.98); }
  .dock-item svg{ display:block; fill: currentColor; opacity:.95; }
  .dock-label{ line-height: 1; }

  /* Icon colors (only icons, not tiles) */
  .mobile-dock .dock-item:nth-child(1) svg{ color: var(--brand-primary);} /* Home */
  .dock-ams svg{ color: var(--brand-success);}                           /* AMS / Property Search */
  .mobile-dock .dock-item:nth-child(4) svg{ color: var(--brand-amber);}   /* DST/CRM */
  .mobile-dock .dock-item:nth-child(5) svg{ color: var(--brand-cyan);}    /* Profile */

  .dock-spacer{ opacity: 0; pointer-events: none; }

  /* Floating Action Button (+) */
  .fab-add{
    position: absolute; left: 50%; bottom: 36px; transform: translateX(-50%);
    width: 56px; height: 56px; border-radius: 50%;
    display: grid; place-items: center;
    background: linear-gradient(135deg, #4f46e5, #22d3ee);
    box-shadow: 0 12px 22px rgba(79,70,229,.35);
    text-decoration: none;
  }
  .fab-add svg{ display:block; }
</style>

<script>
/* Resolution-aware links (desktop vs mobile) */
(function () {
  const mq = matchMedia('(max-width: 767.98px)');
  function applySmartLinks(){
    const isMobile = mq.matches;
    document.querySelectorAll('a.smart-link, button.smart-link').forEach(el=>{
      const href = isMobile ? (el.dataset.hrefMobile || el.getAttribute('href'))
                            : (el.dataset.hrefDesktop || el.getAttribute('href'));
      if (href && el.tagName === 'A') el.setAttribute('href', href);
      const label = isMobile ? el.dataset.labelMobile : el.dataset.labelDesktop;
      if (label) {
        const t = el.querySelector('.dock-label,.link-text,.nav-text'); if (t) t.textContent = label;
        el.setAttribute('aria-label', label); el.title = label;
      }
    });
  }
  applySmartLinks();
  mq.addEventListener?.('change', applySmartLinks);
  addEventListener('resize', applySmartLinks);
  addEventListener('orientationchange', applySmartLinks);
})();
</script>
<?php endif; ?>