<?php
/* =============================================================================
 * File   : home.php
 * Goal   : Combined homepage using Base/style/base.php; "Property For Rent"
 *          section shows ALL Projects (no rent-only filter) with UNIFORM CARD
 *          HEIGHTS (image+text+box same size for every slide/card).
 * Rules  : - Database connection is in config.php (MySQLi; NO PDO)
 *          - DB charset/collation set to utf8mb4 / utf8mb4_general_ci
 *          - Full file from 1st line to last line
 * ============================================================================= */

@session_start();

/* ------------------------- Centralized config ------------------------- */
$cfgs = [
  __DIR__ . '/config.php',
  __DIR__ . '/../config.php',
  __DIR__ . '/../../config.php'
];
$cfgFound = false;
foreach ($cfgs as $c) {
  if (file_exists($c)) { require_once $c; $cfgFound = true; break; }
}
if (!$cfgFound) {
  http_response_code(500);
  die('config.php not found');
}

/* -------------------------- Pick a mysqli link ------------------------ */
$DB = null;
if (isset($conn_crm)        && $conn_crm instanceof mysqli)        $DB = $conn_crm;
elseif (isset($conn_crmplus) && $conn_crmplus instanceof mysqli)   $DB = $conn_crmplus;
elseif (isset($conn)         && $conn instanceof mysqli)           $DB = $conn;
elseif (isset($conn_sa)      && $conn_sa instanceof mysqli)        $DB = $conn_sa;

if (!$DB instanceof mysqli) { http_response_code(500); exit('DB connection not found - check config.php'); }

/* ---------------------- Charset / Collation (required) ---------------- */
@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* ---------------------------- Helpers --------------------------------- */
if (!function_exists('h')) { function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); } }
function q($s){ global $DB; return mysqli_real_escape_string($DB,(string)$s); }
function base_url(){
  if (!empty($GLOBALS['baseUrl'])) return rtrim($GLOBALS['baseUrl'],'/');
  $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS']!=='off') ? 'https' : 'http';
  $host  = $_SERVER['HTTP_HOST'] ?? ($_SERVER['SERVER_NAME'] ?? 'localhost');
  $dir   = rtrim(dirname($_SERVER['SCRIPT_NAME'] ?? '/'),'/');
  return rtrim("$proto://$host$dir",'/');
}
function table_exists($name){ global $DB; $name = q($name); $r = @mysqli_query($DB,"SHOW TABLES LIKE '{$name}'"); $ok = ($r && mysqli_num_rows($r)>0); if($r) @mysqli_free_result($r); return $ok; }
function col_exists($tbl,$col){ global $DB; $tbl = q($tbl); $col = q($col); $r = @mysqli_query($DB,"SHOW COLUMNS FROM `{$tbl}` LIKE '{$col}'"); $ok = ($r && mysqli_num_rows($r)>0); if($r) @mysqli_free_result($r); return $ok; }
function full_url_from($path){
  $p = trim((string)$path);
  if ($p === '') return $p;
  if (preg_match('~^https?://~i', $p)) return $p;
  if ($p[0] === '/') return $p;                    // already absolute to domain root
  return base_url().'/'.$p;                        // relative -> make absolute
}
function inr_format($num){
  $n = (float)$num;
  if ($n >= 10000000) return '₹ '.round($n/10000000,2).' Cr';
  if ($n >= 100000)   return '₹ '.round($n/100000,2).' L';
  return '₹ '.number_format($n, 0);
}

/* ------------------- Resolve Projects / Media tables ------------------- */
$PROJECTS_TABLE = null;
foreach (['projects','project','tbl_projects','pp_projects'] as $t) {
  if (table_exists($t)) { $PROJECTS_TABLE = $t; break; }
}
$MEDIA_TABLE = table_exists('project_media') ? 'project_media' : null;
if (!$PROJECTS_TABLE) { http_response_code(500); exit('Projects table not found.'); }

/* ------------------------------ Paths ---------------------------------- */
$base_dir   = __DIR__ . '/Base';
$static_url = '/Resido/assets'; // ensure this matches your theme's public assets

/* -------------------------- Shared Navbar ------------------------------ */
ob_start();
include "$base_dir/navbar.php";
$navlink_content = ob_get_clean();
$page  = 'nav';
$fpage = 'foot';

/* ---------------------- Project -> Images helper ----------------------- */
function project_images($project_id, $limit = 3){
  global $DB, $MEDIA_TABLE, $PROJECTS_TABLE;
  $images = [];

  // Try project_media (preferred)
  if ($MEDIA_TABLE) {
    $urlCol    = col_exists($MEDIA_TABLE,'url') ? 'url' : (col_exists($MEDIA_TABLE,'file_url') ? 'file_url' : null);
    $sortCol   = col_exists($MEDIA_TABLE,'sort_order') ? 'sort_order' : (col_exists($MEDIA_TABLE,'id') ? 'id' : null);
    $isActCond = col_exists($MEDIA_TABLE,'is_active') ? 'COALESCE(is_active,1)=1 AND ' : '';
    $hasMediaType = col_exists($MEDIA_TABLE,'media_type');
    if ($urlCol) {
      $pid = (int)$project_id;
      $where = "{$isActCond} project_id={$pid}";
      if ($hasMediaType) { $where .= " AND LOWER(media_type)='image'"; }
      $orderPieces = [];
      if ($sortCol) $orderPieces[] = $sortCol . (strcasecmp($sortCol,'sort_order')===0 ? ' ASC' : ' DESC');
      $orderSql = $orderPieces ? (' ORDER BY '.implode(', ',$orderPieces)) : '';
      $limitSql = ' LIMIT '.(int)$limit;

      $sql = "SELECT {$urlCol} AS file_url FROM `{$MEDIA_TABLE}` WHERE {$where}{$orderSql}{$limitSql}";
      if ($rs = mysqli_query($DB,$sql)) {
        while($r = mysqli_fetch_assoc($rs)) {
          if (!empty($r['file_url'])) $images[] = full_url_from($r['file_url']);
        }
        @mysqli_free_result($rs);
      }
    }
  }

  // Fallback: cover_image on projects
  if (count($images) < $limit && col_exists($PROJECTS_TABLE,'cover_image')) {
    $pid = (int)$project_id;
    $r2  = mysqli_query($DB,"SELECT cover_image FROM `{$PROJECTS_TABLE}` WHERE id={$pid} LIMIT 1");
    if ($r2 && mysqli_num_rows($r2)) {
      $row = mysqli_fetch_assoc($r2);
      if (!empty($row['cover_image'])) $images[] = full_url_from($row['cover_image']);
    }
    if ($r2) @mysqli_free_result($r2);
  }

  // Final fallback: theme placeholders
  while (count($images) < $limit) {
    $images[] = '/Resido/assets/img/p-'.(count($images)+1).'.jpg';
  }
  return array_slice(array_values(array_unique($images)),0,$limit);
}

/* ------------------- Project -> Text/labels helpers -------------------- */
function project_price_text($row){
  if (!empty($row['price_text'])) return $row['price_text'];
  $min = isset($row['price_min']) ? (float)$row['price_min'] : null;
  $max = isset($row['price_max']) ? (float)($row['price_max'] ?? 0) : null;
  if ($min && $max && $max >= $min) return inr_format($min).' - '.inr_format($max);
  if ($min) return inr_format($min);
  return 'Price on request';
}
function project_location_text($row){
  $parts = [];
  foreach (['address','city','state'] as $c) {
    if (!empty($row[$c])) $parts[] = trim($row[$c]);
  }
  return $parts ? implode(', ', $parts) : 'Location not specified';
}
function project_config_text($row){
  if (!empty($row['configuration'])) return $row['configuration'];     // e.g., "2 & 3 BHK"
  if (!empty($row['bedrooms']))     return (int)$row['bedrooms'].' BHK';
  return '—';
}
function project_area_text($row){
  if (!empty($row['area_sqft'])) {
    $u = !empty($row['area_unit']) ? $row['area_unit'] : 'SQFT';
    return (float)$row['area_sqft'].' '.$u;
  }
  return '—';
}
function project_span_badge($row){
  $now = time();
  if (!empty($row['created_at'])) {
    $t = strtotime((string)$row['created_at']);
    if ($t && ($now - $t) <= (30*24*3600)) return 'SuperAgent';
  }
  if (isset($row['is_verified']) && (string)$row['is_verified']==='1') return 'Verified';
  if (isset($row['is_superagent']) && (string)$row['is_superagent']==='1') return 'SuperAgent';
  return 'Verified';
}
function project_tag1($row){
  foreach (['property_type','type','category'] as $c) {
    if (!empty($row[$c])) return $row[$c];
  }
  return 'Apartment';
}
function project_tag_rent_or_sell($row){
  if (isset($row['for_rent']) && (int)$row['for_rent']===1) return ['For Rent','label bg-light-success text-success prt-type me-2'];
  if (isset($row['is_rent']) && (int)$row['is_rent']===1)   return ['For Rent','label bg-light-success text-success prt-type me-2'];
  if (isset($row['rent_available']) && (int)$row['rent_available']===1) return ['For Rent','label bg-light-success text-success prt-type me-2'];
  if (!empty($row['sale_type']) && stripos($row['sale_type'],'rent')!==false) return ['For Rent','label bg-light-success text-success prt-type me-2'];
  if (!empty($row['status_label']) && stripos($row['status_label'],'rent')!==false) return ['For Rent','label bg-light-success text-success prt-type me-2'];
  return ['For Sell','label bg-light-danger text-danger prt-type me-2'];
}

/* -------------- Fetch ALL projects (NO rent-only filter) --------------- */
function fetch_projects_for_rent($limit=0){
  global $DB, $PROJECTS_TABLE;

  $w = [];
  if (col_exists($PROJECTS_TABLE,'is_active')) $w[] = "COALESCE(is_active,1)=1";

  // NOTE: No rent-specific conditions here – we are fetching ALL projects.
  $where = $w ? ('WHERE '.implode(' AND ', $w)) : '';

  $order = 'ORDER BY ';
  if (col_exists($PROJECTS_TABLE,'sort_order')) $order .= 'sort_order ASC, ';
  if (col_exists($PROJECTS_TABLE,'created_at')) $order .= 'created_at DESC, ';
  $order .= 'id DESC';

  $limitSql = '';
  if ((int)$limit > 0) $limitSql = ' LIMIT '.(int)$limit;

  $sql = "SELECT * FROM `{$PROJECTS_TABLE}` {$where} {$order}{$limitSql}";
  $out = [];
  if ($rs = mysqli_query($DB,$sql)) {
    while($r = mysqli_fetch_assoc($rs)) $out[] = $r;
    @mysqli_free_result($rs);
  }
  return $out;
}

/* ========================== HERO + CONTENT START ======================== */
ob_start();
?>
<!-- ============================ Hero Banner  Start================================== -->
<style>
/* === Uniform Card Sizing for "Property For Rent" & Sell sections === */
:root{
  --card-img-h: clamp(180px, 28vw, 260px); /* responsive fixed image window */
  --card-title-lines: 2;
  --card-loc-lines: 2;
}
.single-slide{height:100%}
.property-listing.card.card-fixed{display:flex;flex-direction:column;height:100%}
.property-listing.card.card-fixed .listing-img-wrapper{height:var(--card-img-h)}
.property-listing.card.card-fixed .list-img-slide,
.property-listing.card.card-fixed .click{height:100%}
.property-listing.card.card-fixed .click>div{height:100%}
.property-listing.card.card-fixed .click img{width:100%;height:100%;object-fit:cover;display:block}
.property-listing.card.card-fixed .listing-caption-wrapper{display:flex;flex-direction:column;gap:.4rem;flex:1 1 auto}
.property-listing.card.card-fixed .listing-detail-wrapper{flex:0 0 auto}
.property-listing.card.card-fixed .listing-short-detail .listing-name{
  display:-webkit-box;-webkit-line-clamp:var(--card-title-lines);-webkit-box-orient:vertical;overflow:hidden;
  min-height:calc(1.25rem * var(--card-title-lines) + .5rem);
}
.property-listing.card.card-fixed .listing-short-detail .prt-location{
  display:-webkit-box;-webkit-line-clamp:var(--card-loc-lines);-webkit-box-orient:vertical;overflow:hidden;
  min-height:2.25rem;
}
.property-listing.card.card-fixed .price-features-wrapper{margin-top:auto}
.property-listing.card.card-fixed .listing-detail-footer{padding-top:1rem;margin-top:.5rem;border-top:1px solid rgba(0,0,0,.06)}
/* tighten tiny differences coming from badges/buttons */
.property-listing.card.card-fixed .label{line-height:1;padding:.25rem .5rem;font-size:.72rem;border-radius:.4rem}
</style>

<div class="image-bottom hero-banner bg-primary-dark" style="background:url(<?php echo $static_url; ?>/img/banner-bg2.jpg) no-repeat;" data-overlay="0">
	<div class="container">
		<div class="row justify-content-center">
			<div class="col-lg-9 col-md-11 col-sm-12">
				<div class="inner-banner-text text-center mb-2">
					<h2 class="mb-4"><span class="font-normal">Ready. Set. Sold</span></h2>
					<p class="fs-5 fw-light px-xl-4 px-lg-4">Discover verified projects with transparent pricing and honest details curated by our team.</p>
				</div>
				<div class="full-search-2 eclip-search italian-search hero-search-radius shadow-hard mt-5">
					<div class="hero-search-content">
						<div class="row">
							<div class="col-lg-3 col-md-4 col-sm-12 b-r">
								<div class="form-group">
									<div class="choose-propert-type">
										<ul>
											<li>
												<div class="form-check">
													<input class="form-check-input" type="radio" id="typbuy" name="typeprt">
													<label class="form-check-label" for="typbuy">For Buy</label>
												</div>
											</li>
											<li>
												<div class="form-check">
													<input class="form-check-input" type="radio" id="typrent" name="typeprt" checked>
													<label class="form-check-label" for="typrent">For Rent</label>
												</div>
											</li>
										</ul>
									</div>
								</div>
							</div>
							<div class="col-lg-7 col-md-5 col-sm-12 p-md-0 elio">
								<div class="form-group border-start borders">
									<div class="position-relative">
										<input type="text" class="form-control border-0 ps-5" placeholder="Search for a location">
										<div class="position-absolute top-50 start-0 translate-middle-y ms-2">
											<span class="svg-icon text-primary svg-icon-2hx">
												<svg width="25" height="25" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
													<path opacity="0.3" d="M18.0624 15.3453L13.1624 20.7453C12.5624 21.4453 11.5624 21.4453 10.9624 20.7453L6.06242 15.3453C4.56242 13.6453 3.76242 11.4453 4.06242 8.94534C4.56242 5.34534 7.46242 2.44534 11.0624 2.04534C15.8624 1.54534 19.9624 5.24534 19.9624 9.94534C20.0624 12.0453 19.2624 13.9453 18.0624 15.3453Z" fill="currentColor"/>
													<path d="M12.0624 13.0453C13.7193 13.0453 15.0624 11.7022 15.0624 10.0453C15.0624 8.38849 13.7193 7.04535 12.0624 7.04535C10.4056 7.04535 9.06241 8.38849 9.06241 10.0453C9.06241 11.7022 10.4056 13.0453 12.0624 13.0453Z" fill="currentColor"/>
												</svg>
											</span>
										</div>
									</div>
								</div>
							</div>
							<div class="col-lg-2 col-md-3 col-sm-12">
								<div class="form-group">
									<a href="#" class="btn btn-dark full-width">Search</a>
								</div>
							</div>
						</div><!--/row-->
					</div>
				</div>
			</div>
		</div>
	</div>
</div>
<!-- ============================ Hero Banner End ================================== -->

<!-- ============================ Property For Rent Start ================================== -->
<section>
	<div class="container">
		<div class="row justify-content-start">
			<div class="col-lg-7 col-md-10">
				<div class="small-heads">
					<h2 class="fs-5 mb-0">Property For Rent</h2>
					<p>All Projects (no rent filter)</p>
				</div>
			</div>
		</div>
		<?php
		// Pass 0 to fetch ALL projects without limiting the count
		$rentProjects = fetch_projects_for_rent(0);
		?>
		<div class="row justify-content-center">
			<div class="col-lg-12 col-md-12">
				<div class="floral-wraps">
					<div class="slide-livok top-arrow">
						<?php if (!empty($rentProjects)): ?>
							<?php foreach ($rentProjects as $row): 
								$id       = (int)($row['id'] ?? 0);
								$title    = !empty($row['title']) ? $row['title'] : ('Project #'.$id);
								$location = project_location_text($row);
								$ptype    = project_tag1($row);
								list($tagTxt,$tagClass) = project_tag_rent_or_sell($row);
								$spanLbl  = project_span_badge($row);
								$priceTxt = project_price_text($row);
								$config   = project_config_text($row);
								$areaTxt  = project_area_text($row);
								$bedsTxt  = !empty($row['bedrooms']) ? ((int)$row['bedrooms'].' Beds') : '—';
								$imgs     = project_images($id, 3);
								$linkSlug = urlencode(str_replace(' ', '-', strtolower($title)));
							?>
							<div class="single-slide">
								<div class="property-listing card border rounded-3 card-fixed">
									<div class="listing-img-wrapper p-2">
										<div class="list-img-slide position-relative">
											<div class="position-absolute top-0 left-0 ms-3 mt-3 z-1">
												<?php if ($spanLbl === 'Verified'): ?>
													<div class="label bg-success text-light d-inline-flex align-items-center justify-content-center">
														<span class="svg-icon text-light svg-icon-2hx me-1">
															<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
																<path opacity="0.3" d="M20.5543 4.37824L12.1798 2.02473C12.0626 1.99176 11.9376 1.99176 11.8203 2.02473L3.44572 4.37824C3.18118 4.45258 3 4.6807 3 4.93945V13.569C3 14.6914 3.48509 15.8404 4.4417 16.984C5.17231 17.8575 6.18314 18.7345 7.446 19.5909C9.56752 21.0295 11.6566 21.912 11.7445 21.9488C11.8258 21.9829 11.9129 22 12.0001 22C12.0872 22 12.1744 21.983 12.2557 21.9488C12.3435 21.912 14.4326 21.0295 16.5541 19.5909C17.8169 18.7345 18.8277 17.8575 19.5584 16.984C20.515 15.8404 21 14.6914 21 13.569V4.93945C21 4.6807 20.8189 4.45258 20.5543 4.37824Z" fill="currentColor"></path>
																<path d="M14.854 11.321C14.7568 11.2282 14.6388 11.1818 14.4998 11.1818H14.3333V10.2272C14.3333 9.61741 14.1041 9.09378 13.6458 8.65628C13.1875 8.21876 12.639 8 12 8C11.361 8 10.8124 8.21876 10.3541 8.65626C9.89574 9.09378 9.66663 9.61739 9.66663 10.2272V11.1818H9.49999C9.36115 11.1818 9.24306 11.2282 9.14583 11.321C9.0486 11.4138 9 11.5265 9 11.6591V14.5227C9 14.6553 9.04862 14.768 9.14583 14.8609C9.24306 14.9536 9.36115 15 9.49999 15H14.5C14.6389 15 14.7569 14.9536 14.8542 14.8609C14.9513 14.768 15 14.6553 15 14.5227V11.6591C15.0001 11.5265 14.9513 11.4138 14.854 11.321ZM13.3333 11.1818H10.6666V10.2272C10.6666 9.87594 10.7969 9.57597 11.0573 9.32743C11.3177 9.07886 11.6319 8.9546 12 8.9546C12.3681 8.9546 12.6823 9.07884 12.9427 9.32743C13.2031 9.57595 13.3333 9.87594 13.3333 10.2272V11.1818Z" fill="currentColor"></path>
															</svg>
														</span>Verified
													</div>
												<?php elseif ($spanLbl === 'SuperAgent'): ?>
													<div class="label bg-purple text-light d-inline-flex align-items-center justify-content-center">
														<span class="svg-icon text-light svg-icon-2hx me-1">
															<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
																<path opacity="0.3" d="M20.5543 4.37824L12.1798 2.02473C12.0626 1.99176 11.9376 1.99176 11.8203 2.02473L3.44572 4.37824C3.18118 4.45258 3 4.6807 3 4.93945V13.569C3 14.6914 3.48509 15.8404 4.4417 16.984C5.17231 17.8575 6.18314 18.7345 7.446 19.5909C9.56752 21.0295 11.6566 21.912 11.7445 21.9488C11.8258 21.9829 11.9129 22 12.0001 22C12.0872 22 12.1744 21.983 12.2557 21.9488C12.3435 21.912 14.4326 21.0295 16.5541 19.5909C17.8169 18.7345 18.8277 17.8575 19.5584 16.984C20.515 15.8404 21 14.6914 21 13.569V4.93945C21 4.6807 20.8189 4.45258 20.5543 4.37824Z" fill="currentColor"/>
																<path d="M12.0006 11.1542C13.1434 11.1542 14.0777 10.22 14.0777 9.0771C14.0777 7.93424 13.1434 7 12.0006 7C10.8577 7 9.92348 7.93424 9.92348 9.0771C9.92348 10.22 10.8577 11.1542 12.0006 11.1542Z" fill="currentColor"/>
																<path d="M15.5652 13.814C15.5108 13.6779 15.4382 13.551 15.3566 13.4331C14.9393 12.8163 14.2954 12.4081 13.5697 12.3083C13.479 12.2993 13.3793 12.3174 13.3067 12.3718C12.9257 12.653 12.4722 12.7981 12.0006 12.7981C11.5289 12.7981 11.0754 12.653 10.6944 12.3718C10.6219 12.3174 10.5221 12.2902 10.4314 12.3083C9.70578 12.4081 9.05272 12.8163 8.64456 13.4331C8.56293 13.551 8.49036 13.687 8.43595 13.814C8.40875 13.8684 8.41781 13.9319 8.44502 13.9864C8.51759 14.1133 8.60828 14.2403 8.68991 14.3492C8.81689 14.5215 8.95295 14.6757 9.10715 14.8208C9.23413 14.9478 9.37925 15.0657 9.52439 15.1836C10.2409 15.7188 11.1026 15.9999 11.9915 15.9999C12.8804 15.9999 13.7421 15.7188 14.4586 15.1836C14.6038 15.0748 14.7489 14.9478 14.8759 14.8208C15.021 14.6757 15.1661 14.5215 15.2931 14.3492C15.3838 14.2312 15.4655 14.1133 15.538 13.9864C15.5833 13.9319 15.5924 13.8684 15.5652 13.814Z" fill="currentColor"/>
														</span>SuperAgent
													</div>
												<?php else: /* New */ ?>
													<div class="label bg-danger text-light d-inline-flex align-items-center justify-content-center">
														<span class="svg-icon text-light svg-icon-2hx me-1">
															<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
																<path d="M19.0647 5.43757C19.3421 5.43757 19.567 5.21271 19.567 4.93534C19.567 4.65796 19.3421 4.43311 19.0647 4.43311C18.7874 4.43311 18.5625 4.65796 18.5625 4.93534C18.5625 5.21271 18.7874 5.43757 19.0647 5.43757Z" fill="currentColor"/>
																<path d="M20.0692 9.48884C20.3466 9.48884 20.5714 9.26398 20.5714 8.98661C20.5714 8.70923 20.3466 8.48438 20.0692 8.48438C19.7918 8.48438 19.567 8.70923 19.567 8.98661C19.567 9.26398 19.7918 9.48884 20.0692 9.48884Z" fill="currentColor"/>
																<path d="M12.0335 20.5714C15.6943 20.5714 18.9426 18.2053 20.1168 14.7338C20.1884 14.5225 20.1114 14.289 19.9284 14.161C19.746 14.034 19.5003 14.0418 19.3257 14.1821C18.2432 15.0546 16.9371 15.5156 15.5491 15.5156C12.2257 15.5156 9.48884 12.8122 9.48884 9.48886C9.48884 7.41079 10.5773 5.47137 12.3449 4.35752C12.5342 4.23832 12.6 4.00733 12.5377 3.79251C12.4759 3.57768 12.2571 3.42859 12.0335 3.42859C7.32556 3.42859 3.42857 7.29209 3.42857 12C3.42857 16.7079 7.32556 20.5714 12.0335 20.5714Z" fill="currentColor"/>
																<path d="M13.0379 7.47998C13.8688 7.47998 14.5446 8.15585 14.5446 8.98668C14.5446 9.26428 14.7693 9.48891 15.0469 9.48891C15.3245 9.48891 15.5491 9.26428 15.5491 8.98668C15.5491 8.15585 16.225 7.47998 17.0558 7.47998C17.3334 7.47998 17.558 7.25535 17.558 6.97775C17.558 6.70015 17.3334 6.47552 17.0558 6.47552C16.225 6.47552 15.5491 5.76616 15.5491 4.93534C15.5491 4.65774 15.3245 4.43311 15.0469 4.43311C14.7693 4.43311 14.5446 4.65774 14.5446 4.93534C14.5446 5.76616 13.8688 6.47552 17.0558 6.47552C17.3334 6.47552 17.558 6.97775 17.558 6.70015Z" fill="currentColor"/>
															</svg>
														</span>New
													</div>
												<?php endif; ?>
											</div>
											<div class="click rounded-3 overflow-hidden mb-0">
												<?php foreach ($imgs as $img): ?>
													<div>
														<a href="single-property-1.php?title=<?php echo $linkSlug; ?>">
															<img src="<?php echo h($img); ?>" class="img-fluid" alt="<?php echo h($title); ?>" />
														</a>
													</div>
												<?php endforeach; ?>
											</div>
										</div>
									</div>

									<div class="listing-caption-wrapper px-3">
										<div class="listing-detail-wrapper">
											<div class="listing-short-detail-wrap">
												<div class="listing-short-detail">
													<div class="d-flex align-items-center">
														<span class="<?php echo h($tagClass); ?>"><?php echo h($tagTxt); ?></span>
														<span class="label bg-light-purple text-purple property-cats ms-1"><?php echo h($ptype); ?></span>
													</div>
													<h4 class="listing-name fw-semibold fs-5 mb-2 mt-3">
														<a href="single-property-1.php?title=<?php echo $linkSlug; ?>"><?php echo h($title); ?></a>
													</h4>
													<div class="prt-location text-muted-2">
														<span class="svg-icon svg-icon-2hx">
															<svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
																<path opacity="0.3" d="M18.0624 15.3453L13.1624 20.7453C12.5624 21.4453 11.5624 21.4453 10.9624 20.7453L6.06242 15.3453C4.56242 13.6453 3.76242 11.4453 4.06242 8.94534C4.56242 5.34534 7.46242 2.44534 11.0624 2.04534C15.8624 1.54534 19.9624 5.24534 19.9624 9.94534C20.0624 12.0453 19.2624 13.9453 18.0624 15.3453Z" fill="currentColor"/>
																<path d="M12.0624 13.0453C13.7193 13.0453 15.0624 11.7022 15.0624 10.0453C15.0624 8.38849 13.7193 7.04535 12.0624 7.04535C10.4056 7.04535 9.06241 8.38849 9.06241 10.0453C9.06241 11.7022 10.4056 13.0453 12.0624 13.0453Z" fill="currentColor"/>
															</svg>
														</span>
														<?php echo h($location); ?>
													</div>
												</div>
											</div>
										</div>

										<div class="price-features-wrapper">
											<div class="list-fx-features d-flex align-items-center justify-content-between">
												<div class="listing-card d-flex align-items-center">
													<div class="square--30 text-muted-2 fs-sm circle gray-simple me-2">
														<i class="fa-solid fa-building-shield fs-sm"></i>
													</div>
													<span class="text-muted-2"><?php echo h($config); ?></span>
												</div>
												<div class="listing-card d-flex align-items-center">
													<div class="square--30 text-muted-2 fs-sm circle gray-simple me-2">
														<i class="fa-solid fa-bed fs-sm"></i>
													</div>
													<span class="text-muted-2"><?php echo h($bedsTxt); ?></span>
												</div>
												<div class="listing-card d-flex align-items-center">
													<div class="square--30 text-muted-2 fs-sm circle gray-simple me-2">
														<i class="fa-solid fa-clone fs-sm"></i>
													</div>
													<span class="text-muted-2"><?php echo h($areaTxt); ?></span>
												</div>
											</div>
										</div>

										<div class="listing-detail-footer d-flex align-items-center justify-content-between py-4">
											<div class="listing-short-detail-flex">
												<h6 class="listing-card-info-price m-0"><?php echo h($priceTxt); ?></h6>
											</div>
											<div class="footer-flex">
												<a href="property-detail.php?id=<?php echo $id; ?>" class="prt-view">
													<span class="svg-icon text-primary svg-icon-2hx">
														<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
															<path d="M15.43 8.56949L10.744 15.1395C10.6422 15.282 10.5804 15.4492 10.5651 15.6236C10.5498 15.7981 10.5815 15.9734 10.657 16.1315L13.194 21.4425C13.2737 21.6097 13.3991 21.751 13.5557 21.8499C13.7123 21.9488 13.8938 22.0014 14.079 22.0015H14.117C14.3087 21.9941 14.4941 21.9307 14.6502 21.8191C14.8062 21.7075 14.9261 21.5526 14.995 21.3735L21.933 3.33649C22.0011 3.15918 22.0164 2.96594 21.977 2.78013C21.9376 2.59432 21.8452 2.4239 21.711 2.28949L15.43 8.56949Z" fill="currentColor"/>
															<path opacity="0.3" d="M20.664 2.06648L2.62602 9.00148C2.44768 9.07085 2.29348 9.19082 2.1824 9.34663C2.07131 9.50244 2.00818 9.68731 2.00074 9.87853C1.99331 10.0697 2.04189 10.259 2.14054 10.4229C2.23919 10.5869 2.38359 10.7185 2.55601 10.8015L7.86601 13.3365C8.02383 13.4126 8.19925 13.4448 8.37382 13.4297C8.54839 13.4145 8.71565 13.3526 8.85801 13.2505L15.43 8.56548L21.711 2.28448C21.5762 2.15096 21.4055 2.05932 21.2198 2.02064C21.034 1.98196 20.8409 1.99788 20.664 2.06648Z" fill="currentColor"/>
														</svg>
													</span>
												</a>
											</div>
										</div>
									</div><!--/caption-->
								</div><!--/card-->
							</div><!--/single-slide-->
							<?php endforeach; ?>
						<?php else: ?>
							<p class="text-center text-muted py-5">No projects found.</p>
						<?php endif; ?>
					</div><!--/slide-livok-->
				</div><!--/floral-wraps-->
			</div>
		</div>
	</div>
</section>
<div class="clearfix"></div>
<!-- ============================ Property For Rent End ================================== -->

<?php
/* --------------------- (Optional) Static SELL section -------------------- */
$propertys = [
  [
    'id'=>1,'img'=>'/img/p-1.jpg','img1'=>'/img/p-9.jpg','img2'=>'/img/p-10.jpg',
    'tag'=>'For Rent','class'=>'label bg-light-success text-success prt-type me-2',
    'tag1'=>'Apartment','title'=>'The Green Canton Chrysler','location'=>'210 Zirak Road, Canada',
    'price'=>'$80,000','span'=>'Verified'
  ],
  [
    'id'=>2,'img'=>'/img/p-2.jpg','img1'=>'/img/p-6.jpg','img2'=>'/img/p-8.jpg',
    'tag'=>'For Sell','class'=>'label bg-light-danger text-danger prt-type me-2',
    'tag1'=>'House','title'=>'Purple Flatiron House','location'=>'210 Zirak Road, Canada',
    'price'=>'$67,000','span'=>'SuperAgent'
  ],
  [
    'id'=>3,'img'=>'/img/p-3.jpg','img1'=>'/img/p-5.jpg','img2'=>'/img/p-7.jpg',
    'tag'=>'For Rent','class'=>'label bg-light-success text-success prt-type me-2',
    'tag1'=>'Building','title'=>'Rustic Reunion Tower','location'=>'210 Zirak Road, Canada',
    'price'=>'$92,500','span'=>'New'
  ],
];
function render_property_cards_static($propertys,$static_url){
  foreach ($propertys as $item): ?>
    <div class="single-slide">
      <div class="property-listing card border rounded-3 card-fixed">
        <div class="listing-img-wrapper p-2">
          <div class="list-img-slide position-relative">
            <div class="position-absolute top-0 left-0 ms-3 mt-3 z-1">
              <?php if ($item['span'] === 'Verified'): ?>
                <div class="label bg-success text-light d-inline-flex align-items-center justify-content-center">
                  <span class="svg-icon text-light svg-icon-2hx me-1"><svg width="14" height="14" viewBox="0 0 24 24"><path opacity="0.3" d="M20.55 4.38 12.18 2.02c-.12-.03-.25-.03-.36 0L3.45 4.38C3.18 4.45 3 4.68 3 4.94v8.63c0 1.12.49 2.27 1.45 3.42.73.87 1.74 1.75 3 2.61 2.12 1.44 4.21 2.32 4.29 2.36.08.03.17.05.25.05s.17-.03.25-.07c.09-.04 2.18-.93 4.3-2.36 1.26-.86 2.27-1.74 3-2.61.96-1.15 1.45-2.3 1.45-3.42V4.94c0-.26-.18-.49-.45-.56Z" fill="currentColor"/><path d="M14.85 11.32c-.1-.09-.22-.14-.35-.14h-.17V10.23c0-.61-.23-1.13-.69-1.57C13.19 8.22 12.64 8 12 8c-.64 0-1.19.22-1.65.66-.46.44-.69.96-.69 1.57v.95h-.17c-.13 0-.25.05-.35.14-.1.09-.15.21-.15.34v2.86c0 .13.05.25.15.34.1.09.22.14.35.14H14.5c.14 0 .26-.05.36-.14.1-.09.15-.21.15-.34v-2.86c.01-.13-.04-.25-.16-.34ZM13.33 11.18h-2.67v-.95c0-.35.13-.65.39-.9.26-.25.57-.38.95-.38.37 0 .69.13.95.38.26.25.39.56.39.9v.95Z" fill="currentColor"/></svg></span>Verified
                </div>
              <?php elseif ($item['span'] === 'SuperAgent'): ?>
                <div class="label bg-purple text-light d-inline-flex align-items-center justify-content-center">
                  <span class="svg-icon text-light svg-icon-2hx me-1"><svg width="14" height="14" viewBox="0 0 24 24"><path opacity="0.3" d="M20.55 4.38 12.18 2.02c-.12-.03-.25-.03-.36 0L3.45 4.38C3.18 4.45 3 4.68 3 4.94v8.63c0 1.12.49 2.27 1.45 3.42.73.87 1.74 1.75 3 2.61 2.12 1.44 4.21 2.32 4.29 2.36.08.03.17.05.25.05s.17-.03.25-.07c.09-.04 2.18-.93 4.3-2.36 1.26-.86 2.27-1.74 3-2.61.96-1.15 1.45-2.3 1.45-3.42V4.94c0-.26-.18-.49-.45-.56Z" fill="currentColor"/><path d="M12 11.15c1.14 0 2.08-.94 2.08-2.08S13.14 7 12 7s-2.08.94-2.08 2.08.94 2.07 2.08 2.07Z" fill="currentColor"/><path d="M15.57 13.81c-.05-.14-.12-.27-.2-.38-.42-.62-1.07-1.03-1.79-1.13-.09-.01-.19.01-.26.06-.38.28-.83.43-1.31.43-.47 0-.92-.15-1.31-.43a.45.45 0 0 0-.26-.06c-.72.1-1.37.51-1.79 1.13-.08.11-.14.24-.2.38-.03.05-.02.12.01.17.08.15.17.28.26.39.13.17.26.31.4.45.72.53 1.58.82 2.49.82.9 0 1.77-.29 2.49-.82.15-.11.28-.25.4-.45.09-.11.18-.24.26-.39.03-.05.04-.12.01-.17Z" fill="currentColor"/></svg></span>SuperAgent
                </div>
              <?php else: ?>
                <div class="label bg-danger text-light d-inline-flex align-items-center justify-content-center"><span class="svg-icon text-light svg-icon-2hx me-1"></span>New</div>
              <?php endif; ?>
            </div>
            <div class="click rounded-3 overflow-hidden mb-0">
              <div><a href="single-property-1.php?title=<?php echo urlencode(str_replace(' ', '-', strtolower($item['title']))); ?>"><img src="<?php echo $static_url, $item['img']; ?>" class="img-fluid" alt="" /></a></div>
              <div><a href="single-property-1.php?title=<?php echo urlencode(str_replace(' ', '-', strtolower($item['title']))); ?>"><img src="<?php echo $static_url, $item['img1']; ?>" class="img-fluid" alt="" /></a></div>
              <div><a href="single-property-1.php?title=<?php echo urlencode(str_replace(' ', '-', strtolower($item['title']))); ?>"><img src="<?php echo $static_url, $item['img2']; ?>" class="img-fluid" alt="" /></a></div>
            </div>
          </div>
        </div>
        <div class="listing-caption-wrapper px-3">
          <div class="listing-detail-wrapper">
            <div class="listing-short-detail-wrap">
              <div class="listing-short-detail">
                <div class="d-flex align-items-center"><span class="<?php echo $item['class']; ?>"><?php echo $item['tag']; ?></span><span class="label bg-light-purple text-purple property-cats"><?php echo $item['tag1']; ?></span></div>
                <h4 class="listing-name fw-semibold fs-5 mb-2 mt-3"><a href="single-property-1.php?title=<?php echo urlencode(str_replace(' ', '-', strtolower($item['title']))); ?>"><?php echo $item['title']; ?></a></h4>
                <div class="prt-location text-muted-2"><?php echo $item['location']; ?></div>
              </div>
            </div>
          </div>
          <div class="price-features-wrapper">
            <div class="list-fx-features d-flex align-items-center justify-content-between">
              <div class="listing-card d-flex align-items-center"><div class="square--30 text-muted-2 fs-sm circle gray-simple me-2"><i class="fa-solid fa-building-shield fs-sm"></i></div><span class="text-muted-2">4BHK</span></div>
              <div class="listing-card d-flex align-items-center"><div class="square--30 text-muted-2 fs-sm circle gray-simple me-2"><i class="fa-solid fa-bed fs-sm"></i></div><span class="text-muted-2">3 Beds</span></div>
              <div class="listing-card d-flex align-items-center"><div class="square--30 text-muted-2 fs-sm circle gray-simple me-2"><i class="fa-solid fa-clone fs-sm"></i></div><span class="text-muted-2">1800 SQFT</span></div>
            </div>
          </div>
          <div class="listing-detail-footer d-flex align-items-center justify-content-between py-4">
            <div class="listing-short-detail-flex"><h6 class="listing-card-info-price m-0"><?php echo $item['price']; ?></h6></div>
            <div class="footer-flex"><a href="property-detail.php" class="prt-view"><span class="svg-icon text-primary svg-icon-2hx"></span></a></div>
          </div>
        </div>
      </div>
    </div>
  <?php endforeach;
}
?>

<!-- ============================ Property For Sell Start ================================== -->
<section class="pt-0">
	<div class="container">
		<div class="row justify-content-start">
			<div class="col-lg-7 col-md-10">
				<div class="small-heads">
					<h2 class="fs-5 mb-0">Properties For Sell</h2>
					<p>Best Properties For Sell</p>
				</div>
			</div>
		</div>
		<div class="row justify-content-center">
			<div class="col-lg-12 col-md-12">
				<div class="floral-wraps">
					<div class="slide-livok top-arrow">
						<?php render_property_cards_static($propertys, $static_url); ?>
					</div>
				</div>
			</div>
		</div>
	</div>
</section>
<div class="clearfix"></div>
<!-- ============================ Property For Sell End ================================== -->

<!-- ============================ Explore Top Places Start ================================== -->
<section class="pt-0">
	<div class="container">
		<div class="row justify-content-start">
			<div class="col-lg-7 col-md-10">
				<div class="small-heads">
					<h2 class="fs-5 mb-0">Explore Top Places</h2>
					<p>Best Places</p>
				</div>
			</div>
		</div>
		<div class="row justify-content-center">
			<div class="col-lg-12 col-md-12">
				<div class="floral-wraps">
					<div class="slide-livok top-arrow">
						<?php include "$base_dir/Components/Home/top-places.php"; ?>
					</div>
				</div>
			</div>
		</div>
	</div>
</section>
<div class="clearfix"></div>
<!-- ============================ Explore Top Places End ================================== -->

<!-- ============================ New Projects Start ================================== -->
<section class="pt-0">
	<div class="container">
		<div class="row justify-content-center">
			<div class="col-lg-12 col-md-12">
				<div class="bg-primary rounded-4 py-md-5 px-md-5 p-4">
					<div class="row align-items-center justify-content-between py-5">
						<div class="col-xl-9 col-lg-8 col-md-7 my-2">
							<h2 class="fs-1 text-light lh-base">New Projects in United States</h2>
							<p class="text-light m-0">New projects, verified by our quality control teams.</p>
						</div>
						<div class="col-xl-3 col-lg-4 col-md-5 my-2">
							<a href="#" class="btn btn-lg btn-whites full-width fw-medium">View New Projects</a>
						</div>
					</div>
				</div>
			</div>
		</div>
	</div>
</section>
<div class="clearfix"></div>
<!-- ============================ New project End ================================== -->

<!-- ============================ Explore Top Agents Start ================================== -->
<section class="pt-0">
	<div class="container">
		<div class="row justify-content-start">
			<div class="col-lg-7 col-md-10">
				<div class="small-heads">
					<h2 class="fs-5 mb-0">Meet Top Agents</h2>
					<p>Top Agents</p>
				</div>
			</div>
		</div>
		<div class="row justify-content-center">
			<div class="col-lg-12 col-md-12">
				<div class="floral-wraps">
					<div class="slide-livok top-arrow">
						<?php include "$base_dir/Components/Home/agents1.php"; ?>
					</div>
				</div>
			</div>
		</div>
	</div>
</section>
<div class="clearfix"></div>
<!-- ============================ Explore Top Agents End ================================== -->

<!-- ============================ Smart Testimonials ================================== -->
<section class="gray-bg">
	<div class="container">
		<div class="row justify-content-center">
			<div class="col-lg-7 col-md-10 text-center">
				<div class="sec-heading center">
					<h2>Good Reviews by Customers</h2>
					<p>At vero eos et accusamus et iusto odio dignissimos ducimus qui blanditiis praesentium voluptatum deleniti atque corrupti quos dolores</p>
				</div>
			</div>
		</div>
		<div class="row justify-content-center">
			<div class="col-lg-12 col-md-12">
				<div class="smart-textimonials smart-center" id="smart-textimonials">
					<?php include "$base_dir/Components/Home/reviews.php"; ?>
				</div>
			</div>
		</div>
	</div>
</section>
<!-- ============================ Smart Testimonials End ================================== -->

<!-- ============================ Price Table Start ================================== -->
<section>
	<div class="container">
		<div class="row justify-content-center">
			<div class="col-lg-7 col-md-10 text-center">
				<div class="sec-heading center">
					<h2>See our packages</h2>
					<p>At vero eos et accusamus et iusto odio dignissimos ducimus qui blanditiis praesentium voluptatum deleniti atque corrupti quos dolores</p>
				</div>
			</div>
		</div>
		<div class="row align-items-center justify-content-center g-lg-4 g-md-3 g-4">
			<?php include "$base_dir/Components/Home/packages.php"; ?>
		</div>
	</div>	
</section>
<!-- ============================ Price Table End ================================== -->

<!-- ============================ Call To Action ================================== -->
<section class="bg-primary call-to-act-wrap">
	<div class="container">
		<?php include "$base_dir/Components/Home/estate-agent.php"; ?>
	</div>
</section>
<!-- ============================ Call To Action End ================================== -->

<?php
$hero_content = ob_get_clean();

/* ----------------------------- Render base ----------------------------- */
include "$base_dir/style/base.php";
