<?php
/* =============================================================================
 * File   : projects_mobile.php
 * Goal   : Mobile Projects list (Featured / All) + Filters (search/price/city/…)
 * Updated: header fixed to top + sidebar & sheet safe offsets + bookmarking
 * ============================================================================= */

@session_start();
date_default_timezone_set('Asia/Kolkata');

/* ---------------------- Ensure body class safe default -------------------- */
$bodyClass = $_SESSION['theme_mode'] ?? '';

/* ---- Bootstrap auth + config + helpers (provides h(), $DB link helpers) ---- */
require_once '/home/broriserin/public_html/mobile/auth_bootstrap.php'; // has h(), config.php
$DB = isset($conn_crm) && $conn_crm instanceof mysqli ? $conn_crm : (isset($conn) ? $conn : null);
if (!($DB instanceof mysqli)) { http_response_code(500); exit('DB connection missing.'); }
@mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET NAMES 'utf8mb4'");
@mysqli_query($DB, "SET collation_connection='utf8mb4_general_ci'");

/* ------------------------------- Helpers -------------------------------- */
function q($s){ global $DB; return mysqli_real_escape_string($DB,(string)$s); }
function one($sql){ global $DB; $r=@mysqli_query($DB,$sql); if($r && ($row=mysqli_fetch_assoc($r))){ @mysqli_free_result($r); return $row; } return null; }
function all($sql){ global $DB; $o=[]; if($r=@mysqli_query($DB,$sql)){ while($row=mysqli_fetch_assoc($r)) $o[]=$row; @mysqli_free_result($r);} return $o; }
function table_exists($name){ global $DB; $name=q($name); $r=@mysqli_query($DB,"SHOW TABLES LIKE '{$name}'"); return ($r && mysqli_num_rows($r)>0); }
function col_exists($table,$col){ global $DB; $t=q($table); $c=q($col); $r=@mysqli_query($DB,"SHOW COLUMNS FROM `{$t}` LIKE '{$c}'"); return ($r && mysqli_num_rows($r)>0); }
function distinct_vals($table,$col,$limit=120){
  $table = str_replace('`','',$table);
  $col   = str_replace('`','',$col);
  return array_map(fn($r)=>$r['v'], all("SELECT DISTINCT `$col` AS v FROM `$table` WHERE COALESCE(`$col`,'')<>'' ORDER BY 1 LIMIT ".(int)$limit));
}
function format_inr_compact($n){
  $n = (float)$n;
  if ($n >= 10000000) {
    $v = $n / 10000000; $s = rtrim(rtrim(number_format($v, 1), '0'), '.'); return $s . ' Cr Onwards';
  } elseif ($n >= 100000) {
    $v = $n / 100000; $s = rtrim(rtrim(number_format($v, 1), '0'), '.'); return $s . ' Lakhs Onwards';
  } else {
    return number_format($n) . ' Onwards';
  }
}

/* ----------------- Bookmark AJAX handler + table auto-create (supports guests) ----------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && (($_POST['action'] ?? '') === 'toggle_bookmark')) {
    header('Content-Type: application/json; charset=utf-8');

    // Parse project id
    $project_id = (int)($_POST['project_id'] ?? 0);
    if ($project_id <= 0) {
        http_response_code(400);
        echo json_encode(['ok'=>false,'error'=>'invalid_project']);
        exit;
    }

    // Determine actor: prefer logged-in user id; else fallback to a session guest token
    $user_id = $_SESSION['user_id'] ?? $_SESSION['uid'] ?? null;

    // Ensure a stable guest token in session for non-logged users
    if (!$user_id) {
        if (empty($_SESSION['guest_token'])) {
            try {
                $_SESSION['guest_token'] = bin2hex(random_bytes(16));
            } catch (Exception $e) {
                $_SESSION['guest_token'] = substr(md5(uniqid('', true)), 0, 32);
            }
        }
        $guest_token = $_SESSION['guest_token'];
    } else {
        $guest_token = null;
    }

    // ensure bookmarks table exists (allows user_id NULL and guest_token)
    $bm_table = 'project_bookmarks';
    if (!table_exists($bm_table)) {
        $create = "
          CREATE TABLE IF NOT EXISTS `{$bm_table}` (
            `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            `user_id` BIGINT UNSIGNED DEFAULT NULL,
            `guest_token` VARCHAR(64) DEFAULT NULL,
            `project_id` BIGINT UNSIGNED NOT NULL,
            `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `uk_user_project` (`user_id`,`project_id`),
            UNIQUE KEY `uk_guest_project` (`guest_token`,`project_id`)
          ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
        ";
        @mysqli_query($DB, $create);
    }

    // Build safe WHERE / existence check
    if ($user_id) {
        $check_sql = "SELECT id FROM `{$bm_table}` WHERE user_id = ".(int)$user_id." AND project_id = ".(int)$project_id." LIMIT 1";
    } else {
        $t = mysqli_real_escape_string($DB, $guest_token);
        $check_sql = "SELECT id FROM `{$bm_table}` WHERE guest_token = '{$t}' AND project_id = ".(int)$project_id." LIMIT 1";
    }

    $exists = one($check_sql);
    if ($exists) {
        // delete
        $del_sql = "DELETE FROM `{$bm_table}` WHERE id = ".(int)$exists['id']." LIMIT 1";
        $ok = @mysqli_query($DB, $del_sql);
        if ($ok) echo json_encode(['ok'=>true,'bookmarked'=>false]);
        else { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'db_delete']); }
    } else {
        // insert
        if ($user_id) {
            $ins = "INSERT INTO `{$bm_table}` (user_id, project_id) VALUES (".(int)$user_id.",".(int)$project_id.")";
        } else {
            $t = mysqli_real_escape_string($DB, $guest_token);
            $ins = "INSERT INTO `{$bm_table}` (guest_token, project_id) VALUES ('{$t}',".(int)$project_id.")";
        }
        $ok = @mysqli_query($DB, $ins);
        if ($ok) echo json_encode(['ok'=>true,'bookmarked'=>true]);
        else { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'db_insert']); }
    }
    exit;
}

/* --------------------------- Detect table/columns --------------------- */
$PROJECTS_TABLE=null;
foreach (['projects','project','tbl_projects','pp_projects'] as $t) {
  if (table_exists($t)) { $PROJECTS_TABLE=$t; break; }
}
if(!$PROJECTS_TABLE && table_exists('property_listings')) { $PROJECTS_TABLE='property_listings'; }
if(!$PROJECTS_TABLE){ http_response_code(500); exit('No projects table found.'); }

$HAS = fn($c)=>col_exists($PROJECTS_TABLE,$c);
$C = [
  'id'        => $HAS('id') ? 'id' : ($HAS('project_id')?'project_id':'id'),
  'title'     => $HAS('title')?'title':($HAS('name')?'name':($HAS('project_name')?'project_name':'title')),
  'city'      => $HAS('city')?'city':($HAS('location_city')?'location_city':null),
  'region'    => $HAS('region')?'region':($HAS('zone')?'zone':null),
  'locality'  => $HAS('location')?'location':($HAS('locality')?'locality':null),
  'price_min' => $HAS('price_min')?'price_min':null,
  'price_txt' => $HAS('price_text')?'price_text':($HAS('price_label')?'price_label':null),
  'image'     => $HAS('cover_image')?'cover_image':($HAS('image')?'image':($HAS('thumbnail')?'thumbnail':($HAS('photo')?'photo':null))),
  'config'    => $HAS('configuration')?'configuration':($HAS('property_type')?'property_type':null),
  'tags'      => $HAS('tags')?'tags':null,
  'active'    => $HAS('is_active')?'is_active':null,
  'project_tag'=> $HAS('project_tag')?'project_tag':null,
];

/* ------------------------------ Inputs -------------------------------- */
$tab = ($_GET['tab'] ?? 'featured') === 'all' ? 'all' : 'featured';
$q   = trim($_GET['q'] ?? '');
$chips_city = trim($_GET['chips_city'] ?? '');

$flt = [
  'city'      => (array)($_GET['city'] ?? []),
  'region'    => (array)($_GET['region'] ?? []),
  'locality'  => (array)($_GET['locality'] ?? []),
  'config'    => (array)($_GET['configuration'] ?? ($_GET['property_type'] ?? [])),
  'tags'      => (array)($_GET['tags'] ?? []),
  'pmin'      => $_GET['price_min'] ?? '',
  'pmax'      => $_GET['price_max'] ?? '',
];
if($chips_city!==''){ $flt['city'][]=$chips_city; }

/* ---------------------------- WHERE builder ---------------------------- */
$W = ['1=1'];
if($C['active']){ $W[]="COALESCE(`{$C['active']}`,1)=1"; }
if ($tab==='featured' && $C['project_tag']) {
  $W[]="LOWER(COALESCE(`{$C['project_tag']}`,'')) LIKE '%featured%'";
}
if($q!==''){
  $like = '%'.q($q).'%';
  $or = [];
  foreach ([$C['title'],$C['locality'],$C['city'],$C['region'],$C['config']] as $col) {
    if($col){ $or[]="`$col` LIKE '{$like}'"; }
  }
  if($or) $W[]='('.implode(' OR ',$or).')';
}
$mk_in = function($arr){ return implode(',', array_map(fn($v)=>"'".q($v)."'", array_unique(array_filter(array_map('trim',$arr))))); };
if($C['city'] && $flt['city'])         $W[]="`{$C['city']}` IN (".$mk_in($flt['city']).")";
if($C['region'] && $flt['region'])     $W[]="`{$C['region']}` IN (".$mk_in($flt['region']).")";
if($C['locality'] && $flt['locality']) $W[]="`{$C['locality']}` IN (".$mk_in($flt['locality']).")";
if($C['config'] && $flt['config'])     $W[]="`{$C['config']}` IN (".$mk_in($flt['config']).")";
if($C['price_min']){
  $expr="`{$C['price_min']}`";
  if($flt['pmin']!=='') $W[]="$expr >= ".(float)$flt['pmin'];
  if($flt['pmax']!=='') $W[]="$expr <= ".(float)$flt['pmax'];
}
$where_sql = 'WHERE '.implode(' AND ',$W);

/* ------------------------------- Counts -------------------------------- */
$totalWhere = ['1=1'];
if($C['active']) $totalWhere[]="COALESCE(`{$C['active']}`,1)=1";
$total_sql = 'WHERE '.implode(' AND ',$totalWhere);
$total = (int)(one("SELECT COUNT(*) c FROM `$PROJECTS_TABLE` $total_sql")['c'] ?? 0);
$shown = (int)(one("SELECT COUNT(*) c FROM `$PROJECTS_TABLE` $where_sql")['c'] ?? 0);

/* ------------------------------- Data ---------------------------------- */
$list = all("SELECT * FROM `$PROJECTS_TABLE` $where_sql ORDER BY `{$C['id']}` DESC LIMIT 20");

/* ---------------- Build bookmarks map for current actor ----------------- */
$bookmarks_map = [];
$actor_user = $_SESSION['user_id'] ?? $_SESSION['uid'] ?? null;
$actor_guest = $_SESSION['guest_token'] ?? null;
$bm_table = 'project_bookmarks';
if (($actor_user || $actor_guest) && table_exists($bm_table)) {
    if ($actor_user) {
        $rows_bm = all("SELECT project_id FROM `{$bm_table}` WHERE user_id = ".(int)$actor_user);
    } else {
        $t = mysqli_real_escape_string($DB, $actor_guest);
        $rows_bm = all("SELECT project_id FROM `{$bm_table}` WHERE guest_token = '{$t}'");
    }
    foreach ($rows_bm as $b) { $bookmarks_map[(int)$b['project_id']] = true; }
}

/* ------------------------------- Options -------------------------------- */
$chipCities = $C['city'] ? distinct_vals($PROJECTS_TABLE, $C['city'], 40) : [];
$opt_city   = $C['city'] ? distinct_vals($PROJECTS_TABLE, $C['city'], 120) : [];
$opt_region = $C['region'] ? distinct_vals($PROJECTS_TABLE, $C['region'], 120) : [];
$opt_local  = $C['locality'] ? distinct_vals($PROJECTS_TABLE, $C['locality'], 160) : [];
$opt_conf   = $C['config'] ? distinct_vals($PROJECTS_TABLE, $C['config'], 80) : [];
$opt_tags   = $C['project_tag'] ? distinct_vals($PROJECTS_TABLE, $C['project_tag'], 30) : [];

if($C['price_min']){
  $rng = one("SELECT MIN(`{$C['price_min']}`) mn, MAX(`{$C['price_min']}`) mx FROM `$PROJECTS_TABLE` $total_sql");
  $price_min_default = isset($rng['mn']) ? max(0,(int)$rng['mn']) : 0;
  $price_max_default = isset($rng['mx']) ? max($price_min_default,(int)$rng['mx']) : 100000000;
} else {
  $price_min_default = 0; $price_max_default = 100000000;
}

/* ------------------------------- SEO/UI -------------------------------- */
$PAGE_TITLE   = ($tab==='featured'?'Featured Projects':'All Projects');
$PAGE_DESC    = 'Browse projects (search, city, region, location, configuration, tags & price).';
$ACTIVE_FOOTER= 'projects';
$BASE = defined('BASE_URL') ? rtrim(BASE_URL,'/').'/' : '/';
$ASSET = $BASE . 'mobile/';
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1, viewport-fit=cover">
<title><?php echo h($PAGE_TITLE); ?></title>
<meta name="description" content="<?php echo h($PAGE_DESC); ?>">
<meta name="robots" content="index,follow">
<meta name="theme-color" content="#ffffff">
<link rel="stylesheet" href="<?php echo h($ASSET.'assets/css/style.css'); ?>">

<style>
:root{
  --header-h: 60px; /* header height (change if header markup differs) */
}

/* Fixed header — transparent so page gradient shows through */
.header, header.header {
  position: fixed;
  top: 0; left: 0; right: 0;
  height: var(--header-h);
  z-index: 1305;
  background: transparent;
  border-bottom: none;
}
.header { padding-top: env(safe-area-inset-top, 0); box-sizing: border-box; }

/* main content offset so header doesn't cover it */
.container-tight, main.page-content, main.wrap {
  padding-top: calc(var(--header-h) + -50px); /* small gap under header */
}

/* Filter sheet (account for header) */
.sheet-panel{ top: 0; padding-top: calc(var(--header-h) + env(safe-area-inset-top, 0)); box-sizing: border-box; }

/* Mobile Sidebar adjustments */
#mobileSidebar { top: calc(var(--header-h) + env(safe-area-inset-top, 0)); height: calc(100vh - calc(var(--header-h) + env(safe-area-inset-top,0))); }
#mobileSidebarBackdrop { top: calc(var(--header-h) + env(safe-area-inset-top,0)); height: calc(100vh - calc(var(--header-h) + env(safe-area-inset-top,0))); }

/* Compact / denser card layout (reduced margins & image height) */
body { background: #f6f7fb; }
.container-tight{ max-width: 540px; margin: 0 auto; }
.tabs-bar{ display:flex; gap:12px; padding:8px 12px 4px; align-items:flex-end; margin-top:0; }
.tabs-bar a{ font-weight:800; font-size:17px; color:#6b7380; }
.tabs-bar a.active{ color:#FE9063; position:relative; }
.tabs-bar a.active:after{ content:''; position:absolute; left:0; right:0; bottom:-6px; height:4px; background:#111; border-radius:999px; }

/* Search card minor tweak */
.search-card{ background:#fff; border:1px solid #e9edf3; border-radius:14px; padding:10px; margin:6px 12px; box-shadow:0 8px 20px rgba(17,23,32,.04); }
.search-row{ display:grid; grid-template-columns:1fr 44px 44px; gap:8px; align-items:center; }
.search-row .form-control{ height:42px; border-radius:12px; border:1px solid #eceff4; background:#fbfdff; }
.icon-btn{ height:42px; border-radius:12px; border:1px solid #eceff4; background:#fbfdff; display:flex; align-items:center; justify-content:center; }
.icon-btn svg{ width:18px; height:18px; stroke:#111; }

/* chips */
.hchips{ display:flex; gap:10px; padding:8px 12px 0; overflow:auto; }
.chip{ background:#fff; border:1px solid #e5e8ef; color:#111; padding:7px 12px; border-radius:999px; white-space:nowrap; font-weight:700; font-size:13px; }
.chip.active{ background:#111; color:#fff; border-color:#111; }

/* counter */
.counter{ padding:8px 12px; color:#6b7380; font-weight:700; font-size:14px; }

/* card reduced size and margins */
.card-proj{
  background:#fff; border:1px solid #e9edf3; border-radius:14px; overflow:hidden;
  margin:8px 12px; box-shadow: 0 8px 18px rgba(17,23,32,.05); display:block;
  transition: box-shadow .18s, transform .12s, border-color .12s;
}
.card-proj:hover{ transform: translateY(-2px); box-shadow: 0 12px 30px rgba(17,23,32,.07); }

/* bookmarked (blue) visual */
.card-proj.bookmarked{ border-color:#2b8df8; box-shadow: 0 10px 30px rgba(43,141,248,0.06); }

/* image area smaller */
.card-proj .pimg{ position:relative; height:160px; background:#eef2f7; }
.card-proj .pimg img{ width:100%; height:100%; object-fit:cover; display:block; }

/* bookmark button */
.card-proj .pbookmark{
  position:absolute; left:10px; top:10px; width:36px; height:36px; border-radius:10px;
  border:1px solid rgba(255,255,255,.9); background:rgba(255,255,255,.96); display:flex; align-items:center; justify-content:center;
  box-shadow:0 6px 18px rgba(0,0,0,.06);
  transition: background .12s, transform .12s;
}
.card-proj .pbookmark svg{ width:18px; height:18px; fill: #111; }
.card-proj.bookmarked .pbookmark{ background:#fff; border-color:#2b8df8; transform:scale(1.02); }
.card-proj.bookmarked .pbookmark svg{ fill:#2b8df8; }

/* featured chip */
.card-proj .pchip{
  position:absolute; right:10px; bottom:10px; display:inline-flex; gap:6px; align-items:center;
  background:#FE9063; color:#fff; font-weight:800; font-size:12px; padding:6px 10px; border-radius:999px;
}

/* body text */
.card-proj .body{ padding:10px; }
.card-proj .title{ font-size:16px; font-weight:800; margin:6px 0; }
.card-proj .sub{ color:#6b7380; font-size:13px; margin-bottom:6px; }
.card-proj .price{ display:flex; gap:8px; align-items:baseline; margin-top:6px; }
.card-proj .price .lbl{ font-size:12px; color:#6b7380; font-weight:800; text-transform:uppercase; }
.card-proj .price .val{ font-size:18px; font-weight:900; color:#FE9063; }
.card-proj .tags{ display:flex; flex-wrap:wrap; gap:8px; margin-top:8px; }
.card-proj .tag{ background:#f8fafc; border:1px solid #e9edf3; border-radius:999px; font-size:12px; font-weight:700; padding:6px 10px; color:#111; }

/* sheet & sidebar basics */
.sheet-mask{ position:fixed; inset:0; background:rgba(0,0,0,.4); display:none; z-index:1050; }
.sheet-panel{ position:absolute; inset:0; background:#fff; display:flex; flex-direction:column; max-width:560px; margin:0 auto; }
.sheet-top{ height:56px; display:flex; align-items:center; justify-content:space-between; padding:0 12px; border-bottom:1px solid #e5e8ef; font-weight:800; }
.sheet-body{ flex:1; display:grid; grid-template-columns:36vw 1fr; min-height:0; }
.sheet-left{ border-right:1px solid #e5e8ef; overflow:auto; }
.sheet-left a{ display:block; padding:14px 12px; border-bottom:1px solid #f0f3f7; color:#111; font-weight:700; }
.sheet-left a.active{ background:#f8fafc; }
.sheet-right{ overflow:auto; padding:0 6px 6px; }
.sheet-opt{ display:flex; align-items:center; gap:12px; padding:12px 8px; border-bottom:1px solid #f0f3f7; }
.sheet-bottom{ height:76px; display:flex; gap:12px; align-items:center; justify-content:space-between; padding:12px; border-top:1px solid #e5e8ef; background:#fff; }
.hidden{ display:none !important; }

/* mobile sidebar */
#mobileSidebarBackdrop{ position:fixed; inset:0; z-index:1200; display:none; background:rgba(0,0,0,.35); transition:opacity .18s ease; }
#mobileSidebar{ position:fixed; left:0; bottom:0; width:86vw; max-width:320px; background:#fff; z-index:1210; transform:translateX(-110%); transition:transform .22s cubic-bezier(.2,.9,.2,1); box-shadow: 6px 0 24px rgba(0,0,0,.12); overflow:auto; }
body.sidebar-open #mobileSidebar{ transform:translateX(0); }
body.sidebar-open #mobileSidebarBackdrop{ display:block; opacity:1; }
</style>
</head>
<body class="bg-gradient-2 <?php echo h($bodyClass); ?>">
    <div class="page-wraper header-fixed">

<?php
/* ---- Use your improved header if present (it will be fixed by CSS above) ---- */
$header_candidate = __DIR__ . '/../header_mobile.php';
if (file_exists($header_candidate)) {
    require $header_candidate;
} else {
    // fallback fixed header
    ?>
    <header class="header" role="banner" style="display:flex;align-items:center;box-sizing:border-box">
      <div style="max-width:540px;margin:0 auto;display:flex;align-items:center;justify-content:space-between;padding:0 12px;height:var(--header-h);">
        <div style="font-weight:900">Projects</div>
        <div style="display:flex;gap:8px;align-items:center">
          <button aria-label="Filters" id="openFiltersBtn" style="height:36px;border-radius:10px;border:1px solid #e5e8ef;background:#f8fafc;padding:6px 10px">Filters</button>
          <button class="menu-toggler" aria-label="Open menu" style="height:36px;border-radius:10px;border:1px solid #e5e8ef;background:#f8fafc;padding:6px 8px">
            <svg viewBox="0 0 24 24" width="20" height="20" fill="currentColor"><path d="M3 6h18M3 12h18M3 18h18"/></svg>
          </button>
        </div>
      </div>
    </header>
    <?php
}

/* ---- Sidebar: include if available, else render fallback sidebar ---- */
$sider = __DIR__ . '/../sidebar_mobile.php';
if (file_exists($sider)) {
    require $sider;
} else {
    // fallback sidebar (fixed top offset)
    ?>
    <div id="mobileSidebar" aria-hidden="true" role="complementary">
      <div style="display:flex;align-items:center;justify-content:space-between;padding:10px;border-bottom:1px solid #f0f3f7">
        <div style="font-weight:800">Menu</div>
        <button class="close-btn" id="mobileSidebarClose" aria-label="Close menu">
          <svg viewBox="0 0 24 24" width="16" height="16" fill="currentColor"><path d="M18 6L6 18M6 6l12 12"/></svg>
        </button>
      </div>
      <nav style="padding:10px">
        <a href="projects_mobile.php" style="display:block;padding:12px 6px;font-weight:700">Projects</a>
        <a href="crm_mobile.php" style="display:block;padding:12px 6px;font-weight:700">CRM</a>
        <a href="ams/index.php" style="display:block;padding:12px 6px;font-weight:700">AMS</a>
        <a href="profile.php" style="display:block;padding:12px 6px;font-weight:700">Profile</a>
      </nav>
    </div>
    <div id="mobileSidebarBackdrop" aria-hidden="true"></div>
    <?php
}
?>

<main class="container-tight" style="padding-bottom:96px">
  <!-- Tabs -->
  <div class="tabs-bar" style="margin-top:6px">
    <a class="<?php echo $tab==='featured'?'active':''; ?>" href="?tab=featured">Featured</a>
    <a class="<?php echo $tab==='all'?'active':''; ?>" href="?tab=all">All Projects</a>
  </div>

  <!-- Search -->
  <form class="search-card" method="get" action="">
    <input type="hidden" name="tab" value="<?php echo h($tab); ?>">
    <div class="search-row">
      <input class="form-control" type="text" name="q" value="<?php echo h($q); ?>" placeholder="Search project / locality / city" autocomplete="off">
      <button class="icon-btn" type="submit" aria-label="Search">
        <svg viewBox="0 0 24 24" fill="none"><circle cx="11" cy="11" r="7" stroke="currentColor" stroke-width="2"/><path d="M21 21l-4-4" stroke="currentColor" stroke-width="2"/></svg>
      </button>
      <button class="icon-btn" type="button" id="openFilters" aria-label="Open filters">
        <svg viewBox="0 0 24 24" fill="none"><path d="M3 5h18M6 12h12M10 19h4" stroke="currentColor" stroke-width="2" stroke-linecap="round"/></svg>
      </button>
    </div>
  </form>

  <!-- City chips -->
  <div class="hchips">
    <a class="chip <?php echo $chips_city===''?'active':''; ?>" href="?tab=<?php echo h($tab); ?>">All</a>
    <?php foreach($chipCities as $c){ $is=($chips_city===$c); ?>
      <a class="chip <?php echo $is?'active':''; ?>" href="?tab=<?php echo h($tab); ?>&chips_city=<?php echo urlencode($c); ?>"><?php echo h($c); ?></a>
    <?php } ?>
  </div>

  <div class="counter">Showing <?php echo min($shown,20); ?> of <?php echo (int)$total; ?> projects</div>

  <!-- List -->
  <?php
  if(!$list){
    echo '<div style="padding:12px 16px;color:#6b7380">No projects found. Try switching tabs or clearing filters.</div>';
  }
  foreach($list as $r):
    $id    = (int)($r[$C['id']] ?? 0);
    $title = $r[$C['title']] ?? 'Project';
    $city  = $C['city'] ? ($r[$C['city']] ?? '') : '';
    $loc   = $C['locality'] ? ($r[$C['locality']] ?? '') : '';
    $reg   = $C['region'] ? ($r[$C['region']] ?? '') : '';
    $sub   = trim(($loc? $loc : '').($city? ', '.$city:'').($reg? ', '.$reg:''));
    $priceText = '';
    if ($C['price_min'] && is_numeric($r[$C['price_min']] ?? null)){
      $amount = (float)$r[$C['price_min']];
      $priceText = '₹ ' . format_inr_compact($amount);
    } elseif ($C['price_txt'] && !empty($r[$C['price_txt']])){
      $priceText = (string)$r[$C['price_txt']];
    }
    $img = $C['image'] ? ($r[$C['image']] ?? '') : '';
    $tag = $C['project_tag'] ? strtolower(trim((string)($r[$C['project_tag']] ?? ''))) : '';
    $conf = $C['config'] ? ($r[$C['config']] ?? '') : '';

    // bookmarked state for this project
    $is_booked = !empty($bookmarks_map[$id]);
    $book_class = $is_booked ? 'bookmarked' : '';
  ?>
    <a class="card-proj <?php echo $book_class; ?>" href="project_details?id=<?php echo urlencode($id); ?>">
      <figure class="pimg" style="position:relative">
        <?php if($img){ ?>
          <img src="<?php echo h($img); ?>" alt="<?php echo h($title); ?>" loading="lazy" onerror="this.closest('.pimg').style.background='#dde3ea'">
        <?php } ?>
        <?php if($tag && strpos($tag,'featured')!==false){ ?>
          <span class="pchip" aria-hidden="true">
            <svg viewBox="0 0 24 24" aria-hidden="true" style="width:14px;height:14px;"><path d="M12 2l3 7h7l-5.5 4.2L18 21l-6-4-6 4 1.5-7.8L2 9h7z" fill="currentColor"/></svg>
            Featured
          </span>
        <?php } ?>
        <button class="pbookmark" type="button" tabindex="-1" aria-label="Save" aria-pressed="<?php echo $is_booked ? 'true' : 'false'; ?>" data-project-id="<?php echo h($id); ?>">
          <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M6 2h12a2 2 0 0 1 2 2v18l-8-4-8 4V4a2 2 0 0 1 2-2z"/></svg>
        </button>
      </figure>

      <div class="body">
        <h3 class="title"><?php echo h($title); ?></h3>
        <?php if($sub){ ?><div class="sub"><?php echo h($sub); ?></div><?php } ?>

        <?php if($priceText){ ?>
          <div class="price">
            <span class="lbl">Price</span>
            <span class="val"><?php echo h($priceText); ?></span>
          </div>
        <?php } ?>

        <div class="tags">
          <?php if($conf){ ?><span class="tag"><?php echo h($conf); ?></span><?php } ?>
          <?php if($tag){ ?><span class="tag"><?php echo h(ucwords($tag)); ?></span><?php } ?>
        </div>
      </div>
    </a>
  <?php endforeach; ?>
</main>

<?php
if (file_exists(__DIR__.'/../footer_mobile.php')) require __DIR__.'/../footer_mobile.php';
?>

<!-- ================== FILTER SHEET ================== -->
<div class="sheet-mask" id="sheet" aria-hidden="true">
  <form class="sheet-panel" method="get" action="">
    <input type="hidden" name="tab" value="<?php echo h($tab); ?>">
    <div class="sheet-top">
      <div>Filters</div>
      <button type="button" id="clearAll" class="btn-outline" style="height:34px;">Clear All</button>
    </div>
    <div class="sheet-body">
      <div class="sheet-left" id="catList">
        <a href="#" data-cat="price" class="active">Price</a>
        <?php if($C['city']){ ?><a href="#" data-cat="city">City</a><?php } ?>
        <?php if($C['region']){ ?><a href="#" data-cat="region">Region</a><?php } ?>
        <?php if($C['locality']){ ?><a href="#" data-cat="locality">Location</a><?php } ?>
        <?php if($C['config']){ ?><a href="#" data-cat="config">Configuration</a><?php } ?>
        <?php if($C['project_tag']){ ?><a href="#" data-cat="tags">Tags</a><?php } ?>
      </div>
      <div class="sheet-right">
        <div id="pane-price">
          <div style="padding:14px 8px">
            <div style="display:flex;justify-content:space-between;color:#6b7380;margin-bottom:6px;font-weight:800">
              <div>Minimum</div><div id="minVal"><?php echo (int)($flt['pmin']!==''?$flt['pmin']:$price_min_default); ?></div>
            </div>
            <input type="range" name="price_min" id="rMin" min="0" max="<?php echo (int)$price_max_default; ?>" step="1000" value="<?php echo (int)($flt['pmin']!==''?$flt['pmin']:$price_min_default); ?>" oninput="document.getElementById('minVal').innerText=this.value">
          </div>
          <div style="padding:14px 8px">
            <div style="display:flex;justify-content:space-between;color:#6b7380;margin-bottom:6px;font-weight:800">
              <div>Maximum</div><div id="maxVal"><?php echo (int)($flt['pmax']!==''?$flt['pmax']:$price_max_default); ?></div>
            </div>
            <input type="range" name="price_max" id="rMax" min="0" max="<?php echo (int)$price_max_default; ?>" step="1000" value="<?php echo (int)($flt['pmax']!==''?$flt['pmax']:$price_max_default); ?>" oninput="document.getElementById('maxVal').innerText=this.value">
          </div>
        </div>

        <?php if($C['city']){ ?>
        <div id="pane-city" class="hidden">
          <?php foreach($opt_city as $v){ $checked=in_array($v,$flt['city'],true); ?>
            <label class="sheet-opt">
              <input type="checkbox" name="city[]" value="<?php echo h($v); ?>" <?php echo $checked?'checked':''; ?>>
              <span style="font-size:16px"><?php echo h($v); ?></span>
            </label>
          <?php } ?>
        </div>
        <?php } ?>

        <?php if($C['region']){ ?>
        <div id="pane-region" class="hidden">
          <?php foreach($opt_region as $v){ $checked=in_array($v,$flt['region'],true); ?>
            <label class="sheet-opt">
              <input type="checkbox" name="region[]" value="<?php echo h($v); ?>" <?php echo $checked?'checked':''; ?>>
              <span style="font-size:16px"><?php echo h($v); ?></span>
            </label>
          <?php } ?>
        </div>
        <?php } ?>

        <?php if($C['locality']){ ?>
        <div id="pane-locality" class="hidden">
          <?php foreach($opt_local as $v){ $checked=in_array($v,$flt['locality'],true); ?>
            <label class="sheet-opt">
              <input type="checkbox" name="locality[]" value="<?php echo h($v); ?>" <?php echo $checked?'checked':''; ?>>
              <span style="font-size:16px"><?php echo h($v); ?></span>
            </label>
          <?php } ?>
        </div>
        <?php } ?>

        <?php if($C['config']){ ?>
        <div id="pane-config" class="hidden">
          <?php foreach($opt_conf as $v){ $checked=in_array($v,$flt['config'],true); ?>
            <label class="sheet-opt">
              <input type="checkbox" name="configuration[]" value="<?php echo h($v); ?>" <?php echo $checked?'checked':''; ?>>
              <span style="font-size:16px"><?php echo h($v); ?></span>
            </label>
          <?php } ?>
        </div>
        <?php } ?>

        <?php if($C['project_tag']){ ?>
        <div id="pane-tags" class="hidden">
          <?php foreach($opt_tags as $v){ $checked=in_array($v,$flt['tags'],true); ?>
            <label class="sheet-opt">
              <input type="checkbox" name="tags[]" value="<?php echo h($v); ?>" <?php echo $checked?'checked':''; ?>>
              <span style="font-size:16px"><?php echo h($v); ?></span>
            </label>
          <?php } ?>
        </div>
        <?php } ?>
      </div>
    </div>
    <div class="sheet-bottom">
      <button type="button" class="btn-outline" id="closeSheet">Close</button>
      <button type="submit" class="btn-primary">Apply Filters</button>
    </div>
  </form>
</div>

<!-- Mobile sidebar backdrop (already rendered above if fallback used) -->
<div id="mobileSidebarBackdrop" style="display:none" aria-hidden="true"></div>
</div>

<script>
/* Filter sheet open/close + left menu switch + clear + iOS 100vh fix */
const sheet=document.getElementById('sheet');
const openBtn=document.getElementById('openFilters');
const closeBtn=document.getElementById('closeSheet');
const clearBtn=document.getElementById('clearAll');

function lockScroll(){ document.documentElement.style.overflow='hidden'; document.body.style.overflow='hidden'; }
function unlockScroll(){ document.documentElement.style.overflow=''; document.body.style.overflow=''; }

if(openBtn){ openBtn.addEventListener('click', ()=>{ sheet.style.display='block'; lockScroll(); sheet.setAttribute('aria-hidden','false'); }); }
// header variant "Filters" button (if present)
const openFiltersBtn = document.getElementById('openFiltersBtn');
if(openFiltersBtn){ openFiltersBtn.addEventListener('click', ()=>{ sheet.style.display='block'; lockScroll(); sheet.setAttribute('aria-hidden','false'); }); }

if(closeBtn){ closeBtn.addEventListener('click', ()=>{ sheet.style.display='none'; unlockScroll(); sheet.setAttribute('aria-hidden','true'); }); }
sheet.addEventListener('click', (e)=>{ if(e.target===sheet){ sheet.style.display='none'; unlockScroll(); sheet.setAttribute('aria-hidden','true'); }});

/* Left category click */
document.querySelectorAll('#catList a').forEach(a=>{
  a.addEventListener('click', (e)=>{
    e.preventDefault();
    const cat=a.getAttribute('data-cat');
    document.querySelectorAll('#catList a').forEach(x=>x.classList.remove('active'));
    a.classList.add('active');
    document.querySelectorAll('[id^="pane-"]').forEach(p=>p.classList.add('hidden'));
    const pane=document.getElementById('pane-'+cat);
    if(pane) pane.classList.remove('hidden');
  });
});

/* Clear all */
if(clearBtn){
  clearBtn.addEventListener('click', ()=>{
    const frm=sheet.querySelector('form');
    frm.querySelectorAll('input[type=checkbox]').forEach(i=>i.checked=false);
    frm.querySelectorAll('input[type=text]').forEach(i=>i.value='');
    const rMin=document.getElementById('rMin'), rMax=document.getElementById('rMax');
    if(rMin&&rMax){
      rMin.value=<?php echo (int)$price_min_default; ?>;
      rMax.value=<?php echo (int)$price_max_default; ?>;
      document.getElementById('minVal').innerText=rMin.value;
      document.getElementById('maxVal').innerText=rMax.value;
    }
  });
}

/* iOS 100vh fix */
(function(){ const setH=()=>{ document.querySelectorAll('.sheet-panel').forEach(p=>p.style.height=(window.innerHeight - (parseInt(getComputedStyle(document.documentElement).getPropertyValue('--header-h')||60)))+'px'); }; setH(); window.addEventListener('resize', setH); })();

/* ---------------- Sidebar open/close logic ---------------- */
(function(){
  const body = document.body;
  const backdrop = document.getElementById('mobileSidebarBackdrop');
  let sidebar = document.getElementById('mobileSidebar');

  if(!sidebar) sidebar = document.querySelector('[role="complementary"], .sidebar, #mobileSidebar');

  function openSidebar(){
    body.classList.add('sidebar-open');
    if(backdrop) { backdrop.style.display='block'; backdrop.setAttribute('aria-hidden','false'); }
    if(sidebar) sidebar.setAttribute('aria-hidden','false');
  }
  function closeSidebar(){
    body.classList.remove('sidebar-open');
    if(backdrop) { backdrop.style.display='none'; backdrop.setAttribute('aria-hidden','true'); }
    if(sidebar) sidebar.setAttribute('aria-hidden','true');
  }

  document.querySelectorAll('.menu-toggler, .menu-toggle, .hamb, .menu-btn').forEach(btn=>{
    btn.addEventListener('click', function(e){
      e.preventDefault();
      if(body.classList.contains('sidebar-open')) closeSidebar(); else openSidebar();
    });
  });

  document.querySelectorAll('#mobileSidebarClose, .close-btn').forEach(btn=>{
    btn.addEventListener('click', function(e){ e.preventDefault(); closeSidebar(); });
  });

  if(backdrop) backdrop.addEventListener('click', closeSidebar);
  document.addEventListener('keydown', function(e){ if(e.key === 'Escape'){ closeSidebar(); } });

  if(sidebar) sidebar.setAttribute('aria-hidden', 'true');
  if(backdrop) backdrop.setAttribute('aria-hidden', 'true');
})();

/* ---------------- Bookmark toggle (delegated) ---------------- */
document.addEventListener('click', function(e){
  const btn = e.target.closest('.pbookmark');
  if(!btn) return;
  e.preventDefault();
  e.stopPropagation();

  const card = btn.closest('.card-proj');
  const pid = btn.getAttribute('data-project-id');
  if(!pid) return;

  btn.disabled = true;
  const form = new FormData();
  form.append('action','toggle_bookmark');
  form.append('project_id', pid);

  fetch(window.location.pathname, {
    method:'POST',
    credentials:'same-origin',
    body: form,
    headers: { 'X-Requested-With': 'XMLHttpRequest' }
  }).then(r => {
    if(!r.ok) return r.json().catch(()=>({ok:false, status:r.status}));
    return r.json();
  }).then(json => {
    if(json && json.ok){
      if(json.bookmarked){
        card.classList.add('bookmarked');
        btn.setAttribute('aria-pressed','true');
      } else {
        card.classList.remove('bookmarked');
        btn.setAttribute('aria-pressed','false');
      }
    } else {
      // optional: simple alert when unauthenticated or error
      if(json && json.error === 'invalid_project') {
        alert('Invalid project.');
      } else if(json && json.error) {
        console.warn('bookmark error', json.error);
      } else {
        console.warn('bookmark toggle failed');
      }
    }
  }).catch(err => { console.error(err); })
    .finally(()=>{ btn.disabled = false; });
});
</script>

</body>
</html>
