<?php
/* ============================================================================
 * File   : editor.php
 * Purpose: Code editor + preview (70% editor, 30% preview)
 * ========================================================================== */
require_once __DIR__ . '/functions.php';
require_login();
$user = current_user();

$project_id = isset($_GET['project_id']) ? (int)$_GET['project_id'] : 0;
$file_id    = isset($_GET['file_id']) ? (int)$_GET['file_id'] : 0;

if ($project_id <= 0) {
    die('Invalid project.');
}

// Fetch project
$stmtP = mysqli_prepare($conn, "SELECT * FROM projects WHERE id = ? LIMIT 1");
if (!$stmtP) die('DB error.');
mysqli_stmt_bind_param($stmtP, "i", $project_id);
mysqli_stmt_execute($stmtP);
$resP = mysqli_stmt_get_result($stmtP);
$project = mysqli_fetch_assoc($resP);
mysqli_stmt_close($stmtP);

if (!$project) {
    die('Project not found.');
}

// If no file specified, pick the first file of project
if ($file_id <= 0) {
    $resF = mysqli_query($conn, "SELECT id FROM project_files WHERE project_id = " . (int)$project_id . " ORDER BY relative_path ASC LIMIT 1");
    if ($resF && ($rowF = mysqli_fetch_assoc($resF))) {
        $file_id = (int)$rowF['id'];
    }
}

// Fetch file
$stmtF = mysqli_prepare($conn, "SELECT * FROM project_files WHERE id = ? AND project_id = ? LIMIT 1");
if (!$stmtF) die('DB error.');
mysqli_stmt_bind_param($stmtF, "ii", $file_id, $project_id);
mysqli_stmt_execute($stmtF);
$resF2 = mysqli_stmt_get_result($stmtF);
$file = mysqli_fetch_assoc($resF2);
mysqli_stmt_close($stmtF);

if (!$file) {
    die('File not found or not selected.');
}

// Permission check
$perm = get_file_permission($conn, $project_id, $file_id, $user['id']);
if (!$perm) {
    die('You do not have access to this file.');
}

$projectRoot = get_project_root_path($project);
$relPath     = clean_relative_path($file['relative_path']);
$fullPath    = $projectRoot . '/' . $relPath;

if (!file_exists($fullPath)) {
    $fileContents = '';
} else {
    $fileContents = file_get_contents($fullPath);
    if ($fileContents === false) $fileContents = '';
}

// List files for sidebar
$files = [];
$resList = mysqli_query($conn, "SELECT id, relative_path FROM project_files WHERE project_id = " . (int)$project_id . " ORDER BY relative_path ASC");
if ($resList) {
    while ($row = mysqli_fetch_assoc($resList)) {
        $files[] = $row;
    }
}

$canEdit = ($perm === 'edit');
$previewUrl = rtrim($project['base_url'], '/') . '/' . $relPath;
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Editor - <?php echo h($project['name']); ?></title>

<link rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/codemirror.min.css">
<link rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/theme/material-darker.min.css">

<style>
body {
    margin:0;
    font-family:Arial, sans-serif;
    background:#111;
    color:#eee;
}
.topbar2 {
    background:#222;
    padding:8px 15px;
    display:flex;
    align-items:center;
    justify-content:space-between;
    font-size:13px;
}
.topbar2 a {
    color:#fff;
    text-decoration:none;
    margin-right:8px;
}
.topbar2 a:hover {
    text-decoration:underline;
}
.main-layout {
    display:flex;
    height:calc(100vh - 40px);
}
.sidebar {
    width:20%;
    background:#181818;
    border-right:1px solid #444;
    overflow:auto;
    font-size:13px;
    padding:8px;
}
.sidebar h3 {
    margin-top:0;
    font-size:13px;
}
.sidebar a {
    color:#ddd;
    text-decoration:none;
}
.sidebar a.active {
    color:#4caf50;
    font-weight:bold;
}
.sidebar li {
    margin-bottom:3px;
}
.editor-preview {
    width:80%;
    display:flex;
}
.preview-pane {
    width:30%;
    border-right:1px solid #444;
    background:#fff;
}
.editor-pane {
    width:70%;
}
.CodeMirror {
    height:100%;
    font-size:14px;
}
.btn {
    display:inline-block;
    padding:5px 10px;
    border-radius:3px;
    font-size:12px;
    text-decoration:none;
    border:none;
    cursor:pointer;
}
.btn-save {
    background:#28a745;
    color:#fff;
}
.btn-save[disabled] {
    background:#555;
    cursor:not-allowed;
}
.btn-refresh {
    background:#007bff;
    color:#fff;
}
</style>
</head>
<body>
<div class="topbar2">
    <div>
        <a href="dashboard.php">&larr; Dashboard</a>
        <span>Project: <b><?php echo h($project['name']); ?></b></span>
        <span style="margin-left:10px;">File: <b><?php echo h($file['relative_path']); ?></b></span>
    </div>
    <div>
        <?php if ($canEdit): ?>
            <button id="btnSave" class="btn btn-save">Save</button>
        <?php else: ?>
            <button id="btnSave" class="btn btn-save" disabled title="View only">Save (no rights)</button>
        <?php endif; ?>
        <button id="btnRefresh" class="btn btn-refresh">Refresh Preview</button>
    </div>
</div>

<div class="main-layout">
    <div class="sidebar">
        <h3>Files</h3>
        <ul style="list-style:none; padding-left:0;">
            <?php foreach ($files as $f): ?>
                <li>
                    <a href="editor.php?project_id=<?php echo (int)$project_id; ?>&file_id=<?php echo (int)$f['id']; ?>"
                       class="<?php echo ((int)$f['id'] === $file_id ? 'active' : ''); ?>">
                        <?php echo h($f['relative_path']); ?>
                    </a>
                </li>
            <?php endforeach; ?>
        </ul>
    </div>

    <div class="editor-preview">
        <div class="preview-pane">
            <iframe id="previewFrame"
                    src="<?php echo h($previewUrl); ?>"
                    style="width:100%;height:100%;border:0;"></iframe>
        </div>
        <div class="editor-pane">
            <textarea id="code"><?php echo htmlspecialchars($fileContents, ENT_QUOTES, 'UTF-8'); ?></textarea>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/codemirror.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/mode/php/php.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/mode/htmlmixed/htmlmixed.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/mode/xml/xml.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/mode/javascript/javascript.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.13/mode/css/css.min.js"></script>

<script>
var editor = CodeMirror.fromTextArea(document.getElementById("code"), {
    mode: "application/x-httpd-php",
    theme: "material-darker",
    lineNumbers: true,
    lineWrapping: true,
    readOnly: <?php echo $canEdit ? 'false' : 'true'; ?>,
});

document.getElementById('btnSave').addEventListener('click', function () {
    <?php if (!$canEdit): ?>
    return;
    <?php endif; ?>

    var btn = this;
    btn.disabled = true;
    var xhr = new XMLHttpRequest();
    xhr.open("POST", "save_file.php");
    xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");

    var params = "project_id=<?php echo (int)$project_id; ?>&file_id=<?php echo (int)$file_id; ?>"+
        "&content=" + encodeURIComponent(editor.getValue());

    xhr.onreadystatechange = function () {
        if (xhr.readyState === XMLHttpRequest.DONE) {
            btn.disabled = false;
            if (xhr.status === 200) {
                // reload preview
                refreshPreview();
                alert("File saved.");
            } else {
                alert("Error saving file: " + xhr.responseText);
            }
        }
    };
    xhr.send(params);
});

function refreshPreview() {
    var frame = document.getElementById('previewFrame');
    frame.src = "<?php echo h($previewUrl); ?>" + (<?php echo strpos($previewUrl, '?') === false ? '?"' : '"&'; ?>t=" + new Date().getTime());
}

document.getElementById('btnRefresh').addEventListener('click', function () {
    refreshPreview();
});
</script>
</body>
</html>
