<?php
/* ============================================================================
 * File   : functions.php
 * Purpose: Common helper functions (auth, permissions, etc.)
 * ========================================================================== */

require_once __DIR__ . '/config.php';

function h($str) {
    return htmlspecialchars($str, ENT_QUOTES, 'UTF-8');
}

function redirect($url) {
    header('Location: ' . $url);
    exit;
}

function current_user() {
    if (!isset($_SESSION['user_id'])) {
        return null;
    }
    return [
        'id'   => $_SESSION['user_id'],
        'name' => $_SESSION['user_name'] ?? '',
        'role' => $_SESSION['user_role'] ?? 'viewer',
    ];
}

function require_login() {
    if (!isset($_SESSION['user_id'])) {
        redirect('login.php');
    }
}

function is_admin() {
    $user = current_user();
    return $user && $user['role'] === 'admin';
}

/**
 * Check if a user has permission on a given project/file
 * Returns: 'edit', 'view', or null (no access)
 */
function get_file_permission($conn, $project_id, $file_id, $user_id) {
    // Admin has full rights
    $user = current_user();
    if ($user && $user['role'] === 'admin') {
        return 'edit';
    }

    $perm = null;
    $sql = "SELECT permission FROM project_permissions 
            WHERE project_id = ? AND file_id = ? AND user_id = ? LIMIT 1";
    if ($stmt = mysqli_prepare($conn, $sql)) {
        mysqli_stmt_bind_param($stmt, "iii", $project_id, $file_id, $user_id);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_bind_result($stmt, $permission);
        if (mysqli_stmt_fetch($stmt)) {
            $perm = $permission;
        }
        mysqli_stmt_close($stmt);
    }
    return $perm;
}

function get_projects_for_user($conn, $user) {
    if (!$user) return [];

    if ($user['role'] === 'admin') {
        $sql = "SELECT * FROM projects ORDER BY created_at DESC";
        $res = mysqli_query($conn, $sql);
        $rows = [];
        if ($res) {
            while ($row = mysqli_fetch_assoc($res)) {
                $rows[] = $row;
            }
        }
        return $rows;
    } else {
        // Non-admin: only projects with at least one permission
        $rows = [];
        $sql = "SELECT DISTINCT p.* 
                FROM projects p
                INNER JOIN project_permissions pp ON pp.project_id = p.id
                WHERE pp.user_id = ?
                ORDER BY p.created_at DESC";
        if ($stmt = mysqli_prepare($conn, $sql)) {
            mysqli_stmt_bind_param($stmt, "i", $user['id']);
            mysqli_stmt_execute($stmt);
            $res = mysqli_stmt_get_result($stmt);
            if ($res) {
                while ($row = mysqli_fetch_assoc($res)) {
                    $rows[] = $row;
                }
            }
            mysqli_stmt_close($stmt);
        }
        return $rows;
    }
}

function get_project_root_path($project) {
    global $APP_ROOT;
    $rel = trim($project['base_path'], '/');
    return $APP_ROOT . '/' . $rel;
}

function clean_relative_path($path) {
    // basic security: normalize & prevent directory traversal
    $path = str_replace(['\\', '..'], ['/', ''], $path);
    $path = ltrim($path, '/');
    return $path;
}
?>
