<?php
/* ============================================================================
 * File   : project_create.php
 * Purpose: Admin - create project, upload ZIP of code + optional SQL
 * ========================================================================== */
require_once __DIR__ . '/functions.php';
require_login();
if (!is_admin()) {
    die('Access denied');
}

$errors = [];
$info   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');

    if ($name === '') {
        $errors[] = 'Project name is required.';
    }

    if (empty($_FILES['project_zip']['name'])) {
        $errors[] = 'Project ZIP file is required.';
    } elseif (pathinfo($_FILES['project_zip']['name'], PATHINFO_EXTENSION) !== 'zip') {
        $errors[] = 'Project file must be a .zip archive.';
    }

    if (empty($errors)) {
        $now = date('Y-m-d H:i:s');

        // Insert project row (base_path/base_url temp)
        $stmt = mysqli_prepare($conn, "INSERT INTO projects (name, base_path, base_url, created_at) VALUES (?,?,?,?)");
        if ($stmt) {
            $empty = '';
            mysqli_stmt_bind_param($stmt, "ssss", $name, $empty, $empty, $now);
            if (!mysqli_stmt_execute($stmt)) {
                $errors[] = 'Failed to create project record.';
            }
            $project_id = mysqli_insert_id($conn);
            mysqli_stmt_close($stmt);
        } else {
            $errors[] = 'DB error.';
        }

        if (empty($errors)) {
            // Prepare project directory
            $projectsRoot = __DIR__ . '/projects';
            if (!is_dir($projectsRoot)) {
                mkdir($projectsRoot, 0775, true);
            }
            $projectFolderName = 'project_' . $project_id;
            $projectPath       = $projectsRoot . '/' . $projectFolderName;

            if (!mkdir($projectPath, 0775, true)) {
                $errors[] = 'Failed to create project folder.';
            } else {
                // Move uploaded ZIP
                $tmpZip = $projectPath . '/upload.zip';
                if (!move_uploaded_file($_FILES['project_zip']['tmp_name'], $tmpZip)) {
                    $errors[] = 'Failed to move uploaded ZIP.';
                } else {
                    // Extract ZIP
                    $zip = new ZipArchive();
                    if ($zip->open($tmpZip) === TRUE) {
                        $zip->extractTo($projectPath);
                        $zip->close();
                        unlink($tmpZip);
                    } else {
                        $errors[] = 'Failed to extract ZIP.';
                    }
                }
            }

            // Optional DB SQL
            if (empty($errors) && !empty($_FILES['db_sql']['name'])) {
                if (pathinfo($_FILES['db_sql']['name'], PATHINFO_EXTENSION) !== 'sql') {
                    $errors[] = 'Database file must be .sql';
                } else {
                    $tmpSql = $projectPath . '/db.sql';
                    if (!move_uploaded_file($_FILES['db_sql']['tmp_name'], $tmpSql)) {
                        $errors[] = 'Failed to move uploaded SQL file.';
                    } else {
                        $sqlContent = file_get_contents($tmpSql);
                        if ($sqlContent !== false && trim($sqlContent) !== '') {
                            if (!mysqli_multi_query($conn, $sqlContent)) {
                                $errors[] = 'Error importing SQL: ' . mysqli_error($conn);
                            } else {
                                // Flush multi_query results
                                while (mysqli_more_results($conn) && mysqli_next_result($conn)) {;}
                            }
                        }
                    }
                }
            }

            if (empty($errors)) {
                // Update base_path & base_url
                $base_path = 'projects/' . $projectFolderName;
                $base_url  = 'projects/' . $projectFolderName;
                $stmt2 = mysqli_prepare($conn, "UPDATE projects SET base_path = ?, base_url = ? WHERE id = ?");
                if ($stmt2) {
                    mysqli_stmt_bind_param($stmt2, "ssi", $base_path, $base_url, $project_id);
                    mysqli_stmt_execute($stmt2);
                    mysqli_stmt_close($stmt2);
                }

                // Scan files & insert into project_files
                $rii = new RecursiveIteratorIterator(
                    new RecursiveDirectoryIterator($projectPath, RecursiveDirectoryIterator::SKIP_DOTS),
                    RecursiveIteratorIterator::SELF_FIRST
                );

                $allowedExt = ['php','html','htm','css','js'];
                foreach ($rii as $file) {
                    if ($file->isDir()) continue;
                    $ext = strtolower(pathinfo($file->getFilename(), PATHINFO_EXTENSION));
                    if (!in_array($ext, $allowedExt, true)) continue;

                    $full  = str_replace('\\', '/', $file->getPathname());
                    $rel   = substr($full, strlen($projectPath) + 1);
                    $rel   = ltrim($rel, '/');

                    $stmtF = mysqli_prepare($conn, "INSERT INTO project_files (project_id, relative_path, is_editable) VALUES (?,?,1)");
                    if ($stmtF) {
                        mysqli_stmt_bind_param($stmtF, "iss", $project_id, $rel, $dummy = 1);
                        mysqli_stmt_execute($stmtF);
                        mysqli_stmt_close($stmtF);
                    }
                }

                $info = 'Project created successfully with ID ' . $project_id . '. Now set file permissions.';
            }
        }
    }
}

?>
<?php include __DIR__ . '/header.php'; ?>

<h2>Create New Project</h2>

<?php if ($info): ?>
    <div style="color:#28a745; font-size:13px; margin-bottom:10px;"><?php echo h($info); ?></div>
<?php endif; ?>
<?php if ($errors): ?>
    <div style="color:#ff6b6b; font-size:13px; margin-bottom:10px;">
        <?php foreach ($errors as $e) echo h($e) . "<br>"; ?>
    </div>
<?php endif; ?>

<form method="post" enctype="multipart/form-data">
    <div class="form-group">
        <label>Project Name</label><br>
        <input type="text" name="name" required>
    </div>
    <div class="form-group">
        <label>Project ZIP (all files)</label><br>
        <input type="file" name="project_zip" required>
    </div>
    <div class="form-group">
        <label>Database SQL (.sql) (optional)</label><br>
        <input type="file" name="db_sql">
    </div>
    <button class="btn btn-success" type="submit">Create Project</button>
</form>

<p style="font-size:12px; margin-top:10px;">
Upload your full project source as a ZIP. System will extract into <b>/projects/project_ID/</b> and index all PHP/HTML/CSS/JS files.
Optional SQL upload will import DB schema/tables.
</p>

<?php include __DIR__ . '/footer.php'; ?>
