<?php
/* ============================================================================
 * File   : users.php
 * Purpose: Admin - manage users
 * ========================================================================== */
require_once __DIR__ . '/functions.php';
require_login();
if (!is_admin()) {
    die('Access denied');
}

$errors = [];
$info   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name  = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $pass  = trim($_POST['password'] ?? '');
    $role  = trim($_POST['role'] ?? 'viewer');

    if ($name === '' || $email === '' || $pass === '') {
        $errors[] = 'Name, Email, and Password are required.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email format.';
    } elseif (!in_array($role, ['admin','editor','viewer'], true)) {
        $errors[] = 'Invalid role.';
    } else {
        $hash = password_hash($pass, PASSWORD_DEFAULT);
        $now  = date('Y-m-d H:i:s');
        $stmt = mysqli_prepare($conn, "INSERT INTO users (name, email, password, role, created_at) VALUES (?,?,?,?,?)");
        if ($stmt) {
            mysqli_stmt_bind_param($stmt, "sssss", $name, $email, $hash, $role, $now);
            if (mysqli_stmt_execute($stmt)) {
                $info = 'User created successfully.';
            } else {
                $errors[] = 'Failed to create user. Maybe email already exists.';
            }
            mysqli_stmt_close($stmt);
        } else {
            $errors[] = 'DB error.';
        }
    }
}

$users = [];
$res = mysqli_query($conn, "SELECT id, name, email, role, created_at FROM users ORDER BY id ASC");
if ($res) {
    while ($row = mysqli_fetch_assoc($res)) {
        $users[] = $row;
    }
}
?>
<?php include __DIR__ . '/header.php'; ?>

<h2>Users</h2>

<?php if ($info): ?>
    <div style="color:#28a745; font-size:13px; margin-bottom:10px;"><?php echo h($info); ?></div>
<?php endif; ?>
<?php if ($errors): ?>
    <div style="color:#ff6b6b; font-size:13px; margin-bottom:10px;">
        <?php foreach ($errors as $e) echo h($e) . "<br>"; ?>
    </div>
<?php endif; ?>

<h3>Create New User</h3>
<form method="post" action="">
    <div class="form-group">
        <label>Name</label><br>
        <input type="text" name="name" required>
    </div>
    <div class="form-group">
        <label>Email</label><br>
        <input type="text" name="email" required>
    </div>
    <div class="form-group">
        <label>Password</label><br>
        <input type="text" name="password" required>
    </div>
    <div class="form-group">
        <label>Role</label><br>
        <select name="role">
            <option value="admin">Admin</option>
            <option value="editor">Editor</option>
            <option value="viewer" selected>Viewer</option>
        </select>
    </div>
    <button class="btn btn-success" type="submit">Add User</button>
</form>

<h3 style="margin-top:20px;">Existing Users</h3>
<table class="table">
    <tr>
        <th>ID</th><th>Name</th><th>Email</th><th>Role</th><th>Created</th>
    </tr>
    <?php if (empty($users)): ?>
        <tr><td colspan="5">No users found.</td></tr>
    <?php else: ?>
        <?php foreach ($users as $u): ?>
            <tr>
                <td><?php echo (int)$u['id']; ?></td>
                <td><?php echo h($u['name']); ?></td>
                <td><?php echo h($u['email']); ?></td>
                <td><?php echo h($u['role']); ?></td>
                <td><?php echo h($u['created_at']); ?></td>
            </tr>
        <?php endforeach; ?>
    <?php endif; ?>
</table>

<?php include __DIR__ . '/footer.php'; ?>
